import numpy as np
import os

from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase
from adari_core.plots.points import LinePlot
from adari_core.plots.panel import Panel

from .pionier_utils import PionierSetupInfo, PionierReportMixin

class PionierDarkReport(PionierReportMixin, AdariReportBase):
    def __init__(self):
        super().__init__("pionier_dark")
        self.XMIN = 200
        self.XMAX = 5000

    def parse_sof(self):

        raw_dark = None
        master_dark = None

        for filename, catg in self.inputs:
            if catg == "DARK" and master_dark is None:
                master_dark = filename
            elif catg == "DARK_CALIBRATION" and raw_dark is None:
                raw_dark = filename

        # Build and return the file name list
        file_lists = []
        if raw_dark is not None:
            if master_dark is not None:
                file_lists.append(
                    {
                        "master_im": master_dark,
                        "raw_im": raw_dark,
                    }
                )

        return file_lists

    def generate_panels(self, **kwargs):

        panels = {}

        master_im = self.hdus[0]["master_im"]
        ext = "IMAGING_DATA"
        for i in range(len(master_im)):
            if str(master_im[i].header.get("EXTNAME")) == ext:
                master_data = master_im[i].data

        # selecting outputs to plot
        data = []
        for i in master_data[self.XMIN - 1 : self.XMAX]:
            data.append(i[6:])

        # calculating averages for each output
        res = []
        for i in range(len(data)):
            res2 = [np.mean(data[i][j]) for j in range(len(data[i]))]
            res.append(res2)

        res_trans = np.array(res).T

        hdr = master_im[0].header
        procatg = hdr.get("HIERARCH ESO PRO CATG")
        p = Panel(
            4,
            14,
            height_ratios=[
                0.25,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
                0.2,
            ],
            x_stretch=2,
            y_stretch=0.4,
        )

        vspace = 0.2

        t1 = TextPlot(columns=1, v_space=vspace)
        fname = master_im.filename()
        col1 = (
            "INSTRUME: " + str(hdr.get("INSTRUME")),
            "EXTNAME: " + ext,
            "DPR.TYPE: " + str(hdr.get("HIERARCH ESO DPR TYPE")),
            "FILE NAME: " + os.path.basename(fname),
        )
        t1.add_data(col1)
        p.assign_plot(t1, 0, 0, xext=2)

        metadata = PionierSetupInfo.dark(master_im)
        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        col2 = metadata
        t2.add_data(col2)
        p.assign_plot(t2, 1, 0, xext=2)

        # Generating plots except the last
        for i in np.arange(26):
            x_label = ""
            x_major_ticks = []
            if (i % 13) + 1 == 13:
                x_label = "frame number"
                x_major_ticks = None
            line_plot = LinePlot(
                # title="Resolution",
                legend=False,
                x_label=x_label,
                x_major_ticks=x_major_ticks,
                y_label="DATA" + str(i + 1),
            )
            line_plot.add_data(
                (np.arange(self.XMIN, self.XMAX + 1), res_trans[i]),
                label="data" + str(i + 1),
                color="blue",
            )
            p.assign_plot(line_plot, 2 * (i // 13), 1 + i % 13, xext=2)

        addme = {
            "ext": ext,
            "report_name": f"pionier_{procatg}_{str(ext)}",
            "report_description": f"PIONIER wavelength calibration panel - " f"{ext}",
            "report_tags": [],
        }

        panels[p] = addme

        return panels


rep = PionierDarkReport()
