/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_imcombine_lite.h"
#include "qmost_linear.h"
#include "qmost_lininfo.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_ccdproc   qmost_ccdproc
 *
 * Basic CCD image processing
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_ccdproc.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/* Structure to store readout information for a single amplifier. */

struct amp_info {
    int biassec[4];
    int trimsec[4];
    int outsec[4];
    float osmed;
    float ossig;
    float gain;  /* ADU / e- */
    float ronvar;  /* e-^2 */
};

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_raw_qc (
    cpl_image *raw_img,
    cpl_propertylist *qclist);

static cpl_error_code qmost_flip (
    cpl_image **img,
    int swapx);

static cpl_error_code qmost_detector_regions(
    cpl_propertylist *arm_plist,
    int amp_idx,
    int namps,
    int binx,
    int biny,
    int biassec[4],
    int trimsec[4],
    int *insert_x,
    int *insert_y);

static cpl_error_code qmost_detector_get_namps(
    cpl_propertylist *pri_plist,
    cpl_propertylist *arm_plist,
    int *namps);

static cpl_error_code qmost_detector_get_bin(
    cpl_propertylist *pri_plist,
    cpl_propertylist *arm_plist,
    int *binx,
    int *biny);

static double qmost_detector_get_gain(
    cpl_propertylist *arm_plist,
    int amp_idx);

static double qmost_detector_get_ron(
    cpl_propertylist *arm_plist,
    int amp_idx);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read input image and do 2D CCD image reduction.
 *
 * The image specified by the given frame and extension is read from
 * the input file and the standard 2D CCD image reduction steps
 * performed to remove the detector signature.  The standard
 * "CCD equation" noise model is evaluated based on the gain and
 * readout noise given in the FITS headers, and used to populate a
 * variance array.  The results are returned as a pair of cpl_image
 * objects.
 *
 * The 2D image reduction steps performed are, in order:
 *  - Linearity correction, if lin_when=QMOST_LIN_BEFORE_BIAS.
 *  - Overscan correction using the overscan region of each amplifier
 *    to determine overscan level and noise.  The overscan level is
 *    then subtracted from the pixels of the amplifier.
 *  - Trimming to remove the non-illuminated regions of the detector
 *    and stitching into a single contiguous image.
 *  - Population of the bad pixel mask of the cpl_image using the
 *    master bad pixel mask.
 *  - Bias correction to remove any residual 2-D bias non-uniformity
 *    not removed by overscan correction.
 *  - Linearity correction, if lin_when=QMOST_LIN_AFTER_BIAS.
 *  - Dark correction, using a master dark frame scaled by the
 *    relative exposure times of the image being processed to the
 *    master dark frame.
 *  - Division by the detector flat field.  Due to the way the
 *    detector flat field frame is made, this also corrects gain
 *    differences between the amplifiers.  Pixels with a value of zero
 *    in the detector flat are flagged as bad in the bad pixel mask
 *    and also in the variance array by setting the variance to 0.
 *  - Axis-flipping the images to produce a consistent orientation of
 *    the spectral and spatial axes for further processing.
 *
 * Most of the individual steps above can be enabled or disabled
 * either by giving a NULL pointer for the corresponding calibration
 * frame input, or with separate flags.
 *
 * The calibration frames used in this processing should be properly
 * matched to the frame being processed, but may have different
 * binning if desired, provided this binning is compatible with that
 * of the image being processed.  Specifically, it must be possible to
 * reconcile the difference in binning by applying an integer binning
 * factor to the calibration frame (alternatively, the binning of the
 * calibration frame must be an integer divisor of the binning of the
 * raw frame).  This is typically used to process binned raw data
 * using unbinned (1x1) calibration frames.
 *
 * @param   raw_frame         (Given)    Raw image frame to process.
 * @param   extension         (Given)    FITS extension of the raw
 *                                       image to process.
 * @param   master_bpm        (Given)    Master bad pixel mask (NULL =
 *                                       none, all pixels assumed
 *                                       good).
 * @param   master_bias_img   (Given)    Master bias frame (NULL = no
 *                                       debias).
 * @param   master_bias_var   (Given)    The corresponding variance
 *                                       array for the master bias.
 * @param   master_dark_img   (Given)    Master dark frame (NULL = no
 *                                       dedark).
 * @param   master_dark_var   (Given)    The corresponding variance
 *                                       array for the master dark.
 * @param   master_detflat_img (Given)   Master flat frame (NULL = no
 *                                       flat fielding).
 * @param   master_detflat_var (Given)   The corresponding variance
 *                                       array for the master flat.
 * @param   flip              (Given)    If set, the axes will be
 *                                       swapped, to make the spectra
 *                                       parallel to the y axis.
 * @param   swapx_table       (Given)    A string of length 9
 *                                       characters specifying whether
 *                                       we should flip the image in
 *                                       in x first before swapping
 *                                       the axes, or NULL to use the
 *                                       default.  This is done to
 *                                       correct the direction of the
 *                                       spectral axis such that
 *                                       wavelength increases with
 *                                       increasing pixel coordinate.
 *                                       See note.
 * @param   oscor             (Given)    If set, subtract overscan.
 * @param   linearity         (Given)    Table of non-linearity
 *                                       coefficients (NULL = no
 *                                       linearity correction).
 * @param   lin_when          (Given)    QMOST_LIN_NEVER: no linearity
 *                                       correction.
 *                                       QMOST_LIN_BEFORE_BIAS: do the
 *                                       linearity correction before
 *                                       overscan/bias correction.
 *                                       QMOST_LIN_AFTER_BIAS: do the
 *                                       linearity correction after
 *                                       overscan/bias correction.
 * @param   processed_img     (Returned) The processed image.
 * @param   processed_var     (Returned) The corresponding variance
 *                                       image.  Can be NULL if not
 *                                       needed.
 * @param   qclist            (Modified) A propertylist to receive QC
 *                                       and DRS header information.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the computed image
 *                                         sections for the amplifier,
 *                                         overscan or illuminated
 *                                         regions are out of bounds,
 *                                         indicating a problem with
 *                                         the input FITS headers 
 *                                         specifying the detector
 *                                         geometry.
 * @retval  CPL_ERROR_BAD_FILE_FORMAT  If the image cannot be loaded
 *                                     from the file.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the input image extension
 *                                    doesn't exist or a required
 *                                    FITS header keyword was
 *                                    missing.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the swap table was invalid,
 *                                    or an input parameter was out of
 *                                    range.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT   If the calibration frames
 *                                         aren't compatible with
 *                                         (can't be rebinned by an
 *                                         integer factor to match)
 *                                         the raw frame being
 *                                         processed.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If one of the FITS header
 *                                    keywords had an incorrect
 *                                    data type (for example,
 *                                    strings in integer
 *                                    keywords).
 * @retval  QMOST_ERROR_DUMMY         If the image is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>EXPTIME</b>
 *   - <b>EXTNAME</b>
 *   - <b>ESO DET BINX</b>
 *   - <b>ESO DET BINY</b>
 *   - <b>ESO DET CHIP LIVE</b>
 *   - <b>ESO DET CHIP NX</b>
 *   - <b>ESO DET CHIP NY</b>
 *   - <b>ESO DET CHIPS</b>
 *   - <b>ESO DET OUTn GAIN</b>
 *   - <b>ESO DET OUTn INDEX</b>
 *   - <b>ESO DET OUTn NX</b>
 *   - <b>ESO DET OUTn NY</b>
 *   - <b>ESO DET OUTn OVSCX</b>
 *   - <b>ESO DET OUTn OVSCY</b>
 *   - <b>ESO DET OUTn PRSCX</b>
 *   - <b>ESO DET OUTn PRSCY</b>
 *   - <b>ESO DET OUTn RON</b>
 *   - <b>ESO DET OUTn X</b>
 *   - <b>ESO DET OUTn Y</b>
 *   - <b>ESO DET OUTPUTS</b>
 *   - <b>ESO EXPTIME</b>
 *   - <b>ESO INS PATH</b>
 *
 * @par Output DRS Headers:
 *   - <b>AMPn SECT</b>: The section of the input raw image containing
 *     amplifier n.
 *   - <b>BIASSECn</b>: The section of the input raw image containing
 *     the overscan region used for amplifier n.
 *   - <b>NAMPS</b>: The number of amplifiers per detector chip.
 *   - <b>SPATBIN</b>: The spatial binning of the image.
 *   - <b>SPECBIN</b>: The spectral binning of the image.
 *   - <b>TRIMSECn</b>: The The section of the input raw image
 *     containing the illuminated region of amplifier n.
 *
 * @par Output QC Parameters:
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *
 * @note    For bias correction: If a master bias is specified then a
 *          full 2d bias correction will be done. The values in the 2d
 *          bias image will be modified by the overscan strip levels
 *          in the science image. If no 2d master bias image is
 *          specified, then the bias will be estimated from the
 *          overscan regions of the input science image. The master
 *          bias can have matched binning or can be unbinned, and will
 *          be averaged down to match the binning of the data frame if
 *          needed. Such unmatched binning is supported, but is not
 *          recommended for bias frames where the properties of the 2D
 *          bias might depend on pixel readout timing and therefore be
 *          different for different binning or readout rate settings.
 *
 * @note    For dark correction: the master dark image should be in
 *          units of dark counts per second. This will be scaled by
 *          the exposure time of the science image. The master dark
 *          can have matched binning or can be unbinned, and will be
 *          binned down to match the binning of the data frame if
 *          needed.
 *
 * @note    For detector flat correction: this is a straight
 *          division. The master flat can have matched binning or can
 *          be unbinned, and will be averaged down to match the
 *          binning of the data frame if needed.
 *
 * @note    The binning needed for the four types of 2D calibration
 *          frames (master bias, dark, detector flat, and BPM) is
 *          inferred from the ratio of their size to the size of the
 *          data frame. This was done to avoid having to pass in
 *          extra cpl_propertylist arguments containing the FITS
 *          headers.
 *
 * @note    The string swapx_table contains 3 characters for each
 *          spectrograph, in the following order:
 *          HRS(R) HRS(G) HRS(B)
 *          LRS-A(R) LRS-A(G) LRS-A(B)
 *          LRS-B(R) LRS-B(G) LRS-B(B)
 *          where the character should be '1' to swap and '0' to not
 *          swap.  The default setting is QMOST_DEFAULT_SWAPX_TABLE.
 *
 * @author  Jonathan Irwin, CASU
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_ccdproc (
    const cpl_frame *raw_frame,
    int extension,
    cpl_mask *master_bpm,
    cpl_image *master_bias_img,
    cpl_image *master_bias_var,
    cpl_image *master_dark_img,
    cpl_image *master_dark_var,
    cpl_image *master_detflat_img,
    cpl_image *master_detflat_var,
    int flip,
    const char *swapx_table,
    int oscor,
    cpl_table *linearity,
    int lin_when,
    cpl_image **processed_img,
    cpl_image **processed_var,
    cpl_propertylist *qclist)
{
    const char *raw_filename;
    cpl_image *raw_img = NULL;
    cpl_propertylist *raw_pri_hdr = NULL;
    cpl_propertylist *raw_ext_hdr = NULL;

    int nxin, nyin;
    int detlive;
    float *indata = NULL;

    int namps, binx, biny;
    int phys_nx, phys_ny, nxout, nyout, npixout;

    struct amp_info *ampinfos = NULL;
    struct amp_info *ampinfo;
    int iamp;
    int insert_x, insert_y;
    float ron;

    cpl_image *tmp_img = NULL;
    cpl_image *tmp_var = NULL;

    float *odata = NULL;
    float *ovar = NULL;
    cpl_binary *obpm = NULL;
    int ixin, iyin, ixout, iyout, indin, indout;

    cpl_mask *tmp_bpm = NULL;
    int naxis_cal[2], naxis_var[2], blkfac[2];

    qmost_lininfo *lininfos = NULL;
    int iampt, nampslin = 0;

    float *bdata = NULL;
    float *bvar = NULL;
    float *ddata = NULL;
    float *dvar = NULL;
    float *fdata = NULL;
    float *fvar = NULL;
    int ipix;
    double dexptime;
    float exptime;
    float fac, gain, fsq, rvf;

    int spec, arm;
    char swapx;
    int specbin, spatbin, itmp;

    char *qcname = NULL;
    char *qcval = NULL;

    /* Check input and output pointers */
    cpl_ensure_code(raw_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(processed_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialize outputs to NULL for garbage collection */
    *processed_img = NULL;

    if(processed_var != NULL) {
        *processed_var = NULL;
    }

#undef TIDY
#define TIDY                                                    \
    if(raw_img != NULL) {                                       \
        cpl_image_delete(raw_img);                              \
        raw_img = NULL;                                         \
    }                                                           \
    if(raw_pri_hdr != NULL) {                                   \
        cpl_propertylist_delete(raw_pri_hdr);                   \
        raw_pri_hdr = NULL;                                     \
    }                                                           \
    if(raw_ext_hdr != NULL) {                                   \
        cpl_propertylist_delete(raw_ext_hdr);                   \
        raw_ext_hdr = NULL;                                     \
    }                                                           \
    if(ampinfos != NULL) {                                      \
        cpl_free(ampinfos);                                     \
        ampinfos = NULL;                                        \
    }                                                           \
    if(tmp_img != NULL) {                                       \
        cpl_image_delete(tmp_img);                              \
        tmp_img = NULL;                                         \
    }                                                           \
    if(tmp_var != NULL) {                                       \
        cpl_image_delete(tmp_var);                              \
        tmp_var = NULL;                                         \
    }                                                           \
    if(tmp_bpm != NULL) {                                       \
        cpl_mask_delete(tmp_bpm);                               \
        tmp_bpm = NULL;                                         \
    }                                                           \
    if(lininfos != NULL) {                                      \
        for(iampt = 0; iampt < nampslin; iampt++) {             \
            qmost_lindelinfo(lininfos + iampt);                 \
        }                                                       \
        cpl_free(lininfos);                                     \
        lininfos = NULL;                                        \
    }                                                           \
    if(*processed_img != NULL) {                                \
        cpl_image_delete(*processed_img);                       \
        *processed_img = NULL;                                  \
    }                                                           \
    if(processed_var != NULL && *processed_var != NULL) {       \
        cpl_image_delete(*processed_var);                       \
        *processed_var = NULL;                                  \
    }                                                           \
    if(qcname != NULL) {                                        \
        cpl_free(qcname);                                       \
        qcname = NULL;                                          \
    }                                                           \
    if(qcval != NULL) {                                         \
        cpl_free(qcval);                                        \
        qcval = NULL;                                           \
    }

    /* Load headers */
    raw_filename = cpl_frame_get_filename(raw_frame);
    if(raw_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get filename for "
                                     "input frame");
    }
    
    /* Load primary header */
    raw_pri_hdr = cpl_propertylist_load(raw_filename,
                                        0);
    if(raw_pri_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load input FITS primary "
                                     "header from %s",
                                     raw_filename);
    }
    
    /* Load extension header */
    raw_ext_hdr = cpl_propertylist_load(raw_filename,
                                        extension);
    if(raw_ext_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load input FITS header "
                                     "%s[%d]",
                                     raw_filename,
                                     extension);
    }
    
    /* Is detector live? */
    if(qmost_pfits_get_detlive(raw_ext_hdr, &detlive) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read live flag for %s[%d]",
                                     raw_filename,
                                     extension);
    }

    if(!detlive) {
        /* We shouldn't get to here if the detector isn't live, so
         * raise an error. */
        TIDY;
        return cpl_error_set_message(cpl_func, QMOST_ERROR_DUMMY,
                                     "detector is not live for %s[%d], "
                                     "inconsistent with first frame",
                                     raw_filename,
                                     extension);
    }

    /* Load the image */
    raw_img = cpl_image_load(raw_filename,
                             CPL_TYPE_FLOAT,
                             0,
                             extension);
    if(raw_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load input image %s[%d]",
                                     raw_filename,
                                     extension);
    }

    nxin = cpl_image_get_size_x(raw_img);
    nyin = cpl_image_get_size_y(raw_img);

    indata = cpl_image_get_data_float(raw_img);
    if(indata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer to "
                                     "input image %s[%d]",
                                     raw_filename,
                                     extension);
    }

    /* Read header information */
    if(qmost_detector_get_namps(raw_pri_hdr, raw_ext_hdr,
                                &namps) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of amplifiers "
                                     "for image 1");
    }

    if(qmost_detector_get_bin(raw_pri_hdr, raw_ext_hdr,
                              &binx, &biny) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get binning for %s[%d]",
                                     raw_filename,
                                     extension);
    }

    /* XXX should probably be doing this some other way such as bounds
     * of outsec? */
    if(qmost_cpl_propertylist_get_int(raw_ext_hdr,
                                      "ESO DET CHIP NX",
                                      &phys_nx) != CPL_ERROR_NONE ||
       qmost_cpl_propertylist_get_int(raw_ext_hdr,
                                      "ESO DET CHIP NY",
                                      &phys_ny) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get chip size for %s[%d]",
                                     raw_filename,
                                     extension);
    }

    nxout = phys_nx / binx;
    nyout = phys_ny / biny;

    npixout = nxout * nyout;

    ampinfos = cpl_calloc(namps, sizeof(struct amp_info));

    for(iamp = 0; iamp < namps; iamp++) {
        ampinfo = ampinfos + iamp;

        if(qmost_detector_regions(raw_ext_hdr,
                                  iamp+1,
                                  namps,
                                  binx,
                                  biny,
                                  ampinfo->biassec,
                                  ampinfo->trimsec,
                                  &insert_x,
                                  &insert_y) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not determine "
                                         "illuminated region for "
                                         "amplifier %d",
                                         iamp+1);
        }

        /* Bounds check */
        if(ampinfo->biassec[0] < 1 || ampinfo->biassec[0] > nxin ||
           ampinfo->biassec[1] < 1 || ampinfo->biassec[1] > nyin ||
           ampinfo->biassec[2] < 1 || ampinfo->biassec[2] > nxin ||
           ampinfo->biassec[3] < 1 || ampinfo->biassec[3] > nyin) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "overscan region is out of "
                                         "bounds for amplifier %d",
                                         iamp+1);
        }

        if(ampinfo->trimsec[0] < 1 || ampinfo->trimsec[0] > nxin ||
           ampinfo->trimsec[1] < 1 || ampinfo->trimsec[1] > nyin ||
           ampinfo->trimsec[2] < 1 || ampinfo->trimsec[2] > nxin ||
           ampinfo->trimsec[3] < 1 || ampinfo->trimsec[3] > nyin) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "illuminated region is out of "
                                         "bounds for amplifier %d",
                                         iamp+1);
        }

        ampinfo->outsec[0] = insert_x;
        ampinfo->outsec[1] = insert_y;
        ampinfo->outsec[2] = insert_x + ampinfo->trimsec[2] - ampinfo->trimsec[0];
        ampinfo->outsec[3] = insert_y + ampinfo->trimsec[3] - ampinfo->trimsec[1];

        if(ampinfo->outsec[0] < 1 || ampinfo->outsec[0] > nxout ||
           ampinfo->outsec[1] < 1 || ampinfo->outsec[1] > nyout ||
           ampinfo->outsec[2] < 1 || ampinfo->outsec[2] > nxout ||
           ampinfo->outsec[3] < 1 || ampinfo->outsec[3] > nyout) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "output region is out of "
                                         "bounds for amplifier %d",
                                         iamp+1);
        }

        /* Gain, ESO convention (ADU/e-) */
        ampinfo->gain = qmost_detector_get_gain(raw_ext_hdr, iamp+1);

        /* RON, e- */
        ron = qmost_detector_get_ron(raw_ext_hdr, iamp+1);

        /* Read noise variance, ADU^2 */
        ampinfo->ronvar = ron*ron;
    }

    /* QC on raw image (count of saturated pixels) XXX - bring inline
     * and figure out what to do about BPM, maybe we should be
     * looking only inside trimsec or something?  Think about it. */
    if(qmost_raw_qc(raw_img, qclist) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "error doing raw image QC for "
                                     "%s[%d]",
                                     raw_filename,
                                     extension);
    }

    /* Load linearity table if requested */
    if(lin_when != QMOST_LIN_NEVER && linearity != NULL) {
        if(qmost_linchk(linearity) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "invalid linearity table");
        }
        
        if(qmost_linread(linearity, &lininfos, &nampslin) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read linearity table");
        }
        
        /* Check we have enough linearity coefficients for all amps */
        if(namps > nampslin) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "linearity table doesn't have "
                                         "enough amps: image %d table %d",
                                         namps, nampslin);
        }
    }

    /* Linearity, if requested before overscan */
    if(lin_when == QMOST_LIN_BEFORE_BIAS && linearity != NULL) {
        for(iamp = 0; iamp < namps; iamp++) {
            ampinfo = ampinfos + iamp;

            qmost_lincor1_float(indata,
                                nxin,
                                ampinfo->biassec[0]-1,
                                ampinfo->biassec[2]-1,
                                ampinfo->biassec[1]-1,
                                ampinfo->biassec[3]-1,
                                lininfos + iamp);

            qmost_lincor1_float(indata,
                                nxin,
                                ampinfo->trimsec[0]-1,
                                ampinfo->trimsec[2]-1,
                                ampinfo->trimsec[1]-1,
                                ampinfo->trimsec[3]-1,
                                lininfos + iamp);
        }
    }
    
    /* Compute overscan.  This copies the overscan region, which could
     * potentially be eliminated by altering skylevel_image to take a
     * region.  See if this approach would be useful elsewhere and if
     * so we can refactor it later. */
    for(iamp = 0; iamp < namps; iamp++) {
        ampinfo = ampinfos + iamp;

        tmp_img = cpl_image_extract(raw_img,
                                     ampinfo->biassec[0],
                                     ampinfo->biassec[1],
                                     ampinfo->biassec[2],
                                     ampinfo->biassec[3]);
        if(tmp_img == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not extract overscan "
                                         "region for amp %d in "
                                         "%s[%d]",
                                         iamp+1,
                                         raw_filename,
                                         extension);
        }

        if(qmost_skylevel_image(tmp_img,
                                0, 65535,
                                -FLT_MAX, FLT_MAX, 0,
                                &(ampinfo->osmed),
                                &(ampinfo->ossig)) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute overscan stats "
                                         "for amp %d in %s[%d]",
                                         iamp+1,
                                         raw_filename,
                                         extension);
        }

        cpl_image_delete(tmp_img);
        tmp_img = NULL;
    }
    
    /* Create output */
    *processed_img = cpl_image_new(nxout, nyout, CPL_TYPE_FLOAT);
    if(*processed_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create %dx%d output "
                                     "image",
                                     nxout, nyout);
    }

    /* Overscan correction, trimming and stitching */
    odata = cpl_image_get_data_float(*processed_img);
    if(odata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer "
                                     "to output image");
    }

    if(oscor) {
        for(iamp = 0; iamp < namps; iamp++) {
            ampinfo = ampinfos + iamp;

            for(iyin = ampinfo->trimsec[1]-1, iyout = ampinfo->outsec[1]-1; iyin < ampinfo->trimsec[3]; iyin++, iyout++) {
                indin = iyin * nxin;
                indout = iyout * nxout;

                for(ixin = ampinfo->trimsec[0]-1, ixout = ampinfo->outsec[0]-1; ixin < ampinfo->trimsec[2]; ixin++, ixout++) {
                    odata[indout+ixout] = indata[indin+ixin]
                                          - ampinfo->osmed;
                }
            }
        }
    }
    else {
        for(iamp = 0; iamp < namps; iamp++) {
            ampinfo = ampinfos + iamp;

            for(iyin = ampinfo->trimsec[1]-1, iyout = ampinfo->outsec[1]-1; iyin < ampinfo->trimsec[3]; iyin++, iyout++) {
                indin = iyin * nxin;
                indout = iyout * nxout;

                for(ixin = ampinfo->trimsec[0]-1, ixout = ampinfo->outsec[0]-1; ixin < ampinfo->trimsec[2]; ixin++, ixout++) {
                    odata[indout+ixout] = indata[indin+ixin];
                }
            }
        }
    }

    /* Create variance array if requested */
    if(processed_var != NULL) {
        *processed_var = cpl_image_new(nxout, nyout, CPL_TYPE_FLOAT);
        if(*processed_var == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not create %dx%d output "
                                         "variance image",
                                         nxout, nyout);
        }
        
        ovar = cpl_image_get_data_float(*processed_var);
        if(ovar == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer "
                                         "to output variance image");
        }

        for(ipix = 0; ipix < npixout; ipix++) {
            ovar[ipix] = 0;
        }
    }

    /* Set up bad pixel mask */
    if(master_bpm != NULL) {
        /* Infer binning factor needed from the relative size of the
         * 2D calibration frame to the data frame.  It must be an
         * integer, throw an error if not. */
        naxis_cal[0] = cpl_mask_get_size_x(master_bpm);
        naxis_cal[1] = cpl_mask_get_size_y(master_bpm);
        
        blkfac[0] = naxis_cal[0] / nxout;
        blkfac[1] = naxis_cal[1] / nyout;
        
        if(naxis_cal[0] != nxout*blkfac[0] ||
           naxis_cal[1] != nyout*blkfac[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image (%d, %d), bpm (%d, %d) "
                                         "dimensions don't match for "
                                         "image %s[%d]",
                                         nxout*blkfac[0], nyout*blkfac[1],
                                         naxis_cal[0], naxis_cal[1],
                                         raw_filename,
                                         extension);
        }
        
        if(blkfac[0] != 1 || blkfac[1] != 1) {
            /* Bin down BPM to match image */
            tmp_bpm = qmost_maskblk(master_bpm, blkfac);
            if(!tmp_bpm) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not bin BPM to match "
                                             "image %s[%d]",
                                             raw_filename,
                                             extension);
            }
            
            /* Set BPM in image */
            if(cpl_image_reject_from_mask(*processed_img,
                                          tmp_bpm) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set BPM for %s[%d]",
                                             raw_filename,
                                             extension);
            }
            
            cpl_mask_delete(tmp_bpm);
            tmp_bpm = NULL;
        }
        else {
            /* Set BPM in image */
            if(cpl_image_reject_from_mask(*processed_img,
                                          master_bpm) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set BPM for %s[%d]",
                                             raw_filename,
                                             extension);
            }
        }
    }

    obpm = cpl_mask_get_data(cpl_image_get_bpm(*processed_img));
    if(obpm == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get BPM for %s[%d]",
                                     raw_filename,
                                     extension);
    }

    /* Bias subtraction */
    if(master_bias_img != NULL) {
        /* Infer binning factor needed from the relative size of the
         * 2D calibration frame to the data frame.  It must be an
         * integer, throw an error if not. */
        naxis_cal[0] = cpl_image_get_size_x(master_bias_img);
        naxis_cal[1] = cpl_image_get_size_y(master_bias_img);

        if(master_bias_var != NULL) {
            /* Check variance is the same size */
            naxis_var[0] = cpl_image_get_size_x(master_bias_var);
            naxis_var[1] = cpl_image_get_size_y(master_bias_var);

            if(naxis_cal[0] != naxis_var[0] ||
               naxis_cal[1] != naxis_var[1]) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_INCOMPATIBLE_INPUT,
                                             "bias image (%d, %d) "
                                             "and variance (%d, %d) "
                                             "dimensions don't match",
                                             naxis_cal[0], naxis_cal[1],
                                             naxis_var[0], naxis_var[1]);
            }
        }

        blkfac[0] = naxis_cal[0] / nxout;
        blkfac[1] = naxis_cal[1] / nyout;

        if(naxis_cal[0] != nxout*blkfac[0] ||
           naxis_cal[1] != nyout*blkfac[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image (%d,%d) and bias frame "
                                         "(%d, %d) dimensions don't "
                                         "match",
                                         nxout*blkfac[0], nyout*blkfac[1],
                                         naxis_cal[0], naxis_cal[1]);
        }

        if(blkfac[0] != 1 || blkfac[1] != 1) {
            tmp_img = cpl_image_rebin(master_bias_img, 1, 1,
                                      blkfac[0], blkfac[1]);
            if(tmp_img == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not bin %dx%d "
                                             "master bias image",
                                             blkfac[0], blkfac[1]);
            }

            bdata = cpl_image_get_data_float(tmp_img);
            if(bdata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "master bias image");
            }

            if(master_bias_var != NULL) {
                tmp_var = cpl_image_rebin(master_bias_var, 1, 1,
                                          blkfac[0], blkfac[1]);
                if(tmp_var == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not bin %dx%d "
                                                 "master bias variance",
                                                 blkfac[0], blkfac[1]);
                }

                bvar = cpl_image_get_data_float(tmp_var);
            }

            fac = 1.0 / (blkfac[0] * blkfac[1]);
        }
        else {
            bdata = cpl_image_get_data_float(master_bias_img);
            if(bdata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "master bias image");
            }

            if(master_bias_var != NULL) {
                bvar = cpl_image_get_data_float(master_bias_var);
            }

            fac = 1.0;
        }

        if(master_bias_var != NULL && bvar == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "master bias variance");
        }

        for(ipix = 0; ipix < npixout; ipix++) {
            odata[ipix] -= bdata[ipix] * fac;
            if(processed_var != NULL && master_bias_var != NULL) {
                ovar[ipix] += bvar[ipix] * fac * fac;
            }
        }

        if(tmp_img != NULL) {
            cpl_image_delete(tmp_img);
            tmp_img = NULL;
        }

        if(tmp_var != NULL) {
            cpl_image_delete(tmp_var);
            tmp_var = NULL;
        }
    }

    /* Linearity, if requested here */
    if(lin_when == QMOST_LIN_AFTER_BIAS && linearity != NULL) {
        for(iamp = 0; iamp < namps; iamp++) {
            ampinfo = ampinfos + iamp;

            qmost_lincor1_float(odata,
                                nxout,
                                ampinfo->outsec[0]-1,
                                ampinfo->outsec[2]-1,
                                ampinfo->outsec[1]-1,
                                ampinfo->outsec[3]-1,
                                lininfos + iamp);
        }
    }

    /* Poisson and read noise contributions to variance */
    if(processed_var != NULL) {
        for(iamp = 0; iamp < namps; iamp++) {
            ampinfo = ampinfos + iamp;

            for(iyout = ampinfo->outsec[1]-1; iyout < ampinfo->outsec[3]; iyout++) {
                indout = iyout * nxout;

                for(ixout = ampinfo->outsec[0]-1; ixout < ampinfo->outsec[2]; ixout++) {
                    ipix = indout + ixout;

                    /* The gain to use is a bit subtle.  The detector
                     * flat gain corrects all of the amps to the gain
                     * of amp 1, so if we're applying the detector
                     * flat, the correct gain to use for the overall
                     * gain^2 scaling is the gain for amp 1, which
                     * then gets changed into the correct amp by the
                     * gain correction from the detector flat below. */
                    gain = (master_detflat_img != NULL ?
                            ampinfos[0].gain :
                            ampinfo->gain);

                    /* Var(e-^2) = RON(e-)^2 + ADU / GAIN(ADU/e-)
                     * where the gain here is the gain for the
                     * amplifier in question.  Then:
                     * Var(ADU^2) = GAIN(ADU/e-)^2 Var(e-^2)
                     * where the gain squared here is from above. */
                    ovar[ipix] += gain*gain * (ampinfo->ronvar +
                                               (odata[ipix] > 0 ?
                                                odata[ipix] / ampinfo->gain :
                                                0));
                }
            }
        }
    }

    /* Dark subtraction */
    if(master_dark_img != NULL) {
        /* Get exposure time.  This is converted to float explicitly
         * here so the multiplication by the exposure time in the dark
         * correction loop below doesn't get promoted to double. */
        if(qmost_pfits_get_exptime(raw_pri_hdr,
                                   &dexptime) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read exposure "
                                         "time for %s",
                                         raw_filename);                
        }

        exptime = dexptime;

        /* Infer binning factor needed from the relative size of the
         * 2D calibration frame to the data frame.  It must be an
         * integer, throw an error if not. */
        naxis_cal[0] = cpl_image_get_size_x(master_dark_img);
        naxis_cal[1] = cpl_image_get_size_y(master_dark_img);

        if(master_dark_var != NULL) {
            /* Check variance is the same size */
            naxis_var[0] = cpl_image_get_size_x(master_dark_var);
            naxis_var[1] = cpl_image_get_size_y(master_dark_var);

            if(naxis_cal[0] != naxis_var[0] ||
               naxis_cal[1] != naxis_var[1]) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_INCOMPATIBLE_INPUT,
                                             "dark image (%d, %d) "
                                             "and variance (%d, %d) "
                                             "dimensions don't match",
                                             naxis_cal[0], naxis_cal[1],
                                             naxis_var[0], naxis_var[1]);
            }
        }

        blkfac[0] = naxis_cal[0] / nxout;
        blkfac[1] = naxis_cal[1] / nyout;

        if(naxis_cal[0] != nxout*blkfac[0] ||
           naxis_cal[1] != nyout*blkfac[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image (%d,%d) and dark frame "
                                         "(%d, %d) dimensions don't "
                                         "match",
                                         nxout*blkfac[0], nyout*blkfac[1],
                                         naxis_cal[0], naxis_cal[1]);
        }

        if(blkfac[0] != 1 || blkfac[1] != 1) {
            tmp_img = cpl_image_rebin(master_dark_img, 1, 1,
                                      blkfac[0], blkfac[1]);
            if(tmp_img == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not bin %dx%d "
                                             "master dark image",
                                             blkfac[0], blkfac[1]);
            }

            ddata = cpl_image_get_data_float(tmp_img);
            if(ddata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "master dark image");
            }

            if(master_dark_var != NULL) {
                tmp_var = cpl_image_rebin(master_dark_var, 1, 1,
                                          blkfac[0], blkfac[1]);
                if(tmp_var == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not bin %dx%d "
                                                 "master dark variance",
                                                 blkfac[0], blkfac[1]);
                }

                dvar = cpl_image_get_data_float(tmp_var);
            }
        }
        else {
            ddata = cpl_image_get_data_float(master_dark_img);
            if(ddata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "master dark image");
            }

            if(master_dark_var != NULL) {
                dvar = cpl_image_get_data_float(master_dark_var);
            }
        }

        if(master_dark_var != NULL && dvar == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "master dark variance");
        }

        for(ipix = 0; ipix < npixout; ipix++) {
            odata[ipix] -= ddata[ipix] * exptime;
            if(processed_var != NULL && master_dark_var != NULL) {
                ovar[ipix] += dvar[ipix] * exptime * exptime;
            }
        }

        if(tmp_img != NULL) {
            cpl_image_delete(tmp_img);
            tmp_img = NULL;
        }

        if(tmp_var != NULL) {
            cpl_image_delete(tmp_var);
            tmp_var = NULL;
        }
    }

    /* Detector flat */
    if(master_detflat_img != NULL) {
        /* Infer binning factor needed from the relative size of the
         * 2D calibration frame to the data frame.  It must be an
         * integer, throw an error if not. */
        naxis_cal[0] = cpl_image_get_size_x(master_detflat_img);
        naxis_cal[1] = cpl_image_get_size_y(master_detflat_img);

        if(master_detflat_var != NULL) {
            /* Check variance is the same size */
            naxis_var[0] = cpl_image_get_size_x(master_detflat_var);
            naxis_var[1] = cpl_image_get_size_y(master_detflat_var);

            if(naxis_cal[0] != naxis_var[0] ||
               naxis_cal[1] != naxis_var[1]) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_INCOMPATIBLE_INPUT,
                                             "flat image (%d, %d) "
                                             "and variance (%d, %d) "
                                             "dimensions don't match",
                                             naxis_cal[0], naxis_cal[1],
                                             naxis_var[0], naxis_var[1]);
            }
        }

        blkfac[0] = naxis_cal[0] / nxout;
        blkfac[1] = naxis_cal[1] / nyout;

        if(naxis_cal[0] != nxout*blkfac[0] ||
           naxis_cal[1] != nyout*blkfac[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image (%d,%d) and flat frame "
                                         "(%d, %d) dimensions don't "
                                         "match",
                                         nxout*blkfac[0], nyout*blkfac[1],
                                         naxis_cal[0], naxis_cal[1]);
        }

        if(blkfac[0] != 1 || blkfac[1] != 1) {
            tmp_img = cpl_image_rebin(master_detflat_img, 1, 1,
                                      blkfac[0], blkfac[1]);
            if(tmp_img == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not bin %dx%d "
                                             "master flat image",
                                             blkfac[0], blkfac[1]);
            }

            fdata = cpl_image_get_data_float(tmp_img);
            if(fdata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "master flat image");
            }

            if(master_detflat_var != NULL) {
                tmp_var = cpl_image_rebin(master_detflat_var, 1, 1,
                                          blkfac[0], blkfac[1]);
                if(tmp_var == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not bin %dx%d "
                                                 "master flat variance",
                                                 blkfac[0], blkfac[1]);
                }

                fvar = cpl_image_get_data_float(tmp_var);
            }

            fac = 1.0 / (blkfac[0] * blkfac[1]);
        }
        else {
            fdata = cpl_image_get_data_float(master_detflat_img);
            if(fdata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "master flat image");
            }

            if(master_detflat_var != NULL) {
                fvar = cpl_image_get_data_float(master_detflat_var);
            }

            fac = 1.0;
        }

        if(master_detflat_var != NULL && fvar == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "master flat variance");
        }

        /* Apply flat field.  Bad pixels or zero pixels in flat are
         * not flat fielded, and are instead flagged as bad in the
         * output.  Not flat fielding the bad pixels helps to prevent
         * large outliers in the resulting image if these pixels are
         * insensitive to light and are near zero in both image and
         * flat field.  It should otherwise be harmless given that
         * all subsequent analysis skips them. */
        if(processed_var != NULL) {
            for(ipix = 0; ipix < npixout; ipix++) {
                if(obpm[ipix] == 0 && fdata[ipix] != 0) {
                    /* d = i / f */
                    odata[ipix] /= (fdata[ipix] * fac);

                    /* var(d) = d^2 (var(i)/i^2 + var(f)/f^2)
                     *        = (i^2 / f^2) (var(i)/i^2 + var(f)/f^2)
                     *        = var(i) / f^2 + d^2 var(f) / f^2
                     * This computes the first term, which is the same
                     * whether we have a flat variance or not. */
                    fsq = fdata[ipix] * fdata[ipix] * fac * fac;
                    ovar[ipix] /= fsq;

                    /* This computes the second term */
                    if(master_detflat_var != NULL) {
                        rvf = fvar[ipix] / fsq;
                        ovar[ipix] += odata[ipix] * odata[ipix] * rvf;
                    }
                }
                else {
                    obpm[ipix] = 1;
                    ovar[ipix] = 0;
                }
            }
        }
        else {
            for(ipix = 0; ipix < npixout; ipix++) {
                if(obpm[ipix] == 0 && fdata[ipix] != 0) {
                    odata[ipix] /= (fdata[ipix] * fac);
                }
                else {
                    obpm[ipix] = 1;
                }
            }
        }

        if(tmp_img != NULL) {
            cpl_image_delete(tmp_img);
            tmp_img = NULL;
        }

        if(tmp_var != NULL) {
            cpl_image_delete(tmp_var);
            tmp_var = NULL;
        }
    }

    /* Set variance to zero to flag bad pixels */
    if(processed_var != NULL) {
        for(ipix = 0; ipix < npixout; ipix++) {
            if(obpm[ipix]) {
                ovar[ipix] = 0;
            }
        }
    }

    if(flip) {
        /* Determine spectrograph and check for x swap needed */
        if(qmost_pfits_get_spectrograph(raw_pri_hdr,
                                        &spec) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "spectrograph we're processing");
        }

        if(qmost_pfits_get_arm(raw_ext_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        /* Check table and supply default if needed */
        if(swapx_table == NULL) {
            swapx_table = QMOST_DEFAULT_SWAPX_TABLE;
        }

        if(strlen(swapx_table) != 9) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "invalid swap table");
        }

        if(spec >= 1 && spec <= 3 &&
           arm >= 1 && arm <= 3) {
            swapx = swapx_table[(spec-1) * 3 + (arm-1)];
        }
        else {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "unexpected spectrograph / arm: "
                                         "%d / %d in %s[%d]",
                                         spec, arm,
                                         raw_filename,
                                         extension);
        }
        
        if(qmost_flip(processed_img, swapx == '1') != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not flip image "
                                         "%s[%d]",
                                         raw_filename,
                                         extension);
        }

        if(processed_var != NULL) {
            if(qmost_flip(processed_var, swapx == '1') != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not flip variance "
                                             "%s[%d]",
                                             raw_filename,
                                             extension);
            }
        }
        
        for(iamp = 0; iamp < namps; iamp++) {
            ampinfo = ampinfos + iamp;

            qmost_swap(ampinfo->outsec[0], ampinfo->outsec[1], itmp);
            qmost_swap(ampinfo->outsec[2], ampinfo->outsec[3], itmp);

            if(swapx == '1') {
                itmp = ampinfo->outsec[0];
                ampinfo->outsec[0] = nyout - ampinfo->outsec[2] + 1;
                ampinfo->outsec[2] = nyout - itmp + 1;
            }
        }
    }

    specbin = binx;
    spatbin = biny;

    /* Set DRS keywords for our later use */
    cpl_propertylist_update_int(qclist, "ESO DRS NAMPS", namps);
    cpl_propertylist_set_comment(qclist, "ESO DRS NAMPS",
                                 "Number of amplifiers");
    
    cpl_propertylist_update_int(qclist, "ESO DRS SPECBIN", specbin);
    cpl_propertylist_set_comment(qclist, "ESO DRS SPECBIN",
                                 "Spectral binning");
    cpl_propertylist_update_int(qclist, "ESO DRS SPATBIN", spatbin);
    cpl_propertylist_set_comment(qclist, "ESO DRS SPATBIN",
                                 "Spatial binning");

    /* Populate QC and DRS keywords for each amp */
    for(iamp = 0; iamp < namps; iamp++) {
        ampinfo = ampinfos + iamp;

        qcname = cpl_sprintf("ESO DRS BIASSEC%d", iamp+1);
        if(qcname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for BIASSEC keyword %d", iamp+1);
        }

        qcval = cpl_sprintf("[%d:%d,%d:%d]",
                            ampinfo->biassec[0],
                            ampinfo->biassec[2],
                            ampinfo->biassec[1],
                            ampinfo->biassec[3]);
        if(qcval == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for BIASSEC value %d", iamp+1);
        }

        cpl_propertylist_update_string(qclist, qcname, qcval);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "Overscan region used");

        cpl_free(qcname);
        qcname = NULL;

        cpl_free(qcval);
        qcval = NULL;

        qcname = cpl_sprintf("ESO DRS TRIMSEC%d", iamp+1);
        if(qcname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for TRIMSEC keyword %d", iamp+1);
        }

        qcval = cpl_sprintf("[%d:%d,%d:%d]",
                            ampinfo->trimsec[0],
                            ampinfo->trimsec[2],
                            ampinfo->trimsec[1],
                            ampinfo->trimsec[3]);
        if(qcval == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for TRIMSEC value %d", iamp+1);
        }

        cpl_propertylist_update_string(qclist, qcname, qcval);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "Illuminated region used");

        cpl_free(qcname);
        qcname = NULL;

        cpl_free(qcval);
        qcval = NULL;
        
        qcname = cpl_sprintf("ESO DRS AMP%d SECT", iamp+1);
        if(qcname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for AMP%d SECT keyword", iamp+1);
        }

        qcval = cpl_sprintf("[%d:%d,%d:%d]",
                            ampinfo->outsec[0],
                            ampinfo->outsec[2],
                            ampinfo->outsec[1],
                            ampinfo->outsec[3]);
        if(qcval == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for AMP%d SECT value", iamp+1);
        }

        cpl_propertylist_update_string(qclist, qcname, qcval);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "Location of amp in result");

        cpl_free(qcname);
        qcname = NULL;

        cpl_free(qcval);
        qcval = NULL;
        
        qcname = cpl_sprintf("ESO QC OS MED AMP%d", iamp+1);
        if(qcname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for OS MED keyword %d", iamp+1);
        }

        cpl_propertylist_update_float(qclist, qcname,
                                      ampinfo->osmed);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "[ADU] Overscan level");

        cpl_free(qcname);
        qcname = NULL;
        
        qcname = cpl_sprintf("ESO QC OS RMS AMP%d", iamp+1);
        if(qcname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format string "
                                         "for OS RMS keyword %d", iamp+1);
        }

        cpl_propertylist_update_float(qclist, qcname,
                                      ampinfo->ossig);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "[ADU] Overscan RMS");

        cpl_free(qcname);
        qcname = NULL;
    }

    cpl_image_delete(raw_img);
    raw_img = NULL;
    
    cpl_propertylist_delete(raw_pri_hdr);
    raw_pri_hdr = NULL;
    
    cpl_propertylist_delete(raw_ext_hdr);
    raw_ext_hdr = NULL;

    cpl_free(ampinfos);
    ampinfos = NULL;

    if(lininfos != NULL) {
        for(iampt = 0; iampt < nampslin; iampt++) {
            qmost_lindelinfo(lininfos + iampt);
        }

        cpl_free(lininfos);
        lininfos = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   CCD process and then combine images.
 *
 * This utility function processes a set of raw frames and then stacks
 * them, discarding the individual processed images and returning the
 * stack and the FITS headers for the first file.  This is a fairly
 * common operation needed when making master calibration frames.
 * This is simply a wrapper calling qmost_ccdproc followed by
 * qmost_imcombine_lite.
 *
 * @param   raw_frames       (Given)    Frameset of raw images to
 *                                      process.
 * @param   extension        (Given)    Extension to process.
 * @param   master_bpm       (Given)    Master bad pixel mask (NULL =
 *                                      none).
 * @param   master_bias_img  (Given)    Master bias frame (NULL = no
 *                                      debias).
 * @param   master_bias_var  (Given)    The corresponding variance
 *                                      array for the master bias.
 * @param   master_dark_img  (Given)    Master dark frame (NULL = no
 *                                      dedark).
 * @param   master_dark_var  (Given)    The corresponding variance
 *                                      array for the master dark.
 * @param   master_detflat_img (Given)   Master flat frame (NULL = no
 *                                       flat fielding).
 * @param   master_detflat_var (Given)   The corresponding variance
 *                                       array for the master flat.
 * @param   flip             (Given)    If set, the axes will be
 *                                      swapped, to make the spectra
 *                                      parallel to the y axis.
 * @param   swapx_table       (Given)   A string of length 9
 *                                      characters specifying whether
 *                                      we should also flip the image
 *                                      in x after swapping the axes,
 *                                      or NULL to use the default.
 * @param   oscor            (Given)    If set, subtract overscan.
 * @param   linearity        (Given)    Table of non-linearity
 *                                      coefficients.
 * @param   lin_when         (Given)    QMOST_LIN_NEVER: no linearity
 *                                      correction.
 *                                      QMOST_LIN_BEFORE_BIAS: do the
 *                                      linearity correction before
 *                                      overscan/bias correction.
 *                                      QMOST_LIN_AFTER_BIAS: do the
 *                                      linearity correction after
 *                                      overscan/bias correction.
 * @param   combtype         (Given)    MEANCALC: the output pixel
 *                                      will be the mean of the input
 *                                      pixels.  MEDIANCALC: the
 *                                      output pixel will be the
 *                                      median of the input pixels.
 * @param   scaletype        (Given)    0: no biasing or scaling is
 *                                      done.  1: input images are
 *                                      biased by the relative offset
 *                                      of their background values
 *                                      before combining.  2: input
 *                                      images are scaled by the
 *                                      relative ratios of their
 *                                      background values before
 *                                      combining.  3: input images
 *                                      are scaled by their exposure
 *                                      times.
 * @param   fibre            (Given)    If true, treat as fibre
 *                                      spectra where the majority of
 *                                      pixels aren't illuminated.
 * @param   xrej             (Given)    If true, then an extra
 *                                      rejection cycle is performed.
 * @param   thresh           (Given)    The threshold level in terms
 *                                      of background noise for
 *                                      rejection.
 * @param   out_img          (Returned) The resulting combined image.
 * @param   out_var          (Returned) The corresponding variance
 *                                      array, or NULL if not needed.
 * @param   qclist           (Modified) A propertylist to receive QC
 *                                      and DRS header information.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the computed image
 *                                         sections for the amplifier,
 *                                         overscan or illuminated
 *                                         regions are out of bounds,
 *                                         indicating a problem with
 *                                         the input FITS headers 
 *                                         specifying the detector
 *                                         geometry.
 * @retval  CPL_ERROR_BAD_FILE_FORMAT  If the image cannot be loaded
 *                                     from the file.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no input frames in
 *                                    the frameset, an input image
 *                                    extension doesn't exist, or a
 *                                    required FITS header keyword was
 *                                    missing.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the swap table was invalid,
 *                                    or an input parameter was out of
 *                                    range.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT   If the calibration frames
 *                                         aren't compatible with
 *                                         (can't be rebinned by an
 *                                         integer factor to match)
 *                                         the raw frame being
 *                                         processed.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If one of the FITS header
 *                                    keywords had an incorrect
 *                                    data type (for example,
 *                                    strings in integer
 *                                    keywords).
 * @retval  QMOST_ERROR_DUMMY         If the image is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>EXPTIME</b>
 *   - <b>EXTNAME</b>
 *   - <b>ESO DET BINX</b>
 *   - <b>ESO DET BINY</b>
 *   - <b>ESO DET CHIP LIVE</b>
 *   - <b>ESO DET CHIP NX</b>
 *   - <b>ESO DET CHIP NY</b>
 *   - <b>ESO DET CHIPS</b>
 *   - <b>ESO DET OUTn GAIN</b>
 *   - <b>ESO DET OUTn INDEX</b>
 *   - <b>ESO DET OUTn NX</b>
 *   - <b>ESO DET OUTn NY</b>
 *   - <b>ESO DET OUTn OVSCX</b>
 *   - <b>ESO DET OUTn OVSCY</b>
 *   - <b>ESO DET OUTn PRSCX</b>
 *   - <b>ESO DET OUTn PRSCY</b>
 *   - <b>ESO DET OUTn RON</b>
 *   - <b>ESO DET OUTn X</b>
 *   - <b>ESO DET OUTn Y</b>
 *   - <b>ESO DET OUTPUTS</b>
 *   - <b>ESO EXPTIME</b>
 *   - <b>ESO INS PATH</b>
 *
 * @par Output DRS Headers:
 *   - <b>AMPn SECT</b>: The section of the output processed image
 *     containing amplifier n.
 *   - <b>BIASSECn</b>: The section of the input raw image containing
 *     the overscan region used for amplifier n.
 *   - <b>NAMPS</b>: The number of amplifiers per detector chip.
 *   - <b>SPATBIN</b>: The spatial binning of the image.
 *   - <b>SPECBIN</b>: The spectral binning of the image.
 *   - <b>TRIMSECn</b>: The The section of the input raw image
 *     containing the illuminated region of amplifier n.
 *
 * @par Output QC Parameters:
 *   - <b>IMCOMBINE MAX</b>: The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MEAN</b>: The mean background level of the frames
 *     that were combined.
 *   - <b>IMCOMBINE MIN</b>: The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b>: The average RMS of the background
 *     in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>IMCOMBINE RMS</b>: The RMS of the background levels over the
 *     frames that were combined, as a measure of how consistent the
 *     background levels were.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_ccdproc_and_combine (
    cpl_frameset *raw_frames,
    int extension,
    cpl_mask *master_bpm,
    cpl_image *master_bias_img,
    cpl_image *master_bias_var,
    cpl_image *master_dark_img,
    cpl_image *master_dark_var,
    cpl_image *master_detflat_img,
    cpl_image *master_detflat_var,
    int flip,
    const char *swapx_table,
    int oscor,
    cpl_table *linearity,
    int lin_when,
    int combtype,
    int scaletype,
    int fibre,
    int xrej,
    float thresh,
    cpl_image **out_img,
    cpl_image **out_var,
    cpl_propertylist *qclist)
{
    int ifile, nfiles;
    cpl_imagelist *processed_imglist = NULL;
    cpl_imagelist *processed_varlist = NULL;
    double *explist = NULL;

    const cpl_frame *raw_frame = NULL;
    const char *raw_filename = NULL;

    cpl_propertylist *qctmp = NULL;
    cpl_propertylist *qcuse;

    cpl_image *processed_img = NULL;
    cpl_image *processed_var = NULL;

    cpl_propertylist *raw_pri_hdr = NULL;

    cpl_ensure_code(raw_frames != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    *out_img = NULL;

    if(out_var != NULL) {
        *out_var = NULL;
    }

#undef TIDY
#define TIDY                                            \
    if(processed_imglist != NULL) {                     \
        cpl_imagelist_delete(processed_imglist);        \
        processed_imglist = NULL;                       \
    }                                                   \
    if(processed_varlist != NULL) {                     \
        cpl_imagelist_delete(processed_varlist);        \
        processed_varlist = NULL;                       \
    }                                                   \
    if(explist != NULL) {                               \
        cpl_free(explist);                              \
        explist = NULL;                                 \
    }                                                   \
    if(qctmp != NULL) {                                 \
        cpl_propertylist_delete(qctmp);                 \
        qctmp = NULL;                                   \
    }                                                   \
    if(processed_img != NULL) {                         \
        cpl_image_delete(processed_img);                \
        processed_img = NULL;                           \
    }                                                   \
    if(processed_var != NULL) {                         \
        cpl_image_delete(processed_var);                \
        processed_var = NULL;                           \
    }                                                   \
    if(raw_pri_hdr != NULL) {                           \
        cpl_propertylist_delete(raw_pri_hdr);           \
        raw_pri_hdr = NULL;                             \
    }                                                   \
    if(*out_img != NULL) {                              \
        cpl_image_delete(*out_img);                     \
        *out_img = NULL;                                \
    }                                                   \
    if(out_var != NULL && *out_var != NULL) {           \
        cpl_image_delete(*out_var);                     \
        *out_var = NULL;                                \
    }

    /* Get and check number of files */
    nfiles = cpl_frameset_get_size(raw_frames);
    if(nfiles < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "there were no input images");
    }

    /* Set up imagelists for processed images */
    processed_imglist = cpl_imagelist_new();

    if(out_var != NULL) {
        processed_varlist = cpl_imagelist_new();
    }

    /* If scaling by exposure time, we need that */
    if(scaletype == 3) {
        explist = cpl_calloc(nfiles, sizeof(double));
    }

    /* Process all files */
    for(ifile = 0; ifile < nfiles; ifile++) {
        raw_frame = cpl_frameset_get_position(raw_frames, ifile);
        if(raw_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get input frame %d "
                                         "from frameset",
                                         ifile+1);
        }

        raw_filename = cpl_frame_get_filename(raw_frame);

        if(ifile > 0) {
            qctmp = cpl_propertylist_duplicate(qclist);
            qcuse = qctmp;
        }
        else {
            qcuse = qclist;
        }
        
        if(qmost_ccdproc(raw_frame,
                         extension,
                         master_bpm,
                         master_bias_img,
                         master_bias_var,
                         master_dark_img,
                         master_dark_var,
                         master_detflat_img,
                         master_detflat_var,
                         flip,
                         swapx_table,
                         oscor,
                         linearity,
                         lin_when,
                         &processed_img,
                         out_var != NULL ?
                         &processed_var : NULL,
                         qcuse) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "CCD processing failed "
                                         "for file %d = %s[%d]",
                                         ifile+1,
                                         raw_filename,
                                         extension);
        }

        cpl_imagelist_set(processed_imglist, processed_img, ifile);
        processed_img = NULL;  /* now owned by the imagelist */

        if(out_var != NULL) {
            cpl_imagelist_set(processed_varlist, processed_var, ifile);
            processed_var = NULL;
        }

        if(qctmp != NULL) {
            cpl_propertylist_delete(qctmp);
            qctmp = NULL;
        }

        if(explist != NULL) {
            /* Get the exposure time from the PHDU */
            raw_pri_hdr = cpl_propertylist_load(raw_filename,
                                                0);
            if(raw_pri_hdr == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not load input FITS "
                                             "primary header for file "
                                             "%d = %s",
                                             ifile+1,
                                             raw_filename);
            }
            
            if(qmost_pfits_get_exptime(raw_pri_hdr,
                                       explist + ifile) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not read exposure "
                                             "time for file %d = %s",
                                             ifile+1,
                                             raw_filename);                
            }

            if(explist[ifile] < 0) {
                explist[ifile] = 1.0;
            }

            cpl_propertylist_delete(raw_pri_hdr);
            raw_pri_hdr = NULL;
        }
    }

    if(nfiles > 1) {
        /* Combine images */
        if(qmost_imcombine_lite(processed_imglist,
                                processed_varlist,
                                explist,
                                combtype,
                                scaletype,
                                fibre,
                                xrej,
                                thresh,
                                out_img,
                                out_var != NULL ?
                                out_var : NULL,
                                qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "stacking failed");
        }
    }
    else {
        /* Claim the image from the imagelist */
        *out_img = cpl_imagelist_unset(processed_imglist, 0);

        if(out_var != NULL) {
            *out_var = cpl_imagelist_unset(processed_varlist, 0);
        }
    }

    cpl_imagelist_delete(processed_imglist);
    processed_imglist = NULL;

    if(out_var != NULL) {
        cpl_imagelist_delete(processed_varlist);
        processed_varlist = NULL;
    }

    if(explist != NULL) {
        cpl_free(explist);
        explist = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do QC on an input raw image.
 *
 * Determines the number of saturated pixels in the image.
 *
 * @param   raw_img           (Given)    Raw image to process.
 * @param   qclist            (Modified) QC header information.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 *
 * @par Output QC Parameters:
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_raw_qc (
    cpl_image *raw_img,
    cpl_propertylist *qclist)
{
    cpl_image *tmp_img = NULL;
    cpl_image *img = NULL;
    float *img_buf = NULL;
    cpl_binary *bpm_buf = NULL;
    cpl_size nx, ny, ipix, npix;
    int nsat;

#undef TIDY
#define TIDY                                    \
    if(tmp_img != NULL) {                       \
        cpl_image_delete(tmp_img);              \
        tmp_img = NULL;                         \
    }

    /* Convert to float if it isn't already */
    if(cpl_image_get_type(raw_img) != CPL_TYPE_FLOAT) {
        tmp_img = cpl_image_cast(raw_img, CPL_TYPE_FLOAT);
        if(tmp_img == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not cast image to float");
        }

        img = tmp_img;
    }
    else {
        img = raw_img;
    }

    /* Get size */
    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);

    npix = nx * ny;

    /* Retrieve pointers */
    img_buf = cpl_image_get_data_float(img);
    if(img_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get image data");
    }

    bpm_buf = cpl_mask_get_data(cpl_image_get_bpm(img));
    if(bpm_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get BPM data");
    }

    /* Do the count */
    nsat = 0;

    for(ipix = 0; ipix < npix; ipix++) {
        if(bpm_buf[ipix])
            continue;

        if(img_buf[ipix] >= QMOST_SATURATE)
            nsat++;
    }

    /* Emit QC */
    cpl_propertylist_update_int(qclist,
                                "ESO QC NUM SAT",
                                nsat);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC NUM SAT",
                                 "Number of saturated pixels");

    /* Release temporary float image if we have one */
    if(tmp_img) {
        cpl_image_delete(tmp_img);
        tmp_img = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Transpose and optionally also x flip image.
 *
 * The given image is transposed and optionally x flipped using
 * block method to reduce cache misses.
 *
 * @param   img              (Modified)   Pointer to the image to
 *                                        flip.  Will be replaced by
 *                                        the flipped image.
 * @param   swapx            (Given)      If true, the image will be
 *                                        flipped in x before
 *                                        transposing.
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_flip (
    cpl_image **img,
    int swapx)
{
    const int blocksize = 16;

    int nx, ny;
    cpl_image *tmp_img = NULL;
    float *inarr, *tmparr;
    cpl_binary *inbpm, *tmpbpm;

    int x, y, iind, oind;
    int ibx, iby, nbx, nby;
    int xsub, ysub, nxsub, nysub;

    nx = cpl_image_get_size_x(*img);
    ny = cpl_image_get_size_y(*img);

    inarr = cpl_image_get_data_float(*img);
    if(inarr == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer "
                                     "to image data");
    }

    inbpm = cpl_mask_get_data(cpl_image_get_bpm(*img));

    tmp_img = cpl_image_new(ny, nx, CPL_TYPE_FLOAT);

    tmparr = cpl_image_get_data_float(tmp_img);
    tmpbpm = cpl_mask_get_data(cpl_image_get_bpm(tmp_img));

    nbx = nx / blocksize;
    if(nx % blocksize > 0) {
        nbx++;
    }

    nby = ny / blocksize;
    if(ny % blocksize > 0) {
        nby++;
    }

    for(iby = 0; iby < nby; iby++) {
        for(ibx = 0; ibx < nbx; ibx++) {
            y = iby*blocksize;

            nysub = ny - y;
            if(nysub > blocksize) {
                nysub = blocksize;
            }

            for(ysub = 0; ysub < nysub; ysub++, y++) {
                x = ibx*blocksize;

                nxsub = nx - x;
                if(nxsub > blocksize) {
                    nxsub = blocksize;
                }

                for(xsub = 0; xsub < nxsub; xsub++, x++) {
                    iind = y * nx + x;
                    oind = (swapx ? nx - x - 1 : x) * ny + y;

                    tmparr[oind] = inarr[iind];
                    tmpbpm[oind] = inbpm[iind];
                }
            }
        }
    }

    cpl_image_delete(*img);
    *img = tmp_img;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine overscan and trim sections from FITS header.
 *
 * @param   arm_plist        (Given)      The image extension FITS
 *                                        header.
 * @param   amp_idx          (Given)      Index of amplifier,
 *                                        numbering from 1.
 * @param   namps            (Given)      The total number of
 *                                        amplifiers read out.
 * @param   binx             (Given)      The detector binning in x.
 * @param   biny             (Given)      The detector binning in y.
 * @param   biassec          (Modified)   Overscan region (aka
 *                                        BIASSEC), specified as an
 *                                        array of integers
 *                                        {llx,lly,urx,ury} suitable
 *                                        to be passed in as the
 *                                        arguments to
 *                                        cpl_image_extract.
 * @param   trimsec          (Modified)   Illuminated region (aka
 *                                        TRIMSEC).
 * @param   insert_x         (Returned)   The x location where the
 *                                        amplifier should be inserted
 *                                        when stitching together the
 *                                        reduced image.
 * @param   insert_y         (Returned)   The y location where the
 *                                        amplifier should be inserted
 *                                        when stitching together the
 *                                        reduced image.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If an input value was out of
 *                                    range, or the resulting solution
 *                                    for the image sections is
 *                                    invalid.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If a required FITS header
 *                                    keyword was missing.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DET OUTn NX</b>
 *   - <b>ESO DET OUTn NY</b>
 *   - <b>ESO DET OUTn OVSCX</b>
 *   - <b>ESO DET OUTn OVSCY</b>
 *   - <b>ESO DET OUTn PRSCX</b>
 *   - <b>ESO DET OUTn PRSCY</b>
 *   - <b>ESO DET OUTn X</b>
 *   - <b>ESO DET OUTn Y</b>
 *
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_detector_regions(
    cpl_propertylist *arm_plist,
    int amp_idx,
    int namps,
    int binx,
    int biny,
    int biassec[4],
    int trimsec[4],
    int *insert_x,
    int *insert_y)
{
    char *kw = NULL;
    int amp_prscx, amp_prscy, amp_ovscx, amp_ovscy;
    int amp_nx, amp_ny, amp_lx, amp_ly;
    int amp_nxbin, amp_nybin;
    int amp_nxraw, amp_nyraw, iampx, iampy, amp_llxraw, amp_llyraw;
    int dir_x, dir_y, discard;
    int llx, lly, urx, ury;
    int llx_data, lly_data, urx_data, ury_data;

    struct {
        char *fmt;
        int *dst;
    } amp_keywords[] = {
        { "ESO DET OUT%d PRSCX", &amp_prscx },
        { "ESO DET OUT%d PRSCY", &amp_prscy },
        { "ESO DET OUT%d OVSCX", &amp_ovscx },
        { "ESO DET OUT%d OVSCY", &amp_ovscy },
        { "ESO DET OUT%d NX", &amp_nx },
        { "ESO DET OUT%d NY", &amp_ny },
        { "ESO DET OUT%d X", &amp_lx },
        { "ESO DET OUT%d Y", &amp_ly },
    };
    int ikw, nkw;
    cpl_error_code code;

    cpl_ensure_code(arm_plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(namps >= 1, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(amp_idx >= 1 && amp_idx <= namps, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(binx >= 1, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(biny >= 1, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(insert_x, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(insert_y, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(kw) {                                    \
        cpl_free(kw);                           \
        kw = NULL;                              \
    }

    /* Region defining the prescan and overscan
     * - ESO DET OUT* PRSC*: prescan
     * - ESO DET OUT* OVSC*: overscan
     * - ESO DET OUT* N*: number of pixels per axis
     *
     * Row structures:
     *
     * Top:
     * |OUT4 PRSCX|OUT4 NX|OUT4 OVSCX|OUT3 OVSCX|OUT3 NX|OUT3 PRSCX|
     * Bottom:
     * |OUT1 PRSCX|OUT1 NX|OUT1 OVSCX|OUT2 OVSCX|OUT2 NX|OUT2 PRSCX|
     *
     * Overall image structure:
     *
     * OUT3/4 PRSCY rows of "top"
     * OUT3/4 NY rows of "top"
     * OUT3/4 OVSCY rows of "top"
     * OUT1/2 OVSCY rows of "bottom"
     * OUT1/2 NY rows of "bottom"
     * OUT1/2 PRSCY rows of "bottom"
     *
     * But the correct way to figure where in the output things go
     * and the readout direction is to use X and Y which give the
     * location of the output amplifier in physical (unbinned) image
     * pixels.
     * */
    nkw = sizeof(amp_keywords) / sizeof(amp_keywords[0]);

    for(ikw = 0; ikw < nkw; ikw++) {
        kw = cpl_sprintf(amp_keywords[ikw].fmt, amp_idx);
        if(kw == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format amplifier "
                                         "keyword string");
        }

        if(!cpl_propertylist_has(arm_plist, kw)) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "overscan detector parameter "
                                         "not present");
        }

        code = qmost_cpl_propertylist_get_int(arm_plist,
                                              kw,
                                              amp_keywords[ikw].dst);

        if(code != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, code,
                                         "could not read amplifier "
                                         "keyword value");
        }

        cpl_free(kw);
        kw = NULL;
    }

    /* All quantities in the raw file are in unbinned (physical)
     * detector pixels.  Convert to binned pixels (as in raw image). */
    amp_prscx /= binx;
    amp_prscy /= biny;
    amp_ovscx /= binx;
    amp_ovscy /= biny;
    amp_nxbin = amp_nx / binx;
    amp_nybin = amp_ny / biny;

    /* Determine size of an amp in raw image.  We assume here that the
       amps are the same size.  I think this should be true for any
       realistic detector system given that the clocks are usually
       shared but we should probably check the headers satisfy our
       assumptions. */
    amp_nxraw = amp_prscx + amp_nxbin + amp_ovscx;
    amp_nyraw = amp_prscy + amp_nybin + amp_ovscy;

    /* Determine amplifier location within array */
    iampx = (amp_lx-1) / (binx * amp_nxbin);
    iampy = (amp_ly-1) / (biny * amp_nybin);

    /* Calculate position of lower left corner in raw file */
    amp_llxraw = iampx * amp_nxraw + 1;
    amp_llyraw = iampy * amp_nyraw + 1;

    /* Calculate position of lower left corner in output */
    *insert_x = iampx * amp_nxbin + 1;
    *insert_y = iampy * amp_nybin + 1;

    /* Determine which direction the readout goes.  This makes quite a
       lot of assumptions but as far as I can tell we have no choice
       given the limited header information.  Here we assume for x:

       left half of detector: amp at left
       right half of detector: amp at right

       and for y:

       top half of detector: amp at top
       bottom half of detector: amp at bottom */ 
    dir_x = iampx < (namps/2)/2 ? 1 : -1;
    dir_y = iampy < 1 ? 1 : -1;

    /* Number of overscan pixels to discard */
    discard = (3 + binx - 1) / binx;

#ifdef QMOST_USE_OVERSCAN
    /* Overscan region */
    if(dir_x > 0) {
        llx = amp_llxraw + amp_prscx + amp_nxbin + discard;
        urx = amp_llxraw + amp_prscx + amp_nxbin + amp_ovscx - 1;
    }
    else {
        llx = amp_llxraw;
        urx = amp_llxraw + amp_ovscx - discard - 1;
    }
#else
    /* Prescan region */
    if(dir_x > 0) {
        llx = amp_llxraw;
        urx = amp_llxraw + amp_prscx - discard - 1;
    }
    else {
        llx = amp_llxraw + amp_ovscx + amp_nxbin + discard;
        urx = amp_llxraw + amp_ovscx + amp_nxbin + amp_prscx - 1;
    }
#endif

    if(dir_y > 0) {
        lly = amp_llyraw + amp_prscy;
        ury = amp_llyraw + amp_prscy + amp_nybin - 1;
    }
    else {
        lly = amp_llyraw + amp_ovscy;
        ury = amp_llyraw + amp_ovscy + amp_nybin - 1;
    }

    /* ToDo add all checks
     * - oscan region same height than amplifier
     * - oscan region with more than 3 (?) pixels in width. MIN
     *   Value good enough?
     * - others?
     *
     * Check if error codes are right
     * */
    cpl_ensure_code((urx - llx) >= 3, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((ury - lly + 1) == amp_nybin, CPL_ERROR_ILLEGAL_INPUT);

    /* Define (oscan) rectangular region
     *
     * remember rhis is a double pointer given as input parameter
     * */
    biassec[0] = llx;
    biassec[1] = lly;
    biassec[2] = urx;
    biassec[3] = ury;

    /* ===========
     * Data region
     * */

    /* Region defining the data section that should be corrected */
    if(dir_x > 0) {
        llx_data = amp_llxraw + amp_prscx;
        urx_data = amp_llxraw + amp_prscx + amp_nxbin - 1;
    }
    else {
        llx_data = amp_llxraw + amp_ovscx;
        urx_data = amp_llxraw + amp_ovscx + amp_nxbin - 1;
    }

    if(dir_y > 0) {
        lly_data = amp_llyraw + amp_prscy;
        ury_data = amp_llyraw + amp_prscy + amp_nybin - 1;
    }
    else {
        lly_data = amp_llyraw + amp_ovscy;
        ury_data = amp_llyraw + amp_ovscy + amp_nybin - 1;
    }

    /* Check if the data region has the same number of pixels as the
     * physical pix defined in the header
     *
     * ToDo:
     * - shouldn't be this test in the unit test codes?
     * */
    cpl_ensure_code((urx_data - llx_data + 1) == amp_nxbin,
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((ury_data - lly_data + 1) == amp_nybin,
                    CPL_ERROR_ILLEGAL_INPUT);

    /* Finally: the data region is assigned */
    /* This variable is a double pointer given as input argument */
    trimsec[0] = llx_data;
    trimsec[1] = lly_data;
    trimsec[2] = urx_data;
    trimsec[3] = ury_data;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determines the number of amplifiers from the headers.
 *
 * @param   pri_plist   (Given)    The FITS primary header.
 * @param   arm_plist   (Given)    The FITS image extension header.
 * @param   namps       (Returned) The number of amplifiers.
 *
 * @return  cpl_error_code
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DET CHIPS</b>
 *   - <b>ESO DET OUTn INDEX</b>
 *   - <b>ESO DET OUTPUTS</b>
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_detector_get_namps(
    cpl_propertylist *pri_plist,
    cpl_propertylist *arm_plist,
    int *namps)
{
    int nouts, nchips;
    cpl_propertylist *out_ids = NULL;
    int iid, nids, out_id;
    const cpl_property *prop;
    cpl_error_code code;

#undef TIDY
#define TIDY                                                    \
    if(out_ids != NULL) {                                       \
        cpl_propertylist_delete(out_ids);                       \
        out_ids = NULL;                                         \
    }

    /* We need to figure out how many outputs there are.  The proper
     * way to do this are the keywords from the primary HDU, but these
     * might not be present. */
    if(cpl_propertylist_has(pri_plist, "ESO DET OUTPUTS") &&
       cpl_propertylist_has(pri_plist, "ESO DET CHIPS")) {
        /* ESO DET OUTPUTS is the total number of outputs across all
         * chips so we need to divide by ESO DET CHIPS. */
        if(qmost_cpl_propertylist_get_int(pri_plist,
                                          "ESO DET OUTPUTS",
                                          &nouts) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get number of outputs");
        }

        if(qmost_cpl_propertylist_get_int(pri_plist,
                                          "ESO DET CHIPS",
                                          &nchips) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get number of chips");
        }

        *namps = nouts / nchips;
    }
    else {
        /* We have to do it the hard way.  Look at the ESO DET OUT*
         * INDEX keywords and find the maximum. */
        out_ids = cpl_propertylist_new();
        code = cpl_propertylist_copy_property_regexp(
            out_ids,
            arm_plist,
            "^ESO DET OUT[0-9][0-9]* INDEX$",
            0);
        if(code != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not search output indexes");
        }

        nids = cpl_propertylist_get_size(out_ids);
        if(nids <= 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "could not find output indexes");
        }

        *namps = -1;
        
        for(iid = 0; iid < nids; iid++) {
            prop = cpl_propertylist_get_const(out_ids, iid);
            if(prop != NULL) {
                out_id = cpl_property_get_int(prop);
                if(out_id > *namps)
                    *namps = out_id;
            }
        }
            
        cpl_propertylist_delete(out_ids);
        out_ids = NULL;

        cpl_msg_warning(cpl_func,
                        "DET keywords are missing. "
                        "Counted %d amps, proceeding.",
                        *namps);
    }

    /* For 4MOST, this should be 4 so issue a warning if it's not
     * divisible by 4.  We try to proceed anyway but the output might
     * be garbled. */
    if(*namps % 4) {
        cpl_msg_warning(cpl_func,
                        "number of amps is not divisible by 4: %d", *namps);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determines the binning setting from the headers.
 *
 * @param   pri_plist   (Given)    The FITS primary header.
 * @param   arm_plist   (Given)    The FITS image extension header.
 * @param   binx        (Returned) The x axis binning.
 * @param   biny        (Returned) The y axis binning.
 *
 * @return  cpl_error_code
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DET BINX</b>
 *   - <b>ESO DET BINY</b>
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_detector_get_bin(
    cpl_propertylist *pri_plist,
    cpl_propertylist *arm_plist,
    int *binx,
    int *biny)
{
    /* Get binning.  This wasn't here in the simulated files, so
     * we default to 1 if the keywords aren't present. */
    if(cpl_propertylist_has(pri_plist, "ESO DET BINX")) {
        if(qmost_cpl_propertylist_get_int(pri_plist,
                                          "ESO DET BINX",
                                          binx) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO DET BINX "
                                         "from PHDU");
        }
    }
    else if(cpl_propertylist_has(arm_plist, "ESO DET BINX")) {
        if(qmost_cpl_propertylist_get_int(arm_plist,
                                          "ESO DET BINX",
                                          binx) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO DET BINX "
                                         "from IMAGE extension");
        }
    }
    else {
        *binx = 1;
    }

    if(cpl_propertylist_has(pri_plist, "ESO DET BINY")) {
        if(qmost_cpl_propertylist_get_int(pri_plist,
                                          "ESO DET BINY",
                                          biny) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO DET BINY "
                                         "from PHDU");
        }
    }
    else if(cpl_propertylist_has(arm_plist, "ESO DET BINY")) {
        if(qmost_cpl_propertylist_get_int(arm_plist,
                                          "ESO DET BINY",
                                          biny) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO DET BINY "
                                         "from IMAGE extension");
        }
    }
    else {
        *biny = 1;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Returns the amplifier GAIN value, as taken from the header.
 *
 * @param   arm_plist   (Given)    The image extension FITS header.
 * @param   amp_idx     (Given)    Index of amplifier, numbering from
 *                                 1.
 *
 * @return  GAIN value for the amplifier (ADU/e-), or a dummy value of
 *          1.0 if not present.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DET OUTn GAIN</b>
 */
/*----------------------------------------------------------------------------*/

static double qmost_detector_get_gain(
    cpl_propertylist *arm_plist,
    int amp_idx)
{
    /* Check inputs */
    cpl_ensure_code(arm_plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(amp_idx >= 1, CPL_ERROR_ILLEGAL_INPUT);

    double amp_gain = 1.0;
    char * kw_gain  = cpl_sprintf("%s%d %s", "ESO DET OUT", amp_idx, "GAIN");
    if(kw_gain == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format GAIN keyword string");
    }

    if(cpl_propertylist_has(arm_plist, kw_gain)){
        qmost_cpl_propertylist_get_double(arm_plist, kw_gain, &amp_gain);
    } else{
        cpl_msg_error(cpl_func, "GAIN amplifier parameter missing. Set to dummy.");
    }

    cpl_free(kw_gain);

    return amp_gain;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Returns the amplifier RON value, as taken from the header.
 *
 * @param   arm_plist   (Given)    The image extension FITS header.
 * @param   amp_idx     (Given)    Index of amplifier, numbering from
 *                                 1.
 *
 * @return  RON value for the amplifier (e-), or a dummy value of 2.5
 *          if not present.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DET OUTn RON</b>
 */
/*----------------------------------------------------------------------------*/

static double qmost_detector_get_ron(
    cpl_propertylist *arm_plist,
    int amp_idx)
{
    /* Check inputs */
    cpl_ensure_code(arm_plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(amp_idx >= 1, CPL_ERROR_ILLEGAL_INPUT);

    char * kw_ron  = cpl_sprintf("%s%d %s", "ESO DET OUT", amp_idx, "RON");
    if(kw_ron == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format RON keyword string");
    }

    double amp_ron = 2.5;

    if(cpl_propertylist_has(arm_plist, kw_ron)){
        qmost_cpl_propertylist_get_double(arm_plist, kw_ron, &amp_ron);
    } else {
        cpl_msg_error(cpl_func, "RON amplifier parameter missing. Set to dummy.");
    }

    cpl_free(kw_ron);

    return amp_ron;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine detector gain and readout noise.
 * 
 * Given a pair of detector flat frames and, optionally, a pair of
 * bias frames, measures the detector gain and readout noise.  The
 * results are returned as QC headers in a propertylist.
 *
 * If no bias frames are given, the overscan regions are used to
 * measure the bias level and readout noise.
 *
 * @param   images                 (Given)    A pair of detector
 *                                            flats with the same
 *                                            exposure time and
 *                                            illumination level,
 *                                            optionally followed by a
 *                                            pair of biases.  These
 *                                            images must all be taken
 *                                            with the same readout
 *                                            settings.
 * @param   pri_hdr                (Given)    The FITS primary
 *                                            header for each image.
 * @param   ext_hdr                (Given)    The FITS header from the
 *                                            image extension for each
 *                                            image.  This can be the
 *                                            same set of property
 *                                            lista as the previous
 *                                            argument if the
 *                                            necessary DET keywords
 *                                            have already been
 *                                            aggregated into a single
 *                                            propertylist.
 * @param   nimgin                            The number of input
 *                                            images.  Must be 2 or 4.
 * @param   master_bpm             (Given)    Master bad pixel mask,
 *                                            or NULL if none.
 * @param   qclist                 (Modified) QC header information.
 *                                            Caller allocated and
 *                                            will be filled in.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND     If a required FITS header
 *                                       keyword was missing.
 * @retval  CPL_ERROR_NULL_INPUT         If one of the required input
 *                                       or output pointers was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT      If an input value was out of
 *                                       range.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT If the input images don't
 *                                       match.  The things checked
 *                                       are: number of amplifiers,
 *                                       binning, and illuminated
 *                                       region ("TRIMSEC").
 * @retval  CPL_ERROR_TYPE_MISMATCH      If one of the FITS header
 *                                       keywords had an incorrect
 *                                       data type (for example,
 *                                       strings in integer
 *                                       keywords).
 *
 * @note    This routine should be fairly generic for multi amplifier
 *          CCD data.
 *
 * @par Input FITS Header Information:
 *   - <b>EXTNAME</b>
 *   - <b>ESO DET BINX</b>
 *   - <b>ESO DET BINY</b>
 *   - <b>ESO DET CHIPS</b>
 *   - <b>ESO DET OUTn INDEX</b>
 *   - <b>ESO DET OUTn NX</b>
 *   - <b>ESO DET OUTn NY</b>
 *   - <b>ESO DET OUTn OVSCX</b>
 *   - <b>ESO DET OUTn OVSCY</b>
 *   - <b>ESO DET OUTn PRSCX</b>
 *   - <b>ESO DET OUTn PRSCY</b>
 *   - <b>ESO DET OUTn X</b>
 *   - <b>ESO DET OUTn Y</b>
 *   - <b>ESO DET OUTPUTS</b>
 *
 * @par Output QC Parameters:
 *   - <b>BIAS MED AMPn</b> (ADU): The median level of the bias in
 *     amplifier n.
 *   - <b>BIAS DIFF AMPn</b> (ADU): The median level of the bias
 *     difference image for amplifier n.
 *   - <b>FLAT MED AMPn</b> (ADU): The median bias-corrected flat
 *     level in amplifier n.
 *   - <b>FLAT DIFF AMPn</b> (ADU): The median level of the flat
 *     difference image for amplifier n.
 *   - <b>CONAD AMPn</b> (e-/ADU): The measured conversion gain for
 *     amplifier n.
 *   - <b>READNOISE AMPn</b> (ADU): The measured readout noise for
 *     amplifier n.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_findgain (
    cpl_image **images,
    cpl_propertylist **pri_hdr,
    cpl_propertylist **ext_hdr,
    int nimgin,
    cpl_mask *master_bpm,
    cpl_propertylist *qclist)
{
    int have_bias;
    int iamp, namps = 0, binx = 0, biny = 0;
    int this_namps, this_binx, this_biny;
    const char *extname = NULL, *this_extname;

    cpl_mask *binned_bpm = NULL;
    cpl_mask *tmp_bpm = NULL;
    int blkfac[2];

    int biassec[4], trimsec[4], this_biassec[4], this_trimsec[4];
    int insert_x, insert_y;

    cpl_image *amp_images[4] = { NULL, NULL, NULL, NULL };
    int iimg, iimgt;

    /* Things to compute */
    float flatsummed, flatsumsig, flatdiffmed, flatdiffsig;
    float biassummed, biassumsig, biasdiffmed, biasdiffsig;

    /* Table of operations needed to produce them */
    struct {
        /* Indices of inputs in amp_images array */
        int ina;
        int inb;
        /* This unreadable mess declares a function pointer suitable
         * for the two CPL functions performing the image arithmetic
         * operations.  Gotta love C syntax for declaring function
         * pointers. */ 
        cpl_image *(*op) (const cpl_image *, const cpl_image *);
        /* Name when printing things */
        char *name;
        /* Range of image counts to histogram */
        int minlev;
        int maxlev;
        /* Pointers to where result should go */
        float *med;
        float *sig;
    } ops_to_do[4] = {
        { 0, 1, cpl_image_add_create, "flat sum", 0, 131071,
          &flatsummed, &flatsumsig },
        { 0, 1, cpl_image_subtract_create, "flat difference", -65536, 65535,
          &flatdiffmed, &flatdiffsig },
        { 2, 3, cpl_image_add_create, "bias sum", 0, 131071,
          &biassummed, &biassumsig },
        { 2, 3, cpl_image_subtract_create, "bias difference", -65536, 65535,
          &biasdiffmed, &biasdiffsig }
    };
    int iop, nops;

    cpl_image *tmp_image = NULL;
    char *qcname = NULL;

    float noisvar, conad, rdnois;

    /* Check the required pair of detector flats are present */
    for(iimg = 0; iimg < nimgin; iimg++) {
        cpl_ensure_code(images[iimg] != NULL, CPL_ERROR_NULL_INPUT);
        cpl_ensure_code(pri_hdr[iimg] != NULL, CPL_ERROR_NULL_INPUT);
        cpl_ensure_code(ext_hdr[iimg] != NULL, CPL_ERROR_NULL_INPUT);
    }

    /* Other checks */
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(binned_bpm != NULL) {                    \
        cpl_mask_delete(binned_bpm);            \
        binned_bpm = NULL;                      \
    }                                           \
    if(tmp_bpm != NULL) {                       \
        cpl_mask_delete(tmp_bpm);               \
        tmp_bpm = NULL;                         \
    }                                           \
    for(iimgt = 0; iimgt < 4; iimgt++) {        \
        if(amp_images[iimgt] != NULL) {         \
            cpl_image_delete(amp_images[iimgt]); \
            amp_images[iimgt] = NULL;           \
        }                                       \
    }                                           \
    if(tmp_image != NULL) {                     \
        cpl_image_delete(tmp_image);            \
        tmp_image = NULL;                       \
    }                                           \
    if(qcname != NULL) {                        \
        cpl_free(qcname);                       \
        qcname = NULL;                          \
    }

    /* Were we given bias frames? */
    if(nimgin == 4) {
        have_bias = 1;
    }
    else if(nimgin == 2) {
        have_bias = 0;
    }
    else {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "invalid number of images: %d, "
                                     "must be 2 or 4",
                                     nimgin);
    }

    /* Read headers, checking for consistency with first file */
    for(iimg = 0; iimg < nimgin; iimg++) {
        if(qmost_detector_get_namps(pri_hdr[iimg], ext_hdr[iimg],
                                    &this_namps) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get number of amplifiers "
                                         "for image %d", iimg+1);
        }
        
        if(iimg == 0) {
            namps = this_namps;
        }
        else if(this_namps != namps) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "number of amps does not match "
                                         "for image %d: %d != %d",
                                         iimg+1, this_namps, namps);
        }

        if(qmost_detector_get_bin(pri_hdr[iimg], ext_hdr[iimg],
                                  &this_binx, &this_biny)  != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get binning "
                                         "for image %d", iimg+1);
        }

        if(iimg == 0) {
            binx = this_binx;
            biny = this_biny;
        }
        else if(this_binx != binx || this_biny != biny) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "binning does not match "
                                         "for image %d: %dx%d != %dx%d",
                                         iimg+1,
                                         this_binx, this_biny,
                                         binx, biny);
        }

        if(cpl_propertylist_has(ext_hdr[iimg], "EXTNAME")) {
            this_extname = cpl_propertylist_get_string(ext_hdr[iimg],
                                                       "EXTNAME");
        }
        else {
            this_extname = "UNKNOWN";
        }

        if(iimg == 0) {
            extname = this_extname;
        }
        else if(strcmp(this_extname, extname) != 0) {
            cpl_msg_warning(cpl_func,
                            "EXTNAME does not match for image %d: "
                            "%s != %s.  This might mean arm order "
                            "in file doesn't match causing results "
                            "to be incorrect.",
                            iimg+1, this_extname, extname);
        }
    }

    /* Bin bad pixel mask to match image */
    if(master_bpm != NULL) {
        blkfac[0] = binx;
        blkfac[1] = biny;
        
        binned_bpm = qmost_maskblk(master_bpm, blkfac);
        if(!binned_bpm) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not bin BPM to match "
                                         "images");
        }
    }

    /* Initialize these to prevent compiler warning */
    memset(biassec, 0, sizeof(biassec));
    memset(trimsec, 0, sizeof(trimsec));

    /* Loop over amplifiers */
    for(iamp = 0; iamp < namps; iamp++) {
        /* Loop over images */
        for(iimg = 0; iimg < nimgin; iimg++) {
            /* Overscan and illuminated regions for this amp */
            if(qmost_detector_regions(ext_hdr[iimg],
                                      iamp+1,
                                      namps,
                                      binx,
                                      biny,
                                      this_biassec,
                                      this_trimsec,
                                      &insert_x,
                                      &insert_y) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not determine "
                                             "illuminated region for "
                                             "image %d amplifier %d",
                                             iimg+1, iamp+1);
            }

            if(iimg == 0) {
                memcpy(biassec, this_biassec, sizeof(biassec));
                memcpy(trimsec, this_trimsec, sizeof(trimsec));
            }
            else if(this_trimsec[0] != trimsec[0] ||
                    this_trimsec[1] != trimsec[1] ||
                    this_trimsec[2] != trimsec[2] ||
                    this_trimsec[3] != trimsec[3]) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_INCOMPATIBLE_INPUT,
                                             "illuminated region does not "
                                             "match for image %d",
                                             iimg+1);
            }

            /* Extract illuminated region */
            amp_images[iimg] = cpl_image_extract(images[iimg],
                                                 this_trimsec[0],
                                                 this_trimsec[1],
                                                 this_trimsec[2],
                                                 this_trimsec[3]);
            if(amp_images[iimg] == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not extract illuminated "
                                             "region for image %d amp %d",
                                             iimg+1, iamp+1);
            }

            /* If we're not using bias frames, check and extract the
             * overscan region and put it into the slot for the bias. */
            if(have_bias == 0) {
                if(this_biassec[0] != biassec[0] ||
                   this_biassec[1] != biassec[1] ||
                   this_biassec[2] != biassec[2] ||
                   this_biassec[3] != biassec[3]) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 CPL_ERROR_INCOMPATIBLE_INPUT,
                                                 "overscan region does not "
                                                 "match for image %d",
                                                 iimg+1);
                }
                
                amp_images[iimg+2] = cpl_image_extract(images[iimg],
                                                       this_biassec[0],
                                                       this_biassec[1],
                                                       this_biassec[2],
                                                       this_biassec[3]);
                if(amp_images[iimg+2] == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not extract overscan "
                                                 "region for image %d amp %d",
                                                 iimg+1, iamp+1);
                }
            }

            /* Extract BPM and set in image.  The BPM is for illuminated
             * pixels only so is already in the "trimmed" coordinate
             * system. */
            if(binned_bpm != NULL) {
                tmp_bpm = cpl_mask_extract(
                    binned_bpm,
                    insert_x,
                    insert_y,
                    insert_x + (this_trimsec[2] - this_trimsec[0]),
                    insert_y + (this_trimsec[3] - this_trimsec[1]));
                if(tmp_bpm == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not extract BPM at "
                                                 "%d,%d for image %d amp %d, "
                                                 "sizes may be incompatible",
                                                 insert_x, insert_y,
                                                 iimg+1, iamp+1);
                }
                
                if(cpl_image_reject_from_mask(amp_images[iimg],
                                              tmp_bpm) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not set BPM "
                                                 "for image %d amp %d",
                                                 iimg+1, iamp+1);
                }
                
                cpl_mask_delete(tmp_bpm);
                tmp_bpm = NULL;
            }
        }

        /* This loop computes all of the sums and differences we need,
         * per the table of operations defined above. */
        nops = sizeof(ops_to_do) / sizeof(ops_to_do[0]);

        for(iop = 0; iop < nops; iop++) {
            tmp_image = ops_to_do[iop].op(amp_images[ops_to_do[iop].ina],
                                          amp_images[ops_to_do[iop].inb]);
            if(tmp_image == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not compute %s "
                                             "for amp %d",
                                             ops_to_do[iop].name,
                                             iamp+1);
            }

            if(qmost_skylevel_image(tmp_image,
                                    ops_to_do[iop].minlev,
                                    ops_to_do[iop].maxlev,
                                    -FLT_MAX, FLT_MAX, 0,
                                    ops_to_do[iop].med,
                                    ops_to_do[iop].sig) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not compute %s "
                                             "statistics for amp %d",
                                             ops_to_do[iop].name,
                                             iamp+1);
            }
            
            cpl_image_delete(tmp_image);
            tmp_image = NULL;
        }
        
        /* Write QC headers */
        qcname = cpl_sprintf("ESO QC BIAS MED AMP%d", iamp+1);

        cpl_propertylist_update_float(qclist, qcname,
                                      0.5*biassummed);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "[ADU] Bias level");

        cpl_free(qcname);
        qcname = NULL;

        qcname = cpl_sprintf("ESO QC BIAS DIFF AMP%d", iamp+1);

        cpl_propertylist_update_float(qclist, qcname,
                                      biasdiffmed);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "[ADU] Median bias difference");

        cpl_free(qcname);
        qcname = NULL;

        qcname = cpl_sprintf("ESO QC FLAT MED AMP%d", iamp+1);

        cpl_propertylist_update_float(qclist, qcname,
                                      0.5*(flatsummed - biassummed));
        cpl_propertylist_set_comment(qclist, qcname,
                                     "[ADU] Bias corrected flat level");

        cpl_free(qcname);
        qcname = NULL;

        qcname = cpl_sprintf("ESO QC FLAT DIFF AMP%d", iamp+1);

        cpl_propertylist_update_float(qclist, qcname,
                                      flatdiffmed);
        cpl_propertylist_set_comment(qclist, qcname,
                                     "[ADU] Median flat difference");

        cpl_free(qcname);
        qcname = NULL;

        /* Calculate poisson noise variance in flat difference = 2 *
         * Poisson noise in flat by taking off the read noise. */
        noisvar = flatdiffsig*flatdiffsig - biasdiffsig*biasdiffsig;

        if(noisvar > 0) {
            /* The variable noisvar is twice the measured Poisson
             * noise in the flat (the Poission noise in each flat,
             * added in quadrature).  If the gain was 1 then this
             * should equal twice the counts in the flat, or (more
             * conveniently) the sum of the flat counts.  This is
             * equal to flatsummed - biassummed to correct the bias
             * level.  If the gain is not 1 then the ratio of these
             * quantities gives the gain.  Which way up is a matter of
             * taste, the standard in astronomy is to call e-/ADU gain
             * but ESO call it CONAD and call ADU/e- GAIN.  This is
             * technically more correct, higher numbers would then
             * correspond to more gain in the electronics, but this is
             * astronomy so the conventions often don't make sense.
             * Confused yet? */
            conad = (flatsummed - biassummed) / noisvar;

            /* Read noise in electrons */
            rdnois = conad * biasdiffsig * CPL_MATH_SQRT1_2;

            /* Print result */
            cpl_msg_info(cpl_func,
                         "%s Amplifier %d "
                         "CONAD = %.3f e-/ADU "
                         "READNOISE = %.3f e-",
                         extname, iamp+1, conad, rdnois);

            /* Write QC headers */
            qcname = cpl_sprintf("ESO QC CONAD AMP%d", iamp+1);

            cpl_propertylist_update_float(qclist, qcname, conad);
            cpl_propertylist_set_comment(qclist, qcname,
                                         "[e-/ADU] Amplifier gain");
            
            cpl_free(qcname);
            qcname = NULL;

            qcname = cpl_sprintf("ESO QC READNOISE AMP%d", iamp+1);

            cpl_propertylist_update_float(qclist, qcname, rdnois);
            cpl_propertylist_set_comment(qclist, qcname,
                                         "[e-] Readout noise");
            
            cpl_free(qcname);
            qcname = NULL;
        }
        else {
            /* Otherwise, can't compute, which is recorded by not
             * producing an answer. */
            cpl_msg_warning(cpl_func,
                            "Can't compute gain for amp %d, "
                            "flat variance was too low",
                            iamp+1);
        }

        /* Clean up */
        for(iimg = 0; iimg < 4; iimg++) {
            cpl_image_delete(amp_images[iimg]);
            amp_images[iimg] = NULL;
        }
    }

    if(binned_bpm != NULL) {
        cpl_mask_delete(binned_bpm);
        binned_bpm = NULL;
    }

    return CPL_ERROR_NONE;
}

/**@}*/
