/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include <string.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_dfs  qmost_dfs
 *
 * DFS related functions
 * @par Synopsis:
 * @code
 *   #include "qmost_dfs.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                        Defines
 -----------------------------------------------------------------------------*/

/* This isn't exported from CPL ?! */
#undef PRO_DID
#define PRO_DID "PRO-1.16"

/*----------------------------------------------------------------------------*/
/**
 * @brief    Set the group as RAW or CALIB in a frameset
 *
 * @param    set    The frameset to process
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If the frameset was NULL.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code qmost_dfs_set_groups(cpl_frameset *set) {
    /* Check entries
     * cpl_ensure_code: set an error code and return it iff a boolean
     * expression is false.
     * Not sure
     * */
    cpl_ensure_code(set, CPL_ERROR_NULL_INPUT);

    /* Initialize */
    cpl_size nframes = cpl_frameset_get_size(set);

    /* Loop on frames */
    for (cpl_size i = 0; i < nframes; i++) {

        cpl_frame *cur_frame = cpl_frameset_get_position(set, i);
        const char *tag = cpl_frame_get_tag(cur_frame);

        if (tag == NULL) {

            /* tag not defined */
            cpl_msg_warning(cpl_func,
                            "Frame %d of %d has no tag", 1 + (int) i, (int) nframes);
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_NONE);
            /* Predefined macros, from CPL manual S 4.2.6:
             * - CPL_FRAME_GROUP_CALIB_ID "CALIB"
             * - CPL_FRAME_GROUP_PRODUCT_ID "PRODUCT"
             * - CPL_FRAME_GROUP_RAW_ID "RAW"
             * - CPL_FRAME_GROUP_NONE
             * */

        } else if (
                !strcmp(tag, QMOST_RAW_BIAS) ||
                !strcmp(tag, QMOST_RAW_DARK) ||
                !strcmp(tag, QMOST_RAW_DETECTOR_FLAT) ||
                !strcmp(tag, QMOST_RAW_DETECTOR_FLAT_LIN) ||
                !strcmp(tag, QMOST_RAW_DETECTOR_FLAT_MON) ||
                !strcmp(tag, QMOST_RAW_FIBRE_FLAT_DAY) ||
                !strcmp(tag, QMOST_RAW_FIBRE_FLAT_NIGHT) ||
                !strcmp(tag, QMOST_RAW_FIBRE_FLAT_SKY) ||
                !strcmp(tag, QMOST_RAW_FIBRE_WAVE_DAY) ||
                !strcmp(tag, QMOST_RAW_FIBRE_WAVE_NIGHT) ||
                !strcmp(tag, QMOST_RAW_FIBRE_WAVE_SIMUARC) ||
                !strcmp(tag, QMOST_RAW_FIBRE_WAVE_SIMUFPE) ||
                !strcmp(tag, QMOST_RAW_SCIENCE)) {

            /* RAW frames */
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_RAW);

        } else if (
                !strcmp(tag, QMOST_CALIB_MASTER_BPM) ||
                !strcmp(tag, QMOST_CALIB_REFERENCE_BIAS) ||
                !strcmp(tag, QMOST_CALIB_REFERENCE_DARK) ||
                !strcmp(tag, QMOST_CALIB_REFERENCE_DETECTOR_FLAT) ||
                !strcmp(tag, QMOST_CALIB_REFERENCE_FIBRE_TRACE) ||
                !strcmp(tag, QMOST_CALIB_REFERENCE_WAVE) ||
                !strcmp(tag, QMOST_CALIB_REFERENCE_FIBRE_FLAT) ||
                !strcmp(tag, QMOST_CALIB_SLIT_MASK) ||
                !strcmp(tag, QMOST_CALIB_MASTER_WAVE_MAP) ||
                !strcmp(tag, QMOST_CALIB_ARC_LINELIST) ||
                !strcmp(tag, QMOST_CALIB_SENSITIVITY) ||
                !strcmp(tag, QMOST_PRO_MASTER_BIAS) ||
                !strcmp(tag, QMOST_PRO_DIFFIMG_BIAS) ||
                !strcmp(tag, QMOST_PRO_MASTER_DARK) ||
                !strcmp(tag, QMOST_PRO_DIFFIMG_DARK) ||
                !strcmp(tag, QMOST_PRO_MASTER_DETECTOR_FLAT) ||
                !strcmp(tag, QMOST_PRO_DIFFIMG_DETECTOR_FLAT) ||
                !strcmp(tag, QMOST_PRO_UNFILTERED_DETECTOR_FLAT) ||
                !strcmp(tag, QMOST_PRO_LINEARITY) ||
                !strcmp(tag, QMOST_PRO_BPM) ||
                !strcmp(tag, QMOST_PRO_CRMASK) ||
                !strcmp(tag, QMOST_PRO_FIBRE_TRACE) ||
                !strcmp(tag, QMOST_PRO_FIBRE_MASK) ||
                !strcmp(tag, QMOST_PROC_FIBRE_FLAT) ||
                !strcmp(tag, QMOST_PRO_FPE_LINELIST) ||
                !strcmp(tag, QMOST_PRO_FPE_SPECTRUM) ||
                !strcmp(tag, QMOST_PROC_FIBRE_WAVE) ||
                !strcmp(tag, QMOST_PRO_MASTER_ARC) ||
                !strcmp(tag, QMOST_PRO_MASTER_WAVE) ||
                !strcmp(tag, QMOST_PRO_MASTER_WARC) ||
                !strcmp(tag, QMOST_PRO_OB_ARC) ||
                !strcmp(tag, QMOST_PRO_OB_WAVE) ||
                !strcmp(tag, QMOST_PRO_OB_WARC) ||
                !strcmp(tag, QMOST_PRO_SIMUARC_ARC) ||
                !strcmp(tag, QMOST_PRO_SIMUARC_WAVE) ||
                !strcmp(tag, QMOST_PRO_SIMUARC_WARC) ||
                !strcmp(tag, QMOST_PRO_MASTER_PSF) ||
                !strcmp(tag, QMOST_PRO_MASTER_FIBRE_FLAT) ||
                !strcmp(tag, QMOST_PRO_EXTRACTED_FIBRE_FLAT) ||
                !strcmp(tag, QMOST_PRO_OB_FIBRE_FLAT) ||
                !strcmp(tag, QMOST_PRO_SKY_FIBRE_FLAT) ||
                !strcmp(tag, QMOST_PRO_READGAIN)) {

            /* CALIB frames */
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_CALIB);

        } else {

            /* unknown tag frame */
            cpl_msg_warning(cpl_func, "Frame:%lld with tag:%s, unknown!", i, tag);
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_NONE);
        }
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Generate default filename for a product file.
 *
 * This routine generates the usual filename for a product file
 * following a consistent naming scheme.
 *
 * The files are named QMOST_procatg_spec.fits where "spec" is one of
 * the strings HRS, LRS-A or LRS-B respectively, so for example the
 * science output of LRS-A is named QMOST_SCIENCE_LRS-A.fits etc.
 *
 * @param   ispec      (Given)    One of the QMOST_SPEC_* values
 *                                specifying which spectrograph the
 *                                file is for.
 * @param   procatg    (Given)    The product category (PRO.CATG).
 *
 * @return  char * giving the resulting filename.  This should be
 *          freed by the caller.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

char *qmost_dfs_product_filename (
    const int ispec,
    const char *procatg)
{
    const char *spec = NULL;

    switch(ispec) {
    case QMOST_SPEC_HRS:
        spec = "HRS";
        break;
    case QMOST_SPEC_LRS_A:
        spec = "LRS-A";
        break;
    case QMOST_SPEC_LRS_B:
        spec = "LRS-B";
        break;
    default:
        spec = "UNKNOWN";
        break;
    }

    return cpl_sprintf("QMOST_%s_%s.fits", procatg, spec);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Set up output product frame with dummy primary HDU.
 *
 * @param   frameset   (Modified) The list of input frames for the
 *                                recipe.  The created product frame
 *                                will be added to the framset.
 * @param   parlist    (Given)    The input parameters to the recipe.
 * @param   recipe     (Given)    The recipe name.
 * @param   filename   (Given)    The name of the output file to
 *                                create.
 * @param   procatg    (Given)    The product category (PRO.CATG).
 * @param   protype    (Given)    The product type.
 * @param   applist    (Given)    Any other headers to append, or NULL
 *                                if none.
 *
 * @return  A pointer to the created product frame.  The product frame
 *          is owned by the frameset so should not be modified or
 *          freed by the caller.
 *
 * @par CPL error codes raised
 *   - CPL_ERROR_DATA_NOT_FOUND if the frameset is empty.
 *   - CPL_ERROR_NULL_INPUT if one of the required inputs is NULL.
 *   - CPL_ERROR_TYPE_MISMATCH if applist contains a property
 *     overriding one already populated by
 *     cpl_dfs_setup_product_header but the data type doesn't match.
 *
 * @par Output FITS Headers:
 *   - <b>EXPTIME</b>
 *   - <b>HIERARCH ESO DRS DID</b>
 *   - <b>HIERARCH ESO QC DID</b>
 *   - All standard headers populated by CPL routine
 *     cpl_dfs_setup_product_header().
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

const cpl_frame *qmost_dfs_setup_product (
    cpl_frameset *frameset,
    const cpl_parameterlist *parlist,
    const char *recipe,
    const char *filename,
    const char *procatg,
    const cpl_frame_type protype,
    const cpl_propertylist *applist)
{
    cpl_frame *frame = NULL;
    cpl_propertylist *hdr = NULL;

    int ifr, nfr, iuse;
    const cpl_frame *raw_frame = NULL;
    cpl_errorstate prestate;
    cpl_error_code code;
    cpl_propertylist *raw_hdr = NULL;

#undef TIDY
#define TIDY                                    \
    if(frame) {                                 \
        cpl_frame_delete(frame);                \
        frame = NULL;                           \
    }                                           \
    if(hdr) {                                   \
        cpl_propertylist_delete(hdr);           \
        hdr = NULL;                             \
    }

    /* Create product frame */
    frame = cpl_frame_new();

    if(cpl_frame_set_filename(frame, filename) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not set frame filename to %s",
                              filename);
        return NULL;
    }
    if(cpl_frame_set_tag(frame, procatg) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not set frame tag to %s",
                              procatg);
        return NULL;
    }
    if(cpl_frame_set_type(frame, protype) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not set frame type");
        return NULL;
    }
    if(cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not set frame group to product");
        return NULL;
    }
    if(cpl_frame_set_level(frame, CPL_FRAME_LEVEL_FINAL) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not set frame level to final");
        return NULL;
    }

    /* Create header */
    hdr = cpl_propertylist_new();

    cpl_propertylist_update_string(hdr,
                                   CPL_DFS_PRO_CATG,
                                   procatg);

    if(!strcmp(procatg, QMOST_PRO_SCIENCE)) {
        cpl_propertylist_update_bool(hdr,
                                     CPL_DFS_PRO_SCIENCE,
                                     1);
        cpl_propertylist_set_comment(hdr,
                                     CPL_DFS_PRO_SCIENCE,
                                     "Scientific product if T");
    }

    cpl_propertylist_update_string(hdr,
                                   "ESO DRS DID",
                                   "ESO-VLT-DIC.QMOST_DRS-0.2");
    cpl_propertylist_set_comment(hdr,
                                 "ESO DRS DID",
                                 "Data dictionary for DRS");

    cpl_propertylist_update_string(hdr,
                                   "ESO QC DID",
                                   "ESO-DFS-DIC.QMOST_QC-0.4");
    cpl_propertylist_set_comment(hdr,
                                 "ESO QC DID",
                                 "Data dictionary for QC");

    /* Add dataflow keywords */
    if(cpl_dfs_setup_product_header(hdr,
                                    frame,
                                    frameset,
                                    parlist,
                                    recipe,
                                    PACKAGE "/" PACKAGE_VERSION,
                                    PRO_DID,
                                    NULL) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not setup FITS header "
                              "product dataflow keywords");
        return NULL;
    }

    /* For REFERENCE frames, override PRO TYPE to STATIC since we are
     * making a new static calibration frame. */
    if(!strncmp(procatg, "REFERENCE_", strlen("REFERENCE_"))) {
        cpl_propertylist_update_string(hdr,
                                       "ESO PRO TYPE",
                                       "STATIC");
    }

    /* Append any other headers we need.  This might override
     * something that was already populated, so we can't use
     * cpl_propertylist_append, which doesn't check for duplicates.
     * Instead we use copy_property_regexp with a regular expression
     * that matches anything.  An error would be raised if there's a
     * type mismatch.  Fixing this is a lot of trouble so it's left as
     * a caveat for the caller to be careful about it. */
    if(applist != NULL) {
        if(cpl_propertylist_copy_property_regexp(hdr,
                                                 applist,
                                                 ".*",
                                                 0) != CPL_ERROR_NONE) {
            TIDY;
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "could not append FITS headers");
            return NULL;
        }
    }

    /* EXPTIME is stripped by cpl_dfs_setup_product_header().  Set it
     * here if we didn't set it explicitly already (currently this is
     * done by science_process to give the total exposure time when
     * stacking).  Other recipes that stack do averaging, so the
     * effective exposure time is preserved. */
    if(!cpl_propertylist_has(hdr, "EXPTIME")) {
        /* First raw */
        nfr = cpl_frameset_get_size(frameset);

        iuse = -1;

        for(ifr = 0; ifr < nfr; ifr++) {
            raw_frame = cpl_frameset_get_position_const(frameset, ifr);
            if(cpl_frame_get_group(raw_frame) == CPL_FRAME_GROUP_RAW) {
                iuse = ifr;
                break;
            }
        }

        if(iuse >= 0) {
            raw_frame = cpl_frameset_get_position_const(frameset, iuse);

            /* Copy EXPTIME, ignoring any errors */
            prestate = cpl_errorstate_get();

            raw_hdr = cpl_propertylist_load(cpl_frame_get_filename(raw_frame),
                                            0);
            if(raw_hdr == NULL) {
                cpl_errorstate_set(prestate);
            }
            else {
                code = cpl_propertylist_copy_property(hdr,
                                                      raw_hdr,
                                                      "EXPTIME");
                if(code != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                }

                cpl_propertylist_delete(raw_hdr);
                raw_hdr = NULL;
            }
        }
    }

    /* Create PHDU */
    if(cpl_propertylist_save(hdr,
                             filename,
                             CPL_IO_CREATE) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not create product FITS file: %s",
                              filename);
        return NULL;
    }

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Register frame in frameset */
    if(cpl_frameset_insert(frameset, frame) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "couldn't add product frame "
                              "to frameset");
        return NULL;
    }
    /* the frameset now owns it */

    return frame;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Set up output product frame with default name.
 *
 * This is a convenience function combining qmost_dfs_product_filename
 * and qmost_dfs_setup_product to assign the product filenames
 * automatically.  The reader is referred to the documentation of
 * these two routines for details and return types / values.
 *
 * @param   frameset   (Modified) The list of input frames for the
 *                                recipe.  The created product frame
 *                                will be added to the framset.
 * @param   parlist    (Given)    The input parameters to the recipe.
 * @param   recipe     (Given)    The recipe name.
 * @param   ispec      (Given)    One of the QMOST_SPEC_* values
 *                                specifying which spectrograph the
 *                                file is for.
 * @param   procatg    (Given)    The product category (PRO.CATG).
 * @param   protype    (Given)    The product type.
 * @param   applist    (Given)    Any other headers to append, or NULL
 *                                if none.
 *
 * @par Output FITS Headers:
 *   - <b>EXPTIME</b>
 *   - <b>HIERARCH ESO DRS DID</b>
 *   - <b>HIERARCH ESO QC DID</b>
 *   - All standard headers populated by CPL routine
 *     cpl_dfs_setup_product_header().
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

const cpl_frame *qmost_dfs_setup_product_default (
    cpl_frameset *frameset,
    const cpl_parameterlist *parlist,
    const char *recipe,
    const int ispec,
    const char *procatg,
    const cpl_frame_type protype,
    const cpl_propertylist *applist)
{
    char *filename = NULL;
    const cpl_frame *frame = NULL;

    filename = qmost_dfs_product_filename(ispec, procatg);
    if(filename == NULL) {
        return NULL;
    }

    frame = qmost_dfs_setup_product(frameset,
                                    parlist,
                                    recipe,
                                    filename,
                                    procatg,
                                    protype,
                                    applist);

    cpl_free(filename);
    filename = NULL;

    return frame;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Set up output product extension header.
 *
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   dummy      (Given)    True if dummy product, else false.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 *
 * @return  The new extension header.
 *
 * @par CPL error codes raised
 *   - CPL_ERROR_NULL_INPUT if one of the required inputs is NULL.
 *   - CPL_ERROR_TYPE_MISMATCH if qclist contains a property
 *     overriding one already populated from the input file, but the
 *     data type doesn't match.
 *
 * @par Output FITS Headers:
 *   - <b>EXTNAME</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_propertylist *qmost_dfs_setup_product_extension_header (
    cpl_propertylist *inhdr,
    const char *extname,
    int dummy,
    cpl_propertylist *qclist)
{
    cpl_propertylist *outhdr = NULL;

#undef TIDY
#define TIDY                                    \
    if(outhdr) {                                \
        cpl_propertylist_delete(outhdr);        \
        outhdr = NULL;                          \
    }

    /* Create header */
    outhdr = cpl_propertylist_new();

    /* Set EXTNAME */
    if(cpl_propertylist_update_string(outhdr,
                                      "EXTNAME",
                                      extname) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not set FITS header EXTNAME = %s",
                              extname);
        return NULL;
    }

    if(inhdr != NULL) {
        /* Copy in other headers */
        if(cpl_propertylist_copy_property_regexp(outhdr,
                                                 inhdr,
                                                 QMOST_REGEXP_HDUCOPY,
                                                 1) != CPL_ERROR_NONE) {
            TIDY;
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "could not copy input file headers");
            return NULL;
        }
    }

    /* Set dummy product flag */
    cpl_propertylist_update_bool(outhdr, "ESO DRS DUMMY", dummy);
    cpl_propertylist_set_comment(outhdr, "ESO DRS DUMMY",
                                 "True if extension is a dummy product");

    /* Copy in QC */
    if(qclist != NULL) {
        if(cpl_propertylist_copy_property_regexp(outhdr,
                                                 qclist,
                                                 ".*",
                                                 0) != CPL_ERROR_NONE) {
            TIDY;
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "could not copy QC headers");
            return NULL;
        }
    }

    return outhdr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save an image extension to an output frame.
 *
 * @param   frame      (Given)    The output frame to save to.
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 * @param   image      (Given)    The image to save, or NULL to create
 *                                a dummy HDU.
 * @param   type       (Given)    The data type to be used to store
 *                                the image in the file.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the filename isn't set in the
 *                                    frame.
 * @retval  CPL_ERROR_FILE_IO         If the file can't be written or
 *                                    isn't a valid FITS file.
 * @retval  CPL_ERROR_NULL_INPUT      If the frame was NULL.
 *
 * @par Output FITS Headers:
 *   - <b>EXTNAME</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_save_image_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_image *image,
    cpl_type type)
{
    cpl_propertylist *outhdr = NULL;
    const char *filename;

    cpl_ensure_code(frame, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(outhdr) {                                \
        cpl_propertylist_delete(outhdr);        \
        outhdr = NULL;                          \
    }

    outhdr = qmost_dfs_setup_product_extension_header(inhdr,
                                                      extname,
                                                      image == NULL,
                                                      qclist);
    if(outhdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not setup extension header %s",
                                     extname);
    }

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    if(cpl_image_save(image,
                      filename,
                      type,
                      outhdr,
                      CPL_IO_EXTEND) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save %s image extension to %s",
                                     extname,
                                     filename);
    }

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save image and variance to extensions in an output frame.
 *
 * The image is saved to the given extname and the variance array to
 * extname_var, with the required cross-reference keywords added to
 * the headers.
 *
 * @param   frame      (Given)    The output frame to save to.
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 * @param   image      (Given)    The image to save, or NULL to create
 *                                a dummy HDU.
 * @param   var        (Given)    The variance image to save, or NULL
 *                                to create a dummy HDU.
 * @param   type       (Given)    The data type to be used to store
 *                                the image in the file.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the filename isn't set in the
 *                                    frame.
 * @retval  CPL_ERROR_FILE_IO         If the file can't be written or
 *                                    isn't a valid FITS file.
 * @retval  CPL_ERROR_NULL_INPUT      If the frame was NULL.
 *
 * @par Output FITS Headers:
 *   - <b>ERRDATA</b>
 *   - <b>EXTNAME</b>
 *   - <b>HDUCLAS1</b>
 *   - <b>HDUCLAS2</b>
 *   - <b>HDUCLAS3</b>
 *   - <b>HDUCLASS</b>
 *   - <b>HDUDOC</b>
 *   - <b>HDUVERS</b>
 *   - <b>SCIDATA</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_save_image_and_var (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_image *image,
    cpl_image *var,
    cpl_type type)
{
    char *var_extname = NULL;
    cpl_propertylist *qctmp = NULL;

    cpl_ensure_code(frame, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(var_extname != NULL) {                   \
        cpl_free(var_extname);                  \
        var_extname = NULL;                     \
    }                                           \
    if(qctmp != NULL) {                         \
        cpl_propertylist_delete(qctmp);         \
        qctmp = NULL;                           \
    }

    /* EXTNAME for the variance array */
    var_extname = cpl_sprintf("%s_var", extname);
    if(var_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format EXTNAME string "
                                     "for variance extension");
    }

    /* Adjust header for image to add the required classification and
       cross reference keywords per VLT-SPE-ESO-19500-5667_DataFormat. */
    if(qclist != NULL) {
        qctmp = cpl_propertylist_duplicate(qclist);
    }
    else {
        qctmp = cpl_propertylist_new();
    }
    if(qctmp == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not initialise image "
                                     "extension header");
    }

    cpl_propertylist_update_string(qctmp, "HDUCLASS", "ESO");
    cpl_propertylist_update_string(qctmp, "HDUDOC", "DICD");
    cpl_propertylist_update_string(qctmp, "HDUVERS", "DICD version 6");
    cpl_propertylist_update_string(qctmp, "HDUCLAS1", "IMAGE");
    cpl_propertylist_update_string(qctmp, "HDUCLAS2", "DATA");
    cpl_propertylist_update_string(qctmp, "ERRDATA", var_extname);

    if(qmost_dfs_save_image_extension(frame,
                                      inhdr,
                                      extname,
                                      qctmp,
                                      image,
                                      type) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save image extension");
    }

    cpl_propertylist_delete(qctmp);
    qctmp = NULL;

    /* Now the variance array */
    if(qclist != NULL) {
        qctmp = cpl_propertylist_duplicate(qclist);
    }
    else {
        qctmp = cpl_propertylist_new();
    }
    if(qctmp == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not initialise variance "
                                     "extension header");
    }

    cpl_propertylist_update_string(qctmp, "HDUCLASS", "ESO");
    cpl_propertylist_update_string(qctmp, "HDUDOC", "DICD");
    cpl_propertylist_update_string(qctmp, "HDUVERS", "DICD version 6");
    cpl_propertylist_update_string(qctmp, "HDUCLAS1", "IMAGE");
    cpl_propertylist_update_string(qctmp, "HDUCLAS2", "ERROR");
    cpl_propertylist_update_string(qctmp, "HDUCLAS3", "MSE");
    cpl_propertylist_update_string(qctmp, "SCIDATA", extname);

    if(qmost_dfs_save_image_extension(frame,
                                      inhdr,
                                      var_extname,
                                      qctmp,
                                      var,
                                      type) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save variance extension");
    }

    cpl_propertylist_delete(qctmp);
    qctmp = NULL;

    cpl_free(var_extname);
    var_extname = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save image and inverse variance to extensions in an output
 *          frame.
 *
 * The image is saved to the given extname.  An inverse variance
 * array is computed from the given variance array, converting zero
 * variance in the variance array to zero inverse variance (used as a
 * flag).  The inverse variance array is saved to an extension named
 * by replacing _DATA with _IVAR in the given extname.
 *
 * This is a programmatic means of achieving the peculiar extension
 * naming scheme used in 4MOST (which appears to not be designed with
 * the need to do this in mind).  If the extname doesn't contain _DATA
 * then an error is raised.
 *
 * @param   frame      (Given)    The output frame to save to.
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 * @param   image      (Given)    The image to save, or NULL to create
 *                                a dummy HDU.
 * @param   var        (Given)    The variance image to save, or NULL
 *                                to create a dummy HDU.  The data
 *                                type must be CPL_TYPE_FLOAT.
 * @param   type       (Given)    The data type to be used to store
 *                                the image in the file.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the filename isn't set in the
 *                                    frame.
 * @retval  CPL_ERROR_FILE_IO         If the file can't be written or
 *                                    isn't a valid FITS file.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the input EXTNAME doesn't
 *                                    contain the string "_DATA".
 * @retval  CPL_ERROR_NULL_INPUT      If a required input was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the image data types aren't
 *                                    supported.
 *
 * @par Output FITS Headers:
 *   - <b>ERRDATA</b>
 *   - <b>EXTNAME</b>
 *   - <b>HDUCLAS1</b>
 *   - <b>HDUCLAS2</b>
 *   - <b>HDUCLAS3</b>
 *   - <b>HDUCLASS</b>
 *   - <b>HDUDOC</b>
 *   - <b>HDUVERS</b>
 *   - <b>SCIDATA</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_save_image_and_ivar (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_image *image,
    cpl_image *var,
    cpl_type type)
{
    char *ivar_extname = NULL;
    cpl_propertylist *qctmp = NULL;
    cpl_image *ivar = NULL;

    char *p;

    double *ddata = NULL;
    float *fdata = NULL;
    int ipix, npix, nx, ny;

    cpl_ensure_code(frame, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(extname, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(ivar_extname != NULL) {                  \
        cpl_free(ivar_extname);                 \
        ivar_extname = NULL;                    \
    }                                           \
    if(qctmp != NULL) {                         \
        cpl_propertylist_delete(qctmp);         \
        qctmp = NULL;                           \
    }                                           \
    if(ivar != NULL) {                          \
        cpl_image_delete(ivar);                 \
        ivar = NULL;                            \
    }

    /* EXTNAME for the inverse variance array */
    ivar_extname = cpl_strdup(extname);

    p = strstr(ivar_extname, "_DATA");
    if(p != NULL) {
        p[1] = 'I';
        p[2] = 'V';
        p[3] = 'A';
        p[4] = 'R';
    }
    else {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "EXTNAME string doesn't contain "
                                     "_DATA");
    }

    if(var != NULL) {
        /* Compute the inverse variance array */
        nx = cpl_image_get_size_x(var);
        ny = cpl_image_get_size_y(var);
        npix = nx * ny;
        
        ivar = cpl_image_duplicate(var);
        
        switch(cpl_image_get_type(ivar)) {
        case CPL_TYPE_DOUBLE:
            ddata = cpl_image_get_data_double(ivar);
            if(ddata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get double pointer "
                                             "to inverse variance image");
            }
            
            for(ipix = 0; ipix < npix; ipix++) {
                if(ddata[ipix] != 0) {
                    ddata[ipix] = 1.0 / ddata[ipix];
                }
            }
            
            break;
        case CPL_TYPE_FLOAT:
            fdata = cpl_image_get_data_float(ivar);
            if(fdata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer "
                                             "to inverse variance image");
            }
            
            for(ipix = 0; ipix < npix; ipix++) {
                if(fdata[ipix] != 0) {
                    fdata[ipix] = 1.0 / fdata[ipix];
                }
            }
            
            break;
        default:
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "only double or float variance "
                                         "images are supported");
        }
    }

    /* Adjust header for image to add the required classification and
       cross reference keywords per VLT-SPE-ESO-19500-5667_DataFormat. */
    if(qclist != NULL) {
        qctmp = cpl_propertylist_duplicate(qclist);
    }
    else {
        qctmp = cpl_propertylist_new();
    }
    if(qctmp == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not initialise image "
                                     "extension header");
    }

    cpl_propertylist_update_string(qctmp, "HDUCLASS", "ESO");
    cpl_propertylist_update_string(qctmp, "HDUDOC", "DICD");
    cpl_propertylist_update_string(qctmp, "HDUVERS", "DICD version 6");
    cpl_propertylist_update_string(qctmp, "HDUCLAS1", "IMAGE");
    cpl_propertylist_update_string(qctmp, "HDUCLAS2", "DATA");
    cpl_propertylist_update_string(qctmp, "ERRDATA", ivar_extname);

    if(qmost_dfs_save_image_extension(frame,
                                      inhdr,
                                      extname,
                                      qctmp,
                                      image,
                                      type) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save image extension");
    }

    cpl_propertylist_delete(qctmp);
    qctmp = NULL;

    /* Now the inverse variance array */
    if(qclist != NULL) {
        qctmp = cpl_propertylist_duplicate(qclist);
    }
    else {
        qctmp = cpl_propertylist_new();
    }
    if(qctmp == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not initialise variance "
                                     "extension header");
    }

    cpl_propertylist_update_string(qctmp, "HDUCLASS", "ESO");
    cpl_propertylist_update_string(qctmp, "HDUDOC", "DICD");
    cpl_propertylist_update_string(qctmp, "HDUVERS", "DICD version 6");
    cpl_propertylist_update_string(qctmp, "HDUCLAS1", "IMAGE");
    cpl_propertylist_update_string(qctmp, "HDUCLAS2", "ERROR");
    cpl_propertylist_update_string(qctmp, "HDUCLAS3", "INVMSE");
    cpl_propertylist_update_string(qctmp, "SCIDATA", extname);

    if(qmost_dfs_save_image_extension(frame,
                                      inhdr,
                                      ivar_extname,
                                      qctmp,
                                      ivar,
                                      type) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save inverse variance "
                                     "extension");
    }

    if(ivar != NULL) {
        cpl_image_delete(ivar);
        ivar = NULL;
    }

    cpl_propertylist_delete(qctmp);
    qctmp = NULL;

    cpl_free(ivar_extname);
    ivar_extname = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save an imagelist as a 3D IMAGE extension.
 *
 * @param   frame      (Given)    The output frame to save to.
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 * @param   imagelist  (Given)    The imagelist to save, or NULL to
 *                                create a dummy HDU.
 * @param   type       (Given)    The data type to be used to store
 *                                the image in the file.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the filename isn't set in the
 *                                    frame.
 * @retval  CPL_ERROR_FILE_IO         If the file can't be written or
 *                                    isn't a valid FITS file.
 * @retval  CPL_ERROR_NULL_INPUT      If a required input was NULL.
 *
 * @par Output FITS Headers:
 *   - <b>EXTNAME</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_save_imagelist_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_imagelist *imagelist,
    cpl_type type)
{
    cpl_propertylist *outhdr = NULL;
    const char *filename;

    cpl_ensure_code(frame, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(outhdr) {                                \
        cpl_propertylist_delete(outhdr);        \
        outhdr = NULL;                          \
    }

    outhdr = qmost_dfs_setup_product_extension_header(inhdr,
                                                      extname,
                                                      imagelist == NULL,
                                                      qclist);
    if(outhdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not setup extension header %s",
                                     extname);
    }

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    if(imagelist != NULL) {
        if(cpl_imagelist_save(imagelist,
                              filename,
                              type,
                              outhdr,
                              CPL_IO_EXTEND) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not save %s 3D image "
                                         "extension to %s",
                                         extname,
                                         filename);
        }
    }
    else {
        if(cpl_propertylist_save(outhdr,
                                 filename,
                                 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not save %s dummy "
                                         "extension to %s",
                                         extname,
                                         filename);
        }
    }

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save a mask to an extension in an output frame.
 *
 * @param   frame      (Given)    The output frame to save to.
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 * @param   mask       (Given)    The mask to save, or NULL to create
 *                                a dummy HDU.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the filename isn't set in the
 *                                    frame.
 * @retval  CPL_ERROR_FILE_IO         If the file can't be written or
 *                                    isn't a valid FITS file.
 * @retval  CPL_ERROR_NULL_INPUT      If a required input was NULL.
 *
 * @par Output FITS Headers:
 *   - <b>EXTNAME</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_save_mask_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_mask *mask)
{
    cpl_propertylist *outhdr = NULL;
    const char *filename;

    cpl_ensure_code(frame, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(outhdr) {                                \
        cpl_propertylist_delete(outhdr);        \
        outhdr = NULL;                          \
    }

    outhdr = qmost_dfs_setup_product_extension_header(inhdr,
                                                      extname,
                                                      mask == NULL,
                                                      qclist);
    if(outhdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not setup extension header %s",
                                     extname);
    }

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    if(mask != NULL) {
        if(cpl_mask_save(mask,
                         filename,
                         outhdr,
                         CPL_IO_EXTEND) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not save %s BPM "
                                         "extension to %s",
                                         extname,
                                         filename);
        }
    }
    else {
        if(cpl_propertylist_save(outhdr,
                                 filename,
                                 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not save %s dummy "
                                         "extension to %s",
                                         extname,
                                         filename);
        }
    }

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save a table extension to an output frame.
 *
 * @param   frame      (Given)    The output frame to save to.
 * @param   inhdr      (Given)    The FITS header from the input
 *                                file.
 * @param   extname    (Given)    Extension name.
 * @param   qclist     (Given)    QC parameters to append to the
 *                                header.
 * @param   table      (Given)    The table to save, or NULL to create
 *                                a dummy HDU.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the filename isn't set in the
 *                                    frame.
 * @retval  CPL_ERROR_FILE_NOT_CREATED  If the file can't be written
 *                                      or isn't a valid FITS file.
 * @retval  CPL_ERROR_NULL_INPUT      If a required input was NULL.
 *
 * @par Output FITS Headers:
 *   - <b>EXTNAME</b>
 *
 * @par Output DRS Headers:
 *   - <b>DUMMY</b>: Boolean flag set to true if the extension
 *     contains a dummy product.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_save_table_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_table *table)
{
    cpl_propertylist *outhdr = NULL;
    const char *filename;
    cpl_table *tmp_table = NULL;
    cpl_table *tosave;

    cpl_ensure_code(frame, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(outhdr) {                                \
        cpl_propertylist_delete(outhdr);        \
        outhdr = NULL;                          \
    }                                           \
    if(tmp_table) {                             \
        cpl_table_delete(tmp_table);            \
        tmp_table = NULL;                       \
    }

    outhdr = qmost_dfs_setup_product_extension_header(inhdr,
                                                      extname,
                                                      table == NULL,
                                                      qclist);
    if(outhdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not setup extension header %s",
                                     extname);
    }

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    /* When saving a dummy, create an empty table and save that so we
     * get a BINTABLE extension.  The normal means of saving an empty
     * FITS extension by saving the propertylist would otherwise
     * produce an IMAGE extension, which might break something
     * downstream expecting consistent HDU types in the file. */
    if(table != NULL) {
        tosave = table;
    }
    else {
        tmp_table = cpl_table_new(0);
        tosave = tmp_table;
    }

    if(cpl_table_save(tosave,
                      NULL,
                      outhdr,
                      filename,
                      CPL_IO_EXTEND) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save %s table "
                                     "extension to %s",
                                     extname,
                                     filename);
    }

    if(tmp_table != NULL) {
        cpl_table_delete(tmp_table);
        tmp_table = NULL;
    }

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    return CPL_ERROR_NONE;
}

/**@}*/
