/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_extract_tram.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_extract_tram  qmost_extract_tram
 * 
 * Tramline extraction of fibre spectra.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_extract_tram.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void qmost_extract_spec (
    qmost_traceinfo tr,
    long *naxis,
    float *indata,
    float *invar,
    unsigned char *inbpmdata,
    int iwidth,
    int miny,
    int maxy,
    int specbin,
    int spatbin,
    float **spec,
    float **varmap);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do tramline extraction of spectra from a 2D image.
 *
 * This routine performs a standard unweighted boxcar spectral
 * extraction, using the given spectral trace to define the centre of
 * an extraction aperture of width specified by the argument iwidth.
 * The extracted spectrum at each spectral pixel is simply the sum of
 * the spatial pixels within the aperture, weighting partial pixels by
 * the fraction of the pixel that falls within the aperture.
 *
 * @param   in_image     (Given)    The input image, including bad
 *                                  pixel mask.  The data type must be
 *                                  CPL_TYPE_FLOAT.
 * @param   in_var       (Given)    The input variance image.  The data
 *                                  type must be CPL_TYPE_FLOAT.
 * @param   in_hdr       (Given)    The FITS header of the input
 *                                  image.
 * @param   trace_tbl    (Given)    The relevant trace table for the
 *                                  2d spectra.
 * @param   trace_hdr    (Given)    The FITS header of the trace
 *                                  table.
 * @param   iwidth       (Given)    The number of pixels over which
 *                                  the summation will be done at each
 *                                  spectral position.
 * @param   out_spec_img (Returned) The output image containing the
 *                                  extracted spectra.  NULL is
 *                                  returned if there are no fibres in
 *                                  the trace file.
 * @param   out_spec_var (Returned) The corresponding variance array
 *                                  for the extracted spectra.  NULL
 *                                  is returned if there are no fibres
 *                                  in the trace file.
 * @param   out_spec_hdr (Modified) A caller allocated property list
 *                                  to receive output DRS FITS
 *                                  headers.  Some of these are needed
 *                                  by subsequent processing stages if
 *                                  binning was used.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If one of the required input
 *                                    FITS header keywords was not
 *                                    found.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the trace file doesn't make
 *                                    sense or iwidth <= 0.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required inputs or
 *                                    outputs is NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input spectrum data type
 *                                    was not float, or if one of the
 *                                    required input FITS header
 *                                    keyword values had an incorrect
 *                                    data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *
 * @par Output DRS Headers:
 *   - <b>MAXYFN</b>: The last physical detector spectral pixel in the
 *     extracted spectra.
 *   - <b>MINYST</b>: The first physical detector spectral pixel in
 *     the extracted spectra.
 *   - <b>WVCRV</b>: The physical (unbinned) spectral pixel coordinate
 *     of the middle of the first (binned) pixel in the extracted
 *     spectrum.  Needed for wavelength calibration of binned data.
 *   - <b>WVNOFF</b>: Physical (unbinned) spectral pixel offset of
 *     (binned) trace start position.  Needed for wavelength
 *     calibration of binned data.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_extract_tram (
    cpl_image *in_image,
    cpl_image *in_var,
    cpl_propertylist *in_hdr,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    int iwidth,
    cpl_image **out_spec_img,
    cpl_image **out_spec_var,
    cpl_propertylist *out_spec_hdr)
{
    int itrace,i,specbin,spatbin,isbinned;
    long naxis[2],nps,miny,maxy;

    cpl_mask *in_bpm;

    float tcrv;
    int noff;

    float *indata, *invar;
    cpl_binary *inbpmdata;
    float *out_spec_buf, *out_var_buf;
    cpl_binary *out_spec_bpm_buf, *out_var_bpm_buf;
    float *this_spec, *this_var;
    cpl_binary *this_spec_bpm, *this_var_bpm;

    /* Separate these out to make it easier to do the garbage collection */
    int ntrace = 0;
    qmost_traceinfo *tr = NULL;
    float *spec = NULL;
    float *wtmap = NULL;

    /* Check required inputs and outputs */
    cpl_ensure_code(in_image, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_var, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(iwidth > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(out_spec_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_var, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_hdr, CPL_ERROR_NULL_INPUT);

    /* Need to set these to NULL prior to handling any errors */
    *out_spec_img = NULL;
    *out_spec_var = NULL;

#undef TIDY
#define TIDY                                    \
    if(tr) {                                    \
        qmost_trclose(ntrace, &tr);             \
        tr = NULL;                              \
    }                                           \
    if(*out_spec_img) {                         \
        cpl_image_delete(*out_spec_img);        \
        *out_spec_img = NULL;                   \
    }                                           \
    if(*out_spec_var) {                         \
        cpl_image_delete(*out_spec_var);        \
        *out_spec_var = NULL;                   \
    }                                           \
    if(spec) {                                  \
        cpl_free(spec);                         \
        spec = NULL;                            \
    }                                           \
    if(wtmap) {                                 \
        cpl_free(wtmap);                        \
        wtmap = NULL;                           \
    }

    /* Dissect the input */
    in_bpm = cpl_image_get_bpm(in_image);

    /* Get sizes */
    naxis[0] = cpl_image_get_size_x(in_image);
    naxis[1] = cpl_image_get_size_y(in_image);

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    /* Get pointers to image data */
    indata = cpl_image_get_data_float(in_image);
    if(indata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    inbpmdata = cpl_mask_get_data(in_bpm);
    if(inbpmdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "input BPM");
    }

    invar = cpl_image_get_data_float(in_var);
    if(invar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input variance image");
    }

    /* Read trace table */
    if(qmost_tropen(trace_tbl, trace_hdr, &ntrace, &tr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to load trace table");
    }

    /* If there's nothing to do, get out of here now */
    if(ntrace < 1) {
        TIDY;
        return CPL_ERROR_NONE;
    }

    /* Get size of output spectra */
    miny = tr[0].minyst / ((float) specbin);
    maxy = tr[0].maxyfn / ((float) specbin);
    nps = maxy - miny + 1;

    /* Offset due to binning */
    if(specbin > 1) {
        noff = (tr[0].minyst - 1) % specbin;
        tcrv = 0.5 * specbin + 0.5 - noff;
    }
    else {
        noff = 0;
        tcrv = 1.0;
    }

    /* Create output */
    *out_spec_img = cpl_image_new(nps, ntrace, CPL_TYPE_FLOAT);
    *out_spec_var = cpl_image_new(nps, ntrace, CPL_TYPE_FLOAT);
    if(*out_spec_img == NULL || *out_spec_var == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output spectrum "
                                     "array of size (%ld, %d)",
                                     nps, ntrace);
    }

    out_spec_buf = cpl_image_get_data_float(*out_spec_img);
    out_spec_bpm_buf = cpl_mask_get_data(cpl_image_get_bpm(*out_spec_img));
    out_var_buf = cpl_image_get_data_float(*out_spec_var);
    out_var_bpm_buf = cpl_mask_get_data(cpl_image_get_bpm(*out_spec_var));
    if(out_spec_buf == NULL || out_spec_bpm_buf == NULL ||
       out_var_buf == NULL || out_var_bpm_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get pointer to "
                                     "output spectrum array");
    }

    /* Loop for all spectra */
    for(itrace = 0; itrace < ntrace; itrace++) {
        this_spec = out_spec_buf + itrace * nps;
        this_spec_bpm = out_spec_bpm_buf + itrace * nps;
        this_var = out_var_buf + itrace * nps;
        this_var_bpm = out_var_bpm_buf + itrace * nps;

	/* If this isn't a live fibre, set everything to zero and
           flag pixels so they are excluded from statistics. */
	if(!tr[itrace].live) {
            for(i = 0; i < nps; i++) {
                this_spec[i] = 0;
                this_spec_bpm[i] = 1;
                this_var[i] = 0;
                this_var_bpm[i] = 1;
            }
	    continue;
	}

	/* Do the extraction of the current spectrum */
	qmost_extract_spec(tr[itrace],naxis,indata,invar,inbpmdata,
                           iwidth,miny,maxy,specbin,spatbin,&spec,
                           &wtmap);

	/* Write the spectrum to the buffer */
	for (i = 0; i < nps; i++) {
	    this_spec[i] = spec[i];
	    this_var[i] = wtmap[i];
            if(wtmap[i] <= 0) {
                /* If there was no measurement, flag bad so pixel is
                   excluded from statistics. */
                this_spec_bpm[i] = 1;
                this_var_bpm[i] = 1;
            }
	}

        cpl_free(spec);
        spec = NULL;

        cpl_free(wtmap);
        wtmap = NULL;
    }

    /* Write FITS headers */
    cpl_propertylist_update_int(out_spec_hdr,
                                "ESO DRS MINYST",
                                tr[0].minyst);
    cpl_propertylist_set_comment(out_spec_hdr,
                                 "ESO DRS MINYST",
                                 "[pix] Minimum yst");
    cpl_propertylist_update_int(out_spec_hdr,
                                "ESO DRS MAXYFN",
                                tr[0].maxyfn);
    cpl_propertylist_set_comment(out_spec_hdr,
                                 "ESO DRS MAXYFN",
                                 "[pix] Maximum yfn");

    if(specbin > 1) {
        cpl_propertylist_update_float(out_spec_hdr,
                                      "ESO DRS WVCRV",
                                      tcrv);
        cpl_propertylist_set_comment(out_spec_hdr,
                                     "ESO DRS WVCRV",
                                     "[pix] Extraction zeropoint offset");

        cpl_propertylist_update_int(out_spec_hdr,
                                    "ESO DRS WVNOFF",
                                    noff);
        cpl_propertylist_set_comment(out_spec_hdr,
                                     "ESO DRS WVNOFF",
                                     "[pix] Extraction blocking offset");
    }

    qmost_trclose(ntrace, &tr);
    tr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Extract a spectrum along a given trace.
 *
 * Extract a spectrum by summing the pixels between two tramlines
 * running along the position given by the trace.
 *
 * @param   tr         (Given)    The trace information for a given
 *                                fibre.
 * @param   naxis      (Given)    The lengths of the axes of the input
 *                                2d image arrays.
 * @param   indata     (Given)    The input 2d spectral array.
 * @param   invar      (Given)    The input 2d spectral variance
 *                                array.
 * @param   inbpmdata  (Given)    The input 2d bad pixel mask.
 * @param   iwidth     (Given)    The number of pixels between the
 *                                tramlines to integrate.
 * @param   miny       (Given)    The minimum value of the starting
 *                                point for each spectrum.
 * @param   maxy       (Given)    The maximum value of the ending
 *                                point for each spectrum.
 * @param   specbin    (Given)    The binning along the spectral
 *                                axis.
 * @param   spatbin    (Given)    The binning along the spatial axis.
 * @param   spec       (Returned) The output array of spectra. Each
 *                                spectrum is a row.
 * @param   varmap     (Returned) The output array of spectral
 *                                variances. 
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void qmost_extract_spec (
    qmost_traceinfo tr,
    long *naxis,
    float *indata,
    float *invar,
    unsigned char *inbpmdata,
    int iwidth,
    int miny,
    int maxy,
    int specbin,
    int spatbin,
    float **spec,
    float **varmap)
{
    int nps,j,ix1,ix2,i,ind,jp;
    unsigned char *bpm;
    float xmid,x1,x2,sum1,sum2,data,var,wt,w2;
    double yy;

    /* Set up a few variables */

    w2 = 0.5*iwidth;

    /* Get space for the output spectrum and for a bpm array */
    nps = maxy - miny + 1;

    *spec = cpl_calloc(nps,sizeof(float));
    *varmap = cpl_calloc(nps,sizeof(float));

    /* Loop for each point in the wavelength direction */

    jp = tr.yst/(float)specbin - miny;
    for (j = 1; j <= naxis[1]; j++) {
        yy = 0.5*((double)(specbin*(2*j - 1) + 1.0));
        if (yy < tr.yst || yy > tr.yfn) {
            continue;
        } else {
            jp++;
            xmid = (float)qmost_tracexpos(tr,yy);

            /* Work out extraction spatial limits at this wavelength */

            x1 = (xmid - w2 - 0.5)/(float)spatbin + 0.5;
            x2 = (xmid + w2 - 0.5)/(float)spatbin + 0.5;
            ix1 = qmost_min(naxis[0],qmost_max(1,(int)(x1 + 0.5)));
            ix2 = qmost_min(naxis[0],(int)(x2 + 0.5));

            /* Initialise the bpm at this point */

            bpm = inbpmdata + (j-1)*naxis[0];

            /* First sum up across this profile */

            sum1 = 0.0;
            sum2 = 0.0;
            for (i = ix1; i <= ix2; i++) {
                ind = (j-1)*naxis[0] + i - 1;
                if (bpm[i-1])
                    continue;
                data = indata[ind];
                var = invar[ind];

                /* Sort out the weights */

                if (i == ix1) 
                    wt = (float)i + 0.5 - x1;
                else if (i == ix2)
                    wt = x2 - (float)i + 0.5;
                else
                    wt = 1.0;
                sum1 += wt*data;
                sum2 += wt*var;
            }

            /* Normalise the result */

            if (sum2 != 0) {
                (*spec)[jp-1] = sum1;
                (*varmap)[jp-1] = sum2;      /* changed to variance */
            } else {
                (*spec)[jp-1] = 0.0;
                (*varmap)[jp-1] = 0.0;
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Populate QC using extracted spectrum.
 *
 * @param   spec        (Given)    The extracted 2D spectra as an
 *                                 image.  The data type must be
 *                                 CPL_TYPE_FLOAT.
 * @param   v_spec      (Given)    The corresponding 2D variance
 *                                 spectra.  The data type must be
 *                                 CPL_TYPE_FLOAT.
 * @param   qclist      (Modified) The output FITS header to receive
 *                                 the QC parameters.
 * @param   fibinfo_tbl (Modified) Output FIBINFO table to receive
 *                                 per-fibre QC, or NULL for none.
 * @param   arm         (Given)    One of the QMOST_ARM_* constants
 *                                 specifying which arm we're
 *                                 processing.  Only used if
 *                                 fibinfo_tbl != NULL and can be
 *                                 given as 0 otherwise.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If an output FIBINFO table is
 *                                    given but the output columns are
 *                                    not found.  These must be
 *                                    created first by calling
 *                                    qmost_fibtab_newcols(). 
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the parameter arm is invalid.
 * @retval  CPL_ERROR_INVALID_TYPE    If one of the output FIBINFO
 *                                    table columns was not numerical.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required inputs or
 *                                    outputs is NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input spectrum data type
 *                                    was not float.
 *
 * @par Output QC Parameters:
 *   - <b>EXT FLUX MAX</b> (ADU): The maximum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MAXSPC</b>: The fibre with the maximum extracted
 *     flux.
 *   - <b>EXT FLUX MED</b> (ADU): The median of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MIN</b> (ADU): The minimum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MINSPC</b>: The fibre with the minimum extracted
 *     flux.
 *   - <b>EXT FLUX RMS</b> (ADU): The robustly-estimated RMS of
 *     average extracted flux in each fibre.
 *   - <b>EXT SN MAX</b>: The maximum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MAXSPC</b>: The fibre with the maximum signal to
 *     noise ratio.
 *   - <b>EXT SN MED</b>: The median of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MIN</b>: The minimum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MINSPC</b>: The fibre with the minimum signal to
 *     noise ratio.
 *   - <b>EXT SN RMS</b>: The robustly-estimated RMS of the average
 *     signal to noise ratio in each fibre.
 *
 * @par Output FIBINFO Table Columns:
 *   - <b>MEDFLUX_a</b> (ADU): The median flux (ADU per pixel) in arm
 *     "a" (R, G or B, depending on the value of parameter arm).
 *   - <b>SNR_a</b>: The average signal to noise ratio (per pixel) in
 *     arm "a" (R, G or B, depending on the value of parameter arm).
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_extract_qc (
    cpl_image *spec,
    cpl_image *v_spec,
    cpl_propertylist *qclist,
    cpl_table *fibinfo_tbl,
    int arm)
{
    float *trace_medflux = NULL;
    float *trace_medsnr = NULL;
    float *snrbuf = NULL;
    unsigned char *snrbpm = NULL;

    const char *arm_extname;
    char arm_ltr;
    char *colname_snr = NULL;
    char *colname_meanflux = NULL;

    float *spec_buf, *this_spec;
    float *var_buf, *this_var;
    cpl_binary *bpm_buf, *this_bpm;

    float trace_medflux_min, trace_medflux_max;
    int trace_medflux_minspc, trace_medflux_maxspc, ntrace_medflux;
    float trace_medsnr_min, trace_medsnr_max;
    int trace_medsnr_minspc, trace_medsnr_maxspc, ntrace_medsnr;

    int itrace, ntrace, ips, nps;
    float medflux, medsnr, medmed, rmsmed;
    float meanflux, sigflux;

    cpl_errorstate prestate;
    cpl_error_code code;

    cpl_ensure_code(qclist, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(trace_medflux) {                         \
        cpl_free(trace_medflux);                \
        trace_medflux = NULL;                   \
    }                                           \
    if(trace_medsnr) {                          \
        cpl_free(trace_medsnr);                 \
        trace_medsnr = NULL;                    \
    }                                           \
    if(snrbuf) {                                \
        cpl_free(snrbuf);                       \
        snrbuf = NULL;                          \
    }                                           \
    if(snrbpm) {                                \
        cpl_free(snrbpm);                       \
        snrbpm = NULL;                          \
    }                                           \
    if(colname_snr) {                           \
        cpl_free(colname_snr);                  \
        colname_snr = NULL;                     \
    }                                           \
    if(colname_meanflux) {                      \
        cpl_free(colname_meanflux);             \
        colname_meanflux = NULL;                \
    }

    /* Check for NULL spectrum input, meaning there were no fibres in
     * the trace file.  This is not an error.  In such cases the
     * routine is a no-op so we can return now. */
    if(spec == NULL) {
        return CPL_ERROR_NONE;
    }

    /* Check variance was supplied if spectra were */
    cpl_ensure_code(v_spec, CPL_ERROR_NULL_INPUT);

    /* Get size */
    nps = cpl_image_get_size_x(spec);
    ntrace = cpl_image_get_size_y(spec);

    /* Allocate workspace */
    trace_medflux = cpl_malloc(ntrace * sizeof(float));
    trace_medsnr = cpl_malloc(ntrace * sizeof(float));
    snrbuf = cpl_malloc(nps * sizeof(float));
    snrbpm = cpl_malloc(nps * sizeof(unsigned char));

    /* Get inputs */
    spec_buf = cpl_image_get_data_float(spec);
    if(spec_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "2D spectrum");
    }

    var_buf = cpl_image_get_data_float(v_spec);
    if(var_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "2D spectrum variance");
    }

    bpm_buf = cpl_mask_get_data(cpl_image_get_bpm(spec));
    if(bpm_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "2D spectrum BPM");
    }

    /* Set up FIBINFO columns if given */
    if(fibinfo_tbl != NULL) {
        /* Get the appropriate letter for the arm */
        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not determine EXTNAME "
                                         "for arm %d", arm);
        }
    
        arm_ltr = arm_extname[0];

        colname_snr = cpl_sprintf("SNR_%c", arm_ltr);
        if(colname_snr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format column "
                                         "names for arm %d", arm);
        }

        colname_meanflux = cpl_sprintf("MEANFLUX_%c", arm_ltr);
        if(colname_meanflux == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format column "
                                         "names for arm %d", arm);
        }
    }

    /* Loop for all spectra */
    trace_medflux_min = 0;
    trace_medflux_minspc = -1;
    trace_medflux_max = 0;
    trace_medflux_maxspc = -1;
    ntrace_medflux = 0;

    trace_medsnr_min = 0;
    trace_medsnr_minspc = -1;
    trace_medsnr_max = 0;
    trace_medsnr_maxspc = -1;
    ntrace_medsnr = 0;

    for(itrace = 0; itrace < ntrace; itrace++) {
        this_spec = spec_buf + itrace * nps;
        this_var = var_buf + itrace * nps;
        this_bpm = bpm_buf + itrace * nps;

        /* Compute median flux */
        prestate = cpl_errorstate_get();

        code = qmost_med(this_spec, this_bpm, nps, &medflux);
        if(code != CPL_ERROR_NONE) {
            if(code == CPL_ERROR_DATA_NOT_FOUND) {
                cpl_errorstate_set(prestate);  /* clear */
            }
            else {
                TIDY;
                return cpl_error_set_message(cpl_func, code,
                                             "couldn't calculate "
                                             "median flux for trace %d",
                                             itrace+1);
            }
        }
        else {
            if(trace_medflux_minspc < 0 ||
               medflux < trace_medflux_min) {
                trace_medflux_min = medflux;
                trace_medflux_minspc = itrace+1;
            }
            if(trace_medflux_maxspc < 0 ||
               medflux > trace_medflux_max) {
                trace_medflux_max = medflux;
                trace_medflux_maxspc = itrace+1;
            }

            trace_medflux[ntrace_medflux] = medflux;
            ntrace_medflux++;
        }

        /* Compute SNR */
        for(ips = 0; ips < nps; ips++) {
            if(this_bpm[ips] == 0 &&
               this_var[ips] > 0) {
                snrbuf[ips] = this_spec[ips] / sqrt(this_var[ips]);
                snrbpm[ips] = 0;
            }
            else {
                snrbuf[ips] = 0;
                snrbpm[ips] = 1;
            }
        }

        /* Compute median SNR */
        prestate = cpl_errorstate_get();

        code = qmost_med(snrbuf, snrbpm, nps, &medsnr);
        if(code != CPL_ERROR_NONE) {
            if(code == CPL_ERROR_DATA_NOT_FOUND) {
                cpl_errorstate_set(prestate);  /* clear */
            }
            else {
                TIDY;
                return cpl_error_set_message(cpl_func, code,
                                             "couldn't calculate "
                                             "median SNR for trace %d",
                                             itrace+1);
            }
        }
        else {
            if(trace_medsnr_minspc < 0 ||
               medsnr < trace_medsnr_min) {
                trace_medsnr_min = medsnr;
                trace_medsnr_minspc = itrace+1;
            }
            if(trace_medsnr_maxspc < 0 ||
               medsnr > trace_medsnr_max) {
                trace_medsnr_max = medsnr;
                trace_medsnr_maxspc = itrace+1;
            }

            trace_medsnr[ntrace_medsnr] = medsnr;
            ntrace_medsnr++;

            if(fibinfo_tbl != NULL) {
                if(cpl_table_set(fibinfo_tbl,
                                 colname_snr,
                                 itrace,
                                 medsnr) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't set SNR in "
                                                 "FIBINFO for trace %d",
                                                 itrace+1);
                }
            }
        }

        if(fibinfo_tbl != NULL) {
            /* Compute mean flux */
            prestate = cpl_errorstate_get();
            
            code = qmost_meansig(this_spec, this_bpm, nps,
                                 &meanflux, &sigflux);
            if(code != CPL_ERROR_NONE) {
                if(code == CPL_ERROR_DATA_NOT_FOUND) {
                    cpl_errorstate_set(prestate);  /* clear */
                }
                else {
                    TIDY;
                    return cpl_error_set_message(cpl_func, code,
                                                 "couldn't calculate "
                                                 "mean flux for trace %d",
                                                 itrace+1);
                }
            }
            else {
                if(cpl_table_set(fibinfo_tbl,
                                 colname_meanflux,
                                 itrace,
                                 meanflux) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't set MEANFLUX in "
                                                 "FIBINFO for trace %d",
                                                 itrace+1);
                }
            }
        }
    }

    cpl_free(snrbuf);
    snrbuf = NULL;

    cpl_free(snrbpm);
    snrbpm = NULL;

    if(ntrace_medflux > 0) {
        prestate = cpl_errorstate_get();

        if(qmost_medmad(trace_medflux, NULL, ntrace_medflux,
                        &medmed, &rmsmed) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            medmed = 0;
            rmsmed = 0;
        }

        rmsmed *= CPL_MATH_STD_MAD;

        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT FLUX MED",
                                      medmed);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT FLUX MED",
                                     "[ADU] Median of extracted counts over fibres");
        
        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT FLUX RMS",
                                      rmsmed);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT FLUX RMS",
                                     "[ADU] RMS of extracted counts over fibres");
        
        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT FLUX MIN",
                                      trace_medflux_min);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT FLUX MIN",
                                     "[ADU] Minimum extracted counts over fibres");
        
        cpl_propertylist_update_int(qclist,
                                    "ESO QC EXT FLUX MINSPC",
                                    trace_medflux_minspc);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT FLUX MINSPC",
                                     "Fibre with minimum extracted counts");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT FLUX MAX",
                                      trace_medflux_max);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT FLUX MAX",
                                     "[ADU] Maximum extracted counts over fibres");

        cpl_propertylist_update_int(qclist,
                                    "ESO QC EXT FLUX MAXSPC",
                                    trace_medflux_maxspc);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT FLUX MAXSPC",
                                     "Fibre with maximum extracted counts");
    }

    if(ntrace_medsnr > 0) {
        prestate = cpl_errorstate_get();

        if(qmost_medmad(trace_medsnr, NULL, ntrace_medsnr,
                        &medmed, &rmsmed) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            medmed = 0;
            rmsmed = 0;
        }

        rmsmed *= CPL_MATH_STD_MAD;

        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT SN MED",
                                      medmed);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT SN MED",
                                     "Median SNR over fibres");
        
        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT SN RMS",
                                      rmsmed);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT SN RMS",
                                     "RMS of SNR over fibres");
        
        cpl_propertylist_update_float(qclist,
                                      "ESO QC EXT SN MIN",
                                      trace_medsnr_min);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT SN MIN",
                                     "Minimum SNR over fibres");
        
        cpl_propertylist_update_int(qclist,
                                    "ESO QC EXT SN MINSPC",
                                    trace_medsnr_minspc);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT SN MINSPC",
                                     "Fibre with minimum SNR");

        cpl_propertylist_update_float(qclist,
                                       "ESO QC EXT SN MAX",
                                       trace_medsnr_max);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT SN MAX",
                                     "Maximum SNR over fibres");

        cpl_propertylist_update_int(qclist,
                                    "ESO QC EXT SN MAXSPC",
                                    trace_medsnr_maxspc);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC EXT SN MAXSPC",
                                     "Fibre with minimum SNR");
    }

    cpl_free(trace_medflux);
    trace_medflux = NULL;

    cpl_free(trace_medsnr);
    trace_medsnr = NULL;

    if(colname_snr) {
        cpl_free(colname_snr);
        colname_snr = NULL;
    }

    if(colname_meanflux) {
        cpl_free(colname_meanflux);
        colname_meanflux = NULL;
    }

    return CPL_ERROR_NONE;
}

/**@}*/

/*

$Log$
Revision 1.7  20230306  mji
changed profile width computation to float to give correct profile
width extraction, jrl integer method lost 1 pixel in integration

Revision 1.6  20210107  mji
Changed varmap output to variance for consistency with other calib
data and with extname 

Revision 1.5  2019/02/25 10:35:05  jrl
New memory allocation scheme. Added binning

Revision 1.4  2017/03/14 11:02:19  jim
Now gets information on the ends of the trace from the trace structure

Revision 1.3  2017/01/17 08:59:15  jim
Modified to fix bug with placement of first pixel

Revision 1.2  2016/10/23 15:55:38  jim
Added docs

Revision 1.1  2016/07/06 11:05:48  jim
New file


*/
