/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_LININFO_H
#define QMOST_LININFO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/**
 * @file qmost_lininfo.h
 */

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @ingroup qmost_lininfo
 * @struct qmost_lininfo
 * Non-linearity correction structure.
 */
/*----------------------------------------------------------------------------*/

typedef struct {
    /** @brief Amplifier this row is for */
    int amp;
    /** @brief x pixel lower limit of flat region analysed */
    int xmin;
    /** @brief x pixel upper limit of flat region analysed */
    int xmax;
    /** @brief y pixel lower limit of flat region analysed */
    int ymin;
    /** @brief y pixel upper limit of flat region analysed */
    int ymax;
    /** @brief Degree of polynomial. */
    int nord;
    /** @brief Polynomial solution */
    cpl_polynomial *coefs;
    /** @brief Fit RMS (ADU) */
    double rms;
    /** @brief Number of flat files in fit */
    int nfiles;
    /** @brief Number of good points included in fit */
    int ngood;
    /** @brief Array of median ADU level for each flat in fit */
    double *med;
    /** @brief Array of exposure times for each flat in fit */
    double *exptime;
    /** @brief Array of correction factors from monitor sequence */
    double *moncorr;
    /** @brief Array of linearised counts in ADU for each flat in fit */
    double *linmed;
    /** @brief Boolean array, true if point was flagged bad in fit. */
    unsigned char *fit_flag;
} qmost_lininfo;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_lincrtab (
    cpl_table **tptr,
    int nfiles,
    int nord);

cpl_error_code qmost_lincrinfo (
    int nfiles,
    int nord,
    qmost_lininfo *lin);

void qmost_lindelinfo (
    qmost_lininfo *lin);

cpl_error_code qmost_linchk (
    cpl_table *tptr);

cpl_error_code qmost_linwrite (
    cpl_table *tptr,
    cpl_propertylist *hdr,
    qmost_lininfo *lins,
    int namps);

cpl_error_code qmost_linread (
    cpl_table *tptr,
    qmost_lininfo **lins,
    int *namps);

CPL_END_DECLS

#endif  /* QMOST_LININFO_H */
