/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_sort.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_sort      qmost_sort
 *
 * Sorting routines
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort float array.
 *
 * @param   a          (Modified) Float array to sort.
 * @param   n          (Given)    Length of array.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_f (
    float *a,
    int n)
{
    int iii,ii,i,ifin,j;
    float ta;

    iii = 4;
    while (iii < n)
        iii *= 2;
    iii = qmost_min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a[i] > a[j]) {
                ta = a[j];
                while (1) {
                    a[j] = a[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a[i] <= ta) 
                        break;
                }
                a[j] = ta;
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel float arrays.
 *
 * @param   a          (Modified) Float array to sort on.
 * @param   b          (Modified) Float array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_ff (
    float *a,
    float *b,
    int n)
{
    int iii,ii,i,ifin,j;
    float ta,tb;

    iii = 4;
    while (iii < n)
        iii *= 2;
    iii = qmost_min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a[i] > a[j]) {
                ta = a[j];
                tb = b[j];
                while (1) {
                    a[j] = a[i];
		    b[j] = b[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a[i] <= ta) 
                        break;
                }
                a[j] = ta;
		b[j] = tb;
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel float arrays.
 *
 * @param   a          (Modified) Float array to sort on.
 * @param   b          (Modified) Float array to update in parallel.
 * @param   c          (Modified) Float array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_fff (
    float *a,
    float *b,
    float *c,
    int n)
{
    int iii,ii,i,ifin,j;
    float ta,tb,tc;

    iii = 4;
    while (iii < n)
        iii *= 2;
    iii = qmost_min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a[i] > a[j]) {
                ta = a[j];
                tb = b[j];
                tc = c[j];
                while (1) {
                    a[j] = a[i];
		    b[j] = b[i];
		    c[j] = c[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a[i] <= ta) 
                        break;
                }
                a[j] = ta;
		b[j] = tb;
		c[j] = tc;
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel float arrays.
 *
 * @param   a          (Modified) Float array to sort on.
 * @param   b          (Modified) Float array to update in parallel.
 * @param   c          (Modified) Float array to update in parallel.
 * @param   d          (Modified) Float array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_ffff (
    float *a,
    float *b,
    float *c,
    float *d,
    int n)
{
    int iii,ii,i,ifin,j;
    float ta,tb,tc,td;

    iii = 4;
    while (iii < n)
        iii *= 2;
    iii = qmost_min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a[i] > a[j]) {
                ta = a[j];
                tb = b[j];
                tc = c[j];
                td = d[j];
                while (1) {
                    a[j] = a[i];
		    b[j] = b[i];
		    c[j] = c[i];
		    d[j] = d[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a[i] <= ta) 
                        break;
                }
                a[j] = ta;
		b[j] = tb;
		c[j] = tc;
		d[j] = td;
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel float and integer arrays.
 *
 * @param   ia         (Modified) Float array to sort on.
 * @param   ib         (Modified) Integer array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_fi (
    float *ia,
    int *ib,
    int n)
{
    int i,j,ii,jj,ifin,iu;
    float it;

    jj = 4;
    while(jj < n) 
	jj = 2 * jj;
    jj = qmost_min(n,(3*jj)/4 - 1);
    while (jj > 1) {
	jj = jj/2;
	ifin = n - jj;
	for (ii = 0; ii < ifin; ii++) {
	    i = ii;
	    j = i + jj;
	    if(ia[i] <= ia[j]) 
		continue;
	    it = ia[j];
	    iu = ib[j];
	    do {
		ia[j] = ia[i];
		ib[j] = ib[i];
		j = i;
		i = i - jj;
		if (i < 0)
		    break;
	    } while(ia[i] > it);
	    ia[j] = it;
	    ib[j] = iu;
	}
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel float, unsigned char and integer arrays.
 *
 * @param   a1         (Modified) Float array to sort on.
 * @param   a2         (Modified) Unsigned char array to update in
 *                                parallel.
 * @param   a3         (Modified) Integer array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_fui (
    float *a1,
    unsigned char *a2,
    int *a3,
    int n)
{
    int iii,ii,i,ifin,j,b3;
    unsigned char b2;
    float b1;

    iii = 4;
    while (iii < n)
        iii *= 2;
    iii = qmost_min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a1[i] > a1[j]) {
                b1 = a1[j];
                b2 = a2[j];
                b3 = a3[j];
                while (1) {
                    a1[j] = a1[i];
                    a2[j] = a2[i];
                    a3[j] = a3[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a1[i] <= b1) 
                        break;
                }
                a1[j] = b1;
                a2[j] = b2;
                a3[j] = b3;
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel double arrays.
 *
 * @param   ia         (Modified) Double array to sort on.
 * @param   ib         (Modified) Double array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_dd (
    double *ia,
    double *ib,
    int n)
{
    int i,j,ii,jj,ifin;
    double it,iu;

    jj = 4;
    while(jj < n) 
	jj = 2 * jj;
    jj = qmost_min(n,(3*jj)/4 - 1);
    while (jj > 1) {
	jj = jj/2;
	ifin = n - jj;
	for (ii = 0; ii < ifin; ii++) {
	    i = ii;
	    j = i + jj;
	    if(ia[i] <= ia[j]) 
		continue;
	    it = ia[j];
	    iu = ib[j];
	    do {
		ia[j] = ia[i];
		ib[j] = ib[i];
		j = i;
		i = i - jj;
		if (i < 0)
		    break;
	    } while(ia[i] > it);
	    ia[j] = it;
	    ib[j] = iu;
	}
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel double and integer arrays.
 *
 * @param   ia         (Modified) Double array to sort on.
 * @param   ib         (Modified) Integer array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_di (
    double *ia,
    int *ib,
    int n)
{
    int i,j,ii,jj,ifin,iu;
    double it;

    jj = 4;
    while(jj < n) 
	jj = 2 * jj;
    jj = qmost_min(n,(3*jj)/4 - 1);
    while (jj > 1) {
	jj = jj/2;
	ifin = n - jj;
	for (ii = 0; ii < ifin; ii++) {
	    i = ii;
	    j = i + jj;
	    if(ia[i] <= ia[j]) 
		continue;
	    it = ia[j];
	    iu = ib[j];
	    do {
		ia[j] = ia[i];
		ib[j] = ib[i];
		j = i;
		i = i - jj;
		if (i < 0)
		    break;
	    } while(ia[i] > it);
	    ia[j] = it;
	    ib[j] = iu;
	}
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort parallel double and integer arrays in descending
 *          order.
 *
 * @param   ia         (Modified) Double array to sort on.
 * @param   ic         (Modified) Integer array to update in parallel.
 * @param   n          (Given)    Length of arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sort_rev_di (
    double *ia,
    int *ic,
    int n)
{
    int i,j,ii,jj,ifin,iu;
    double it;

    jj = 4;
    while(jj < n) 
	jj = 2 * jj;
    jj = qmost_min(n,(3*jj)/4 - 1);
    while (jj > 1) {
	jj = jj/2;
	ifin = n - jj;
	for (ii = 0; ii < ifin; ii++) {
	    i = ii;
	    j = i + jj;
	    if(ia[i] >= ia[j]) 
		continue;
	    it = ia[j];
	    iu = ic[j];
	    do {
		ia[j] = ia[i];
		ic[j] = ic[i];
		j = i;
		i = i - jj;
		if (i < 0)
		    break;
	    } while(ia[i] < it);
	    ia[j] = it;
	    ic[j] = iu;
	}
    }
}

/**@}*/
