/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_constants.h"
#include "qmost_filt1d.h"
#include "qmost_spec_combine.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_spec_combine  qmost_spec_combine
 * 
 * Stacking of spectra.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_spec_combine.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void sc_comb4 (
    float *in,
    float *inv,
    int nfiles,
    long nspix, 
    float thresh,
    float *out,
    float *outv);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Stack a list of spectra.
 *
 * Spectra from a pair of imagelists, each containing the extracted 
 * spectrum and corresponding variance, are stacked.  This routine is
 * intended for stacking at OB level, so the object spectra are
 * assumed to be in the same fibres in each input and have the same
 * wavelength scale.
 *
 * The stacking is weighted by signal to noise ratio, and incorporates
 * cosmic rejection.  Cosmic rays are detected by first removing the
 * continuum and calculating the RMS of the continuum removed
 * spectrum.  The continuum removed spectra are then searched for
 * pixels that are outliers above the threshold level defined by the
 * argument thresh.  If a given wavelength bin is an outlier in only
 * one of the spectra being stacked, it is flagged as a cosmic ray and
 * rejected.  If a wavelength bin is an outlier in more than one of
 * the spectra being stacked, it is assumed to be a real emission
 * feature and retained.
 *
 * @param   in_imglist   (Given)    An imagelist with the input
 *                                  spectra in the 2D
 *                                  "ANCILLARY.MOSSPECTRA" 
 *                                  layout.  The data type must be
 *                                  CPL_TYPE_FLOAT.
 * @param   in_varlist   (Given)    An imagelist with the
 *                                  corresponding variance arrays.
 * @param   thresh       (Given)    The rejection threshold in units
 *                                  of the RMS.
 * @param   out_img      (Returned) The output stacked spectrum.
 * @param   out_var      (Returned) The output variance array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If there were no input
 *                                        images.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the image or variance
 *                                        dimensions don't match.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the data type of an input
 *                                        image or variance array was
 *                                        not float.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_spec_combine_lite (
    cpl_imagelist *in_imglist,
    cpl_imagelist *in_varlist,
    float thresh,
    cpl_image **out_img,
    cpl_image **out_var)
{
    int ifile, nfiles, nfiles_var;
    cpl_image *in_img, *in_var;
    int nwave, ispec, nspec, this_nwave, this_nspec;

    float **in_img_buf = NULL;
    float **in_var_buf = NULL;

    float *out_img_buf, *out_var_buf;

    float *in_specs = NULL;
    float *in_specsv = NULL;

    cpl_ensure_code(in_imglist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_varlist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_var != NULL, CPL_ERROR_NULL_INPUT);

    *out_img = NULL;
    *out_var = NULL;

#undef TIDY
#define TIDY                                    \
    if(in_img_buf != NULL) {                    \
        cpl_free(in_img_buf);                   \
        in_img_buf = NULL;                      \
    }                                           \
    if(in_var_buf != NULL) {                    \
        cpl_free(in_var_buf);                   \
        in_var_buf = NULL;                      \
    }                                           \
    if(*out_img != NULL) {                      \
        cpl_image_delete(*out_img);             \
        *out_img = NULL;                        \
    }                                           \
    if(*out_var != NULL) {                      \
        cpl_image_delete(*out_var);             \
        *out_var = NULL;                        \
    }                                           \
    if(in_specs != NULL) {                      \
        cpl_free(in_specs);                     \
        in_specs = NULL;                        \
    }                                           \
    if(in_specsv != NULL) {                     \
        cpl_free(in_specsv);                    \
        in_specsv = NULL;                       \
    }
    
    /* Check the number of images */
    nfiles = cpl_imagelist_get_size(in_imglist);
    if(nfiles < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "there were no input images");
    }

    nfiles_var = cpl_imagelist_get_size(in_varlist);
    if(nfiles != nfiles_var) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "image and variance list sizes don't "
                                     "match: %d != %d",
                                     nfiles, nfiles_var);
    }

    /* Get size */
    in_img = cpl_imagelist_get(in_imglist, 0);
    if(in_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't fetch first image");
    }
    
    nwave = cpl_image_get_size_x(in_img);
    nspec = cpl_image_get_size_y(in_img);

    /* Check compatibility and get pointers to all input data */
    in_img_buf = cpl_calloc(nfiles, sizeof(float *));
    in_var_buf = cpl_calloc(nfiles, sizeof(float *));

    for(ifile = 0; ifile < nfiles; ifile++) {
        /* Get image */
        in_img = cpl_imagelist_get(in_imglist, ifile);
        if(in_img == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't fetch image %d",
                                         ifile);
        }

        /* Check size */
        this_nwave = cpl_image_get_size_x(in_img);
        this_nspec = cpl_image_get_size_y(in_img);
        if(this_nwave != nwave ||
           this_nspec != nspec) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image %d size doesn't match: "
                                         "(%d, %d) != (%d, %d)",
                                         ifile,
                                         this_nwave, this_nspec,
                                         nwave, nspec);
        }

        /* Get pointer */
        in_img_buf[ifile] = cpl_image_get_data_float(in_img);
        if(in_img_buf[ifile] == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "input image %d",
                                         ifile);
        }

        /* Get variance */
        in_var = cpl_imagelist_get(in_varlist, ifile);
        if(in_var == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't fetch variance image %d",
                                         ifile);
        }

        /* Check size */
        this_nwave = cpl_image_get_size_x(in_var);
        this_nspec = cpl_image_get_size_y(in_var);
        if(this_nwave != nwave ||
           this_nspec != nspec) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "variance image %d size doesn't "
                                         "match: (%d, %d) != (%d, %d)",
                                         ifile,
                                         this_nwave, this_nspec,
                                         nwave, nspec);
        }

        /* Get pointer */
        in_var_buf[ifile] = cpl_image_get_data_float(in_var);
        if(in_var_buf[ifile] == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "input variance image %d",
                                         ifile);
        }
    }

    /* Create output */
    *out_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    *out_var = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    if(*out_img == NULL || *out_var == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output spectrum "
                                     "array of size (%d, %d)",
                                     nwave, nspec);
    }

    out_img_buf = cpl_image_get_data_float(*out_img);
    out_var_buf = cpl_image_get_data_float(*out_var);
    if(out_img_buf == NULL || out_var_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer to "
                                     "output spectrum array");
    }

#pragma omp parallel default(none) private(ifile, in_specs, in_specsv) shared(in_img_buf, in_var_buf, nfiles, nspec, nwave, out_img_buf, out_var_buf, thresh)
    {
        /* Allocate workspace */
        in_specs = cpl_malloc(nwave * nfiles * sizeof(float));
        in_specsv = cpl_malloc(nwave * nfiles * sizeof(float));

#pragma omp for
        for(ispec = 0; ispec < nspec; ispec++) {
            for(ifile = 0; ifile < nfiles; ifile++) {
                /* Copy image and variance */
                memcpy(in_specs + ifile*nwave,
                       in_img_buf[ifile] + ispec*nwave,
                       nwave * sizeof(float));

                memcpy(in_specsv + ifile*nwave,
                       in_var_buf[ifile] + ispec*nwave,
                       nwave * sizeof(float));
            }

            /* Do the combination */
            sc_comb4(in_specs, in_specsv, nfiles, nwave, thresh,
                     out_img_buf + ispec*nwave,
                     out_var_buf + ispec*nwave);
        }

        cpl_free(in_specs);
        in_specs = NULL;
        
        cpl_free(in_specsv);
        in_specsv = NULL;
    }
    
    cpl_free(in_img_buf);
    in_img_buf = NULL;

    cpl_free(in_var_buf);
    in_var_buf = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Combine spectra with rejection and weighting by s/v.
 *
 * Spectra are combined with rejection and weighted by s/v. Cosmic
 * rays are detected by removing the continuum first and working out
 * the RMS of the spectrum. Then we go through the flattened spectra
 * and look for pixels that stick above the threshold level. If only
 * pixel at a given wavelength bin sticks up, then it's assumed to be
 * a cosmic ray and it's flagged.
 *
 * @param   in           (Given)    A 1d array with all of the input
 *                                  spectra.
 * @param   inv          (Given)    A 1d array with the input variance
 *                                  spectra.
 * @param   nfiles       (Given)    The number of input spectra.
 * @param   nspix        (Given)    The number of pixels per spectrum.
 * @param   thresh       (Given)    The rejection threshold in units
 *                                  of the RMS.
 * @param   out          (Modified) The output combined spectrum.
 * @param   outv         (Modified) The output combined variance
 *                                  array.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void sc_comb4 (
    float *in,
    float *inv,
    int nfiles,
    long nspix, 
    float thresh,
    float *out,
    float *outv)
{
    float *ispec,*invar,*fspec;
    float sum,sumv,sigma,sumw,wt;
    int i,j,nfud;
    unsigned char *b;
    cpl_errorstate prestate;

    unsigned char *bpm = NULL;
    float *smed = NULL;
    float *vmed = NULL;
    float *psi = NULL;
    float *filtered = NULL;
    float *cont = NULL;
    float *lines = NULL;
    float *mask = NULL;
    float *rmss = NULL;

    /* Do nothing for the situation where you only have 1 spectrum */

    if (nfiles == 1) {
	memcpy(out,in,nspix*sizeof(float));

	if (inv != NULL) {
            if (outv != NULL) {
                memcpy(outv,inv,nspix*sizeof(float));
            }
        }
        else {
            if (outv != NULL) {
                memset(outv,0,nspix*sizeof(float));
            }
        }

	return;
    }

    /* Get a bad pixel mask array. If there are variance estimates
       then flag the pixels where that is zero */
    
    bpm = (unsigned char *) cpl_calloc(nspix*nfiles,sizeof(unsigned char));

    if (inv != NULL) {
        for (i = 0; i < nfiles*nspix; i++)
            if (inv[i] == 0.0)
                bpm[i] = 1;
    }

    /* Median signal <S> and median variance <V> */

    smed = (float *) cpl_malloc(nfiles*sizeof(float));
    vmed = (float *) cpl_malloc(nfiles*sizeof(float));

    sum = 0;
    sumv = 0;

    for (i = 0; i < nfiles; i++) {
        ispec = in + i*nspix;
        b = bpm + i*nspix;

        /* Median signal, not less than zero */
        prestate = cpl_errorstate_get();

        if(qmost_med(ispec,b,(int)nspix,&(smed[i])) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            smed[i] = 0;
        }

        if(smed[i] < 0.0) {
            smed[i] = 0.0;
        }

        if(inv != NULL) {
            /* Median variance */
            invar = inv + i*nspix;

            if(qmost_med(invar,b,(int)nspix,&(vmed[i])) != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);
                vmed[i] = 0;
            }
        }
        else {
            /* Unweighted sum */
            vmed[i] = smed[i];
        }

        sum += smed[i];
        sumv += vmed[i];
    }

    /* Decide weighting scheme and calculate weights.  The <= ensures
     * 0 <= 0 triggers inverse variance. */

    psi = (float *) cpl_malloc(nfiles*sizeof(float));

    if(inv != NULL &&
       sum <= 50 * sqrt(0.5 * CPL_MATH_PI * nfiles * sumv / nspix)) {
        /* Inverse variance weights */
        for(i = 0; i < nfiles; i++) {
            psi[i] = 1.0 / nfiles;
        }
    }
    else {
        /* S/V weights */
        for(i = 0; i < nfiles; i++) {
            psi[i] = smed[i] / sum;
        }
    }

    /* Create normalised spectra */

    filtered = cpl_malloc(nspix*nfiles*sizeof(float));
    cont = cpl_malloc(nspix*sizeof(float));
    lines = cpl_malloc(nspix*sizeof(float));
    mask = cpl_malloc(nspix*sizeof(float));
    rmss = cpl_malloc(nfiles*sizeof(float));

    for (i = 0; i < nfiles; i++) {
	ispec = in + i*nspix;
	fspec = filtered + i*nspix;
        b = bpm + i*nspix;
	memset(mask,0,nspix*sizeof(float));
	qmost_skyfilt(nspix,ispec,cont,lines,mask,&sigma,501,255,3,3.0,3.0);
	for (j = 0; j < nspix; j++) 
	    fspec[j] = ispec[j] - cont[j];

        prestate = cpl_errorstate_get();

	if(qmost_medmad(fspec,b,nspix,&sum,&sumv) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            sum = 0;
            sumv = 0;
        }

	sumv *= 1.48*thresh;
	rmss[i] = sumv;
    }

    cpl_free(cont);
    cont = NULL;

    cpl_free(lines);
    lines = NULL;

    cpl_free(mask);
    mask = NULL;

    /* For each output pixel look at the contributing input pixels 
       and see which ones stick above the threshold. If only 1 contribution
       sticks above the threshold, then we say it's a cosmic ray. Otherwise
       we assume it's real */

    for (j = 0; j < nspix; j++) {
	nfud = 0;
	for (i = 0; i < nfiles; i++) {
	    b = bpm + i*nspix;
            if (b[j] != 0)
                continue;
	    fspec = filtered + i*nspix;
	    if (fspec[j] > rmss[i]) 
		nfud++;
	}
	if (nfud <= 1) {
	    for (i = 0; i < nfiles; i++) {
		b = bpm + i*nspix;
                if (b[j] != 0)
                    continue;
		fspec = filtered + i*nspix;
		if (fspec[j] > rmss[i]) 
		    b[j] = 1;
	    }
	}
    }

    cpl_free(rmss);
    rmss = NULL;

    /* Now form the output spectrum */
    
    memset(out,0,nspix*sizeof(float));

    if (outv != NULL)  {
        memset(outv,0,nspix*sizeof(float));
    }

    for (j = 0; j < nspix; j++) {
	sum = 0.0;
	sumv = 0.0;
        sumw = 0.0;
	for (i = 0; i < nfiles; i++) {
	    ispec = in + i*nspix;
	    if (inv != NULL) 
		invar = inv + i*nspix;
            b = bpm + i*nspix;
            if (b[j] != 0)
                continue;

            wt = psi[i] / vmed[i];

	    sum += wt * ispec[j];
	    if (inv != NULL)
		sumv += wt*wt * invar[j];
            sumw += wt * psi[i];
	}
        if (sumw != 0.0) {
            out[j] = sum/sumw;
            if (outv != NULL) 
                outv[j] = sumv/(sumw*sumw);
        } else {
            out[j] = 0.0;
            if (outv != NULL)
                outv[j] = 0.0;
        }
    }

    /* Tidy up */

    cpl_free(filtered);
    filtered = NULL;

    cpl_free((void *) psi);
    psi = NULL;

    cpl_free((void *) smed);
    smed = NULL;

    cpl_free((void *) vmed);
    vmed = NULL;

    cpl_free(bpm);
    bpm = NULL;
}	

/**@}*/

/*

$Log$
Revision 1.16  20191125  mji
altered output format for superstack to match MOS format and removed
surplus 2nd axis header keywords from image extensions

Revision 1.15  20190830  mji 
fixed bug in flux scaling, possible dynamic range issue with flux invar
hence internal scaling and changed output to flux error array
Added bunit cards to all extensions

Revision 1.14  2019/02/25 10:48:09  jrl
New memory allocation scheme, removed disused routines

Revision 1.13  2018/10/12 10:10:58  jrl
added superstack routine and sc_comb4a

Revision 1.12  2018/08/01 09:10:42  jim
Fixed problem with RUNSXXXX keywords

Revision 1.11  2018/06/27 09:53:08  jim
Adds a list of runs of contributing spectra to the primary

Revision 1.10  2017/08/02 08:57:07  jim
Modified to update the exposure time to the total time and to include
NCOMB (number of combined images)

Revision 1.9  2017/06/07 10:58:05  jim
Fixed scaling for low s/n

Revision 1.8  2017/06/05 16:52:39  jim
Modified the scaling of the output variance

Revision 1.7  2017/06/05 11:03:16  jim
Fixed output variance estimate so that we conserve the variance

Revision 1.6  2017/06/02 14:19:26  jim
Fixed bug where variance in regions with no data was not being set to zero

Revision 1.5  2017/05/31 11:54:18  jim
Fixed so that places where there are no contributions from any spectra are
now set to zero rather than some inter/extrapolation

Revision 1.4  2017/03/14 11:31:48  jim
Fixed so that the variance input is treated as an inverse variance

Revision 1.3  2016/10/26 11:20:40  jim
Added docs

Revision 1.2  2016/07/21 08:34:37  jim
fixed bug causing problems with output file extension

Revision 1.1  2016/07/14 07:11:18  jim
New file


*/
