/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_stats.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_stats     qmost_stats
 *
 * Statistics convenience routines
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find median of a float data array.  Ignore bad pixels.
 *
 * The median of a float data array is found, ignoring bad pixels (if
 * a BPM is supplied).  Wraps CPL routines.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   value      (Returned) Median of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If npts <= 0.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_med (
    float *data,
    unsigned char *bpm,
    int npts,
    float *value)
{
    cpl_array *tmparr = NULL;
    int i, nbad;
    cpl_errorstate prestate;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(value != NULL, CPL_ERROR_NULL_INPUT);

    tmparr = cpl_array_wrap_float(data, npts);
    if(tmparr == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not wrap input data");
    }

    if(bpm != NULL) {
        nbad = 0;

        for(i = 0; i < npts; i++) {
            if(bpm[i]) {
                cpl_array_set_invalid(tmparr, i);
                nbad++;
            }
        }

        if(nbad >= npts) {
            cpl_array_unwrap(tmparr);
            tmparr = NULL;

            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "no good pixels");
        }
    }

    prestate = cpl_errorstate_get();

    *value = cpl_array_get_median(tmparr);
    if(!cpl_errorstate_is_equal(prestate) &&
        cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_array_unwrap(tmparr);
        tmparr = NULL;

        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not compute median");
    }

    cpl_array_unwrap(tmparr);
    tmparr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find median of a double data array.  Ignore bad pixels.
 *
 * The median of a double data array is found, ignoring bad pixels (if
 * a BPM is supplied).  Wraps CPL routines.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   value      (Returned) Median of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If npts <= 0.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dmed (
    double *data,
    unsigned char *bpm,
    int npts,
    double *value)
{
    cpl_array *tmparr = NULL;
    int i, nbad;
    cpl_errorstate prestate;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(value != NULL, CPL_ERROR_NULL_INPUT);

    tmparr = cpl_array_wrap_double(data, npts);
    if(tmparr == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not wrap input data");
    }

    if(bpm != NULL) {
        nbad = 0;

        for(i = 0; i < npts; i++) {
            if(bpm[i]) {
                cpl_array_set_invalid(tmparr, i);
                nbad++;
            }
        }

        if(nbad >= npts) {
            cpl_array_unwrap(tmparr);
            tmparr = NULL;

            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "no good pixels");
        }
    }

    prestate = cpl_errorstate_get();

    *value = cpl_array_get_median(tmparr);
    if(!cpl_errorstate_is_equal(prestate) &&
        cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_array_unwrap(tmparr);
        tmparr = NULL;

        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not compute median");
    }

    cpl_array_unwrap(tmparr);
    tmparr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find median and median absolute deviation of data array.
 *
 * An array of values is given.  A median is found.  The MAD is
 * defined as the median absolute residual from the ensemble median
 * for the array.  Bad pixels are ignored if a BPM is supplied.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   medval     (Returned) Median of data array.
 * @param   madval     (Returned) Median absolute deviation of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If npts <= 0.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_medmad (
    float *data,
    unsigned char *bpm,
    int npts,
    float *medval,
    float *madval)
{
    float *absdev = NULL;
    int i;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(medval != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(madval != NULL, CPL_ERROR_NULL_INPUT);

    if(qmost_med(data, bpm, npts, medval) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not compute median");
    }

    absdev = cpl_malloc(npts * sizeof(float));

    for (i = 0; i < npts; i++)
        absdev[i] = fabsf(data[i] - (*medval));

    if(qmost_med(absdev, bpm, npts, madval) != CPL_ERROR_NONE) {
        cpl_free(absdev);
        absdev = NULL;

        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not compute MAD");
    }

    cpl_free(absdev);
    absdev = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find median and median absolute deviation of data array.
 *
 * An array of values is given.  A median is found.  The MAD is
 * defined as the median absolute residual from the ensemble median
 * for the array.  Bad pixels are ignored if a BPM is supplied.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   medval     (Returned) Median of data array.
 * @param   madval     (Returned) Median of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If npts <= 0.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dmedmad (
    double *data,
    unsigned char *bpm,
    int npts,
    double *medval,
    double *madval)
{
    double *absdev = NULL;
    int i;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(medval != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(madval != NULL, CPL_ERROR_NULL_INPUT);

    if(qmost_dmed(data, bpm, npts, medval) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not compute median");
    }

    absdev = cpl_malloc(npts * sizeof(double));

    for (i = 0; i < npts; i++)
        absdev[i] = fabs(data[i] - (*medval));

    if(qmost_dmed(absdev, bpm, npts, madval) != CPL_ERROR_NONE) {
        cpl_free(absdev);
        absdev = NULL;

        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not compute MAD");
    }

    cpl_free(absdev);
    absdev = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find median and median absolute deviation of data array
 *          with windowing.
 *
 * An array of values is given.  The array is windowed by low and high
 * threshold values.  A median is found.  The MAD is defined as the
 * median absolute residual from the ensemble median for the array.
 * Bad pixels are ignored if a BPM is supplied.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   lcut       (Given)    The lower threshold for the cut.
 * @param   hcut       (Given)    The upper threshold for the cut.
 * @param   medval     (Returned) Median of data array.
 * @param   madval     (Returned) Median of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels
 *                                    within the given window.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_medmadcut (
    float *data,
    unsigned char *bpm,
    int npts, 
    float lcut,
    float hcut,
    float *medval, 
    float *madval)
{
    float *ndata = NULL;
    int i, ng;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(medval != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(madval != NULL, CPL_ERROR_NULL_INPUT);

    /* First get some workspace */

    ndata = cpl_malloc(npts * sizeof(float));

    /* Now transfer any good data across */

    ng = 0;
    if (bpm != NULL) {
        for (i = 0; i < npts; i++) {
            if (data[i] < lcut || data[i] > hcut || bpm[i])
                continue;
            ndata[ng++] = data[i];
        } 
    } else {
        for (i = 0; i < npts; i++) {
            if (data[i] < lcut || data[i] > hcut)
                continue;
            ndata[ng++] = data[i];
        } 
    }

    /* Are there any pixels left? */

    if (ng == 0) {
        *medval = FLT_MAX;
        *madval = FLT_MAX;

        cpl_free(ndata);
        ndata = NULL;

        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "all values were cut");
    }

    /* Find median and mad */

    if(qmost_medmad(ndata, NULL, ng, medval, madval) != CPL_ERROR_NONE) {
        cpl_free(ndata);
        ndata = NULL;
        return cpl_error_get_code();
    }

    cpl_free(ndata);
    ndata = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find mean and standard deviation of data array.
 *
 * The mean and standard deviation of a float data array are found in
 * the usual way, while ignoring bad pixels (if a BPM is supplied).
 * Wraps CPL routines.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   mean       (Returned) Mean of data array.
 * @param   sig        (Returned) Unbiased estimator of the standard
 *                                deviation of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_meansig (
    float *data,
    unsigned char *bpm,
    int npts,
    float *mean,
    float *sig)
{
    cpl_array *tmparr = NULL;
    cpl_array *tmparr2 = NULL;
    int i;
    cpl_error_code code;
    cpl_errorstate prestate;
    float var;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mean != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sig != NULL, CPL_ERROR_NULL_INPUT);

    tmparr = cpl_array_new(npts, CPL_TYPE_FLOAT);
    if(tmparr == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not wrap input data");
    }

    code = cpl_array_copy_data_float(tmparr, data);
    if(code != CPL_ERROR_NONE) {
        cpl_array_delete(tmparr);
        tmparr = NULL;

        return cpl_error_set_message(cpl_func, code,
                                     "could not copy input data");
    }

    if(bpm != NULL) {
        for(i = 0; i < npts; i++) {
            if(bpm[i]) {
                cpl_array_set_invalid(tmparr, i);
            }
        }
    }

    if(cpl_array_has_valid(tmparr)) {
        prestate = cpl_errorstate_get();

        *mean = cpl_array_get_mean(tmparr);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            cpl_array_delete(tmparr);
            tmparr = NULL;
            
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute mean");
        }
        
        cpl_array_subtract_scalar(tmparr, *mean);

        tmparr2 = cpl_array_duplicate(tmparr);
        cpl_array_multiply(tmparr, tmparr2);
        cpl_array_delete(tmparr2);
        tmparr2 = NULL;

        var = cpl_array_get_mean(tmparr);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            cpl_array_delete(tmparr);
            tmparr = NULL;
            
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute stdev");
        }

        *sig = var > 0.0 ? sqrt(var) : 0.0;
    }
    else {
        *mean = FLT_MAX;
        *sig = FLT_MAX;

        cpl_array_delete(tmparr);
        tmparr = NULL;
        
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "no good pixels");
    }

    cpl_array_delete(tmparr);
    tmparr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find mean and standard deviation of data array with
 *          clipping.
 *
 * The mean and standard deviation of a float data array are found in
 * the usual way, while ignoring bad pixels (if a BPM is supplied).
 * Pixels that are outside a predefined data window are also removed.
 * Wraps CPL routines.
 *
 * @param   data       (Given)    Input data.
 * @param   bpm        (Given)    The input BPM, or NULL.
 * @param   npts       (Given)    Number of data points.
 * @param   lcut       (Given)    The lowest allowable value.
 * @param   hcut       (Given)    The highest allowable value.
 * @param   mean       (Returned) Mean of data array.
 * @param   sig        (Returned) Unbiased estimator of the standard
 *                                deviation of data array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no good pixels.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_meansigcut (
    float *data,
    unsigned char *bpm,
    int npts,
    float lcut,
    float hcut,
    float *mean,
    float *sig)
{
    cpl_array *tmparr = NULL;
    cpl_array *tmparr2 = NULL;
    int i;
    cpl_error_code code;
    cpl_errorstate prestate;
    float var;

    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mean != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sig != NULL, CPL_ERROR_NULL_INPUT);

    tmparr = cpl_array_new(npts, CPL_TYPE_FLOAT);
    if(tmparr == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not wrap input data");
    }

    code = cpl_array_copy_data_float(tmparr, data);
    if(code != CPL_ERROR_NONE) {
        cpl_array_delete(tmparr);
        tmparr = NULL;

        return cpl_error_set_message(cpl_func, code,
                                     "could not copy input data");
    }

    if(bpm != NULL) {
        for(i = 0; i < npts; i++) {
            if(bpm[i]) {
                cpl_array_set_invalid(tmparr, i);
            }
            else if(data[i] < lcut || data[i] > hcut) {
                cpl_array_set_invalid(tmparr, i);
            }
        }
    }
    else {
        for(i = 0; i < npts; i++) {
            if(data[i] < lcut || data[i] > hcut) {
                cpl_array_set_invalid(tmparr, i);
            }
        }
    }

    if(cpl_array_has_valid(tmparr)) {
        prestate = cpl_errorstate_get();

        *mean = cpl_array_get_mean(tmparr);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            cpl_array_delete(tmparr);
            tmparr = NULL;
            
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute mean");
        }
        
        cpl_array_subtract_scalar(tmparr, *mean);

        tmparr2 = cpl_array_duplicate(tmparr);
        cpl_array_multiply(tmparr, tmparr2);
        cpl_array_delete(tmparr2);
        tmparr2 = NULL;

        var = cpl_array_get_mean(tmparr);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            cpl_array_delete(tmparr);
            tmparr = NULL;
            
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute stdev");
        }

        *sig = var > 0.0 ? sqrt(var) : 0.0;
    }
    else {
        *mean = FLT_MAX;
        *sig = FLT_MAX;

        cpl_array_delete(tmparr);
        tmparr = NULL;
        
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "no good pixels");
    }

    cpl_array_delete(tmparr);
    tmparr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find the sum of all the values in a BPM.
 *
 * Find the sum of all values in a BPM. Useful for seeing of all your
 * values are bad or not.
 *
 * @param   bpm        (Given)    The input BPM.
 * @param   npts       (Given)    Number of values in the input BPM
 *                                array.
 * @param   sumb       (Modified) The sum of the BPM.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_sumbpm (
    unsigned char *bpm,
    int npts,
    int *sumb)
{
    int j;

    *sumb = 0;
    for (j = 0; j < npts; j++)
        *sumb += bpm[j];
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find the first good pixel in a BPM.
 *
 * @param   bpm        (Given)    The input BPM.
 * @param   npts       (Given)    Number of values in the input BPM
 *                                array.
 *
 * @return  int        The index of the first good pixel in the BPM,
 *                     or npts if there were no good pixels.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

int qmost_firstgood (
    unsigned char *bpm,
    int npts)
{
    int i;
    for (i = 0; i < npts; i++) {
        if (bpm[i] == 0)
            break;
    }
    return(i);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find the last good pixel in a BPM.
 *
 * @param   bpm        (Given)    The input BPM.
 * @param   npts       (Given)    Number of values in the input BPM
 *                                array.
 *
 * @return  int        The index of the last good pixel in the BPM,
 *                     or -1 if there were no good pixels.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

int qmost_lastgood (
    unsigned char *bpm,
    int npts)
{
    int i;
    for (i = npts-1; i >= 0; i--) {
        if (bpm[i] == 0)
            break;
    }
    return(i);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate median and sigma of histogrammed integer image
 *          data.
 *
 * This routine is usually called through the driver routine
 * qmost_skylevel_image() to process the histogram, but can be called
 * directly.  The background level and noise are calculated using
 * a robust iteratively clipped median and 1.48*MAD, where the
 * clipping threshold and number of iterations can be specified using
 * parameters, or clipping disabled entirely by setting the
 * appropriate values given in the parameter descriptions.
 *
 * @param   ihist      (Given)    The input histogram.
 * @param   ihmin      (Given)    The lowest populated index in the
 *                                histogram, or 0 if unknown.
 * @param   ihmax      (Given)    The highest populated index in the
 *                                histogram, or the highest index
 *                                (length - 1) in the histogram.
 * @param   mpix       (Given)    The total number of pixels in the
 *                                histogram (sum of the histogram).
 * @param   clip_low   (Given)    The number of sigma below which a
 *                                point should be considered and
 *                                outlier and rejected, or -FLT_MAX
 *                                for no lower envelope clipping.
 * @param   clip_high  (Given)    The number of sigma above which a
 *                                point should be considered and
 *                                outlier and rejected, or FLT_MAX for
 *                                no upper envelope clipping.
 * @param   clip_niter (Given)    The number of outlier rejection
 *                                clipping iterations to perform, or 0
 *                                for no clipping.
 * @param   skylev     (Returned) The resulting median sky level.
 * @param   sigma      (Returned) The resulting sigma ("1.48*MAD").
 * @param   toplev     (Returned) The 90th percentile.  Can be given
 *                                as NULL pointer if not required.
 *
 * @return  void
 *
 * @note    The "sigma" is nominally 1.48*MAD but the MAD is estimated
 *          using the upper quartile only.  The routine is usually
 *          used for sky background estimation where the one sided
 *          estimate uses the higher / longer tail of the distribution.
 *
 * @note    The arguments ihmin and ihmax can be used to improve
 *          efficiency by keeping track of which range of histogram
 *          bins are populated in the caller and allow the calculation
 *          to skip the iterations needed to loop past the
 *          bias/pedestal level or brightest pixel, respectively.  The
 *          full range of the histogram can be given if unknown or
 *          don't care.
 *
 * @note    This routine is optimised for sky levels close to ihmin
 *          and low noise levels relative to the total range.  Both
 *          are typically true for astronomical data.
 *
 * @author  Jonathan Irwin, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_skylevel (
    int *ihist,
    int ihmin,
    int ihmax,
    int mpix,
    float clip_low,
    float clip_high,
    int clip_niter,
    float *skylev,
    float *sigma,
    float *toplev)
{
    int i, s, n, iloop;
    int tclipl, tcliph, iclipl, icliph, irej, mcpix;
    float skymed, sigmed, skysig, skymedc, sigmedc, sigmac;
    float fclipl, fcliph;

    /* Find points for sigma, then median */
    n = (mpix+3)/4;

    i = ihmin;
    s = ihist[i];

    while(s <= n) {
        i++;
        s += ihist[i];
    }

    sigmed = i + (n - s) / ((float) ihist[i]) + 0.5;

    n = (mpix+1)/2;  /* guaranteed >= previous "n" */

    while(s <= n) {
        i++;
        s += ihist[i];
    }

    skymed = i + (n - s) / ((float) ihist[i]) + 0.5;

    if(toplev != NULL) {
        n = mpix - (mpix + 9) / 10;

        while(s <= n) {
            i++;
            s += ihist[i];
        }

        *toplev = i + (n - s) / ((float) ihist[i]) + 0.5;
    }

    /* the 1.48 converts MAD to rms equivalent */
    skysig = 1.48 * (skymed - sigmed);
    if(skysig < 0.5)
        skysig = 0.5;

    /* do an iterative clip to give a more robust estimator */
    iclipl = ihmin;
    icliph = ihmax;
    mcpix = mpix;
    skymedc = skymed;
    sigmac = skysig;

    for(iloop = 0; iloop < clip_niter; iloop++) {
        /* Clip thresholds, range check and then convert to integer */
        fclipl = skymedc + clip_low * sigmac + 1;
        fcliph = skymedc + clip_high * sigmac - 1;

        if(fclipl < ihmin)
            tclipl = ihmin;
        else if(fclipl > ihmax)
            tclipl = ihmax;
        else
            tclipl = fclipl + 0.5;   /* nearest integer, positive quantity */

        if(fcliph < ihmin)
            tcliph = ihmin;
        else if(fcliph > ihmax)
            tcliph = ihmax;
        else
            tcliph = fcliph + 0.5;   /* nearest integer, positive quantity*/

        if(iloop == 0) {
            /* First time, just count between the limits.  This should
             * usually be quicker provided the clipping range is small
             * compared to [ihmin, ihmax]. */
            mcpix = 0;

            for(i = tclipl; i <= tcliph; i++)
                mcpix += ihist[i];

            iclipl = tclipl;
            icliph = tcliph;
        }
        else {
            /* How many points to reject? */
            irej = 0;

            for(i = iclipl; i < tclipl; i++)
                irej += ihist[i];

            for(i = tcliph+1; i <= icliph; i++)
                irej += ihist[i];
      
            /* Is there any work to do? */
            if(irej == 0)
                break;

            iclipl = tclipl;
            icliph = tcliph;

            /* New number of points */
            mcpix -= irej;
        }

        /* Find points for sigma, then median */
        n = (mcpix+3)/4;

        i = iclipl;
        s = ihist[i];
    
        while(s <= n) {
            i++;
            s += ihist[i];
        }
    
        sigmedc = i + (n - s) / ((float) ihist[i]) + 0.5;
    
        n = (mcpix+1)/2;  /* guaranteed >= previous "n" */
    
        while(s <= n) {
            i++;
            s += ihist[i];
        }
    
        skymedc = i + (n - s) / ((float) ihist[i]) + 0.5;

        sigmac = 1.48 * (skymedc - sigmedc);
        if(sigmac < 0.5)
            sigmac = 0.5;
    }

    *skylev = skymedc;
    *sigma = sigmac;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate median and sigma of integer image data.
 *
 * Histogramming is used to compute a precise median and robustly
 * estimated sigma of integer image data.  The histogram is computed
 * over the range specified by "minlev" and "maxlev", any pixels
 * outside this range are clamped to these limits.  Usually for
 * unsigned 16-bit integer data, minlev = 0 and maxlev = 65535 but
 * there are other special use cases found elsewhere in the pipeline, 
 * for example in the qmost_detector_noise recipe where the routine is
 * used on sum and difference images with different ranges.
 *
 * @param   in_image   (Given)    Input image to process.  The data
 *                                type must be one that can be cast to
 *                                float.
 * @param   minlev     (Given)    The lower limit of the histogram.
 * @param   maxlev     (Given)    The upper limit of the histogram.
 * @param   clip_low   (Given)    The number of sigma below which a
 *                                point should be considered and
 *                                outlier and rejected, or -FLT_MAX
 *                                for no lower envelope clipping.
 * @param   clip_high  (Given)    The number of sigma above which a
 *                                point should be considered and
 *                                outlier and rejected, or FLT_MAX for
 *                                no upper envelope clipping.
 * @param   clip_niter (Given)    The number of outlier rejection
 *                                clipping iterations to perform, or 0
 *                                for no clipping.
 * @param   skylev     (Returned) The resulting median sky level.
 * @param   sigma      (Returned) The resulting sigma (1.48*MAD).
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If minlev > maxlev.
 * @retval  CPL_ERROR_NULL_INPUT      If the image or one of the
 *                                    output pointers are NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input image data type was
 *                                    not float.
 *
 * If there are no pixels within the specified range, skylev and sigma
 * are left unchanged but no error is raised.  This condition can be
 * detected by setting them to a known value that won't be seen in the
 * image (for example, NAN).
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_skylevel_image (
    cpl_image *in_image,
    int minlev,
    int maxlev,
    float clip_low,
    float clip_high,
    int clip_niter,
    float *skylev,
    float *sigma)
{
    cpl_image *tmp_image = NULL;
    cpl_image *image;

    float *map = NULL;
    cpl_binary *mask = NULL;
    int npix;

    int *hist = NULL;
    int nhist, hmin, hmax, mpix, v, p;
 
    float f;
    
    cpl_ensure_code(in_image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skylev != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sigma != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(tmp_image) {                             \
        cpl_image_delete(tmp_image);            \
        tmp_image = NULL;                       \
    }                                           \
    if(hist) {                                  \
        cpl_free(hist);                         \
        hist = NULL;                            \
    }

    /* Check if image is float, cast if not */
    if(cpl_image_get_type(in_image) == CPL_TYPE_FLOAT) {
        image = in_image;
    }
    else {
        tmp_image = cpl_image_cast(in_image, CPL_TYPE_FLOAT);
        if(tmp_image == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't cast input image "
                                         "to float");
        }

        image = tmp_image;
    }

    /* Get image */
    map = cpl_image_get_data_float(image);
    if(map == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    npix = cpl_image_get_size_x(image) * cpl_image_get_size_y(image);

    mask = cpl_mask_get_data(cpl_image_get_bpm(image));
    if(mask == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get bad pixel mask "
                                     "for input image");
    }

    /* Compute size of histogram, allocate workspace and clear */
    nhist = maxlev - minlev + 1;

    if(nhist < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "couldn't get bad pixel mask "
                                     "for input image");
    }

    hist = cpl_calloc(nhist, sizeof(int));

    /* Accumulate pixels */
    hmin = nhist-1;
    hmax = 0;
    
    mpix = 0;
    
    for(p = 0; p < npix; p++) {
        /* Good, finite pixels only, ignore NaNs and infinities */
        if(mask[p] == 0 && isfinite(map[p])) {
            f = rintf(map[p]) - minlev;
            
            if(f < 0) {
                v = 0;
            }
            else if(f > nhist-1) {
                v = nhist-1;
            }
            else {
                v = f;
            }
            
            hist[v]++;
            mpix++;
            
            if(v < hmin) {
                hmin = v;
            }
            if(v > hmax) {
                hmax = v;
            }
        }
    }
    
    if(mpix > 0) {
        qmost_skylevel(hist, hmin, hmax, mpix,
                       clip_low, clip_high, clip_niter,
                       skylev, sigma, NULL);
        
        *skylev += minlev;
    }

    TIDY;

    return CPL_ERROR_NONE;
}

/**@}*/
