/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_WAVEINFO_H
#define QMOST_WAVEINFO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/**
 * @file qmost_waveinfo.h
 */

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @ingroup qmost_waveinfo
 * @struct qmost_waveinfo
 * Wavelength solution structure.
 */
/*----------------------------------------------------------------------------*/

typedef struct {
    /** @brief Spectrum number in table. */
    int            specnum;
    /** @brief Is fibre live? */
    int            live;
    /** @brief Degree of polynomial. */
    int            nord;
    /** @brief Polynomial wavelength solution */
    cpl_polynomial *coefs;
    /** @brief Reference spectral pixel coordinate for wavelength
      * solution. */
    double         xref;
    /** @brief Number of good arc lines included in fit. */
    int           ngood;
    /** @brief Median fit residual (A). */
    double        medresid;
    /** @brief Fit RMS residual (A). */
    double        fit_rms;
    /** @brief Wavelength at centre of spectral pixel 1 */
    double        wave1;
    /** @brief Wavelength at centre of spectral pixel n */
    double        waven;
    /** @brief Disperson (A / pixel). */
    double        dwave;
    /** @brief Number of lines measured. */
    int           nlines;
    /** @brief Spectral pixel coordinate of line. */
    double        *xpos;
    /** @brief Arc line FWHM (A). */
    float         *fwhm;
    /** @brief Calculated arc line wavelength from master wavelength
      * solution (A). */
    double        *wave_calc;
    /** @brief True arc line wavelength from line list (A). */
    double        *wave_true;
    /** @brief Boolean array, true if point was flagged bad in fit. */
    unsigned char *fit_flag;
    /** @brief Wavelength residual (correction) for line in OB arc
      * compared to master wavelength solution (A). */
    double        *wave_cor;
    /** @brief Peak counts in line (ADU) */
    float         *peak;
    /** @brief Measure of contrast of peak compared to continuum */
    float         *contrast;
} qmost_waveinfo;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wvcreate(
    cpl_table **fptr,
    int nord,
    int nlinesmax);

cpl_error_code qmost_wvopen(
    cpl_table *tptr,
    int *nwv,
    qmost_waveinfo **wv);

cpl_error_code qmost_wvread1(
    cpl_table *tptr,
    int row,
    qmost_waveinfo *wv);

cpl_error_code qmost_wvwrite1(
    cpl_table *tptr,
    int row,
    qmost_waveinfo wv);

cpl_error_code qmost_wvchk(
    cpl_table *tptr);

double qmost_wvmeandisp(
    int nwv,
    qmost_waveinfo *wv);

void qmost_wvendpoints(
    int nwv,
    qmost_waveinfo *wv,
    double *minwave1,
    double *maxwave1,
    double *minwaven, 
    double *maxwaven);

cpl_error_code qmost_wave_qc (
    cpl_table *wave_tbl,
    cpl_table *ref_wave_tbl,
    int obsol,
    cpl_propertylist *qclist);

void qmost_wvclose(
    int nwv,
    qmost_waveinfo **wv);

CPL_END_DECLS

#endif  /* QMOST_WAVEINFO_H */
