/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_diffimg.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_diffimg_test  Unit test of qmost_diffimg
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_diffimg
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_diffimg(void)
{
    int x, nx = 256;
    int y, ny = 256;

    cpl_image *img = NULL;
    cpl_image *test_img = NULL;
    float *buf;
    cpl_binary *bpm;

    cpl_image *diff_img = NULL;
    cpl_propertylist *qclist = NULL;

    cpl_error_code code;

    /* Make a test image */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    buf = cpl_image_get_data_float(img);
    cpl_test_nonnull(buf);

    bpm = cpl_mask_get_data(cpl_image_get_bpm(img));
    cpl_test_nonnull(bpm);

    if(buf != NULL && bpm != NULL) {
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                buf[y*nx+x] += 1024 + (x-nx/2) + (y-ny/2);
                bpm[y*nx+x] = (x+y) % 1000 == 0;
            }
        }

        /* And a larger block of badness so we can test the no valid
         * pixels in block case */
        for(y = 0; y < 16; y++) {
            for(x = 0; x < 16; x++) {
                bpm[y*nx+x] = 1;
            }
        }
    }

    qclist = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_diffimg(NULL, img, 256, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_diffimg(img, NULL, 256, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_diffimg(img, img, 256, 0, NULL, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_diffimg(img, img, 256, 0, &diff_img, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Bad ncells */
    code = qmost_diffimg(img, img, 0, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Different sizes */
    test_img = cpl_image_extract(img, 1, 1, nx, ny/2);

    code = qmost_diffimg(test_img, img, 256, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(test_img);
    test_img = NULL;

    /* This should work, the difference is zero */
    code = qmost_diffimg(img, img, 256, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(diff_img, 0.0, 0, 0.1);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MED"),
                 0.0, 0.1);

    cpl_image_delete(diff_img);
    diff_img = NULL;

    /* Now divide, which should give a ratio of 1 */
    code = qmost_diffimg(img, img, 256, 1, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(diff_img, 1.0, 0, 0.1);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MED"),
                 1.0, 0.1);

    cpl_image_delete(diff_img);
    diff_img = NULL;

    /* Now try something where there's actually a difference */
    test_img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_window(test_img, 1, 1, nx, ny, 1024.0);

    code = qmost_diffimg(img, test_img, 256, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MED"),
                 0.0, 5.0);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE BIN MIN"),
                 -240.0, 50.0);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE BIN MAX"),
                 240.0, 50.0);

    cpl_image_delete(test_img);
    test_img = NULL;

    cpl_image_delete(diff_img);
    diff_img = NULL;

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_image_delete(img);
    img = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_diffimg_fib
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_diffimg_fib(void)
{
    int x, nx = 256;
    int y, ny = 64;

    cpl_image *img = NULL;
    cpl_image *test_img = NULL;
    float *buf;
    cpl_binary *bpm;

    cpl_image *diff_img = NULL;
    cpl_propertylist *qclist = NULL;

    cpl_error_code code;

    /* Make a test image */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    buf = cpl_image_get_data_float(img);
    cpl_test_nonnull(buf);

    bpm = cpl_mask_get_data(cpl_image_get_bpm(img));
    cpl_test_nonnull(bpm);

    if(buf != NULL && bpm != NULL) {
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                buf[y*nx+x] += 1024 + (y-ny/2);
                bpm[y*nx+x] = (x+y) % 1000 == 0;
            }
        }

        /* And a bad fibre */
        y = 42;
        
        for(x = 0; x < nx; x++) {
            bpm[y*nx+x] = 1;
        }
    }

    qclist = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_diffimg_fib(NULL, img, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_diffimg_fib(img, NULL, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_diffimg_fib(img, img, 0, NULL, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_diffimg_fib(img, img, 0, &diff_img, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Different sizes */
    test_img = cpl_image_extract(img, 1, 1, nx, ny/2);

    code = qmost_diffimg_fib(test_img, img, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(test_img);
    test_img = NULL;

    /* This should work, the difference is zero */
    code = qmost_diffimg_fib(img, img, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(diff_img, 0.0, 0, 0.1);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MEAN"),
                 0.0, 0.1);

    cpl_image_delete(diff_img);
    diff_img = NULL;

    /* Now divide, which should give a ratio of 1 */
    code = qmost_diffimg_fib(img, img, 1, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(diff_img, 1.0, 0, 0.1);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MEAN"),
                 1.0, 0.1);

    cpl_image_delete(diff_img);
    diff_img = NULL;

    /* Now try something where there's actually a difference */
    test_img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_window(test_img, 1, 1, nx, ny, 1024.0);

    code = qmost_diffimg_fib(img, test_img, 0, &diff_img, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MEAN"),
                 0.0, 5.0);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MIN"),
                 -32.0, 5.0);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC COMPARE MAX"),
                 32.0, 5.0);

    cpl_image_delete(test_img);
    test_img = NULL;

    cpl_image_delete(diff_img);
    diff_img = NULL;

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_image_delete(img);
    img = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_diffimg module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_diffimg();
    test_qmost_diffimg_fib();

    return cpl_test_end(0);
}

/**@}*/
