/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_extract_tram.h"
#include "qmost_traceinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_extract_tram_test  Unit test of qmost_extract_tram
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief  Helper for unit test of qmost_extract_tram.
 *
 * @param  xcen         (Given)  Array of reference x positions for
 *                               simulated fibre traces.
 * @param  nobj         (Given)  Number of fibre images to inject
 *                               (length of xcen array).
 * @param  coef         (Given)  Polynomial coefficients for traces.
 *                               The first coefficient (zeroth degree)
 *                               must be zero.
 * @param  nord         (Given)  Degree of polynomial (array coef
 *                               should have nord+1 elements).
 * @param  replmissing  (Given)  Add blank entries to trace table for
 *                               missing fibre images and attempt to
 *                               recover them if this flag is set.
 *
 * @return void
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_extract_tram_internal (
    int *xcen,
    double *pkht,
    int *fiblive,
    int nobj,
    double *coef,
    int nord,
    int specbin)
{
    int nx = 1024;
    int ny = 2048;
    int yref = 1024;
    int xfwhm = 3;
    int width = 2*xfwhm;

    cpl_polynomial *thetrace = NULL;
    int i;

    double yrel, xpoly;

    cpl_image *img = NULL;
    cpl_image *var = NULL;
    cpl_image *bintmp;
    int nyb;

    float *img_data;
    int x, y, iobj;
    double dx, xw;

    cpl_propertylist *in_hdr = NULL;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;

    qmost_traceinfo tr;
    double xpdummy[1] = { 0 };
    double ypdummy[1] = { 0 };
    float fwhmdummy[1] = { 3 };
    float peakdummy[1] = { 10000 };
    float contrastdummy[1] = { 1.0 };
    cpl_size icoef;

    int xl, xh;

    cpl_error_code code;

    cpl_image *spec_img = NULL;
    cpl_image *this_spec_img = NULL;
    cpl_image *spec_var = NULL;
    cpl_image *this_spec_var = NULL;
    cpl_propertylist *spec_hdr = NULL;
    double gnorm;

    float expect_img, expect_img_tol;
    float expect_var, expect_var_tol;
    cpl_binary expect_bad;

    /* Set up polynomial */
    thetrace = cpl_polynomial_new(1);

    for(i = nord; i >= 0; i--) {
        icoef = i;
        cpl_polynomial_set_coeff(thetrace, &icoef, coef[i]);
    }

    /* Generate a test image with specified number of traces */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    var = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);

    cpl_image_fill_noise_uniform(img, -8, 8);
    cpl_image_fill_window(var, 1, 1, nx, ny, 4096.0/9.0);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        xw = 0.5*CPL_MATH_FWHM_SIG*CPL_MATH_FWHM_SIG / (xfwhm*xfwhm);

        for(y = 0; y < ny; y++) {
            yrel = (y+1 - yref) / ((float) yref);
            xpoly = cpl_polynomial_eval_1d(thetrace, yrel, NULL);

            for(iobj = 0; iobj < nobj; iobj++) {
                xl = xcen[iobj]+xpoly - 3*xfwhm;
                if(xl < 0)
                    xl = 0;
                if(xl >= nx)
                    xl = nx-1;
                
                xh = xcen[iobj]+xpoly + 3*xfwhm;
                if(xh < 0)
                    xh = 0;
                if(xh >= nx)
                    xh = nx-1;
                
                for(x = xl; x <= xh; x++) {
                    dx = (x+1) - (xcen[iobj]+xpoly);
                    img_data[y*nx+x] += pkht[iobj] * exp(-xw * dx*dx);
                }
            }
        }
    }

    in_hdr = cpl_propertylist_new();
    cpl_test_nonnull(in_hdr);

    /* Binning, if requested */
    if(specbin != 1) {
        bintmp = cpl_image_rebin(img, 1, 1, 1, specbin);
        cpl_image_delete(img);
        img = bintmp;

        bintmp = cpl_image_rebin(var, 1, 1, 1, specbin);
        cpl_image_delete(var);
        var = bintmp;
    }

    cpl_propertylist_update_int(in_hdr, "ESO DRS SPECBIN", specbin);
    cpl_propertylist_update_int(in_hdr, "ESO DRS SPATBIN", 1);

    nyb = ny / specbin;

    /* Set up trace table */
    code = qmost_trcreate(&trace_tbl, nord, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(trace_tbl);

    if(trace_tbl) {
        for(iobj = 0; iobj < nobj; iobj++) {
            memset(&tr, 0, sizeof(tr));
            tr.specnum = iobj+1;
            tr.live = fiblive[iobj];
            tr.yst = 1;
            tr.yfn = ny;
            tr.yref = yref;
            tr.trrms = 0;
            tr.nord = nord;

            icoef = 0;
            cpl_polynomial_set_coeff(thetrace, &icoef, xcen[iobj]+coef[0]);

            tr.coefs = thetrace;
            tr.npos = 1;
            tr.xpos = xpdummy;
            tr.ypos = ypdummy;
            tr.fwhm = fwhmdummy;
            tr.peak = peakdummy;
            tr.contrast = contrastdummy;

            qmost_trwrite1(trace_tbl, tr, iobj+1);
        }
    }

    trace_hdr = cpl_propertylist_new();
    cpl_test_nonnull(trace_hdr);
    cpl_propertylist_update_int(trace_hdr, "MINYST", 1);
    cpl_propertylist_update_int(trace_hdr, "MAXYFN", ny);

    spec_hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_extract_tram(NULL, var, in_hdr, trace_tbl, trace_hdr, width,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, NULL, in_hdr, trace_tbl, trace_hdr, width,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, var, NULL, trace_tbl, trace_hdr, width,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, var, in_hdr, NULL, trace_hdr, width,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, var, in_hdr, trace_tbl, NULL, width,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, var, in_hdr, trace_tbl, trace_hdr, width,
                              NULL, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, var, in_hdr, trace_tbl, trace_hdr, width,
                              &spec_img, NULL, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_extract_tram(img, var, in_hdr, trace_tbl, trace_hdr, width,
                              &spec_img, &spec_var, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid width */
    code = qmost_extract_tram(img, var, in_hdr, trace_tbl, trace_hdr, -1,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Valid inputs, runs an actual extraction */
    code = qmost_extract_tram(img, var, in_hdr, trace_tbl, trace_hdr, width,
                              &spec_img, &spec_var, spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Result should be reasonably close to pkht * sqrt(2 pi sigma^2) */
    gnorm = sqrt(2*CPL_MATH_PI) * xfwhm / CPL_MATH_FWHM_SIG;

    for(iobj = 0; iobj < nobj; iobj++) {
        this_spec_img = cpl_image_extract(spec_img, 1, iobj+1, nyb, iobj+1);
        cpl_test_nonnull(this_spec_img);

        this_spec_var = cpl_image_extract(spec_var, 1, iobj+1, nyb, iobj+1);
        cpl_test_nonnull(this_spec_var);

        if(fiblive[iobj]) {
            expect_img = specbin * 0.98*gnorm*pkht[iobj];
            expect_img_tol = specbin * 16*width;
            expect_var = specbin * width*4096.0/9.0;
            expect_var_tol = specbin * 25;
            expect_bad = 0;
        }
        else {
            expect_img = 0;
            expect_img_tol = FLT_EPSILON;
            expect_var = 0;
            expect_var_tol = FLT_EPSILON;
            expect_bad = 1;
        }

        /* Should get about 98% of the integral of the Gaussian 
           within noise margin */
        qmost_test_image_float_abs(this_spec_img, expect_img, expect_bad, expect_img_tol);

        /* Should get the noise we put in times sqrt number of pixels
           summed. */
        qmost_test_image_float_abs(this_spec_var, expect_var, expect_bad, expect_var_tol);

        cpl_image_delete(this_spec_img);
        this_spec_img = NULL;

        cpl_image_delete(this_spec_var);
        this_spec_var = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    /* Clean up */
    cpl_image_delete(img);
    img = NULL;

    cpl_image_delete(var);
    var = NULL;

    cpl_propertylist_delete(in_hdr);
    in_hdr = NULL;

    cpl_table_delete(trace_tbl);
    trace_tbl = NULL;

    cpl_propertylist_delete(trace_hdr);
    trace_hdr = NULL;

    cpl_polynomial_delete(thetrace);
    thetrace = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_extract_tram
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_extract_tram(void)
{
    int xcen1[1] = { 512 };
    double pkht1[1] = { 512 };
    double coef1[1] = { 0 };

    int xcen2[1] = { 512 };
    double pkht2[1] = { 512 };
    double coef2[3] = { 0, 10.0, -10.0 };

    int xcen3[2] = { 512, 640 };
    double pkht3[2] = { 512, 1024 };
    double coef3[1] = { 0 };

    int xcen4[2] = { 512, 640 };
    double pkht4[2] = { 512, 1024 };
    double coef4[3] = { 0, 10.0, -10.0 };

    int fiblive12[1] = { 1 };
    int fiblive34[2] = { 1, 1 };
    int fiblive5[2] = { 1, 0 };
    int fiblive6[2] = { 0, 1 };

    /* Normal test of extraction */
    test_qmost_extract_tram_internal(xcen1, pkht1, fiblive12, 1, coef1, 0, 1);
    test_qmost_extract_tram_internal(xcen2, pkht2, fiblive12, 1, coef2, 2, 1);
    test_qmost_extract_tram_internal(xcen3, pkht3, fiblive34, 2, coef3, 0, 1);
    test_qmost_extract_tram_internal(xcen4, pkht4, fiblive34, 2, coef4, 2, 1);

    /* Run a test to make sure fiblive is honoured */
    test_qmost_extract_tram_internal(xcen3, pkht3, fiblive5, 2, coef3, 0, 1);
    test_qmost_extract_tram_internal(xcen3, pkht3, fiblive6, 2, coef3, 0, 1);

    /* Binning */
    test_qmost_extract_tram_internal(xcen4, pkht4, fiblive34, 2, coef4, 2, 2);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_extract_qc
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_extract_qc(void)
{
    int nx = 2048;
    int ny = 512;

    cpl_image *img = NULL;
    cpl_image *var = NULL;

    float *img_data;
    int x, y;

    cpl_propertylist *qclist = NULL;
    cpl_error_code code;

    cpl_table *fibinfo_tbl = NULL;

    float noise;
    double value;

    /* Create */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    var = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);

    /* Generate a test image with known background level */
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        /* Ramp from 500 at top to 600 at bottom */
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                img_data[y*nx+x] += 500 + (100.0*y) / ny;
            }
        }
    }

    /* Noise */
    noise = 64.0 / 3.0;
    cpl_image_fill_window(var, 1, 1, nx, ny, noise*noise);

    /* Blank out a spectrum to test those branches */
    cpl_image_fill_window(var, 1, 42, nx, 42, 0);

    /* Property list for QC parameters */
    qclist = cpl_propertylist_new();

    /* The extracted spectrum input can be NULL and the routine should
     * be a no-op in this case. */
    code = qmost_extract_qc(NULL, var, qclist, NULL, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_extract_qc(img, NULL, qclist, NULL, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_extract_qc(img, var, NULL, NULL, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Valid inputs, should generate some QC */
    code = qmost_extract_qc(img, var, qclist, NULL, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT FLUX MED");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 550, 1);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT FLUX RMS");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, CPL_MATH_STD_MAD * 25, 1);  /* MAD should be 25 */

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT FLUX MIN");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 500, 10);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT FLUX MAX");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 600, 10);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT SN MED");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 550 / noise, 1 / noise);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT SN RMS");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, CPL_MATH_STD_MAD * 25 / noise, 1 / noise);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT SN MIN");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 500 / noise, 10 / noise);

    value = cpl_propertylist_get_double(qclist, "ESO QC EXT SN MAX");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 600 / noise, 10 / noise);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Test with FIBINFO output */
    qclist = cpl_propertylist_new();

    fibinfo_tbl = cpl_table_new(ny);

    /* Bad arm */
    code = qmost_extract_qc(img, var, qclist, fibinfo_tbl, -1);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Missing columns */
    code = qmost_extract_qc(img, var, qclist, fibinfo_tbl, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_table_new_column(fibinfo_tbl, "SNR_R", CPL_TYPE_DOUBLE);

    code = qmost_extract_qc(img, var, qclist, fibinfo_tbl, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Should work */
    cpl_table_new_column(fibinfo_tbl, "MEANFLUX_R", CPL_TYPE_DOUBLE);

    code = qmost_extract_qc(img, var, qclist, fibinfo_tbl, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_table_get(fibinfo_tbl, "SNR_R", 0, NULL),
                 500 / noise, 10 / noise);

    cpl_test_eq(cpl_table_is_valid(fibinfo_tbl, "SNR_R", 41), 0);

    cpl_test_abs(cpl_table_get(fibinfo_tbl, "SNR_R", ny-1, NULL),
                 600 / noise, 10 / noise);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    /* Clean up */
    cpl_image_delete(img);
    cpl_image_delete(var);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_extract_tram module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_extract_tram();
    test_qmost_extract_qc();

    return cpl_test_end(0);
}

/**@}*/
