/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_ffnorm_fib.h"
#include "qmost_traceinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_ffnorm_fib_test  Unit test of qmost_ffnorm_fib
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_ffnorm_fib
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_ffnorm_fib(void)
{
    cpl_image *in_spec_img = NULL;
    cpl_image *in_spec_var = NULL;

    cpl_image *spec_img = NULL;
    cpl_image *spec_var = NULL;
    cpl_propertylist *spec_hdr = NULL;
    cpl_table *fibinfo_tbl = NULL;

    cpl_image *dummy_var = NULL;
    cpl_image *check_img = NULL;
    cpl_image *this_spec = NULL;

    int badpix = 7;
    int badspec = 8;
    cpl_mask *badspec_msk = NULL;

    cpl_error_code code;

    int nwave = 512;
    int nspec = 16;

    float *buf = NULL;
    float *medlevel = NULL;

    int ispec, iwave;
    float dx, dy, baselevel;
    float value;

    /* Simulate a flat spectrum */
    in_spec_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(in_spec_img, -8, 8);

    buf = cpl_image_get_data_float(in_spec_img);

    medlevel = cpl_malloc(nspec * sizeof(float));

    for(ispec = 0; ispec < nspec; ispec++) {
        /* Put in a parabola but different level in each fibre */
        dy = (2*ispec - nspec) / ((float) nspec);
        baselevel = 1000 * (1.0 - 0.1 * dy*dy);

        for(iwave = 0; iwave < nwave; iwave++) {
            dx = (2*iwave - nwave) / ((float) nwave);
            buf[ispec*nwave+iwave] += baselevel * (1.0 - 0.1 * dx*dx);
        }

        /* Median of the quadratic is the same as dx=0.5,
           so baselevel * (1.0 - 0.1 * 0.5*0.5) */
        medlevel[ispec] = 0.975 * baselevel;
    }

    /* Variance */
    in_spec_var = cpl_image_add_scalar_create(in_spec_img, 64);
    cpl_image_threshold(in_spec_var, 0, FLT_MAX, 0, FLT_MAX);

    /* Add a bad pixel.  Also set in the BPM so we ignore during the
       compare. */
    cpl_image_set(in_spec_var, 42, badpix, 0);
    cpl_mask_set(cpl_image_get_bpm(in_spec_img), 42, badpix, 1);

    /* ...and a bad spectrum */
    cpl_image_fill_window(in_spec_var, 1, badspec, nwave, badspec, 0);

    badspec_msk = cpl_mask_new(nwave, 1);
    cpl_mask_not(badspec_msk);
    cpl_mask_copy(cpl_image_get_bpm(in_spec_img), badspec_msk, 1, badspec);
    cpl_mask_delete(badspec_msk);
    badspec_msk = NULL;

    /* Outputs */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);
    spec_hdr = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_ffnorm_fib(NULL, spec_var, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ffnorm_fib(spec_img, NULL, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Size mismatch test */
    dummy_var = cpl_image_extract(in_spec_var, 1, 1, 1, 1);

    code = qmost_ffnorm_fib(spec_img, dummy_var, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(dummy_var);
    dummy_var = NULL;

    /* All flagged bad */
    dummy_var = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_window(dummy_var, 1, 1, nwave, nspec, 0);
    
    code = qmost_ffnorm_fib(spec_img, dummy_var, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_image_delete(dummy_var);
    dummy_var = NULL;

    /* FIBINFO without FIB_ID */
    fibinfo_tbl = cpl_table_new(nspec);

    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    /* FIBINFO with FIB_ID, FIB_ST, FIB_USE but wrong length */
    fibinfo_tbl = cpl_table_new(1);
    cpl_table_new_column(fibinfo_tbl, "FIB_ID", CPL_TYPE_INT);
    cpl_table_new_column(fibinfo_tbl, "FIB_ST", CPL_TYPE_INT);
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);

    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    /* Make proper input */
    fibinfo_tbl = cpl_table_new(nspec);
    cpl_table_new_column(fibinfo_tbl, "FIB_ID", CPL_TYPE_INT);
    cpl_table_new_column(fibinfo_tbl, "FIB_ST", CPL_TYPE_INT);
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);

    for(ispec = 0; ispec < nspec; ispec++) {
        /* Might as well use the bad one to test NULL handling */
        if(ispec+1 == badspec)
            continue;

        cpl_table_set_int(fibinfo_tbl, "FIB_ID", ispec, ispec+1);
        cpl_table_set_int(fibinfo_tbl, "FIB_ST", ispec, 2);
        cpl_table_set_int(fibinfo_tbl, "FIB_USE", ispec, 1);
    }

    /* Bad arm */
    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            -1, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* This should work */
    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, fibinfo_tbl,
                            0, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Fibres should be individually normalised but spectral variation
       should still be there.  Check by taking ratio. */
    check_img = cpl_image_divide_create(in_spec_img, spec_img);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        this_spec = cpl_image_extract(check_img, 1, ispec+1, nwave, ispec+1);

        qmost_test_image_float_abs(this_spec, medlevel[ispec], 0, 5);

        cpl_image_delete(this_spec);
        this_spec = NULL;
    }

    /* Also check reported levels in FIBINFO */
    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        value = cpl_table_get_float(fibinfo_tbl, "MED1_R", ispec, NULL);
        cpl_test_abs(value, medlevel[ispec], 5);

        value = cpl_table_get_float(fibinfo_tbl, "MED2_R", ispec, NULL);
        cpl_test_abs(value, medlevel[ispec], 5);

        value = cpl_table_get_float(fibinfo_tbl, "MED3_R", ispec, NULL);
        cpl_test_abs(value, 1.0, 0.1);

        value = cpl_table_get_float(fibinfo_tbl, "NORMLEVEL_R", ispec, NULL);
        cpl_test_abs(value, 1.0/medlevel[ispec], 1e-4);
    }
        
    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    cpl_image_delete(check_img);
    check_img = NULL;

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    /* Now test with resptrack=1, everything should be normalised to 1. */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);
    spec_hdr = cpl_propertylist_new();

    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, NULL,
                            1, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        this_spec = cpl_image_extract(spec_img, 1, ispec+1, nwave, ispec+1);

        qmost_test_image_float_abs(this_spec, 1, 0, 0.1);

        cpl_image_delete(this_spec);
        this_spec = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    /* Add a some pixels that are bad in all the spectra.  We do the
       first and last to trigger those branches and one in middle. */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);
    spec_hdr = cpl_propertylist_new();

    for(ispec = 0; ispec < nspec; ispec++) {
        cpl_image_set(spec_var, 1, ispec+1, 0);
        cpl_mask_set(cpl_image_get_bpm(spec_img), 1, ispec+1, 1);

        cpl_image_set(spec_var, 64, ispec+1, 0);
        cpl_mask_set(cpl_image_get_bpm(spec_img), 64, ispec+1, 1);

        cpl_image_set(spec_var, nwave, ispec+1, 0);
        cpl_mask_set(cpl_image_get_bpm(spec_img), nwave, ispec+1, 1);
    }

     code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, NULL,
                            1, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        this_spec = cpl_image_extract(spec_img, 1, ispec+1, nwave, ispec+1);

        qmost_test_image_float_abs(this_spec, 1, 0, 0.1);

        cpl_image_delete(this_spec);
        this_spec = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    /* Also with smoothing, should just lower noise */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);
    spec_hdr = cpl_propertylist_new();

    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, NULL,
                            1, 3, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(spec_img, 1, 0, 0.1);

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    /* Now test with rescale=1, fibres should be normalised to 1
       globally so their ratios should be their relative throughputs. */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);
    spec_hdr = cpl_propertylist_new();

    code = qmost_ffnorm_fib(spec_img, spec_var, spec_hdr,
                            QMOST_ARM_RED, NULL,
                            1, 0, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        this_spec = cpl_image_extract(spec_img, 1, ispec+1, nwave, ispec+1);

        qmost_test_image_float_abs(this_spec, medlevel[ispec]/975, 0, 5);

        cpl_image_delete(this_spec);
        this_spec = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    cpl_image_delete(in_spec_img);
    cpl_image_delete(in_spec_var);
    cpl_free(medlevel);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_ffnorm_fib
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_obffcor_fib(void)
{
    cpl_image *in_spec_img = NULL;
    cpl_image *in_spec_var = NULL;
    cpl_table *trace_tbl = NULL;
    cpl_table *fibinfo_tbl = NULL;

    int nwave = 512;
    int nspec = 16;
    int badspec = 8;

    int ispec;
    float level;
    qmost_traceinfo *tr = NULL;

    cpl_image *spec_img = NULL;
    cpl_image *spec_var = NULL;
    cpl_image *dummy_var = NULL;
    cpl_table *dummy_tbl = NULL;
    cpl_image *this_img = NULL;

    cpl_error_code code;

    /* Set up a ramp such that the correct result is to flatten it */
    in_spec_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);

    in_spec_var = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_window(in_spec_var, 1, 1, nwave, nspec, 0.1);

    fibinfo_tbl = cpl_table_new(nspec);
    cpl_table_new_column(fibinfo_tbl, "FIB_ST", CPL_TYPE_INT);
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);
    cpl_table_new_column(fibinfo_tbl, "MED3_R", CPL_TYPE_FLOAT);

    for(ispec = 0; ispec < nspec; ispec++) {
        level = 0.5 + ispec / ((float) nspec);

        cpl_image_fill_window(in_spec_img, 1, ispec+1, nwave, ispec+1,
                              level);

        cpl_table_set_int(fibinfo_tbl, "FIB_ST", ispec, 2);
        cpl_table_set_int(fibinfo_tbl, "FIB_USE", ispec, 1);
        cpl_table_set_float(fibinfo_tbl, "MED3_R", ispec, level);
    }

    /* Add a bad pixel so we trigger those branches */
    cpl_image_set(in_spec_var, 42, 7, 0);

    /* This is needed so we can compare excluding the bad pixel */
    cpl_mask_set(cpl_image_get_bpm(in_spec_img), 42, 7, 1);

    /* Trace table */
    qmost_trcreate(&trace_tbl, 0, 2);

    for(ispec = 0; ispec < nspec; ispec++) {
        tr = cpl_calloc(1, sizeof(qmost_traceinfo));

        tr->live = (ispec+1 == badspec ? 0 : 1);
        tr->coefs = cpl_polynomial_new(1);
        tr->npos = 2;
        tr->xpos = cpl_calloc(2, sizeof(double));
        tr->ypos = cpl_calloc(2, sizeof(double));
        tr->fwhm = cpl_calloc(2, sizeof(float));
        tr->peak = cpl_calloc(2, sizeof(float));
        tr->contrast = cpl_calloc(2, sizeof(float));

        qmost_trwrite1(trace_tbl, *tr, ispec+1);

        qmost_trclose(1, &tr);
    }

    /* NULL input tests */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);

    code = qmost_obffcor_fib(NULL, spec_var,
                             trace_tbl,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_obffcor_fib(spec_img, NULL,
                             trace_tbl,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_obffcor_fib(spec_img, spec_var,
                             NULL,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_obffcor_fib(spec_img, spec_var,
                             trace_tbl,
                             QMOST_ARM_RED,
                             NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Size mismatch of error array */
    dummy_var = cpl_image_new(1, 1, CPL_TYPE_FLOAT);

    code = qmost_obffcor_fib(spec_img, dummy_var,
                             trace_tbl,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(dummy_var);
    dummy_var = NULL;

    /* Size mismatch of FIBINFO */
    dummy_tbl = cpl_table_new(1);

    code = qmost_obffcor_fib(spec_img, spec_var,
                             trace_tbl,
                             QMOST_ARM_RED,
                             dummy_tbl);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    /* Invalid trace table */
    code = qmost_obffcor_fib(spec_img, spec_var,
                             dummy_tbl,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_table_delete(dummy_tbl);
    dummy_tbl = NULL;

    /* Invalid arm */
    code = qmost_obffcor_fib(spec_img, spec_var,
                             trace_tbl,
                             -1,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Invalid FIBINFO but correct size */
    dummy_tbl = cpl_table_new(nspec);

    code = qmost_obffcor_fib(spec_img, spec_var,
                             trace_tbl,
                             QMOST_ARM_RED,
                             dummy_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_table_delete(dummy_tbl);
    dummy_tbl = NULL;

    /* Valid but too short trace table */
    dummy_tbl = cpl_table_extract(trace_tbl, 0, 1);

    code = qmost_obffcor_fib(spec_img, spec_var,
                             dummy_tbl,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_table_delete(dummy_tbl);
    dummy_tbl = NULL;

    /* This should work */
    code = qmost_obffcor_fib(spec_img, spec_var,
                             trace_tbl,
                             QMOST_ARM_RED,
                             fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        this_img = cpl_image_extract(spec_img, 1, ispec+1, nwave, ispec+1);

        qmost_test_image_float_abs(this_img, 1.0, 0, 0.1);

        cpl_image_delete(this_img);
        this_img = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_image_delete(in_spec_img);
    cpl_image_delete(in_spec_var);
    cpl_table_delete(fibinfo_tbl);
    cpl_table_delete(trace_tbl);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_ffdiv_fib
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_ffdiv_fib(void)
{
    cpl_image *in_spec_img = NULL;
    cpl_image *in_spec_var = NULL;
    cpl_table *trace_tbl = NULL;

    int nwave = 512;
    int nspec = 16;
    int badspec = 8;

    int ispec;
    float level;
    qmost_traceinfo *tr = NULL;

    cpl_image *spec_img = NULL;
    cpl_image *spec_var = NULL;
    cpl_image *ffnorm_img = NULL;
    cpl_image *ffnorm_var = NULL;

    cpl_propertylist *spec_hdr = NULL;
    cpl_image *dummy_img = NULL;
    cpl_image *dummy_var = NULL;
    cpl_table *dummy_tbl = NULL;
    cpl_image *this_img = NULL;

    int nwbinned;

    cpl_error_code code;

    /* Set up a ramp such that the correct result is to flatten it */
    in_spec_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);

    in_spec_var = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_window(in_spec_var, 1, 1, nwave, nspec, 0.1);

    for(ispec = 0; ispec < nspec; ispec++) {
        level = 0.5 + ispec / ((float) nspec);

        cpl_image_fill_window(in_spec_img, 1, ispec+1, nwave, ispec+1,
                              level);
    }

    /* Simple way to make normalised flat */
    ffnorm_img = cpl_image_divide_scalar_create(in_spec_img, 1.0);
    ffnorm_var = cpl_image_divide_scalar_create(in_spec_var, 1.0);

    /* Add bad pixel in image, so we trigger those branches */
    cpl_image_set(in_spec_var, 42, 7, 0);

    /* This is needed so we can compare excluding the bad pixels */
    cpl_mask_set(cpl_image_get_bpm(in_spec_img), 42, 7, 1);

    /* Add zero pixel in flat */
    cpl_image_set(ffnorm_img, 42, 6, 0);

    /* Also set it bad for compare */
    cpl_mask_set(cpl_image_get_bpm(in_spec_img), 42, 6, 1);

    /* Trace table */
    qmost_trcreate(&trace_tbl, 0, 2);

    for(ispec = 0; ispec < nspec; ispec++) {
        tr = cpl_calloc(1, sizeof(qmost_traceinfo));

        tr->live = (ispec+1 == badspec ? 0 : 1);
        tr->coefs = cpl_polynomial_new(1);
        tr->npos = 2;
        tr->xpos = cpl_calloc(2, sizeof(double));
        tr->ypos = cpl_calloc(2, sizeof(double));
        tr->fwhm = cpl_calloc(2, sizeof(float));
        tr->peak = cpl_calloc(2, sizeof(float));
        tr->contrast = cpl_calloc(2, sizeof(float));

        qmost_trwrite1(trace_tbl, *tr, ispec+1);

        qmost_trclose(1, &tr);
    }

    /* Empty header */
    spec_hdr = cpl_propertylist_new();

    /* NULL input tests */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);

    code = qmost_ffdiv_fib(NULL, spec_var, spec_hdr,
                           trace_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ffdiv_fib(spec_img, NULL, spec_hdr,
                           trace_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ffdiv_fib(spec_img, spec_var, NULL,
                           trace_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           NULL, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           trace_tbl, NULL, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           trace_tbl, ffnorm_img, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Size mismatch of error array */
    dummy_var = cpl_image_new(1, 1, CPL_TYPE_FLOAT);

    code = qmost_ffdiv_fib(spec_img, dummy_var, spec_hdr,
                           trace_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(dummy_var);
    dummy_var = NULL;

    /* Size mismatch of flat array */
    dummy_img = cpl_image_new(1, 1, CPL_TYPE_FLOAT);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           trace_tbl, dummy_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(dummy_img);
    dummy_img = NULL;

    /* Size mismatch of flat error array */
    dummy_var = cpl_image_new(1, 1, CPL_TYPE_FLOAT);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           trace_tbl, ffnorm_img, dummy_var);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_image_delete(dummy_var);
    dummy_var = NULL;

    /* Invalid trace table */
    dummy_tbl = cpl_table_new(1);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           dummy_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_table_delete(dummy_tbl);
    dummy_tbl = NULL;

    /* Valid but too short trace table */
    dummy_tbl = cpl_table_extract(trace_tbl, 0, 1);

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           dummy_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_table_delete(dummy_tbl);
    dummy_tbl = NULL;

    /* This should work */
    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           trace_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        this_img = cpl_image_extract(spec_img, 1, ispec+1, nwave, ispec+1);

        qmost_test_image_float_abs(this_img, 1.0, 0, 0.1);

        cpl_image_delete(this_img);
        this_img = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    /* Test of spectral binning */
    spec_img = cpl_image_rebin(in_spec_img, 1, 1, 2, 1);
    spec_var = cpl_image_rebin(in_spec_var, 1, 1, 2, 1);

    cpl_propertylist_update_int(spec_hdr, "ESO DRS SPECBIN", 2);
    cpl_propertylist_update_int(spec_hdr, "ESO DRS SPATBIN", 1);

    nwbinned = nwave / 2;

    code = qmost_ffdiv_fib(spec_img, spec_var, spec_hdr,
                           trace_tbl, ffnorm_img, ffnorm_var);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec+1 == badspec)
            continue;

        /* Binned image so use nwbinned and all values are doubled */
        this_img = cpl_image_extract(spec_img, 1, ispec+1, nwbinned, ispec+1);

        qmost_test_image_float_abs(this_img, 2.0, 0, 0.1);

        cpl_image_delete(this_img);
        this_img = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    cpl_image_delete(in_spec_img);
    cpl_image_delete(in_spec_var);
    cpl_image_delete(ffnorm_img);
    cpl_image_delete(ffnorm_var);
    cpl_table_delete(trace_tbl);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_ffnorm_fib module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_ffnorm_fib();
    test_qmost_obffcor_fib();
    test_qmost_ffdiv_fib();

    return cpl_test_end(0);
}

/**@}*/
