/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#define _POSIX_C_SOURCE   200809L   /* For mkstemp()       */
#define _DARWIN_C_SOURCE

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include "qmost_constants.h"
#include "qmost_fibtab.h"
#include "qmost_pfits.h"
#include "qmost_testutil.h"
#include "qmost_waveinfo.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_fibtab_test  Unit test of qmost_fibtab
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fibtabload
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fibtabload(void)
{
    cpl_frame *frame = NULL;
    char tmpfile[] = "test-qmost_fibtab_XXXXXX";
    int fd;

    cpl_table *outtab = NULL;
    cpl_propertylist *outhdr = NULL;

    cpl_table *table = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_error_code code;

    int i;

    /* Create temporary file in the wrong format */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Make an empty frame */
    frame = cpl_frame_new();

    /* NULL input */
    code = qmost_fibtabload(NULL, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtabload(frame, 0, NULL, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtabload(frame, 0, &table, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename */
    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Not a FITS file */
    cpl_frame_set_filename(frame, tmpfile);
    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* Create dummy PHDU so file is now valid FITS */
    outhdr = cpl_propertylist_new();

    code = cpl_propertylist_save(outhdr, tmpfile, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    /* No FIBINFO extension.  This is not an error and should just
     * NULL the pointers out. */
    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_null(table);
    cpl_test_null(hdr);

    /* Make a FIBINFO extension with a dummy IMAGE HDU in it so the
       table read fails instead. */
    outhdr = cpl_propertylist_new();

    cpl_propertylist_update_string(outhdr,
                                   "EXTNAME",
                                   QMOST_FIBINFO_EXTNAME);

    cpl_propertylist_save(outhdr, tmpfile, CPL_IO_EXTEND);

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Start over with a fresh file, this time with a valid table HDU
       in the extension as it should be, but some of the columns we
       need are missing. */
    outtab = cpl_table_new(1);
    cpl_table_new_column(outtab, "FIB_ST", CPL_TYPE_INT);
    cpl_table_set_int(outtab, "FIB_ST", 0, 0);
    outhdr = cpl_propertylist_new();

    cpl_propertylist_update_string(outhdr,
                                   "EXTNAME",
                                   QMOST_FIBINFO_EXTNAME);

    cpl_table_save(outtab, NULL, outhdr, tmpfile, CPL_IO_CREATE);

    cpl_table_delete(outtab);
    outtab = NULL;
    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    /* Reading it without shuffle should now work */
    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_int(table, "FIB_ST", 0, NULL), 0);
    cpl_test_eq_string(cpl_propertylist_get_string(hdr, "EXTNAME"),
                       QMOST_FIBINFO_EXTNAME);

    cpl_table_delete(table);
    table = NULL;
    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Reading it with shuffle shouldn't */
    code = qmost_fibtabload(frame, 1, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Do it again with a valid table for shuffle */
    outtab = cpl_table_new(5);
    cpl_table_new_column(outtab, "FIB_ID", CPL_TYPE_INT);
    cpl_table_new_column(outtab, "FIB_ROOT", CPL_TYPE_INT);
    cpl_table_new_column(outtab, "FIB_ST", CPL_TYPE_INT);
    cpl_table_new_column(outtab, "SLIT_POS", CPL_TYPE_INT);
    cpl_table_new_column(outtab, "FIB_USE", CPL_TYPE_INT);

    /* Fibre 1, wrong root (should get filtered) */
    cpl_table_set_int(outtab, "FIB_ID", 0, 1);
    cpl_table_set_int(outtab, "FIB_ROOT", 0, 0);
    cpl_table_set_int(outtab, "FIB_ST", 0, 1);
    cpl_table_set_int(outtab, "SLIT_POS", 0, 10);
    cpl_table_set_int(outtab, "FIB_USE", 0, 1);

    /* Fibre 2, wrong root (should get filtered) */
    cpl_table_set_int(outtab, "FIB_ID", 1, 2);
    cpl_table_set_int(outtab, "FIB_ROOT", 1, 2);
    cpl_table_set_int(outtab, "FIB_ST", 1, 2);
    cpl_table_set_int(outtab, "SLIT_POS", 1, 9);
    cpl_table_set_int(outtab, "FIB_USE", 1, 1);

    /* Fibres 3-5, okay to test sort */
    cpl_table_set_int(outtab, "FIB_ID", 2, 3);
    cpl_table_set_int(outtab, "FIB_ROOT", 2, 1);
    cpl_table_set_int(outtab, "FIB_ST", 2, 2);
    cpl_table_set_int(outtab, "SLIT_POS", 2, 8);
    cpl_table_set_int(outtab, "FIB_USE", 2, 1);

    cpl_table_set_int(outtab, "FIB_ID", 3, 4);
    cpl_table_set_int(outtab, "FIB_ROOT", 3, 1);
    cpl_table_set_int(outtab, "FIB_ST", 3, 1);
    cpl_table_set_int(outtab, "SLIT_POS", 3, 7);
    cpl_table_set_int(outtab, "FIB_USE", 3, 1);

    cpl_table_set_int(outtab, "FIB_ID", 4, 5);
    cpl_table_set_int(outtab, "FIB_ROOT", 4, 1);
    cpl_table_set_int(outtab, "FIB_ST", 4, 0);
    cpl_table_set_int(outtab, "SLIT_POS", 4, 6);
    cpl_table_set_int(outtab, "FIB_USE", 4, 1);

    outhdr = cpl_propertylist_new();

    cpl_propertylist_update_string(outhdr,
                                   "EXTNAME",
                                   QMOST_FIBINFO_EXTNAME);

    cpl_table_save(outtab, NULL, outhdr, tmpfile, CPL_IO_CREATE);

    cpl_table_delete(outtab);
    outtab = NULL;
    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    /* Load that without shuffle first */
    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_nrow(table), 5);

    cpl_table_delete(table);
    table = NULL;
    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Now load it with shuffle, we should get the 3 selected rows in
       the correct order plus the 5 simucal entries at each end.  */
    code = qmost_fibtabload(frame, 1, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_nrow(table), 13);

    /* Check simucal */
    for(i = 0; i < 5; i++) {
        cpl_test_eq(cpl_table_get_int(table, "FIB_USE", i, NULL), 0);
        cpl_test_eq(cpl_table_get_int(table, "FIB_ROOT", i, NULL), 1);
        cpl_test_eq(cpl_table_get_int(table, "SLIT_POS", i, NULL), i+1);
    }

    /* Check real fibres */
    cpl_test_eq(cpl_table_get_int(table, "SLIT_POS", 5, NULL), 6);
    cpl_test_eq(cpl_table_get_int(table, "SLIT_POS", 6, NULL), 7);
    cpl_test_eq(cpl_table_get_int(table, "SLIT_POS", 7, NULL), 8);

    /* Check simucal */
    for(i = 8; i < 13; i++) {
        cpl_test_eq(cpl_table_get_int(table, "FIB_USE", i, NULL), 0);
        cpl_test_eq(cpl_table_get_int(table, "FIB_ROOT", i, NULL), 1);
        cpl_test_eq(cpl_table_get_int(table, "SLIT_POS", i, NULL), i+1);
    }

    cpl_table_delete(table);
    table = NULL;
    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Add METROLOGY */
    outtab = cpl_table_new(5);
    cpl_table_new_column(outtab, "FIB_ID", CPL_TYPE_INT);
    cpl_table_new_column(outtab, "TEST", CPL_TYPE_INT);
    cpl_table_set_column_unit(outtab, "TEST", "UNIT");

    cpl_table_set_int(outtab, "FIB_ID", 2, 3);
    cpl_table_set_int(outtab, "TEST", 2, 42);

    cpl_table_set_int(outtab, "FIB_ID", 1, 99);
    cpl_table_set_int(outtab, "TEST", 1, 64);

    outhdr = cpl_propertylist_new();

    cpl_propertylist_update_string(outhdr,
                                   "EXTNAME",
                                   QMOST_METROLOGY_EXTNAME);

    cpl_table_save(outtab, NULL, outhdr, tmpfile, CPL_IO_EXTEND);

    cpl_table_delete(outtab);
    outtab = NULL;
    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    code = qmost_fibtabload(frame, 1, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_nrow(table), 13);

    cpl_test_eq(cpl_table_get_int(table, "TEST", 7, NULL), 42);

    cpl_table_delete(table);
    table = NULL;
    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Clean up */
    cpl_frame_delete(frame);
    frame = NULL;

    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fibtabsave
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fibtabsave(void)
{
    cpl_frame *frame = NULL;
    char tmpfile[] = "test-qmost_fibtab_XXXXXX";
    int fd;

    cpl_table *outtab = NULL;
    cpl_propertylist *outhdr = NULL;

    cpl_table *table = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_error_code code;

    /* Create temporary file */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Make an empty frame */
    frame = cpl_frame_new();

    /* Make a simple table to write */
    outtab = cpl_table_new(1);
    cpl_table_new_column(outtab, "FIB_ST", CPL_TYPE_INT);
    cpl_table_set_int(outtab, "FIB_ST", 0, 0);

    /* NULL input */
    code = qmost_fibtabsave(NULL, NULL, frame);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtabsave(outtab, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename */
    code = qmost_fibtabsave(outtab, NULL, frame);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid input, but file not initialized */
    cpl_frame_set_filename(frame, tmpfile);

    code = qmost_fibtabsave(outtab, NULL, frame);
    cpl_test_eq_error(code, CPL_ERROR_FILE_NOT_CREATED);

    /* Initialize the file */
    outhdr = cpl_propertylist_new();

    code = cpl_propertylist_save(outhdr, tmpfile, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    /* It should work now we have a PHDU */
    code = qmost_fibtabsave(outtab, NULL, frame);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Read it back */
    code = qmost_fibtabload(frame, 0, &table, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_int(table, "FIB_ST", 0, NULL), 0);
    cpl_test_eq_string(cpl_propertylist_get_string(hdr, "EXTNAME"),
                       QMOST_FIBINFO_EXTNAME);

    cpl_table_delete(table);
    table = NULL;
    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Write another with additional headers.  This should work but
       appends another HDU with the same EXTNAME. */
    outhdr = cpl_propertylist_new();

    code = qmost_fibtabsave(outtab, outhdr, frame);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_table_delete(outtab);
    outtab = NULL;
    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    /* Clean up */
    cpl_frame_delete(frame);
    frame = NULL;

    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fibtab_dummy
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fibtab_dummy(void)
{
    cpl_table *tbl = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_error_code code;

    /* NULL inputs */
    code = qmost_fibtab_dummy(5, NULL, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtab_dummy(5, &tbl, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* This should work */
    code = qmost_fibtab_dummy(5, &tbl, &hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_nonnull(tbl);
    cpl_test_nonnull(hdr);

    cpl_test_eq(cpl_table_get_nrow(tbl), 5);
    cpl_test(cpl_table_has_column(tbl, "FIB_ID"));

    cpl_table_delete(tbl);
    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fibtab_newcols
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fibtab_newcols(void)
{
    cpl_table *in_tbl = NULL;
    cpl_table *out_tbl = NULL;
    cpl_propertylist *pri_hdr = NULL;
    cpl_error_code code;
    int isnull;

    double exptime = 1.0;
    double mjdmid = 59000.0;

    in_tbl = cpl_table_new(5);
    pri_hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_fibtab_newcols(NULL, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtab_newcols(in_tbl, NULL, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty inputs */
    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Start adding headers we need one by one */
    cpl_propertylist_update_double(pri_hdr, "ESO TEL GEOLAT", -30.0);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_update_double(pri_hdr, "ESO TEL GEOLON", -70.0);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* This is the last one.  It should now run up to the point of
     * trying to retrieve the FIB_USE column so this is where the
     * error comes from. */
    cpl_propertylist_update_double(pri_hdr, "ESO TEL GEOELEV", 2500.0);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add FIB_USE */
    cpl_table_new_column(in_tbl, "FIB_USE", CPL_TYPE_INT);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add OBJ_RA */
    cpl_table_new_column(in_tbl, "OBJ_RA", CPL_TYPE_DOUBLE);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add OBJ_DEC, this should now be enough to make it all the way
     * but it won't do anything due to all the values being NULL. */
    cpl_table_new_column(in_tbl, "OBJ_DEC", CPL_TYPE_DOUBLE);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(out_tbl);

    cpl_test_eq(cpl_table_get_int(out_tbl, "NSPEC", 2, NULL), 3);
    cpl_test_abs(cpl_table_get_float(out_tbl, "EXPTIME", 1, NULL), exptime, DBL_EPSILON);

    cpl_table_get_float(out_tbl, "HELIO_COR", 0, &isnull);
    cpl_test(isnull > 0);

    cpl_table_delete(out_tbl);
    out_tbl = NULL;

    /* We can now test the "badcoords" case by adding PHDU RA/DEC.
     * First, something it doesn't understand in each position, then
     * an actual set of coordinates. */
    cpl_propertylist_update_string(pri_hdr, "RA", "BLIMEY");
    cpl_propertylist_update_string(pri_hdr, "DEC", "CRIKEY");

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_erase(pri_hdr, "RA");
    cpl_propertylist_update_double(pri_hdr, "RA", 148.888);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_erase(pri_hdr, "DEC");
    cpl_propertylist_update_double(pri_hdr, "DEC", 69.065);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* At this point we should have a populated output where all of
     * the corrections are the same (the ones for the pointing
     * centre).  This value of the correction was calculated using my
     * "bcv" routine so should differ slightly due to inclusion of
     * relativistic effects at the < 10 m/s level.  We therefore use
     * a comparison threshold of 50 m/s. */
    cpl_test_abs(cpl_table_get_float(out_tbl, "HELIO_COR", 0, NULL),
                 -14.100207,
                 0.05);

    cpl_table_delete(out_tbl);
    out_tbl = NULL;

    /* Now test with populated coordinates, this object should be
     * close enough to not get replaced. */
    cpl_table_set_double(in_tbl, "OBJ_RA", 0, 148.968);
    cpl_table_set_double(in_tbl, "OBJ_DEC", 0, 69.680);

    /* This one is too far so should get replaced */
    cpl_table_set_double(in_tbl, "OBJ_RA", 1, 140.0);
    cpl_table_set_double(in_tbl, "OBJ_DEC", 1, 60.0);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_table_get_float(out_tbl, "HELIO_COR", 0, NULL),
                 -13.830897,
                 0.05);

    cpl_test_abs(cpl_table_get_float(out_tbl, "HELIO_COR", 1, NULL),
                 -14.100207,
                 0.05);

    cpl_table_delete(out_tbl);
    out_tbl = NULL;

    /* Check simucals get HELIO_COR = 0 */
    cpl_table_set_int(in_tbl, "FIB_USE", 0, 0);

    code = qmost_fibtab_newcols(in_tbl, pri_hdr, exptime, mjdmid, &out_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_table_get_float(out_tbl, "HELIO_COR", 0, NULL),
                 0,
                 0.05);

    cpl_test_abs(cpl_table_get_float(out_tbl, "HELIO_COR", 1, NULL),
                 -14.100207,
                 0.05);

    cpl_table_delete(out_tbl);
    out_tbl = NULL;

    cpl_table_delete(in_tbl);
    in_tbl = NULL;

    cpl_propertylist_delete(pri_hdr);
    pri_hdr = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fibtab_arcqc
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fibtab_arcqc(void)
{
    cpl_table *fibinfo_tbl = NULL;

    cpl_table *master_wave_tbl = NULL;
    qmost_waveinfo wv;
    cpl_size zero;

    cpl_table *ob_wave_tbl = NULL;
    cpl_error_code code;

    fibinfo_tbl = cpl_table_new(5);
    master_wave_tbl = cpl_table_new(5);
    ob_wave_tbl = cpl_table_new(5);

    /* NULL inputs */
    code = qmost_fibtab_arcqc(NULL, QMOST_ARM_RED, master_wave_tbl, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fibtab_arcqc(fibinfo_tbl, QMOST_ARM_RED, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Bad arm */
    code = qmost_fibtab_arcqc(fibinfo_tbl, -1, master_wave_tbl, NULL);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Bad master waveinfo table */
    code = qmost_fibtab_arcqc(fibinfo_tbl,
                              QMOST_ARM_RED,
                              master_wave_tbl,
                              NULL);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_table_delete(master_wave_tbl);
    master_wave_tbl = NULL;

    /* Now create waveinfo so that succeeds.  It can be shorter than
     * the fibinfo, any extra rows just don't get populated.  Here we
     * put in two rows, first a valid one and then one with no lines
     * to test those branches.*/
    code = qmost_wvcreate(&master_wave_tbl, 1, 2);

    wv.specnum = 1;
    wv.live = 1;
    wv.nord = 0;
    wv.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(wv.coefs, &zero, 0);
    wv.xref = 0;
    wv.ngood = 1;
    wv.medresid = 0;
    wv.fit_rms = 1.0e-3;
    wv.wave1 = 6000;
    wv.waven = 7000;
    wv.dwave = 1;
    wv.nlines = 1;
    wv.xpos = cpl_calloc(1, sizeof(double));
    wv.fwhm = cpl_calloc(1, sizeof(float));
    wv.wave_calc = cpl_calloc(1, sizeof(double));
    wv.wave_true = cpl_calloc(1, sizeof(double));
    wv.fit_flag = cpl_calloc(1, sizeof(unsigned char));
    wv.wave_cor = cpl_calloc(1, sizeof(double));
    wv.peak = cpl_calloc(1, sizeof(float));
    wv.contrast = cpl_calloc(1, sizeof(float));

    code = qmost_wvwrite1(master_wave_tbl, 1, wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    wv.specnum = 2;
    wv.nlines = 0;

    code = qmost_wvwrite1(master_wave_tbl, 2, wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_polynomial_delete(wv.coefs);
    cpl_free(wv.xpos);
    cpl_free(wv.fwhm);
    cpl_free(wv.wave_calc);
    cpl_free(wv.wave_true);
    cpl_free(wv.fit_flag);
    cpl_free(wv.wave_cor);
    cpl_free(wv.peak);
    cpl_free(wv.contrast);

    /* We should now get to the end and populate some stuff */
    code = qmost_fibtab_arcqc(fibinfo_tbl,
                              QMOST_ARM_RED,
                              master_wave_tbl,
                              NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_int(fibinfo_tbl, "NLIN_ARC_R", 0, NULL), 1);
    cpl_test_abs(cpl_table_get(fibinfo_tbl, "RMS_ARC_R", 0, NULL),
                 1.0e-3, FLT_EPSILON);

    /* Test with bad OB waveinfo table */
    code = qmost_fibtab_arcqc(fibinfo_tbl,
                              QMOST_ARM_RED,
                              master_wave_tbl,
                              ob_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Test with mismatched OB waveinfo table */
    cpl_table_delete(ob_wave_tbl);
    ob_wave_tbl = NULL;

    ob_wave_tbl = cpl_table_extract(master_wave_tbl, 0, 1);

    code = qmost_fibtab_arcqc(fibinfo_tbl,
                              QMOST_ARM_RED,
                              master_wave_tbl,
                              ob_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    /* Now with a properly matched OB waveinfo table */
    cpl_table_delete(ob_wave_tbl);
    ob_wave_tbl = NULL;

    ob_wave_tbl = cpl_table_duplicate(master_wave_tbl);

    code = qmost_fibtab_arcqc(fibinfo_tbl,
                              QMOST_ARM_RED,
                              master_wave_tbl,
                              ob_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_table_get(fibinfo_tbl, "WAVE_COR_R", 0, NULL),
                 0, FLT_EPSILON);
    cpl_test_abs(cpl_table_get(fibinfo_tbl, "WAVE_CORRMS_R", 0, NULL),
                 1.0e-3, FLT_EPSILON);

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    cpl_table_delete(master_wave_tbl);
    master_wave_tbl = NULL;

    cpl_table_delete(ob_wave_tbl);
    ob_wave_tbl = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_fibtab module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_fibtabload();
    test_qmost_fibtabsave();
    test_qmost_fibtab_dummy();
    test_qmost_fibtab_newcols();
    test_qmost_fibtab_arcqc();

    return cpl_test_end(0);
}

/**@}*/
