/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_gaincor.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_gaincor_test  Unit test of qmost_gaincor
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_gaincor
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_gaincor(void)
{
    int iampx, nampsx = 2;
    int iampy, nampsy = 2;
    int iamp, namps;

    int nxamp = 256;
    int nyamp = 512;

    int nximg, nyimg;

    cpl_image *img = NULL;
    cpl_image *amp_img = NULL;
    cpl_propertylist *hdr = NULL;

    float basects = 10000, flatcts, flatsig, gains[4];

    cpl_error_code code;

    char *key = NULL;
    float thisgain;

    cpl_mask *mask = NULL;

    /* Simulate flat */
    nximg = nampsx * nxamp;
    nyimg = nampsy * nyamp;

    namps = nampsx * nampsy;

    img = cpl_image_new(nximg, nyimg, CPL_TYPE_FLOAT);

    for(iampy = 0; iampy < nampsy; iampy++) {
        for(iampx = 0; iampx < nampsx; iampx++) {
            if(iampy % 2) {
                iamp = iampy * nampsx + nampsx - iampx - 1;
            }
            else {
                iamp = iampy * nampsx + iampx;
            }

            flatcts = basects + iamp * 1000;
            flatsig = sqrt(flatcts);

            amp_img = cpl_image_new(nxamp, nyamp, CPL_TYPE_FLOAT);
            qmost_test_fill_noise_rounded_gauss(amp_img, flatcts, flatsig);
            cpl_image_threshold(amp_img, 0, 65535, 0, 65535);

            cpl_image_copy(img, amp_img, iampx*nxamp+1, iampy*nyamp+1);

            gains[iamp] = flatcts / basects;

            cpl_image_delete(amp_img);
            amp_img = NULL;
        }
    }

    /* Empty header to start */
    hdr = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_gaincor(NULL, hdr, 16);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_gaincor(img, NULL, 16);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Missing namps */
    code = qmost_gaincor(img, hdr, 16);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* No op */
    cpl_propertylist_update_int(hdr, "ESO DRS NAMPS", 1);

    code = qmost_gaincor(img, hdr, 16);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Invalid number of amps */
    cpl_propertylist_update_int(hdr, "ESO DRS NAMPS", 3);

    code = qmost_gaincor(img, hdr, 16);
    cpl_test_eq_error(code, CPL_ERROR_UNSUPPORTED_MODE);

    /* Populate header with correct number of amps, this should work */
    cpl_propertylist_update_int(hdr, "ESO DRS NAMPS", namps);

    code = qmost_gaincor(img, hdr, 16);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iampy = 0; iampy < nampsy; iampy++) {
        for(iampx = 0; iampx < nampsx; iampx++) {
            if(iampy % 2) {
                iamp = iampy * nampsx + nampsx - iampx - 1;
            }
            else {
                iamp = iampy * nampsx + iampx;
            }

            amp_img = cpl_image_extract(img,
                                        iampx*nxamp+1, iampy*nyamp+1,
                                        (iampx+1)*nxamp, (iampy+1)*nyamp);
            cpl_test_nonnull(amp_img);

            cpl_test_abs(cpl_image_get_median(amp_img), basects, 100.0);

            cpl_image_delete(amp_img);
            amp_img = NULL;

            key = cpl_sprintf("ESO DRS AMP%d GAINCOR", iamp+1);
            
            thisgain = cpl_propertylist_get_float(hdr, key);
            cpl_test_error(CPL_ERROR_NONE);
            
            cpl_test_abs(thisgain, gains[iamp], 0.01);
            
            cpl_free(key);
            key = NULL;
        }
    }

    /* Singular matrix case can be triggered by setting the entire
     * image bad so all the weights are zero. */
    mask = cpl_mask_new(nximg, nyimg);
    cpl_mask_not(mask);

    cpl_image_reject_from_mask(img, mask);

    cpl_mask_delete(mask);
    mask = NULL;

    code = qmost_gaincor(img, hdr, 16);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* ...and results should all be 1.0 */
    for(iamp = 0; iamp < namps; iamp++) {
        key = cpl_sprintf("ESO DRS AMP%d GAINCOR", iamp+1);
            
        thisgain = cpl_propertylist_get_float(hdr, key);
        cpl_test_error(CPL_ERROR_NONE);
        
        cpl_test_abs(thisgain, 1.0, 0.01);
        
        cpl_free(key);
        key = NULL;
    }

    cpl_image_delete(img);
    img = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_gaincor module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_gaincor();

    return cpl_test_end(0);
}

/**@}*/
