/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_skysub.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_skysub_test  Unit test of qmost_skysub
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_skysub_vshiftpca
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_skysub_vshiftpca(void)
{
    cpl_image *in_spec_img = NULL;
    cpl_image *in_spec_var = NULL;

    cpl_image *spec_img = NULL;
    cpl_image *spec_var = NULL;
    cpl_propertylist *spec_hdr = NULL;
    cpl_table *fibinfo_tbl = NULL;

    cpl_image *this_in_spec = NULL;
    cpl_image *this_spec = NULL;

    cpl_error_code code;

    int nwave = 512;
    int nspec = 64;
    double xfwhm = 3;

    int iline;
    int nlines = 32;
    double xline;

    int ispec;
    double xw, dx;
    int xl, xh, x;

    float *buf = NULL;

    int fib_use;

    qmost_skysub_diags diags;

    /* Simulate spectra */
    in_spec_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(in_spec_img, -8, 8);
    cpl_image_add_scalar(in_spec_img, 100.0);

    buf = cpl_image_get_data_float(in_spec_img);

    xw = 0.5*CPL_MATH_FWHM_SIG*CPL_MATH_FWHM_SIG / (xfwhm*xfwhm);

    for(ispec = 0; ispec < nspec; ispec++) {
        /* Put in some sky lines */
        for(iline = 0; iline < nlines; iline++) {
            xline = 16*(iline+0.5);

            xl = floor(xline - 3*xfwhm - 1);
            if(xl < 0)
                xl = 0;
            if(xl >= nwave)
                xl = nwave-1;
            
            xh = ceil(xline + 3*xfwhm - 1);
            if(xh < 0)
                xh = 0;
            if(xh >= nwave)
                xh = nwave-1;
            
            for(x = xl; x <= xh; x++) {
                dx = (x+1) - xline;
                buf[ispec*nwave+x] += 10*(5 + iline % 5) * exp(-xw * dx*dx);
            }
        }
    }

    /* Variance */
    in_spec_var = cpl_image_add_scalar_create(in_spec_img, 64);
    cpl_image_threshold(in_spec_var, 0, FLT_MAX, 0, FLT_MAX);

    /* Add a bad pixel common to all fibres */
    for(ispec = 0; ispec < nspec; ispec++) {
        cpl_image_reject(in_spec_img, 42, ispec+1);
        cpl_image_set(in_spec_var, 42, ispec+1, 0);
    }

    /* FIBINFO, empty for now */
    fibinfo_tbl = cpl_table_new(0);

    /* Outputs */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);
    spec_hdr = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_skysub_vshiftpca(NULL, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_skysub_vshiftpca(spec_img, NULL, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, NULL, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, NULL,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Missing headers, add them one by one */
    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_update_float(spec_hdr, "CRVAL1", 5000.0);
    
    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_update_float(spec_hdr, "CRPIX1", 1.0);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_update_float(spec_hdr, "CD1_1", 2.0);
    
    /* FIBINFO isn't the right size */
    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    /* FIBINFO right size but missing column */
    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = cpl_table_new(nspec);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Set up FIB_USE to have NULL, 1, 2, 3, 4 */
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);

    for(ispec = 0; ispec < nspec; ispec++) {
        if(ispec % 5) {
            cpl_table_set_int(fibinfo_tbl, "FIB_USE", ispec, ispec % 5);
        }
    }

    /* This should work */
    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check result.  FIB_USE = 1, 3, 4 should be sky subtracted,
     * others should not. */
    for(ispec = 0; ispec < nspec; ispec++) {
        this_in_spec = cpl_image_extract(in_spec_img, 1, ispec+1, nwave, ispec+1);
        cpl_test_nonnull(this_in_spec);

        this_spec = cpl_image_extract(spec_img, 1, ispec+1, nwave, ispec+1);
        cpl_test_nonnull(this_spec);

        fib_use = cpl_table_get_int(fibinfo_tbl, "FIB_USE", ispec, NULL);
        if(fib_use == 1 || fib_use == 3 || fib_use == 4) {
            qmost_test_image_float_abs(this_spec, 0, 0, 20);
        }
        else {
            cpl_test_image_abs(this_spec, this_in_spec, 20);
        }

        cpl_image_delete(this_in_spec);
        this_in_spec = NULL;

        cpl_image_delete(this_spec);
        this_spec = NULL;
    }

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    /* Try doscale=1 */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 1, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    /* Try the extra residual filtering */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 1, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    /* Test with less than 25 sky lines by using half the image */
    spec_img = cpl_image_extract(in_spec_img, 1, 1, nwave/2, nspec);
    spec_var = cpl_image_extract(in_spec_var, 1, 1, nwave/2, nspec);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    /* Test diagnostic outputs */
    spec_img = cpl_image_duplicate(in_spec_img);
    spec_var = cpl_image_duplicate(in_spec_var);

    code = qmost_skysub_vshiftpca(spec_img, spec_var, spec_hdr, fibinfo_tbl,
                                  100, 10, 0, 0, 0, 0, &diags);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(spec_img);
    spec_img = NULL;

    cpl_image_delete(spec_var);
    spec_var = NULL;

    cpl_test_nonnull(diags.eigenvectors);
    cpl_image_delete(diags.eigenvectors);
    diags.eigenvectors = NULL;

    cpl_test_nonnull(diags.eigeninfo);
    cpl_table_delete(diags.eigeninfo);
    diags.eigeninfo = NULL;

    cpl_test_nonnull(diags.orig_img);
    cpl_image_delete(diags.orig_img);
    diags.orig_img = NULL;

    cpl_test_nonnull(diags.orig_var);
    cpl_image_delete(diags.orig_var);
    diags.orig_var = NULL;

    cpl_test_nonnull(diags.skyinfo);
    cpl_table_delete(diags.skyinfo);
    diags.skyinfo = NULL;

    cpl_test_nonnull(diags.comb_img);
    cpl_image_delete(diags.comb_img);
    diags.comb_img = NULL;

    cpl_test_nonnull(diags.comb_var);
    cpl_image_delete(diags.comb_var);
    diags.comb_var = NULL;

    cpl_test_nonnull(diags.subt_img);
    cpl_image_delete(diags.subt_img);
    diags.subt_img = NULL;

    cpl_test_nonnull(diags.subt_var);
    cpl_image_delete(diags.subt_var);
    diags.subt_var = NULL;

    /* Clean up */
    cpl_propertylist_delete(spec_hdr);
    spec_hdr = NULL;

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    cpl_image_delete(in_spec_img);
    cpl_image_delete(in_spec_var);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_skysub module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_skysub_vshiftpca();

    return cpl_test_end(0);
}

/**@}*/
