/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_spec_combine.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_spec_combine_test  Unit test of qmost_spec_combine
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_spec_combine_lite
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_spec_combine_lite(void)
{
    cpl_imagelist *in_imglist = NULL;
    cpl_imagelist *in_varlist = NULL;

    cpl_imagelist *tmp_imglist = NULL;
    cpl_imagelist *tmp_varlist = NULL;

    cpl_image *in_spec_img = NULL;
    cpl_image *in_spec_var = NULL;

    cpl_error_code code;

    int nwave = 512;
    int nspec = 64;

    int ifile, ispec;

    cpl_image *out_img = NULL;
    cpl_image *out_var = NULL;

    float imglevel, expect_img, expect_var;
    cpl_mask *expect_msk = NULL;

    /* Create image lists */
    in_imglist = cpl_imagelist_new();
    in_varlist = cpl_imagelist_new();

    expect_img = 0;
    expect_var = 0;

    expect_msk = cpl_mask_new(nwave, nspec);

    for(ifile = 0; ifile < 5; ifile++) {
        /* Spectra */
        imglevel = 100.0 + 10.0*ifile;

        in_spec_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
        cpl_image_fill_noise_uniform(in_spec_img, -8, 8);
        cpl_image_add_scalar(in_spec_img, imglevel);
        
        expect_img += imglevel;

        /* plus a few outliers */
        cpl_image_set(in_spec_img, 50*ifile+1, 10*ifile+1, 1000.0);
        cpl_mask_set(expect_msk, 50*ifile+1, 10*ifile+1, 1);

        /* Uncertainty */
        in_spec_var = cpl_image_add_scalar_create(in_spec_img, 64);
        cpl_image_threshold(in_spec_var, 0, FLT_MAX, 0, FLT_MAX);

        expect_var += imglevel + 64;

        /* Flag some bad pixels by var = 0 */
        for(ispec = 0; ispec < nspec; ispec++) {
            cpl_image_set(in_spec_var, 100+ispec, 10*ifile+1, 0.0);
            cpl_mask_set(expect_msk, 100+ispec, 10*ifile+1, 1);
        }

        cpl_imagelist_set(in_imglist, in_spec_img, ifile);
        cpl_imagelist_set(in_varlist, in_spec_var, ifile);
    }

    /* NULL input tests */
    code = qmost_spec_combine_lite(NULL, in_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_spec_combine_lite(in_imglist, NULL, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_spec_combine_lite(in_imglist, in_varlist, 5.0,
                                   NULL, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_spec_combine_lite(in_imglist, in_varlist, 5.0,
                                   &out_img, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty image list */
    tmp_imglist = cpl_imagelist_new();
    tmp_varlist = cpl_imagelist_new();

    code = qmost_spec_combine_lite(tmp_imglist, tmp_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Mismatched number of images and variances */
    code = qmost_spec_combine_lite(in_imglist, tmp_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);
    
    /* Size mismatch */
    for(ifile = 0; ifile < 5; ifile++) {
        in_spec_var = cpl_imagelist_get(in_varlist, ifile);
        cpl_imagelist_set(tmp_varlist,
                          cpl_image_extract(in_spec_var,
                                            1, 1, nwave/2, nspec/2),
                          ifile);
    }

    code = qmost_spec_combine_lite(in_imglist, tmp_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_imagelist_delete(tmp_imglist);
    tmp_imglist = NULL;

    cpl_imagelist_delete(tmp_varlist);
    tmp_varlist = NULL;

    /* This should work, normal case */
    code = qmost_spec_combine_lite(in_imglist, in_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(out_img, expect_img, 0, 100);

    cpl_image_reject_from_mask(out_var, expect_msk);
    qmost_test_image_float_abs(out_var, expect_var, 0, 100);

    cpl_image_delete(out_img);
    out_img = NULL;

    cpl_image_delete(out_var);
    out_var = NULL;

    /* Zero counts, returns zero */
    tmp_imglist = cpl_imagelist_duplicate(in_imglist);
    cpl_imagelist_multiply_scalar(tmp_imglist, 0);

    tmp_varlist = cpl_imagelist_duplicate(in_varlist);
    cpl_imagelist_multiply_scalar(tmp_varlist, 0);

    code = qmost_spec_combine_lite(tmp_imglist, tmp_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(out_img, 0, 0, FLT_EPSILON);

    cpl_image_reject_from_mask(out_var, expect_msk);
    qmost_test_image_float_abs(out_var, 0, 0, FLT_EPSILON);

    cpl_imagelist_delete(tmp_imglist);
    tmp_imglist = NULL;

    cpl_imagelist_delete(tmp_varlist);
    tmp_varlist = NULL;

    cpl_image_delete(out_img);
    out_img = NULL;

    cpl_image_delete(out_var);
    out_var = NULL;

    /* Low SNR */
    tmp_imglist = cpl_imagelist_duplicate(in_imglist);
    cpl_imagelist_multiply_scalar(tmp_imglist, 0.01);

    code = qmost_spec_combine_lite(tmp_imglist, in_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    qmost_test_image_float_abs(out_img, 0.01 * expect_img, 0, 1.0);

    cpl_image_reject_from_mask(out_var, expect_msk);
    qmost_test_image_float_abs(out_var, expect_var, 0, 100);

    cpl_imagelist_delete(tmp_imglist);
    tmp_imglist = NULL;

    cpl_image_delete(out_img);
    out_img = NULL;

    cpl_image_delete(out_var);
    out_var = NULL;

    /* One input, no-op */
    tmp_imglist = cpl_imagelist_new();
    tmp_varlist = cpl_imagelist_new();

    cpl_imagelist_set(tmp_imglist,
                      cpl_image_duplicate(cpl_imagelist_get(in_imglist, 0)),
                      0);
    cpl_imagelist_set(tmp_varlist,
                      cpl_image_duplicate(cpl_imagelist_get(in_varlist, 0)),
                      0);

    code = qmost_spec_combine_lite(tmp_imglist, tmp_varlist, 5.0,
                                   &out_img, &out_var);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_image_abs(cpl_imagelist_get(in_imglist, 0),
                       out_img,
                       FLT_EPSILON);
    cpl_test_image_abs(cpl_imagelist_get(in_varlist, 0),
                       out_var,
                       FLT_EPSILON);

    cpl_imagelist_delete(tmp_imglist);
    tmp_imglist = NULL;

    cpl_imagelist_delete(tmp_varlist);
    tmp_varlist = NULL;

    cpl_image_delete(out_img);
    out_img = NULL;

    cpl_image_delete(out_var);
    out_var = NULL;

    cpl_imagelist_delete(in_imglist);
    in_imglist = NULL;

    cpl_imagelist_delete(in_varlist);
    in_varlist = NULL;

    cpl_mask_delete(expect_msk);
    expect_msk = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_spec_combine module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_spec_combine_lite();

    return cpl_test_end(0);
}

/**@}*/
