/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#define _POSIX_C_SOURCE   200809L   /* For mkstemp()       */
#define _DARWIN_C_SOURCE

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_utils.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_utils_test  Unit test of qmost_utils
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_get_license
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_get_license(void)
{
    const char *str = NULL;

    str = qmost_get_license();
    cpl_test_nonnull(str);
    qmost_test_gt(strlen(str), 0);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_check_and_set_groups
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_check_and_set_groups(void)
{
    cpl_frameset *frameset = NULL;
    cpl_frame *frame = NULL;
    cpl_propertylist *hdr = NULL;
    char tmpfile[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    /* Empty frameset */
    frameset = cpl_frameset_new();

    /* NULL input */
    code = qmost_check_and_set_groups(NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty frameset, should work */
    code = qmost_check_and_set_groups(frameset);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Empty file (not valid FITS) */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Add to frameset */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, tmpfile);
    cpl_frame_set_tag(frame, QMOST_RAW_BIAS);
    cpl_frameset_insert(frameset, frame);

    code = qmost_check_and_set_groups(frameset);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* Populate with FITS header */
    hdr = cpl_propertylist_new();
    cpl_propertylist_save(hdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(hdr);

    /* Should now work */
    code = qmost_check_and_set_groups(frameset);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Add invalid to frameset */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, tmpfile);
    cpl_frame_set_tag(frame, "INVALID_TAG");
    cpl_frameset_insert(frameset, frame);

    code = qmost_check_and_set_groups(frameset);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_frameset_delete(frameset);
    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_propertylist_get_double
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_propertylist_get_double(void)
{
    cpl_propertylist *plist = NULL;
    cpl_error_code code;
    double value;

    plist = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_cpl_propertylist_get_double(NULL, "DOUBLE", &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_propertylist_get_double(plist, NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_propertylist_get_double(plist, "DOUBLE", NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Not found (blank plist) */
    code = qmost_cpl_propertylist_get_double(plist, "DOUBLE", &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Now populate it */
    cpl_propertylist_update_bool(plist, "BOOL", 1);
    cpl_propertylist_update_char(plist, "CHAR", '2');
    cpl_propertylist_update_double(plist, "DOUBLE", 3);
    cpl_propertylist_update_float(plist, "FLOAT", 4);
    cpl_propertylist_update_int(plist, "INT", 5);
    cpl_propertylist_update_long(plist, "LONG", 6);
    cpl_propertylist_update_long_long(plist, "LONGLONG", 7);

    /* These should all work */
    code = qmost_cpl_propertylist_get_double(plist, "BOOL", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 1, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "DOUBLE", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 3, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "FLOAT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 4, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "INT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 5, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "LONG", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 6, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "LONGLONG", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 7, DBL_EPSILON);

    /* This should not */
    code = qmost_cpl_propertylist_get_double(plist, "CHAR", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* Test strings, none of these should work */
    cpl_propertylist_update_string(plist, "STRING1", "");
    cpl_propertylist_update_string(plist, "STRING2", " ");
    cpl_propertylist_update_string(plist, "STRING3", "U0");
    cpl_propertylist_update_string(plist, "STRING4", "0U");

    code = qmost_cpl_propertylist_get_double(plist, "STRING1", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_double(plist, "STRING2", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_double(plist, "STRING3", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_double(plist, "STRING4", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* These should work */
    cpl_propertylist_update_string(plist, "STRING5", "42");
    cpl_propertylist_update_string(plist, "STRING6", "64.5");
    cpl_propertylist_update_string(plist, "STRING7", "1.0e3");
    cpl_propertylist_update_string(plist, "STRING8", " 9");
    cpl_propertylist_update_string(plist, "STRING9", "9 ");
    cpl_propertylist_update_string(plist, "STRING10", "10 ");

    code = qmost_cpl_propertylist_get_double(plist, "STRING5", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 42, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "STRING6", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 64.5, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "STRING7", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 1.0e3, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "STRING8", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 9, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "STRING9", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 9, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_double(plist, "STRING10", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 10, DBL_EPSILON);

    cpl_propertylist_delete(plist);
    plist = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_propertylist_get_float
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_propertylist_get_float(void)
{
    cpl_propertylist *plist = NULL;
    cpl_error_code code;
    float value;

    plist = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_cpl_propertylist_get_float(NULL, "FLOAT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_propertylist_get_float(plist, NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_propertylist_get_float(plist, "FLOAT", NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Not found (blank plist) */
    code = qmost_cpl_propertylist_get_float(plist, "FLOAT", &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Now populate it */
    cpl_propertylist_update_bool(plist, "BOOL", 1);
    cpl_propertylist_update_char(plist, "CHAR", '2');
    cpl_propertylist_update_double(plist, "DOUBLE", 3);
    cpl_propertylist_update_float(plist, "FLOAT", 4);
    cpl_propertylist_update_int(plist, "INT", 5);
    cpl_propertylist_update_long(plist, "LONG", 6);
    cpl_propertylist_update_long_long(plist, "LONGLONG", 7);

    /* These should all work */
    code = qmost_cpl_propertylist_get_float(plist, "BOOL", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 1, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "DOUBLE", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 3, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "FLOAT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 4, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "INT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 5, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "LONG", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 6, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "LONGLONG", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 7, DBL_EPSILON);

    /* This should not */
    code = qmost_cpl_propertylist_get_float(plist, "CHAR", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* Test strings, none of these should work */
    cpl_propertylist_update_string(plist, "STRING1", "");
    cpl_propertylist_update_string(plist, "STRING2", " ");
    cpl_propertylist_update_string(plist, "STRING3", "U0");
    cpl_propertylist_update_string(plist, "STRING4", "0U");

    code = qmost_cpl_propertylist_get_float(plist, "STRING1", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_float(plist, "STRING2", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_float(plist, "STRING3", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_float(plist, "STRING4", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* These should work */
    cpl_propertylist_update_string(plist, "STRING5", "42");
    cpl_propertylist_update_string(plist, "STRING6", "64.5");
    cpl_propertylist_update_string(plist, "STRING7", "1.0e3");
    cpl_propertylist_update_string(plist, "STRING8", " 9");
    cpl_propertylist_update_string(plist, "STRING9", "9 ");
    cpl_propertylist_update_string(plist, "STRING10", "10 ");

    code = qmost_cpl_propertylist_get_float(plist, "STRING5", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 42, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "STRING6", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 64.5, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "STRING7", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 1.0e3, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "STRING8", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 9, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "STRING9", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 9, DBL_EPSILON);

    code = qmost_cpl_propertylist_get_float(plist, "STRING10", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 10, DBL_EPSILON);

    cpl_propertylist_delete(plist);
    plist = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_propertylist_get_int
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_propertylist_get_int(void)
{
    cpl_propertylist *plist = NULL;
    cpl_error_code code;
    int value;

    plist = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_cpl_propertylist_get_int(NULL, "INT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_propertylist_get_int(plist, NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_propertylist_get_int(plist, "INT", NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Not found (blank plist) */
    code = qmost_cpl_propertylist_get_int(plist, "INT", &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Now populate it */
    cpl_propertylist_update_bool(plist, "BOOL", 1);
    cpl_propertylist_update_char(plist, "CHAR", '2');
    cpl_propertylist_update_double(plist, "DOUBLE", 3);
    cpl_propertylist_update_int(plist, "FLOAT", 4);
    cpl_propertylist_update_int(plist, "INT", 5);
    cpl_propertylist_update_long(plist, "LONG", 6);
    cpl_propertylist_update_long_long(plist, "LONGLONG", 7);

    /* These should all work */
    code = qmost_cpl_propertylist_get_int(plist, "BOOL", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 1);

    code = qmost_cpl_propertylist_get_int(plist, "DOUBLE", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 3);

    code = qmost_cpl_propertylist_get_int(plist, "FLOAT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 4);

    code = qmost_cpl_propertylist_get_int(plist, "INT", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 5);

    code = qmost_cpl_propertylist_get_int(plist, "LONG", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 6);

    code = qmost_cpl_propertylist_get_int(plist, "LONGLONG", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 7);

    /* This should not */
    code = qmost_cpl_propertylist_get_int(plist, "CHAR", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* Test strings, none of these should work */
    cpl_propertylist_update_string(plist, "STRING1", "");
    cpl_propertylist_update_string(plist, "STRING2", " ");
    cpl_propertylist_update_string(plist, "STRING3", "U0");
    cpl_propertylist_update_string(plist, "STRING4", "0U");
    cpl_propertylist_update_string(plist, "STRING5", "64.5");
    cpl_propertylist_update_string(plist, "STRING6", "1.0e3");

    code = qmost_cpl_propertylist_get_int(plist, "STRING1", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_int(plist, "STRING2", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_int(plist, "STRING3", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_int(plist, "STRING4", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_int(plist, "STRING5", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_propertylist_get_int(plist, "STRING6", &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* These should work */
    cpl_propertylist_update_string(plist, "STRING7", "42");
    cpl_propertylist_update_string(plist, "STRING8", " 9");
    cpl_propertylist_update_string(plist, "STRING9", "9 ");
    cpl_propertylist_update_string(plist, "STRING10", "10 ");

    code = qmost_cpl_propertylist_get_int(plist, "STRING7", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 42);

    code = qmost_cpl_propertylist_get_int(plist, "STRING8", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 9);

    code = qmost_cpl_propertylist_get_int(plist, "STRING9", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 9);

    code = qmost_cpl_propertylist_get_int(plist, "STRING10", &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 10);

    cpl_propertylist_delete(plist);
    plist = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_copy_cell
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_copy_cell(void)
{
    cpl_table *intbl = NULL;
    cpl_table *outtbl = NULL;
    cpl_error_code code;
    int inputs[5] = { 1, 2, 3, 4, 5 };
    cpl_array *arr = NULL;
    
    /* Create test tables */
    intbl = cpl_table_new(5);
    cpl_test_nonnull(intbl);

    outtbl = cpl_table_new(5);
    cpl_test_nonnull(outtbl);

    code = cpl_table_new_column_array(intbl, "TEST_A", CPL_TYPE_INT, 3);
    code = cpl_table_new_column_array(outtbl, "TEST_A", CPL_TYPE_INT, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(intbl, "TEST_D", CPL_TYPE_DOUBLE);
    code = cpl_table_new_column(outtbl, "TEST_D", CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(intbl, "TEST_F", CPL_TYPE_FLOAT);
    code = cpl_table_new_column(outtbl, "TEST_F", CPL_TYPE_FLOAT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(intbl, "TEST_I", CPL_TYPE_INT);
    code = cpl_table_new_column(outtbl, "TEST_I", CPL_TYPE_INT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(intbl, "TEST_LL", CPL_TYPE_LONG_LONG);
    code = cpl_table_new_column(outtbl, "TEST_LL", CPL_TYPE_LONG_LONG);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(intbl, "TEST_S", CPL_TYPE_STRING);
    code = cpl_table_new_column(outtbl, "TEST_S", CPL_TYPE_STRING);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(outtbl, "MISMATCH_A", CPL_TYPE_INT, 2);
    code = cpl_table_new_column_array(outtbl, "MISMATCH_B", CPL_TYPE_DOUBLE, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    arr = cpl_array_wrap_int(inputs, 3);
    code = cpl_table_set_array(intbl, "TEST_A", 2, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_table_set_double(intbl, "TEST_D", 0, CPL_MATH_PI);
    cpl_table_set_float(intbl, "TEST_F", 1, CPL_MATH_2PI);
    cpl_table_set_int(intbl, "TEST_I", 2, 42);
    cpl_table_set_long_long(intbl, "TEST_LL", 3, 43);
    cpl_table_set_string(intbl, "TEST_S", 4, "Basil");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_cpl_table_copy_cell(NULL, "TEST_I", 0,
                                     intbl, "TEST_I", 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_copy_cell(outtbl, NULL, 0,
                                     intbl, "TEST_I", 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_I", 0,
                                     NULL, "TEST_I", 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_I", 0,
                                     intbl, NULL, 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Column doesn't exist */
    code = qmost_cpl_table_copy_cell(outtbl, "TEST_I", 0,
                                     intbl, "NONEXISTENT", 0);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    code = qmost_cpl_table_copy_cell(outtbl, "NONEXISTENT", 0,
                                     intbl, "TEST_I", 0);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    code = qmost_cpl_table_copy_cell(outtbl, "NONEXISTENT", 0,
                                     intbl, "TEST_I", 2);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Row doesn't exist, tested for every input type */
    code = qmost_cpl_table_copy_cell(outtbl, "TEST_I", 6,
                                     intbl, "TEST_I", 0);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_I", 0,
                                     intbl, "TEST_I", 7);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Array length and type mismatches */
    code = qmost_cpl_table_copy_cell(outtbl, "MISMATCH_A", 0,
                                     intbl, "TEST_A", 2);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    code = qmost_cpl_table_copy_cell(outtbl, "MISMATCH_B", 0,
                                     intbl, "TEST_A", 2);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* These should all succeed and yield the correct data */
    code = qmost_cpl_table_copy_cell(outtbl, "TEST_A", 0,
                                     intbl, "TEST_A", 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_array_abs(cpl_table_get_array(outtbl, "TEST_A", 0),
                       arr,
                       1.0e-10);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_D", 1,
                                     intbl, "TEST_D", 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_table_get_double(outtbl, "TEST_D", 1, NULL),
                 CPL_MATH_PI,
                 1.0e-10);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_F", 2,
                                     intbl, "TEST_F", 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_table_get_float(outtbl, "TEST_F", 2, NULL),
                 CPL_MATH_2PI,
                 1.0e-5);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_I", 3,
                                     intbl, "TEST_I", 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_int(outtbl, "TEST_I", 3, NULL),
                42);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_LL", 4,
                                     intbl, "TEST_LL", 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(cpl_table_get_long_long(outtbl, "TEST_LL", 4, NULL),
                43);

    code = qmost_cpl_table_copy_cell(outtbl, "TEST_S", 0,
                                     intbl, "TEST_S", 4);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq_string(cpl_table_get_string(outtbl, "TEST_S", 0),
                       "Basil");

    cpl_table_delete(intbl);
    cpl_table_delete(outtbl);
    cpl_array_unwrap(arr);
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_get_byte_array
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_get_byte_array(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    int inputs[5] = { 1, 2, 3, 4, 5 };
    unsigned char values[5] = { 0, 0, 0, 0, 0 };
    float blank_floats[2] = { 0, 0 };
    cpl_array *arr, **parr;
    
    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_B", CPL_TYPE_INT, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_B", CPL_TYPE_INT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_F", CPL_TYPE_FLOAT, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_F", CPL_TYPE_FLOAT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    arr = cpl_array_wrap_int(inputs, 3);
    code = cpl_table_set_array(table, "TEST_B", 2, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_B", 3, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_array_unwrap(arr);

    cpl_table_set_int(table, "TEST_ONE_B", 2, inputs[0]);
    cpl_table_set_int(table, "TEST_ONE_B", 3, inputs[1]);

    arr = cpl_array_wrap_float(blank_floats, 2);
    code = cpl_table_set_array(table, "TEST_F", 0, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_F", 1, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_F", 2, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_F", 3, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_F", 4, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_array_unwrap(arr);

    cpl_table_set_int(table, "TEST_ONE_F", 2, blank_floats[0]);
    cpl_table_set_int(table, "TEST_ONE_F", 3, blank_floats[1]);

    /* NULL inputs */
    code = qmost_cpl_table_get_byte_array(NULL, "TEST_B", 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_get_byte_array(table, NULL, 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_get_byte_array(table, "TEST_B", 2, NULL, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_get_byte_array(table, "TEST_B", -1, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_get_byte_array(table, "TEST_B", 6, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_get_byte_array(table, "TEST_B", 2, values, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_get_byte_array(table, "TEST_F", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    code = qmost_cpl_table_get_byte_array(table, "TEST_ONE_F", 2, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* A correct call, this should work and set 2 elements (not the 3rd) */
    code = qmost_cpl_table_get_byte_array(table, "TEST_B", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], inputs[0]);
    cpl_test_eq(values[1], inputs[1]);
    cpl_test_eq(values[2], 0);

    /* Check NULL flagging, first if the column was never set */
    memset(values, 0, sizeof(values));

    code = qmost_cpl_table_get_byte_array(table, "TEST_B", 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], 0);
    cpl_test_eq(values[1], 0);
    cpl_test_eq(values[2], 0);

    /* Now invalidate an element of a row we did set */
    memset(values, 0, sizeof(values));

    parr = cpl_table_get_data_array(table, "TEST_B");
    cpl_test_nonnull(parr);
    code = cpl_array_set_invalid(parr[2], 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    code = qmost_cpl_table_get_byte_array(table, "TEST_B", 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], inputs[0]);
    cpl_test_eq(values[1], 0);
    cpl_test_eq(values[2], inputs[2]);

    /* Test the scalar case */
    values[0] = 255;

    code = qmost_cpl_table_get_byte_array(table, "TEST_ONE_B", 0, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = qmost_cpl_table_get_byte_array(table, "TEST_ONE_B", 2, values, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], 255);

    code = qmost_cpl_table_get_byte_array(table, "TEST_ONE_B", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_get_byte_array(table, "TEST_ONE_B", 2, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], inputs[0]);

    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_get_double_array
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_get_double_array(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    double inputs[5] = { 1, 2, 3, 4, 5 };
    double values[5] = { 0, 0, 0, 0, 0 };
    cpl_array *arr, **parr;
    int irow;

    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_D", CPL_TYPE_DOUBLE, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_D", CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_S", CPL_TYPE_STRING, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_S", CPL_TYPE_STRING);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    arr = cpl_array_wrap_double(inputs, 3);
    code = cpl_table_set_array(table, "TEST_D", 2, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_D", 3, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_array_unwrap(arr);

    cpl_table_set_double(table, "TEST_ONE_D", 2, inputs[0]);
    cpl_table_set_double(table, "TEST_ONE_D", 3, inputs[1]);

    arr = cpl_array_new(2, CPL_TYPE_STRING);
    for(irow = 0; irow < 5; irow++) {
        cpl_table_set_array(table, "TEST_S", 2, arr);
    }
    cpl_array_delete(arr);

    /* NULL inputs */
    code = qmost_cpl_table_get_double_array(NULL, "TEST_D", 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_get_double_array(table, NULL, 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_get_double_array(table, "TEST_D", 2, NULL, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_get_double_array(table, "TEST_D", -1, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_get_double_array(table, "TEST_D", 6, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_get_double_array(table, "TEST_D", 2, values, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_get_double_array(table, "TEST_S", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    code = qmost_cpl_table_get_double_array(table, "TEST_ONE_S", 2, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    /* A correct call, this should work and set 2 elements (not the 3rd) */
    code = qmost_cpl_table_get_double_array(table, "TEST_D", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(values[0], inputs[0], DBL_EPSILON);
    cpl_test_abs(values[1], inputs[1], DBL_EPSILON);
    cpl_test_abs(values[2], 0, DBL_EPSILON);

    /* Check NULL flagging, first if the column was never set */
    code = qmost_cpl_table_get_double_array(table, "TEST_D", 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test(isnan(values[0]));
    cpl_test(isnan(values[1]));
    cpl_test(isnan(values[2]));

    memset(values, 0, sizeof(values));

    /* Now invalidate an element of a row we did set */
    parr = cpl_table_get_data_array(table, "TEST_D");
    cpl_test_nonnull(parr);
    code = cpl_array_set_invalid(parr[2], 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    code = qmost_cpl_table_get_double_array(table, "TEST_D", 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(values[0], inputs[0], DBL_EPSILON);
    cpl_test(isnan(values[1]));
    cpl_test_abs(values[2], inputs[2], DBL_EPSILON);

    /* Test the scalar case */
    code = qmost_cpl_table_get_double_array(table, "TEST_ONE_D", 0, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test(isnan(values[0]));

    values[0] = 255.0;

    code = qmost_cpl_table_get_double_array(table, "TEST_ONE_D", 2, values, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], 255.0);

    code = qmost_cpl_table_get_double_array(table, "TEST_ONE_D", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_get_double_array(table, "TEST_ONE_D", 2, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], inputs[0]);

    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_get_float_array
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_get_float_array(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    float inputs[5] = { 1, 2, 3, 4, 5 };
    float values[5] = { 0, 0, 0, 0, 0 };
    cpl_array *arr, **parr;
    int irow;
    
    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_F", CPL_TYPE_FLOAT, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_F", CPL_TYPE_FLOAT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_S", CPL_TYPE_STRING, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_S", CPL_TYPE_STRING);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    arr = cpl_array_wrap_float(inputs, 3);
    code = cpl_table_set_array(table, "TEST_F", 2, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_F", 3, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_array_unwrap(arr);

    cpl_table_set_float(table, "TEST_ONE_F", 2, inputs[0]);
    cpl_table_set_float(table, "TEST_ONE_F", 3, inputs[1]);

    arr = cpl_array_new(2, CPL_TYPE_STRING);
    for(irow = 0; irow < 5; irow++) {
        cpl_table_set_array(table, "TEST_S", 2, arr);
    }
    cpl_array_delete(arr);

    /* NULL inputs */
    code = qmost_cpl_table_get_float_array(NULL, "TEST_F", 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_get_float_array(table, NULL, 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_get_float_array(table, "TEST_F", 2, NULL, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_get_float_array(table, "TEST_F", -1, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_get_float_array(table, "TEST_F", 6, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_get_float_array(table, "TEST_F", 2, values, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_get_float_array(table, "TEST_S", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    code = qmost_cpl_table_get_float_array(table, "TEST_ONE_S", 2, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    /* A correct call, this should work and set 2 elements (not the 3rd) */
    code = qmost_cpl_table_get_float_array(table, "TEST_F", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(values[0], inputs[0], DBL_EPSILON);
    cpl_test_abs(values[1], inputs[1], DBL_EPSILON);
    cpl_test_abs(values[2], 0, DBL_EPSILON);

    /* Check NULL flagging, first if the column was never set */
    code = qmost_cpl_table_get_float_array(table, "TEST_F", 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test(isnan(values[0]));
    cpl_test(isnan(values[1]));
    cpl_test(isnan(values[2]));

    memset(values, 0, sizeof(values));

    /* Now invalidate an element of a row we did set */
    parr = cpl_table_get_data_array(table, "TEST_F");
    cpl_test_nonnull(parr);
    code = cpl_array_set_invalid(parr[2], 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    code = qmost_cpl_table_get_float_array(table, "TEST_F", 2, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(values[0], inputs[0], DBL_EPSILON);
    cpl_test(isnan(values[1]));
    cpl_test_abs(values[2], inputs[2], DBL_EPSILON);

    /* Test the scalar case */
    code = qmost_cpl_table_get_float_array(table, "TEST_ONE_F", 0, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test(isnan(values[0]));

    values[0] = 255.0;

    code = qmost_cpl_table_get_float_array(table, "TEST_ONE_F", 2, values, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], 255.0);

    code = qmost_cpl_table_get_float_array(table, "TEST_ONE_F", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_get_float_array(table, "TEST_ONE_F", 2, values, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(values[0], inputs[0]);


    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_get_polynomial
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_get_polynomial(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    double inputs[5] = { 1, 2, 3, 4, 5 };
    cpl_array *arr, **parr;
    cpl_polynomial *poly = NULL;
    cpl_size deg;
    int irow;

    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_D", CPL_TYPE_DOUBLE, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_D", CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_S", CPL_TYPE_STRING, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column(table, "TEST_ONE_S", CPL_TYPE_STRING);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    arr = cpl_array_wrap_double(inputs, 3);
    code = cpl_table_set_array(table, "TEST_D", 2, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_table_set_array(table, "TEST_D", 3, arr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_array_unwrap(arr);

    cpl_table_set_double(table, "TEST_ONE_D", 2, inputs[0]);
    cpl_table_set_double(table, "TEST_ONE_D", 3, inputs[1]);

    arr = cpl_array_new(2, CPL_TYPE_STRING);
    for(irow = 0; irow < 5; irow++) {
        cpl_table_set_array(table, "TEST_S", 2, arr);
    }
    cpl_array_delete(arr);

    /* NULL inputs */
    poly = qmost_cpl_table_get_polynomial(NULL, "TEST_D", 2, 2);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    poly = qmost_cpl_table_get_polynomial(table, NULL, 2, 2);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    poly = qmost_cpl_table_get_polynomial(table, "TEST_D", -1, 2);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    poly = qmost_cpl_table_get_polynomial(table, "TEST_D", 6, 2);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    poly = qmost_cpl_table_get_polynomial(table, "TEST_D", 2, 5);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    poly = qmost_cpl_table_get_polynomial(table, "TEST_S", 2, 1);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    poly = qmost_cpl_table_get_polynomial(table, "TEST_ONE_S", 2, 0);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    /* A correct call, this should work and set 2 elements (not the 3rd) */
    poly = qmost_cpl_table_get_polynomial(table, "TEST_D", 2, 1);
    cpl_test_nonnull(poly);

    deg = 0;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), inputs[0], DBL_EPSILON);
    deg = 1;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), inputs[1], DBL_EPSILON);
    deg = 2;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), 0, DBL_EPSILON);

    cpl_polynomial_delete(poly);

    /* Check NULL flagging, first if the column was never set */
    poly = qmost_cpl_table_get_polynomial(table, "TEST_D", 0, 2);
    cpl_test_nonnull(poly);

    deg = 0;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), 0, DBL_EPSILON);
    deg = 1;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), 0, DBL_EPSILON);
    deg = 2;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), 0, DBL_EPSILON);

    cpl_polynomial_delete(poly);

    /* Now invalidate an element of a row we did set */
    parr = cpl_table_get_data_array(table, "TEST_D");
    cpl_test_nonnull(parr);
    code = cpl_array_set_invalid(parr[2], 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    poly = qmost_cpl_table_get_polynomial(table, "TEST_D", 2, 2);
    cpl_test_nonnull(poly);

    deg = 0;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), inputs[0], DBL_EPSILON);
    deg = 1;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), 0, DBL_EPSILON);
    deg = 2;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), inputs[2], DBL_EPSILON);

    cpl_polynomial_delete(poly);

    /* Test the scalar case */
    poly = qmost_cpl_table_get_polynomial(table, "TEST_ONE_D", 0, 0);
    cpl_test_nonnull(poly);

    deg = 0;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), 0, DBL_EPSILON);

    cpl_polynomial_delete(poly);

    poly = qmost_cpl_table_get_polynomial(table, "TEST_ONE_D", 2, 1);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    poly = qmost_cpl_table_get_polynomial(table, "TEST_ONE_D", 2, 0);
    cpl_test_nonnull(poly);

    deg = 0;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &deg), inputs[0], DBL_EPSILON);

    cpl_polynomial_delete(poly);

    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_set_byte_array
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_set_byte_array(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    unsigned char values[5] = { 1, 2, 0, 4, 5 };
    cpl_array *answer;
    
    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_B", CPL_TYPE_INT, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_F", CPL_TYPE_FLOAT, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_cpl_table_set_byte_array(NULL, "TEST_B", 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_byte_array(table, NULL, 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_byte_array(table, "TEST_B", 0, NULL, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_set_byte_array(table, "TEST_B", -1, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_set_byte_array(table, "TEST_B", 6, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_set_byte_array(table, "TEST_B", 0, values, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_set_byte_array(table, "TEST_F", 0, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* A correct call */
    code = qmost_cpl_table_set_byte_array(table, "TEST_B", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    answer = cpl_array_new(3, CPL_TYPE_INT);
    cpl_array_set_int(answer, 0, values[0]);
    cpl_array_set_int(answer, 1, values[1]);
    cpl_array_set_int(answer, 2, values[2]);

    cpl_test_array_abs(cpl_table_get_array(table, "TEST_B", 2),
                       answer,
                       DBL_EPSILON);

    cpl_array_delete(answer);

    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_set_double_array
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_set_double_array(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    double values[5] = { 1, 2, NAN, 4, 5 };
    cpl_array *answer;
    
    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_D", CPL_TYPE_DOUBLE, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_S", CPL_TYPE_STRING, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_cpl_table_set_double_array(NULL, "TEST_D", 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_double_array(table, NULL, 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_double_array(table, "TEST_D", 0, NULL, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_set_double_array(table, "TEST_D", -1, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_set_double_array(table, "TEST_D", 6, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_set_double_array(table, "TEST_D", 0, values, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_set_double_array(table, "TEST_S", 0, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    /* A correct call */
    code = qmost_cpl_table_set_double_array(table, "TEST_D", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    answer = cpl_array_new(3, CPL_TYPE_DOUBLE);
    cpl_array_set_double(answer, 0, values[0]);
    cpl_array_set_double(answer, 1, values[1]);
    cpl_array_set_invalid(answer, 2);

    cpl_test_array_abs(cpl_table_get_array(table, "TEST_D", 2),
                       answer,
                       DBL_EPSILON);

    cpl_array_delete(answer);

    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_set_float_array
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_set_float_array(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    float values[5] = { 1, 2, NAN, 4, 5 };
    cpl_array *answer;
    
    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_F", CPL_TYPE_FLOAT, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_S", CPL_TYPE_STRING, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_cpl_table_set_float_array(NULL, "TEST_F", 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_float_array(table, NULL, 0, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_float_array(table, "TEST_F", 0, NULL, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_set_float_array(table, "TEST_F", -1, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_set_float_array(table, "TEST_F", 6, values, 3);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_set_float_array(table, "TEST_F", 0, values, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_set_float_array(table, "TEST_S", 0, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    /* A correct call */
    code = qmost_cpl_table_set_float_array(table, "TEST_F", 2, values, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    answer = cpl_array_new(3, CPL_TYPE_FLOAT);
    cpl_array_set_float(answer, 0, values[0]);
    cpl_array_set_float(answer, 1, values[1]);
    cpl_array_set_invalid(answer, 2);

    cpl_test_array_abs(cpl_table_get_array(table, "TEST_F", 2),
                       answer,
                       DBL_EPSILON);

    cpl_array_delete(answer);

    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_cpl_table_set_polynomial
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_cpl_table_set_polynomial(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    double values[5] = { 1, 2, 3, 4, 5 };
    cpl_array *answer;
    cpl_polynomial *poly = NULL;
    cpl_size deg;

    poly = cpl_polynomial_new(1);

    for(deg = 0; deg < 5; deg++) {
        cpl_polynomial_set_coeff(poly, &deg, values[deg]);
    }

    /* Create a test table */
    table = cpl_table_new(5);
    cpl_test_nonnull(table);

    code = cpl_table_new_column_array(table, "TEST_D", CPL_TYPE_DOUBLE, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_table_new_column_array(table, "TEST_S", CPL_TYPE_STRING, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_cpl_table_set_polynomial(NULL, "TEST_D", 0, poly, 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_polynomial(table, NULL, 0, poly, 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_cpl_table_set_polynomial(table, "TEST_D", 0, NULL, 2);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row */
    code = qmost_cpl_table_set_polynomial(table, "TEST_D", -1, poly, 2);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    code = qmost_cpl_table_set_polynomial(table, "TEST_D", 6, poly, 2);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Too long */
    code = qmost_cpl_table_set_polynomial(table, "TEST_D", 0, poly, 5);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Wrong type */
    code = qmost_cpl_table_set_polynomial(table, "TEST_S", 0, poly, 1);
    cpl_test_eq_error(code, CPL_ERROR_INVALID_TYPE);

    /* A correct call */
    code = qmost_cpl_table_set_polynomial(table, "TEST_D", 2, poly, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    answer = cpl_array_new(3, CPL_TYPE_DOUBLE);
    cpl_array_set_double(answer, 0, values[0]);
    cpl_array_set_double(answer, 1, values[1]);
    cpl_array_set_double(answer, 2, values[2]);

    cpl_test_array_abs(cpl_table_get_array(table, "TEST_D", 2),
                       answer,
                       DBL_EPSILON);

    cpl_array_delete(answer);

    cpl_table_delete(table);
    cpl_polynomial_delete(poly);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_load_master_mask
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_load_master_mask(void)
{
    cpl_frame *frame = NULL;
    const char *extname = "MASK";
    const char *extdummy = "DUMMY";
    cpl_table *intbl = NULL;
    cpl_mask *inmask = NULL;
    cpl_mask *outmask = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_propertylist *outhdr = NULL;
    char tmpfile[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    /* Empty file (not valid FITS) */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* NULL inputs */
    code = qmost_load_master_mask(NULL, extname, &outmask, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_mask(frame, NULL, &outmask, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_mask(frame, extname, NULL, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set */
    code = qmost_load_master_mask(frame, extname, &outmask, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Filename set but not FITS */
    cpl_frame_set_filename(frame, tmpfile);

    code = qmost_load_master_mask(frame, extname, &outmask, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* FITS but extension doesn't exist */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_save(inhdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_mask(frame, extname, &outmask, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Populate it with a table (wrong data type), should fail */
    intbl = cpl_table_new(1);
    cpl_table_new_column(intbl, "TEST", CPL_TYPE_INT);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST");

    cpl_table_save(intbl, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_table_delete(intbl);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_mask(frame, "TEST", &outmask, NULL);
    cpl_test_eq_error(code, CPL_ERROR_BAD_FILE_FORMAT);

    /* Populate it with an actual mask and correct EXTNAME */
    inmask = cpl_mask_new(256, 512);
    cpl_mask_set(inmask, 42, 42, 1);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname);

    cpl_mask_save(inmask, tmpfile, inhdr, CPL_IO_EXTEND);

    /* Also make a dummy */
    cpl_propertylist_update_string(inhdr, "EXTNAME", extdummy);
    cpl_propertylist_update_bool(inhdr, "ESO DRS DUMMY", 1);

    cpl_mask_save(inmask, tmpfile, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);

    /* Should now work, with or without header arg */
    code = qmost_load_master_mask(frame, extname, &outmask, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outmask);

    cpl_test_eq_mask(inmask, outmask);

    cpl_mask_delete(outmask);

    code = qmost_load_master_mask(frame, extname, &outmask, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outmask);
    cpl_test_nonnull(outhdr);

    cpl_test_eq_mask(inmask, outmask);

    cpl_mask_delete(outmask);
    outmask = NULL;

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    cpl_mask_delete(inmask);
    inmask = NULL;

    /* Attempting to load a dummy should raise an error */
    code = qmost_load_master_mask(frame, extdummy, &outmask, NULL);
    cpl_test_eq_error(code, QMOST_ERROR_DUMMY);
    cpl_test_null(outmask);

    cpl_frame_delete(frame);
    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_load_master_image
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_load_master_image(void)
{
    cpl_frame *frame = NULL;
    const char *extname = "IMAGE";
    const char *extdummy = "DUMMY";
    cpl_table *intbl = NULL;
    cpl_image *inimage = NULL;
    cpl_image *outimage = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_propertylist *outhdr = NULL;
    char tmpfile[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    /* Empty file (not valid FITS) */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* NULL inputs */
    code = qmost_load_master_image(NULL, extname, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_image(frame, NULL, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_image(frame, extname, CPL_TYPE_DOUBLE, NULL, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set */
    code = qmost_load_master_image(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Filename set but not FITS */
    cpl_frame_set_filename(frame, tmpfile);

    code = qmost_load_master_image(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* FITS but extension doesn't exist */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_save(inhdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_image(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Populate it with a table (wrong data type), should fail */
    intbl = cpl_table_new(1);
    cpl_table_new_column(intbl, "TEST", CPL_TYPE_INT);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST");

    cpl_table_save(intbl, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_table_delete(intbl);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_image(frame, "TEST", CPL_TYPE_DOUBLE, &outimage, NULL);
    cpl_test_eq_error(code, CPL_ERROR_BAD_FILE_FORMAT);

    /* Populate it with an actual image and correct EXTNAME */
    inimage = cpl_image_fill_test_create(256, 512);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname);

    cpl_image_save(inimage, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    /* Also make a dummy */
    cpl_propertylist_update_string(inhdr, "EXTNAME", extdummy);
    cpl_propertylist_update_bool(inhdr, "ESO DRS DUMMY", 1);

    cpl_image_save(inimage, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);

    /* Should now work, with or without header arg */
    code = qmost_load_master_image(frame, extname, CPL_TYPE_DOUBLE, &outimage, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimage);

    cpl_test_image_abs(inimage, outimage, 0.1);

    cpl_image_delete(outimage);

    code = qmost_load_master_image(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimage);
    cpl_test_nonnull(outhdr);

    cpl_test_image_abs(inimage, outimage, 0.1);

    cpl_image_delete(outimage);
    outimage = NULL;

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    cpl_image_delete(inimage);
    inimage = NULL;

    /* Attempting to load a dummy should raise an error */
    code = qmost_load_master_image(frame, extdummy, CPL_TYPE_DOUBLE, &outimage, &outhdr);
    cpl_test_eq_error(code, QMOST_ERROR_DUMMY);
    cpl_test_null(outimage);
    cpl_test_null(outhdr);

    cpl_frame_delete(frame);
    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_load_master_image_and_var
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_load_master_image_and_var(void)
{
    cpl_frame *frame = NULL;
    const char *extname = "IMAGE";
    const char *extname_error = "IMAGE_ERROR";
    const char *extname_var = "IMAGE_var";
    cpl_table *intbl = NULL;
    cpl_image *inimage = NULL;
    cpl_image *inerror = NULL;
    cpl_image *invar = NULL;
    cpl_image *outimage = NULL;
    cpl_image *outvar = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_propertylist *outhdr = NULL;
    char tmpfile[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    /* Empty file (not valid FITS) */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* NULL inputs */
    code = qmost_load_master_image_and_var(NULL, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_image_and_var(frame, NULL, CPL_TYPE_DOUBLE, &outimage, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, NULL, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, NULL, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set */
    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Filename set but not FITS */
    cpl_frame_set_filename(frame, tmpfile);

    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* FITS but extension doesn't exist */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_save(inhdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Populate it with a table (wrong data type), should fail */
    intbl = cpl_table_new(1);
    cpl_table_new_column(intbl, "TEST", CPL_TYPE_INT);
    inhdr = cpl_propertylist_new();

    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST1");
    cpl_table_save(intbl, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST1_var");
    cpl_table_save(intbl, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_table_delete(intbl);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_image_and_var(frame, "TEST1", CPL_TYPE_DOUBLE, &outimage, &outvar, NULL);
    cpl_test_eq_error(code, CPL_ERROR_BAD_FILE_FORMAT);

    /* Make test image */
    inimage = cpl_image_fill_test_create(256, 512);

    /* Image correct but variance not */
    intbl = cpl_table_new(1);
    cpl_table_new_column(intbl, "TEST", CPL_TYPE_INT);
    inhdr = cpl_propertylist_new();

    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST2");
    cpl_image_save(inimage, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST2_var");
    cpl_table_save(intbl, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_table_delete(intbl);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_image_and_var(frame, "TEST2", CPL_TYPE_DOUBLE, &outimage, &outvar, NULL);
    cpl_test_eq_error(code, CPL_ERROR_BAD_FILE_FORMAT);

    /* Populate it with an actual image and correct EXTNAME */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname);

    cpl_image_save(inimage, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);

    /* No variance should return zeros in variance array */
    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimage);
    cpl_test_nonnull(outvar);

    cpl_test_image_abs(inimage, outimage, 0.1);
    qmost_test_image_float_abs(outvar, 0, 0, 0.1);

    cpl_image_delete(outimage);
    cpl_image_delete(outvar);

    /* Populate variance */
    invar = cpl_image_fill_test_create(256, 512);
    cpl_image_threshold(invar, 0, FLT_MAX, 0, FLT_MAX);
    inerror = cpl_image_power_create(invar, 0.5);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname_var);

    cpl_image_save(invar, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);
    inhdr = NULL;

    /* Should now work, with or without header arg */
    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimage);
    cpl_test_nonnull(outvar);

    cpl_test_image_abs(inimage, outimage, 0.1);
    cpl_test_image_abs(invar, outvar, 0.1);

    cpl_image_delete(outimage);
    cpl_image_delete(outvar);

    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimage);
    cpl_test_nonnull(outvar);
    cpl_test_nonnull(outhdr);

    cpl_test_image_abs(inimage, outimage, 0.1);
    cpl_test_image_abs(invar, outvar, 0.1);

    cpl_image_delete(outimage);
    cpl_image_delete(outvar);
    cpl_propertylist_delete(outhdr);

    /* Also check error */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname_error);

    cpl_image_save(inerror, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_image_and_var(frame, extname, CPL_TYPE_DOUBLE, &outimage, &outvar, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimage);
    cpl_test_nonnull(outvar);

    cpl_test_image_abs(inimage, outimage, 0.1);
    cpl_test_image_abs(invar, outvar, 0.1);

    cpl_image_delete(outimage);
    cpl_image_delete(outvar);

    cpl_image_delete(inimage);
    cpl_image_delete(invar);
    cpl_image_delete(inerror);

    cpl_frame_delete(frame);
    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_load_master_imagelist
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_load_master_imagelist(void)
{
    cpl_frame *frame = NULL;
    const char *extname = "PROFILES";
    const char *extdummy = "DUMMY";
    cpl_table *intbl = NULL;
    cpl_imagelist *inimagelist = NULL;
    cpl_image *inimage = NULL;
    cpl_imagelist *outimagelist = NULL;
    cpl_image *outimage = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_propertylist *outhdr = NULL;
    char tmpfile[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    int iimg, nimg = 2;

    /* Empty file (not valid FITS) */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* NULL inputs */
    code = qmost_load_master_imagelist(NULL, extname, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_imagelist(frame, NULL, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_imagelist(frame, extname, CPL_TYPE_DOUBLE, NULL, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set */
    code = qmost_load_master_imagelist(frame, extname, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Filename set but not FITS */
    cpl_frame_set_filename(frame, tmpfile);

    code = qmost_load_master_imagelist(frame, extname, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* FITS but extension doesn't exist */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_save(inhdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_imagelist(frame, extname, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Populate it with a table (wrong data type), should fail */
    intbl = cpl_table_new(1);
    cpl_table_new_column(intbl, "TEST", CPL_TYPE_INT);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST");

    cpl_table_save(intbl, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_table_delete(intbl);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_imagelist(frame, "TEST", CPL_TYPE_DOUBLE, &outimagelist, NULL);
    cpl_test_eq_error(code, CPL_ERROR_BAD_FILE_FORMAT);

    /* Populate it with an actual imagelist with a couple of planes,
     * and correct EXTNAME. */
    inimagelist = cpl_imagelist_new();

    for(iimg = 0; iimg < nimg; iimg++) {
        cpl_imagelist_set(inimagelist,
                          cpl_image_fill_test_create(256, 512),
                          iimg);
    }

    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname);

    cpl_imagelist_save(inimagelist, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    /* Also make a dummy */
    cpl_propertylist_update_string(inhdr, "EXTNAME", extdummy);
    cpl_propertylist_update_bool(inhdr, "ESO DRS DUMMY", 1);

    cpl_imagelist_save(inimagelist, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);

    /* Should now work, with or without header arg */
    code = qmost_load_master_imagelist(frame, extname, CPL_TYPE_DOUBLE, &outimagelist, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimagelist);

    for(iimg = 0; iimg < nimg; iimg++) {
        inimage = cpl_imagelist_get(inimagelist, iimg);
        outimage = cpl_imagelist_get(outimagelist, iimg);
        cpl_test_nonnull(outimage);

        cpl_test_image_abs(inimage, outimage, 0.1);
    }

    cpl_imagelist_delete(outimagelist);
    outimagelist = NULL;
    outimage = NULL;

    code = qmost_load_master_imagelist(frame, extname, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outimagelist);
    cpl_test_nonnull(outhdr);

    cpl_test_eq(cpl_imagelist_get_size(outimagelist), nimg);

    for(iimg = 0; iimg < nimg; iimg++) {
        inimage = cpl_imagelist_get(inimagelist, iimg);
        outimage = cpl_imagelist_get(outimagelist, iimg);
        cpl_test_nonnull(outimage);

        cpl_test_image_abs(inimage, outimage, 0.1);
    }

    cpl_imagelist_delete(outimagelist);
    outimagelist = NULL;
    outimage = NULL;

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    cpl_imagelist_delete(inimagelist);
    inimagelist = NULL;

    /* Attempting to load a dummy should raise an error */
    code = qmost_load_master_imagelist(frame, extdummy, CPL_TYPE_DOUBLE, &outimagelist, &outhdr);
    cpl_test_eq_error(code, QMOST_ERROR_DUMMY);
    cpl_test_null(outimagelist);
    cpl_test_null(outhdr);

    cpl_frame_delete(frame);
    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_load_master_table
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_load_master_table(void)
{
    cpl_frame *frame = NULL;
    const char *extname = "TABLE";
    const char *extdummy = "DUMMY";
    cpl_image *inimage = NULL;
    cpl_table *intable = NULL;
    cpl_table *outtable = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_propertylist *outhdr = NULL;
    char tmpfile[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    /* Empty file (not valid FITS) */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* NULL inputs */
    code = qmost_load_master_table(NULL, extname, &outtable, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_table(frame, NULL, &outtable, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_load_master_table(frame, extname, NULL, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set */
    code = qmost_load_master_table(frame, extname, &outtable, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Filename set but not FITS */
    cpl_frame_set_filename(frame, tmpfile);

    code = qmost_load_master_table(frame, extname, &outtable, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    /* FITS but extension doesn't exist */
    inhdr = cpl_propertylist_new();
    cpl_propertylist_save(inhdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(inhdr);

    code = qmost_load_master_table(frame, extname, &outtable, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Populate it with an image (wrong data type), should fail */
    inimage = cpl_image_fill_test_create(256, 512);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", "TEST");

    cpl_image_save(inimage, tmpfile, CPL_TYPE_DOUBLE, inhdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);
    cpl_image_delete(inimage);

    code = qmost_load_master_table(frame, "TEST", &outtable, NULL);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Populate it with an actual table and correct EXTNAME */
    intable = cpl_table_new(1);
    cpl_table_new_column(intable, "TEST", CPL_TYPE_INT);
    inhdr = cpl_propertylist_new();
    cpl_propertylist_update_string(inhdr, "EXTNAME", extname);

    cpl_table_save(intable, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    /* Also make a dummy */
    cpl_propertylist_update_string(inhdr, "EXTNAME", extdummy);
    cpl_propertylist_update_bool(inhdr, "ESO DRS DUMMY", 1);

    cpl_table_save(intable, NULL, inhdr, tmpfile, CPL_IO_EXTEND);

    cpl_propertylist_delete(inhdr);

    /* Should now work, with or without header arg */
    code = qmost_load_master_table(frame, extname, &outtable, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outtable);

    cpl_test_eq(cpl_table_compare_structure(intable, outtable), 0);

    cpl_table_delete(outtable);

    code = qmost_load_master_table(frame, extname, &outtable, &outhdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(outtable);
    cpl_test_nonnull(outhdr);

    cpl_test_eq(cpl_table_compare_structure(intable, outtable), 0);

    cpl_table_delete(outtable);
    outtable = NULL;

    cpl_propertylist_delete(outhdr);
    outhdr = NULL;

    cpl_table_delete(intable);
    intable = NULL;

    /* Attempting to load a dummy should raise an error */
    code = qmost_load_master_table(frame, extdummy, &outtable, &outhdr);
    cpl_test_eq_error(code, QMOST_ERROR_DUMMY);
    cpl_test_null(outtable);
    cpl_test_null(outhdr);

    cpl_frame_delete(frame);
    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_load_master_table
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_copy_sensfunc(void)
{
    cpl_frame *science_frame = NULL;
    cpl_frame *sensfunc_frame = NULL;
    cpl_propertylist *raw_hdr = NULL;
    cpl_propertylist *ext_hdr = NULL;
    double delwave = 0.2;
    double gain = 2.0;
    double exptime = 4.0;

    char science_filename[] = "test-qmost_utils_XXXXXX";
    char sensfunc_filename[] = "test-qmost_utils_XXXXXX";
    int fd = -1;
    cpl_error_code code;

    cpl_image *sens_image = NULL;
    cpl_propertylist *sens_hdr = NULL;

    cpl_image *test_image = NULL;
    cpl_propertylist *test_hdr = NULL;

    /* Create empty FITS files */
    fd = mkstemp(science_filename);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    fd = mkstemp(sensfunc_filename);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create frames */
    science_frame = cpl_frame_new();
    cpl_frame_set_filename(science_frame, science_filename);

    sensfunc_frame = cpl_frame_new();
    cpl_frame_set_filename(sensfunc_frame, sensfunc_filename);

    /* Create empty headers */
    raw_hdr = cpl_propertylist_new();
    ext_hdr = cpl_propertylist_new();

    /* Save to create primary HDU in files */
    code = cpl_propertylist_save(raw_hdr, science_filename, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_propertylist_save(raw_hdr, sensfunc_filename, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_copy_sensfunc(NULL, sensfunc_frame,
                               QMOST_ARM_RED, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_copy_sensfunc(science_frame, NULL,
                               QMOST_ARM_RED, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               QMOST_ARM_RED, exptime,
                               NULL, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid arm */
    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               -1, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* CD1_1 missing in extension */
    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               QMOST_ARM_RED, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add it */
    cpl_propertylist_update_double(ext_hdr, "CD1_1", delwave);

    /* GAIN missing in raw */
    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               QMOST_ARM_RED, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add it */
    cpl_propertylist_update_double(raw_hdr, "ESO DET OUT1 GAIN", gain);

    /* Can't find sensfunc extension */
    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               QMOST_ARM_RED, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add it */
    sens_image = cpl_image_new(256, 256, CPL_TYPE_FLOAT);
    cpl_test_nonnull(sens_image);

    cpl_image_fill_window(sens_image, 1, 1, 256, 256,
                          gain * exptime * delwave);

    sens_hdr = cpl_propertylist_new();

    cpl_propertylist_update_string(sens_hdr, "EXTNAME",
                                   "sens_RED_DATA");
    cpl_image_save(sens_image, sensfunc_filename, CPL_TYPE_FLOAT,
                   sens_hdr, CPL_IO_EXTEND);

    cpl_propertylist_update_string(sens_hdr, "EXTNAME",
                                   "sens_GREEN_DATA");
    cpl_image_save(sens_image, sensfunc_filename, CPL_TYPE_FLOAT,
                   sens_hdr, CPL_IO_EXTEND);

    cpl_propertylist_update_string(sens_hdr, "EXTNAME",
                                   "sens_BLUE_DATA");
    cpl_image_save(sens_image, sensfunc_filename, CPL_TYPE_FLOAT,
                   sens_hdr, CPL_IO_EXTEND);

    cpl_propertylist_delete(sens_hdr);
    sens_hdr = NULL;

    cpl_image_delete(sens_image);
    sens_image = NULL;

    /* This should work */
    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               QMOST_ARM_RED, exptime,
                               raw_hdr, ext_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    test_image = cpl_image_load(science_filename, CPL_TYPE_FLOAT, 0, 1);
    cpl_test_nonnull(test_image);

    qmost_test_image_float_abs(test_image, 1.0, 0, FLT_EPSILON);

    cpl_image_delete(test_image);
    test_image = NULL;

    test_hdr = cpl_propertylist_load(science_filename, 1);

    cpl_test_eq_string(cpl_propertylist_get_string(test_hdr, "EXTNAME"),
                       "RED_SENSFUNC");

    cpl_propertylist_delete(test_hdr);
    test_hdr = NULL;

    /* Test dummy on green */
    code = qmost_copy_sensfunc(science_frame, sensfunc_frame,
                               QMOST_ARM_GREEN, exptime,
                               raw_hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    test_image = cpl_image_load(science_filename, CPL_TYPE_FLOAT, 0, 2);
    cpl_test_nonnull(test_image);

    qmost_test_image_float_abs(test_image, 0.2, 0, FLT_EPSILON);

    cpl_image_delete(test_image);
    test_image = NULL;

    test_hdr = cpl_propertylist_load(science_filename, 2);

    cpl_test_eq_string(cpl_propertylist_get_string(test_hdr, "EXTNAME"),
                       "GREEN_SENSFUNC");

    cpl_propertylist_delete(test_hdr);
    test_hdr = NULL;

    /* Clean up */
    cpl_propertylist_delete(raw_hdr);
    cpl_propertylist_delete(ext_hdr);

    cpl_frame_delete(science_frame);
    unlink(science_filename);

    cpl_frame_delete(sensfunc_frame);
    unlink(sensfunc_filename);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_utils module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_get_license();
    test_qmost_check_and_set_groups();
    test_qmost_cpl_propertylist_get_double();
    test_qmost_cpl_propertylist_get_float();
    test_qmost_cpl_propertylist_get_int();
    test_qmost_cpl_table_copy_cell();
    test_qmost_cpl_table_get_byte_array();
    test_qmost_cpl_table_get_double_array();
    test_qmost_cpl_table_get_float_array();
    test_qmost_cpl_table_get_polynomial();
    test_qmost_cpl_table_set_byte_array();
    test_qmost_cpl_table_set_double_array();
    test_qmost_cpl_table_set_float_array();
    test_qmost_cpl_table_set_polynomial();
    test_qmost_load_master_mask();
    test_qmost_load_master_image();
    test_qmost_load_master_image_and_var();
    test_qmost_load_master_imagelist();
    test_qmost_load_master_table();
    test_qmost_copy_sensfunc();

    return cpl_test_end(0);
}

/**@}*/
