/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_waveinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_waveinfo_test  Unit test of qmost_waveinfo
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define TOL 1.0e-15
#define FTOL 1.0e-5

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvcreate
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvcreate(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    /* NULL input */
    code = qmost_wvcreate(NULL, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid dimensions */
    code = qmost_wvcreate(&table, -1, 10);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    code = qmost_wvcreate(&table, 7, -1);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Create a simple table */
    code = qmost_wvcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Test it has the right format */
    code = qmost_wvchk(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Delete table */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvopen
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvopen(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    qmost_waveinfo dummy;
    qmost_waveinfo *wv = NULL;
    int nwv = 0;

    /* Empty table */
    table = cpl_table_new(0);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_wvopen(NULL, &nwv, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_wvopen(table, NULL, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_wvopen(table, &nwv, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Attempt to open empty table */
    code = qmost_wvopen(table, &nwv, &wv);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table_delete(table);
    table = NULL;

    /* Create a simple table with the correct format */
    code = qmost_wvcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* And add a dummy entry */
    memset(&dummy, 0, sizeof(dummy));
    code = qmost_wvwrite1(table, 1, dummy);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Correct table format now so should work */
    code = qmost_wvopen(table, &nwv, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(nwv, 1);
    cpl_test_nonnull(wv);

    if(wv) {
        cpl_test_eq(wv[0].live, 0);
    }

    /* Clean up */
    qmost_wvclose(nwv, &wv);

    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvread1
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvread1(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    qmost_waveinfo dummy, wvlive, wv;
    cpl_size zero;
    double value;

    int icol, ncols;
    const char *colname;
    cpl_array *colnames = NULL;
    cpl_table *test_tbl = NULL;

    /* Fill struct with nonsense */
    memset(&wv, 0xa5, sizeof(wv));

    /* Make empty table */
    code = qmost_wvcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_wvread1(NULL, 1, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_wvread1(table, 1, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row number */
    code = qmost_wvread1(table, 0, &wv);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Valid row number beyond end of table */
    code = qmost_wvread1(table, 1, &wv);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    
    /* Now populate table and run again, should work this time */
    memset(&dummy, 0, sizeof(dummy));
    code = qmost_wvwrite1(table, 1, dummy);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = qmost_wvread1(table, 1, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(wv.live, 0);

    /* Add a live fibre */
    wvlive.specnum = 2;
    wvlive.live = 1;
    wvlive.nord = 0;
    wvlive.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(wvlive.coefs, &zero, 7);
    wvlive.xref = 3;
    wvlive.ngood = 1;
    wvlive.medresid = 4;
    wvlive.fit_rms = 5;
    wvlive.wave1 = 6;
    wvlive.waven = 7;
    wvlive.dwave = 8;
    wvlive.nlines = 1;
    wvlive.xpos = cpl_malloc(sizeof(double));
    wvlive.xpos[0] = 9;
    wvlive.fwhm = cpl_malloc(sizeof(float));
    wvlive.fwhm[0] = 10;
    wvlive.wave_calc = cpl_malloc(sizeof(double));
    wvlive.wave_calc[0] = 11;
    wvlive.wave_true = cpl_malloc(sizeof(double));
    wvlive.wave_true[0] = 12;
    wvlive.fit_flag = cpl_malloc(sizeof(unsigned char));
    wvlive.fit_flag[0] = 1;
    wvlive.wave_cor = cpl_malloc(sizeof(double));
    wvlive.wave_cor[0] = 13;
    wvlive.peak = cpl_malloc(sizeof(float));
    wvlive.peak[0] = 14;
    wvlive.contrast = cpl_malloc(sizeof(float));
    wvlive.contrast[0] = 15;

    code = qmost_wvwrite1(table, 2, wvlive);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* and read it back */
    code = qmost_wvread1(table, 2, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(wv.specnum, wvlive.specnum);
    cpl_test_eq(wv.nord, wvlive.nord);
    cpl_test_nonnull(wv.coefs);
    zero = 0;
    value = cpl_polynomial_get_coeff(wv.coefs, &zero);
    cpl_test_abs(value, 7, TOL);
    cpl_test_abs(wv.xref, wvlive.xref, TOL);
    cpl_test_eq(wv.ngood, wvlive.ngood);
    cpl_test_abs(wv.medresid, wvlive.medresid, TOL);
    cpl_test_abs(wv.fit_rms, wvlive.fit_rms, TOL);
    cpl_test_abs(wv.wave1, wvlive.wave1, TOL);
    cpl_test_abs(wv.waven, wvlive.waven, TOL);
    cpl_test_abs(wv.dwave, wvlive.dwave, TOL);
    cpl_test_eq(wv.nlines, 1);
    cpl_test_nonnull(wv.xpos);
    if(wv.xpos) {
        cpl_test_abs(wv.xpos[0], wvlive.xpos[0], TOL);
    }
    cpl_test_nonnull(wv.fwhm);
    if(wv.fwhm) {
        cpl_test_abs(wv.fwhm[0], wvlive.fwhm[0], FTOL);
    }
    cpl_test_nonnull(wv.wave_calc);
    if(wv.wave_calc) {
        cpl_test_abs(wv.wave_calc[0], wvlive.wave_calc[0], TOL);
    }
    cpl_test_nonnull(wv.wave_true);
    if(wv.wave_true) {
        cpl_test_abs(wv.wave_true[0], wvlive.wave_true[0], TOL);
    }
    cpl_test_nonnull(wv.fit_flag);
    if(wv.fit_flag) {
        cpl_test_eq(wv.fit_flag[0], wvlive.fit_flag[0]);
    }
    cpl_test_nonnull(wv.wave_cor);
    if(wv.wave_cor) {
        cpl_test_abs(wv.wave_cor[0], wvlive.wave_cor[0], TOL);
    }
    cpl_test_nonnull(wv.peak);
    if(wv.peak) {
        cpl_test_abs(wv.peak[0], wvlive.peak[0], TOL);
    }
    cpl_test_nonnull(wv.contrast);
    if(wv.contrast) {
        cpl_test_abs(wv.contrast[0], wvlive.contrast[0], TOL);
    }

    cpl_polynomial_delete(wv.coefs);
    cpl_free(wv.xpos);
    cpl_free(wv.fwhm);
    cpl_free(wv.wave_calc);
    cpl_free(wv.wave_true);
    cpl_free(wv.fit_flag);
    cpl_free(wv.wave_cor);
    cpl_free(wv.peak);
    cpl_free(wv.contrast);

    /* Clean up */
    cpl_polynomial_delete(wvlive.coefs);
    cpl_free(wvlive.xpos);
    cpl_free(wvlive.fwhm);
    cpl_free(wvlive.wave_calc);
    cpl_free(wvlive.wave_true);
    cpl_free(wvlive.fit_flag);
    cpl_free(wvlive.wave_cor);
    cpl_free(wvlive.peak);
    cpl_free(wvlive.contrast);

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column reads.  Need to use the live fibre so we do all of
     * the tests. */
    colnames = cpl_table_get_column_names(table);
    ncols = cpl_table_get_ncol(table);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);

        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_wvread1(test_tbl, 2, &wv);
        if(icol >= ncols - 2) {
            /* peak and contrast are optional */
            cpl_test_eq_error(code, CPL_ERROR_NONE);
            cpl_polynomial_delete(wv.coefs);
            cpl_free(wv.xpos);
            cpl_free(wv.fwhm);
            cpl_free(wv.wave_calc);
            cpl_free(wv.wave_true);
            cpl_free(wv.fit_flag);
            cpl_free(wv.wave_cor);
            cpl_free(wv.peak);
            cpl_free(wv.contrast);
        }
        else {
            cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);
        }

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvwrite1
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvwrite1(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    qmost_waveinfo wvlive, wv;
    cpl_size zero;
    double value;

    int icol, ncols;
    const char *colname;
    cpl_array *colnames = NULL;
    cpl_table *test_tbl = NULL;

    const char *dead_colnames[] = { "nord", "xref", "ngood",
                                    "medresid", "fit_rms", "wave1",
                                    "waven", "dwave", "nlines" };

    /* Fill struct with nonsense */
    memset(&wv, 0xa5, sizeof(wv));

    /* Make empty table */
    code = qmost_wvcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_wvwrite1(NULL, 1, wv);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row number */
    code = qmost_wvwrite1(table, 0, wv);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* A live fibre */
    wvlive.specnum = 2;
    wvlive.live = 1;
    wvlive.nord = 0;
    wvlive.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(wvlive.coefs, &zero, 7);
    wvlive.xref = 3;
    wvlive.ngood = 1;
    wvlive.medresid = 4;
    wvlive.fit_rms = 5;
    wvlive.wave1 = 6;
    wvlive.waven = 7;
    wvlive.dwave = 8;
    wvlive.nlines = 1;
    wvlive.xpos = cpl_malloc(sizeof(double));
    wvlive.xpos[0] = 9;
    wvlive.fwhm = cpl_malloc(sizeof(float));
    wvlive.fwhm[0] = 10;
    wvlive.wave_calc = cpl_malloc(sizeof(double));
    wvlive.wave_calc[0] = 11;
    wvlive.wave_true = cpl_malloc(sizeof(double));
    wvlive.wave_true[0] = 12;
    wvlive.fit_flag = cpl_malloc(sizeof(unsigned char));
    wvlive.fit_flag[0] = 1;
    wvlive.wave_cor = cpl_malloc(sizeof(double));
    wvlive.wave_cor[0] = 13;
    wvlive.peak = cpl_malloc(sizeof(float));
    wvlive.peak[0] = 14;
    wvlive.contrast = cpl_malloc(sizeof(float));
    wvlive.contrast[0] = 15;

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column writes. */
    colnames = cpl_table_get_column_names(table);
    ncols = cpl_table_get_ncol(table);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);
        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_wvwrite1(test_tbl, 2, wvlive);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    /* Finally write it properly */
    code = qmost_wvwrite1(table, 2, wvlive);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* and read it back */
    code = qmost_wvread1(table, 2, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(wv.specnum, wvlive.specnum);
    cpl_test_eq(wv.live, wvlive.live);
    cpl_test_nonnull(wv.coefs);
    zero = 0;
    value = cpl_polynomial_get_coeff(wv.coefs, &zero);
    cpl_test_abs(value, 7, TOL);
    cpl_test_abs(wv.xref, wvlive.xref, TOL);
    cpl_test_eq(wv.ngood, wvlive.ngood);
    cpl_test_abs(wv.medresid, wvlive.medresid, TOL);
    cpl_test_abs(wv.fit_rms, wvlive.fit_rms, TOL);
    cpl_test_abs(wv.wave1, wvlive.wave1, TOL);
    cpl_test_abs(wv.waven, wvlive.waven, TOL);
    cpl_test_abs(wv.dwave, wvlive.dwave, TOL);
    cpl_test_eq(wv.nlines, 1);
    cpl_test_nonnull(wv.xpos);
    if(wv.xpos) {
        cpl_test_abs(wv.xpos[0], wvlive.xpos[0], TOL);
    }
    cpl_test_nonnull(wv.fwhm);
    if(wv.fwhm) {
        cpl_test_abs(wv.fwhm[0], wvlive.fwhm[0], FTOL);
    }
    cpl_test_nonnull(wv.wave_calc);
    if(wv.wave_calc) {
        cpl_test_abs(wv.wave_calc[0], wvlive.wave_calc[0], TOL);
    }
    cpl_test_nonnull(wv.wave_true);
    if(wv.wave_true) {
        cpl_test_abs(wv.wave_true[0], wvlive.wave_true[0], TOL);
    }
    cpl_test_nonnull(wv.fit_flag);
    if(wv.fit_flag) {
        cpl_test_eq(wv.fit_flag[0], wvlive.fit_flag[0]);
    }
    cpl_test_nonnull(wv.wave_cor);
    if(wv.wave_cor) {
        cpl_test_abs(wv.wave_cor[0], wvlive.wave_cor[0], TOL);
    }
    cpl_test_nonnull(wv.peak);
    if(wv.peak) {
        cpl_test_abs(wv.peak[0], wvlive.peak[0], TOL);
    }
    cpl_test_nonnull(wv.contrast);
    if(wv.contrast) {
        cpl_test_abs(wv.contrast[0], wvlive.contrast[0], TOL);
    }

    cpl_polynomial_delete(wv.coefs);
    cpl_free(wv.xpos);
    cpl_free(wv.fwhm);
    cpl_free(wv.wave_calc);
    cpl_free(wv.wave_true);
    cpl_free(wv.fit_flag);
    cpl_free(wv.wave_cor);
    cpl_free(wv.peak);
    cpl_free(wv.contrast);

    /* a dead fibre */
    wvlive.specnum = 4;
    wvlive.live = 0;

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column writes. */
    ncols = sizeof(dead_colnames) / sizeof(dead_colnames[0]);

    for(icol = 0; icol < ncols; icol++) {
        colname = dead_colnames[icol];
        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_wvwrite1(test_tbl, 4, wvlive);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    code = qmost_wvwrite1(table, 4, wvlive);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* and read it back */
    code = qmost_wvread1(table, 4, &wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(wv.specnum, wvlive.specnum);
    cpl_test_eq(wv.live, wvlive.live);
    cpl_test_eq(wv.nord, 0);
    cpl_test_null(wv.coefs);
    cpl_test_eq(wv.ngood, 0);
    cpl_test_eq(wv.nlines, 0);
    cpl_test_null(wv.xpos);
    cpl_test_null(wv.fwhm);
    cpl_test_null(wv.wave_calc);
    cpl_test_null(wv.wave_true);
    cpl_test_null(wv.fit_flag);
    cpl_test_null(wv.wave_cor);
    cpl_test_null(wv.peak);
    cpl_test_null(wv.contrast);

    /* Clean up */
    cpl_polynomial_delete(wvlive.coefs);
    cpl_free(wvlive.xpos);
    cpl_free(wvlive.fwhm);
    cpl_free(wvlive.wave_calc);
    cpl_free(wvlive.wave_true);
    cpl_free(wvlive.fit_flag);
    cpl_free(wvlive.wave_cor);
    cpl_free(wvlive.peak);
    cpl_free(wvlive.contrast);

    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvchk
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvchk(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    int i;
    char *colname = NULL;

    /* NULL input */
    code = qmost_wvchk(NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Create an empty table */
    table = cpl_table_new(0);
    cpl_test_nonnull(table);
    
    /* Empty table */
    code = qmost_wvchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Incorrect number of columns */
    code = cpl_table_new_column(table, "BADNAME", CPL_TYPE_INT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    code = qmost_wvchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Correct number of columns but incorrect names */
    for(i = 0; i < 17; i++) {
        colname = cpl_sprintf("BAD%d", i);
        code = cpl_table_new_column(table, colname, CPL_TYPE_INT);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        cpl_free(colname);
    }
    
    code = qmost_wvchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table_delete(table);
    table = NULL;

    /* Proper format */
    code = qmost_wvcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Test it has the right format */
    code = qmost_wvchk(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Delete table */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvmeandisp
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvmeandisp(void)
{
    qmost_waveinfo wv[2];
    double value;

    memset(wv, 0, sizeof(wv));

    wv[0].dwave = 1.0;
    wv[1].dwave = 2.0;

    /* None live */
    value = qmost_wvmeandisp(2, wv);
    cpl_test_abs(value, 0.0, TOL);

    /* One live */
    wv[1].live = 1;

    value = qmost_wvmeandisp(2, wv);
    cpl_test_abs(value, 2.0, TOL);

    /* Both live */
    wv[0].live = 1;

    value = qmost_wvmeandisp(2, wv);
    cpl_test_abs(value, 1.5, TOL);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvendpoints
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvendpoints(void)
{
    qmost_waveinfo wv[2];
    double minwave1, maxwave1, minwaven, maxwaven;

    memset(wv, 0, sizeof(wv));

    wv[0].wave1 = 1.0;
    wv[0].waven = 2.0;
    wv[1].wave1 = 0.0;
    wv[1].waven = 3.0;

    /* One live */
    wv[0].live = 1;

    qmost_wvendpoints(2, wv, &minwave1, &maxwave1, &minwaven, &maxwaven);

    cpl_test_abs(minwave1, 1.0, TOL);
    cpl_test_abs(maxwave1, 1.0, TOL);
    cpl_test_abs(minwaven, 2.0, TOL);
    cpl_test_abs(maxwaven, 2.0, TOL);

    /* Both live */
    wv[1].live = 1;

    qmost_wvendpoints(2, wv, &minwave1, &maxwave1, &minwaven, &maxwaven);

    cpl_test_abs(minwave1, 0.0, TOL);
    cpl_test_abs(maxwave1, 1.0, TOL);
    cpl_test_abs(minwaven, 2.0, TOL);
    cpl_test_abs(maxwaven, 3.0, TOL);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wave_qc
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wave_qc(void)
{
    cpl_table *table = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_error_code code;

    qmost_waveinfo wv;
    cpl_size zero;
    int n;

    /* Make empty table */
    code = qmost_wvcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_wave_qc(NULL, NULL, 0, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_wave_qc(table, NULL, 0, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty table */
    qclist = cpl_propertylist_new();

    code = qmost_wave_qc(table, NULL, 0, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    n = cpl_propertylist_get_int(qclist, "ESO QC WAVE LINES TOT");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(n, 0);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Add a live fibre with wave points flagged bad */
    wv.specnum = 1;
    wv.live = 1;
    wv.nord = 0;
    wv.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(wv.coefs, &zero, 0);
    wv.xref = 0;
    wv.ngood = 1;
    wv.medresid = 0;
    wv.fit_rms = 1.0e-3;
    wv.wave1 = 6000;
    wv.waven = 7000;
    wv.dwave = 1;
    wv.nlines = 1;
    wv.xpos = cpl_calloc(1, sizeof(double));
    wv.fwhm = cpl_calloc(1, sizeof(float));
    wv.wave_calc = cpl_calloc(1, sizeof(double));
    wv.wave_true = cpl_calloc(1, sizeof(double));
    wv.fit_flag = cpl_malloc(sizeof(unsigned char));
    wv.fit_flag[0] = 1;
    wv.wave_cor = cpl_calloc(1, sizeof(double));
    wv.peak = cpl_calloc(1, sizeof(float));
    wv.contrast = cpl_calloc(1, sizeof(float));

    code = qmost_wvwrite1(table, 1, wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_polynomial_delete(wv.coefs);
    cpl_free(wv.xpos);
    cpl_free(wv.fwhm);
    cpl_free(wv.wave_calc);
    cpl_free(wv.wave_true);
    cpl_free(wv.fit_flag);
    cpl_free(wv.wave_cor);
    cpl_free(wv.peak);
    cpl_free(wv.contrast);

    /* Populated, but all bad */
    qclist = cpl_propertylist_new();

    code = qmost_wave_qc(table, NULL, 0, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    n = cpl_propertylist_get_int(qclist, "ESO QC WAVE LINES TOT");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(n, 0);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Add a live fibre with wave points flagged good */
    wv.specnum = 2;
    wv.live = 1;
    wv.nord = 0;
    wv.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(wv.coefs, &zero, 0);
    wv.xref = 0;
    wv.ngood = 1;
    wv.medresid = 0;
    wv.fit_rms = 1.0e-3;
    wv.wave1 = 6000;
    wv.waven = 7000;
    wv.dwave = 1;
    wv.nlines = 1;
    wv.xpos = cpl_calloc(1, sizeof(double));
    wv.fwhm = cpl_calloc(1, sizeof(float));
    wv.wave_calc = cpl_calloc(1, sizeof(double));
    wv.wave_true = cpl_calloc(1, sizeof(double));
    wv.fit_flag = cpl_calloc(1, sizeof(unsigned char));
    wv.wave_cor = cpl_calloc(1, sizeof(double));
    wv.peak = cpl_calloc(1, sizeof(float));
    wv.contrast = cpl_calloc(1, sizeof(float));

    code = qmost_wvwrite1(table, 2, wv);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_polynomial_delete(wv.coefs);
    cpl_free(wv.xpos);
    cpl_free(wv.fwhm);
    cpl_free(wv.wave_calc);
    cpl_free(wv.wave_true);
    cpl_free(wv.fit_flag);
    cpl_free(wv.wave_cor);
    cpl_free(wv.peak);
    cpl_free(wv.contrast);

    /* Populated, should generate some QC */
    qclist = cpl_propertylist_new();

    code = qmost_wave_qc(table, NULL, 0, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    n = cpl_propertylist_get_int(qclist, "ESO QC WAVE LINES TOT");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(n, 1);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Also check comparison to reference */
    qclist = cpl_propertylist_new();

    code = qmost_wave_qc(table, table, 0, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* ...and OB solution */
    qclist = cpl_propertylist_new();

    code = qmost_wave_qc(table, NULL, 1, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC WAVE OFFSET MED"),
                 0,
                 1.0e-3);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Clean up */
    cpl_table_delete(table);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_wvclose
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_wvclose(void)
{
    qmost_waveinfo *wv = NULL;
    int iwv, nwv = 2;

    /* NULL argument */
    qmost_wvclose(0, &wv);

    /* Empty argument, should free "wv" and null it */
    wv = cpl_calloc(nwv, sizeof(qmost_waveinfo));
    qmost_wvclose(nwv, &wv);
    cpl_test_null(wv);

    /* Populated argument, should free internal references as well */
    wv = cpl_calloc(nwv, sizeof(qmost_waveinfo));
    for(iwv = 0; iwv < nwv; iwv++) {
        wv[iwv].coefs = cpl_polynomial_new(1);
        wv[iwv].xpos = cpl_malloc(2 * sizeof(double));
        wv[iwv].fwhm = cpl_malloc(2 * sizeof(float));
        wv[iwv].wave_calc = cpl_malloc(2 * sizeof(double));
        wv[iwv].wave_true = cpl_malloc(2 * sizeof(double));
        wv[iwv].fit_flag = cpl_malloc(2 * sizeof(unsigned char));
        wv[iwv].wave_cor = cpl_malloc(2 * sizeof(double));
        wv[iwv].peak = cpl_malloc(2 * sizeof(float));
        wv[iwv].contrast = cpl_malloc(2 * sizeof(float));
    }

    qmost_wvclose(2, &wv);
    cpl_test_null(wv);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_waveinfo module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_wvcreate();
    test_qmost_wvopen();
    test_qmost_wvread1();
    test_qmost_wvwrite1();
    test_qmost_wvchk();
    test_qmost_wvmeandisp();
    test_qmost_wvendpoints();
    test_qmost_wave_qc();
    test_qmost_wvclose();

    return cpl_test_end(0);
}

/**@}*/
