/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_diffimg.h"
#include "qmost_imcombine_lite.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_bias_combine"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_bias_combine_description[] =
    "Process a set of bias frames to form a master bias.  The input\n"
    "exposures are overscan corrected, trimmed and then stacked.\n"
    "The median and rms background levels are computed in both the\n"
    "overscan region and the light sensitive part of the detector.\n\n"
    "Optionally, the generated master calibration can be compared\n"
    "to a reference to look for statistical or spatial variations\n"
    "in the bias.\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                  Req/Opt? Tag\n"
    "---------------------------- -------- --------------------\n"
    "Bias frames                  Required " QMOST_RAW_BIAS "\n"
    "Master bad pixel mask        Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Reference bias frame         Optional " QMOST_CALIB_REFERENCE_BIAS "\n"
    "\n"
    "Outputs:\n\n"
    "Description                  Tag\n"
    "---------------------------- -----------------\n"
    "Master bias frame            " QMOST_PRO_MASTER_BIAS "\n"
    "or\n"
    "Reference bias frame         " QMOST_CALIB_REFERENCE_BIAS "\n"
    "\n"
    "The type of output depends on the recipe parameter reference.\n"
    "\n"
    "Additional outputs if reference frame is given:\n\n"
    "Description                  Tag\n"
    "---------------------------- -----------------\n"
    "Master bias comparison       " QMOST_PRO_DIFFIMG_BIAS "\n"
    "\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the master bias frame\n"
    "for LRS-A would be QMOST_" QMOST_PRO_MASTER_BIAS "_LRS-A.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_bias_combine,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Process a set of bias frames to form a master bias.",
                  qmost_bias_combine_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_bias_combine    qmost_bias_combine
 *
 * @brief    Process a set of bias frames to form a master bias.
 *
 * @par Name:
 *   qmost_bias_combine
 * @par Purpose:
 *   Process a set of bias frames to form a master bias.  The input
 *   exposures are overscan corrected, trimmed and then stacked.
 *   The median and rms background levels are computed in both the
 *   overscan region and the light sensitive part of the detector.
 *   Optionally, the generated master calibration can be compared
 *   to a reference to look for statistical or spatial variations
 *   in the bias.
 * @par Type:
 *   Detector calibration
 * @par Parameters:
 *   - @b imcombine.combtype (int, default 1):
 *     Determines the type of combination that is done to form the
 *     stacked bias frame.  Can take the following values:
 *       - (1): The output pixels are kappa sigma clipped means of
 *              the input pixels.
 *       - (2): The output pixels are medians of the input pixels.
 *   - @b imcombine.scaletype (int, default 1):
 *     Scaling to use when combining multiple images to form the
 *     stacked bias frame.  Can take the following values:
 *       - (0): No biasing or scaling is done.
 *       - (1): An additive offset is applied to remove any
 *              difference in background levels.
 *       - (2): A multiplicative scaling is applied to remove any
 *              difference in background levels.
 *       - (3): Images are scaled by their relative exposure times
 *              and an additive offset is applied to remove any
 *              remaining difference in background levels.
 *   - @b imcombine.xrej (bool, default false):
 *     If set, then an extra rejection cycle will be run.
 *   - @b imcombine.thresh (float, default 5.0):
 *     The rejection threshold for clipping during the combination in
 *     units of the background sigma.
 *   - @b diffimg.ncells (int, default 256):
 *     If a difference image is being formed between the current
 *     combined bias frame and a library reference bias product, then
 *     this is the number of cells in which to divide the image to
 *     compute statistics.  The value should be a power of 2 up to and
 *     including 256.
 *   - @b reference (bool, default false):
 *     Emit a reference frame rather than a master frame.  Caution:
 *     the resulting frame must be inspected and approved before being
 *     allowed to enter the calibration database or used as a
 *     reference.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b BIAS: Raw bias frames.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b REFERENCE_BIAS (optional): A library reference master bias
 *         frame.  If given, then the new master bias produced by the
 *         recipe will be compared to this reference and additional QC
 *         headers giving the results of the comparison are written to
 *         the output files.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b MASTER_BIAS: The master bias frame, if recipe parameter
 *          reference is false.
 *     - @b REFERENCE_BIAS: The new reference bias frame, if recipe
 *          parameter reference is true.
 *     - @b DIFFIMG_BIAS: If a REFERENCE_BIAS is included in the
 *          inputs, this output is the difference image formed by
 *          subtracting the reference from the new master bias.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     master bias frame for LRS-A would be @c
 *     QMOST_MASTER_BIAS_LRS-A.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean residual background level
 *     in the illuminated region of the overscan subtracted frames
 *     that were combined.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>COMPARE MED</b> (ADU): The median of the pixels in the
 *     difference image.
 *   - <b>COMPARE RMS</b> (ADU): The robustly-estimated RMS of the
 *     pixels in the difference image.
 *   - <b>COMPARE BIN MEAN</b> (ADU): The mean over all of the cells
 *     of the median of the pixels in each cell.
 *   - <b>COMPARE BIN RMS</b> (ADU): The RMS over all of the cells of
 *     the median of the pixels in each cell.
 *   - <b>COMPARE BIN MIN</b> (ADU): The minimum over all of the cells
 *     of the median of the pixels in each cell.
 *   - <b>COMPARE BIN MAX</b> (ADU): The maximum over all of the cells
 *     of the median of the pixels in each cell.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No raw bias frames in the input frameset.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 * @par Non-Fatal Error Conditions:
 *   - Missing reference bias frame (no comparison is done).
 *   - Missing bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 *   - Dummy extension in reference file.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 *
 *   MASTER_BPM -> qmost_ccdproc_and_combine [style="dashed"];
 *   REFERENCE_BIAS -> qmost_diffimg [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     BIAS -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> MASTER_BIAS;
 *   }
 *
 *   qmost_ccdproc_and_combine -> qmost_diffimg;
 *
 *   qmost_diffimg -> DIFFIMG_BIAS;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_diffimg -> QC1;
 *
 *   BIAS [shape="box" fillcolor="#eeeeee"]
 *   MASTER_BPM [shape="box" fillcolor="#fff5ce"]
 *   REFERENCE_BIAS [shape="box" fillcolor="#fff5ce"]
 *   MASTER_BIAS [shape="box" fillcolor="#fff5ce"]
 *   DIFFIMG_BIAS [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_bias_combine(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    const cpl_frame *master_bpm_frame = NULL;
    const cpl_frame *reference_bias_frame = NULL;

    const cpl_frame *master_bias_frame = NULL;
    const char *master_bias_filename = NULL;
    const cpl_frame *diffimg_bias_frame = NULL;
    const char *diffimg_bias_filename = NULL;

    cpl_frameset *biases_raw = NULL;
    int iin, nin, nfiles;

    const cpl_frame *this_frame;
    const char *this_tag;

    const cpl_frame *first_bias_frame;
    const char *first_bias_filename = NULL;
    int spec;

    const cpl_parameter *par;
    int combtype = QMOST_MEANCALC;
    int scaletype = 1;
    int xrej = 0;
    double combthresh = 5.0;
    int ncells = 256;
    int make_reference = 0;

    cpl_size raw_extension;
    int arm, detlive;
    const char *arm_extname;

    cpl_mask *master_bpm = NULL;

    cpl_propertylist *bias_hdr = NULL;

    cpl_propertylist *qclist = NULL;

    cpl_image *bias_img = NULL;
    cpl_image *bias_var = NULL;

    cpl_image *reference_bias_img = NULL;
    cpl_image *diff_img = NULL;
    cpl_errorstate prestate;

#undef TIDY
#define TIDY                                    \
    if(biases_raw != NULL) {                    \
        cpl_frameset_delete(biases_raw);        \
        biases_raw = NULL;                      \
    }                                           \
    if(master_bpm != NULL) {                    \
        cpl_mask_delete(master_bpm);            \
        master_bpm = NULL;                      \
    }                                           \
    if(bias_hdr != NULL) {                      \
        cpl_propertylist_delete(bias_hdr);      \
        bias_hdr = NULL;                        \
    }                                           \
    if(qclist != NULL) {                        \
        cpl_propertylist_delete(qclist);        \
        qclist = NULL;                          \
    }                                           \
    if(bias_img != NULL) {                      \
        cpl_image_delete(bias_img);             \
        bias_img = NULL;                        \
    }                                           \
    if(bias_var != NULL) {                      \
        cpl_image_delete(bias_var);             \
        bias_var = NULL;                        \
    }                                           \
    if(reference_bias_img != NULL) {            \
        cpl_image_delete(reference_bias_img);   \
        reference_bias_img = NULL;              \
    }                                           \
    if(diff_img != NULL) {                      \
        cpl_image_delete(diff_img);             \
        diff_img = NULL;                        \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);
    reference_bias_frame = cpl_frameset_find_const(frameset,
                                                   QMOST_CALIB_REFERENCE_BIAS);

    /* Create frameset of inputs */
    biases_raw = cpl_frameset_new();

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_BIAS)) {
            cpl_frameset_insert(biases_raw,
                                cpl_frame_duplicate(this_frame));
        }
    }

    /* Check we have some */
    nfiles = cpl_frameset_get_size(biases_raw);
    if(nfiles < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_RAW_BIAS);
    }

    /* Retrieve imcombine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.combtype");
    if(par != NULL)
        combtype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.scaletype");
    if(par != NULL)
        scaletype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.xrej");
    if(par != NULL)
        xrej = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.thresh");
    if(par != NULL)
        combthresh = cpl_parameter_get_double(par);

    /* Retrieve diffimg parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".diffimg.ncells");
    if(par != NULL)
        ncells = cpl_parameter_get_int(par);

    /* Reference parameter */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".reference");
    if(par != NULL)
        make_reference = cpl_parameter_get_bool(par);

    /* Get first file to use as basis for output */
    first_bias_frame = cpl_frameset_get_position(biases_raw, 0);
    if(first_bias_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first bias frame "
                                     "from SOF");
    }

    /* Get filename of first bias frame */
    first_bias_filename = cpl_frame_get_filename(first_bias_frame);
    if(first_bias_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first bias frame");
    }

    /* Extract the primary FITS header of the reference */
    bias_hdr = cpl_propertylist_load(first_bias_filename, 0);
    if(bias_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(bias_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    cpl_propertylist_delete(bias_hdr);
    bias_hdr = NULL;

    /* Create outputs */
    master_bias_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        make_reference ?
        QMOST_CALIB_REFERENCE_BIAS :
        QMOST_PRO_MASTER_BIAS,
        CPL_FRAME_TYPE_TABLE,
        NULL);
    if(master_bias_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     make_reference ?
                                     QMOST_CALIB_REFERENCE_BIAS :
                                     QMOST_PRO_MASTER_BIAS);
    }

    master_bias_filename = cpl_frame_get_filename(master_bias_frame);

    if(reference_bias_frame != NULL) {
        diffimg_bias_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PRO_DIFFIMG_BIAS,
            CPL_FRAME_TYPE_TABLE,
            NULL);
        if(diffimg_bias_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PRO_DIFFIMG_BIAS);
        }
        
        diffimg_bias_filename = cpl_frame_get_filename(diffimg_bias_frame);
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        bias_hdr = cpl_propertylist_load(
            first_bias_filename,
            raw_extension);
        if(bias_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(bias_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(bias_hdr, &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            /* Detector is not live, write out dummy HDUs to products
             * and move on. */
            if(reference_bias_frame != NULL) {
                if(qmost_dfs_save_image_extension(diffimg_bias_frame,
                                                  bias_hdr,
                                                  arm_extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy bias "
                                                 "difference image %s[%s]",
                                                 diffimg_bias_filename,
                                                 arm_extname);
                }
            }

            if(qmost_dfs_save_image_and_var(master_bias_frame,
                                            bias_hdr,
                                            arm_extname,
                                            NULL,
                                            NULL,
                                            NULL,
                                            CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy master "
                                             "bias frame %s[%s]",
                                             master_bias_filename,
                                             arm_extname);
            }

            cpl_propertylist_delete(bias_hdr);
            bias_hdr = NULL;

            continue;
        }

        cpl_msg_info(cpl_func,
                     "Processing extension %lld = %s",
                     raw_extension, arm_extname);
        cpl_msg_indent_more();

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(master_bpm_frame != NULL) {
            if(qmost_load_master_mask(master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* CCD processing and stacking */
        if(qmost_ccdproc_and_combine(biases_raw,
                                     raw_extension,
                                     master_bpm,
                                     NULL,
                                     NULL,
                                     NULL,
                                     NULL,
                                     NULL,
                                     NULL,
                                     0,
                                     NULL,
                                     1,
                                     NULL,
                                     QMOST_LIN_NEVER,
                                     combtype,
                                     scaletype,
                                     0,
                                     xrej,
                                     combthresh,
                                     &bias_img,
                                     &bias_var,
                                     qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "CCD processing failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(master_bpm) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        /* Difference, if reference was given.  This needs to be done
         * before we save the master so the FITS header includes the
         * QC from the comparison. */
        if(reference_bias_frame != NULL) {
            /* Load reference.  Failure to do so is non-fatal and just
             * produces a warning and a dummy HDU in the difference
             * image file.  This includes the case where the
             * reference was a dummy. */
            prestate = cpl_errorstate_get();

            if(qmost_load_master_image(reference_bias_frame,
                                       arm_extname,
                                       CPL_TYPE_FLOAT,
                                       &reference_bias_img,
                                       NULL) != CPL_ERROR_NONE) {
                cpl_msg_warning(cpl_func,
                                "couldn't load reference bias frame "
                                "%s[%s]: %s",
                                cpl_frame_get_filename(reference_bias_frame),
                                arm_extname,
                                cpl_error_get_message());

                cpl_errorstate_set(prestate);
                reference_bias_img = NULL;
            }

            if(reference_bias_img != NULL) {
                if(qmost_diffimg(bias_img,
                                 reference_bias_img,
                                 ncells,
                                 0,
                                 &diff_img,
                                 qclist) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "couldn't difference "
                                                 "bias frame");
                }
            }
            else {
                /* Produce a dummy HDU */
                diff_img = NULL;
            }

            if(qmost_dfs_save_image_extension(diffimg_bias_frame,
                                              bias_hdr,
                                              arm_extname,
                                              qclist,
                                              diff_img,
                                              CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save bias "
                                             "difference image %s[%s]",
                                             diffimg_bias_filename,
                                             arm_extname);
            }

            cpl_image_delete(diff_img);
            diff_img = NULL;

            cpl_image_delete(reference_bias_img);
            reference_bias_img = NULL;
        }

        /* Save result */
        if(qmost_dfs_save_image_and_var(master_bias_frame,
                                        bias_hdr,
                                        arm_extname,
                                        qclist,
                                        bias_img,
                                        bias_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save master "
                                         "bias frame %s[%s]",
                                         master_bias_filename,
                                         arm_extname);
        }

        /* Clean up */
        cpl_propertylist_delete(bias_hdr);
        bias_hdr = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        cpl_image_delete(bias_img);
        bias_img = NULL;

        cpl_image_delete(bias_var);
        bias_var = NULL;

        cpl_msg_indent_less();
    }

    /* Clean up */
    cpl_frameset_delete(biases_raw);
    biases_raw = NULL;

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_bias_combine_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* imcombine parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.combtype",
                                  CPL_TYPE_INT,
                                  "Stacking method when combining multiple "
                                  "images.  1: mean, 2: median.",
                                  RECIPE_NAME,
                                  QMOST_MEANCALC);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.combtype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.scaletype",
                                  CPL_TYPE_INT,
                                  "Scaling to use when combining multiple "
                                  "images.  0: no biasing or scaling, "
                                  "1: bias by relative offset in background "
                                  "levels, 2: scale by ratio of background "
                                  "levels, 3: scale by exposure time and "
                                  "bias by relative offset in background.",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.scaletype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.xrej",
                                  CPL_TYPE_BOOL,
                                  "Do extra rejection cycle when combining "
                                  "images.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.xrej");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold in terms of "
                                  "background noise for rejection when "
                                  "combining images.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* diffimg parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".diffimg.ncells",
                                  CPL_TYPE_INT,
                                  "Number of smoothing cells to use "
                                  "for difference image statistics.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "diffimg.ncells");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* reference parameter */
    par = cpl_parameter_new_value(RECIPE_NAME".reference",
                                  CPL_TYPE_BOOL,
                                  "Emit reference rather than master frame.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "reference");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/**@}*/
