/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_detector_noise"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_detector_noise_description[] =
    "Use a pair of bias frames and a pair of detector flat frames to\n"
    "measure the detector gain and readout noise. Record the results\n"
    "in a readgain file.\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                 Req/Opt? Tag\n"
    "--------------------------- -------- --------------------\n"
    "Detector flat images        Required " QMOST_RAW_DETECTOR_FLAT "\n"
    "Bias frames                 Optional " QMOST_RAW_BIAS "\n"
    "Master bad pixel mask       Optional " QMOST_CALIB_MASTER_BPM "\n\n"
    "The first two frames of each input type are used.  Any additional\n"
    "inputs will be ignored.\n\n"
    "Outputs:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Readgain file               " QMOST_PRO_READGAIN "\n\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the readgain file for\n"
    "HRS would be QMOST_" QMOST_PRO_READGAIN "_HRS.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_detector_noise,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Measure detector gain and readout noise.",
                  qmost_detector_noise_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_detector_noise    qmost_detector_noise
 *
 * @brief Measure detector gain and readout noise.
 *
 * @par Name:
 *   qmost_detector_noise
 * @par Purpose:
 *   Use a pair of bias frames and a pair of detector flat frames to
 *   measure the detector gain and readout noise. Record the results
 *   in a readgain file.
 * @par Type:
 *   Detector calibration
 * @par Parameters:
 *   None.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b DETECTOR_FLAT: Raw detector flat frames.  Exactly two
 *         frames are required.  Any additional frames given will be
 *         ignored.
 *    - @b BIAS (optional): Raw bias frames.  If given, exactly two
 *         frames are required.  Any additional frames given will be
 *         ignored.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b READGAIN: A file summarising the results of the analysis,
 *          which are given as QC headers (see below).  The file has a
 *          dummy (NAXIS=0) IMAGE extension for each detector since it
 *          only contains FITS headers.
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     readgain file for HRS would be @c QMOST_READGAIN_HRS.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>BIAS MED AMPn</b> (ADU): The median level of the bias in
 *     amplifier n.
 *   - <b>BIAS DIFF AMPn</b> (ADU): The median level of the bias
 *     difference image for amplifier n.
 *   - <b>FLAT MED AMPn</b> (ADU): The median bias-corrected flat
 *     level in amplifier n.
 *   - <b>FLAT DIFF AMPn</b> (ADU): The median level of the flat
 *     difference image for amplifier n.
 *   - <b>CONAD AMPn</b> (e-/ADU): The measured conversion gain for
 *     amplifier n.
 *   - <b>READNOISE AMPn</b> (ADU): The measured readout noise for
 *     amplifier n.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - There are not 2 or more detector flat frames in the input
 *     frameset.
 *   - One bias frame is given in the input frameset (zero or two are
 *     required).
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 * @par Non-Fatal Error Conditions:
 *   - Missing bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 *
 *   MASTER_BPM -> qmost_findgain [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     DETECTOR_FLAT -> qmost_findgain;
 *     BIAS -> qmost_findgain [style="dashed"]
 *
 *     qmost_findgain -> READGAIN;
 *   }
 *
 *   qmost_findgain -> QC1;
 *
 *   DETECTOR_FLAT [shape="box" fillcolor="#eeeeee"]
 *   BIAS [shape="box" fillcolor="#eeeeee"]
 *   MASTER_BPM [shape="box" fillcolor="#fff5ce"]
 *   READGAIN [shape="box" fillcolor="#fff5ce"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_detector_noise(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    cpl_frameset *flats = NULL;
    cpl_frameset *biases = NULL;

    int iin, nin;
    int spec;
    const cpl_frame *this_frame;
    const char *this_tag;

    const cpl_frame *master_bpm_frame = NULL;

    const cpl_frame *frames[4];
    cpl_image *images[4] = { NULL, NULL, NULL, NULL };
    cpl_propertylist *pri_hdr[4] = { NULL, NULL, NULL, NULL };
    cpl_propertylist *ext_hdr[4] = { NULL, NULL, NULL, NULL };
    int iimg, iimgt, nimg = 0;
    int detlive, all_detlive;

    cpl_mask *master_bpm = NULL;

    const cpl_frame *readgain_frame = NULL;
    const char *readgain_filename = NULL;

    const char *filename;

    int raw_extension;
    const char *extname;

    cpl_propertylist *qclist = NULL;

#undef TIDY
#define TIDY                                            \
    if(flats != NULL) {                                 \
        cpl_frameset_delete(flats);                     \
        flats = NULL;                                   \
    }                                                   \
    if(biases != NULL) {                                \
        cpl_frameset_delete(biases);                    \
        biases = NULL;                                  \
    }                                                   \
    for(iimgt = 0; iimgt < nimg; iimgt++) {             \
        if(images[iimgt] != NULL) {                     \
            cpl_image_delete(images[iimgt]);            \
            images[iimgt] = NULL;                       \
        }                                               \
        if(pri_hdr[iimgt] != NULL) {                    \
            cpl_propertylist_delete(pri_hdr[iimgt]);    \
            pri_hdr[iimgt] = NULL;                      \
        }                                               \
        if(ext_hdr[iimgt] != NULL) {                    \
            cpl_propertylist_delete(ext_hdr[iimgt]);    \
            ext_hdr[iimgt] = NULL;                      \
        }                                               \
    }                                                   \
    if(master_bpm != NULL) {                            \
        cpl_mask_delete(master_bpm);                    \
        master_bpm = NULL;                              \
    }                                                   \
    if(qclist != NULL) {                                \
        cpl_propertylist_delete(qclist);                \
        qclist = NULL;                                  \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Create framesets of flats and biases */
    flats = cpl_frameset_new();
    biases = cpl_frameset_new();

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_DETECTOR_FLAT)) {
            cpl_frameset_insert(flats, cpl_frame_duplicate(this_frame));
        }
        else if(!strcmp(this_tag, QMOST_RAW_BIAS)) {
            cpl_frameset_insert(biases, cpl_frame_duplicate(this_frame));
        }
    }

    /* Make sure there are enough flats */
    if(cpl_frameset_get_size(flats) < 2) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have two images tagged %s",
                                     QMOST_RAW_DETECTOR_FLAT);
    }

    /* Assign pointers */
    frames[0] = cpl_frameset_get_position_const(flats, 0);
    frames[1] = cpl_frameset_get_position_const(flats, 1);

    if(cpl_frameset_get_size(biases) < 2) {
        cpl_msg_info(cpl_func, "Using overscan for bias calculation");
        nimg = 2;
    }
    else {
        cpl_msg_info(cpl_func, "Using bias frames for bias calculation");
        nimg = 4;

        frames[2] = cpl_frameset_get_position_const(biases, 0);
        frames[3] = cpl_frameset_get_position_const(biases, 1);
    }

    /* Master bad pixel mask, optional */
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);

    /* Read primary headers */
    for(iimg = 0; iimg < nimg; iimg++) {
        filename = cpl_frame_get_filename(frames[iimg]);

        pri_hdr[iimg] = cpl_propertylist_load(filename, 0);
        if(pri_hdr[iimg] == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS primary header "
                                         "from input file %s",
                                         filename);
        }
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(pri_hdr[0],
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    /* Create output */
    readgain_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_READGAIN,
        CPL_FRAME_TYPE_ANY,
        NULL);
    if(readgain_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_READGAIN);
    }

    readgain_filename = cpl_frame_get_filename(readgain_frame);

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Load images and extension headers */
        all_detlive = 1;

        for(iimg = 0; iimg < nimg; iimg++) {
            filename = cpl_frame_get_filename(frames[iimg]);

            ext_hdr[iimg] = cpl_propertylist_load(filename, raw_extension);
            if(ext_hdr[iimg] == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load FITS image "
                                             "extension header from "
                                             "input %s[%d]",
                                             filename, raw_extension);
            }

            /* Is detector live? */
            if(qmost_pfits_get_detlive(ext_hdr[iimg],
                                       &detlive) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not read live flag for "
                                             "input %s[%d]",
                                             filename, raw_extension);
            }

            if(!detlive) {
                /* Detector is not live */
                all_detlive = 0;
                break;
            }
            
            images[iimg] = cpl_image_load(filename, CPL_TYPE_FLOAT,
                                          0, raw_extension);
            if(images[iimg] == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load image "
                                             "from input %s[%d]",
                                             filename, raw_extension);
            }
        }

        /* Get EXTNAME */
        if(cpl_propertylist_has(ext_hdr[0], "EXTNAME")) {
            extname = cpl_propertylist_get_string(ext_hdr[0],
                                                  "EXTNAME");
        }
        else {
            extname = "READGAIN";
        }

        if(!all_detlive) {
            /* Detector is not live, write out dummy HDUs to products
             * and move on. */
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %d",
                         raw_extension);

            if(qmost_dfs_save_image_extension(readgain_frame,
                                              ext_hdr[0],
                                              extname,
                                              NULL,
                                              NULL,
                                              CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy readgain "
                                             "file extension %s[%d]",
                                             readgain_filename,
                                             raw_extension);
            }

            for(iimg = 0; iimg < nimg; iimg++) {
                if(images[iimg] != NULL) {
                    cpl_image_delete(images[iimg]);
                    images[iimg] = NULL;
                }
                
                if(ext_hdr[iimg] != NULL) {
                    cpl_propertylist_delete(ext_hdr[iimg]);
                    ext_hdr[iimg] = NULL;
                }
            }

            continue;
        }

        /* Load master BPM if given */
        if(master_bpm_frame != NULL) {
            master_bpm = cpl_mask_load(cpl_frame_get_filename(master_bpm_frame),
                                       0,
                                       raw_extension);
            if(master_bpm == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM for "
                                             "extension %d",
                                             raw_extension);
            }
        }

        /* Compute result */
        qclist = cpl_propertylist_new();

        if(qmost_findgain(images, pri_hdr, ext_hdr, nimg, master_bpm,
                          qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "findgain failed for "
                                         "extension %d",
                                         raw_extension);
        }

        /* Save result.  This is a somewhat special case, the readgain
         * file is just a bunch of FITS headers without any image
         * data.  Dummy extensions can be created using cpl_image_save
         * and passing image = NULL so we use the "save image" DFS
         * function to do this. */
        if(qmost_dfs_save_image_extension(readgain_frame,
                                          ext_hdr[0],
                                          extname,
                                          qclist,
                                          NULL,
                                          CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save output readgain "
                                         "file extension %s[%d]",
                                         readgain_filename,
                                         raw_extension);
        }

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        for(iimg = 0; iimg < nimg; iimg++) {
            cpl_image_delete(images[iimg]);
            images[iimg] = NULL;

            cpl_propertylist_delete(ext_hdr[iimg]);
            ext_hdr[iimg] = NULL;
        }

        if(master_bpm != NULL) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }
    }

    /* Clean up */
    cpl_frameset_delete(flats);
    flats = NULL;

    cpl_frameset_delete(biases);
    biases = NULL;

    for(iimg = 0; iimg < nimg; iimg++) {
        cpl_propertylist_delete(pri_hdr[iimg]);
        pri_hdr[iimg] = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_detector_noise_fill_parameterlist(
    cpl_parameterlist *self)
{
    /* This recipe has no parameters, so this is a no-op. */
    if(self != NULL) {
        ;
    }

    return CPL_ERROR_NONE;
}

/**@}*/
