/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_extract_psf.h"
#include "qmost_extract_tram.h"
#include "qmost_ffnorm_fib.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_rebin_spectra.h"
#include "qmost_scattered.h"
#include "qmost_skysub.h"
#include "qmost_spec_combine.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_science_process"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                 Typedefs: Structs and enum types
 */
/*----------------------------------------------------------------------------*/

struct qmost_science_process_calibs {
    const cpl_frame *master_bias_frame;
    const cpl_frame *master_dark_frame;
    const cpl_frame *master_detflat_frame;
    const cpl_frame *master_bpm_frame;
    const cpl_frame *master_lintab_frame;
    const cpl_frame *fibre_trace_frame;
    const cpl_frame *fibre_mask_frame;
    const cpl_frame *fibre_psf_frame;
    const cpl_frame *master_wave_frame;
    const cpl_frame *ob_wave_frame;
    const cpl_frame *master_fibre_flat_frame;
    const cpl_frame *ob_fibre_flat_frame;
    const cpl_frame *sky_fibre_flat_frame;
    const cpl_frame *sensfunc_frame;
};

struct qmost_science_process_params {
    int level;
    int keep;
    const char *swapx_table;
    int scattered_enable;
    int scattered_nbsize;
    int extract_crosstalk;
    float extract_crrej_thr;
    int extract_niter;
    int extract_width;
    int skysub_enable;
    int skysub_neigen;
    float skysub_smoothing;
    float combine_thresh;
};


/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_science_process_get_expinfo(
    cpl_frameset *raw_frames,
    int spec,
    double *exptime,
    double *mjdstart,
    double *mjdmid,
    double *mjdend);

static cpl_error_code qmost_science_process_extract_one(
    cpl_image *processed_img,
    cpl_image *processed_var,
    cpl_propertylist *qclist,
    cpl_table *fibinfo_tbl,
    int spec,
    int arm,
    struct qmost_science_process_calibs calibs,
    struct qmost_science_process_params params,
    const cpl_frame *proc_science_frame,
    const cpl_frame *uncal_science_frame,
    const cpl_frame *eigen_frame,
    const cpl_frame *sky_frame,
    cpl_propertylist *raw_hdr,
    cpl_image **extract_rej,
    cpl_image **wsp_img,
    cpl_image **wsp_var,
    cpl_propertylist **wsp_hdr,
    cpl_image **noss_img,
    cpl_image **noss_var,
    cpl_propertylist **noss_hdr);

static cpl_error_code qmost_science_process_zmag_qc(
    cpl_image *spec_img,
    cpl_image *spec_var,
    cpl_propertylist *spec_hdr,
    cpl_table *fibinfo_tbl,
    int arm,
    double exptime,
    cpl_propertylist *qclist);

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_science_process_description[] =
    "Raw science images are debiased, trimmed, dark corrected, linearity\n"
    "corrected, and detector flat field corrected.  Spectra are then\n"
    "extracted, wavelength calibrated, and flat fielded using fibre flat\n"
    "fields.  The resulting spectra are combined with a static\n"
    "sensitivity function and emitted into a 2D \"ANCILLARY.MOSSPECTRA\"\n"
    "format output file with each row of the 2D image array containing\n"
    "the extracted spectrum for a single fibre.  Optionally, the\n"
    "extracted spectra can also be corrected for sky background.\n\n"
    "All raw images passed in a single SOF must be from the same OB, and\n"
    "must match in terms of spectrograph and fibre configuration.  OB\n"
    "level stacking is supported and will be invoked automatically if\n"
    "multiple raw images are given.  These are extracted separately and\n"
    "stacked after spectral extraction, wavelength calibration, fibre\n"
    "flat field correction and sky subtraction (if enabled).\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                 Req/Opt? Tag\n"
    "--------------------------- -------- --------------------\n"
    "Raw images                  Required " QMOST_RAW_SCIENCE "\n"
    "Master bias frame           Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame           Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master detector flat        Optional " QMOST_PRO_MASTER_DETECTOR_FLAT "\n"
    "Master bad pixel mask       Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Master linearity table      Optional " QMOST_PRO_LINEARITY "\n"
    "Master fibre trace table    Required " QMOST_PRO_FIBRE_TRACE "\n"
    "Master fibre mask           Required " QMOST_PRO_FIBRE_MASK "\n"
    "Master PSF                  Req/Opt  " QMOST_PRO_MASTER_PSF "\n"
    "Master wavelength solution  Required " QMOST_PRO_MASTER_WAVE "\n"
    "OB wavelength solution      Optional " QMOST_PRO_OB_WAVE "\n"
    "Master fibre flat           Optional " QMOST_PRO_MASTER_FIBRE_FLAT "\n"
    "OB fibre flat               Optional " QMOST_PRO_OB_FIBRE_FLAT "\n"
    "Twilight fibre flat         Optional " QMOST_PRO_SKY_FIBRE_FLAT "\n"
    "Sensitivity function        Optional " QMOST_CALIB_SENSITIVITY "\n"
    "\n"
    "Note: Master PSF is only required for QC1 (--level=1).\n"
    "\n"
    "Outputs:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Extracted science spectra   " QMOST_PRO_SCIENCE "\n"
    "\n"
    "Optional processed outputs if keep=true:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Processed 2D image          " QMOST_PROC_SCIENCE "\n"
    "Uncalibrated ext. spec.     " QMOST_PRO_UNCALIBRATED_SCIENCE "\n"
    "Cosmic ray mask             " QMOST_PRO_CRMASK "\n"
    "Sky eigenvectors            " QMOST_PRO_EIGEN "\n"
    "Sky diagnostics file        " QMOST_PRO_SKY "\n"
    "\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the extracted science\n"
    "spectrum file for LRS-B would be QMOST_" QMOST_PRO_SCIENCE "_LRS-B.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_science_process,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Extract science spectra.",
                  qmost_science_process_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_science_process    qmost_science_process
 *
 * @brief Extract science spectra.
 *
 * @par Name:
 *   qmost_science_process
 * @par Purpose
 *   Raw science images are debiased, trimmed, dark corrected,
 *   linearity corrected, and detector flat field corrected.  Spectra
 *   are then extracted, wavelength calibrated, and flat fielded using
 *   fibre flat fields.  The resulting spectra are combined with a
 *   static sensitivity function and emitted into a 2D
 *   "ANCILLARY.MOSSPECTRA" format output file with each row of the 2D
 *   image array containing the extracted spectrum for a single fibre.
 *   Optionally, the extracted spectra can also be corrected for sky
 *   background.  All raw images passed in a single SOF must be from
 *   the same OB, and must match in terms of spectrograph and fibre
 *   configuration.  OB level stacking is supported and will be
 *   invoked automatically if multiple raw images are given.  These
 *   are extracted separately and stacked after spectral extraction,
 *   wavelength calibration, fibre flat field correction and sky
 *   subtraction (if enabled).
 * @par Type:
 *   Science
 * @par Parameters:
 *   - @b level (int, default 1):
 *     Determines the level of the analysis:
 *       - (0): QC0 analysis.
 *       - (1): QC1 analysis.
 *   - @b keep (bool, default false):
 *     If this flag is set, optional products generated during
 *     execution of the recipe are also saved.  In the case of
 *     qmost_science_process, this saves the processed science
 *     image, cosmic ray mask, and sky diagnostics files if sky
 *     subtraction is enabled.
 *   - @b ccdproc.swapx (string, default 100010101):
 *     String of 0 or 1 specifying for each arm of each spectrograph
 *     if we should flip the x axis to correct the wavelength order of
 *     the spectral axis.
 *   - @b scattered.enable (bool, default true):
 *     Enable scattered light removal.
 *   - @b scattered.nbsize (int, default 256):
 *     Size of the background smoothing cells for determining
 *     scattered light in pixels.
 *   - @b extract.crosstalk (bool, default true):
 *     Correct for crosstalk between fibres using the overlap of the
 *     fibre PSFs if true.  Can be set to false to disable crosstalk
 *     correction.
 *   - @b extract.cr_thresh (float, default 5.0):
 *     Rejection threshold as number of sigma used to define cosmic
 *     ray hits during PSF extraction.
 *   - @b extract.niter (int, default 3):
 *     Number of rejection iterations for PSF spectral extraction.
 *   - @b extract.width (int, default 6):
 *     Boxcar width in pixels over which the summation will be done
 *     at each spectral pixel during extraction (QC0 only).
 *   - @b skysub.enable (int)
 *     If true, enable sky subtraction.  If false, disable sky
 *     subtraction.  If not specified or negative, selected based on
 *     the QC level, where the default is to enable sky subtraction
 *     for QC1 analysis and disable it for QC0 analysis.
 *   - @b skysub.neigen (int)
 *     The maximum number of eigenvectors to use for PCA analysis in
 *     sky subtraction.  If not specified or negative, a suitable
 *     default value is chosen automatically.
 *   - @b skysub.smoothing (float, default 50.0)
 *     Smoothing box in Angstroms to be used in smoothing the fitted
 *     sky continuum.
 *   - @b combine.rej_thresh (float, default 5.0)
 *     Threshold for outlier rejection when combining spectra.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b OBJECT: Raw science frames.  If more than one, they must
 *         all be from the same OB.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_DETECTOR_FLAT (optional): Master detector flat.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b LINEARITY (optional): Master linearity table.
 *    - @b FIBRE_TRACE (required): Trace table.
 *    - @b FIBRE_MASK (required): Fibre mask.
 *    - @b MASTER_WAVE (required): Master wavelength solution.
 *    - @b OB_WAVE (optional): OB-level wavelength solution.
 *    - @b MASTER_FIBRE_FLAT (optional): Master fibre flat.
 *    - @b OB_FIBRE_FLAT (optional): OB-level fibre flat.
 *    - @b SKY_FIBRE_FLAT (optional): Twilight fibre flat.
 *    - @b SENSITIVITY (optional): Static sensitivity function file.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b SCIENCE: The extracted, wavelength calibrated science
 *          spectra and variance arrays.
 *
 *   - The following intermediate product files are optional and are
 *     only emitted if keep=true is specified in the recipe parameters.
 *     - @b PROC_SCIENCE: Processed 2D science image (prior to
 *          spectral extraction).  Can be used to assess the quality 
 *          of the initial CCD processing, such as bias, dark,
 *          detector flat field correction, scattered light
 *          correction.
 *     - @b UNCALIBRATED_SCIENCE: The extracted science spectra prior
 *          to wavelength calibration, and without flat fielding or
 *          sky subtraction.
 *     - @b COSMIC_RAY_MASK: An optional mask showing which pixels of
 *          the processed 2D science image in the previous output were
 *          flagged as cosmic rays during PSF extraction (QC1 only).
 *          This is an integer image using the enumerated constants
 *          QMOST_WMASK_*, the values of which can be found in the
 *          documentation for qmost_constants.
 *     - @b EIGEN: Optional sky subtraction diagnostics file giving
 *          the eigenvectors and eigenvalues used in the PCA
 *          solution.
 *     - @b SKY: Optional sky subtraction diagnostics file giving
 *          information about the derived sky spectrum and sky line
 *         residuals in each science fibre.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     extracted science spectrum file for LRS-B would be @c
 *     QMOST_SCIENCE_LRS-B.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>EXT FLUX MED</b> (ADU): The median of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX RMS</b> (ADU): The robustly-estimated RMS of
 *     average extracted flux in each fibre.
 *   - <b>EXT FLUX MIN</b> (ADU): The minimum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MINSPC</b>: The fibre with the minimum extracted
 *     flux.
 *   - <b>EXT FLUX MAX</b> (ADU): The maximum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MAXSPC</b>: The fibre with the maximum extracted
 *     flux.
 *   - <b>EXT SN MED</b>: The median of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN RMS</b>: The robustly-estimated RMS of the average
 *     signal to noise ratio in each fibre.
 *   - <b>EXT SN MIN</b>: The minimum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MINSPC</b>: The fibre with the minimum signal to
 *     noise ratio.
 *   - <b>EXT SN MAX</b>: The maximum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MAXSPC</b>: The fibre with the maximum signal to
 *     noise ratio.
 *   - <b>EXT GOF MED</b>: The median goodness of fit of the profile
 *     in PSF extraction.
 *   - <b>EXT NUM REJECTED</b>: The number of rejected pixels during
 *     PSF extraction.
 *   - <b>SCATTL MED</b> (ADU): The median scattered light level on
 *     the detector outside the region illuminated by the fibres.
 *   - <b>SCATTL RMS</b> (ADU): The robustly-estimated RMS of the
 *     scattered light level on the detector outside the region
 *     illuminated by the fibres.
 *   - <b>SCATTL RESID MED</b> (ADU): The median residual scattered
 *     light background remaining after subtraction.
 *   - <b>SCATTL RESID RMS</b> (ADU): The RMS residual of the
 *     scattered light background remaining after subtraction.
 *   - <b>SKY CONT MED</b> (ADU): Median sky continuum level in the
 *     sky fibres. 
 *   - <b>SKY CONT RMS</b> (ADU): RMS variation in sky level between
 *     the sky fibres. 
 *   - <b>SKY NUM</b>: The number of skies found, including sky fibres
 *     and any weakly exposed targets used to supplement the sky
 *     information for the sky residual PCA.
 *   - <b>SKY NUSED</b>: The number of skies used in the sky residual
 *     PCA.
 *   - <b>SKY RESID MED</b>: The median of the sky emission line
 *     residuals after correction, relative to the expected noise.
 *   - <b>SKY RESID RMS</b>: The RMS variation of the sky emission
 *     line residuals after correction, relative to the expected
 *     noise.
 *   - <b>ZMAG MED</b> (mag): Median magnitude zero point, defined as
 *     the magnitude of a source giving 1 ADU/second total extracted
 *     spectral counts in this arm of the spectrograph.
 *   - <b>ZMAG RMS</b> (mag): Robustly-estimated RMS dispersion of the
 *     magnitude zero point, or equivalently the dispersion in the
 *     difference between a synthetic magnitude computed from the
 *     spectrum and expected magnitude from target catalogue.
 *   - <b>ZMAG NUM</b>: The number of spectra used in the calculation
 *     of the median and RMS magnitude zero point.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No raw science frames in the input frameset.
 *   - Mandatory calibration images/tables not specified in SOF or
 *     unreadable.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 *   - If the trace table for an active detector is empty.
 * @par Non-Fatal Error Conditions:
 *   - No bad pixel mask (all pixels assumed to be good).
 *   - If sky subtraction was requested but there were no usable sky
 *     fibres (no sky subtraction is performed).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 *   - In the EIGEN and SKY output files, if sky subtraction was
 *     requested but there were no usable sky fibres.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 *
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> "qmost_extract_psf\nqmost_extract_tram";
 *   MASTER_PSF -> "qmost_extract_psf\nqmost_extract_tram" [style="dashed"];
 *   MASTER_WAVE -> qmost_rebin_spectra_off;
 *   OB_WAVE -> qmost_rebin_spectra_off [style="dashed"];
 *   MASTER_FIBRE_FLAT -> qmost_ffdiv_fib [style="dashed"];
 *   OB_FIBRE_FLAT -> qmost_obffcor_fib [style="dashed"];
 *   SKY_FIBRE_FLAT -> qmost_obffcor_fib [style="dashed"];
 *   SENSITIVITY -> qmost_copy_sensfunc [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     OBJECT -> qmost_ccdproc;
 *
 *     qmost_ccdproc -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> "qmost_extract_psf\nqmost_extract_tram";
 *     "qmost_extract_psf\nqmost_extract_tram" -> qmost_rebin_spectra_off;
 *     qmost_rebin_spectra_off -> qmost_ffdiv_fib;
 *     qmost_ffdiv_fib -> qmost_obffcor_fib;
 *     qmost_obffcor_fib -> qmost_skysub_vshiftpca;
 *     qmost_skysub_vshiftpca -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_copy_sensfunc;
 *     qmost_copy_sensfunc -> qmost_spec_combine [label=" x nscience"];
 *
 *     qmost_spec_combine -> SCIENCE;
 *   }
 *
 *   qmost_scattered -> PROC_SCIENCE;
 *   "qmost_extract_psf\nqmost_extract_tram" -> "COSMIC_RAY_MASK\nUNCALIBRATED_SCIENCE";
 *   qmost_skysub_vshiftpca -> EIGEN;
 *   qmost_skysub_vshiftpca -> SKY;
 *
 *   qmost_ccdproc -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   "qmost_extract_psf\nqmost_extract_tram" -> QC1;
 *   qmost_extract_qc -> QC1;
 *
 *   OBJECT [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   MASTER_PSF [shape="box" fillcolor="#fff5ce"]
 *   MASTER_WAVE [shape="box" fillcolor="#fff5ce"]
 *   OB_WAVE [shape="box" fillcolor="#fff5ce"]
 *   MASTER_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   OB_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   SKY_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   SENSITIVITY [shape="box" fillcolor="#afd095"]
 *   SCIENCE [shape="box" fillcolor="#b4c7dc"]
 *   PROC_SCIENCE [shape="box" fillcolor="#ffffff"]
 *   "COSMIC_RAY_MASK\nUNCALIBRATED_SCIENCE" [shape="box" fillcolor="#ffffff"]
 *   EIGEN [shape="box" fillcolor="#ffffff"]
 *   SKY [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_science_process(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    struct qmost_science_process_calibs calibs = {
        .master_bias_frame = NULL,
        .master_dark_frame = NULL,
        .master_detflat_frame = NULL,
        .master_bpm_frame = NULL,
        .master_lintab_frame = NULL,
        .fibre_trace_frame = NULL,
        .fibre_mask_frame = NULL,
        .fibre_psf_frame = NULL,
        .master_wave_frame = NULL,
        .ob_wave_frame = NULL,
        .master_fibre_flat_frame = NULL,
        .ob_fibre_flat_frame = NULL,
        .sky_fibre_flat_frame = NULL,
        .sensfunc_frame = NULL
    };

    const cpl_frame *proc_science_frame = NULL;
    const char *proc_science_filename = NULL;
    const cpl_frame *uncal_science_frame = NULL;
    const char *uncal_science_filename = NULL;
    const cpl_frame *crmask_frame = NULL;
    const char *crmask_filename = NULL;
    const cpl_frame *eigen_frame = NULL;
    const cpl_frame *sky_frame = NULL;
    const cpl_frame *science_frame = NULL;
    const char *science_filename = NULL;

    cpl_frameset *raw_frames = NULL;
    int nin, iin;

    const cpl_frame *this_frame;
    const char *this_tag;

    cpl_frame *raw_ref;
    const char *raw_ref_filename;

    const cpl_parameter *par;
    struct qmost_science_process_params params = {
        .level = 1,
        .keep = 0,
        .swapx_table = QMOST_DEFAULT_SWAPX_TABLE,
        .scattered_enable = 1,
        .scattered_nbsize = 256,
        .extract_crosstalk = 1,
        .extract_crrej_thr = 5.0,
        .extract_niter = 3,
        .extract_width = 6,
        .skysub_enable = -1,
        .skysub_neigen = -1,
        .skysub_smoothing = 50.0,
        .combine_thresh = 5.0
    };

    cpl_size raw_extension;
    int spec, arm, detlive;
    double exptime, mjdstart, mjdmid, mjdend;
    const char *arm_extname;
    char *extname = NULL;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_image *master_detflat_img = NULL;
    cpl_image *master_detflat_var = NULL;
    cpl_table *master_lintab = NULL;

    cpl_propertylist *raw_hdr = NULL;

    cpl_propertylist *qclist = NULL;
    cpl_propertylist *qctmp = NULL;
    cpl_propertylist *qcuse;

    cpl_image *processed_img = NULL;
    cpl_image *processed_var = NULL;

    cpl_table *fibinfo_tbl = NULL;
    cpl_table *tmp_fibinfo_tbl = NULL;
    cpl_propertylist *fibinfo_hdr = NULL;

    cpl_propertylist *applist = NULL;

    cpl_image *extract_rej = NULL;
    cpl_image *wsp_img = NULL;
    cpl_image *wsp_var = NULL;
    cpl_propertylist *wsp_hdr = NULL;
    cpl_propertylist *out_hdr = NULL;

    cpl_image *noss_img = NULL;
    cpl_image *noss_var = NULL;
    cpl_propertylist *noss_hdr = NULL;

    cpl_imagelist *wsp_imglist = NULL;
    cpl_imagelist *wsp_varlist = NULL;
    cpl_propertylist **wsp_hdrlist = NULL;
    cpl_imagelist *noss_imglist = NULL;
    cpl_imagelist *noss_varlist = NULL;
    cpl_propertylist **noss_hdrlist = NULL;
    int ifile, ifilet, nfiles = 0;

#undef TIDY
#define TIDY                                                    \
    if(raw_frames) {                                            \
        cpl_frameset_delete(raw_frames);                        \
        raw_frames = NULL;                                      \
    }                                                           \
    if(fibinfo_tbl) {                                           \
        cpl_table_delete(fibinfo_tbl);                          \
        fibinfo_tbl = NULL;                                     \
    }                                                           \
    if(fibinfo_hdr) {                                           \
        cpl_propertylist_delete(fibinfo_hdr);                   \
        fibinfo_hdr = NULL;                                     \
    }                                                           \
    if(applist) {                                               \
        cpl_propertylist_delete(applist);                       \
        applist = NULL;                                         \
    }                                                           \
    if(master_bpm) {                                            \
        cpl_mask_delete(master_bpm);                            \
        master_bpm = NULL;                                      \
    }                                                           \
    if(master_bias_img) {                                       \
        cpl_image_delete(master_bias_img);                      \
        master_bias_img = NULL;                                 \
    }                                                           \
    if(master_bias_var) {                                       \
        cpl_image_delete(master_bias_var);                      \
        master_bias_var = NULL;                                 \
    }                                                           \
    if(master_dark_img) {                                       \
        cpl_image_delete(master_dark_img);                      \
        master_dark_img = NULL;                                 \
    }                                                           \
    if(master_dark_var) {                                       \
        cpl_image_delete(master_dark_var);                      \
        master_dark_var = NULL;                                 \
    }                                                           \
    if(master_detflat_img) {                                    \
        cpl_image_delete(master_detflat_img);                   \
        master_detflat_img = NULL;                              \
    }                                                           \
    if(master_detflat_var) {                                    \
        cpl_image_delete(master_detflat_var);                   \
        master_detflat_var = NULL;                              \
    }                                                           \
    if(master_lintab) {                                         \
        cpl_table_delete(master_lintab);                        \
        master_lintab = NULL;                                   \
    }                                                           \
    if(raw_hdr) {                                               \
        cpl_propertylist_delete(raw_hdr);                       \
        raw_hdr = NULL;                                         \
    }                                                           \
    if(qclist) {                                                \
        cpl_propertylist_delete(qclist);                        \
        qclist = NULL;                                          \
    }                                                           \
    if(qctmp) {                                                 \
        cpl_propertylist_delete(qctmp);                         \
        qctmp = NULL;                                           \
    }                                                           \
    if(processed_img) {                                         \
        cpl_image_delete(processed_img);                        \
        processed_img = NULL;                                   \
    }                                                           \
    if(processed_var) {                                         \
        cpl_image_delete(processed_var);                        \
        processed_var = NULL;                                   \
    }                                                           \
    if(extname) {                                               \
        cpl_free(extname);                                      \
        extname = NULL;                                         \
    }                                                           \
    if(wsp_imglist) {                                           \
        cpl_imagelist_delete(wsp_imglist);                      \
        wsp_imglist = NULL;                                     \
    }                                                           \
    if(wsp_hdrlist != NULL) {                                   \
        for(ifilet = 0; ifilet < nfiles; ifilet++) {            \
            if(wsp_hdrlist[ifilet] != NULL) {                   \
                cpl_propertylist_delete(wsp_hdrlist[ifilet]);   \
            }                                                   \
        }                                                       \
        cpl_free(wsp_hdrlist);                                  \
        wsp_hdrlist = NULL;                                     \
    }                                                           \
    if(wsp_varlist) {                                           \
        cpl_imagelist_delete(wsp_varlist);                      \
        wsp_varlist = NULL;                                     \
    }                                                           \
    if(noss_imglist) {                                          \
        cpl_imagelist_delete(noss_imglist);                     \
        noss_imglist = NULL;                                    \
    }                                                           \
    if(noss_varlist) {                                          \
        cpl_imagelist_delete(noss_varlist);                     \
        noss_varlist = NULL;                                    \
    }                                                           \
    if(noss_hdrlist != NULL) {                                  \
        for(ifilet = 0; ifilet < nfiles; ifilet++) {            \
            if(noss_hdrlist[ifilet] != NULL) {                  \
                cpl_propertylist_delete(noss_hdrlist[ifilet]);  \
            }                                                   \
        }                                                       \
        cpl_free(noss_hdrlist);                                 \
        noss_hdrlist = NULL;                                    \
    }                                                           \
    if(extract_rej) {                                           \
        cpl_image_delete(extract_rej);                          \
        extract_rej = NULL;                                     \
    }                                                           \
    if(wsp_img) {                                               \
        cpl_image_delete(wsp_img);                              \
        wsp_img = NULL;                                         \
    }                                                           \
    if(wsp_var) {                                               \
        cpl_image_delete(wsp_var);                              \
        wsp_var = NULL;                                         \
    }                                                           \
    if(out_hdr) {                                               \
        cpl_propertylist_delete(out_hdr);                       \
        out_hdr = NULL;                                         \
    }                                                           \
    if(noss_img) {                                              \
        cpl_image_delete(noss_img);                             \
        noss_img = NULL;                                        \
    }                                                           \
    if(noss_var) {                                              \
        cpl_image_delete(noss_var);                             \
        noss_var = NULL;                                        \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    calibs.master_bias_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_MASTER_BIAS);
    calibs.master_dark_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_MASTER_DARK);
    calibs.master_detflat_frame = cpl_frameset_find_const(frameset,
                                                          QMOST_PRO_MASTER_DETECTOR_FLAT);
    calibs.master_lintab_frame = cpl_frameset_find_const(frameset,
                                                         QMOST_PRO_LINEARITY);
    calibs.master_bpm_frame = cpl_frameset_find_const(frameset,
                                                      QMOST_CALIB_MASTER_BPM);

    /* Trace table and fibre mask, required */
    calibs.fibre_trace_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_FIBRE_TRACE);
    if(calibs.fibre_trace_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_TRACE);
    }

    calibs.fibre_mask_frame = cpl_frameset_find_const(frameset,
                                                      QMOST_PRO_FIBRE_MASK);
    if(calibs.fibre_mask_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_MASK);
    }

    /* Master PSF, required for PSF extraction but not otherwise so
     * check later once we know if we're using PSF extraction. */
    calibs.fibre_psf_frame = cpl_frameset_find_const(frameset,
                                                     QMOST_PRO_MASTER_PSF);

    /* Master wavelength solution, required */
    calibs.master_wave_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_MASTER_WAVE);
    if(calibs.master_wave_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_MASTER_WAVE);
    }

    /* OB wavelength solution, optional */
    calibs.ob_wave_frame = cpl_frameset_find_const(frameset,
                                                   QMOST_PRO_OB_WAVE);

    /* Master fibre flat frame, optional */
    calibs.master_fibre_flat_frame =
        cpl_frameset_find_const(frameset,
                                QMOST_PRO_MASTER_FIBRE_FLAT);

    /* OB fibre flat frame, optional */
    calibs.ob_fibre_flat_frame =
        cpl_frameset_find_const(frameset,
                                QMOST_PRO_OB_FIBRE_FLAT);

    /* Twilight fibre flat frame, optional */
    calibs.sky_fibre_flat_frame =
        cpl_frameset_find_const(frameset,
                                QMOST_PRO_SKY_FIBRE_FLAT);

    /* Sensitivity function, optional */
    calibs.sensfunc_frame = cpl_frameset_find_const(frameset,
                                                    QMOST_CALIB_SENSITIVITY);

    /* Create frameset of input raw images */
    raw_frames = cpl_frameset_new();  /* can't fail? */

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_SCIENCE)) {
            cpl_frameset_insert(raw_frames, cpl_frame_duplicate(this_frame));
        }
    }

    /* Check we have some */
    nfiles = cpl_frameset_get_size(raw_frames);
    if(nfiles < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_RAW_SCIENCE);
    }

    /* Retrieve QC level parameter */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".level");
    if(par != NULL)
        params.level = cpl_parameter_get_int(par);

    if(params.level < 0 || params.level > 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "unknown QC level: %d", params.level);
    }

    /* Check PSF if we're using PSF extraction */
    if(params.level != 0 && calibs.fibre_psf_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_MASTER_PSF);
    }

    /* Switch to save processed products */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".keep");
    if(par != NULL)
        params.keep = cpl_parameter_get_bool(par);

    /* Retrieve ccdproc parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ccdproc.swapx");
    if(par != NULL)
        params.swapx_table = cpl_parameter_get_string(par);

    /* Retrieve scattered light removal parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.enable");
    if(par != NULL)
        params.scattered_enable = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.nbsize");
    if(par != NULL)
        params.scattered_nbsize = cpl_parameter_get_int(par);

    /* Retrieve extraction parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.crosstalk");
    if(par != NULL)
        params.extract_crosstalk = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.cr_thresh");
    if(par != NULL)
        params.extract_crrej_thr = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.niter");
    if(par != NULL)
        params.extract_niter = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.width");
    if(par != NULL)
        params.extract_width = cpl_parameter_get_int(par);

    /* Retrieve skysub parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".skysub.enable");
    if(par != NULL)
        params.skysub_enable = cpl_parameter_get_int(par);

    if(params.skysub_enable < 0) {
        /* Select suitable default based on QC level: QC0 = 0, QC1 = 1 */
        params.skysub_enable = params.level;
    }

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".skysub.neigen");
    if(par != NULL)
        params.skysub_neigen = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".skysub.smoothing");
    if(par != NULL)
        params.skysub_smoothing = cpl_parameter_get_double(par);

    /* spec_combine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".combine.rej_thresh");
    if(par != NULL)
        params.combine_thresh = cpl_parameter_get_double(par);

    /* Get first frame to use as reference */
    raw_ref = cpl_frameset_get_position(raw_frames, 0);
    if(raw_ref == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first raw frame "
                                     "from SOF");
    }

    /* Get filename of first raw image */
    raw_ref_filename = cpl_frame_get_filename(raw_ref);
    if(raw_ref_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first raw frame");
    }

    /* Extract the primary FITS header of the reference */
    raw_hdr = cpl_propertylist_load(raw_ref_filename,
                                    0);
    if(raw_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(raw_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    /* Get fibinfo from reference */
    if(qmost_fibtabload(raw_ref,
                        spec,
                        &fibinfo_tbl,
                        &fibinfo_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO table "
                                     "from first input file");
    }

    /* If there's no FIBINFO, we can't do sky subtraction */
    if(params.skysub_enable && fibinfo_tbl == NULL) {
        cpl_msg_warning(cpl_func,
                        "Sky subtraction disabled due to no FIBINFO");
        params.skysub_enable = 0;
    }

    /* Read other files to work out total exposure time and midpoint */
    if(qmost_science_process_get_expinfo(raw_frames,
                                         spec,
                                         &exptime,
                                         &mjdstart,
                                         &mjdmid,
                                         &mjdend) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "error loading input file "
                                     "primary HDUs");
    }

    if(fibinfo_tbl != NULL) {
        /* Create pipeline-generated columns */
        if(qmost_fibtab_newcols(fibinfo_tbl,
                                raw_hdr,
                                exptime,
                                mjdmid,
                                &tmp_fibinfo_tbl) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't create FIBINFO table "
                                         "columns");
        }
        
        /* Replace table */
        cpl_table_delete(fibinfo_tbl);
        fibinfo_tbl = tmp_fibinfo_tbl;
    }        

    cpl_propertylist_delete(raw_hdr);
    raw_hdr = NULL;

    /* Adjust the exposure time, start and end to account for
     * stacking, and add phase-3 headers needed in science output. */
    applist = cpl_propertylist_new();

    cpl_propertylist_update_double(applist,
                                   "EXPTIME",
                                   exptime);
    cpl_propertylist_set_comment(applist,
                                 "EXPTIME",
                                 "[s] Effective exposure time");

    cpl_propertylist_update_double(applist,
                                   "TEXPTIME",
                                   exptime);
    cpl_propertylist_set_comment(applist,
                                 "TEXPTIME",
                                 "[s] Total exposure time");

    cpl_propertylist_update_double(applist,
                                   "MJD-OBS",
                                   mjdstart);
    cpl_propertylist_set_comment(applist,
                                 "MJD-OBS",
                                 "Obs start");
    
    cpl_propertylist_update_double(applist,
                                   "MJD-END",
                                   mjdend);
    cpl_propertylist_set_comment(applist,
                                 "MJD-END",
                                 "Obs end");

    cpl_propertylist_update_string(applist,
                                   "PRODCATG",
                                   "ANCILLARY.MOSSPECTRA");
    cpl_propertylist_set_comment(applist,
                                 "PRODCATG",
                                 "Data product category");

    cpl_propertylist_update_string(applist,
                                   "FLUXCAL",
                                   "UNCALIBRATED");
    cpl_propertylist_set_comment(applist,
                                 "FLUXCAL",
                                 "Quality of flux calib: "
                                 "ABSOLUTE or UNCALIBRATED");

    cpl_propertylist_update_string(applist,
                                   "SPECSYS",
                                   "BARYCENT");
    cpl_propertylist_set_comment(applist,
                                 "SPECSYS",
                                 "Frame of reference for spectral coordinates");

    /* Create outputs */
    science_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_SCIENCE,
        CPL_FRAME_TYPE_IMAGE,
        applist);
    if(science_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_SCIENCE);
    }

    science_filename = cpl_frame_get_filename(science_frame);

    cpl_propertylist_delete(applist);
    applist = NULL;

    if(params.keep) {
        proc_science_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PROC_SCIENCE,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(proc_science_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PROC_SCIENCE);
        }

        proc_science_filename = cpl_frame_get_filename(proc_science_frame);

        uncal_science_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PRO_UNCALIBRATED_SCIENCE,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(uncal_science_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PRO_UNCALIBRATED_SCIENCE);
        }
            
        uncal_science_filename = 
            cpl_frame_get_filename(uncal_science_frame);

        if(params.level) {
            crmask_frame = qmost_dfs_setup_product_default(
                frameset,
                parlist,
                RECIPE_NAME,
                spec,
                QMOST_PRO_CRMASK,
                CPL_FRAME_TYPE_IMAGE,
                NULL);
            if(crmask_frame == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't set up output product "
                                             "file for %s",
                                             QMOST_PRO_CRMASK);
            }
            
            crmask_filename = 
                cpl_frame_get_filename(crmask_frame);
        }

        if(params.skysub_enable) {
            eigen_frame = qmost_dfs_setup_product_default(
                frameset,
                parlist,
                RECIPE_NAME,
                spec,
                QMOST_PRO_EIGEN,
                CPL_FRAME_TYPE_IMAGE,
                NULL);
            if(eigen_frame == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't set up output product "
                                             "file for %s",
                                             QMOST_PRO_EIGEN);
            }
            
            sky_frame = qmost_dfs_setup_product_default(
                frameset,
                parlist,
                RECIPE_NAME,
                spec,
                QMOST_PRO_SKY,
                CPL_FRAME_TYPE_IMAGE,
                NULL);
            if(sky_frame == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't set up output product "
                                             "file for %s",
                                             QMOST_PRO_SKY);
            }
        }
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        raw_hdr = cpl_propertylist_load(
            raw_ref_filename,
            raw_extension);
        if(raw_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(raw_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(raw_hdr,
                                   &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            if(proc_science_frame != NULL) {
                if(qmost_dfs_save_image_and_var(proc_science_frame,
                                                raw_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "2D image %s[%s]",
                                                 proc_science_filename,
                                                 arm_extname);
                }
            }

            if(uncal_science_frame != NULL) {
                if(qmost_dfs_save_image_and_var(uncal_science_frame,
                                                raw_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "uncalibrated science "
                                                 "spectrum file %s[%s]",
                                                 uncal_science_filename,
                                                 arm_extname);
                }
            }

            if(crmask_frame != NULL) {
                if(qmost_dfs_save_image_extension(crmask_frame,
                                                  raw_hdr,
                                                  arm_extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "cosmic ray mask %s[%s]",
                                                 crmask_filename,
                                                 arm_extname);
                }
            }

            if(eigen_frame != NULL) {
                extname = cpl_sprintf("%s_eigenvectors", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not format dummy "
                                                 "eigenvector EXTNAME");
                }
                
                if(qmost_dfs_save_image_extension(eigen_frame,
                                                  raw_hdr,
                                                  extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_DOUBLE) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "eigenvectors extension "
                                                 "for %s",
                                                 arm_extname);
                }

                cpl_free(extname);
                extname = NULL;
                
                extname = cpl_sprintf("%s_eigeninfo", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not format dummy "
                                                 "eigeninfo EXTNAME");
                }
                
                if(qmost_dfs_save_table_extension(eigen_frame,
                                                  raw_hdr,
                                                  extname,
                                                  NULL,
                                                  NULL) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "eigeninfo extension "
                                                 "for %s",
                                                 arm_extname);
                }

                cpl_free(extname);
                extname = NULL;
            }

            if(sky_frame != NULL) {
                if(qmost_dfs_save_image_and_var(sky_frame,
                                                raw_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "original sky extension "
                                                 "for %s",
                                                 arm_extname);
                }
                
                extname = cpl_sprintf("%s_skyinfo", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not format skyinfo "
                                                 "EXTNAME string");
                }
                
                if(qmost_dfs_save_table_extension(sky_frame,
                                                  raw_hdr,
                                                  extname,
                                                  NULL,
                                                  NULL) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "skyinfo extension "
                                                 "for %s",
                                                 arm_extname);
                }
                
                cpl_free(extname);
                extname = NULL;
                
                extname = cpl_sprintf("%s_comb", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not format mean sky "
                                                 "EXTNAME string");
                }
                
                if(qmost_dfs_save_image_and_var(sky_frame,
                                                raw_hdr,
                                                extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "mean sky extension "
                                                 "for %s",
                                                 arm_extname);
                }
                
                cpl_free(extname);
                extname = NULL;
                
                extname = cpl_sprintf("%s_subt", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not format "
                                                 "subtracted sky "
                                                 "EXTNAME string");
                }
                
                if(qmost_dfs_save_image_and_var(sky_frame,
                                                raw_hdr,
                                                extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "subtracted sky "
                                                 "extension for %s",
                                                 arm_extname);
                }

                cpl_free(extname);
                extname = NULL;
            }

            if(qmost_dfs_save_image_and_ivar(science_frame,
                                             raw_hdr,
                                             arm_extname,
                                             NULL,
                                             NULL,
                                             NULL,
                                             CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "extracted spectra %s[%s]",
                                             science_filename,
                                             arm_extname);
            }

            if(params.skysub_enable) {
                extname = cpl_sprintf("%s_NOSS", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't format EXTNAME for "
                                                 "NOSS spectrum");
                }

                if(qmost_dfs_save_image_and_ivar(science_frame,
                                                 raw_hdr,
                                                 extname,
                                                 NULL,
                                                 NULL,
                                                 NULL,
                                                 CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "extracted spectra "
                                                 "%s[%s_NOSS]",
                                                 science_filename,
                                                 arm_extname);
                }

                cpl_free(extname);
                extname = NULL;
            }

            /* Copy sensitivity function extension */
            if(calibs.sensfunc_frame != NULL) {
                if(qmost_copy_sensfunc(science_frame,
                                       calibs.sensfunc_frame,
                                       arm,
                                       exptime,
                                       raw_hdr,
                                       NULL) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't copy sensitivity "
                                                 "function to output %s",
                                                 science_filename);
                }
            }

            cpl_propertylist_delete(raw_hdr);
            raw_hdr = NULL;

            continue;
        }

        cpl_msg_info(cpl_func,
                     "Processing extension %lld = %s",
                     raw_extension, arm_extname);
        cpl_msg_indent_more();

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(calibs.master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(calibs.master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(calibs.master_detflat_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_detflat_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_detflat_img,
                                               &master_detflat_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master detector "
                                             "flat frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_detflat_img = NULL;
            master_detflat_var = NULL;
        }

        if(calibs.master_bpm_frame != NULL) {
            if(qmost_load_master_mask(calibs.master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        if(calibs.master_lintab_frame != NULL) {
            if(qmost_load_master_table(calibs.master_lintab_frame,
                                       arm_extname,
                                       &master_lintab,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "linearity table");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* Extract spectra individually */
        nfiles = cpl_frameset_get_size(raw_frames);

        wsp_imglist = cpl_imagelist_new();
        wsp_varlist = cpl_imagelist_new();
        wsp_hdrlist = cpl_calloc(nfiles, sizeof(cpl_propertylist *));
        noss_imglist = cpl_imagelist_new();
        noss_varlist = cpl_imagelist_new();
        noss_hdrlist = cpl_calloc(nfiles, sizeof(cpl_propertylist *));

        for(ifile = 0; ifile < nfiles; ifile++) {
            cpl_msg_info(cpl_func,
                         "Processing science spectrum %d of %d",
                         ifile+1, nfiles);
            cpl_msg_indent_more();

            if(ifile > 0) {
                qctmp = cpl_propertylist_duplicate(qclist);
                qcuse = qctmp;
            }
            else {
                qcuse = qclist;
            }

            if(qmost_ccdproc(cpl_frameset_get_position(raw_frames, ifile),
                             raw_extension,
                             master_bpm,
                             master_bias_img,
                             master_bias_var,
                             master_dark_img,
                             master_dark_var,
                             master_detflat_img,
                             master_detflat_var,
                             1,
                             params.swapx_table,
                             1,
                             master_lintab,
                             QMOST_LIN_AFTER_BIAS,
                             &processed_img,
                             &processed_var,
                             qcuse) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "basic CCD processing failed "
                                             "for extension %lld",
                                             raw_extension);
            }

            if(qmost_science_process_extract_one(processed_img,
                                                 processed_var,
                                                 qcuse,
                                                 fibinfo_tbl,
                                                 spec,
                                                 arm,
                                                 calibs,
                                                 params,
                                                 ifile == 0 ?
                                                 proc_science_frame :
                                                 NULL,
                                                 ifile == 0 ?
                                                 uncal_science_frame :
                                                 NULL,
                                                 ifile == 0 ?
                                                 eigen_frame :
                                                 NULL,
                                                 ifile == 0 ?
                                                 sky_frame :
                                                 NULL,
                                                 raw_hdr,
                                                 &extract_rej,
                                                 &wsp_img,
                                                 &wsp_var,
                                                 wsp_hdrlist + ifile,
                                                 &noss_img,
                                                 &noss_var,
                                                 noss_hdrlist + ifile)) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "processing of science frame "
                                             "%d extension %lld failed",
                                             ifile+1,
                                             raw_extension);
            }

            /* Save optional cosmic ray mask */
            if(extract_rej != NULL) {
                if(ifile == 0 && crmask_frame != NULL) {
                    if(qmost_dfs_save_image_extension(crmask_frame,
                                                      raw_hdr,
                                                      arm_extname,
                                                      qclist,
                                                      extract_rej,
                                                      CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                     "couldn't save cosmic ray "
                                                     "mask %s[%s]",
                                                     crmask_filename,
                                                     arm_extname);
                    }
                }

                cpl_image_delete(extract_rej);
                extract_rej = NULL;
            }

            cpl_image_delete(processed_img);
            processed_img = NULL;

            cpl_image_delete(processed_var);
            processed_var = NULL;

            if(qctmp != NULL) {
                cpl_propertylist_delete(qctmp);
                qctmp = NULL;
            }

            cpl_imagelist_set(wsp_imglist, wsp_img, ifile);
            cpl_imagelist_set(wsp_varlist, wsp_var, ifile);

            if(params.skysub_enable) {
                cpl_imagelist_set(noss_imglist, noss_img, ifile);
                cpl_imagelist_set(noss_varlist, noss_var, ifile);
            }

            /* These are now owned by the imagelists */
            wsp_img = NULL;
            wsp_var = NULL;
            noss_img = NULL;
            noss_var = NULL;

            cpl_msg_indent_less();
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_detflat_img != NULL) {
            cpl_image_delete(master_detflat_img);
            master_detflat_img = NULL;
        }

        if(master_detflat_var != NULL) {
            cpl_image_delete(master_detflat_var);
            master_detflat_var = NULL;
        }

        if(master_bpm != NULL) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        if(master_lintab != NULL) {
            cpl_table_delete(master_lintab);
            master_lintab = NULL;
        }

        /* These headers were appended to wsp and noss hdrlist
         * entries, so we are done with them. */
        cpl_propertylist_delete(qclist);
        qclist = NULL;

        /* Stack */
        if(nfiles > 1) {
            cpl_msg_info(cpl_func,
                         "Stacking %d spectra",
                         nfiles);
            cpl_msg_indent_more();

            if(qmost_spec_combine_lite(wsp_imglist, wsp_varlist,
                                       params.combine_thresh,
                                       &wsp_img, &wsp_var) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "stacking failed "
                                             "for extension %lld",
                                             raw_extension);
            }

            wsp_hdr = wsp_hdrlist[0];

            if(params.skysub_enable) {
                if(qmost_spec_combine_lite(noss_imglist, noss_varlist,
                                           params.combine_thresh,
                                           &noss_img, &noss_var)) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "stacking of NOSS failed "
                                                 "for extension %lld",
                                                 raw_extension);
                }

                noss_hdr = noss_hdrlist[0];
            }

            cpl_msg_indent_less();
        }
        else {
            wsp_img = cpl_imagelist_unset(wsp_imglist, 0);
            wsp_var = cpl_imagelist_unset(wsp_varlist, 0);
            wsp_hdr = wsp_hdrlist[0];

            if(params.skysub_enable) {
                noss_img = cpl_imagelist_unset(noss_imglist, 0);
                noss_var = cpl_imagelist_unset(noss_varlist, 0);
                noss_hdr = noss_hdrlist[0];
            }
        }

        cpl_imagelist_delete(wsp_imglist);
        wsp_imglist = NULL;

        cpl_imagelist_delete(wsp_varlist);
        wsp_varlist = NULL;

        cpl_imagelist_delete(noss_imglist);
        noss_imglist = NULL;

        cpl_imagelist_delete(noss_varlist);
        noss_varlist = NULL;

        /* Extracted spectrum QC */
        qclist = cpl_propertylist_new();

        if(qmost_extract_qc(wsp_img,
                            wsp_var,
                            qclist,
                            fibinfo_tbl,
                            arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "extracted spectrum QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        /* If we have enough information to do so, run magnitude
         * comparison.  FIBINFO table required to do this. */
        if(fibinfo_tbl != NULL) {
            if(qmost_science_process_zmag_qc(wsp_img,
                                             wsp_var,
                                             wsp_hdr,
                                             fibinfo_tbl,
                                             arm,
                                             exptime,
                                             qclist) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "synthetic photometry QC "
                                             "failed for extension %lld",
                                             raw_extension);
            }
        }

        /* Add QC to header */
        out_hdr = cpl_propertylist_duplicate(wsp_hdr);
        cpl_propertylist_copy_property_regexp(out_hdr, qclist, ".*", 0);

        /* Save default extracted, wavelength calibrated spectra */
        if(qmost_dfs_save_image_and_ivar(science_frame,
                                         raw_hdr,
                                         arm_extname,
                                         out_hdr,
                                         wsp_img,
                                         wsp_var,
                                         CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save output "
                                         "extracted spectra %s[%s]",
                                         science_filename,
                                         arm_extname);
        }

        /* If we did sky subtraction, also save NOSS versions */
        if(params.skysub_enable) {
            cpl_propertylist_copy_property_regexp(noss_hdr, qclist, ".*", 0);

            extname = cpl_sprintf("%s_NOSS", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't format EXTNAME for "
                                             "NOSS spectrum");
            }

            if(qmost_dfs_save_image_and_ivar(science_frame,
                                             raw_hdr,
                                             extname,
                                             noss_hdr,
                                             noss_img,
                                             noss_var,
                                             CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save output "
                                             "extracted spectra %s[%s_NOSS]",
                                             science_filename,
                                             arm_extname);
            }

            cpl_free(extname);
            extname = NULL;

            cpl_image_delete(noss_img);
            noss_img = NULL;

            cpl_image_delete(noss_var);
            noss_var = NULL;
        }

        cpl_propertylist_delete(out_hdr);
        out_hdr = NULL;

        /* Copy sensitivity function extension */
        if(calibs.sensfunc_frame != NULL) {
            if(qmost_copy_sensfunc(science_frame,
                                   calibs.sensfunc_frame,
                                   arm,
                                   exptime,
                                   raw_hdr,
                                   wsp_hdr) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't copy sensitivity "
                                             "function to output %s",
                                             science_filename);
            }
        }

        cpl_propertylist_delete(raw_hdr);
        raw_hdr = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        cpl_image_delete(wsp_img);
        wsp_img = NULL;

        cpl_image_delete(wsp_var);
        wsp_var = NULL;

        for(ifile = 0; ifile < nfiles; ifile++) {
            if(wsp_hdrlist[ifile] != NULL) {
                cpl_propertylist_delete(wsp_hdrlist[ifile]);
            }
        }
        cpl_free(wsp_hdrlist);
        wsp_hdrlist = NULL;

        for(ifile = 0; ifile < nfiles; ifile++) {
            if(noss_hdrlist[ifile] != NULL) {
                cpl_propertylist_delete(noss_hdrlist[ifile]);
            }
        }
        cpl_free(noss_hdrlist);
        noss_hdrlist = NULL;

        cpl_msg_indent_less();
    }

    /* Output processed 2D image, if requested */
    if(proc_science_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                proc_science_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to processed 2D image %s ",
                                             proc_science_filename);
            }
        }
    }

    if(uncal_science_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                uncal_science_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to uncalibrated science "
                                             "spectrum file %s ",
                                             uncal_science_filename);
            }
        }
    }

    /* Copy fibinfo table from the first frame if there was one */
    if(fibinfo_tbl != NULL) {
        if(qmost_fibtabsave(fibinfo_tbl,
                            fibinfo_hdr,
                            science_frame) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save FIBINFO table "
                                         "to output extracted spectrum "
                                         "frame %s",
                                         science_filename);
        }
    }

    if(sky_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                sky_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to sky file ");
            }
        }
    }

    /* Clean up */
    cpl_frameset_delete(raw_frames);
    raw_frames = NULL;

    if(fibinfo_tbl != NULL) {
        cpl_table_delete(fibinfo_tbl);
        fibinfo_tbl = NULL;
    }

    if(fibinfo_hdr != NULL) {
        cpl_propertylist_delete(fibinfo_hdr);
        fibinfo_hdr = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_science_process_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* QC level parameter */
    par = cpl_parameter_new_value(RECIPE_NAME".level",
                                  CPL_TYPE_INT,
                                  "QC level (0: QC0; 1: QC1)",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "level");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Switch to save processed products */
    par = cpl_parameter_new_value(RECIPE_NAME".keep",
                                  CPL_TYPE_BOOL,
                                  "Save optional processed products",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "keep");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* ccdproc parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".ccdproc.swapx",
                                  CPL_TYPE_STRING,
                                  "String of 0 or 1 specifying for each "
                                  "arm of each spectrograph if we should "
                                  "flip the x axis to correct the "
                                  "wavelength order of the spectral axis.",
                                  RECIPE_NAME,
                                  QMOST_DEFAULT_SWAPX_TABLE);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ccdproc.swapx");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Scattered light removal parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".scattered.enable",
                                  CPL_TYPE_BOOL,
                                  "Enable scattered light removal",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.enable");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".scattered.nbsize",
                                  CPL_TYPE_INT,
                                  "Size of the background smoothing cells "
                                  "for determining scattered light in pixels.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.nbsize");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Spectral extraction parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".extract.crosstalk",
                                  CPL_TYPE_BOOL,
                                  "Enable crosstalk correction",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.crosstalk");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".extract.cr_thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold for PSF "
                                  "spectral extraction.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.cr_thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".extract.niter",
                                  CPL_TYPE_INT,
                                  "Number of rejection iterations for PSF "
                                  "spectral extraction.",
                                  RECIPE_NAME,
                                  3);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.niter");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".extract.width",
                                  CPL_TYPE_INT,
                                  "Boxcar width for spectral extraction.",
                                  RECIPE_NAME,
                                  6);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.width");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Sky subtraction parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".skysub.enable",
                                  CPL_TYPE_INT,
                                  "Enable sky subtraction, or -1 to use "
                                  "appropriate default for QC level.",
                                  RECIPE_NAME,
                                  -1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "skysub.enable");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".skysub.neigen",
                                  CPL_TYPE_INT,
                                  "The maximum number of eigenvectors "
                                  "to use for PCA analysis, or -1 to "
                                  "automatically select a suitable "
                                  "default.",
                                  RECIPE_NAME,
                                  -1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "skysub.neigen");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".skysub.smoothing",
                                  CPL_TYPE_DOUBLE,
                                  "Smoothing box in Angstroms to be "
                                  "used in smoothing the fitted "
                                  "continuum.",
                                  RECIPE_NAME,
                                  50.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "skysub.smoothing");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Stacking parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".combine.rej_thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Threshold for outlier rejection "
                                  "when combining spectra.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "combine.rej_thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate exposure time and midpoint appropriate for
 *          stack.
 *
 * Calculates the total exposure time and mean exposure midpoint for a
 * stack based on primary header information of the individual raw
 * frames comprising the stack.  The spectrograph (ESO INS PATH) is
 * also checked and an error raised if the frames in the stack aren't
 * all from the same spectrograph.
 *
 * @param   raw_frames         (Given)    A frameset containing the
 *                                        raw frames.
 * @param   spec               (Given)    One of the QMOST_SPEC_*
 *                                        constants saying which
 *                                        spectrograph we're
 *                                        processing.
 * @param   exptime            (Returned) The total exposure time.
 * @param   mjdstart           (Returned) The start time of the first
 *                                        exposure.
 * @param   mjdmid             (Returned) The mean midpoint.
 * @param   mjdend             (Returned) The end time of the last
 *                                        exposure.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_science_process_get_expinfo(
    cpl_frameset *raw_frames,
    int spec,
    double *exptime,
    double *mjdstart,
    double *mjdmid,
    double *mjdend)
{
    int ifile, nfiles;
    const cpl_frame *raw_frame;
    const char *raw_filename;
    cpl_propertylist *raw_hdr = NULL;

    int this_spec;
    double this_exptime, this_mjdstart, this_mjdmid, this_mjdend;

#undef TIDY
#define TIDY                                    \
    if(raw_hdr != NULL) {                       \
        cpl_propertylist_delete(raw_hdr);       \
        raw_hdr = NULL;                         \
    }

    nfiles = cpl_frameset_get_size(raw_frames);

    *exptime = 0;
    *mjdstart = 0;
    *mjdmid = 0;
    *mjdend = 0;

    for(ifile = 0; ifile < nfiles; ifile++) {
        /* Get filename */
        raw_frame = cpl_frameset_get_position(raw_frames, ifile);

        raw_filename = cpl_frame_get_filename(raw_frame);
        if(raw_filename == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get filename for "
                                         "raw frame %d",
                                         ifile+1);
        }

        /* Extract the primary FITS header */
        raw_hdr = cpl_propertylist_load(raw_filename, 0);
        if(raw_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS primary header "
                                         "from raw file %d",
                                         ifile+1);
        }

        /* Check spectrograph matches */
        if(qmost_pfits_get_spectrograph(raw_hdr,
                                        &this_spec) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine the "
                                         "spectrograph for raw file %d",
                                         ifile+1);
        }

        if(this_spec != spec) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectrograph doesn't match "
                                         "for raw file %d: %d != %d",
                                         ifile+1, this_spec, spec);
        }

        /* Get exposure time */
        if(qmost_pfits_get_exptime(raw_hdr,
                                   &this_exptime) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get exposure time");
        }

        /* Get start MJD */
        if(qmost_pfits_get_mjd_obs(raw_hdr,
                                   &this_mjdstart) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read MJD-OBS");
        }

        /* Exposure midpoint */
        this_mjdmid = this_mjdstart + 0.5 * this_exptime / 86400.0;
        this_mjdend = this_mjdstart + this_exptime / 86400.0;

        /* Accumulate totals */
        *exptime += this_exptime;
        *mjdmid += this_mjdmid;

        /* Start and end are min and max */
        if(ifile > 0) {
            *mjdstart = qmost_min(*mjdstart, this_mjdstart);
            *mjdend = qmost_max(*mjdend, this_mjdend);
        }
        else {
            *mjdstart = this_mjdstart;
            *mjdend = this_mjdend;
        }

        /* Clean up */
        cpl_propertylist_delete(raw_hdr);
        raw_hdr = NULL;
    }

    /* Resulting midpoint is the mean */
    *mjdmid /= nfiles;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Extract and calibrate spectrum from a single science
 *          frame.
 *
 * This routine contains all the steps needed to go from a processed
 * 2D image to an extracted and calibrated spectrum: scattered light
 * removal, spectral extraction, wavelength calibration, fibre flat
 * fielding, and (optionally) sky subtraction.  This is called in a
 * loop by the main body of qmost_science_process to do the parts of
 * processing that should be done individually on each science frame
 * from an OB prior to OB-level stacking.
 *
 * @param   processed_img        (Given)    The input processed image
 *                                          from qmost_ccdproc.
 * @param   processed_var        (Given)    The input variance array
 *                                          from qmost_ccdproc.
 * @param   qclist               (Modified) A propertylist with QC
 *                                          headers, usually also from
 *                                          qmost_ccdproc.  Will be
 *                                          augmented with appropriate
 *                                          QC from the extraction
 *                                          process.
 * @param   fibinfo_tbl          (Modified) The FIBINFO table.
 * @param   spec                 (Given)    One of the QMOST_SPEC_*
 *                                          constants saying which
 *                                          spectrograph we're
 *                                          processing.
 * @param   arm                  (Given)    One of the QMOST_ARM_*
 *                                          constants saying which arm
 *                                          we're processing.
 * @param   calibs               (Given)    A structure containing all
 *                                          of the calibration frames
 *                                          given to the recipe.
 * @param   params               (Given)    A structure containing all
 *                                          of the recipe parameters.
 * @param   proc_science_frame   (Given)    If non-NULL the final
 *                                          scattered light corrected
 *                                          2D image will be written
 *                                          to this frame.
 * @param   uncal_science_frame  (Given)    If non-NULL the extracted
 *                                          spectra prior to
 *                                          wavelength calibration
 *                                          will be written to this
 *                                          frame.
 * @param   eigen_frame          (Given)    If non-NULL the optional
 *                                          sky subtraction
 *                                          eigenvector diagnostic
 *                                          output will be written to
 *                                          this frame.
 * @param   sky_frame            (Given)    If non-NULL the optional
 *                                          sky subtraction diagnostic
 *                                          output will be written to
 *                                          this frame.
 * @param   raw_hdr              (Given)    FITS header from the IMAGE
 *                                          extension of the raw file
 *                                          to be used when populating
 *                                          the FITS header of the
 *                                          proc_science_frame output.
 * @param   extract_rej          (Returned) The cosmic ray mask
 *                                          produced during spectral
 *                                          extraction.
 * @param   wsp_img              (Returned) The final processed
 *                                          science spectra.
 * @param   wsp_var              (Returned) The uncertainties in the
 *                                          final processed science
 *                                          spectra.
 * @param   wsp_hdr              (Returned) The FITS header for the
 *                                          processed science spectum
 *                                          extension.
 * @param   noss_img             (Returned) If sky subtraction was
 *                                          requested, this returns
 *                                          the non sky subtracted
 *                                          science spectra.
 *                                          Otherwise set to NULL.
 * @param   noss_var             (Returned) The corresponding
 *                                          uncertainties for the
 *                                          noss_img.
 * @param   noss_hdr             (Returned) The FITS header for the
 *                                          non sky subtracted
 *                                          spectrum extension.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_science_process_extract_one(
    cpl_image *processed_img,
    cpl_image *processed_var,
    cpl_propertylist *qclist,
    cpl_table *fibinfo_tbl,
    int spec,
    int arm,
    struct qmost_science_process_calibs calibs,
    struct qmost_science_process_params params,
    const cpl_frame *proc_science_frame,
    const cpl_frame *uncal_science_frame,
    const cpl_frame *eigen_frame,
    const cpl_frame *sky_frame,
    cpl_propertylist *raw_hdr,
    cpl_image **extract_rej,
    cpl_image **wsp_img,
    cpl_image **wsp_var,
    cpl_propertylist **wsp_hdr,
    cpl_image **noss_img,
    cpl_image **noss_var,
    cpl_propertylist **noss_hdr)
{
    const char *arm_extname;
    char *extname = NULL;
    const char *proc_science_filename = NULL;
    const char *uncal_science_filename = NULL;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_mask *fibre_mask = NULL;
    cpl_imagelist *fibre_psf_img = NULL;
    cpl_imagelist *fibre_psf_var = NULL;
    cpl_propertylist *fibre_psf_hdr = NULL;
    cpl_table *master_wave_tbl = NULL;
    cpl_propertylist *master_wave_hdr = NULL;
    cpl_table *ob_wave_tbl = NULL;
    cpl_propertylist *ob_wave_hdr = NULL;

    cpl_image *extracted_img = NULL;
    cpl_image *extracted_var = NULL;

    cpl_image *master_ffnorm_img = NULL;
    cpl_image *master_ffnorm_err = NULL;
    cpl_table *ob_ffnorm_fibinfo = NULL;
    cpl_table *sky_ffnorm_fibinfo = NULL;

    int nfib;
    double *waveoff = NULL;
    double *veloff = NULL;
    double minwave, maxwave, dlam;
    int ifib, isnull;

    cpl_errorstate prestate;
    cpl_error_code code;

    qmost_skysub_diags diags = {
        .eigenvectors = NULL,
        .eigeninfo = NULL,
        .orig_img = NULL,
        .orig_var = NULL,
        .skyinfo = NULL,
        .comb_img = NULL,
        .comb_var = NULL,
        .subt_img = NULL,
        .subt_var = NULL
    };

#undef TIDY
#define TIDY                                    \
    if(trace_tbl) {                             \
        cpl_table_delete(trace_tbl);            \
        trace_tbl = NULL;                       \
    }                                           \
    if(trace_hdr) {                             \
        cpl_propertylist_delete(trace_hdr);     \
        trace_hdr = NULL;                       \
    }                                           \
    if(fibre_mask) {                            \
        cpl_mask_delete(fibre_mask);            \
        fibre_mask = NULL;                      \
    }                                           \
    if(fibre_psf_img) {                         \
        cpl_imagelist_delete(fibre_psf_img);    \
        fibre_psf_img = NULL;                   \
    }                                           \
    if(fibre_psf_var) {                         \
        cpl_imagelist_delete(fibre_psf_var);    \
        fibre_psf_var = NULL;                   \
    }                                           \
    if(fibre_psf_hdr) {                         \
        cpl_propertylist_delete(fibre_psf_hdr); \
        fibre_psf_hdr = NULL;                   \
    }                                           \
    if(master_wave_tbl) {                       \
        cpl_table_delete(master_wave_tbl);      \
        master_wave_tbl = NULL;                 \
    }                                           \
    if(master_wave_hdr) {                       \
        cpl_propertylist_delete(master_wave_hdr); \
        master_wave_hdr = NULL;                 \
    }                                           \
    if(ob_wave_tbl) {                           \
        cpl_table_delete(ob_wave_tbl);          \
        ob_wave_tbl = NULL;                     \
    }                                           \
    if(ob_wave_hdr) {                           \
        cpl_propertylist_delete(ob_wave_hdr);   \
        ob_wave_hdr = NULL;                     \
    }                                           \
    if(extracted_img) {                         \
        cpl_image_delete(extracted_img);        \
        extracted_img = NULL;                   \
    }                                           \
    if(extracted_var) {                         \
        cpl_image_delete(extracted_var);        \
        extracted_var = NULL;                   \
    }                                           \
    if(waveoff) {                               \
        cpl_free(waveoff);                      \
        waveoff = NULL;                         \
    }                                           \
    if(veloff) {                                \
        cpl_free(veloff);                       \
        veloff = NULL;                          \
    }                                           \
    if(extname) {                               \
        cpl_free(extname);                      \
        extname = NULL;                         \
    }                                           \
    if(master_ffnorm_img) {                     \
        cpl_image_delete(master_ffnorm_img);    \
        master_ffnorm_img = NULL;               \
    }                                           \
    if(master_ffnorm_err) {                     \
        cpl_image_delete(master_ffnorm_err);    \
        master_ffnorm_err = NULL;               \
    }                                           \
    if(ob_ffnorm_fibinfo) {                     \
        cpl_table_delete(ob_ffnorm_fibinfo);    \
        ob_ffnorm_fibinfo = NULL;               \
    }                                           \
    if(sky_ffnorm_fibinfo) {                    \
        cpl_table_delete(sky_ffnorm_fibinfo);   \
        sky_ffnorm_fibinfo = NULL;              \
    }                                           \
    if(*extract_rej) {                          \
        cpl_image_delete(*extract_rej);         \
        *extract_rej = NULL;                    \
    }                                           \
    if(*wsp_img) {                              \
        cpl_image_delete(*wsp_img);             \
        *wsp_img = NULL;                        \
    }                                           \
    if(*wsp_var) {                              \
        cpl_image_delete(*wsp_var);             \
        *wsp_var = NULL;                        \
    }                                           \
    if(*wsp_hdr) {                              \
        cpl_propertylist_delete(*wsp_hdr);      \
        *wsp_hdr = NULL;                        \
    }                                           \
    if(*noss_img) {                             \
        cpl_image_delete(*noss_img);            \
        *noss_img = NULL;                       \
    }                                           \
    if(*noss_var) {                             \
        cpl_image_delete(*noss_var);            \
        *noss_var = NULL;                       \
    }                                           \
    if(*noss_hdr) {                             \
        cpl_propertylist_delete(*noss_hdr);     \
        *noss_hdr = NULL;                       \
    }                                           \
    if(diags.eigenvectors != NULL) {            \
        cpl_image_delete(diags.eigenvectors);   \
        diags.eigenvectors = NULL;              \
    }                                           \
    if(diags.eigeninfo != NULL) {               \
        cpl_table_delete(diags.eigeninfo);      \
        diags.eigeninfo = NULL;                 \
    }                                           \
    if(diags.orig_img != NULL) {                \
        cpl_image_delete(diags.orig_img);       \
        diags.orig_img = NULL;                  \
    }                                           \
    if(diags.orig_var != NULL) {                \
        cpl_image_delete(diags.orig_var);       \
        diags.orig_var = NULL;                  \
    }                                           \
    if(diags.skyinfo != NULL) {                 \
        cpl_table_delete(diags.skyinfo);        \
        diags.skyinfo = NULL;                   \
    }                                           \
    if(diags.comb_img != NULL) {                \
        cpl_image_delete(diags.comb_img);       \
        diags.comb_img = NULL;                  \
    }                                           \
    if(diags.comb_var != NULL) {                \
        cpl_image_delete(diags.comb_var);       \
        diags.comb_var = NULL;                  \
    }                                           \
    if(diags.subt_img != NULL) {                \
        cpl_image_delete(diags.subt_img);       \
        diags.subt_img = NULL;                  \
    }                                           \
    if(diags.subt_var != NULL) {                \
        cpl_image_delete(diags.subt_var);       \
        diags.subt_var = NULL;                  \
    }

    /* Get EXTNAME for arm */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine EXTNAME "
                                     "for arm %d", arm);
    }

    /* Get trace table and fibre mask */
    extname = cpl_sprintf("trace_%s", arm_extname);
    if(extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format trace "
                                     "EXTNAME string");
    }
        
    if(qmost_load_master_table(calibs.fibre_trace_frame,
                               extname,
                               &trace_tbl,
                               &trace_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load trace");
    }
        
    cpl_free(extname);
    extname = NULL;

    extname = cpl_sprintf("mask_%s", arm_extname);
    if(extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format fibre mask "
                                     "EXTNAME string");
    }

    if(qmost_load_master_mask(calibs.fibre_mask_frame,
                              extname,
                              &fibre_mask,
                              NULL) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load fibre mask");
    }
        
    cpl_free(extname);
    extname = NULL;

    /* Populate scattered light QC before we remove it */
    if(qmost_scattered_qc(processed_img,
                          qclist,
                          fibre_mask,
                          qclist) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "scattered light QC failed");
    }

    if(params.scattered_enable) {
        /* Remove scattered light if requested */
        if(qmost_scattered(processed_img,
                           qclist,
                           fibre_mask,
                           params.scattered_nbsize,
                           1) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "removal of scattered light "
                                         "with box size %d failed",
                                         params.scattered_nbsize);
        }
    }

    /* Save processed image if requested */
    if(proc_science_frame != NULL) {
        proc_science_filename = cpl_frame_get_filename(proc_science_frame);

        if(qmost_dfs_save_image_and_var(proc_science_frame,
                                        raw_hdr,
                                        arm_extname,
                                        qclist,
                                        processed_img,
                                        processed_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save processed "
                                         "2D image %s[%s]",
                                         proc_science_filename,
                                         arm_extname);
        }
    }

    if(params.level == 0) {
        /* Extract spectra */
        if(qmost_extract_tram(processed_img,
                              processed_var,
                              qclist,
                              trace_tbl,
                              trace_hdr,
                              params.extract_width,
                              &extracted_img,
                              &extracted_var,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectral extraction failed");
        }
    }
    else {
        /* Load PSF */
        extname = cpl_sprintf("prof_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format PSF variance "
                                         "EXTNAME string");
        }
            
        if(qmost_load_master_imagelist(calibs.fibre_psf_frame,
                                       extname,
                                       CPL_TYPE_FLOAT,
                                       &fibre_psf_img,
                                       &fibre_psf_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load master PSF");
        }
            
        cpl_free(extname);
        extname = NULL;
            
        extname = cpl_sprintf("profvar_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format PSF variance "
                                         "EXTNAME string");
        }
            
        if(qmost_load_master_imagelist(calibs.fibre_psf_frame,
                                       extname,
                                       CPL_TYPE_FLOAT,
                                       &fibre_psf_var,
                                       NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load master PSF variance");
        }
            
        cpl_free(extname);
        extname = NULL;

        /* Extract spectra */
        if(qmost_extract_psf(processed_img,
                             processed_var,
                             qclist,
                             NULL,  /* no cosmic in QC */
                             fibre_psf_img,
                             fibre_psf_var,
                             fibre_psf_hdr,
                             trace_tbl,
                             trace_hdr,
                             params.extract_crosstalk,
                             params.extract_niter,
                             params.extract_crrej_thr,
                             &extracted_img,
                             &extracted_var,
                             qclist,
                             extract_rej) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectral extraction failed");
        }

        cpl_imagelist_delete(fibre_psf_img);
        fibre_psf_img = NULL;
            
        cpl_imagelist_delete(fibre_psf_var);
        fibre_psf_var = NULL;
            
        cpl_propertylist_delete(fibre_psf_hdr);
        fibre_psf_hdr = NULL;
    }

    /* Save extracted, uncalibrated spectra if requested */
    if(uncal_science_frame != NULL) {
        uncal_science_filename = cpl_frame_get_filename(uncal_science_frame);

        if(qmost_dfs_save_image_and_var(uncal_science_frame,
                                        raw_hdr,
                                        arm_extname,
                                        qclist,
                                        extracted_img,
                                        extracted_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save uncalibrated "
                                         "extracted science spectrum "
                                         "file %s[%s]",
                                         uncal_science_filename,
                                         arm_extname);
        }
    }

    /* If we got nothing, abort here.  This usually means there were
     * no traces. */
    if(extracted_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "no spectra were extracted in "
                                     "extension %s, check trace file",
                                     arm_extname);
    }
    
    /* Get wavelength solution */
    extname = cpl_sprintf("wave_%s", arm_extname);
    if(extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format wave "
                                     "EXTNAME string");
    }
        
    if(qmost_load_master_table(calibs.master_wave_frame,
                               extname,
                               &master_wave_tbl,
                               &master_wave_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load master wavelength "
                                     "solution");
    }
        
    if(calibs.ob_wave_frame != NULL) {
        if(qmost_load_master_table(calibs.ob_wave_frame,
                                   extname,
                                   &ob_wave_tbl,
                                   &ob_wave_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load OB wavelength "
                                         "solution");
        }
    }
    else {
        ob_wave_tbl = NULL;
        ob_wave_hdr = NULL;
    }

    cpl_free(extname);
    extname = NULL;

    /* Wavelength calibrate extracted spectra */
    nfib = cpl_image_get_size_y(extracted_img);

    if(qmost_get_rebin_params(spec, arm,
                              &minwave,
                              &maxwave,
                              &dlam) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get wavelength "
                                     "calibration rebinning "
                                     "parameters");
    }

    waveoff = cpl_calloc(nfib, sizeof(double));
        
    /* Get barycentric correction */
    veloff = cpl_calloc(nfib, sizeof(double));

    if(fibinfo_tbl != NULL) {
        for(ifib = 0; ifib < nfib; ifib++) {
            veloff[ifib] = cpl_table_get(fibinfo_tbl,
                                         "HELIO_COR",
                                         ifib,
                                         &isnull);
            if(isnull < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not read HELIO_COR "
                                             "for fibre %d", ifib+1);
            }
            else if(isnull > 0) {
                veloff[ifib] = 0;
            }
        }
    }

    *wsp_hdr = cpl_propertylist_new();
    cpl_propertylist_update_string(*wsp_hdr, "BUNIT", "ADU");

    if(qmost_rebin_spectra_off(extracted_img,
                               extracted_var,
                               qclist,
                               master_wave_tbl,
                               master_wave_hdr,
                               ob_wave_tbl,
                               ob_wave_hdr,
                               NULL,
                               NULL,
                               waveoff,
                               veloff,
                               nfib,
                               minwave,
                               maxwave,
                               dlam,
                               wsp_img,
                               wsp_var,
                               *wsp_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "wavelength calibration of "
                                     "extracted spectra failed");
    }

    cpl_free(waveoff);
    waveoff = NULL;

    cpl_free(veloff);
    veloff = NULL;

    /* Add QC to header */
    cpl_propertylist_copy_property_regexp(*wsp_hdr, qclist, ".*", 0);

    /* Divide master fibre flat if given */
    if(calibs.master_fibre_flat_frame != NULL) {
        if(qmost_load_master_image_and_var(calibs.master_fibre_flat_frame,
                                           arm_extname,
                                           CPL_TYPE_FLOAT,
                                           &master_ffnorm_img,
                                           &master_ffnorm_err,
                                           NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load master "
                                         "fibre flat");
        }

        if(qmost_ffdiv_fib(*wsp_img,
                           *wsp_var,
                           *wsp_hdr,
                           trace_tbl,
                           master_ffnorm_img,
                           master_ffnorm_err) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "fibre flat division "
                                         "failed");
        }

        cpl_image_delete(master_ffnorm_img);
        master_ffnorm_img = NULL;

        cpl_image_delete(master_ffnorm_err);
        master_ffnorm_err = NULL;
    }

    /* Apply OB fibre flat throughput correction if given */
    if(calibs.ob_fibre_flat_frame != NULL) {
        if(qmost_load_master_table(calibs.ob_fibre_flat_frame,
                                   QMOST_FIBINFO_EXTNAME,
                                   &ob_ffnorm_fibinfo,
                                   NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load OB "
                                         "fibre flat");
        }

        if(qmost_obffcor_fib(*wsp_img,
                             *wsp_var,
                             trace_tbl,
                             arm,
                             ob_ffnorm_fibinfo) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "OB fibre flat correction "
                                         "failed");
        }

        cpl_table_delete(ob_ffnorm_fibinfo);
        ob_ffnorm_fibinfo = NULL;
    }

    /* Apply twilight fibre flat throughput correction if given */
    if(calibs.sky_fibre_flat_frame != NULL) {
        if(qmost_load_master_table(calibs.sky_fibre_flat_frame,
                                   QMOST_FIBINFO_EXTNAME,
                                   &sky_ffnorm_fibinfo,
                                   NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load twilight "
                                         "fibre flat");
        }

        /* The twilight flat throughput correction is mathematically
         * identical to the OB throughput correction, so we can just
         * use that routine. */
        if(qmost_obffcor_fib(*wsp_img,
                             *wsp_var,
                             trace_tbl,
                             arm,
                             sky_ffnorm_fibinfo) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "twilight fibre flat correction "
                                         "failed");
        }

        cpl_table_delete(sky_ffnorm_fibinfo);
        sky_ffnorm_fibinfo = NULL;
    }

    /* Sky subtraction, if requested */
    if(params.skysub_enable) {
        /* Save NOSS versions */
        *noss_img = cpl_image_duplicate(*wsp_img);
        *noss_var = cpl_image_duplicate(*wsp_var);
        *noss_hdr = cpl_propertylist_duplicate(*wsp_hdr);

        /* Now sky subtract main product.  We need to trap
         * CPL_ERROR_DATA_NOT_FOUND, which is raised by skysub if
         * there are no usable sky fibres.  This is converted into a
         * warning message. */
        cpl_msg_info(cpl_func, "Sky subtraction");

        prestate = cpl_errorstate_get();

        code = qmost_skysub_vshiftpca(*wsp_img,
                                      *wsp_var,
                                      *wsp_hdr,
                                      fibinfo_tbl,
                                      params.skysub_neigen,
                                      params.skysub_smoothing,
                                      0, 0, 0, 0,
                                      eigen_frame != NULL ||
                                      sky_frame != NULL ?
                                      &diags : NULL);
        if(code != CPL_ERROR_NONE) {
            if(code == CPL_ERROR_DATA_NOT_FOUND) {
               cpl_msg_warning(cpl_func,
                               "Data could not be sky subtracted: %s",
                               cpl_error_get_message());

               cpl_errorstate_set(prestate);
            }
            else {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "sky subtraction failed");
            }
        }

        if(eigen_frame != NULL) {
            extname = cpl_sprintf("%s_eigenvectors", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format eigenvector "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_image_extension(eigen_frame,
                                              raw_hdr,
                                              extname,
                                              *wsp_hdr,
                                              diags.eigenvectors,
                                              CPL_TYPE_DOUBLE) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save eigenvectors "
                                             "extension for %s",
                                             arm_extname);
            }

            cpl_image_delete(diags.eigenvectors);
            diags.eigenvectors = NULL;

            cpl_free(extname);
            extname = NULL;

            extname = cpl_sprintf("%s_eigeninfo", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format eigeninfo "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_table_extension(eigen_frame,
                                              raw_hdr,
                                              extname,
                                              *wsp_hdr,
                                              diags.eigeninfo) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save eigeninfo "
                                             "extension for %s",
                                             arm_extname);
            }

            cpl_table_delete(diags.eigeninfo);
            diags.eigeninfo = NULL;
            
            cpl_free(extname);
            extname = NULL;
        }

        if(sky_frame != NULL) {
            if(qmost_dfs_save_image_and_var(sky_frame,
                                            raw_hdr,
                                            arm_extname,
                                            *wsp_hdr,
                                            diags.orig_img,
                                            diags.orig_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save original sky "
                                             "extension for %s",
                                             arm_extname);
            }

            cpl_image_delete(diags.orig_img);
            diags.orig_img = NULL;
            
            cpl_image_delete(diags.orig_var);
            diags.orig_var = NULL;
            
            extname = cpl_sprintf("%s_skyinfo", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format skyinfo "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_table_extension(sky_frame,
                                              raw_hdr,
                                              extname,
                                              *wsp_hdr,
                                              diags.skyinfo) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save skyinfo "
                                             "extension for %s",
                                             arm_extname);
            }

            cpl_table_delete(diags.skyinfo);
            diags.skyinfo = NULL;
            
            cpl_free(extname);
            extname = NULL;

            extname = cpl_sprintf("%s_comb", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format mean sky "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_image_and_var(sky_frame,
                                            raw_hdr,
                                            extname,
                                            *wsp_hdr,
                                            diags.comb_img,
                                            diags.comb_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save mean sky "
                                             "extension for %s",
                                             arm_extname);
            }

            cpl_image_delete(diags.comb_img);
            diags.comb_img = NULL;
            
            cpl_image_delete(diags.comb_var);
            diags.comb_var = NULL;

            cpl_free(extname);
            extname = NULL;
            
            extname = cpl_sprintf("%s_subt", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format subtracted sky "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_image_and_var(sky_frame,
                                            raw_hdr,
                                            extname,
                                            *wsp_hdr,
                                            diags.subt_img,
                                            diags.subt_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save subtracted sky "
                                             "extension for %s",
                                             arm_extname);
            }

            cpl_image_delete(diags.subt_img);
            diags.subt_img = NULL;
            
            cpl_image_delete(diags.subt_var);
            diags.subt_var = NULL;

            cpl_free(extname);
            extname = NULL;
        }
    }

    /* Populate QC */
    if(fibinfo_tbl != NULL) {
        if(qmost_fibtab_arcqc(fibinfo_tbl,
                              arm,
                              master_wave_tbl,
                              ob_wave_tbl) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "adding QC to FIBINFO table "
                                         "failed");
        }
    }

    /* Clean up */
    cpl_table_delete(trace_tbl);
    trace_tbl = NULL;

    cpl_propertylist_delete(trace_hdr);
    trace_hdr = NULL;

    cpl_mask_delete(fibre_mask);
    fibre_mask = NULL;

    cpl_image_delete(extracted_img);
    extracted_img = NULL;

    cpl_image_delete(extracted_var);
    extracted_var = NULL;

    cpl_table_delete(master_wave_tbl);
    master_wave_tbl = NULL;

    cpl_propertylist_delete(master_wave_hdr);
    master_wave_hdr = NULL;

    if(ob_wave_tbl != NULL) {
        cpl_table_delete(ob_wave_tbl);
        ob_wave_tbl = NULL;
        cpl_propertylist_delete(ob_wave_hdr);
        ob_wave_hdr = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Compute magnitude zero point and populate QC.
 *
 * Computes magnitude zero point for the given spectra in the natural
 * bandpass of the spectrograph by comparing the catalogue magnitudes
 * of the sources used with the total flux in the spectrum, based on
 * the information given in the FIBINFO table.  Statistics of the
 * comparison are saved to the output properylist.
 *
 * @param   spec_img        (Given)    The extracted, wavelength
 *                                     calibrated science spectra as a
 *                                     2D array.  Currently, only the
 *                                     size is used but the parameter
 *                                     has been passed for flexibility
 *                                     in case the implementation
 *                                     needs to be changed in the
 *                                     future.
 * @param   spec_var        (Given)    The variance in the science
 *                                     spectra as a 2D array.  Not
 *                                     currently used.
 * @param   spec_hdr        (Given)    The FITS header of the spectrum
 *                                     with WCS information.  Not
 *                                     currently used.
 * @param   fibinfo_tbl     (Given)    The FIBINFO table containing
 *                                     the target magnitude
 *                                     information.
 * @param   arm             (Given)    One of the QMOST_ARM_*
 *                                     constants specifying which arm
 *                                     we're processing.
 * @param   exptime         (Given)    The exposure time in seconds.
 * @param   qclist          (Modified) A propertylist to receive the
 *                                     QC parameters.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE             If everything is OK.
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_ST</b>
 *   - <b>FIB_USE</b>
 *   - <b>MEANFLUX_B</b>
 *   - <b>MEANFLUX_G</b>
 *   - <b>MEANFLUX_R</b>
 *   - <b>OBJ_BL</b>
 *   - <b>OBJ_GR</b>
 *   - <b>OBJ_RD</b>
 *
 * @par Output QC Parameters:
 *   - <b>ZMAG MED</b> (mag): Median magnitude zero point, defined as
 *     the magnitude of a source giving 1 ADU/second total extracted
 *     spectral counts in this arm of the spectrograph.
 *   - <b>ZMAG RMS</b> (mag): Robustly-estimated RMS dispersion of the
 *     magnitude zero point, or equivalently the dispersion in the
 *     difference between a synthetic magnitude computed from the
 *     spectrum and expected magnitude from target catalogue.
 *   - <b>ZMAG NUM</b>: The number of spectra used in the calculation
 *     of the median and RMS magnitude zero point.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_science_process_zmag_qc(
    cpl_image *spec_img,
    cpl_image *spec_var,
    cpl_propertylist *spec_hdr,
    cpl_table *fibinfo_tbl,
    int arm,
    double exptime,
    cpl_propertylist *qclist)
{
    int nwave;

    const char *arm_extname = NULL;
    char *mag_column = NULL;
    char *meanflux_column = NULL;

    int irow, nrows;

    float *zmag = (float *) NULL;
    int nmag;

    int fib_st, fib_use, isnull;
    float tbl_mag, meanflux;

    float med_zmag, rms_zmag;

    cpl_ensure_code(spec_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_var != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fibinfo_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(zmag != NULL) {                          \
        cpl_free(zmag);                         \
        zmag = NULL;                            \
    }

    /* Wavelength axis length */
    nwave = cpl_image_get_size_x(spec_img);

    /* Define columns for this arm */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not determine EXTNAME "
                                     "for arm %d", arm);
    }
    
    switch(arm_extname[0]) {
    case 'R':
        mag_column = "OBJ_RD";
        meanflux_column = "MEANFLUX_R";
        break;
    case 'G':
        mag_column = "OBJ_GR";
        meanflux_column = "MEANFLUX_G";
        break;
    case 'B':
        mag_column = "OBJ_BL";
        meanflux_column = "MEANFLUX_B";
        break;
    default:
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "unrecognised arm: %s", arm_extname);
    }

    /* Check columns exist.  If they don't, it's not an error but we
     * should return now without running the analysis. */
    if(!cpl_table_has_column(fibinfo_tbl, mag_column) ||
       !cpl_table_has_column(fibinfo_tbl, meanflux_column)) {
        TIDY;
        return CPL_ERROR_NONE;
    }

    /* Number of rows to process */
    nrows = cpl_table_get_nrow(fibinfo_tbl);

    /* Allocate workspace */
    zmag = cpl_malloc(nrows * sizeof(float));

    /* Loop over objects */
    nmag = 0;

    for(irow = 0; irow < nrows; irow++) {
        fib_st = cpl_table_get_int(fibinfo_tbl, "FIB_ST", irow, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         "FIB_ST",
                                         irow + 1);
        }
        else if(isnull > 0) {
            /* Skip nulls */
            continue;
        }
        else if(fib_st != 2) {
            /* Skip any fibres that are not live */
            continue;
        }

        fib_use = cpl_table_get_int(fibinfo_tbl, "FIB_USE", irow, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         "FIB_USE",
                                         irow + 1);
        }
        else if(isnull > 0) {
            /* Skip nulls */
            continue;
        }
        else if(fib_use != 1 &&  /* science target */
                fib_use != 4 &&  /* calibration (WD) */
                fib_use != 7 &&  /* telluric standard */
                fib_use != 8) {  /* spectrophotometric standard */
            /* Skip any fibres that are not on targets or standard
             * stars. */
            continue;
        }

        tbl_mag = cpl_table_get(fibinfo_tbl, mag_column, irow, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         mag_column,
                                         irow + 1);
        }
        else if(isnull > 0) {
            /* Skip nulls */
            continue;
        }
        else if(tbl_mag == 0) {
            /* Zeros might also be used to flag as not available */
            continue;
        }

        meanflux = cpl_table_get(fibinfo_tbl, meanflux_column, irow, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         meanflux_column,
                                         irow + 1);
        }
        else if(isnull > 0) {
            /* Skip nulls */
            continue;
        }
        else if(meanflux <= 0) {
            /* Can't compute magnitude if it's negative or zero */
            continue;
        }

        /* Magnitude zero point = magnitude of source giving 1 ADU/s
         * integrated over spectrum.  Equivalent to the usual
         * definition for photometry for a photometric bandpass
         * defined by the spectrograph response.  Total flux is
         * computed as mean flux multiplied by length of spectrum to
         * account for any bad pixels. */
        zmag[nmag] = tbl_mag + 2.5 * log10(meanflux * nwave / exptime);
        nmag++;
    }

    /* Did we get any results? */
    if(nmag > 0) {
        /* Compute statistics */
        if(qmost_medmad(zmag, NULL, nmag,
                        &med_zmag,
                        &rms_zmag) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to compute delta "
                                         "mag statistics");
        }

        rms_zmag *= CPL_MATH_STD_MAD;

        cpl_propertylist_update_float(qclist,
                                      "ESO QC ZMAG MED",
                                      med_zmag);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC ZMAG MED",
                                     "[mag] "
                                     "Median magnitude zero point");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC ZMAG RMS",
                                      rms_zmag);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC ZMAG RMS",
                                     "[mag] "
                                     "RMS magnitude zero point");
    }

    cpl_propertylist_update_int(qclist,
                                "ESO QC ZMAG NUM",
                                nmag);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC ZMAG NUM",
                                 "Number of magnitudes used");

    cpl_free(zmag);
    zmag = NULL;

    return CPL_ERROR_NONE;
}

/**@}*/
