/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_doarcs.h"
#include "qmost_pfits.h"
#include "qmost_polynm.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"
#include "qmost_waveinfo.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_doarcs  qmost_doarcs
 * 
 * Compute wavelength solution from extracted arc spectra.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_doarcs.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Calculation window in pixels used when applying Gaussian kernel
 * convolution for detection. */

#define NW 5

/* Reference wavelength (A) for matchwindow and matchgrid parameters */

#define REFWAVE 6000

/* Aperture size (pixels) for centre of gravity measurement */

#define COGWIDTH 5.0

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/* Structure used to report features identified by findpeaks. */

typedef struct {
    double xfeat;
    float tmax;
    float contrast;
    float fwhm;
    double xfeat_norm;
    double wavematched;
    double errwavematched;
    double wavecalc;
    double wave_resid;
    int matched_index;
    int bad;
} featstruct;

/* Structure used to store parameters of a detected spectral line. */

typedef struct {
    double wave;
    double errwave;
    double pos_guess;
    int matched_index;
} linestruct;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eval_wave_map (
    float *refdata,
    long naxisr[2],
    float xpos,
    float ypos,
    float *waveguess);

static cpl_polynomial *dofit(
    int nfeat,
    featstruct *xfeat,
    int nord,
    float rejthr,
    int *ngood,
    double *rms,
    int fitx);

static void detect_lines(
    float *indata,
    long nx,
    float detthr,
    int *nfeat,
    featstruct **xfeat);

static void findpeaks(
    int nx,
    float *indata,
    float *smoothed,
    float thr,
    int *nfeat,
    featstruct **xfeat);

static void crweights(
    float filtfwhm,
    float *weights);

static void convolve(
    int nx,
    float *indata,
    float filtfwhm,
    float *smoothed);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Identify arc lines and fit wavelength solution based on a
 *          reference image or wavelength solution.
 *
 * Emission line features are identified in spectra. These are then
 * assigned wavelengths based on either a 2D image of the focal plane
 * with an estimate of the wavelength versus x,y position, or a
 * reference wavelength solution table.  A wavelength solution is
 * calculated for each fibre using a robust polynomial fit of the
 * known wavelengths of the detected arc lines versus spectral pixel
 * coordinate.  The results are recorded in a wavelength solution
 * table.
 *
 * @param   in_spec          (Given)    The extracted arc spectra
 *                                      to analyse, as an image with a
 *                                      row per fibre.  The data type
 *                                      must be CPL_TYPE_FLOAT.
 * @param   in_hdr           (Given)    The FITS header of the input
 *                                      spectra with binning
 *                                      information.
 * @param   ref_wave_surface (Given)    A 2D image giving a wavelength
 *                                      estimate at each pixel
 *                                      position.  The data type must
 *                                      be CPL_TYPE_FLOAT.
 * @param   ref_wave_tbl     (Given)    An optional reference
 *                                      wavelength solution table, or
 *                                      NULL.
 * @param   ref_wave_hdr     (Given)    The FITS header for the
 *                                      reference wavelength solution
 *                                      table, or NULL if no table.
 * @param   linelist_tbl     (Given)    The line list for the arc
 *                                      lamp.
 * @param   trace_tbl        (Given)    The trace table used in
 *                                      spectral extraction.
 * @param   trace_hdr        (Given)    The corresponding FITS
 *                                      header for the trace table.
 * @param   detthr           (Given)    The detection threshold for
 *                                      finding emission features in
 *                                      units of the background
 *                                      sigma.
 * @param   rejthr           (Given)    The rejection threshold for
 *                                      the fitting procedure in units
 *                                      of the polynomial fit RMS.
 * @param   nord             (Given)    The degree of the polynomial
 *                                      fit.
 * @param   matchwindow      (Given)    Matching threshold specifying
 *                                      the maximum deviation between
 *                                      the predicted position of an
 *                                      arc line and the position of a
 *                                      detected emission feature in
 *                                      wavelength units (Angstroms)
 *                                      at 6000A for it to be
 *                                      considered a match.  At other
 *                                      wavelengths, it is scaled with
 *                                      wavelength such that the
 *                                      matching window is constant in
 *                                      velocity, or equivalently
 *                                      resolving power, to better
 *                                      match the behaviour of the
 *                                      resolution of the spectrograph
 *                                      itself.
 * @param   matchgrid        (Given)    Search limit for optional
 *                                      initial grid search in
 *                                      wavelength units (Angstroms).
 *                                      An initial grid search out to
 *                                      +/- matchgrid will be run
 *                                      prior to the final match
 *                                      against the line list to deal
 *                                      with shifts of the wavelength
 *                                      solution relative to the
 *                                      reference.  A value of 0
 *                                      disables the grid search.
 *                                      This should be disabled for
 *                                      FPE spectra.
 * @param   out_wave_tbl     (Returned) The resulting wavelength
 *                                      solution table.  This table
 *                                      (described in the Data
 *                                      Reduction Pipeline Description
 *                                      document, Section 6.7)
 *                                      contains a row per fibre with
 *                                      the wavelength solution
 *                                      polynomial coefficients, the
 *                                      measured arc line positions,
 *                                      spectral FWHM, and the true
 *                                      arc line wavelengths used for
 *                                      the fit from the line list.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS header keywords was not
 *                                        found.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the trace table
 *                                        dimensions don't match the
 *                                        spectrum or reference
 *                                        wavelength surface image.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the input image data type
 *                                        was not float, or if an
 *                                        input FITS header keyword
 *                                        value had an incorrect data
 *                                        type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS WVCRV</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *   - <b>WVCRV</b>
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_doarcs_ref(
    cpl_image *in_spec,
    cpl_propertylist *in_hdr,
    cpl_image *ref_wave_surface,
    cpl_table *ref_wave_tbl,
    cpl_propertylist *ref_wave_hdr,
    cpl_table *linelist_tbl,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    float detthr,
    float rejthr,
    int nord,
    float matchwindow,
    float matchgrid,
    cpl_table **out_wave_tbl)
{
    long naxis[2];
    int specbin,spatbin,isbinned;
    float tcrv, cenpix;
    float *in_spec_buf = NULL;
    float *indata = NULL;

    long naxisr[2];
    float *refdata = NULL;

    int i,nfeat,fiblive,k,j,jrem,ngood;
    int nfit,lgrid,lmax,nmatch,nmax,l;
    int iline,nlines,nlinesmax;
    int isnull;
    double xpos,dx,best,zwin,diff,sumdiff,sdmax,rmsfit,ypos,cor;
    double medresid,xx,w1,w2;
    float waveguess,waveref,offset;
    int iref, itmp, foundl, foundh;

    /* Garbage collected stuff */

    int ntr = 0;
    qmost_traceinfo *tr = NULL;

    int nwv_ref = 0;
    qmost_waveinfo *wv_ref = NULL;
    int ref_minyst = 1;

    int nwv_out = 0;
    qmost_waveinfo *wv_out = NULL, *wv;

    featstruct *xfeatstr = NULL;

    double *farray=NULL,*ldata=NULL,*lbest=NULL,*xfit=NULL;
    double *yfit=NULL;
    int *lmatch=NULL;
    cpl_polynomial *rcoefs = NULL;
    cpl_errorstate prestate;

    /* Check for NULL arguments, particularly the output so we can
       initialize it for garbage collection. */
    cpl_ensure_code(in_spec, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_wave_surface, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(linelist_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_wave_tbl, CPL_ERROR_NULL_INPUT);

    /* Initialize this for garbage collection */
    *out_wave_tbl = NULL;

#undef TIDY
#define TIDY                                    \
    if(tr != NULL) {                            \
        qmost_trclose(ntr, &tr);                \
        tr = NULL;                              \
        ntr = 0;                                \
    }                                           \
    if(wv_ref != NULL) {                        \
        qmost_wvclose(nwv_ref, &wv_ref);        \
        wv_ref = NULL;                          \
        nwv_ref = 0;                            \
    }                                           \
    if(wv_out != NULL) {                        \
        qmost_wvclose(nwv_out, &wv_out);        \
        wv_out = NULL;                          \
        nwv_out = 0;                            \
    }                                           \
    if(ldata != NULL) {                         \
        cpl_free(ldata);                        \
        ldata = NULL;                           \
    }                                           \
    if(lbest != NULL) {                         \
        cpl_free(lbest);                        \
        lbest = NULL;                           \
    }                                           \
    if(lmatch != NULL) {                        \
        cpl_free(lmatch);                       \
        lmatch = NULL;                          \
    }                                           \
    if(xfeatstr != NULL) {                      \
        cpl_free(xfeatstr);                     \
        xfeatstr = NULL;                        \
    }                                           \
    if(xfit != NULL) {                          \
        cpl_free(xfit);                         \
        xfit = NULL;                            \
    }                                           \
    if(yfit != NULL) {                          \
        cpl_free(yfit);                         \
        yfit = NULL;                            \
    }                                           \
    if(rcoefs != NULL) {                        \
        cpl_polynomial_delete(rcoefs);          \
        rcoefs = NULL;                          \
    }                                           \
    if(farray != NULL) {                        \
        cpl_free(farray);                       \
        farray = NULL;                          \
    }

    /* Get input 2D spectrum */
    naxis[0] = cpl_image_get_size_x(in_spec);
    naxis[1] = cpl_image_get_size_y(in_spec);

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    tcrv = 1.0;
    if(isbinned) {
        if(cpl_propertylist_has(in_hdr, "ESO DRS WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_hdr,
                                                "ESO DRS WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read ESO DRS WVCRV "
                                             "from input FITS header");
            }
        }
        else if(cpl_propertylist_has(in_hdr, "WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_hdr,
                                                "WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read WVCRV "
                                             "from input FITS header");
            }
        }
    }

    cenpix = 0.5 * specbin * naxis[0];

    /* Get pointer to image data */
    in_spec_buf = cpl_image_get_data_float(in_spec);
    if(in_spec_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Open the trace table to be used to see which fibres are broken. There
       should be the same number of rows in the trace file as there are
       spectra in the input file */

    if (qmost_tropen(trace_tbl,trace_hdr,&ntr,&tr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "problem reading trace table");
    }
    if (ntr != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "trace table ntrace=%d doesn't "
                                     "match nspec=%ld",
                                     ntr, naxis[1]);
    }

    /* Get reference wavelength surface (image of lambda vs x,y) */
    naxisr[0] = cpl_image_get_size_x(ref_wave_surface);
    naxisr[1] = cpl_image_get_size_y(ref_wave_surface);

    refdata = cpl_image_get_data_float(ref_wave_surface);
    if(refdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "reference wavelength surface image");
    }

    /* Read the wave info for the reference arc */

    if(ref_wave_tbl != NULL) {
        if(qmost_wvopen(ref_wave_tbl,&nwv_ref,&wv_ref) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "problem reading reference "
                                         "wavelength solution table");
        }
    }

    if(ntr > 0) {
        ref_minyst = tr[0].minyst;
    }

    if(ref_wave_hdr != NULL) {
        if(qmost_pfits_get_minyst(ref_wave_hdr,
                                  &ref_minyst) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read minimum yst");
        }
    }

    /* Read the line list */

    nlines = cpl_table_get_nrow(linelist_tbl);

    ldata = cpl_malloc(nlines*sizeof(double));
    lbest = cpl_malloc(nlines*sizeof(double));
    lmatch = cpl_malloc(nlines*sizeof(int));

    for(iline = 0; iline < nlines; iline++) {
        ldata[iline] = cpl_table_get(linelist_tbl, "Wavelength",
                                     iline, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read wavelength "
                                         "column for row %d",
                                         iline+1);
        }
        else if(isnull > 0) {
            ldata[iline] = NAN;
        }
    }

    /* Allocate blank array for output waveinfo structures.  This
       needs to be initialized to zero for garbage collection. */
    wv_out = cpl_calloc(naxis[1], sizeof(qmost_waveinfo));
    nwv_out = naxis[1];

    /* Convert match window to redshift */
    zwin = matchwindow / REFWAVE;

    /* Grid search limit */
    lgrid = qmost_nint(2 * matchgrid / matchwindow);
    if(lgrid < 0) {
        lgrid = 0;
    }

    /* Loop for each spectrum in the map */

    for (i = 1; i <= naxis[1]; i++) {
        wv = wv_out + (i-1);

	/* Is this fibre live? */

        fiblive = tr[i-1].live;
	if (! fiblive) {
            wv->specnum = i;
            wv->live = 0;
	    continue;
	}

        /* Check if fibre is live in ref table.  If it is, use it.
         * If not, find the closest live fibre. */
        iref = -1;
            
        if(ref_wave_tbl != NULL) {
            if(i <= nwv_ref && wv_ref[i-1].live) {
                iref = i;
            }
            else {
                /* Search on both sides for the closest live */
                foundl = -1;
                
                for(itmp = qmost_min(i-1, nwv_ref); itmp > 0; itmp--) {
                    if(wv_ref[itmp-1].live) {
                        foundl = itmp;
                    }
                }
                
                foundh = -1;
                
                for(itmp = i+1; itmp <= nwv_ref; itmp++) {
                    if(wv_ref[itmp-1].live) {
                        foundh = itmp;
                    }
                }
                
                /* Take the closer one */
                if(foundl > 0 && foundh > 0) {
                    if(i-foundl <= foundh-i) {
                        iref = foundl;
                    }
                    else {
                        iref = foundh;
                    }
                }
                else if(foundl > 0) {
                    iref = foundl;
                }
                else if(foundh > 0) {
                    iref = foundh;
                }
                /* else fall back to wave map */
            }
        }

	/* Get pointer to the relevant subset of the image */
	
        indata = in_spec_buf + (i-1) * naxis[0];

	/* Do the detection */

	detect_lines(indata,naxis[0],detthr,&nfeat,&xfeatstr);

	/* Initialise some stuff */

	for (k = 0; k < nfeat; k++) {
            /* Transform from binned to unbinned pixels for the rest
             * of the calculation. */
            xfeatstr[k].xfeat = tcrv + specbin * (xfeatstr[k].xfeat - 1);

	    dx = xfeatstr[k].xfeat - cenpix;
	    xfeatstr[k].xfeat_norm = dx;
	    xfeatstr[k].wave_resid = 0.0;
	    xfeatstr[k].matched_index = -1;
	    xfeatstr[k].wavematched = 0.0;
	    xfeatstr[k].errwavematched = 0.0;
	    xfeatstr[k].bad = 1;

            /* Get an initial guess at the wavelength from the 
               reference map */
            
            ypos = xfeatstr[k].xfeat + (double)(tr[i-1].minyst - 1);
            xpos = qmost_tracexpos(tr[i-1],ypos);

            waveguess = 0;  /* prevent uninitialized warning */

            if(eval_wave_map(refdata, naxisr,
                             xpos, ypos,
                             &waveguess) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             cpl_error_get_code(),
                                             "couldn't get initial guess "
                                             "of wavelength from "
                                             "wavelength surface map");
            }

            xfeatstr[k].wavecalc = waveguess;

            /* Alternative get wavelengths from reference wavelength solution */

	    if (ref_wave_tbl != NULL && iref > 0) {
                xpos = qmost_tracexpos(tr[iref-1],ypos);

                /* Get reference map value and use it to correct for
                 * expected wavelength difference between reference x
                 * position and desired x position.  It's not an
                 * error if we can't do the lookup, we just fall back
                 * to assuming the difference is zero. */
                waveref = waveguess;

                prestate = cpl_errorstate_get();

                if(eval_wave_map(refdata, naxisr,
                                 xpos, ypos,
                                 &waveref) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                }

                waveguess += cpl_polynomial_eval_1d(
                    wv_ref[iref-1].coefs,
                    xfeatstr[k].xfeat-wv_ref[iref-1].xref +
                    tr[i-1].minyst - ref_minyst,
                    NULL) - waveref;
                xfeatstr[k].wavecalc = waveguess;
	    }
	}

        /* Do a coarse grid search to find best offset */

        nmax = 0;
        sdmax = -1;
        lmax = 0;
        for (l = -lgrid; l <= lgrid; l++) {
            offset = 0.5*matchwindow*(float)l;
            nmatch = 0;
            sumdiff = 0;
            for (k = 0; k < nfeat; k++) {
                for (j = 0; j < nlines; j++) {
                    diff = fabs(xfeatstr[k].wavecalc + offset - ldata[j]);
                    if (diff < zwin * xfeatstr[k].wavecalc) {
                        nmatch++;
                        sumdiff += diff;
                    }
                }
            }
            if (nmatch > nmax || (nmatch == nmax && sumdiff < sdmax)) {
                nmax = nmatch;
                sdmax = sumdiff;
                lmax = l;
            }
        }
        offset = 0.5*matchwindow*(float)lmax;

        /* Adjust initial guesses for offset */

        for (k = 0; k < nfeat; k++) {
            xfeatstr[k].wavecalc += offset;
        }

        /* Initialise a few more things to keep track of which lines 
           are matched */
        
        for (k = 0; k < nlines; k++) {
            lbest[k] = 1.0e10;
            lmatch[k] = -1;
        }

        /* OK,loop through each of the detected features and try to work out
           which is the best line to match it from the line list. Make sure
           it's only ever a 1 to 1 match */
        
        for (k = 0; k < nfeat; k++) {
            best = 1.0e10;
            jrem = -1;
            for (j = 0; j < nlines; j++) {
                diff = fabs(xfeatstr[k].wavecalc - ldata[j]);
                if (diff < best && diff < zwin * xfeatstr[k].wavecalc) {
                    best = diff;
                    jrem = j;
                }
            }

            /* If no suitable match exists, then move on */
            
            if (jrem == -1)
                continue;

            /* If a match was found and that particular line has already
               been assigned to another peak, then see if this latest one 
               is better or not */
            
            if (lmatch[jrem] != -1) {
                if (fabs(lbest[jrem]) < best) {
                    continue;
                } else {
                    lbest[jrem] = xfeatstr[k].wavecalc - ldata[jrem];
                    lmatch[jrem] = k;
                }
            } else {
                lbest[jrem] = xfeatstr[k].wavecalc - ldata[jrem];
                lmatch[jrem] = k;
            }
        }

        /* Do a low order fit to the residuals versus pixel position. 
           Work out a correction to the wavelength guess and apply it */

        xfit = cpl_malloc(nlines*sizeof(double));
        yfit = cpl_malloc(nlines*sizeof(double));

        nfit = 0;
        for (k = 0; k < nlines; k++) {
            if (lmatch[k] == -1)
                continue;
            xfit[nfit] = xfeatstr[lmatch[k]].xfeat;
            yfit[nfit] = lbest[k];
            nfit++;
        }
        if (nfit > 3) {
            prestate = cpl_errorstate_get();

            rcoefs = qmost_polynm(xfit,yfit,nfit,1,0);
            if(rcoefs == NULL) {
                switch(cpl_error_get_code()) {
                case CPL_ERROR_DATA_NOT_FOUND:
                    cpl_errorstate_set(prestate);
                    
                    cpl_msg_warning(cpl_func,
                                    "not enough valid points for linear "
                                    "fit to fibre %d: %d",
                                    i, nfit);
                    
                    rcoefs = cpl_polynomial_new(1);
                
                    break;
                case CPL_ERROR_SINGULAR_MATRIX:
                    cpl_errorstate_set(prestate);
                    
                    cpl_msg_warning(cpl_func,
                                    "singular matrix in linear "
                                    "fit to fibre %d with %d data points",
                                    i, nfit);
                
                    rcoefs = cpl_polynomial_new(1);
                
                    break;
                default:
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not fit straight line "
                                                 "to %d data points for "
                                                 "fibre %d", nfit, i);
                }
            }

            for (k = 0; k < nfeat; k++) {
                cor = cpl_polynomial_eval_1d(rcoefs,xfeatstr[k].xfeat,NULL);
		xfeatstr[k].wavecalc -= cor;
            }

            cpl_polynomial_delete(rcoefs);
            rcoefs = NULL;
        }

        cpl_free(xfit);
        xfit = NULL;

        cpl_free(yfit);
        yfit = NULL;

        /* Now try reassigning them with the corrected guesses */

        for (k = 0; k < nlines; k++) {
            lbest[k] = 1.0e10;
            lmatch[k] = -1;
        }        
        for (k = 0; k < nfeat; k++) {
            best = 1.0e10;
            jrem = -1;
            for (j = 0; j < nlines; j++) {
                diff = fabs(xfeatstr[k].wavecalc - ldata[j]);
                if (diff < best && diff < zwin * xfeatstr[k].wavecalc) {  
                    best = diff;
                    jrem = j;
                }
            }
                
            if (jrem != -1) {
                if (lmatch[jrem] != -1) {
                    if (fabs(lbest[jrem]) < best) {
                        continue;
                    } else {
                        xfeatstr[lmatch[jrem]].wavematched = 0.0;
                        xfeatstr[lmatch[jrem]].matched_index = -1;
                        xfeatstr[lmatch[jrem]].bad = 1;
                        xfeatstr[k].wavematched = ldata[jrem];
                        xfeatstr[k].matched_index = jrem;
                        xfeatstr[k].bad = 0;
                        lmatch[jrem] = k;
                        lbest[jrem] = xfeatstr[k].wavecalc - ldata[jrem];
                    }
                } else {
                    xfeatstr[k].wavematched = ldata[jrem];
                    xfeatstr[k].matched_index = jrem;
                    xfeatstr[k].bad = 0;
                    lmatch[jrem] = k;
                    lbest[jrem] = xfeatstr[k].wavecalc - ldata[jrem];
                }
            }
        }
        
	/* Do the real wavelength fit */

        wv->coefs = dofit(nfeat,xfeatstr,nord,rejthr,&ngood,&rmsfit,1);
        if(wv->coefs == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not fit wavelength "
                                         "solution for fibre %d",
                                         i);
        }
	for (k = 0; k < nfeat; k++) 
	    xfeatstr[k].wavecalc = cpl_polynomial_eval_1d(wv->coefs,
                                                    xfeatstr[k].xfeat_norm,
                                                    NULL);
        /* Now work out the FWHM in angstroms */

        for (k = 0; k < nfeat; k++) {
            xx = xfeatstr[k].xfeat_norm - 0.5*(xfeatstr[k].fwhm);
            w1 = cpl_polynomial_eval_1d(wv->coefs,xx,NULL);
            xx = xfeatstr[k].xfeat_norm + 0.5*(xfeatstr[k].fwhm);
            w2 = cpl_polynomial_eval_1d(wv->coefs,xx,NULL);
            xfeatstr[k].fwhm = (float)(w2 - w1);
        }

        /* Work out the median residual for the fit */

	farray = cpl_malloc(((unsigned int) nfeat)*sizeof(double));

	j = 0;
	for (k = 0; k < nfeat; k++) {
	    if (xfeatstr[k].bad == 0) 
		farray[j++] = xfeatstr[k].wavematched - xfeatstr[k].wavecalc;
	}

        if(j > 0) {
            if(qmost_dmed(farray,NULL,j,&medresid) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute median "
                                             "fit residual for fibre %d",
                                             i);
            }
        }
        else {
            medresid = 0;
        }

        cpl_free(farray);
        farray = NULL;

        /* Create rest of waveinfo structure */

        wv->specnum = i;
        wv->live = 1;
        wv->nord = nord;
        wv->xref = cenpix;
        wv->ngood = ngood;
        wv->medresid = medresid;
        wv->fit_rms = rmsfit;
	wv->wave1 = cpl_polynomial_eval_1d(
            wv->coefs,
            tcrv - cenpix,
            NULL);
	wv->waven = cpl_polynomial_eval_1d(
            wv->coefs,
            tcrv + specbin * (naxis[0] - 1) - cenpix,
            NULL);
	wv->dwave = (wv->waven - wv->wave1)/(double)((naxis[0]-1)*specbin);
        wv->nlines = nfeat;
        wv->xpos = cpl_malloc(nfeat*sizeof(double));
        wv->fwhm = cpl_malloc(nfeat*sizeof(float));
        wv->wave_calc = cpl_malloc(nfeat*sizeof(double));
        wv->wave_true = cpl_malloc(nfeat*sizeof(double));
        wv->fit_flag = cpl_malloc(nfeat*sizeof(unsigned char));
        wv->wave_cor = cpl_calloc(nfeat,sizeof(double));
        wv->peak = cpl_malloc(nfeat*sizeof(float));
        wv->contrast = cpl_malloc(nfeat*sizeof(float));

	for (k = 0; k < nfeat; k++) {
	    wv->xpos[k] = xfeatstr[k].xfeat;
            if(xfeatstr[k].fwhm > 0) {
                wv->fwhm[k] = qmost_max(wv->dwave,xfeatstr[k].fwhm);
            }
            else {
                wv->fwhm[k] = -1.0;
            }
	    wv->wave_calc[k] = xfeatstr[k].wavecalc;
	    wv->wave_true[k] = xfeatstr[k].wavematched;
            wv->fit_flag[k] = xfeatstr[k].bad;
            wv->peak[k] = xfeatstr[k].tmax;
            wv->contrast[k] = xfeatstr[k].contrast;
        }

	cpl_free(xfeatstr);
        xfeatstr = NULL;
    }

    /* Find maximum number of lines for sizing output table */
    nlinesmax = 1;

    for(i = 1; i <= naxis[1]; i++) {
        wv = wv_out + (i-1);

        if(wv->nlines > nlinesmax)
            nlinesmax = wv->nlines;
    }

    /* Write output */
    if(qmost_wvcreate(out_wave_tbl, nord, nlinesmax) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output "
                                     "waveinfo table with order %d "
                                     "max lines lines %d",
                                     nord, nlinesmax);
    }

    for(i = 1; i <= naxis[1]; i++) {
        wv = wv_out + (i-1);

        if(qmost_wvwrite1(*out_wave_tbl, i, *wv) != CPL_ERROR_NONE) {
            TIDY;

            if(*out_wave_tbl != NULL) {
                cpl_table_delete(*out_wave_tbl);
                *out_wave_tbl = NULL;
            }

            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not write wavelength "
                                         "solution for fibre %d to "
                                         "waveinfo table",
                                         i);
        }
    }

    /* Right, close things up */
    TIDY;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do OB-level arc line identification and compute correction
 *          to master wavelength solution.
 *
 * Emission line features are identified in spectra. These are then
 * assigned wavelengths based on a master wavelength solution. The 
 * resulting OB-level wavelength correction is then calculated for
 * each fibre from the residuals of the OB-level arc compared to the
 * master wavelength solution.  The results are recorded in a
 * wavelength solution table.
 *
 * @param   in_spec          (Given)    The extracted arc spectra
 *                                      to analyse, as an image.  The
 *                                      data type must be
 *                                      CPL_TYPE_FLOAT.
 * @param   in_hdr           (Given)    The FITS header of the input
 *                                      spectra with binning
 *                                      information.
 * @param   ref_wave_tbl     (Given)    The master wavelength solution
 *                                      table.
 * @param   ref_wave_hdr     (Given)    The FITS header for the
 *                                      master wavelength solution
 *                                      table.
 * @param   linelist_tbl     (Given)    The line list for the arc
 *                                      lamp.
 * @param   trace_tbl        (Given)    The trace table used in
 *                                      spectral extraction.
 * @param   trace_hdr        (Given)    The corresponding FITS
 *                                      header for the trace table.
 * @param   detthr           (Given)    The detection threshold for
 *                                      finding emission features in
 *                                      units of the background
 *                                      sigma.
 * @param   rejthr           (Given)    The rejection threshold for
 *                                      the fitting procedure in units
 *                                      of the polynomial fit RMS.
 * @param   nord             (Given)    The degree of the polynomial
 *                                      fit.
 * @param   matchwindow      (Given)    Matching threshold specifying
 *                                      the maximum deviation between
 *                                      the predicted position of an
 *                                      arc line and the position of a
 *                                      detected emission feature in
 *                                      wavelength units (Angstroms)
 *                                      at 6000A for it to be
 *                                      considered a match.  At other
 *                                      wavelengths, it is scaled with
 *                                      wavelength such that the
 *                                      matching window is constant in
 *                                      velocity, or equivalently
 *                                      resolving power, to better
 *                                      match the behaviour of the
 *                                      resolution of the spectrograph
 *                                      itself.
 * @param   matchgrid        (Given)    Search limit for optional
 *                                      initial grid search in
 *                                      wavelength units (Angstroms).
 *                                      An initial grid search out to
 *                                      +/- matchgrid will be run
 *                                      prior to the final match
 *                                      against the line list to deal
 *                                      with shifts of the wavelength
 *                                      solution relative to the
 *                                      master.  A value of 0
 *                                      disables the grid search.
 *                                      This should be disabled for
 *                                      FPE spectra.
 * @param   out_wave_tbl     (Returned) The resulting wavelength
 *                                      solution table containing the
 *                                      correction to the master
 *                                      wavelength solution.  This
 *                                      table (described in the Data
 *                                      Reduction Pipeline Description
 *                                      document, Section 6.7)
 *                                      contains a row per fibre with
 *                                      the wavelength solution
 *                                      correction polynomial
 *                                      coefficients, the measured arc
 *                                      line positions, spectral FWHM,
 *                                      and the true arc line
 *                                      wavelengths used for the fit
 *                                      from the line list.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS header keywords was not
 *                                        found, or if there were no
 *                                        live fibres in the reference
 *                                        wavelength solution table.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the trace table
 *                                        dimensions don't match the
 *                                        spectrum or reference
 *                                        wavelength surface image.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the input image data type
 *                                        was not float, or if an
 *                                        input FITS header keyword
 *                                        value had an incorrect data
 *                                        type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS WVCRV</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *   - <b>WVCRV</b>
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_doarcs_ob(
    cpl_image *in_spec,
    cpl_propertylist *in_hdr,
    cpl_table *ref_wave_tbl,
    cpl_propertylist *ref_wave_hdr,
    cpl_table *linelist_tbl,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    float detthr,
    float rejthr,
    int nord,
    float matchwindow,
    float matchgrid,
    cpl_table **out_wave_tbl)
{
    long naxis[2];
    int specbin,spatbin,isbinned;
    float tcrv, cenpix;
    float *in_spec_buf = NULL;
    float *indata = NULL;

    int i,nfeat,nnofeat,fiblive,k,j,jrem,ngood;
    int lgrid,lmax,nmatch,nmax,l;
    int iline,nlines,nlinesmax;
    int isnull;
    double dx,best,zwin,diff,sumdiff,sdmax,rmsfit;
    double medresid,xx,w1,w2;
    float waveguess,offset;
    double dval;
    int iref, itmp, foundl, foundh;
    
    /* Garbage collected stuff */

    int ntr = 0;
    qmost_traceinfo *tr = NULL;

    int nwv_ref = 0;
    qmost_waveinfo *wv_ref = NULL;
    int ref_minyst = 1;

    int nwv_out = 0;
    qmost_waveinfo *wv_out = NULL, *wv;

    featstruct *xfeatstr = NULL;

    double *farray=NULL,*ldata=NULL,*lbest=NULL;
    int *lmatch=NULL;

    /* Check for NULL arguments, particularly the output so we can
       initialize it for garbage collection. */
    cpl_ensure_code(in_spec, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_wave_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_wave_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(linelist_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_wave_tbl, CPL_ERROR_NULL_INPUT);

    /* Initialize this for garbage collection */
    *out_wave_tbl = NULL;

#undef TIDY
#define TIDY                                    \
    if(tr != NULL) {                            \
        qmost_trclose(ntr, &tr);                \
        tr = NULL;                              \
        ntr = 0;                                \
    }                                           \
    if(wv_ref != NULL) {                        \
        qmost_wvclose(nwv_ref, &wv_ref);        \
        wv_ref = NULL;                          \
        nwv_ref = 0;                            \
    }                                           \
    if(wv_out != NULL) {                        \
        qmost_wvclose(nwv_out, &wv_out);        \
        wv_out = NULL;                          \
        nwv_out = 0;                            \
    }                                           \
    if(ldata != NULL) {                         \
        cpl_free(ldata);                        \
        ldata = NULL;                           \
    }                                           \
    if(lbest != NULL) {                         \
        cpl_free(lbest);                        \
        lbest = NULL;                           \
    }                                           \
    if(lmatch != NULL) {                        \
        cpl_free(lmatch);                       \
        lmatch = NULL;                          \
    }                                           \
    if(xfeatstr != NULL) {                      \
        cpl_free(xfeatstr);                     \
        xfeatstr = NULL;                        \
    }                                           \
    if(farray != NULL) {                        \
        cpl_free(farray);                       \
        farray = NULL;                          \
    }

    /* Get input 2D spectrum */
    naxis[0] = cpl_image_get_size_x(in_spec);
    naxis[1] = cpl_image_get_size_y(in_spec);

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    tcrv = 1.0;
    if(isbinned) {
        if(cpl_propertylist_has(in_hdr, "ESO DRS WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_hdr,
                                                "ESO DRS WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read ESO DRS WVCRV "
                                             "from input FITS header");
            }
        }
        else if(cpl_propertylist_has(in_hdr, "WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_hdr,
                                                "WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read WVCRV "
                                             "from input FITS header");
            }
        }
    }

    cenpix = 0.5 * specbin * naxis[0];

    /* Get pointer to image data */
    in_spec_buf = cpl_image_get_data_float(in_spec);
    if(in_spec_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Open the trace table to be used to see which fibres are broken. There
       should be the same number of rows in the trace file as there are
       spectra in the input file */

    if (qmost_tropen(trace_tbl,trace_hdr,&ntr,&tr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "problem reading trace table");
    }
    if (ntr != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "trace table ntrace=%d doesn't "
                                     "match nspec=%ld",
                                     ntr, naxis[1]);
    }

    /* Read the wave info for the reference arc */

    if(qmost_wvopen(ref_wave_tbl,&nwv_ref,&wv_ref) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "problem reading reference "
                                     "wavelength solution table");
    }

    if(ntr > 0) {
        ref_minyst = tr[0].minyst;
    }

    if(qmost_pfits_get_minyst(ref_wave_hdr,
                              &ref_minyst) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read minimum yst");
    }

    /* Read the line list */

    nlines = cpl_table_get_nrow(linelist_tbl);

    ldata = cpl_malloc(nlines*sizeof(double));
    lbest = cpl_malloc(nlines*sizeof(double));
    lmatch = cpl_malloc(nlines*sizeof(int));

    for(iline = 0; iline < nlines; iline++) {
        ldata[iline] = cpl_table_get(linelist_tbl, "Wavelength",
                                     iline, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read wavelength "
                                         "column for row %d",
                                         iline+1);
        }
        else if(isnull > 0) {
            ldata[iline] = NAN;
        }
    }

    /* Allocate blank array for output waveinfo structures.  This
       needs to be initialized to zero for garbage collection. */
    wv_out = cpl_calloc(naxis[1], sizeof(qmost_waveinfo));
    nwv_out = naxis[1];

    /* Convert match window to redshift */
    zwin = matchwindow / REFWAVE;

    /* Grid search limit */
    lgrid = qmost_nint(2 * matchgrid / matchwindow);
    if(lgrid < 0) {
        lgrid = 0;
    }

    /* Loop for each spectrum in the map */

    nnofeat = 0;
    for (i = 1; i <= naxis[1]; i++) {
        wv = wv_out + (i-1);

	/* Is this fibre live? */

        fiblive = tr[i-1].live;
	if (! fiblive) {
            wv->specnum = i;
            wv->live = 0;
	    continue;
	}

        /* Check if fibre is live in ref table.  If it is, use it.
         * If not, find the closest live fibre. */
        iref = -1;
            
        if(i <= nwv_ref && wv_ref[i-1].live) {
            iref = i;
        }
        else {
            /* Search on both sides for the closest live */
            foundl = -1;
                
            for(itmp = i-1; itmp > 0; itmp--) {
                if(wv_ref[itmp-1].live) {
                    foundl = itmp;
                }
            }
                
            foundh = -1;
                
            for(itmp = i+1; itmp <= nwv_ref; itmp++) {
                if(wv_ref[itmp-1].live) {
                    foundh = itmp;
                }
            }
                
            /* Take the closer one */
            if(foundl > 0 && foundh > 0) {
                if(i-foundl <= foundh-i) {
                    iref = foundl;
                }
                else {
                    iref = foundh;
                }
            }
            else if(foundl > 0) {
                iref = foundl;
            }
            else if(foundh > 0) {
                iref = foundh;
            }
            else {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_DATA_NOT_FOUND,
                                             "there were no live fibres "
                                             "in the reference wavelength "
                                             "solution table");
            }
        }

	/* Get pointer to the relevant subset of the image */
	
        indata = in_spec_buf + (i-1) * naxis[0];

	/* Do the detection */

	detect_lines(indata,naxis[0],detthr,&nfeat,&xfeatstr);

        if(nfeat == 0) {
            wv->specnum = i;
            wv->live = 1;
            wv->nord = -1;
            wv->ngood = 0;
            wv->nlines = 0;
            nnofeat++;
            continue;
        }
        
	/* Initialise some stuff */

	for (k = 0; k < nfeat; k++) {
            /* Transform from binned to unbinned pixels for the rest
             * of the calculation. */
            xfeatstr[k].xfeat = tcrv + specbin * (xfeatstr[k].xfeat - 1);

	    dx = xfeatstr[k].xfeat - cenpix;
	    xfeatstr[k].xfeat_norm = dx;
	    xfeatstr[k].wave_resid = 0.0;
	    xfeatstr[k].matched_index = -1;
	    xfeatstr[k].wavematched = 0.0;
	    xfeatstr[k].errwavematched = 0.0;
	    xfeatstr[k].bad = 1;

            /* Get initial guess at wavelength from the reference */

            waveguess = cpl_polynomial_eval_1d(
                wv_ref[iref-1].coefs,
                xfeatstr[k].xfeat-wv_ref[iref-1].xref +
                tr[i-1].minyst - ref_minyst,
                NULL);
            xfeatstr[k].wavecalc = waveguess;
	}

        /* Do a coarse grid search to find best offset */

        nmax = 0;
        sdmax = -1;
        lmax = 0;
        for (l = -lgrid; l <= lgrid; l++) {
            offset = 0.5*matchwindow*(float)l;
            nmatch = 0;
            sumdiff = 0;
            for (k = 0; k < nfeat; k++) {
                for (j = 0; j < nlines; j++) {
                    diff = fabs(xfeatstr[k].wavecalc + offset - ldata[j]);
                    if (diff < matchwindow) {
                        nmatch++;
                        sumdiff += diff;
                    }
                }
            }
            if (nmatch > nmax || (nmatch == nmax && sumdiff < sdmax)) {
                nmax = nmatch;
                sdmax = sumdiff;
                lmax = l;
            }
        }
        offset = 0.5*matchwindow*(float)lmax;

        /* Initialise a few more things to keep track of which lines 
           are matched */
        
        for (k = 0; k < nlines; k++) {
            lbest[k] = 1.0e10;
            lmatch[k] = -1;
        }

        for (k = 0; k < nfeat; k++) {
            best = 1.0e10;
            jrem = -1;
            for (j = 0; j < nlines; j++) {
                diff = fabs(xfeatstr[k].wavecalc + offset - ldata[j]);
                if (diff < best && diff < zwin * xfeatstr[k].wavecalc) {  
                    best = diff;
                    jrem = j;
                }
            }
                
            if (jrem != -1) {
                if (lmatch[jrem] != -1) {
                    if (fabs(lbest[jrem]) < best) {
                        continue;
                    } else {
                        xfeatstr[lmatch[jrem]].wavematched = 0.0;
                        xfeatstr[lmatch[jrem]].matched_index = -1;
                        xfeatstr[lmatch[jrem]].bad = 1;
                        xfeatstr[k].wavematched = ldata[jrem];
                        xfeatstr[k].matched_index = jrem;
                        xfeatstr[k].bad = 0;
                        lmatch[jrem] = k;
                        lbest[jrem] = xfeatstr[k].wavecalc + offset - ldata[jrem];
                    }
                } else {
                    xfeatstr[k].wavematched = ldata[jrem];
                    xfeatstr[k].matched_index = jrem;
                    xfeatstr[k].bad = 0;
                    lmatch[jrem] = k;
                    lbest[jrem] = xfeatstr[k].wavecalc + offset - ldata[jrem];
                }
            }
        }
        
        /* Calculate what the wavelengths for these pixel positions 
           would be using the reference wavelength solution */

        for (k = 0; k < nfeat; k++) {
            dval = cpl_polynomial_eval_1d(
                wv_ref[iref-1].coefs,
                xfeatstr[k].xfeat-wv_ref[iref-1].xref +
                tr[i-1].minyst - ref_minyst,
                NULL);
            xfeatstr[k].wavecalc = dval;
            if (! xfeatstr[k].bad) 
                xfeatstr[k].wave_resid = xfeatstr[k].wavematched - dval;
            else 
                xfeatstr[k].wave_resid = 0.0;
        }

        /* Do the fit to the offsets */

	wv->coefs = dofit(nfeat,xfeatstr,nord,rejthr,&ngood,&rmsfit,2);
        if(wv->coefs == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not fit wavelength "
                                         "solution for fibre %d",
                                         i);
        }

        /* Now work out the FWHM in angstroms */

        for (k = 0; k < nfeat; k++) {
            xx = xfeatstr[k].xfeat_norm - 0.5*(xfeatstr[k].fwhm);
            dval = cpl_polynomial_eval_1d(wv_ref[iref-1].coefs,
                                          xx + tr[i-1].minyst - ref_minyst,
                                          NULL);
            w1 = dval + cpl_polynomial_eval_1d(wv->coefs,xx,NULL);
            xx = xfeatstr[k].xfeat_norm + 0.5*(xfeatstr[k].fwhm);
            dval = cpl_polynomial_eval_1d(wv_ref[iref-1].coefs,
                                          xx + tr[i-1].minyst - ref_minyst
                                          ,NULL);
            w2 = dval + cpl_polynomial_eval_1d(wv->coefs,xx,NULL);
            xfeatstr[k].fwhm = (float)(w2 - w1);
        }

        /* Work out the median residual for the fit */

	farray = cpl_malloc(((unsigned int) nfeat)*sizeof(double));

	j = 0;
	for (k = 0; k < nfeat; k++) {
	    if (xfeatstr[k].bad == 0) 
		farray[j++] = xfeatstr[k].wavematched - xfeatstr[k].wavecalc;
	}

        if(j > 0) {
            if(qmost_dmed(farray,NULL,j,&medresid) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute median "
                                             "fit residual for fibre %d",
                                             i);
            }
        }
        else {
            medresid = 0;
        }

        cpl_free(farray);
        farray = NULL;

        /* Create rest of waveinfo structure */

        wv->specnum = i;
        wv->live = 1;
        wv->nord = nord;
        wv->xref = cenpix;
        wv->ngood = ngood;
        wv->medresid = medresid;
        wv->fit_rms = rmsfit;
        xx = tcrv;
        dval = cpl_polynomial_eval_1d(wv_ref[iref-1].coefs,
                                      xx-wv_ref[iref-1].xref +
                                      tr[i-1].minyst - ref_minyst,
                                      NULL);
	wv->wave1 = dval + cpl_polynomial_eval_1d(wv->coefs,xx-cenpix,NULL);
	xx = tcrv + specbin * (naxis[0] - 1);
        dval = cpl_polynomial_eval_1d(wv_ref[iref-1].coefs,
                                      xx-wv_ref[iref-1].xref +
                                      tr[i-1].minyst - ref_minyst,
                                      NULL);
	wv->waven = dval + cpl_polynomial_eval_1d(wv->coefs,xx-cenpix,NULL);
	wv->dwave = (wv->waven - wv->wave1)/(double)((naxis[0]-1)*specbin);
        wv->nlines = nfeat;
        wv->xpos = cpl_malloc(nfeat*sizeof(double));
        wv->fwhm = cpl_malloc(nfeat*sizeof(float));
        wv->wave_calc = cpl_malloc(nfeat*sizeof(double));
        wv->wave_true = cpl_malloc(nfeat*sizeof(double));
        wv->fit_flag = cpl_malloc(nfeat*sizeof(unsigned char));
        wv->wave_cor = cpl_calloc(nfeat,sizeof(double));
        wv->peak = cpl_malloc(nfeat*sizeof(float));
        wv->contrast = cpl_malloc(nfeat*sizeof(float));

	for (k = 0; k < nfeat; k++) {
	    wv->xpos[k] = xfeatstr[k].xfeat;
            if(xfeatstr[k].fwhm > 0) {
                wv->fwhm[k] = qmost_max(wv->dwave,xfeatstr[k].fwhm);
            }
            else {
                wv->fwhm[k] = -1.0;
            }
	    wv->wave_calc[k] = xfeatstr[k].wavecalc;
	    wv->wave_true[k] = xfeatstr[k].wavematched;
            wv->fit_flag[k] = xfeatstr[k].bad;
            wv->wave_cor[k] = xfeatstr[k].wave_resid;
            wv->peak[k] = xfeatstr[k].tmax;
            wv->contrast[k] = xfeatstr[k].contrast;
        }

	cpl_free(xfeatstr);
        xfeatstr = NULL;
    }

    if(nnofeat == naxis[1]) {
        cpl_msg_warning(cpl_func,
                        "no arc features found above threshold %.1f",
                        detthr);
    }
    
    /* Find maximum number of lines for sizing output table */
    nlinesmax = 1;

    for(i = 1; i <= naxis[1]; i++) {
        wv = wv_out + (i-1);

        if(wv->nlines > nlinesmax)
            nlinesmax = wv->nlines;
    }

    /* Write output */
    if(qmost_wvcreate(out_wave_tbl, nord, nlinesmax) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output "
                                     "waveinfo table with order %d "
                                     "max lines lines %d",
                                     nord, nlinesmax);
    }

    for(i = 1; i <= naxis[1]; i++) {
        wv = wv_out + (i-1);

        if(qmost_wvwrite1(*out_wave_tbl, i, *wv) != CPL_ERROR_NONE) {
            TIDY;

            if(*out_wave_tbl != NULL) {
                cpl_table_delete(*out_wave_tbl);
                *out_wave_tbl = NULL;
            }

            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not write wavelength "
                                         "solution for fibre %d to "
                                         "waveinfo table",
                                         i);
        }
    }

    /* Right, close things up */
    TIDY;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Measure Fabry-Perot line wavelengths and make line list.
 *
 * Emission line features are identified in the provided Fabry-Perot
 * Etalon (FPE) spectra, and assigned wavelengths based on a master
 * wavelength solution derived from ThAr arc lamp spectra.  This
 * results in a set of measured wavelengths for the FPE lines for each
 * of the 10 simultaneous calibration fibres illuminated by the FPE.
 * The results for each fibre are cross-matched, combined, filtered to
 * remove any outliers or spurious detections, and averaged to produce
 * a final set of line wavelengths, which are emitted as a line list
 * table for use by qmost_arc_analyse.
 *
 * @param   in_spec          (Given)    The extracted arc spectra
 *                                      to analyse, as an image.  The
 *                                      data type must be
 *                                      CPL_TYPE_FLOAT.
 * @param   in_hdr           (Given)    The FITS header of the input
 *                                      spectra with binning
 *                                      information.
 * @param   ref_wave_tbl     (Given)    The master wavelength solution
 *                                      table.
 * @param   ref_wave_hdr     (Given)    The FITS header for the
 *                                      master wavelength solution
 *                                      table.
 * @param   detthr           (Given)    The detection threshold for
 *                                      finding emission features in
 *                                      units of the background
 *                                      sigma.
 * @param   matchwindow      (Given)    Maximum deviation between the
 *                                      measured positions of an arc
 *                                      line between fibres for it to
 *                                      be considered the same line,
 *                                      in wavelength units
 *                                      (Angstroms) at 6000A.  At
 *                                      other wavelengths, it is
 *                                      scaled with wavelength such
 *                                      that the matching window is
 *                                      constant in velocity, or
 *                                      equivalently resolving power,
 *                                      to better match the behaviour
 *                                      of the resolution of the
 *                                      spectrograph itself.
 * @param   out_linelist_tbl (Returned) The resulting line list
 *                                      table.  This table is
 *                                      described in the Reduction
 *                                      Pipeline Description
 *                                      document, Section 6.11.
 * @param   qclist           (Modified) A caller-allocated property
 *                                      list to receive QC headers.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the reference wavelength
 *                                        table dimensions don't match
 *                                        the spectrum.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the input image data type
 *                                        was not float.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS WVCRV</b>
 *   - <b>WVCRV</b>
 *
 * @par Output QC Parameters:
 *   - <b>FPE NFIB MAX</b>: The maximum number of fibres an FPE line
 *     was detected in.
 *   - <b>FPE NFIB MEAN</b>: The mean number of fibres each FPE line
 *     was detected in.
 *   - <b>FPE RMS MAX</b> (A): The RMS scatter of the wavelength for
 *     the FPE line with the highest RMS scatter.
 *   - <b>FPE RMS MEAN</b> (A): The mean RMS scatter of the wavelength
 *     in an FPE line.
 *   - <b>FPE RMS MIN</b> (A): The RMS scatter of the wavelength for
 *     the FPE line with the smallest RMS scatter.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fpmeasure(
    cpl_image *in_spec,
    cpl_propertylist *in_hdr,
    cpl_table *ref_wave_tbl,
    cpl_propertylist *ref_wave_hdr,
    float detthr,
    float matchwindow,
    cpl_table **out_linelist_tbl,
    cpl_propertylist *qclist)
{
    long naxis[2];
    int specbin,spatbin,isbinned;
    float tcrv;
    float *in_spec_buf = NULL;
    float *indata = NULL;

    int i,nfeat,fiblive,k,koth;
    double wave;
    int minyst = 1;

    double *linewave = NULL, *thislinewave;
    int *linen = NULL;
    int *linematch = NULL;

    double *featwave = NULL;
    int *featmatch = NULL;
    double *featbest = NULL;
    
    int nlive, iline, ibest, nlines, nmax;
    double zwin, dbest, delta, med, sig, var, rms;
    int iwave, nwave;

    cpl_errorstate prestate;

    double rmssum, rmsmin, rmsmax;
    int nsum, rmsminline, rmsmaxline;

    /* Garbage collected stuff */

    int nwv_ref = 0;
    qmost_waveinfo *wv_ref = NULL;
    int ref_minyst = 1;

    featstruct *xfeatstr = NULL;
    cpl_propertylist *reflist = NULL;

    /* Check for NULL arguments, particularly the output so we can
       initialize it for garbage collection. */
    cpl_ensure_code(in_spec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_wave_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_wave_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_linelist_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialize this for garbage collection */
    *out_linelist_tbl = NULL;

#undef TIDY
#define TIDY                                    \
    if(wv_ref != NULL) {                        \
        qmost_wvclose(nwv_ref, &wv_ref);        \
        wv_ref = NULL;                          \
        nwv_ref = 0;                            \
    }                                           \
    if(xfeatstr != NULL) {                      \
        cpl_free(xfeatstr);                     \
        xfeatstr = NULL;                        \
    }                                           \
    if(linewave != NULL) {                      \
        cpl_free(linewave);                     \
        linewave = NULL;                        \
    }                                           \
    if(linen != NULL) {                         \
        cpl_free(linen);                        \
        linen = NULL;                           \
    }                                           \
    if(linematch != NULL) {                     \
        cpl_free(linematch);                    \
        linematch = NULL;                       \
    }                                           \
    if(featwave != NULL) {                      \
        cpl_free(featwave);                     \
        featwave = NULL;                        \
    }                                           \
    if(featmatch != NULL) {                     \
        cpl_free(featmatch);                    \
        featmatch = NULL;                       \
    }                                           \
    if(featbest != NULL) {                      \
        cpl_free(featbest);                     \
        featbest = NULL;                        \
    }                                           \
    if(*out_linelist_tbl != NULL) {             \
        cpl_table_delete(*out_linelist_tbl);    \
        *out_linelist_tbl = NULL;               \
    }                                           \
    if(reflist) {                               \
        cpl_propertylist_delete(reflist);       \
        reflist = NULL;                         \
    }

    /* Get input 2D spectrum */
    naxis[0] = cpl_image_get_size_x(in_spec);
    naxis[1] = cpl_image_get_size_y(in_spec);

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    tcrv = 1.0;
    if(isbinned) {
        if(cpl_propertylist_has(in_hdr, "ESO DRS WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_hdr,
                                                "ESO DRS WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read ESO DRS WVCRV "
                                             "from input FITS header");
            }
        }
        else if(cpl_propertylist_has(in_hdr, "WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_hdr,
                                                "WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read WVCRV "
                                             "from input FITS header");
            }
        }
    }

    /* Get pointer to image data */
    in_spec_buf = cpl_image_get_data_float(in_spec);
    if(in_spec_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Start y-pos for extraction */
    if(qmost_pfits_get_minyst(in_hdr,
                              &minyst) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read minimum yst");
    }

    /* Read the master wavelength solution */
    if(qmost_wvopen(ref_wave_tbl,&nwv_ref,&wv_ref) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "problem reading reference "
                                     "wavelength solution table");
    }

    if(nwv_ref != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "wave table nwave=%d doesn't "
                                     "match nspec=%ld",
                                     nwv_ref, naxis[1]);
    }

    /* Start y-pos defaults to same if not given in wave file */
    ref_minyst = minyst;

    if(qmost_pfits_get_minyst(ref_wave_hdr,
                              &ref_minyst) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read minimum yst");
    }

    /* How many fibres are live? */
    nlive = 0;

    for (i = 1; i <= naxis[1]; i++) {
	/* Is this fibre live? */
        fiblive = wv_ref[i-1].live;
	if (! fiblive) {
	    continue;
	}

        nlive++;
    }

    /* Convert match window to redshift */
    zwin = matchwindow / REFWAVE;

    /* Initialize line list */
    linewave = NULL;
    linen = NULL;
    linematch = NULL;
    nlines = 0;

    /* Loop for each spectrum in the map */
    for (i = 1; i <= naxis[1]; i++) {
	/* Is this fibre live? */
        fiblive = wv_ref[i-1].live;
	if (! fiblive) {
	    continue;
	}

        /* ...and is there a valid wavelength solution? */
        if(wv_ref[i-1].ngood <= wv_ref[i-1].nord) {
            continue;
        }
        
	/* Get pointer to the relevant subset of the image */
        indata = in_spec_buf + (i-1) * naxis[0];

	/* Do the detection */
	detect_lines(indata,naxis[0],detthr,&nfeat,&xfeatstr);

        if(nfeat == 0) {
            continue;
        }

        /* If this is the first time, initialize lists */
        if(linewave == NULL) {
            linewave = cpl_malloc(nfeat * nlive * sizeof(double));
            linen = cpl_malloc(nfeat * sizeof(int));
            linematch = cpl_malloc(nfeat * sizeof(int));

            for(k = 0; k < nfeat; k++) {
                wave = cpl_polynomial_eval_1d(
                    wv_ref[i-1].coefs,
                    tcrv + specbin * (xfeatstr[k].xfeat - 1) -
                    wv_ref[i-1].xref +
                    minyst - ref_minyst,
                    NULL);
                linewave[k * nlive] = wave;
                linen[k] = 1;
            }

            nlines = nfeat;
        }
        else {
            /* Otherwise, we need to update the list by matching what
             * we got for this fibre to the existing list. */
            featwave = cpl_malloc(nfeat * sizeof(double));
            featmatch = cpl_malloc(nfeat * sizeof(int));
            featbest = cpl_malloc(nfeat * sizeof(double));

            for(iline = 0; iline < nlines; iline++) {
                linematch[iline] = -1;
            }
            
            for(k = 0; k < nfeat; k++) {
                wave = cpl_polynomial_eval_1d(
                    wv_ref[i-1].coefs,
                    tcrv + specbin * (xfeatstr[k].xfeat - 1) -
                    wv_ref[i-1].xref +
                    minyst - ref_minyst,
                    NULL);

                ibest = -1;
                dbest = -1;
                for(iline = 0; iline < nlines; iline++) {
                    delta = fabs(wave - linewave[iline * nlive]);
                    if(delta > zwin * wave) {
                        continue;
                    }

                    if(ibest < 0 || delta < dbest) {
                        ibest = iline;
                        dbest = delta;
                    }
                }
             
                if(ibest >= 0) {
                    /* Check if another feature already matched to
                     * this entry in the line list */
                    koth = linematch[ibest];
                    if(koth >= 0) {  /* yes */
                        if(dbest < featbest[koth]) {
                            /* It's better, unflag the other one */
                            featmatch[koth] = -1;
                            linematch[ibest] = k;
                        }
                        else {
                            /* It's not better, unflag the present one */
                            ibest = -1;
                        }
                    }
                    else {
                        linematch[ibest] = k;
                    }
                }

                featwave[k] = wave;
                featmatch[k] = ibest;
                featbest[k] = dbest;
            }

            for(k = 0; k < nfeat; k++) {
                ibest = featmatch[k];
                
                if(ibest >= 0) {
                    /* If we matched, accumulate in list */
                    linewave[ibest * nlive + linen[ibest]] = featwave[k];
                    linen[ibest]++;
                }
                else {
                    /* If we didn't match, create a new entry */
                    linewave = cpl_realloc(linewave,
                                           (nlines+1)*nlive*sizeof(double));
                    linen = cpl_realloc(linen,
                                        (nlines+1)*sizeof(int));
                    linematch = cpl_realloc(linematch,
                                            (nlines+1)*sizeof(int));
                    
                    linewave[nlines * nlive] = featwave[k];
                    linen[nlines] = 1;
                    linematch[nlines] = -1;
                    nlines++;
                }
            }

            cpl_free(featwave);
            featwave = NULL;

            cpl_free(featmatch);
            featmatch = NULL;

            cpl_free(featbest);
            featbest = NULL;
        }

	cpl_free(xfeatstr);
        xfeatstr = NULL;
    }

    /* Decide how many lines to emit.  False detections are filtered
     * by requiring each line we emit to be detected in at least half
     * of the fibres. */
    nmax = 0;

    for(iline = 0; iline < nlines; iline++) {
        nmax = qmost_max(nmax, linen[iline]);
    }

    nfeat = 0;
    for(iline = 0; iline < nlines; iline++) {
        if(linen[iline] >= nmax/2) {
            nfeat++;
        }
    }

    /* Create output */
    *out_linelist_tbl = cpl_table_new(nfeat);
    if(*out_linelist_tbl == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create linelist");
    }
    
    if(cpl_table_new_column(*out_linelist_tbl,
                            "Wavelength",
                            CPL_TYPE_DOUBLE) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create Wavelength column");
    }
    
    if(cpl_table_set_column_unit(*out_linelist_tbl,
                                 "Wavelength",
                                 "angstrom") != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set Wavelength units");
    }

    if(cpl_table_new_column(*out_linelist_tbl,
                            "RMS",
                            CPL_TYPE_DOUBLE) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create RMS column");
    }
    
    if(cpl_table_set_column_unit(*out_linelist_tbl,
                                 "RMS",
                                 "angstrom") != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set RMS units");
    }

    if(cpl_table_new_column(*out_linelist_tbl,
                            "Nfib",
                            CPL_TYPE_INT) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create Nfib column");
    }
    
    /* Calculate wavelengths and emit */
    k = 0;

    nsum = 0;

    rmssum = 0;
    rmsmin = 0;
    rmsminline = -1;
    rmsmax = 0;
    rmsmaxline = -1;

    for(iline = 0; iline < nlines; iline++) {
        thislinewave = linewave + iline * nlive;

        /* Skip if not detected on at least half of the fibres */
        if(linen[iline] < nmax/2) {
            continue;
        }

        /* Compute robust mean and rms wavelength */
        prestate = cpl_errorstate_get();
        
        if(qmost_dmedmad(thislinewave,
                         NULL,
                         linen[iline],
                         &med, &sig) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            
            med = 0;
            sig = DBL_MAX;
        }
        else {
            sig *= CPL_MATH_STD_MAD;
        }

        wave = 0;
        nwave = 0;
        for(iwave = 0; iwave < linen[iline]; iwave++) {
            if(fabs(thislinewave[iwave] - med) <= 5*sig) {
                wave += thislinewave[iwave];
                nwave++;
            }
        }

        /* Skip if there's nothing left after clipping.  This
         * shouldn't be possible but trap it anyway. */
        if(nwave <= 0) {
            continue;
        }

        wave /= nwave;
        
        var = 0;
        for(iwave = 0; iwave < linen[iline]; iwave++) {
            if(fabs(thislinewave[iwave] - med) <= 5*sig) {
                delta = thislinewave[iwave] - wave;
                var += delta*delta;
            }
        }
        
        var /= nwave;
        
        rms = sqrt(qmost_max(0, var));
        
        /* Record results */
        if(cpl_table_set(*out_linelist_tbl,
                         "Wavelength",
                         k,
                         wave) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set Wavelength "
                                         "for row %d", k);
        }
        
        if(cpl_table_set(*out_linelist_tbl,
                         "RMS",
                         k,
                         rms) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set RMS "
                                         "for row %d", k);
        }
        
        if(cpl_table_set(*out_linelist_tbl,
                         "Nfib",
                         k,
                         nwave) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set Nfib "
                                         "for row %d", k);
        }
        
        nsum += linen[iline];
        
        rmssum += rms;
        
        if(rmsminline < 0 || rms < rmsmin) {
            rmsmin = rms;
            rmsminline = iline;
        }
        
        if(rmsmaxline < 0 || rms > rmsmax) {
            rmsmax = rms;
            rmsmaxline = iline;
        }
        
        k++;
    }

    cpl_propertylist_update_double(qclist, "ESO QC FPE NFIB MEAN",
                                   ((double) nsum) / qmost_max(1, nfeat));
    cpl_propertylist_set_comment(qclist, "ESO QC FPE NFIB MEAN",
                                 "Mean number of fibres each FPE line "
                                 "was detected in");

    cpl_propertylist_update_int(qclist, "ESO QC FPE NFIB MAX", nmax);
    cpl_propertylist_set_comment(qclist, "ESO QC FPE NFIB MAX",
                                 "Maximum number of fibres an FPE line "
                                 "was detected in");

    cpl_propertylist_update_double(qclist, "ESO QC FPE RMS MEAN",
                                   rmssum / qmost_max(1, nfeat));
    cpl_propertylist_set_comment(qclist, "ESO QC FPE RMS MEAN",
                                 "[angstrom] Mean FPE line RMS");

    cpl_propertylist_update_double(qclist, "ESO QC FPE RMS MIN",
                                   rmsmin);
    cpl_propertylist_set_comment(qclist, "ESO QC FPE RMS MIN",
                                 "[angstrom] Minimum FPE line RMS");

    cpl_propertylist_update_double(qclist, "ESO QC FPE RMS MAX",
                                   rmsmax);
    cpl_propertylist_set_comment(qclist, "ESO QC FPE RMS MAX",
                                 "[angstrom] Maximum FPE line RMS");

    cpl_free(linewave);
    linewave = NULL;

    cpl_free(linen);
    linen = NULL;

    cpl_free(linematch);
    linematch = NULL;

    qmost_wvclose(nwv_ref, &wv_ref);
    wv_ref = NULL;
    nwv_ref = 0;

    /* Sort table */
    reflist = cpl_propertylist_new();
    cpl_propertylist_update_bool(reflist, "Wavelength", 0);

    if(cpl_table_sort(*out_linelist_tbl, reflist) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not sort table");
    }

    cpl_propertylist_delete(reflist);
    reflist = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate correction for FPE pressure variations.
 *
 * The effective gap spacing (and thus, frequency spacing, and
 * ultimately the line wavelengths) of a Fabry-Perot etalon (FPE)
 * depends on the refractive index of the material between the
 * plates.
 *
 * The 4MOST FPE is designed to be operated in vacuo to prevent
 * variations of atmospheric pressure and temperature manifesting as
 * variations in the wavelengths.  Due to vacuum leaks, this is not
 * achieved in practice and a correction is needed based on the
 * difference in pressure measured in the etalon vacuum chamber
 * between when the line wavelengths are measured and when they are
 * applied.  This routine computes the correction based on the
 * pressure given in the FITS headers.
 *
 * If the pressure isn't available in one of the input files, the
 * correction is returned as zero.
 *
 * @param   in_pri_hdr       (Given)    The FITS primary header of the
 *                                      FPE spectrum to be calibrated
 *                                      containing the pressure
 *                                      information.
 * @param   linelist_pri_hdr (Given)    The FITS primary header of the
 *                                      FPE line list containing the
 *                                      pressure information.
 * @param   prescoef         (Given)    The pressure correction
 *                                      coefficient, in units of
 *                                      km/s/mbar.
 * @param   prescorr         (Modified) The correction to be applied,
 *                                      in km/s.  The FPE line list
 *                                      wavelengths should be
 *                                      redshifted by this amount to
 *                                      apply the correction.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If an input FITS header
 *                                        keyword value had an
 *                                        incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO INS PRES16 VAL</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fpcorr(
    cpl_propertylist *in_pri_hdr,
    cpl_propertylist *linelist_pri_hdr,
    float prescoef,
    float *prescorr)
{
    float presin, presout, dp;

    cpl_ensure_code(in_pri_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(linelist_pri_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prescorr != NULL, CPL_ERROR_NULL_INPUT);

#define QMOST_FPCORR_PRESSURE_KEY "ESO INS PRES16 VAL"

    if(cpl_propertylist_has(in_pri_hdr, QMOST_FPCORR_PRESSURE_KEY) &&
       cpl_propertylist_has(linelist_pri_hdr, QMOST_FPCORR_PRESSURE_KEY)) {
        /* Read pressure */
        if(qmost_cpl_propertylist_get_float(in_pri_hdr,
                                            QMOST_FPCORR_PRESSURE_KEY,
                                            &presout) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't read %s from "
                                         "input wavecal",
                                         QMOST_FPCORR_PRESSURE_KEY);
        }

        if(qmost_cpl_propertylist_get_float(linelist_pri_hdr,
                                            QMOST_FPCORR_PRESSURE_KEY,
                                            &presin) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't read %s from "
                                         "input linelist",
                                         QMOST_FPCORR_PRESSURE_KEY);
        }

        /* Compute correction */
        dp = presout - presin;
        *prescorr = prescoef * dp;
    }
    else {
        *prescorr = 0;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Evaluate initial wavelength using reference surface map.
 *
 * @param   refdata    (Given)    The reference wavelength surface map
 *                                (2D image of wavelength as a
 *                                function of position on the
 *                                detector).
 * @param   naxisr     (Given)    The dimensions of the surface map.
 * @param   xpos       (Given)    The x position to evaluate the
 *                                wavelength for, numbering from 1.
 * @param   ypos       (Given)    The y position to evaluate the
 *                                wavelength for, numbering from 1..
 * @param   waveguess  (Returned) The wavelength at the given
 *                                position.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the position specified by
 *                                        xpos, ypos is outside the
 *                                        bounds of the reference map.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eval_wave_map (
    float *refdata,
    long naxisr[2],
    float xpos,
    float ypos,
    float *waveguess)
{
    float wx1,wx2,wy1,wy2;
    int ix1,ix2,iy1,iy2,ind1,ind2,ind3,ind4;

    cpl_ensure_code(refdata != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(waveguess != NULL, CPL_ERROR_NULL_INPUT);

    /* X, Y coordinates of pixels for interpolation, clamped to range.
     * If the requested pixel is out of range, the closest value in
     * range is used. */
    ix1 = (int)xpos;
    ix2 = ix1 + 1;

    if(ix1 < 1) {
        xpos = 1.0;
        ix1 = 1;
        ix2 = 1;
    }
    if(ix2 > naxisr[0]) {
        xpos = naxisr[0];
        ix1 = naxisr[0];
        ix2 = naxisr[0];
    }

    iy1 = (int)ypos;
    iy2 = iy1 + 1;

    if(iy1 < 1) {
        ypos = 1.0;
        iy1 = 1;
        iy2 = 1;
    }
    if(iy2 > naxisr[1]) {
        ypos = naxisr[1];
        iy1 = naxisr[1];
        iy2 = naxisr[1];
    }

    wx2 = xpos - (float)ix1;
    wx1 = 1.0 - wx2;
    wy2 = ypos - (float)iy1;
    wy1 = 1.0 - wy2;
    ind1 = (iy1 - 1)*naxisr[0] + ix1 - 1;
    ind2 = (iy1 - 1)*naxisr[0] + ix2 - 1;
    ind3 = (iy2 - 1)*naxisr[0] + ix1 - 1;
    ind4 = (iy2 - 1)*naxisr[0] + ix2 - 1;
    *waveguess = wx1*wy1*refdata[ind1] + wx2*wy1*refdata[ind2] +
        wx1*wy2*refdata[ind3] + wx2*wy2*refdata[ind4];

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do a fit of wavelength vs position for a spectrum.
 *
 * Given the wavelength and position information for the emission
 * features in a spectrum, do a polynomial fit to determine a
 * wavelength solution.
 *
 * @param   nfeat      (Given)    The number of emission features in
 *                                the given spectrum.
 * @param   xfeat      (Given)    The featstruct data structure for
 *                                the given spectrum.
 * @param   nord       (Given)    The order of the polynomial to be
 *                                fit.
 * @param   rejthr     (Given)    The rejection threshold in units of
 *                                sigma.
 * @param   ngood      (Returned) The number of features used in the
 *                                final fit.
 * @param   rms        (Returned) The RMS of the fit in wavelength
 *                                units (nominally Angstroms).
 * @param   fitx       (Given)    If == 1 then fit to wavelength. If
 *                                == 2 then fit to wavelength
 *                                correction.
 *
 * @return  cpl_polynomial        The resulting polynomial fit, or
 *                                NULL if fitting failed.  Error
 *                                CPL_ERROR_DATA_NOT_FOUND can be
 *                                triggered if the fit was
 *                                underconstrained due to there not
 *                                being enough features in the input
 *                                array.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_polynomial *dofit(
    int nfeat,
    featstruct *xfeat,
    int nord,
    float rejthr,
    int *ngood,
    double *rms,
    int fitx)
{
    double *xcopy = NULL;
    double *ycopy = NULL;
    float *rcopy = NULL;
    cpl_polynomial *poly = NULL;
    cpl_errorstate prestate;

    int iter,niter,n,i,nrej;
    float madval;
    double resid;

    /* Initialise some values */

    *ngood = 0;
    *rms = 0.0;

#undef TIDY
#define TIDY                                    \
    if(xcopy) {                                 \
        cpl_free(xcopy);                        \
        xcopy = NULL;                           \
    }                                           \
    if(ycopy) {                                 \
        cpl_free(ycopy);                        \
        ycopy = NULL;                           \
    }                                           \
    if(rcopy) {                                 \
        cpl_free(rcopy);                        \
        rcopy = NULL;                           \
    }                                           \
    if(poly) {                                  \
        cpl_polynomial_delete(poly);            \
        poly = NULL;                            \
    }

    /* Get some memory */

    xcopy = cpl_malloc(nfeat*sizeof(double));
    ycopy = cpl_malloc(nfeat*sizeof(double));
    rcopy = cpl_malloc(nfeat*sizeof(float));

    n = 0;
    for (i = 0; i < nfeat; i++) {
	if (xfeat[i].bad)
	    continue;
	n++;
    }
    niter = 3;

    /* Do niter iterations, unless there is nothing to be clipped. Start
       by copying the good points over */

    for (iter = 1; iter <= niter; iter++) {
	n = 0;
        /* if (verbose)  */
        /* fprintf(stderr,"iter: %d %d\n",iter,nfeat); */
	for (i = 0; i < nfeat; i++) {
	    if (xfeat[i].bad)
 		continue;
	    xcopy[n] = xfeat[i].xfeat_norm;
            if (fitx == 1) 
                ycopy[n] = xfeat[i].wavematched;
            else
                ycopy[n] = xfeat[i].wave_resid;
            /* if (verbose) */
            /*     fprintf(stderr,"%d %0.5f %0.5f %0.5f\n",i,xfeat[i].xfeat, */
            /*             xfeat[i].xfeat_norm,xfeat[i].wavematched); */
	    n++;
	}

	/* (re-)Do the fit */
        if(poly != NULL) {
            cpl_polynomial_delete(poly);
            poly = NULL;
        }

        prestate = cpl_errorstate_get();

	poly = qmost_polynm(xcopy,ycopy,n,nord+1,0);
        if(poly == NULL) {
            switch(cpl_error_get_code()) {
            case CPL_ERROR_DATA_NOT_FOUND:
                cpl_errorstate_set(prestate);
                
                cpl_msg_warning(cpl_func,
                                "not enough valid points for degree %d "
                                "polynomial fit: %d",
                                nord, n);
                
                poly = cpl_polynomial_new(1);
                
                break;
            case CPL_ERROR_SINGULAR_MATRIX:
                cpl_errorstate_set(prestate);
                
                cpl_msg_warning(cpl_func,
                                "singular matrix in degree %d polynomial "
                                "fit to %d data points",
                                nord, n);
                
                poly = cpl_polynomial_new(1);
                
                break;
            default:
                TIDY;
                cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                      "could not fit degree %d polynomial "
                                      "to %d data points", nord, n);
                return NULL;
            }
        }
	
	/* Evaluate the MAD rms of the fit residuals */

	n = 0;
	for (i = 0; i < nfeat; i++) {
	    if (xfeat[i].bad)
		continue;
            if (fitx == 1) 
                resid = xfeat[i].wavematched - 
                    cpl_polynomial_eval_1d(poly,xfeat[i].xfeat_norm,NULL);
            else
                resid = xfeat[i].wave_resid -
                    cpl_polynomial_eval_1d(poly,xfeat[i].xfeat_norm,NULL);
	    rcopy[n] = fabs(resid);
	    n++;
	}
        if(n > 0) {
            if (qmost_med(rcopy,NULL,n,&madval) != CPL_ERROR_NONE) {
                TIDY;
                cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                      "could not get median of %d values",
                                      n);
                return NULL;
            }
        }
        else {
            madval = 0;
        }
	*rms = 1.48*madval;
	*ngood = n;

	/* If this is not the last iteration then flag bad points */

        if (iter != niter) { 
	    nrej = 0; 
            for (i = 0; i < nfeat; i++) {
	 	if (xfeat[i].bad) 
	 	    continue; 
                if (fitx == 1) 
                    resid = fabs(xfeat[i].wavematched - 
                             cpl_polynomial_eval_1d(poly,xfeat[i].xfeat_norm,
                                                    NULL));
                else
                    resid = fabs(xfeat[i].wave_resid - 
                             cpl_polynomial_eval_1d(poly,xfeat[i].xfeat_norm,
                                                    NULL));
	 	if (resid > rejthr*(*rms)) { 
       	            xfeat[i].bad = 2; 
        	    nrej++; 
	 	} 
	     } 
	     if (nrej == 0) 
	 	break; 
	} 
    }
	
    /* Tidy and exit */

    cpl_free(xcopy);
    xcopy = NULL;

    cpl_free(ycopy);
    ycopy = NULL;

    cpl_free(rcopy);
    rcopy = NULL;

    return poly;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Detect emission features in a spectrum.
 *
 * @param   indata     (Given)    The data array with the input
 *                                spectrum.
 * @param   nx         (Given)    The number of pixels in the input
 *                                spectrum.
 * @param   detthr     (Given)    The detection threshold for the
 *                                emission pixels in units of
 *                                background sigma.
 * @param   nfeat      (Returned) The number of features detected in
 *                                the spectrum.
 * @param   xfeat      (Returned) The featstruct data structures for
 *                                all of the features found in the
 *                                current spectrum.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void detect_lines(
    float *indata,
    long nx,
    float detthr,
    int *nfeat,
    featstruct **xfeat)
{
    float *smoothed;

    /* Smooth the profile gently */

    smoothed = cpl_malloc(nx*sizeof(float));
    convolve(nx,indata,2.35,smoothed);          /* 2.0 is close to Hanning */

    /* Find the peaks */
    
    findpeaks(nx,indata,smoothed,detthr,nfeat,xfeat);

    cpl_free(smoothed);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Identify peaks in a spectrum.
 *
 * Searches the input spectrum for peaks, defining a peak as a local
 * maximum in the smoothed data where the signal:to:noise in the
 * integrated peak flux exceeds the threshold value.
 *
 * @param   nx         (Given)    The number of pixels in the input
 *                                spectrum.
 * @param   indata     (Given)    The data array with the input
 *                                spectrum.
 * @param   smoothed   (Given)    The smoothed input spectrum.
 * @param   thr        (Given)    The detection threshold for the
 *                                emission peaks in units of 
 *                                signal:to:noise.
 * @param   nfeat      (Returned) The number of features detected in
 *                                the spectrum.
 * @param   xfeat      (Returned) The featstruct data structures for
 *                                all of the features found in the
 *                                current spectrum.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void findpeaks(
    int nx,
    float *indata,
    float *smoothed,
    float thr,
    int *nfeat,
    featstruct **xfeat)
{
    float sum,dist,loc1,loc2,xbar,pkflux,fwhm,sumw,sumt,cont,s2n;
    int i,j,jl,jh,jmax,maxi,npeaks,window,win2;
    int found1,found2;
    float b, c, halfmax;

    double xl, xh, wt, sxw, sw;
    int ixl, ixh, ix;
    
    /* Initialise a few things */

    *nfeat = 0;
    *xfeat = NULL;

    /* Get enough memory for all pixels to have a feature */

    *xfeat = cpl_malloc(nx*sizeof(featstruct));
    
    /* Now search for the peaks */

    npeaks = 0;
    window = 11;
    win2 = window/2;
    for (i = win2; i < nx-win2; i++) {
        /* Define a peak as a local maximum in the smoothed data */
	if (smoothed[i] >= smoothed[i-1] && smoothed[i] >= smoothed[i+1] &&
          smoothed[i-1] > smoothed[i-2] && smoothed[i+1] > smoothed[i+2]) {
	  jl = i-2;
	  while (smoothed[jl] < smoothed[jl+1] && jl > 0) {
	    jl = jl-1;
	  }	    
	  jl = jl+1;                                        /* back off */
          jh = i+2;
	  while (smoothed[jh] < smoothed[jh-1] && jh < nx-1) {
	    jh = jh+1;
	  }
	  jh = jh-1;                         
	  if (jl >= win2 && jh < nx-win2) {                 /* ignore edges */
            sum = 0.0;
            sumw = 0.0;
	    sumt = 0.0;
            xbar = (float)(i+1);
	    cont = 0.5*(smoothed[jl]+smoothed[jh]);
	    pkflux = indata[jl]-cont;
            jmax = jl;
            for (j = jl; j <= jh; j++) {
	      sum += qmost_max(0.0,indata[j]-cont)*((float)(j+1)-xbar);
              sumw += qmost_max(0.0,indata[j]-cont);
              sumt += qmost_max(0.0,indata[j]);

              if(indata[j]-cont > pkflux) {
                  pkflux = indata[j]-cont;
                  jmax = j;
              }
            }
            if (sumw <= 0.0 || sumt <= 0.0)
	        continue;
	    s2n = sumw/sqrt(sumt+5.0*2.5*2.5);              /* Poisson + r/o */
            if (s2n < thr)                
	        continue;
            xbar += sum/sumw;
            if (xbar > nx || xbar < 0)
                continue;
            (*xfeat)[npeaks].xfeat = xbar;
            (*xfeat)[npeaks].tmax = pkflux + cont;
            (*xfeat)[npeaks].contrast = pkflux / (pkflux + cont);

            /* Re-measure centre of gravity over fixed +/- FWHM approx.
             * aperture to prevent use of broad wings of FPE lines */
            xl = xbar - 0.5*COGWIDTH - 1;
            xh = xbar + 0.5*COGWIDTH - 1;

            ixl = rint(xl);
            if(ixl < 0)
                ixl = 0;
            if(ixl >= nx)
                ixl = nx-1;

            ixh = rint(xh);
            if(ixh < 0)
                ixh = 0;
            if(ixh >= nx)
                ixh = nx-1;

            /* First partial pixel */
            wt = (indata[ixl] - cont) * (ixl + 0.5 - xl);
            sxw = wt * (ixl + 1 - xbar);
            sw = wt;

            /* Whole pixels */
            for(ix = ixl + 1; ix < ixh; ix++) {
                wt = indata[ix] - cont;
                sxw += wt * (ix + 1 - xbar);
                sw += wt;
            }
            
            /* Last partial pixel */
            wt = (indata[ixh] - cont) * (xh - ixh + 0.5);
            sxw += wt * (ixh + 1 - xbar);
            sw += wt;

            if(sw != 0.0) {
                (*xfeat)[npeaks].xfeat = xbar + sxw / sw;                
            }

            /* Parabolic interpolation to estimate peak height for FWHM */
            if(jmax > jl && jmax < jh) {
                b = indata[jmax-1] - indata[jmax+1];
                c = 0.5 * (indata[jmax-1] + indata[jmax+1]) - indata[jmax];
                halfmax = 0.5 * (pkflux - b * b / (16.0 * c));
            }
            else {
                halfmax = 0.5 * pkflux;
            }

            /* Work out the FWHM of the line */
	    maxi = (int)(xbar+0.5);

            found1 = 0;
            for (j = maxi - 1; j >= jl ; j--) {
                if (indata[j]-cont < halfmax) {
                    found1 = 1;
                    break;
                }
            }

            if(found1 && indata[j+1] != indata[j]) {
                dist = (halfmax - indata[j] + cont)
                    / (indata[j+1] - indata[j]);
            }
            else {
                dist = 0.5;
            }

            loc1 = (float)(j+1) + dist;

            found2 = 0;
            for (j = maxi; j <= jh; j++) {
                if (indata[j]-cont < halfmax) {
                    found2 = 1;
                    break;
                }
            }

            if(found2 && indata[j-1] != indata[j]) {
                dist = (halfmax - indata[j] + cont)
                    / (indata[j-1] - indata[j]);
            }
            else {
                dist = 0.5;
            }

            loc2 = (float)(j+1) - dist;

            if(found1 && found2) {
                fwhm = loc2 - loc1;
            }
            else {
                /* Flag as couldn't determine */
                fwhm = -1.0;
            }

            /* Store away the results */
            
            (*xfeat)[npeaks].fwhm = fwhm;
            (*xfeat)[npeaks].bad = 0;
            (*xfeat)[npeaks].matched_index = -1;
            (*xfeat)[npeaks].wavematched = 0.0;
            (*xfeat)[npeaks].errwavematched = 0.0;
            (*xfeat)[npeaks].wavecalc = 0.0;
            (*xfeat)[npeaks].wave_resid = 0.0;
	    npeaks++;
	  }
	}
    }

    /* Reallocate the resulting array */

    if (npeaks > 0) {
        *xfeat = cpl_realloc(*xfeat,npeaks*sizeof(featstruct));
    } else {
        cpl_free(*xfeat);
        *xfeat = NULL;
    }
    *nfeat = npeaks;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create a weight array for a Gaussian convolution.
 *
 * @param   filtfwhm   (Given)    The FWHM of the Gaussian kernel.
 * @param   weights    (Modified) The weight array. Must be at least
 *                                size NW.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void crweights(
    float filtfwhm,
    float *weights)
{
    int j,nw2,n;
    double gsigsq,dj;
    float renorm;

    /* Get the kernel size */

    nw2 = NW/2;
    
    /* Set the normalisation constants */

    gsigsq = 1.0/(2.0*pow(qmost_max(0.5,(double)filtfwhm)/2.35,2.0));
    renorm = 0.0;

    /* Now work out the weights */

    n = -1;
    for (j = -nw2; j <= nw2; j++) {
	dj = (double)j;
	dj *= gsigsq*dj;
	n++;
	weights[n] = (float)exp(-dj);
	renorm += weights[n];
    }

    /* Now normalise the weights */

    n = -1;
    for (j = -nw2; j <= nw2; j++) {
	n++;
	weights[n] /= renorm;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Smooth a spectrum with a Gaussian convolution.
 *
 * Convolve a spectrum with a gaussian kernel in order to smooth it.
 *
 * @param   nx         (Given)    The number of pixels in the input
 *                                and output spectra.
 * @param   indata     (Given)    The input spectrum.
 * @param   filtfwhm   (Given)    The FWHM of the Gaussian kernel.
 * @param   smoothed   (Modified) The output smoothed spectrum.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void convolve(
    int nx,
    float *indata,
    float filtfwhm,
    float *smoothed)
{
    int i,nw2,ix,jx,n;
    float weights[NW];

    /* If the filter is zero, then just copy the data over */

    if (filtfwhm <= 0.0) {
	memcpy(smoothed,indata,nx*sizeof(float));
	return;
    }
	
    /* Zero the summations */

    for (i = 0; i < nx; i++) 
	smoothed[i] = 0.0;

    /* Now big is the smoothing kernel? */

    nw2 = NW/2;
    crweights(filtfwhm,weights);

    /* Now loop for each column */

    for (ix = nw2; ix < nx-nw2; ix++) {
	n = -1;
	for (jx = ix-nw2; jx <= ix+nw2; jx++) {
	    n++;
	    smoothed[ix] += weights[n]*indata[jx];
	}
    }
    for (ix = 0; ix < nw2; ix++)
	smoothed[ix] = indata[ix];
    for (ix = nx-nw2; ix < nx; ix++)
	smoothed[ix] = indata[ix];
}

/**@}*/

/*

$Log$
Revision 1.22 20210930  mji
need to introduce binning dependent matchwindow and update binning code
put in minimum 1 pixel FWHM for arc lines

Revision 1.21 20201210  mji
added in alternative of extimating wavelengths from reference wave solution

Revision 1.20 20200825  mji
Completely rewrote the peak finder and removed any dependence 
on back_level

Revision 1.19 20191029  mji
Modified the iterative k-sigma dofit clipping and replaced rms 
estimate of sigma with scaled MAD

Revision 1.18  2019/02/25 10:33:39  jrl
New memory allocation scheme. Removed extraneous old routines

Revision 1.17  2018/11/07 13:50:06  jrl
added doarcs_obrigid

Revision 1.16  2018/10/25 06:10:47  jrl
Fixed for case where no lines are detected

Revision 1.15  2018/10/12 10:03:55  jrl
Fixed bug where peak wasn't being estimated properly

Revision 1.14  2018/09/19 11:25:11  jrl
Modify the centring of arc lines to use a weighted first moment. Added
qmost_doarcs_ob.

Revision 1.13  2018/07/15 14:32:23  jim
Better line centring.

Revision 1.12  2017/08/02 08:57:41  jim
Major changes to the way arcs are detected and identified

Revision 1.11  2017/05/22 11:16:46  jim
Fixed _ref routine so that standard wavelengths are assigned to only the best
detected line. Also fixed a bug that crops up when no features are detected.

Revision 1.10  2017/03/14 11:01:03  jim
Now does iterative first guess at shift

Revision 1.9  2017/01/17 08:57:09  jim
Modified how lines are centred and added qc calculations

Revision 1.8  2016/10/26 11:20:02  jim
Modified how the position of the peaks is found

Revision 1.7  2016/10/23 15:55:19  jim
Added some documentation and _ref routine

Revision 1.6  2016/10/03 14:50:59  jim
fiddled with rejection criteria

Revision 1.5  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.4  2016/07/11 14:56:42  jim
Added matchwindow parameter to argument list. Removed trace_xmin and trace_xmax
from consideration as arcs have already been trimmed.

Revision 1.3  2016/07/06 11:02:24  jim
Modified to remove code that we commented out

Revision 1.2  2016/05/20 09:58:57  jim
Uses new version of trace file

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry


*/
