/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_dfs.h"
#include "qmost_filt1d.h"
#include "qmost_sort.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_filt1d    qmost_filt1d
 *
 * Median and mean filtering of a 1d array.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void padext(
    float *x,
    int n,
    float blankval);

static void median(
    float *xbuf,
    int npt,
    int nfilt);

static void linear(
    float *xbuf,
    int npt,
    int nfilt);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do median and mean filtering of a 1d array.
 *
 * @param   ybuf       (Modified) The input spectrum. This will be
 *                                replaced by the smoothed version.
 * @param   mpt        (Given)    The length of the input spectrum.
 * @param   mfilt      (Given)    The length of the smoothing box for
 *                                the median filter. If mfilt == 0
 *                                then no median filtering is done.
 * @param   lfilt      (Given)    The length of the smoothing box for
 *                                the linear filter. If lfilt == 0
 *                                then no linear filtering is done.
 * @param   blankval   (Given)    The flag value to show where the bad
 *                                pixels are in the input spectrum.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_filt1d (
    float *ybuf,
    int mpt,
    int mfilt,
    int lfilt, 
    float blankval)
{
    float *wbuf;
    int i,irc;

    /* Get out of here if no filtering is required...*/

    if (mfilt <= 0 && lfilt <= 0)
        return;
    
    /* Allocate temporary storage */

    wbuf = cpl_malloc(mpt*sizeof(float));

    /* Copy data over so long as it doesn't have the flag value */

    irc = 0;
    for (i = 0; i < mpt; i++){
        if (ybuf[i] != blankval){
            wbuf[irc] = ybuf[i];
            irc++;
        }
    }

    /* If they were all bad, then do nothing more */

    if (irc == 0) {
        cpl_free(wbuf);
        return;
    }

    /* Otherwise apply median filter */

    if (mfilt > 0)
        median(wbuf,irc,mfilt);

    /* Now copy it back */

    irc = 0;
    for (i = 0; i < mpt; i++) {
        if (ybuf[i] != blankval) {
            ybuf[i] = wbuf[irc];
	    irc++;
	}
    }
    padext(ybuf,mpt,blankval);
    if (lfilt > 0)
        linear(ybuf,mpt,lfilt);

    /* Tidy up */
    
    cpl_free(wbuf);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do sky spectrum filtering.
 *
 * Filter a sky spectrum and break it into its continuum and emission
 * line components.
 *
 * @param   nx         (Given)    The number of pixels in the input
 *                                sky spectrum.
 * @param   skyspec    (Given)    The original input sky spectrum.
 * @param   skycont    (Modified) An array to be used to return the
 *                                continuum component.
 * @param   skyline    (Modified) An array to be used to return the
 *                                emission line component.
 * @param   skymask    (Given)    A mask array. If the value at a pixel
 *                                is greater than 40 on input then
 *                                that pixel will be ignored when
 *                                doing stats on the continuum.
 * @param   sigma      (Returned) The sigma of the skyline component.
 * @param   nlin       (Given)    The size of the smoothing box for
 *                                linear filtering.
 * @param   nmed       (Given)    The size of the smoothing box for
 *                                median filtering.
 * @param   niter      (Given)    The number of iterations to be
 *                                performed.
 * @param   cliplow    (Given)    The lower clip level in units of
 *                                sigma.
 * @param   cliphigh   (Given)    The upper clip level in units of
 *                                sigma.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_skyfilt (
    int nx,
    float *skyspec,
    float *skycont, 
    float *skyline,
    float *skymask,
    float *sigma, 
    int nlin,
    int nmed,
    int niter,
    float cliplow,
    float cliphigh)
{
    int iter,irej,irejo,i;
    float *work;
    float blankval;
    cpl_errorstate prestate;

    /* Get a little memory to work out medians */

    blankval = -1.0E6;
    work = cpl_malloc(nx*sizeof(float));

    /* Loop for each iteration */

    irej = 0;
    for (iter = 1; iter <= niter; iter++) {
	irejo = irej;
	irej = 0;

	/* On all iterations but the first one try and mask out sky lines */

	if (iter == 1) {
	    memmove(skycont,skyspec,nx*sizeof(float));
	} else {
	    for (i = 0; i < nx; i++) {
		if (skyline[i] > cliphigh*(*sigma) ||
                    skyline[i] < -cliplow*(*sigma) ||
                    (skymask != NULL && skymask[i] > 40.0)) {
		    skycont[i] = blankval;
		    irej++;
		} else {
		    skycont[i] = skyspec[i];
		}
	    }
	}
        
        /* If all rejected, do nothing */

        if(irej >= nx) {
            memmove(skycont,skyspec,nx*sizeof(float));
            break;
        }

	/* Smooth the continuum */

	qmost_filt1d(skycont,nx,nmed,nlin,blankval);

	/* Now define the skyline array */

	for (i = 0; i < nx; i++) {
	    skyline[i] = skyspec[i] - skycont[i];
	    work[i] = fabs(skyline[i]);
	}

        prestate = cpl_errorstate_get();

	if(qmost_med(work,NULL,nx,sigma) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            *sigma = 0;
        }

	*sigma *= 1.48;
	if (irej == irejo && iter > 1)
	    break;
    }

    /* Finish off by redefining the sky line array */

    for (i = 0; i < nx; i++) 
	skyline[i] = skyspec[i] - skycont[i];
    cpl_free(work);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Remove bad pixels from a 1d array by linear interpolation
 *          and extrapolation.
 *
 * @param   x          (Modified) The input array.
 * @param   n          (Given)    The size of the input array.
 * @param   blankval   (Given)    The flag value to show where the bad
 *                                pixels are in the input array.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/
 
static void padext(
    float *x,
    int n,
    float blankval)
{
    int i,j,ilow,ihih=0,ic;
    float xlow,xhih,slope,t1,t2;

    /* Find the first good value */
    
    i = 0;
    while(x[i] == blankval && i < n-1)
	i++;

    /* Linearly interpolate bad values in the spectrum */
    
    ilow = i;
    for (i = ilow+1; i < n; i++) {
	if (x[i] == blankval) {
	    ic = 1;
	    while (i+ic < n-1 && x[i+ic] == blankval)
		ic++;
	    if (i+ic < n && x[i+ic] != blankval) {
		xlow = x[i-1];
		xhih = x[i+ic];
		for (j = 0; j < ic; j++) {
		    t2 = ((float) j+1)/((float) ic+1);
		    t1 = 1.0 - t2;
		    x[i+j] = t1*xlow+t2*xhih;
		}
	    }
	} else {
	    ihih = i;
	}
    }

    /* Linear extrapolation of ends */

    if (ilow > 0) {
	slope = x[ilow+1] - x[ilow];
        for (i = 0; i < ilow; i++) 
	    x[i] = x[ilow] - slope*(float)(ilow-i);
    }
    if (ihih < n-1) {
	slope = x[ihih] - x[ihih-1];
        for (i = ihih+1; i < n; i++) 
	    x[i] = x[ihih] + slope*(float)(i-ihih);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do median filtering of a 1d array.
 *
 * @param   xbuf       (Modified) The input 1d array.  Replaced by
 *                                smoothed version.
 * @param   npt        (Given)    The size of the input array.
 * @param   nfilt      (Given)    The size of the smoothing box. If
 *                                nfilt == 0, no smoothing will be
 *                                done.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void median (
    float *xbuf,
    int npt,
    int nfilt)
{
    int nfo2p1, nelem;

    float *ybuf = NULL;
    float *array = NULL;
    int *point = NULL;
    int *revptr = NULL;

    int i, il, ilow;
    float xmns, xmnf;
    int j, jl, jh, r;
    float vnew;
    int isp, rmp, im;

    cpl_errorstate prestate;

    /* Make sure the filter number is odd and that there are enough
       data points */

    if (nfilt <= 1)
	return;
    if ((nfilt/2)*2 == nfilt) 
	nfilt++;
    if (npt <= nfilt) 
	return;
    nfo2p1 = nfilt/2;

    /* Get some workspace */

    nelem = npt + nfilt;

    ybuf = cpl_malloc(nelem*sizeof(float));
    array = cpl_malloc(nfilt*sizeof(float));
    point = cpl_malloc(nfilt*sizeof(int));
    revptr = cpl_malloc(nfilt*sizeof(int));

    /* Set first and last edges equal */

    il = nfilt/2;
    ilow = qmost_max(3,nfilt/4);
    ilow = (ilow/2)*2 + 1;
    for (i = 0; i < ilow; i++) 
	array[i] = xbuf[i];

    prestate = cpl_errorstate_get();

    if(qmost_med(array,NULL,ilow,&xmns) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
        xmns = 0;
    }

    for (i = 0; i < ilow; i++) 
	array[i] = xbuf[npt-1-i];

    if(qmost_med(array,NULL,ilow,&xmnf) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
        xmnf = 0;
    }

    /* Reflect edges before filtering */

    for (i = 0; i < il; i++) {
	ybuf[i] = 2.0*xmns - xbuf[il+ilow-1-i];
	ybuf[npt+i+il] = 2.0*xmnf - xbuf[npt-i-ilow-1];
    }
    for (i = 0; i < npt; i++) 
	ybuf[i+il] = xbuf[i];

    /* Load up and sort first kernel width */

    for (i = 0; i < nfilt; i++) {
	array[i] = ybuf[i];
	point[i] = i;
    }

    qmost_sort_fi(array,point,nfilt);

    /* Extract reverse lookup pointers */

    for(i = 0; i < nfilt; i++) {
        revptr[point[i]] = i;
    }

    /* First output pixel */

    xbuf[0] = array[nfo2p1];

    /* Loop over input pixels */

    jl = nfilt;
    jh = nfilt+npt-1;
    for (j = jl; j < jh; j++) {
        /* Index of element we're removing in revptr */
        r = (j-jl) % nfilt;

        /* Fetch the next input pixel */
        vnew = ybuf[j];

        /* Binary search to locate insertion point = isp */
        isp = 0;
        rmp = nfilt;

        while(isp < rmp) {
            im = (isp + rmp) / 2;
            if(array[im] < vnew) {
                isp = im + 1;
            }
            else {
                rmp = im;
            }
        }

        /* Element we're removing */
        rmp = revptr[r];

        /* Adjust to where insertion point will be after we shift
         * array contents for the removal */
        if(isp > rmp) {
            isp--;
        }

        /* Do the removal */
        if(isp < rmp) {
            /* Shift everything up to make room */
            for(i = rmp; i > isp; i--) {
                array[i] = array[i-1];
                point[i] = point[i-1];
                revptr[point[i]] = i;
            }
        }
        else if(isp > rmp) {
            /* Shift everything down to make room */
            for(i = rmp; i < isp; i++) {
                array[i] = array[i+1];
                point[i] = point[i+1];
                revptr[point[i]] = i;
            }
        }

        /* Do the insertion */
        array[isp] = vnew;
        point[isp] = r;
        revptr[r] = isp;

        /* Compute median */
        xbuf[j-jl+1] = array[nfo2p1];
    }

    /* Free temporary arrays */

    cpl_free(revptr);
    cpl_free(point);
    cpl_free(array);
    cpl_free(ybuf);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do mean filtering of a 1d array.
 *
 * @param   xbuf       (Modified) The input 1d array.  Replaced by
 *                                smoothed version.
 * @param   npt        (Given)    The size of the input array.
 * @param   nfilt      (Given)    The size of the smoothing box. If
 *                                nfilt == 0, no smoothing will be
 *                                done.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void linear(
    float *xbuf,
    int npt,
    int nfilt)
{
    int i,il,ilow,nelem;
    float xmns,xmnf;
    float sum,*ybuf,fnfilt;

    /* Make sure you have an odd number of pixels in the filter window */

    if (nfilt <= 1)
	return;
    if (! nfilt % 2) 
	nfilt++;
    if (npt <= nfilt)
	return;

    /* Set first and last edges equal */

    il = nfilt/2;
    ilow = qmost_max(3,nfilt/4);
    ilow = (ilow/2)*2 + 1;
    sum = 0.0;
    for (i = 0; i < ilow; i++)
	sum += xbuf[i];
    xmns = sum/((float)ilow);
    sum = 0.0;
    for (i = 0; i < ilow; i++)
	sum += xbuf[npt-1-i];
    xmnf = sum/(float)ilow;

    /* Allocate ybuf array to the maximum number of elements required */

    nelem = npt + nfilt;

    ybuf = cpl_malloc(nelem*sizeof(float));

    /* Reflect edges before filtering */

    for (i = 0; i < il; i++) {
        ybuf[i] = 2.0*xmns - xbuf[il+ilow-1-i];
        ybuf[npt+i+il] = 2.0*xmnf - xbuf[npt-i-ilow-1];
    } 
    for (i = 0; i < npt; i++)
        ybuf[i+il] = xbuf[i];

    /* Do the filtering now */

    fnfilt = (float)nfilt;
    sum = 0.0;
    for (i = 0; i < nfilt; i++) 
	sum += ybuf[i];
    xbuf[0] = sum/fnfilt;
    for (i = 1; i < npt; i++) {
	sum += (ybuf[i+nfilt-1] - ybuf[i-1]);
	xbuf[i] = sum/fnfilt;
    }
    cpl_free(ybuf);
    return;
}	    

/**@}*/

/*

$Log$
Revision 1.8  20231027  jmi
Fixed linear bug xmns,xmnf should be float not int.

Revision 1.7  20211005  mji
removed min limit on returned sigma as caused 
problems in PCA masking for low sky levels

Revision 1.6  2019/02/25 10:39:08  jrl
New memory allocation scheme

Revision 1.5  2018/10/12 10:06:34  jrl
Modified to allow for asymmetric clipping

Revision 1.4  2018/04/27 15:29:44  jim
Fixed small bug

Revision 1.3  2016/10/23 15:55:38  jim
Added docs

Revision 1.2  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.1  2016/07/14 07:11:18  jim
New file


*/
