/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

#include "qmost_blk.h"
#include "qmost_filt1d.h"
#include "qmost_model_psf.h"
#include "qmost_pfits.h"
#include "qmost_sort.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_model_psf  qmost_model_psf
 *
 * Model PSF extraction from fibre flats.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_model_psf.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void plugholes_par (
    float *data,
    unsigned char *bpm,
    int nx);

static float interp_fwhm (
    float y,
    float *tbl_y,
    float *tbl_fwhm,
    int n);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Model PSFs of the fibres using a fibre flat.
 *
 * An empirical model of the fibre PSF is extracted from a fibre flat
 * by block averaging along the spectral direction in an oversampled
 * pixel grid, with the subsampling of each detector pixel specified
 * by the subsample argument (typically 5).  The resulting averages
 * are used to estimate the oversampled PSF at each spectral pixel of
 * each fibre by linear interpolation between the blocks.
 *
 * By taking advantage of the trace slope and curvature, a
 * sufficiently large averaging block allows the PSF to be subsampled
 * in this way, which can then be used during spectral extraction to
 * extract a spectrum with a slightly different (e.g. offset) trace
 * compared to the fibre flat if necessary, so the PSF measurement
 * doesn't have to be redone for every new trace.
 *
 * Since the fibre flats have all fibres illuminated, it is necessary
 * to correct the derived PSFs for overlap of the adjacent fibre
 * profiles, which is done using a simple Gaussian overlap model where
 * fibres are assumed to be equally illuminated and have spatial FWHM
 * given by the average of the FWHMs reported in the trace table.  The
 * overlap correction is computed for a given pixel of a given fibre
 * by determining the fraction of light in the pixel that is due to
 * the fibre of interest, relative to the total light in the pixel
 * from all fibres, and correcting the measured counts accordingly.
 * This simple overlap model seems to be sufficient for determining
 * the wings of the profiles to the desired width in 4MOST for Quality
 * Control purposes, but might be extended by iterative refinement of
 * the extracted fibre flat spectrum for more demanding applications.
 *
 * The output is a data cube (returned as a cpl_imagelist object)
 * where the PSF of each fibre at each spectral coordinate and the
 * corresponding variance are reported.
 *
 * @param   in_img             (Given)    The input fibre flat image,
 *                                        including bad pixel mask.
 *                                        The data type must be
 *                                        CPL_TYPE_FLOAT.
 * @param   in_hdr             (Given)    The FITS header of the input
 *                                        image.
 * @param   trace_tbl          (Given)    The relevant trace table for
 *                                        the fibre flats.
 * @param   trace_hdr          (Given)    The FITS header of the trace
 *                                        table.
 * @param   profwidth          (Given)    The maximum extent of the
 *                                        profile from its centre in
 *                                        pixels.
 * @param   subsample          (Given)    Number of PSF samples per
 *                                        spatial pixel.  Typically
 *                                        5.
 * @param   sblock             (Given)    The blocking factor in the
 *                                        spectral direction.
 * @param   prof_img           (Returned) An image list containing the
 *                                        fibre PSFs at each spectral
 *                                        pixel.  The structure of
 *                                        this argument is described
 *                                        in a note below.
 * @param   prof_var           (Returned) The corresponding variances
 *                                        of the fibre PSFs.
 * @param   prof_hdr           (Modified) A caller supplied
 *                                        propertylist to receive the
 *                                        FITS header for the output
 *                                        profile file with WCS.
 *
 * @note    The resulting profile and variance images have dimension
 *          subsample * (2 * profwidth + 1) x nfibres x nspec where
 *          the most rapidly varying axis are the oversampled spatial
 *          pixels across the profile, and these are given per fibre,
 *          per pixel in the spectral direction corresponding to the
 *          next two axes (where there are nspec planes in the
 *          output).  The spectral axis is always unbinned in the
 *          output, if the input fibre flat was binned the PSF will be
 *          upsampled by interpolation onto the native unbinned
 *          detector pixels.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If the trace table is
 *                                        empty, or if one of the
 *                                        required input FITS header
 *                                        keywords was not found.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the dimensions of the
 *                                        inputs don't match.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If one of the input FITS
 *                                        header keyword values had an
 *                                        incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *
 * @par Output FITS Headers:
 *   - <b>CD1_1</b>
 *   - <b>CD2_2</b>
 *   - <b>CD3_3</b>
 *   - <b>CRPIX1</b>
 *   - <b>CRPIX2</b>
 *   - <b>CRPIX3</b>
 *   - <b>CRVAL1</b>
 *   - <b>CRVAL2</b>
 *   - <b>CRVAL3</b>
 *
 * @par Output DRS Headers:
 *   - <b>PSUBSAMP</b>: Profile subsampling factor, i.e. the number of
 *     profile samples per physical detector pixel.
 *   - <b>PWIDTH</b> (pix): The extraction window used for the
 *     profile, defined as the half-width, i.e. the maximum extent of
 *     the profile from its centre in native detector pixels.
 *   - <b>WVSCLFAC</b>: Profile scale factor applied prior to writing
 *     the profile to allow it to be represented as integers for
 *     compression.  Always equal to 1 in the QC pipeline.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_model_psf_full (
    cpl_image *in_img,
    cpl_propertylist *in_hdr,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    int profwidth,
    int subsample,
    int sblock,
    cpl_imagelist **prof_img,
    cpl_imagelist **prof_var,
    cpl_propertylist *prof_hdr)
{
    int ifib,nr,n,i,j,ix1,ind,ind1,nr2,np,sblock2;
    int live,istart,m,yst,yfn,ycover,nsblocks=0;
    int ib,jst,jfn,ib1,ib2,j2,j1,k,offset;
    int jtmp;
    long naxis[2],naxis_out[3];
    int specbin,spatbin,isbinned;
    float *imgdata,sum,xp,x1,x2,val,mad,cut1,cut2;
    float bfac2,*profile,*profsig,w1,w2,minval;
    float delprof;
    unsigned char *bpmdata;
    double yref,xpos,ytr,ypos;
    cpl_polynomial *coefs;
    cpl_image *plane;
    cpl_errorstate prestate;

    /* Separate these out for garbage collection */
    int nh = 0;
    qmost_traceinfo *tr = NULL;

    float **prof_imgbufs = NULL;
    float **prof_varbufs = NULL;
    float *xarray = NULL;
    float *yarray = NULL;
    unsigned char *prbpm = NULL;
    float *buf = NULL;
    float *p1 = NULL;
    float **blockprofs = NULL;
    float **blockprofs_sig = NULL;

    cpl_polynomial *coefsa, *coefsb;
    int ipos;
    double yrefa, yrefb;

    float *useytr = NULL;
    float *usefwhm = NULL;
    float *useytra = NULL;
    float *usefwhma = NULL;
    float *useytrb = NULL;
    float *usefwhmb = NULL;
    int medfil, linfil;

    double norm, dx, fwhm, pkht;
    double xposa, yposa, dxa, fwhma, pkhta;
    double xposb, yposb, dxb, fwhmb, pkhtb;

    /* Check for NULL arguments */
    cpl_ensure_code(in_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_var != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_hdr != NULL, CPL_ERROR_NULL_INPUT);

    /* Now initialize outputs to NULL for garbage collection */
    *prof_img = NULL;
    *prof_var = NULL;

#undef TIDY
#define TIDY                                            \
    if(tr != NULL) {                                    \
        qmost_trclose(nh, &tr);                         \
        nh = 0;                                         \
        tr = NULL;                                      \
    }                                                   \
    if(*prof_img != NULL) {                             \
        cpl_imagelist_delete(*prof_img);                \
        *prof_img = NULL;                               \
    }                                                   \
    if(*prof_var != NULL) {                             \
        cpl_imagelist_delete(*prof_var);                \
        *prof_var = NULL;                               \
    }                                                   \
    if(prof_imgbufs != NULL) {                          \
        cpl_free(prof_imgbufs);                         \
        prof_imgbufs = NULL;                            \
    }                                                   \
    if(prof_varbufs != NULL) {                          \
        cpl_free(prof_varbufs);                         \
        prof_varbufs = NULL;                            \
    }                                                   \
    if(xarray != NULL) {                                \
        cpl_free(xarray);                               \
        xarray = NULL;                                  \
    }                                                   \
    if(yarray != NULL) {                                \
        cpl_free(yarray);                               \
        yarray = NULL;                                  \
    }                                                   \
    if(prbpm != NULL) {                                 \
        cpl_free(prbpm);                                \
        prbpm = NULL;                                   \
    }                                                   \
    if(buf != NULL) {                                   \
        cpl_free(buf);                                  \
        buf = NULL;                                     \
    }                                                   \
    if(p1 != NULL) {                                    \
        cpl_free(p1);                                   \
        p1 = NULL;                                      \
    }                                                   \
    if(blockprofs != NULL) {                            \
        for(jtmp = 0; jtmp < nsblocks; jtmp++) {        \
            if(blockprofs[jtmp] != NULL) {              \
                cpl_free(blockprofs[jtmp]);             \
            }                                           \
        }                                               \
        cpl_free(blockprofs);                           \
        blockprofs = NULL;                              \
    }                                                   \
    if(blockprofs_sig != NULL) {                        \
        for(jtmp = 0; jtmp < nsblocks; jtmp++) {        \
            if(blockprofs_sig[jtmp] != NULL) {          \
                cpl_free(blockprofs_sig[jtmp]);         \
            }                                           \
        }                                               \
        cpl_free(blockprofs_sig);                       \
        blockprofs_sig = NULL;                          \
    }                                                   \
    if(useytr != NULL) {                                \
        cpl_free(useytr);                               \
        useytr = NULL;                                  \
    }                                                   \
    if(usefwhm != NULL) {                               \
        cpl_free(usefwhm);                              \
        usefwhm = NULL;                                 \
    }                                                   \
    if(useytra != NULL) {                               \
        cpl_free(useytra);                              \
        useytra = NULL;                                 \
    }                                                   \
    if(usefwhma != NULL) {                              \
        cpl_free(usefwhma);                             \
        usefwhma = NULL;                                \
    }                                                   \
    if(useytrb != NULL) {                               \
        cpl_free(useytrb);                              \
        useytrb = NULL;                                 \
    }                                                   \
    if(usefwhmb != NULL) {                              \
        cpl_free(usefwhmb);                             \
        usefwhmb = NULL;                                \
    }

    /* Get image */
    naxis[0] = cpl_image_get_size_x(in_img);
    naxis[1] = cpl_image_get_size_y(in_img);

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    /* Get pointer to image data */
    imgdata = cpl_image_get_data_float(in_img);
    if(imgdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    /* Get bad pixel mask */
    bpmdata = cpl_mask_get_data(cpl_image_get_bpm(in_img));
    if(bpmdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "input BPM");
    }

    /* Open the trace table */
    if(qmost_tropen(trace_tbl, trace_hdr,
                    &nh, &tr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to load trace table");
    }  
    
    if(nh < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "trace table is empty, nothing to do");
    }

    /* Create the output, this also zeros it */
    np = 2*profwidth + 1;
    nr = subsample*np;
    nr2 = nr/2;
    bfac2 = 1.0/subsample;

    naxis_out[0] = nr;
    naxis_out[1] = nh;
    naxis_out[2] = naxis[1] * specbin;

    *prof_img = cpl_imagelist_new();
    *prof_var = cpl_imagelist_new();

    prof_imgbufs = cpl_calloc(naxis_out[2], sizeof(float *));
    prof_varbufs = cpl_calloc(naxis_out[2], sizeof(float *));

    for(j = 0; j < naxis_out[2]; j++) {
        plane = cpl_image_new(naxis_out[0], naxis_out[1], CPL_TYPE_FLOAT);
        cpl_imagelist_set(*prof_img, plane, j);
        /* now owned by imagelist */
        prof_imgbufs[j] = cpl_image_get_data_float(plane);

        plane = cpl_image_new(naxis_out[0], naxis_out[1], CPL_TYPE_FLOAT);
        cpl_imagelist_set(*prof_var, plane, j);
        /* now owned by imagelist */
        prof_varbufs[j] = cpl_image_get_data_float(plane);
    }

    /* Get some memory for the profiles */

    xarray = cpl_malloc(np*naxis[1]*sizeof(float));
    yarray = cpl_malloc(np*naxis[1]*sizeof(float));
    prbpm = cpl_malloc(nr*sizeof(unsigned char));
    buf = cpl_malloc(naxis[1]*sizeof(float));
    p1 = cpl_malloc(np*sizeof(float));

    /* Loop for each of the fibres in the trace. If this fibre is
       dead then just leave the profile as zeros */

    for (ifib = 1; ifib <= nh; ifib++) {
        live = tr[ifib-1].live;
        if (! live) {
            continue;
        }

        /* Read the trace coefficients */

        coefs = tr[ifib-1].coefs;
        yref = tr[ifib-1].yref;
        yst = tr[ifib-1].yst;
        yfn = tr[ifib-1].yfn;

        /* Ensure limited to bounds of (unbinned) image if smaller */

        if(yst < 1)
            yst = 1;
        if(yfn > naxis_out[2])
            yfn = naxis_out[2];

        ycover = yfn - yst + 1;
        nsblocks = (int)((float)ycover/(float)sblock) + 1;
        sblock2 = ycover/nsblocks;

        /* FWHM and peak height to estimate s.d. and normalisation of
         * Gaussian approximation to trace for overlap correction.
         * The FWHM samples are smoothed */
        useytr = cpl_malloc(tr[ifib-1].npos * sizeof(float));
        usefwhm = cpl_malloc(tr[ifib-1].npos * sizeof(float));

        pkht = 0;
        for(ipos = 0; ipos < tr[ifib-1].npos; ipos++) {
            useytr[ipos] = tr[ifib-1].ypos[ipos];

            if(tr[ifib-1].fwhm[ipos] > 0) {
                usefwhm[ipos] = tr[ifib-1].fwhm[ipos];
            }
            else {
                usefwhm[ipos] = -1000;
            }

            pkht += tr[ifib-1].peak[ipos];
        }

        if(tr[ifib-1].npos > 0) {
            pkht /= tr[ifib-1].npos;
        }
        else {
            pkht = 1;
        }

        qmost_sort_ff(useytr, usefwhm, tr[ifib-1].npos);

        medfil = qmost_max(3, tr[ifib-1].npos / 10);
        linfil = medfil / 3;

        qmost_filt1d(usefwhm, tr[ifib-1].npos, medfil, linfil, -1000);

        /* Information for neighbouring fibres */
        if(ifib >= 2) {
            coefsa = tr[ifib-2].coefs;
            yrefa = tr[ifib-2].yref;

            useytra = cpl_malloc(tr[ifib-2].npos * sizeof(float));
            usefwhma = cpl_malloc(tr[ifib-2].npos * sizeof(float));

            pkhta = 0;
            for(ipos = 0; ipos < tr[ifib-2].npos; ipos++) {
                useytra[ipos] = tr[ifib-2].ypos[ipos];

                if(tr[ifib-2].fwhm[ipos] > 0) {
                    usefwhma[ipos] = tr[ifib-2].fwhm[ipos];
                }
                else {
                    usefwhma[ipos] = -1000;
                }

                pkhta += tr[ifib-2].peak[ipos];
            }

            if(tr[ifib-2].npos > 0) {
                pkhta /= tr[ifib-2].npos;
            }
            else {
                pkhta = 1;
            }

            qmost_sort_ff(useytra, usefwhma, tr[ifib-2].npos);

            medfil = qmost_max(3, tr[ifib-2].npos / 10);
            linfil = medfil / 3;

            qmost_filt1d(usefwhma, tr[ifib-2].npos, medfil, linfil, -1000);
        }
        else {
            coefsa = NULL;
            yrefa = 0;
        }

        if(ifib < nh) {
            coefsb = tr[ifib].coefs;
            yrefb = tr[ifib].yref;

            useytrb = cpl_malloc(tr[ifib].npos * sizeof(float));
            usefwhmb = cpl_malloc(tr[ifib].npos * sizeof(float));

            pkhtb = 0;
            for(ipos = 0; ipos < tr[ifib].npos; ipos++) {
                useytrb[ipos] = tr[ifib].ypos[ipos];

                if(tr[ifib].fwhm[ipos] > 0) {
                    usefwhmb[ipos] = tr[ifib].fwhm[ipos];
                }
                else {
                    usefwhmb[ipos] = -1000;
                }

                pkhtb += tr[ifib].peak[ipos];
            }

            if(tr[ifib].npos > 0) {
                pkhtb /= tr[ifib].npos;
            }
            else {
                pkhtb = 1;
            }

            qmost_sort_ff(useytrb, usefwhmb, tr[ifib].npos);

            medfil = qmost_max(3, tr[ifib].npos / 10);
            linfil = medfil / 3;

            qmost_filt1d(usefwhmb, tr[ifib].npos, medfil, linfil, -1000);
        }
        else {
            coefsb = NULL;
            yrefb = 0;
        }

        blockprofs = cpl_calloc(nsblocks, sizeof(float *));
        blockprofs_sig = cpl_calloc(nsblocks, sizeof(float *));
        for (j = 0; j < nsblocks; j++) {
            blockprofs[j] = cpl_malloc(nr*sizeof(float));
            blockprofs_sig[j] = cpl_malloc(nr*sizeof(float));
        }

        /* Loop for each row and extract the profile */

        for (ib = 0; ib < nsblocks; ib++) {
            n = 0;
            jst = ib*sblock2 / specbin;
            jfn = jst + sblock2 / specbin;
            if (ib == nsblocks - 1)
                jfn = ycover / specbin;
            for (j = jst; j < jfn; j++) {
                ytr = (double)((j+0.5)*specbin-0.5+yst);
                ypos = (ytr - yref)/yref;
                xpos = cpl_polynomial_eval_1d(coefs,ypos,NULL);
                ix1 = (int)(xpos+0.5) - profwidth - 1;
                ind = (j+(yst-1)/specbin)*naxis[0];

                minval = 1.0e10;
                for (i = 0; i < np; i++) {
                    ind1 = ix1 + i;
                    if (ind1 < 0)
                        p1[i] = 0.0;
                    else if (ind1 >= naxis[0])
                        p1[i] = 0.0;
                    else
                        p1[i] = imgdata[ind+ix1+i];
                    minval = (p1[i] < minval ? p1[i] : minval);
                }

                /* Interpolate FWHM in table */
                fwhm = interp_fwhm(ytr,
                                   useytr,
                                   usefwhm,
                                   tr[ifib-1].npos);

                /* Trace table information for neighbours */
                if(coefsa != NULL) {
                    yposa = (ytr - yrefa)/yrefa;
                    xposa = cpl_polynomial_eval_1d(coefsa,yposa,NULL);
                    fwhma = interp_fwhm(ytr,
                                        useytra,
                                        usefwhma,
                                        tr[ifib-2].npos);
                }
                else {
                    xposa = 0;
                    fwhma = 0;
                }
                if(coefsb != NULL) {
                    yposb = (ytr - yrefb)/yrefb;
                    xposb = cpl_polynomial_eval_1d(coefsb,yposb,NULL);
                    fwhmb = interp_fwhm(ytr,
                                        useytrb,
                                        usefwhmb,
                                        tr[ifib].npos);
                }
                else {
                    xposb = 0;
                    fwhmb = 0;
                }

                /* Correct for overlap with immediate neighbours
                 * using simple Gaussian approximation. */
                for (i = 0; i < np; i++) {
                    norm = 1.0;
                    dx = (ix1+i+1 - xpos) * CPL_MATH_FWHM_SIG / fwhm;

                    if(coefsa != NULL) {
                        dxa = (ix1+i+1 - xposa) * CPL_MATH_FWHM_SIG / fwhma;
                        norm += (pkhta/pkht) * exp(0.5*(dx*dx - dxa*dxa));
                    }
                    
                    if(coefsb != NULL) {
                        dxb = (ix1+i+1 - xposb) * CPL_MATH_FWHM_SIG / fwhmb;
                        norm += (pkhtb/pkht) * exp(0.5*(dx*dx - dxb*dxb));
                    }

                    p1[i] /= norm;
                }

                for (i = 0; i < np; i++)
		    p1[i] = qmost_max(0.0,p1[i]);
                /*  p1[i] -= minval; */

                /* Now normalise and distribute it */

                sum = 0.0;
                for (i = 0; i < np; i++)
                    sum += p1[i];
                for (i = 0; i < np; i++) {
                    if(sum != 0) {
                        p1[i] /= sum;
                    }
                    xp = (float)(ix1 + i + 1) - xpos;
                    xarray[n] = xp;
                    yarray[n++] = p1[i];
                }
            }

            /* Now sort by x */

            qmost_sort_ff(xarray,yarray,n);

            /* Create the profile now. Loop for every fraction of a pixel */

            istart = 0;
            for (j = -nr2; j <= nr2; j++) {
                x1 = bfac2*(float)j - bfac2*0.5;
                x2 = x1 + bfac2;
                m = 0;
                for (i = istart; i < n; i++) {
                    if (xarray[i] > x2) {
                        istart = i;
                        break;
                    }
                    buf[m++] = yarray[i];
                }
                if (m >= sblock/20) {
                    prestate = cpl_errorstate_get();
		    if (qmost_meansig(buf,NULL,m,
                                      &val,&mad) != CPL_ERROR_NONE) {
                        cpl_errorstate_set(prestate);
                        val = 0.0;
                        mad = 0.0;
                        prbpm[j+nr2] = 1;
                    } else {
                        cut1 = val - 3.0*mad;
                        cut2 = val + 3.0*mad;
                        if (qmost_meansigcut(buf,NULL,m,cut1,cut2,
                                             &val,&mad) == CPL_ERROR_NONE) {
                            mad = qmost_max(mad,0.0);
                            prbpm[j+nr2] = 0;
                        } else {
                            cpl_errorstate_set(prestate);
                            val = 0.0;
                            mad = 0.0;
                            prbpm[j+nr2] = 1;
                        }
                    }
                } else {
                    val = 0.0;
                    mad = 0.0;
                    prbpm[j+nr2] = 1;
                }
                blockprofs[ib][j+nr2] = qmost_max(0.0,val);
                blockprofs_sig[ib][j+nr2] = mad/sqrt((float)(m-1));
            }

            /* Fill in the parts of the profile that have no information */

            plugholes_par(blockprofs[ib],prbpm,nr);
            plugholes_par(blockprofs_sig[ib],prbpm,nr);

	    /* allow for profile 1/2 pixellation in error estimate */
            for (j = -nr2+1; j < nr2; j++) {
	        delprof = fabs(blockprofs[ib][j+nr2]-blockprofs[ib][j-1+nr2]);
	        delprof += fabs(blockprofs[ib][j+nr2]-blockprofs[ib][j+1+nr2]);
		blockprofs_sig[ib][j+nr2] = 
                    sqrt(delprof*delprof/16.0 +
                         blockprofs_sig[ib][j+nr2]*blockprofs_sig[ib][j+nr2]);
	    }

        }

        /* Let the profile for the region before YST be the same as
           for the first good spectral location */

        for (j = 0; j < yst; j++) {
            profile = prof_imgbufs[j] + (ifib-1)*nr;
            profsig = prof_varbufs[j] + (ifib-1)*nr;

            memcpy(profile, blockprofs[0], nr*sizeof(float));
            memcpy(profsig, blockprofs_sig[0], nr*sizeof(float));
        }

        /* Now interpolate the results between adjacent blocks */

        offset = yst + sblock2/2;
        for (j = yst; j <= yfn; j++) {
            profile = prof_imgbufs[j-1] + (ifib-1)*nr;
            profsig = prof_varbufs[j-1] + (ifib-1)*nr;

            ib1 = (j - offset)/sblock2 + 1;
            ib1 = qmost_min(ib1,nsblocks);
            j1 = offset + (ib1-1)*sblock2;
            ib2 = qmost_min(ib1+1,nsblocks);
            j2 = offset + (ib2-1)*sblock2;

            if (j < offset) {
                memcpy(profile, blockprofs[0], nr*sizeof(float));
                memcpy(profsig, blockprofs_sig[0], nr*sizeof(float));
            } else if (ib1 == nsblocks && j > j1) {
                memcpy(profile, blockprofs[nsblocks-1], nr*sizeof(float));
                memcpy(profsig, blockprofs_sig[nsblocks-1], nr*sizeof(float));
            } else {
                w1 = (float)(j2 - j)/(float)sblock2;
                w2 = 1.0 - w1;
                for (k = 0; k < nr; k++) {
                    profile[k] = qmost_max(0.0,w1*blockprofs[ib1-1][k] +
                                           w2*blockprofs[ib2-1][k]);
                    profsig[k] = qmost_max(0.0,w1*blockprofs_sig[ib1-1][k] +
                                           w2*blockprofs_sig[ib2-1][k]);
                }
            }
        }

        /* Now do the bits after yfn */

        for (j = yfn; j < naxis[1]; j++) {
            profile = prof_imgbufs[j] + (ifib-1)*nr;
            profsig = prof_varbufs[j] + (ifib-1)*nr;

            memcpy(profile,blockprofs[nsblocks-1],nr*sizeof(float));
            memcpy(profsig,blockprofs_sig[nsblocks-1],nr*sizeof(float));
        }

        /* Tidy */

        for (j = 0; j < nsblocks; j++) {
            cpl_free(blockprofs[j]);
            cpl_free(blockprofs_sig[j]);
        }
        cpl_free(blockprofs);
        cpl_free(blockprofs_sig);
        blockprofs = NULL;
        blockprofs_sig = NULL;

        if(useytr != NULL) {
            cpl_free(useytr);
            useytr = NULL;
        }
        if(usefwhm != NULL) {
            cpl_free(usefwhm);
            usefwhm = NULL;
        }
        if(useytra != NULL) {
            cpl_free(useytra);
            useytra = NULL;
        }
        if(usefwhma != NULL) {
            cpl_free(usefwhma);
            usefwhma = NULL;
        }
        if(useytrb != NULL) {
            cpl_free(useytrb);
            useytrb = NULL;
        }
        if(usefwhmb != NULL) {
            cpl_free(usefwhmb);
            usefwhmb = NULL;
        }
    }

    /* Write the WCS for the profiles */
    cpl_propertylist_update_double(prof_hdr, "CRPIX1", 1.0);
    cpl_propertylist_update_double(prof_hdr, "CRPIX2", 1.0);
    cpl_propertylist_update_double(prof_hdr, "CRPIX3", 1.0);
    cpl_propertylist_update_double(prof_hdr, "CRVAL1",
                                   -1.0*bfac2*(double)nr2);
    cpl_propertylist_update_double(prof_hdr, "CRVAL2", 1.0);
    cpl_propertylist_update_double(prof_hdr, "CRVAL3", 1.0);
    cpl_propertylist_update_double(prof_hdr, "CD1_1", bfac2);
    cpl_propertylist_update_double(prof_hdr, "CD2_2", 1.0);
    cpl_propertylist_update_double(prof_hdr, "CD3_3", 1.0);

    /* And other header information needed */
    cpl_propertylist_update_int(prof_hdr, "ESO DRS PWIDTH", profwidth);
    cpl_propertylist_set_comment(prof_hdr, "ESO DRS PWIDTH",
                                 "[pix] Profile width used");

    cpl_propertylist_update_int(prof_hdr, "ESO DRS PSUBSAMP", subsample);
    cpl_propertylist_set_comment(prof_hdr, "ESO DRS PSUBSAMP",
                                 "Profile subsampling");

    cpl_propertylist_update_double(prof_hdr, "ESO DRS WVSCLFAC", 1.0);
    cpl_propertylist_set_comment(prof_hdr, "ESO DRS WVSCLFAC",
                                 "Profile scale factor for compression");

    /* Tidy and exit */

    cpl_free(prof_imgbufs);
    cpl_free(prof_varbufs);
    cpl_free(xarray);
    cpl_free(yarray);
    cpl_free(prbpm);
    cpl_free(buf);
    cpl_free(p1);

    qmost_trclose(nh,&tr);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Remove bad pixel data from an array.
 *
 * Remove bad pixel data from an array and replace using a parabolic
 * interpolation of the 3 nearest good pixels.
 *
 * @param   data             (Modified) The input data array.
 * @param   bpm              (Given)    The input bad pixel mask.
 * @param   nx               (Given)    The size of the input arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void plugholes_par (
    float *data,
    unsigned char *bpm,
    int nx)
{
    int sumbpm,i,i1,i2,i3,ifirst,ilast,ist,ifn,j;
    double x[3],y[3],c[3],xx;
    double x3mx2,x3mx1,x2mx1,x3px2,x2px1,y3my2,y2my1;
    float d1,d2,t1,t2,nc,x0,slope;

    /* If there aren't at least 3 good pixels or if all the pixels are
       good then get out of here */

    qmost_sumbpm(bpm,nx,&sumbpm);
    if (sumbpm < 3 || sumbpm == nx)
 	return;

    /* Find the first 3 good values in the array */

    i = 0;
    while (i < nx && bpm[i] != 0)
        i++;
    ifirst = i;
    i1 = i;
    i++;
    while (i < nx && bpm[i] != 0)
        i++;
    i2 = i;
    i++;
    while (i < nx && bpm[i] != 0)
        i++;
    i3 = i;

    /* Find the last good value in the array */

    i = nx - 1;
    while (i >= 0 && bpm[i] != 0)
        i--;
    ilast = i;

    /* Loop until we reach the last good pixel */

    while (1) {

 	/* Take the current group of 3 and see if there are any bad pixels
 	   in between. */

 	if (i3 - i1 > 2) {

 	    /* If there are, then fit a parabola to the 3 points and
 	       interpolate for all the bad pixels */

 	    x[0] = (double)(i1+1);
 	    x[1] = (double)(i2+1);
 	    x[2] = (double)(i3+1);
 	    y[0] = (double)data[i1];
 	    y[1] = (double)data[i2];
 	    y[2] = (double)data[i3];

	    /* special case exact parabolic fit */

	    x3mx1 = x[2] - x[0];
	    x3mx2 = x[2] - x[1];
	    x2mx1 = x[1] - x[0];
	    x3px2 = x[2] + x[1];
	    x2px1 = x[1] + x[0];
	    y3my2 = y[2] - y[1];
	    y2my1 = y[1] - y[0];
	    c[2] = (y3my2/x3mx2 - y2my1/x2mx1)/x3mx1;
	    c[1] = (y2my1*x3px2/x2mx1 - y3my2*x2px1/x3mx2)/x3mx1;
	    c[0] = y[1] - c[1]*x[1] - c[2]*pow(x[1],2.0);

	    /* line segment conditions to minimise outskirt undershoot */

	    if (i2 <= nx/2) {
	        ist = i2;
		ifn = i3;
	    } else {
	        ist = i1;
		ifn = i2;
	    }

	    /* is there a minimum in range */

	    if( c[2] == 0.0) {
	        x0 = -1.0;
	    } else {
	        x0 = -c[1]/(2.0*c[2]);
	    }
	    if( c[2] > 0.0 && x0 > (float)(ist+1) && x0 < (float)(ifn-1) ) {
  	        nc = (float)(ifn - ist);
	        d1 = data[ist];
	        d2 = data[ifn];
	        for (j = ist+1; j <= ifn-1; j++) {
		    t1 = 1.0 - (float)(j - ist)/nc;
		    t2 = 1.0 - t1;
		    data[j] = qmost_max(0.0,t1*d1 + t2*d2);
	        }
	    } else {
	        for (i = ist+1; i <= ifn-1; i++) {
 		    xx = (double)(i+1);
		    data[i] = qmost_max(0.0,c[0] + c[1]*xx  + c[2]*pow(xx,2.0));
	        }
	    }

	    /* ensure ends dealt with */

	    if (i1 == ifirst) {
  	        nc = (float)(i2 - i1);
	        d1 = data[i1];
	        d2 = data[i2];
	        for (j = i1+1; j <= i2-1; j++) {
		    t1 = 1.0 - (float)(j - i1)/nc;
		    t2 = 1.0 - t1;
		    data[j] = qmost_max(0.0,t1*d1 + t2*d2);
	        }
	    }
	    if (i3 == ilast) {
  	        nc = (float)(i3 - i2);
	        d1 = data[i2];
	        d2 = data[i3];
	        for (j = i2+1; j <= i3-1; j++) {
		    t1 = 1.0 - (float)(j - i2)/nc;
		    t2 = 1.0 - t1;
		    data[j] = qmost_max(0.0,t1*d1 + t2*d2);
	        }
	    }
 	}

 	/* Find the next good pixel */

 	if (i3 == ilast)
 	    break;
 	i1 = i2;
 	i2 = i3;
 	i = i3 + 1;
 	while (i < nx && bpm[i] != 0)
 	    i++;
 	i3 = i;
    }

    /* Now the left bit... */

    if (ifirst > 0) {
        slope = data[ifirst+1] - data[ifirst];
        for (j = 0; j < ifirst; j++)
            data[j] = qmost_max(0.0,slope*(float)(j - ifirst) + data[ifirst]);
    }

    /* Now the right bit... */

    if (ilast < nx - 1) {
        slope = data[ilast] - data[ilast-1];
        for (j = ilast+1; j < nx; j++)
            data[j] = qmost_max(0.0,slope*(float)(j - ilast) + data[ilast]);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Interpolate FWHM in lookup table of FWHM vs y coordinate.
 *
 * @param   y                (Given)    The y coordinate the FWHM is
 *                                      required at.
 * @param   tbl_y            (Given)    The y coordinate column of the
 *                                      lookup table.  The table must
 *                                      be sorted on y.
 * @param   tbl_fwhm         (Given)    The FWHM column of the lookup
 *                                      table.
 * @param   n                (Given)    The number of entries in the
 *                                      table.
 *
 * @return  float                       The interpolated FWHM value.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static float interp_fwhm (
    float y,
    float *tbl_y,
    float *tbl_fwhm,
    int n)
{
    int isp, ifp, im;
    float a, b, d, fwhm;

    /* Find leftmost element */
    isp = 0;
    ifp = n;

    while(isp < ifp) {
        im = (isp + ifp) / 2;
        if(tbl_y[im] < y) {
            isp = im + 1;
        }
        else {
            ifp = im;
        }
    }

    if(isp < 1) {
        /* Before first, set equal to first */
        fwhm = tbl_fwhm[0];
    }
    else if(isp >= n) {
        /* After last, set equal to last */
        fwhm = tbl_fwhm[n-1];
    }
    else {
        /* Linear interpolation */
        a = tbl_y[isp] - y;
        b = y - tbl_y[isp-1];
        d = a + b;
        if(d > FLT_EPSILON) {
            fwhm = (tbl_fwhm[isp-1] * a + tbl_fwhm[isp] * b) / d;
        }
        else {
            /* y coords early equal, take average */
            fwhm = 0.5 * (tbl_fwhm[isp-1] + tbl_fwhm[isp]);
        }
    }

    return fwhm;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate QC statistics for fibre PSF.
 *
 * @param   prof_img           (Given)    The image list containing the
 *                                        fibre PSFs at each spectral
 *                                        pixel.
 * @param   prof_hdr           (Modified) The profile FITS header,
 *                                        will be updated with QC
 *                                        information.
 * @param   arm                (Given)    One of the QMOST_ARM_*
 *                                        constants saying which arm
 *                                        we're processing.
 * @param   startpoint         (Given)    The spectral pixel at which
 *                                        to evaluate the FWHM of the
 *                                        fibre PSF, or -1 to select a
 *                                        suitable default (the middle
 *                                        pixel of the detector).
 * @param   fibinfo_tbl        (Modified) The FIBINFO table to receive
 *                                        QC columns, or NULL if none.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE If the start point is
 *                                        outside the image.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS headers or FIBINFO
 *                                        table columns was not
 *                                        found.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the profile image data
 *                                        type was not CPL_TYPE_FLOAT,
 *                                        or the data type of a FITS
 *                                        header keyword value or
 *                                        FIBINFO table column was not
 *                                        correct.
 *
 * @par Input FITS Header Information:
 *   - <b>CD1_1</b>
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_ST</b>
 *
 * @par Output QC Parameters:
 *   - <b>PSF FWHM MAX</b> (pix): The ensemble maximum FWHM of the
 *     PSFs.
 *   - <b>PSF FWHM MAXSPC</b>: The fibre with the maximum FWHM.
 *   - <b>PSF FWHM MED</b> (pix): The ensemble median FWHM of the
 *     PSFs.
 *   - <b>PSF FWHM MIN</b> (pix): The ensemble minimum FWHM of the
 *     PSFs.
 *   - <b>PSF FWHM MINSPC</b>: The fibre with the minimum FWHM.
 *   - <b>PSF FWHM POS</b> (pix): The coordinate along the spectral
 *     axis at which the PSF FWHM was measured.
 *   - <b>PSF FWHM RMS</b> (pix): The ensemble robustly-estimated RMS
 *     FWHM of the PSFs.
 *
 * @par Output FIBINFO Table Columns:
 *   - <b>FWHM_a</b> (pix): The spatial FWHM of the fibre PSF at the
 *     spectral pixel specified by the startpoint argument (usually
 *     the central spectral pixel) in arm "a" (R, G or B, depending on
 *     the value of the parameter arm).
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_psf_stats (
    cpl_imagelist *prof_img,
    cpl_propertylist *prof_hdr,
    int arm,
    int startpoint,
    cpl_table *fibinfo_tbl)
{
    float pixscl;
    int nsamp, ifib, nfib, nspec;
    cpl_image *this_prof_img;
    float *this_prof_buf, *buf;

    const char *arm_extname;
    char arm_ltr;
    char *colname = NULL;

    int fib_st, anynul;
    float pk, dist, loc1, loc2, fwhm;
    int pkloc, j, found1, found2;

    float *all_fwhm = NULL;
    cpl_errorstate prestate;
    float fwhm_min, fwhm_max, fwhm_med, fwhm_sig;
    int fwhm_minspc, fwhm_maxspc;
    int nfwhm;

    cpl_ensure_code(prof_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_hdr != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(colname != NULL) {                       \
        cpl_free(colname);                      \
        colname = NULL;                         \
    }                                           \
    if(all_fwhm != NULL) {                      \
        cpl_free(all_fwhm);                     \
        all_fwhm = NULL;                        \
    }

    /* Number of spectral pixels */
    nspec = cpl_imagelist_get_size(prof_img);

    /* Start point */
    if(startpoint < 0) {
        startpoint = nspec / 2 + 1;
    }

    if(startpoint < 1 || startpoint > nspec) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                     "start point %d is out of range "
                                     "[%d:%d]",
                                     startpoint,
                                     1, nspec);
    }

    /* Get the appropriate plane for analysis */
    this_prof_img = cpl_imagelist_get(prof_img, startpoint-1);

    nsamp = cpl_image_get_size_x(this_prof_img);
    nfib = cpl_image_get_size_y(this_prof_img);

    this_prof_buf = cpl_image_get_data_float(this_prof_img);
    if(this_prof_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer "
                                     "to PSF");
    }

    /* Get pixel scale */
    if(qmost_cpl_propertylist_get_float(prof_hdr,
                                        "CD1_1",
                                        &pixscl) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get CD1_1 from "
                                     "PSF FITS header");
    }

    if(fibinfo_tbl != NULL) {
        /* Set up column in FIBINFO */
        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not determine EXTNAME "
                                         "for arm %d", arm);
        }
        
        arm_ltr = arm_extname[0];
        
        colname = cpl_sprintf("FWHM_%c", arm_ltr);
        
        if(!cpl_table_has_column(fibinfo_tbl, colname)) {
            if(cpl_table_new_column(fibinfo_tbl,
                                    colname,
                                    CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create FIBINFO column "
                                             "FWHM_%c", arm_ltr);
            }

	    if(cpl_table_set_column_unit(fibinfo_tbl,
					 colname,
					 "pixels") != CPL_ERROR_NONE) {
		TIDY;
		return cpl_error_set_message(cpl_func, cpl_error_get_code(),
					     "could not set FWHM_%c unit",
					     arm_ltr);
	    }
        }
    }

    /* Loop over fibres */
    all_fwhm = cpl_malloc(nfib * sizeof(float));

    fwhm_min = 0;
    fwhm_minspc = -1;
    fwhm_max = 0;
    fwhm_maxspc = -1;
    nfwhm = 0;

    for(ifib = 0; ifib < nfib; ifib++) {
        buf = this_prof_buf + ifib * nsamp;

        if(fibinfo_tbl != NULL) {
            /* Was fibre allocated? */
            fib_st = cpl_table_get_int(fibinfo_tbl, 
                                       "FIB_ST",
                                       ifib,
                                       &anynul);
            if(anynul < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d from FIBINFO",
                                             "FIB_ST",
                                             ifib+1);
            }
            else if(anynul > 0) {  /* NULL */
                fib_st = -1;
            }
            
            if(fib_st != 2)
                continue;
        }

        /* Find the location of the peak */

        pk = buf[0];
        pkloc = 0;
        for (j = 1; j < nsamp; j++) {
            if (buf[j] > pk) {
                pk = buf[j];
                pkloc = j;
            }
        }

        /* Look for the first half light point */

        found1 = 0;
        for (j = pkloc - 1; j > 0; j--) {
            if (buf[j] < 0.5*pk) {
                found1 = 1;
                break;
            }
        }

        if(found1 && buf[j+1] != buf[j]) {
            dist = (0.5*pk - buf[j])/(buf[j+1] - buf[j]);
        }
        else {
            dist = 0.5;
        }

        loc1 = (float)j + dist;

        /* Look for the second half light point */

        found2 = 0;
        for (j = pkloc + 1; j < nsamp; j++) {
            if (buf[j] < 0.5*pk) {
                found2 = 1;
                break;
            }
        }

        if(found2 && buf[j-1] != buf[j]) {
            dist = (0.5*pk - buf[j])/(buf[j-1] - buf[j]);
        }
        else {
            dist = 0.5;
        }

        loc2 = (float)j - dist;

        /* Full distance now */

        if(found1 && found2) {
            fwhm = pixscl*(loc2 - loc1);
        }
        else {
            /* Flag as couldn't determine */
            fwhm = -1.0;
        }

        /* Write FWHM to FIBINFO if present */
        if(fibinfo_tbl != NULL) {
            if(cpl_table_set(fibinfo_tbl,
                             colname,
                             ifib,
                             fwhm) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to write FWHM for row "
                                             "%d to FIBINFO", ifib+1);
            }
        }

        /* Don't include in statistics if invalid */
        if(fwhm <= 0) {
            continue;
        }

        /* Accumulate statistics of FWHM for QC headers */
        if(fwhm_minspc < 0 || fwhm < fwhm_min) {
            fwhm_min = fwhm;
            fwhm_minspc = ifib+1;
        }
        if(fwhm_maxspc < 0 || fwhm > fwhm_max) {
            fwhm_max = fwhm;
            fwhm_maxspc = ifib+1;
        }

        all_fwhm[nfwhm] = fwhm;
        nfwhm++;
    }

    if(colname != NULL) {
        cpl_free(colname);
        colname = NULL;
    }

    /* Write QC headers */
    if(nfwhm > 0) {
        prestate = cpl_errorstate_get();

        if(qmost_medmad(all_fwhm, NULL, nfwhm,
                        &fwhm_med, &fwhm_sig) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            fwhm_med = 0;
            fwhm_sig = 0;
        }

        fwhm_sig *= CPL_MATH_STD_MAD;

        cpl_propertylist_update_float(prof_hdr,
                                      "ESO QC PSF FWHM MED",
                                      fwhm_med);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM MED",
                                     "[pix] Median PSF FWHM");
        
        cpl_propertylist_update_float(prof_hdr,
                                      "ESO QC PSF FWHM RMS",
                                      fwhm_sig);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM RMS",
                                     "[pix] RMS PSF FWHM");
        
        cpl_propertylist_update_float(prof_hdr,
                                      "ESO QC PSF FWHM MIN",
                                      fwhm_min);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM MIN",
                                     "[pix] Minimum PSF FWHM");
        
        cpl_propertylist_update_int(prof_hdr,
                                    "ESO QC PSF FWHM MINSPC",
                                    fwhm_minspc);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM MINSPC",
                                     "Fibre with minimum PSF FWHM");

        cpl_propertylist_update_float(prof_hdr,
                                      "ESO QC PSF FWHM MAX",
                                      fwhm_max);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM MAX",
                                     "[pix] Maximum PSF FWHM");

        cpl_propertylist_update_int(prof_hdr,
                                    "ESO QC PSF FWHM MAXSPC",
                                    fwhm_maxspc);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM MAXSPC",
                                     "Fibre with maximum PSF FWHM");

        cpl_propertylist_update_int(prof_hdr,
                                    "ESO QC PSF FWHM POS",
                                    startpoint);
        cpl_propertylist_set_comment(prof_hdr,
                                     "ESO QC PSF FWHM POS",
                                     "[pix] Spectral pixel where FWHM "
                                     "was evaluated");
    }

    cpl_free(all_fwhm);
    all_fwhm = NULL;

    return CPL_ERROR_NONE;
}

/**@}*/

/*

$Log$
Revision 1.13  20210712  mji
Resurrected parabolic interpolation as special case and fixed
outskirt undershoot and renamed LIFU routine to FULL

Revision 1.12  2019/02/25 10:42:37  jrl
New memory allocation scheme

Revision 1.11  2018/08/28 12:03:11  jrl
Modified to guard against negative profile values

Revision 1.10  2018/07/15 14:34:09  jim
Added lifu routine

Revision 1.9  2017/12/20 16:46:34  jim
commented out some stuff

Revision 1.8  2017/10/05 09:10:07  jim
Modified the way holes are plugged in the profiles

Revision 1.7  2017/03/23 12:10:37  jim
Fixed issue where a NaN could arise

Revision 1.6  2017/03/14 11:30:31  jim
Fixed typo in doc

Revision 1.5  2017/01/17 09:00:40  jim
Added module qmost_psf_stats

Revision 1.4  2016/10/23 15:55:38  jim
Added docs

Revision 1.3  2016/07/06 11:04:18  jim
Modified to change the way input files are specified. Improved speed.

Revision 1.2  2016/05/20 09:58:57  jim
Uses new version of trace file

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry


*/
