/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_pca.h"
#include "qmost_sort.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_pca  qmost_pca
 * 
 * Principal components analysis functions used in sky subtraction.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void qmost_jacobi (
    double **a,
    double **v,
    int n);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Form the covariance matrix for PCA analysis.
 *
 * A covariance matrix of the spectra is formed.  It is assumed to
 * to be symetric.
 *
 * @param   in           (Given)    The array of input spectra.
 * @param   nspec        (Given)    The number of spectra in the input
 *                                  array.
 * @param   skymask      (Given)    The sky mask showing where the
 *                                  sky lines are.
 * @param   npix         (Given)    The number of pixels pera
 *                                  spectrum. 
 *
 * @return  double ** array with the covariance matrix of size
 *          nspec x nspec.  This will need to be freed by the caller.
 *
 * @note    Each element of the matrix is a sum over npix of the
 *          product of two spectra.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

double **qmost_pca_form_covar (
    double *in,
    float *skymask,
    int nspec, 
    int npix)
{
    int i,j,k;
    double **covar = NULL;
    double *work = NULL;
    double *s1,*s2;

    /* Get some space for the covariance matrix */
    covar = cpl_malloc(nspec*sizeof(double *));
    work = cpl_calloc(nspec*nspec,sizeof(double));

    for (i = 0; i < nspec; i++)
        covar[i] = work + i*nspec;

    /* Form the covariance. Just do half and then duplicate it */
    
    for (k = 0; k < nspec; k++) {
        s1 = in + k*npix;
        for (j = 0; j <= k; j++) {
            s2 = in + j*npix;
            covar[k][j] = 0.0;
            for (i = 0; i < npix; i++)
                if (skymask[i] > 0.0)
                    covar[k][j] += s1[i]*s2[i];
            covar[j][k] = covar[k][j];
        }
    }

    return(covar);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate the PCA eigenvalues and eigenvectors.
 *
 * The PCA eigenvalue solution is found using Jacobi's method.  The
 * output vectors and values are sorted in descending order of
 * eigenvalue.
 *
 * @param   covar        (Modified) The input covariance matrix.  This
 *                                  is also used as workspace and will
 *                                  be destroyed.
 * @param   nx           (Given)    The length of the sides of the
 *                                  covariance matrix.
 * @param   eigenvalues  (Returned) A 1d array of eigenvalues, length
 *                                  nx.
 * @param   eigenfrac    (Returned) A 1d array of eigenfractions,
 *                                  length nx.
 * @param   eigenvectors (Returned) A 2d array of eigenvectors, length
 *                                  nx x nx.
 * @param   diagvar      (Given)    Offset to be applied to eigenvalues
 *                                  to allow for noise variance.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_pca_get_eigen (
    double **covar,
    int nx,
    double **eigenvalues,
    double **eigenfrac,
    double ***eigenvectors, 
    float diagvar)
{
    double *tempd,*w,sum,*tempd2;
    int i,*iw,isort,j;

    /* Get some workspace for the eigens */
    *eigenvalues = cpl_malloc(nx*sizeof(double));
    *eigenfrac = cpl_malloc(nx*sizeof(double));
    *eigenvectors = cpl_malloc(nx*sizeof(double *));
    tempd = cpl_malloc(nx*nx*sizeof(double));
    iw = cpl_malloc(nx*sizeof(int));
    w = cpl_malloc(nx*sizeof(double));
    tempd2 = cpl_malloc(nx*nx*sizeof(double));

    for (i = 0; i < nx; i++)
        (*eigenvectors)[i] = tempd + i*nx;

    /* Solve for the eigens */
    qmost_jacobi(covar, *eigenvectors, nx);

    for (i = 0; i < nx; i++)
        (*eigenvalues)[i] = covar[i][i];

    /* constrain diagvar by comparing to covar diag sum via eigenvalue sum */

    sum = 0.0;
    for (i = 0; i < nx; i++)
      sum += (*eigenvalues)[i];
    diagvar = qmost_min(diagvar,0.01*sum);    

    /* Now sort the eigens into the correct order */

    sum = 0.0;
    for (i = 0; i < nx; i++)
        sum += qmost_max(0.0,(*eigenvalues)[i]-diagvar);
    for (i = 0; i < nx; i++) {
        iw[i] = i + 1;
        w[i] = (*eigenvalues)[i];
    }
    qmost_sort_rev_di(w,iw,nx);

    /* Now move the sorted eigens back into the original arrays */
            
    for (i = 0; i < nx; i++) {
        isort = iw[i] - 1;
        for (j = 0; j < nx; j++) 
            tempd2[j*nx+i] = tempd[j*nx+isort];
        (*eigenvalues)[i] = w[i]-diagvar;
        if(sum > 0) {
            (*eigenfrac)[i] = (*eigenvalues)[i]/sum;
        }
        else {
            (*eigenfrac)[i] = (*eigenvalues)[i];
        }
    }
    cpl_free(tempd);
    for (i = 0; i < nx; i++)
        (*eigenvectors)[i] = tempd2 + i*nx;
    cpl_free(iw);
    cpl_free(w);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Transform the eigenvector solutions to proper spectral
 *          solutions.
 *
 * @param   eigen        (Given)    The input eigenvector solutions.
 * @param   data         (Given)    The input sky spectra.
 * @param   skymask      (Given)    The sky mask showing where the sky
 *                                  lines are.
 * @param   ns           (Given)    The number of sky spectra.
 * @param   np           (Given)    The number of pixels in each sky
 *                                  spectrum.
 *
 * @return  double ** 2d array with the transformed eigenvectors.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

double **qmost_pca_trans_eigen (
    double **eigen,
    double *data,
    float *skymask,
    int ns,
    int np)
{
    double **outdata,*temp,sum;
    int i,j,k;

    /* Get some space for the output array */
    outdata = cpl_malloc(ns*sizeof(double *));
    temp = cpl_calloc(ns*np,sizeof(double));

    for (i = 0; i < ns; i++)
        outdata[i] = temp + i*np;

    /* Do the transformation and normalise */

    for (i = 0; i < ns; i++) {
        sum = 0.0;
        for (k = 0; k < np; k++) {
            outdata[i][k] = 0.0;
            if (skymask[k] > 0.0) {
                for (j = 0; j < ns; j++)
                    outdata[i][k] += eigen[j][i]*data[j*np+k];
                sum += outdata[i][k]*outdata[i][k];
            }
        }
        if(sum > 0) {
            sum = sqrt(sum);
            for (k = 0; k < np; k++)
                outdata[i][k] /= sum;
        }
    }

    /* Get out of here */

    return(outdata);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Reconstruct a single spectrum using the eigenvectors.
 *
 * @param   eigen        (Given)    A 2d array with the transformed
 *                                  eigenvectors.
 * @param   indata       (Given)    An array with the input spectrum
 *                                  to be corrected.
 * @param   np           (Given)    The number of pixels in each
 *                                  spectrum.
 * @param   skymask      (Given)    The sky mask showing where the sky
 *                                  lines are.
 * @param   invar        (Given)    An array with the variance in the
 *                                  input spectrum.
 * @param   neigen       (Given)    The number eigenvectors to use in
 *                                  the solution.
 * @param   recon        (Modified) The reconstructed sky spectrum.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_pca_recon_spec (
    double **eigen,
    double *indata, 
    int np,
    float *skymask,
    float *invar,
    int neigen,
    float *recon)
{
    int i,k;
    double sum;

    unsigned char *rejmask = (unsigned char *) NULL;
    int iiter, nrej;
    float resid, ksig = 5.0;

    /* Set up mask */
    rejmask = cpl_malloc(np * sizeof(unsigned char));

    for (i = 0; i < np; i++) {
        rejmask[i] = (skymask[i] > 0.0 && invar[i] != 0.0) ? 0 : 1;
    }

    /* Rejection iteration loop */
    for (iiter = 0; iiter < 5; iiter++) {
        /* Zero reconstructed spectrum */
        memset(recon, 0, np * sizeof(float));

        /* Reconstruct spectrum */
        for (k = 0; k < neigen; k++) {
            /* Projection onto PCA basis */
            sum = 0.0;
            for (i = 0; i < np; i++) {
                if (!rejmask[i]) {
                    sum += eigen[k][i]*indata[i];
                }
            }

            for (i = 0; i < np; i++) {
                recon[i] += (float)(sum*eigen[k][i]);
            }
        }

        /* Outlier rejection based on expected variance */
        nrej = 0;

        for (i = 0; i < np; i++) {
            if(rejmask[i] || invar[i] <= 0.0) {
                continue;
            }

            resid = indata[i] - recon[i];

            if(!rejmask[i] && resid*resid > ksig*ksig * invar[i]) {
                rejmask[i] = 1;
                nrej++;
            }
        }

        /* Get out if we didn't reject anything (solution is done) */
        if(nrej <= 0) {
            break;
        }
    }

    cpl_free(rejmask);
    rejmask = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Compute eigenvalues and eigenvectors of a square symmetric
 *          matrix.
 *
 * Performs a cyclic Jacobi eigenvalue decomposition of square matrix
 * A.  On return, the upper triangle of A is destroyed, the diagonal
 * gives the eigenvalues, and the columns of V give the eigenvectors.
 * The lower triangle of A is not accessed.  The returned eigenvalues
 * and eigenvectors are not sorted, if needed this should be done by
 * the user.
 *
 * @param   a          (Modified) A 2D array containing the matrix A.
 *                                On return will contain the
 *                                eigenvalues of A on the diagonal.
 * @param   v          (Modified) An 2D array to receive the matrix of
 *                                eigenvectors of A, where column j
 *                                will contain eigenvector j.
 * @param   n          (Given)    The dimension of the matrices.
 *
 * @return  void
 *
 * @note    The matrices are stored in row-major order such that Aij
 *          is found at a[i][j].
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void qmost_jacobi (
    double **a,
    double **v,
    int n)
{
    double tol;
    int icnt, nrot;
    int i, j, k;
    double aii, aij, ajj;
    double tau, s, c, t, rho;
    double ti, tj;

    /* Iteration limit */
#define EIGEN_MAXITER  100

    /* Initialize v to identity matrix */
    for(i = 0; i < n; i++) {
        for(j = 0; j < n; j++) {
            v[i][j] = 0;
        }

        v[i][i] = 1;
    }

    tol = DBL_EPSILON;

    /* Main iteration loop */
    for(icnt = 0; icnt < EIGEN_MAXITER; icnt++) {
        /* Loop over off-diagonal matrix elements */
        nrot = 0;

        for(i = 0; i < n; i++) {
            for(j = i+1; j < n; j++) {
                /* Process this off-diagonal */
                aii = a[i][i];
                ajj = a[j][j];
                aij = a[i][j];

                /* Trap small off-diagonals following recommendation
                 * from LAPACK working note 15.  The iteration stops
                 * when all matrix elements satisfy this criterion so
                 * we make no updates.  The comparison used here needs
                 * to be <= to handle the case where the right hand
                 * side is zero. */
                if(fabs(aij) <= tol * sqrt(fabs(aii*ajj))) {
                    continue;
                }

                /* Compute sin-cos group for Jacobi rotation by
                 * solving off-diagonal equation for zero, following
                 * numerically stable method from Golub & van Loan
                 * Sect. 8.4.  The solution of the quadratic needs to
                 * be the smaller value of t. */
                tau = 0.5 * (ajj - aii) / aij;
                t = copysign(1.0, tau) / (fabs(tau) + sqrt(1.0 + tau*tau));
                c = 1.0 / sqrt(1 + t*t);
                s = t * c;
                rho = (1 - c) / s;

                /* Upper triangle of Q^T A Q.  First, the off-diagonal
                 * we eliminated with the Jacobi rotation and the
                 * corresponding two diagonal elements. */
                a[i][i] -= t * aij;
                a[j][j] += t * aij;
                a[i][j] = 0;

                /* Now update the rest of the matrix.  This happens in
                 * three pieces to ensure each loop only accesses the
                 * upper triangle. */
                for(k = 0; k < i; k++) {
                    ti = a[k][i];
                    tj = a[k][j];
                    
                    a[k][i] -= s * (tj + rho * ti);
                    a[k][j] += s * (ti - rho * tj);
                }
                
                for(k = i+1; k < j; k++) {
                    ti = a[i][k];
                    tj = a[k][j];
                    
                    a[i][k] -= s * (tj + rho * ti);
                    a[k][j] += s * (ti - rho * tj);
                }

                for(k = j+1; k < n; k++) {
                    ti = a[i][k];
                    tj = a[j][k];
                    
                    a[i][k] -= s * (tj + rho * ti);
                    a[j][k] += s * (ti - rho * tj);
                }

                /* Q^T V */
                for(k = 0; k < n; k++) {
                    ti = v[k][i];
                    tj = v[k][j];
                    
                    v[k][i] -= s * (tj + rho * ti);
                    v[k][j] += s * (ti - rho * tj);
                }

                nrot++;
            }
        }

        /* If there were no updates, we are done */
        if(nrot < 1) {
            break;
        }
    }

    /* Emit a warning if we failed to converge */
    if(icnt >= EIGEN_MAXITER) {
        cpl_msg_warning(cpl_func,
                        "failed to converge after %d iterations\n",
                        EIGEN_MAXITER);
    }
}

/**@}*/

/*

$Log$
Revision 1.5  2019/04/01 16:06:00  jrl
Small modifications mainly to _recon routine

Revision 1.4  2019/02/25 10:43:20  jrl
New memory allocation scheme

Revision 1.3  2018/10/12 10:07:58  jrl
Added skymask to _covar and _trans_eigen routines

Revision 1.2  2016/10/23 15:55:38  jim
Added docs

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry


*/
