/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <strings.h>
#include <cpl.h>

#include "qmost_constants.h"
#include "qmost_pfits.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_pfits     qmost_pfits
 *
 * FITS header protected access.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_pfits.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine which arm of the spectograph we're processing.
 *
 * @param   plist             (Given)    Extension header to read from.
 * @param   arm               (Returned) One of the QMOST_ARM_* values
 *                                       specifying which arm this is.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the required input FITS
 *                                    header keywords are missing.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If ESO INS PATH isn't one of the
 *                                    recognised values.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input FITS header keyword
 *                                    values had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO INS PATH</b>
 *   - <b>EXTNAME</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_arm (
    cpl_propertylist *plist,
    int *arm)
{
    const char *path;

    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(arm, CPL_ERROR_NULL_INPUT);

    /* Determine arm from extension header.  First try the way it's
     * done in simulated data, which is ESO INS PATH. */
    if(cpl_propertylist_has(plist, "ESO INS PATH")) {
        path = cpl_propertylist_get_string(plist, "ESO INS PATH");
        if(path == NULL) {
            return cpl_error_set_message(cpl_func,
                                         cpl_error_get_code(),
                                         "couldn't read ESO INS PATH "
                                         "from header");
        }
        
        if(!strcasecmp(path, "red")) {
            *arm = QMOST_ARM_RED;
        }
        else if(!strcasecmp(path, "green")) {
            *arm = QMOST_ARM_GREEN;
        }
        else if(!strcasecmp(path, "blue")) {
            *arm = QMOST_ARM_BLUE;
        }
        else {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ILLEGAL_INPUT,
                                         "unexpected PATH in extension: %s",
                                         path);
        }
    }
    else if(cpl_propertylist_has(plist, "EXTNAME")) {
        /* Real instrument data such as lab data use EXTNAME instead.
         * We can't use this for simulated data because the EXTNAMEs
         * aren't correct in the simulated raws. */
        path = cpl_propertylist_get_string(plist, "EXTNAME");
        if(path == NULL) {
            return cpl_error_set_message(cpl_func,
                                         cpl_error_get_code(),
                                         "couldn't read EXTNAME "
                                         "from header");
        }
        
        if(!strcasecmp(path, "RED_DATA")) {
            *arm = QMOST_ARM_RED;
        }
        else if(!strcasecmp(path, "GREEN_DATA")) {
            *arm = QMOST_ARM_GREEN;
        }
        else if(!strcasecmp(path, "BLUE_DATA")) {
            *arm = QMOST_ARM_BLUE;
        }
        else {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ILLEGAL_INPUT,
                                         "unexpected EXTNAME in extension: %s",
                                         path);
        }
    }
    else {
        /* Header has neither.  This shouldn't happen so raise an
         * error. */
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "ESO INS PATH and EXTNAME are both "
                                     "missing in extension header");
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Get the detector live flag.
 *
 * @param   plist             (Given)    FITS extension header to read
 *                                       from.
 * @param   detlive           (Returned) The detector live flag.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT         If the input or output
 *                                       pointers are NULL.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DET CHIP LIVE</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_detlive (
    cpl_propertylist *plist,
    int *detlive)
{
    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(detlive, CPL_ERROR_NULL_INPUT);

    if(cpl_propertylist_has(plist, "ESO DET CHIP LIVE")) {
        if(qmost_cpl_propertylist_get_int(plist,
                                          "ESO DET CHIP LIVE",
                                          detlive) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't read ESO DET CHIP LIVE");
        }
    }
    else {
        /* If not present, assume the detector is live. */
        *detlive = 1;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Get the dummy product flag.
 *
 * This is the QC pipe output corresponding to the DET CHIP LIVE flag,
 * but inverted, and in addition to the raw also factors in whether
 * any of the calibrations used in making the product were dummy.
 * This routine is used to read it back from calibration frames.
 *
 * @param   plist             (Given)    FITS primary header to read
 *                                       from.
 * @param   dummy             (Returned) The dummy product flag.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT         If the input or output
 *                                       pointers are NULL.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_dummy (
    cpl_propertylist *plist,
    int *dummy)
{
    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dummy, CPL_ERROR_NULL_INPUT);

    if(cpl_propertylist_has(plist, "ESO DRS DUMMY")) {

        if(qmost_cpl_propertylist_get_int(plist,
                                          "ESO DRS DUMMY",
                                          dummy) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't read ESO DRS DUMMY");
        }
    }
    else {
        /* If not present, assume product is not dummy */
        *dummy = 0;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine the exposure time.
 *
 * @param   plist             (Given)    FITS primary header to read
 *                                       from.
 * @param   exptime           (Returned) The exposure time (seconds).
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND     If the required input FITS
 *                                       header keywords are missing.
 * @retval  CPL_ERROR_NULL_INPUT         If the input or output
 *                                       pointers are NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input FITS header keyword
 *                                    values had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>EXPTIME</b>
 *   - <b>ESO EXPTIME</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_exptime (
    cpl_propertylist *plist,
    double *exptime)
{
    cpl_error_code code;

    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(exptime, CPL_ERROR_NULL_INPUT);

    /* Try the standard EXPTIME keyword first (preferred) and then
     * ESO EXPTIME.  Francisco's recipes used the latter, but it
     * doesn't exist in e.g. lab data. */
    if(cpl_propertylist_has(plist, "EXPTIME")) {
        code = qmost_cpl_propertylist_get_double(plist, "EXPTIME",
                                                 exptime);
    }
    else if(cpl_propertylist_has(plist, "ESO EXPTIME")) {
        code = qmost_cpl_propertylist_get_double(plist, "ESO EXPTIME",
                                                 exptime);
    }
    else {
        /* Header has neither. */
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "EXPTIME and ESO EXPTIME are both "
                                     "missing in FITS header");
    }

    return code;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Return the image EXTNAME for an arm of the spectrograph.
 *
 * @param   arm               (Given)    One of the QMOST_ARM_* values
 *                                       specifying which arm this is.
 *
 * @return  The image HDU EXTNAME to use for the given arm.
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If ESO INS PATH isn't one of the
 *                                    recognised values.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input FITS header keyword
 *                                    values had an incorrect data type.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

const char *qmost_pfits_get_extname (
    int arm)
{
    /* Table of extension names */
    static const char *qmost_extnames[QMOST_NUM_ARMS] = {
        "RED_DATA", "GREEN_DATA", "BLUE_DATA"
    };

    if(arm > 0 && arm <= QMOST_NUM_ARMS) {
        return qmost_extnames[arm-1];
    }
    else {
        cpl_error_set_message(cpl_func,
                              CPL_ERROR_ILLEGAL_INPUT,
                              "invalid arm: %d", arm);
        return NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read start y position of extracted spectra.
 *
 * The MINYST header used to indicate the minimum start y coordinate
 * in a trace file, or the starting y position of the extracted
 * spectra in an extracted spectrum file, is read.  It's not an error
 * if the header is missing, the return value is simply left
 * unmodified.  The user can set it to an otherwise invalid flag value
 * such as a negative number to check if the headers were missing.
 *
 * @param   plist             (Given)    FITS primary header to read
 *                                       from.
 * @param   result            (Returned) The start y position,
 *                                       numbering from 1.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND     If the required input FITS
 *                                       header keywords are missing.
 * @retval  CPL_ERROR_NULL_INPUT         If the input or output
 *                                       pointers are NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH      If the input FITS header
 *                                       keyword values had an
 *                                       incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MINYST</b>
 *   - <b>MINYST</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_minyst (
    cpl_propertylist *plist,
    int *result)
{
    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);

    if(cpl_propertylist_has(plist, "ESO DRS MINYST")) {
        if(qmost_cpl_propertylist_get_int(plist,
                                          "ESO DRS MINYST",
                                          result) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read ESO DRS MINYST "
                                         "FITS header keyword");
        }
    }
    else if(cpl_propertylist_has(plist, "MINYST")) {
        if(qmost_cpl_propertylist_get_int(plist,
                                          "MINYST",
                                          result) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read MINYST "
                                         "FITS header keyword");
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read end y position of extracted spectra.
 *
 * The MAXYFN header used to indicate the maximum end y coordinate
 * in a trace file, or the ending y position of the extracted
 * spectra in an extracted spectrum file, is read.  It's not an error
 * if the header is missing, the return value is simply left
 * unmodified.  The user can set it to an otherwise invalid flag value
 * such as a negative number to check if the headers were missing.
 *
 * @param   plist             (Given)    FITS primary header to read
 *                                       from.
 * @param   result            (Returned) The end y position,
 *                                       numbering from 1.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND     If the required input FITS
 *                                       header keywords are missing.
 * @retval  CPL_ERROR_NULL_INPUT         If the input or output
 *                                       pointers are NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH      If the input FITS header
 *                                       keyword values had an
 *                                       incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>MAXYFN</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_maxyfn (
    cpl_propertylist *plist,
    int *result)
{
    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);

    if(cpl_propertylist_has(plist, "ESO DRS MAXYFN")) {
        if(qmost_cpl_propertylist_get_int(plist,
                                          "ESO DRS MAXYFN",
                                          result) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read ESO DRS MAXYFN "
                                         "FITS header keyword");
        }
    }
    else if(cpl_propertylist_has(plist, "MAXYFN")) {
        if(qmost_cpl_propertylist_get_int(plist,
                                          "MAXYFN",
                                          result) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read MAXYFN "
                                         "FITS header keyword");
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine the start MJD of observation.
 *
 * @param   plist             (Given)    FITS primary header to read
 *                                       from.
 * @param   mjd               (Returned) The start MJD.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE               If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND     If the required input FITS
 *                                       header keywords are missing.
 * @retval  CPL_ERROR_NULL_INPUT         If the input or output
 *                                       pointers are NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input FITS header keyword
 *                                    values had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>MJD-OBS</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_mjd_obs (
    cpl_propertylist *plist,
    double *mjd)
{
    cpl_error_code code;

    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mjd, CPL_ERROR_NULL_INPUT);

    if(cpl_propertylist_has(plist, "MJD-OBS")) {
        code = qmost_cpl_propertylist_get_double(plist, "MJD-OBS",
                                                 mjd);
    }
    else {
        /* Header has neither. */
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "MJD-OBS missing in FITS header");
    }

    return code;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine which spectrograph we're processing.
 *
 * @param   plist             (Given)    Primary header to read from.
 * @param   fib_root          (Returned) One of the QMOST_SPEC_*
 *                                       values specifying which
 *                                       spectrograph this is.  These
 *                                       can be compared to FIB_ROOT
 *                                       in the FIBINFO table.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If ESO INS PATH isn't one of the
 *                                    recognised values.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input FITS header keyword
 *                                    values had an incorrect data type.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_pfits_get_spectrograph (
    cpl_propertylist *plist,
    int *fib_root)
{
    const char *path;

    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fib_root, CPL_ERROR_NULL_INPUT);

    /* Determine arm from extension header */
    path = cpl_propertylist_get_string(plist, "ESO INS PATH");
    if(path == NULL) {
        return cpl_error_set_message(cpl_func,
                                     cpl_error_get_code(),
                                     "ESO INS PATH not found in header");
    }
    
    if(!strcmp(path, "HRS") || !strcmp(path, "CCDHR")) {
        *fib_root = QMOST_SPEC_HRS;
    }
    else if(!strcmp(path, "LRS-A") || !strcmp(path, "CCDLRA")) {
        *fib_root = QMOST_SPEC_LRS_A;
    }
    else if(!strcmp(path, "LRS-B") || !strcmp(path, "CCDLRB")) {
        *fib_root = QMOST_SPEC_LRS_B;
    }
    else {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "unexpected PATH in PHDU: %s",
                                     path);
    }

    return CPL_ERROR_NONE;
}

/**@}*/
