/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#define _POSIX_C_SOURCE   200809L   /* For mkstemp()       */
#define _DARWIN_C_SOURCE

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_lininfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_ccdproc_test  Unit test of qmost_ccdproc
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* Test image parameters */
#define GAIN      2
#define RON      10
#define BIASLEV 500
#define FLATLEV 10000
#define NXAMP   512
#define NYAMP   1024
#define NAMPSX    2
#define NAMPSY    2
#define NAMPS     (NAMPSX * NAMPSY)

/*----------------------------------------------------------------------------*/
/**
 * @brief   Utility routine to create 4-amplifier test raw image.
 *
 * @param   gaussian    (Given)     If true, use Gaussian read noise.
 *                                  Otherwise use uniform (bounded,
 *                                  can be useful for tests).
 * @param   flat        (Given)     If true, simulate a flat field.
 *                                  Otherwise, simulate a bias.
 * @param   result_img  (Returned)  The resulting raw image.
 * @param   result_hdr  (Returned)  The resulting FITS header.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void make_test_image (
    int gaussian,
    int flat,
    int binx,
    int biny,
    cpl_image **result_img,
    cpl_propertylist **result_hdr)
{
    int npr = 16;
    int nov = 32;

    int nxamp = NXAMP / binx;
    int nyamp = NYAMP / biny;

    int nxampraw = nxamp + npr + nov;
    int nyampraw = nyamp;

    int nx = NAMPSX*nxamp;
    int ny = NAMPSY*nyamp;

    int nxraw = NAMPSX*nxampraw;
    int nyraw = NAMPSY*nyampraw;

    cpl_image *img = NULL;

    int iamp;
    cpl_image *ampimg = NULL;
    cpl_image *flatimg = NULL;
    char *kw;

    cpl_propertylist *hdr = NULL;

    /* Create empty image and FITS header */
    img = cpl_image_new(nxraw, nyraw, CPL_TYPE_FLOAT);
    hdr = cpl_propertylist_new();

    cpl_propertylist_update_string(hdr, "EXTNAME", "TEST_DATA");
    cpl_propertylist_update_string(hdr, "ESO INS PATH", "RED");

    /* Binning */
    cpl_propertylist_update_int(hdr, "ESO DET BINX", binx);
    cpl_propertylist_update_int(hdr, "ESO DET BINY", biny);

    /* Physical active pixels */
    cpl_propertylist_update_int(hdr, "ESO DET CHIP NX", nx * binx);
    cpl_propertylist_update_int(hdr, "ESO DET CHIP NY", ny * biny);

    /* Populate amps with different bias levels */
    for(iamp = 0; iamp < NAMPS; iamp++) {
        ampimg = cpl_image_new(nxampraw, nyampraw, CPL_TYPE_FLOAT);

        if(gaussian) {
            qmost_test_fill_noise_rounded_gauss(ampimg, BIASLEV + 50*iamp, 8);
        }
        else {
            cpl_image_fill_noise_uniform(ampimg, -8, 8);
            cpl_image_add_scalar(ampimg, BIASLEV + 50*iamp);
        }

        if(flat) {
            flatimg = cpl_image_new(nxampraw, nyampraw, CPL_TYPE_FLOAT);
            qmost_test_fill_noise_rounded_gauss(flatimg,
                                                FLATLEV,
                                                sqrt(FLATLEV/GAIN));
            /* Blank prescan and overscan regions so these are
             * preserved at the bias level. */
            if(iamp % NAMPSY == 0) {  /* left */
                cpl_image_fill_window(flatimg,
                                      1, 1, npr, nyampraw,
                                      0);
                cpl_image_fill_window(flatimg,
                                      nxampraw-nov+1, 1, nxampraw, nyampraw,
                                      0);
            }
            else {  /* right */
                cpl_image_fill_window(flatimg,
                                      1, 1, nov, nyampraw,
                                      0);
                cpl_image_fill_window(flatimg,
                                      nxampraw-npr+1, 1, nxampraw, nyampraw,
                                      0);
            }
            cpl_image_add(ampimg, flatimg);
            cpl_image_delete(flatimg);
            flatimg = NULL;
        }

        cpl_image_copy(img, ampimg,
                       1 + nxampraw*(iamp % NAMPSY),
                       1 + nyampraw*(iamp / NAMPSY));

        kw = cpl_sprintf("ESO DET OUT%d INDEX", iamp+1);
        cpl_propertylist_update_int(hdr, kw, iamp+1);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d GAIN", iamp+1);
        cpl_propertylist_update_double(hdr, kw, GAIN);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d RON", iamp+1);
        cpl_propertylist_update_double(hdr, kw, RON);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d PRSCX", iamp+1);
        cpl_propertylist_update_int(hdr, kw, npr * binx);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d PRSCY", iamp+1);
        cpl_propertylist_update_int(hdr, kw, 0);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d OVSCX", iamp+1);
        cpl_propertylist_update_int(hdr, kw, nov * binx);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d OVSCY", iamp+1);
        cpl_propertylist_update_int(hdr, kw, 0);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d NX", iamp+1);
        cpl_propertylist_update_int(hdr, kw, nxamp * binx);
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d NY", iamp+1);
        cpl_propertylist_update_int(hdr, kw, nyamp * biny);
        cpl_free(kw);

        /* These are in physical pixels */
        kw = cpl_sprintf("ESO DET OUT%d X", iamp+1);
        cpl_propertylist_update_int(hdr, kw, 1 + nxamp * binx * (iamp % 2));
        cpl_free(kw);

        kw = cpl_sprintf("ESO DET OUT%d Y", iamp+1);
        cpl_propertylist_update_int(hdr, kw, 1 + nyamp * biny * (iamp / 2));
        cpl_free(kw);

        cpl_image_delete(ampimg);
    }

    *result_img = img;
    *result_hdr = hdr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Utility routine to check a reduced 4-amplifier test image.
 *
 * @param   imgout      (Returned)  The reduced image.
 * @param   varout      (Returned)  The corresponding variance array.
 * @param   hdr         (Returned)  The raw image FITS header.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void check_test_result (
    cpl_image *imgout,
    cpl_image *varout,
    cpl_propertylist *hdr)
{
    int iamp;
    int binx, biny;
    int nxamp, nyamp;

    int xamp, yamp;
    double amplevel;

    cpl_image *ampimg = NULL;

    /* Get amp size from header */
    binx = cpl_propertylist_get_int(hdr, "ESO DET BINX");
    biny = cpl_propertylist_get_int(hdr, "ESO DET BINY");

    nxamp = cpl_propertylist_get_int(hdr, "ESO DET OUT1 NX") / binx;
    nyamp = cpl_propertylist_get_int(hdr, "ESO DET OUT1 NY") / biny;

    /* Check size of output against expected size for trimmed image */
    cpl_test_eq(cpl_image_get_size_x(imgout), NAMPSX*nxamp);
    cpl_test_eq(cpl_image_get_size_y(imgout), NAMPSY*nyamp);

    /* Check results */
    for(iamp = 0; iamp < NAMPS; iamp++) {
        /* Extract this amp from processed image */
        xamp = 1 + nxamp*(iamp % NAMPSY);
        yamp = 1 + nyamp*(iamp / NAMPSY);

        ampimg = cpl_image_extract(imgout,
                                   xamp, yamp,
                                   xamp + nxamp - 1, yamp + nyamp - 1);
        cpl_test_nonnull(ampimg);

        /* Were bias offsets removed? */
        amplevel = cpl_image_get_median(ampimg);
        cpl_test_abs(amplevel, 0, 1);

        /* Is resulting image flat to noise level?  It might not be if
           we used the wrong bias level for part of it, e.g. if the
           geometry of the amps was off. */
        qmost_test_image_float_abs(ampimg, 0, NULL, 10);

        cpl_image_delete(ampimg);
    }

    if(varout != NULL) {
        /* We set the read noise to be quite high, above, so this
           should be roughly what we put in for the read noise
           variance. */
        qmost_test_image_float_abs(varout,
                                   GAIN*GAIN*RON*RON, NULL, 40);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_ccdproc
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_ccdproc(void)
{
    cpl_image *img = NULL;
    cpl_propertylist *hdr = NULL;

    cpl_frame *frame = NULL;
    char tmpfile[] = "test-qmost_ccdproc_XXXXXX";
    int fd;

    cpl_propertylist *qclist = NULL;

    cpl_image *imgout = NULL;
    cpl_image *varout = NULL;

    int binx, biny, nxamp, nyamp, nxcal, nycal;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_image *master_detflat_img = NULL;
    cpl_image *master_detflat_var = NULL;

    cpl_image *binned_img = NULL;
    cpl_propertylist *binned_hdr = NULL;

    int iamp;
    qmost_lininfo lins[4];
    cpl_size degree;

    cpl_table *linearity_tbl = NULL;
    cpl_propertylist *linearity_hdr = NULL;

    cpl_error_code code;

    /* Create temporary file in the wrong format */
    fd = mkstemp(tmpfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Wrap in frame */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, tmpfile);
    cpl_frame_set_tag(frame, "TEST");

    /* Empty QC list */
    qclist = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_ccdproc(NULL, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 0, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 0, NULL, 0, NULL, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 0, NULL, 0, &imgout, &varout, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Not a FITS file */
    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 0, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_BAD_FILE_FORMAT);

    /* Create PHDU, extension still isn't there */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_double(hdr, "EXPTIME", 5.0);
    cpl_propertylist_update_string(hdr, "ESO INS PATH", "HRS");
    cpl_propertylist_save(hdr, tmpfile, CPL_IO_CREATE);
    cpl_propertylist_delete(hdr);
    hdr = NULL;

    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 0, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Make a test raw image and save it to the temporary file */
    make_test_image(0, 0, 1, 1, &img, &hdr);
    cpl_image_save(img, tmpfile, CPL_TYPE_USHORT, hdr, CPL_IO_EXTEND);

    binx = cpl_propertylist_get_int(hdr, "ESO DET BINX");
    biny = cpl_propertylist_get_int(hdr, "ESO DET BINY");

    nxamp = cpl_propertylist_get_int(hdr, "ESO DET OUT1 NX") / binx;
    nyamp = cpl_propertylist_get_int(hdr, "ESO DET OUT1 NY") / biny;

    nxcal = NAMPSX * nxamp;
    nycal = NAMPSY * nyamp;

    /* oscor=0 */
    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 0, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* oscor=1 */
    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 1, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    check_test_result(imgout, varout, hdr);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* also flip */
    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         1, NULL, 1, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    cpl_test_eq(cpl_image_get_size_x(imgout), nycal);  /* make sure flipped */
    cpl_test_eq(cpl_image_get_size_y(imgout), nxcal);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* Smoke test running with the other calibrations */
    master_bpm = cpl_mask_new(NXAMP * NAMPSX,
                              NYAMP * NAMPSY);

    master_bias_img = cpl_image_new(nxcal, nycal, CPL_TYPE_FLOAT);
    master_bias_var = cpl_image_new(nxcal, nycal, CPL_TYPE_FLOAT);

    cpl_image_fill_window(master_bias_img,
                          1,
                          1,
                          nxcal,
                          nycal,
                          0);

    cpl_image_fill_window(master_bias_var,
                          1,
                          1,
                          nxcal,
                          nycal,
                          0);

    master_dark_img = cpl_image_new(nxcal, nycal, CPL_TYPE_FLOAT);
    master_dark_var = cpl_image_new(nxcal, nycal, CPL_TYPE_FLOAT);

    cpl_image_fill_window(master_dark_img,
                          1,
                          1,
                          nxcal,
                          nycal,
                          0);

    cpl_image_fill_window(master_dark_var,
                          1,
                          1,
                          nxcal,
                          nycal,
                          0);

    master_detflat_img = cpl_image_new(nxcal, nycal, CPL_TYPE_FLOAT);
    master_detflat_var = cpl_image_new(nxcal, nycal, CPL_TYPE_FLOAT);

    cpl_image_fill_window(master_detflat_img,
                          1,
                          1,
                          nxcal,
                          nycal,
                          1);

    cpl_image_fill_window(master_detflat_var,
                          1,
                          1,
                          nxcal,
                          nycal,
                          0);
    
    code = qmost_ccdproc(frame, 1,
                         master_bpm,
                         master_bias_img,
                         master_bias_var,
                         master_dark_img,
                         master_dark_var,
                         master_detflat_img,
                         master_detflat_var,
                         0, NULL, 1, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    check_test_result(imgout, varout, hdr);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* ...and again without error array out */
    code = qmost_ccdproc(frame, 1,
                         master_bpm,
                         master_bias_img,
                         master_bias_var,
                         master_dark_img,
                         master_dark_var,
                         master_detflat_img,
                         master_detflat_var,
                         0, NULL, 1, NULL, 0, &imgout, NULL, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);

    check_test_result(imgout, NULL, hdr);

    cpl_image_delete(imgout);
    imgout = NULL;

    /* ...and again with binning in a new extension. */
    make_test_image(0, 0, 2, 4, &binned_img, &binned_hdr);
    cpl_image_save(binned_img, tmpfile, CPL_TYPE_USHORT, binned_hdr, CPL_IO_EXTEND);

    code = qmost_ccdproc(frame, 2,
                         master_bpm,
                         master_bias_img,
                         master_bias_var,
                         master_dark_img,
                         master_dark_var,
                         master_detflat_img,
                         master_detflat_var,
                         0, NULL, 1, NULL, 0, &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    check_test_result(imgout, varout, binned_hdr);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    cpl_image_delete(binned_img);
    binned_img = NULL;

    cpl_propertylist_delete(binned_hdr);
    binned_hdr = NULL;

    cpl_mask_delete(master_bpm);
    cpl_image_delete(master_bias_img);
    cpl_image_delete(master_bias_var);
    cpl_image_delete(master_dark_img);
    cpl_image_delete(master_dark_var);
    cpl_image_delete(master_detflat_img);
    cpl_image_delete(master_detflat_var);

    /* oscor=1, and a linearity correction that doesn't do anything */
    code = qmost_lincrtab(&linearity_tbl, 10, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    linearity_hdr = cpl_propertylist_new();

    for(iamp = 0; iamp < NAMPS; iamp++) {
        code = qmost_lincrinfo(10, 3, &(lins[iamp]));
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        degree = 1;
        cpl_polynomial_set_coeff(lins[iamp].coefs, &degree, 1.0);
    }

    code = qmost_linwrite(linearity_tbl, linearity_hdr, lins, NAMPS);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < NAMPS; iamp++) {
        qmost_lindelinfo(&(lins[iamp]));
    }

    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 1, linearity_tbl, QMOST_LIN_BEFORE_BIAS,
                         &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    check_test_result(imgout, varout, hdr);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    code = qmost_ccdproc(frame, 1,
                         NULL,
                         NULL, NULL, NULL,
                         NULL, NULL, NULL,
                         0, NULL, 1, linearity_tbl, QMOST_LIN_AFTER_BIAS,
                         &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(imgout);
    cpl_test_nonnull(varout);

    check_test_result(imgout, varout, hdr);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    cpl_table_delete(linearity_tbl);
    linearity_tbl = NULL;

    cpl_propertylist_delete(linearity_hdr);
    linearity_hdr = NULL;

    /* Clean up */
    cpl_frame_delete(frame);

    cpl_image_delete(img);
    cpl_propertylist_delete(hdr);
    cpl_propertylist_delete(qclist);

    unlink(tmpfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_ccdproc_and_combine
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_ccdproc_and_combine(void)
{
    cpl_image *img = NULL;
    cpl_propertylist *hdr = NULL;

    cpl_frameset *frames = NULL;
    cpl_frameset *tmp_frames = NULL;
    int ifile, nfiles = 2;

    cpl_frame *frame = NULL;
    char tmpfile1[] = "test-qmost_ccdproc_XXXXXX";
    char tmpfile2[] = "test-qmost_ccdproc_XXXXXX";
    char tmpfile3[] = "test-qmost_ccdproc_XXXXXX";
    char *tmpfiles[3] = { tmpfile1, tmpfile2, tmpfile3 };
    int fd;

    cpl_propertylist *qclist = NULL;

    cpl_image *imgout = NULL;
    cpl_image *varout = NULL;

    cpl_error_code code;

    /* Create some input files */
    frames = cpl_frameset_new();

    for(ifile = 0; ifile < nfiles; ifile++) {
        /* Create temporary file */
        fd = mkstemp(tmpfiles[ifile]);
        qmost_test_ge(fd, 0);
        if(fd >= 0)
            close(fd);

        hdr = cpl_propertylist_new();
        cpl_propertylist_update_double(hdr, "EXPTIME", 5.0);
        cpl_propertylist_update_string(hdr, "ESO INS PATH", "HRS");
        cpl_propertylist_save(hdr, tmpfiles[ifile], CPL_IO_CREATE);
        cpl_propertylist_delete(hdr);
        hdr = NULL;

        make_test_image(0, 0, 1, 1, &img, &hdr);
        cpl_image_save(img, tmpfiles[ifile], CPL_TYPE_USHORT, hdr, CPL_IO_EXTEND);

        cpl_image_delete(img);
        img = NULL;

        cpl_propertylist_delete(hdr);
        hdr = NULL;

        /* Add to frameset */
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, tmpfiles[ifile]);
        cpl_frame_set_tag(frame, "TEST");

        cpl_frameset_insert(frames, frame);
        /* now owned by frameset */
    }

    /* Empty QC list */
    qclist = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_ccdproc_and_combine(NULL, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEANCALC, 0, 0, 0, 5.0,
                                     &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ccdproc_and_combine(frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEANCALC, 0, 0, 0, 5.0,
                                     NULL, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_ccdproc_and_combine(frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEANCALC, 0, 0, 0, 5.0,
                                     &imgout, &varout, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty frameset */
    tmp_frames = cpl_frameset_new();

    code = qmost_ccdproc_and_combine(tmp_frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEANCALC, 0, 0, 0, 5.0,
                                     &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* n=1 case */
    frame = cpl_frameset_get_position(frames, 0);
    cpl_frameset_insert(tmp_frames, cpl_frame_duplicate(frame));

    code = qmost_ccdproc_and_combine(tmp_frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEANCALC, 0, 0, 0, 5.0,
                                     &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* n=2 case */
    frame = cpl_frameset_get_position(frames, 1);
    cpl_frameset_insert(tmp_frames, cpl_frame_duplicate(frame));

    code = qmost_ccdproc_and_combine(tmp_frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEDIANCALC, 0, 0, 0, 5.0,
                                     &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    cpl_frameset_delete(tmp_frames);
    tmp_frames = NULL;

    /* n=3 case */
    code = qmost_ccdproc_and_combine(frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEDIANCALC, 0, 0, 0, 5.0,
                                     &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* scaletype=3 where EXPTIME is needed */
    code = qmost_ccdproc_and_combine(frames, 1,
                                     NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL, NULL,
                                     0, NULL, 0, NULL, 0,
                                     QMOST_MEDIANCALC, 3, 0, 0, 5.0,
                                     &imgout, &varout, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(imgout);
    imgout = NULL;

    cpl_image_delete(varout);
    varout = NULL;

    /* Clean up */
    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_frameset_delete(frames);
    frames = NULL;

    for(ifile = 0; ifile < nfiles; ifile++) {
        unlink(tmpfiles[ifile]);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_findgain
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_findgain(void)
{
    cpl_image *images[4] = { NULL, NULL, NULL, NULL };
    cpl_propertylist *ext_hdr[4] = { NULL, NULL, NULL, NULL };
    cpl_propertylist *qclist = NULL;
    cpl_mask *bpm = NULL;
    int iimg;
    cpl_error_code code;

    cpl_image *test_images[4];
    cpl_propertylist *test_hdr[4];
    cpl_mask *test_bpm = NULL;

    int iamp;
    char *qcname = NULL;
    float conad, readnoise;

    /* Make test images */
    for(iimg = 0; iimg < 4; iimg++) {
        make_test_image(1, iimg < 2, 1, 1, &(images[iimg]), &(ext_hdr[iimg]));
    }

    bpm = cpl_mask_new(NXAMP*2, NYAMP*2);
    qclist = cpl_propertylist_new();

    /* NULL input tests */
    memcpy(test_images, images, sizeof(test_images));
    test_images[0] = NULL;

    code = qmost_findgain(test_images, ext_hdr, ext_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[0] = NULL;

    code = qmost_findgain(images, test_hdr, ext_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_findgain(images, ext_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_findgain(images, ext_hdr, ext_hdr, 4, bpm, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid number of input images */
    code = qmost_findgain(images, ext_hdr, ext_hdr, 1, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Empty first header, can't determine number of amps */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[0] = cpl_propertylist_new();

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_delete(test_hdr[0]);

    /* Empty second header, can't determine number of amps */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[1] = cpl_propertylist_new();

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_delete(test_hdr[1]);

    /* Bad number of amps */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[1] = cpl_propertylist_duplicate(ext_hdr[1]);
    cpl_propertylist_update_int(test_hdr[1], "ESO DET OUTPUTS", 1);
    cpl_propertylist_update_int(test_hdr[1], "ESO DET CHIPS", 3);

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_propertylist_delete(test_hdr[1]);

    /* Invalid binning */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[0] = cpl_propertylist_duplicate(ext_hdr[0]);
    cpl_propertylist_erase(test_hdr[0], "ESO DET BINY");
    cpl_propertylist_update_string(test_hdr[0], "ESO DET BINY", "TEST");

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(test_hdr[0]);

    /* Empty second header, can't determine number of amps */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[1] = cpl_propertylist_duplicate(ext_hdr[1]);
    cpl_propertylist_erase(test_hdr[1], "ESO DET BINX");
    cpl_propertylist_update_string(test_hdr[1], "ESO DET BINX", "TEST");

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(test_hdr[1]);

    /* Mismatched binning */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[1] = cpl_propertylist_duplicate(ext_hdr[1]);
    cpl_propertylist_update_int(test_hdr[1], "ESO DET BINX", 2);

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_propertylist_delete(test_hdr[1]);

    /* Missing detector geometry headers */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[0] = cpl_propertylist_duplicate(ext_hdr[0]);
    cpl_propertylist_erase(test_hdr[0], "ESO DET OUT1 NX");

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_delete(test_hdr[0]);

    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[1] = cpl_propertylist_duplicate(ext_hdr[1]);
    cpl_propertylist_erase(test_hdr[1], "ESO DET OUT3 NY");

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_delete(test_hdr[1]);

    /* Incorrect / mismatching NX */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[1] = cpl_propertylist_duplicate(ext_hdr[1]);
    cpl_propertylist_update_int(test_hdr[1], "ESO DET OUT1 NX", 1);

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_propertylist_delete(test_hdr[1]);

    /* Out of bounds NX */
    memcpy(test_hdr, ext_hdr, sizeof(test_hdr));
    test_hdr[0] = cpl_propertylist_duplicate(ext_hdr[0]);
    cpl_propertylist_update_int(test_hdr[0], "ESO DET OUT1 NX", 2048);

    code = qmost_findgain(images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_propertylist_delete(test_hdr[0]);

    /* One of the images too small to trigger the other extract fail */
    memcpy(test_images, images, sizeof(test_images));
    test_images[1] = cpl_image_extract(images[1], 1, 1, 2, 2);
    
    code = qmost_findgain(test_images, ext_hdr, ext_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_image_delete(test_images[1]);

    /* BPM incorrect size */
    test_bpm = cpl_mask_new(NXAMP/2, NYAMP/2);

    code = qmost_findgain(images, ext_hdr, ext_hdr, 4, test_bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_mask_delete(test_bpm);

    /* Valid inputs, this should work */
    code = qmost_findgain(images, ext_hdr, ext_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* We put in CONAD = 1 and noise of 8 ADU */
    for(iamp = 0; iamp < 4; iamp++) {
        qcname = cpl_sprintf("ESO QC CONAD AMP%d", iamp+1);
        conad = cpl_propertylist_get_float(qclist, qcname);
        cpl_free(qcname);
        qcname = NULL;

        qcname = cpl_sprintf("ESO QC READNOISE AMP%d", iamp+1);
        readnoise = cpl_propertylist_get_float(qclist, qcname);
        cpl_free(qcname);
        qcname = NULL;

        cpl_test_abs(conad, GAIN, 0.1);
        cpl_test_abs(readnoise/conad, 8.0, 0.5);
    }

    cpl_propertylist_delete(qclist);

    /* Overscan rather than bias frames, should get same answer */
    qclist = cpl_propertylist_new();

    code = qmost_findgain(images, ext_hdr, ext_hdr, 2, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < 4; iamp++) {
        qcname = cpl_sprintf("ESO QC CONAD AMP%d", iamp+1);
        conad = cpl_propertylist_get_float(qclist, qcname);
        cpl_free(qcname);
        qcname = NULL;

        qcname = cpl_sprintf("ESO QC READNOISE AMP%d", iamp+1);
        readnoise = cpl_propertylist_get_float(qclist, qcname);
        cpl_free(qcname);
        qcname = NULL;

        cpl_test_abs(conad, GAIN, 0.1);
        cpl_test_abs(readnoise/conad, 8.0, 0.5);
    }

    cpl_propertylist_delete(qclist);

    /* Putting in the bias twice should trigger the noisvar == 0 case */
    test_images[0] = images[2];
    test_images[1] = images[3];
    test_images[2] = images[2];
    test_images[3] = images[3];

    test_hdr[0] = ext_hdr[2];
    test_hdr[1] = ext_hdr[3];
    test_hdr[2] = ext_hdr[2];
    test_hdr[3] = ext_hdr[3];

    qclist = cpl_propertylist_new();

    code = qmost_findgain(test_images, test_hdr, test_hdr, 4, bpm, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Should be no answers */
    for(iamp = 0; iamp < 4; iamp++) {
        qcname = cpl_sprintf("ESO QC CONAD AMP%d", iamp+1);
        cpl_test_zero(cpl_propertylist_has(qclist, qcname));
        cpl_free(qcname);
        qcname = NULL;

        qcname = cpl_sprintf("ESO QC READNOISE AMP%d", iamp+1);
        cpl_test_zero(cpl_propertylist_has(qclist, qcname));
        cpl_free(qcname);
        qcname = NULL;
    }

    cpl_propertylist_delete(qclist);

    /* No BPM should also work */
    qclist = cpl_propertylist_new();

    code = qmost_findgain(test_images, test_hdr, test_hdr, 4, NULL, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_propertylist_delete(qclist);

    /* Clean up */
    for(iimg = 0; iimg < 4; iimg++) {
        cpl_image_delete(images[iimg]);
        images[iimg] = NULL;

        cpl_propertylist_delete(ext_hdr[iimg]);
        ext_hdr[iimg] = NULL;
    }

    cpl_mask_delete(bpm);
    bpm = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_ccdproc module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_ccdproc();
    test_qmost_ccdproc_and_combine();
    test_qmost_findgain();

    return cpl_test_end(0);
}

/**@}*/
