/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_pfits.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_pfits_test  Unit test of qmost_pfits
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_arm
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_arm(void)
{
    cpl_propertylist *hdr = NULL;
    cpl_error_code code;
    int result;

    struct {
        char *input;
        int output;
    } answers[] = {
        { "rEd", QMOST_ARM_RED },
        { "gReen", QMOST_ARM_GREEN },
        { "blUe", QMOST_ARM_BLUE }
    };

    struct {
        char *input;
        int output;
    } extnames[] = {
        { "RED_DATA", QMOST_ARM_RED },
        { "GREEN_DATA", QMOST_ARM_GREEN },
        { "BLUE_DATA", QMOST_ARM_BLUE }
    };

    int ians, nans;

    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_arm(NULL, &result);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_arm(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty header */
    code = qmost_pfits_get_arm(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add a keyword, not the right one */
    cpl_propertylist_update_string(hdr, "FOO", "bar");

    code = qmost_pfits_get_arm(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add the keyword, but wrong type */
    cpl_propertylist_update_int(hdr, "ESO INS PATH", 1);

    code = qmost_pfits_get_arm(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add the keyword, correct type but invalid value */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_string(hdr, "ESO INS PATH", "test");

    code = qmost_pfits_get_arm(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Check the correct values return the correct results */
    nans = sizeof(answers) / sizeof(answers[0]);

    for(ians = 0; ians < nans; ians++) {
        cpl_propertylist_update_string(hdr, "ESO INS PATH",
                                       answers[ians].input);

        result = 42;

        code = qmost_pfits_get_arm(hdr, &result);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_test_eq(result, answers[ians].output);
    }

    cpl_propertylist_delete(hdr);

    /* Same tests for EXTNAME: incorrect type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(hdr, "EXTNAME", 1);

    code = qmost_pfits_get_arm(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Correct type but invalid value */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_string(hdr, "EXTNAME", "test");

    code = qmost_pfits_get_arm(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Check the correct values return the correct results */
    nans = sizeof(extnames) / sizeof(extnames[0]);

    for(ians = 0; ians < nans; ians++) {
        cpl_propertylist_update_string(hdr, "EXTNAME",
                                       extnames[ians].input);

        result = 42;

        code = qmost_pfits_get_arm(hdr, &result);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_test_eq(result, extnames[ians].output);
    }

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_detlive
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_detlive(void)
{
    cpl_propertylist *hdr = NULL;
    int value;
    cpl_error_code code;

    /* Empty header */
    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_detlive(NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_detlive(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With an empty header, should return 1 if not present */
    code = qmost_pfits_get_detlive(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 1);

    /* Add keyword, but wrong type */
    cpl_propertylist_update_string(hdr, "ESO DET CHIP LIVE", "invalid");

    code = qmost_pfits_get_detlive(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add keyword, correct type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_bool(hdr, "ESO DET CHIP LIVE", 0);

    code = qmost_pfits_get_detlive(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 0);

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_dummy
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_dummy(void)
{
    cpl_propertylist *hdr = NULL;
    int value;
    cpl_error_code code;

    /* Empty header */
    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_dummy(NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_dummy(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With an empty header, should return 0 if not present */
    code = qmost_pfits_get_dummy(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 0);

    /* Add keyword, but wrong type */
    cpl_propertylist_update_string(hdr, "ESO DRS DUMMY", "invalid");

    code = qmost_pfits_get_dummy(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add keyword, correct type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_bool(hdr, "ESO DRS DUMMY", 1);

    code = qmost_pfits_get_dummy(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 1);

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_exptime
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_exptime(void)
{
    cpl_propertylist *hdr = NULL;
    double value;
    cpl_error_code code;

    /* Empty header */
    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_exptime(NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_exptime(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With an empty header */
    code = qmost_pfits_get_exptime(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add a keyword, not the right one */
    cpl_propertylist_update_string(hdr, "ESO FOO BAR", "baz");

    code = qmost_pfits_get_exptime(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add EXPTIME keyword, but wrong type */
    cpl_propertylist_update_string(hdr, "EXPTIME", "invalid");

    code = qmost_pfits_get_exptime(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add ESO EXPTIME keyword, but wrong type */
    hdr = cpl_propertylist_new();

    cpl_propertylist_update_string(hdr, "ESO EXPTIME", "invalid");

    code = qmost_pfits_get_exptime(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add ESO EXPTIME, correct type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_double(hdr, "ESO EXPTIME", 42);

    code = qmost_pfits_get_exptime(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 42, DBL_EPSILON);

    /* Add EXPTIME, which should get read instead.  Call should
     * succeed even if it's integer. */
    cpl_propertylist_update_int(hdr, "EXPTIME", 5);

    code = qmost_pfits_get_exptime(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 5, DBL_EPSILON);

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_extname
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_extname(void)
{
    struct {
        int input;
        const char *output;
    } answers[] = {
        { QMOST_ARM_RED, "RED_DATA" },
        { QMOST_ARM_GREEN, "GREEN_DATA" },
        { QMOST_ARM_BLUE, "BLUE_DATA" }
    };
    int ians, nans;
    const char *result;

    /* Invalid inputs */
    result = qmost_pfits_get_extname(0);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    result = qmost_pfits_get_extname(QMOST_NUM_ARMS + 1);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    /* Check the correct values return the correct results */
    nans = sizeof(answers) / sizeof(answers[0]);

    for(ians = 0; ians < nans; ians++) {
        result = qmost_pfits_get_extname(answers[ians].input);
        cpl_test_eq_string(result, answers[ians].output);
        cpl_test_error(CPL_ERROR_NONE);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_minyst
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_minyst(void)
{
    cpl_propertylist *hdr = NULL;
    int value;
    cpl_error_code code;

    /* Empty header */
    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_minyst(NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_minyst(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With an empty header value should be unchanged */
    value = -42;

    code = qmost_pfits_get_minyst(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(value, -42);

    /* Add MINYST keyword, but wrong type */
    cpl_propertylist_update_string(hdr, "MINYST", "invalid");

    code = qmost_pfits_get_minyst(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add ESO DRS MINYST keyword, but wrong type */
    hdr = cpl_propertylist_new();

    cpl_propertylist_update_string(hdr, "ESO DRS MINYST", "invalid");

    code = qmost_pfits_get_minyst(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add MINYST, correct type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(hdr, "MINYST", 42);

    code = qmost_pfits_get_minyst(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 42);

    /* Add ESO DRS MINYST, which should get read instead */
    cpl_propertylist_update_int(hdr, "ESO DRS MINYST", 41);

    code = qmost_pfits_get_minyst(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 41, DBL_EPSILON);

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_maxyfn
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_maxyfn(void)
{
    cpl_propertylist *hdr = NULL;
    int value;
    cpl_error_code code;

    /* Empty header */
    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_maxyfn(NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_maxyfn(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With an empty header value should be unchanged */
    value = -42;

    code = qmost_pfits_get_maxyfn(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(value, -42);

    /* Add MAXYFN keyword, but wrong type */
    cpl_propertylist_update_string(hdr, "MAXYFN", "invalid");

    code = qmost_pfits_get_maxyfn(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add ESO DRS MAXYFN keyword, but wrong type */
    hdr = cpl_propertylist_new();

    cpl_propertylist_update_string(hdr, "ESO DRS MAXYFN", "invalid");

    code = qmost_pfits_get_maxyfn(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add MAXYFN, correct type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(hdr, "MAXYFN", 42);

    code = qmost_pfits_get_maxyfn(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(value, 42);

    /* Add ESO DRS MAXYFN, which should get read instead */
    cpl_propertylist_update_int(hdr, "ESO DRS MAXYFN", 41);

    code = qmost_pfits_get_maxyfn(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 41, DBL_EPSILON);

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_mjd_obs
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_mjd_obs(void)
{
    cpl_propertylist *hdr = NULL;
    double value;
    cpl_error_code code;

    /* Empty header */
    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_mjd_obs(NULL, &value);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_mjd_obs(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With an empty header */
    code = qmost_pfits_get_mjd_obs(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add a keyword, not the right one */
    cpl_propertylist_update_string(hdr, "ESO FOO BAR", "baz");

    code = qmost_pfits_get_mjd_obs(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add MJD-OBS keyword, but wrong type */
    cpl_propertylist_update_string(hdr, "MJD-OBS", "invalid");

    code = qmost_pfits_get_mjd_obs(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add ESO MJD-OBS, correct type */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_double(hdr, "MJD-OBS", 59000);

    code = qmost_pfits_get_mjd_obs(hdr, &value);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, 59000, DBL_EPSILON);

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_pfits_get_spectrograph
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_pfits_get_spectrograph(void)
{
    cpl_propertylist *hdr = NULL;
    cpl_error_code code;
    int result;

    struct {
        char *input;
        int output;
    } answers[] = {
        { "HRS", QMOST_SPEC_HRS },
        { "LRS-A", QMOST_SPEC_LRS_A },
        { "LRS-B", QMOST_SPEC_LRS_B }
    };
    int ians, nans;

    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_pfits_get_spectrograph(NULL, &result);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_pfits_get_spectrograph(hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty header */
    code = qmost_pfits_get_spectrograph(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add a keyword, not the right one */
    cpl_propertylist_update_string(hdr, "FOO", "bar");

    code = qmost_pfits_get_spectrograph(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add the keyword, but wrong type */
    cpl_propertylist_update_int(hdr, "ESO INS PATH", 1);

    code = qmost_pfits_get_spectrograph(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_propertylist_delete(hdr);

    /* Add the keyword, correct type but invalid value */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_string(hdr, "ESO INS PATH", "test");

    code = qmost_pfits_get_spectrograph(hdr, &result);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Check the correct values return the correct results */
    nans = sizeof(answers) / sizeof(answers[0]);

    for(ians = 0; ians < nans; ians++) {
        cpl_propertylist_update_string(hdr, "ESO INS PATH",
                                       answers[ians].input);

        result = 42;

        code = qmost_pfits_get_spectrograph(hdr, &result);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_test_eq(result, answers[ians].output);
    }

    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_pfits module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_pfits_get_arm();
    test_qmost_pfits_get_detlive();
    test_qmost_pfits_get_dummy();
    test_qmost_pfits_get_exptime();
    test_qmost_pfits_get_extname();
    test_qmost_pfits_get_minyst();
    test_qmost_pfits_get_maxyfn();
    test_qmost_pfits_get_mjd_obs();
    test_qmost_pfits_get_spectrograph();

    return cpl_test_end(0);
}

/**@}*/
