/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_sort.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_sort_test  Unit test of qmost_sort
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_f
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_f(void)
{
    float data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    float sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    float test_data[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    qmost_sort_f(test_data, 0);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    qmost_sort_f(test_data, 1);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    qmost_sort_f(test_data, n);

    qmost_test_arrays_float_abs(test_data, sorted_data, n, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_ff
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_ff(void)
{
    float data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    float sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    float test_data[9];

    float floats[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    float sorted_floats[9] = { 11, 12, 13, 14, 15, 16, 17, 18, 19 };
    float test_floats[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats, floats, sizeof(test_floats));
    qmost_sort_ff(test_data, test_floats, 0);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats, floats, n, FLT_EPSILON);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats, floats, sizeof(test_floats));
    qmost_sort_ff(test_data, test_floats, 1);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats, floats, n, FLT_EPSILON);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats, floats, sizeof(test_floats));
    qmost_sort_ff(test_data, test_floats, n);

    qmost_test_arrays_float_abs(test_data, sorted_data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats, sorted_floats, n, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_fff
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_fff(void)
{
    float data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    float sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    float test_data[9];

    float floats1[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    float sorted_floats1[9] = { 11, 12, 13, 14, 15, 16, 17, 18, 19 };
    float test_floats1[9];

    float floats2[9] = { 22, 29, 27, 24, 25, 21, 23, 28, 26 };
    float sorted_floats2[9] = { 21, 22, 23, 24, 25, 26, 27, 28, 29 };
    float test_floats2[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats1, floats1, sizeof(test_floats1));
    memcpy(test_floats2, floats2, sizeof(test_floats2));
    qmost_sort_fff(test_data, test_floats1, test_floats2, 0);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats1, floats1, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats2, floats2, n, FLT_EPSILON);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats1, floats1, sizeof(test_floats1));
    memcpy(test_floats2, floats2, sizeof(test_floats2));
    qmost_sort_fff(test_data, test_floats1, test_floats2, 1);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats1, floats1, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats2, floats2, n, FLT_EPSILON);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats1, floats1, sizeof(test_floats1));
    memcpy(test_floats2, floats2, sizeof(test_floats2));
    qmost_sort_fff(test_data, test_floats1, test_floats2, n);

    qmost_test_arrays_float_abs(test_data, sorted_data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats1, sorted_floats1, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats2, sorted_floats2, n, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_ffff
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_ffff(void)
{
    float data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    float sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    float test_data[9];

    float floats1[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    float sorted_floats1[9] = { 11, 12, 13, 14, 15, 16, 17, 18, 19 };
    float test_floats1[9];

    float floats2[9] = { 22, 29, 27, 24, 25, 21, 23, 28, 26 };
    float sorted_floats2[9] = { 21, 22, 23, 24, 25, 26, 27, 28, 29 };
    float test_floats2[9];

    float floats3[9] = { 32, 39, 37, 34, 35, 31, 33, 38, 36 };
    float sorted_floats3[9] = { 31, 32, 33, 34, 35, 36, 37, 38, 39 };
    float test_floats3[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats1, floats1, sizeof(test_floats1));
    memcpy(test_floats2, floats2, sizeof(test_floats2));
    memcpy(test_floats3, floats3, sizeof(test_floats3));
    qmost_sort_ffff(test_data, test_floats1, test_floats2, test_floats3, 0);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats1, floats1, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats2, floats2, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats3, floats3, n, FLT_EPSILON);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats1, floats1, sizeof(test_floats1));
    memcpy(test_floats2, floats2, sizeof(test_floats2));
    memcpy(test_floats3, floats3, sizeof(test_floats3));
    qmost_sort_ffff(test_data, test_floats1, test_floats2, test_floats3, 1);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats1, floats1, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats2, floats2, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats3, floats3, n, FLT_EPSILON);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_floats1, floats1, sizeof(test_floats1));
    memcpy(test_floats2, floats2, sizeof(test_floats2));
    memcpy(test_floats3, floats3, sizeof(test_floats3));
    qmost_sort_ffff(test_data, test_floats1, test_floats2, test_floats3, n);

    qmost_test_arrays_float_abs(test_data, sorted_data, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats1, sorted_floats1, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats2, sorted_floats2, n, FLT_EPSILON);
    qmost_test_arrays_float_abs(test_floats3, sorted_floats3, n, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_fi
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_fi(void)
{
    float data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    float sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    float test_data[9];

    int integers[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    int sorted_integers[9] = { 11, 12, 13, 14, 15, 16, 17, 18, 19 };
    int test_integers[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_fi(test_data, test_integers, 0);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_int_abs(test_integers, integers, n, 0);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_fi(test_data, test_integers, 1);

    qmost_test_arrays_float_abs(test_data, data, n, FLT_EPSILON);
    qmost_test_arrays_int_abs(test_integers, integers, n, 0);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_fi(test_data, test_integers, n);

    qmost_test_arrays_float_abs(test_data, sorted_data, n, FLT_EPSILON);
    qmost_test_arrays_int_abs(test_integers, sorted_integers, n, 0);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_dd
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_dd(void)
{
    double data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    double sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    double test_data[9];

    double doubles[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    double sorted_doubles[9] = { 11, 12, 13, 14, 15, 16, 17, 18, 19 };
    double test_doubles[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_doubles, doubles, sizeof(test_doubles));
    qmost_sort_dd(test_data, test_doubles, 0);

    qmost_test_arrays_double_abs(test_data, data, n, DBL_EPSILON);
    qmost_test_arrays_double_abs(test_doubles, doubles, n, DBL_EPSILON);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_doubles, doubles, sizeof(test_doubles));
    qmost_sort_dd(test_data, test_doubles, 1);

    qmost_test_arrays_double_abs(test_data, data, n, DBL_EPSILON);
    qmost_test_arrays_double_abs(test_doubles, doubles, n, DBL_EPSILON);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_doubles, doubles, sizeof(test_doubles));
    qmost_sort_dd(test_data, test_doubles, n);

    qmost_test_arrays_double_abs(test_data, sorted_data, n, DBL_EPSILON);
    qmost_test_arrays_double_abs(test_doubles, sorted_doubles, n, DBL_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_di
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_di(void)
{
    double data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    double sorted_data[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    double test_data[9];

    int integers[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    int sorted_integers[9] = { 11, 12, 13, 14, 15, 16, 17, 18, 19 };
    int test_integers[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_di(test_data, test_integers, 0);

    qmost_test_arrays_double_abs(test_data, data, n, DBL_EPSILON);
    qmost_test_arrays_int_abs(test_integers, integers, n, 0);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_di(test_data, test_integers, 1);

    qmost_test_arrays_double_abs(test_data, data, n, DBL_EPSILON);
    qmost_test_arrays_int_abs(test_integers, integers, n, 0);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_di(test_data, test_integers, n);

    qmost_test_arrays_double_abs(test_data, sorted_data, n, DBL_EPSILON);
    qmost_test_arrays_int_abs(test_integers, sorted_integers, n, 0);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sort_rev_di
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sort_rev_di(void)
{
    double data[9] = { 2, 9, 7, 4, 5, 1, 3, 8, 6 };
    double sorted_data[9] = { 9, 8, 7, 6, 5, 4, 3, 2, 1 };
    double test_data[9];

    int integers[9] = { 12, 19, 17, 14, 15, 11, 13, 18, 16 };
    int sorted_integers[9] = { 19, 18, 17, 16, 15, 14, 13, 12, 11 };
    int test_integers[9];

    int n;

    n = sizeof(data) / sizeof(data[0]);

    /* Sort nothing, should do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_rev_di(test_data, test_integers, 0);

    qmost_test_arrays_double_abs(test_data, data, n, DBL_EPSILON);
    qmost_test_arrays_int_abs(test_integers, integers, n, 0);

    /* Sort 1 element, should also do nothing */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_rev_di(test_data, test_integers, 1);

    qmost_test_arrays_double_abs(test_data, data, n, DBL_EPSILON);
    qmost_test_arrays_int_abs(test_integers, integers, n, 0);

    /* Sort n elements and check against pre-sorted list */
    memcpy(test_data, data, sizeof(test_data));
    memcpy(test_integers, integers, sizeof(test_integers));
    qmost_sort_rev_di(test_data, test_integers, n);

    qmost_test_arrays_double_abs(test_data, sorted_data, n, DBL_EPSILON);
    qmost_test_arrays_int_abs(test_integers, sorted_integers, n, 0);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_sort module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_sort_f();
    test_qmost_sort_ff();
    test_qmost_sort_fff();
    test_qmost_sort_ffff();
    test_qmost_sort_fi();
    test_qmost_sort_dd();
    test_qmost_sort_di();
    test_qmost_sort_rev_di();

    return cpl_test_end(0);
}

/**@}*/
