/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_imcombine_lite.h"
#include "qmost_linear.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_linearity_analyse"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_linearity_analyse_description[] =
    "Analyse a set of detector flats of a constant light source taken\n"
    "with an exposure time ramp to measure detector linearity.  Emit\n"
    "a linearity correction table and a detector bad pixel mask.  The\n"
    "linearity ramp exposures can optionally be interleaved with lamp\n"
    "brightness monitoring exposures of fixed exposure time to measure\n"
    "and correct for any variations in lamp brightness.\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                  Req/Opt? Tag\n"
    "---------------------------- -------- --------------------\n"
    "Linearity ramp detector flat Required " QMOST_RAW_DETECTOR_FLAT_LIN "\n"
    "Monitor detector flat        Optional " QMOST_RAW_DETECTOR_FLAT_MON "\n"
    "Master bias frame            Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame            Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master bad pixel mask        Optional " QMOST_CALIB_MASTER_BPM "\n"
    "\n"
    "Outputs:\n\n"
    "Description                  Tag\n"
    "---------------------------- -----------------\n"
    "Linearity table              " QMOST_PRO_LINEARITY "\n"
    "Bad pixel mask               " QMOST_PRO_BPM "\n\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the linearity table\n"
    "for HRS would be QMOST_" QMOST_PRO_LINEARITY "_HRS.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_linearity_analyse,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Measure detector linearity and make bad pixel mask.",
                  qmost_linearity_analyse_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_linearity_analyse    qmost_linearity_analyse
 *
 * @brief  Measure detector linearity and make bad pixel mask.
 *
 * @par Name:
 *   qmost_linearity_analyse
 * @par Purpose:
 *   Analyse a set of detector flats of a constant light source taken
 *   with an exposure time ramp to measure detector linearity.  Emit
 *   a linearity correction table and a detector bad pixel mask.  The
 *   linearity ramp exposures can optionally be interleaved with lamp
 *   brightness monitoring exposures of fixed exposure time to measure
 *   and correct for any variations in lamp brightness.
 * @par Type:
 *   Detector calibration
 * @par Parameters:
 *   - @b linearity.docorr (bool, default true)
 *     If set, then the monitor sequence will be used to correct for
 *     lamp brightness variations.  If unset, the monitor sequence
 *     analysis results are reported in the output file but not used.
 *   - @b linearity.nord (int, default 4)
 *     The order of polynomial to fit to the linearity curve.
 *   - @b linearity.niter (int, default 3)
 *     The number of rejection iterations to use in the linearity
 *     fit.
 *   - @b linearity.clipthr (float, default 5.0)
 *     The number of sigma for outlier rejection in the linearity
 *     fit.
 *   - @b linearity.underexp (float, default 1000)
 *     The flux limit in ADU to define an underexposed image.
 *   - @b linearity.overexp (float, default 60000)
 *     The flux limit in ADU to define an overexposed image.
 *   - @b linearity.mingood (int, default 5)
 *     The minimum number of images required to run the algorithm.  If
 *     too few images are left after removing under- or over-exposed
 *     images, an exception is raised.
 *   - @b bpm.lthr (float, default 8.0)
 *     Lower threshold for flagging a bad pixel in terms of the
 *     background sigma of the ratio image.
 *   - @b bpm.hthr (float, default 8.0)
 *     Upper threshold for flagging a bad pixel in terms of the
 *     background sigma of the ratio image.
 *   - @b bpm.badfrac (float, default 0.25)
 *     The fraction of exposures on which a pixel is discordant in
 *     order for it to be defined as bad (between 0.0 and 1.0)
 *   - @b bpm.nmax (int, default 20)
 *     The maximum number of flats to use in BPM analysis.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b DETECTOR_FLAT_LIN: Raw detector flat frames taken in a
 *         linearity ramp.
 *    - @b DETECTOR_FLAT_MON (optional): Raw detector flat frames with
 *         fixed exposure time to monitor lamp brightness.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b LINEARITY: The linearity correction table.
 *     - @b BPM: The new bad pixel mask.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     linearity table for HRS would be @c QMOST_LINEARITY_HRS.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>DETFLAT MIN</b> (ADU): The minimum level of the flat fields
 *     that were combined to form the linearity table.
 *   - <b>DETFLAT MINFILE</b>: The index (numbering from 1) of the
 *     flat with the lowest count level in the linearity sequence.
 *   - <b>DETFLAT MAX</b> (ADU): The maximum level of the flat fields
 *     that were combined to form the linearity table.
 *   - <b>DETFLAT MAXFILE</b>: The index (numbering from 1) of the
 *     flat with the highest count level in the linearity sequence.
 *   - <b>LIN NL1K</b> (percent): The estimated non-linearity at 1000
 *     ADU averaged over all of the amplifiers.
 *   - <b>LIN NL5K</b> (percent): The estimated non-linearity at 5000
 *     ADU averaged over all of the amplifiers.
 *   - <b>LIN NL10K</b> (percent): The estimated non-linearity at
 *     10000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL20K</b> (percent): The estimated non-linearity at
 *     20000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL30K</b> (percent): The estimated non-linearity at
 *     30000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL40K</b> (percent): The estimated non-linearity at
 *     40000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL50K</b> (percent): The estimated non-linearity at
 *     50000 ADU averaged over all of the amplifiers.
 *   - <b>LIN RMS MEAN</b> (ADU): The RMS of the linearity fit
 *     averaged over all of the amplifiers.
 *   - <b>LIN RMS MIN</b> (ADU): The RMS of the linearity fit for the
 *     amplifier with the best linearity fit.
 *   - <b>LIN RMS MINAMP</b>: The amplifier with the best linearity
 *     fit.
 *   - <b>LIN RMS MAX</b> (ADU): The RMS of the linearity fit for the
 *     amplifier with the worst linearity fit.
 *   - <b>LIN RMS MAXAMP</b>: The amplifier with the worst linearity
 *     fit.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean background level of the
 *     frames that were combined for the bad pixel mask.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined for the bad pixel mask, as a
 *     measure of how consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined for the bad pixel mask.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined for the bad pixel mask.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined for the bad pixel
 *     mask.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined for the bad pixel mask, after rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine for the bad pixel mask,
 *     before rejection of any bad frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination for the bad pixel mask.
 *   - <b>BPM BADFRAC</b>: The fraction of pixels on the detector
 *     flagged as bad.
 *   - <b>BPM NBAD</b>: The number of pixels flagged as bad in the bad
 *     pixel mask.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No linearity detector flat frames in the input frameset.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - Too few images within the under and over exposure limits.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 * @par Non-Fatal Error Conditions:
 *   - Missing master bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 *
 *   "MASTER_BPM\nMASTER_BIAS\nMASTER_DARK" -> "qmost_ccdproc (mon)" [style="dashed"];
 *   "MASTER_BPM\nMASTER_BIAS\nMASTER_DARK" -> qmost_ccdproc [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     DETECTOR_FLAT_MON -> "qmost_ccdproc (mon)" [style="dashed"];
 *     "qmost_ccdproc (mon)" -> "qmost_flatstats_compute (mon)" [style="dashed"];
 *     "qmost_flatstats_compute (mon)" -> "qmost_flatstats_check (mon)" [style="dashed"];
 *
 *     DETECTOR_FLAT_LIN -> qmost_ccdproc;
 *     qmost_ccdproc -> "qmost_flatstats_compute";
 *     "qmost_flatstats_compute" -> "qmost_flatstats_check";
 *
 *     "qmost_flatstats_check (mon)" -> qmost_linfit [style="dashed"];
 *     "qmost_flatstats_check" -> qmost_linfit;
 *
 *     qmost_linfit -> LINEARITY;
 *
 *     qmost_ccdproc -> qmost_gen_bpm;
 *
 *     qmost_gen_bpm -> BPM;
 *   }
 *
 *   qmost_ccdproc -> QC1;
 *   qmost_linfit -> QC1;
 *   qmost_gen_bpm -> QC1;
 *
 *   DETECTOR_FLAT_MON [shape="box" fillcolor="#eeeeee"]
 *   DETECTOR_FLAT_LIN [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BPM\nMASTER_BIAS\nMASTER_DARK" [shape="box" fillcolor="#fff5ce"]
 *   LINEARITY [shape="box" fillcolor="#fff5ce"]
 *   BPM [shape="box" fillcolor="#fff5ce"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_linearity_analyse(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    const cpl_frame *master_bias_frame = NULL;
    const cpl_frame *master_dark_frame = NULL;
    const cpl_frame *master_bpm_frame = NULL;

    int spec;
    const cpl_frame *linearity_frame = NULL;
    const char *linearity_filename = NULL;
    const cpl_frame *bpm_frame = NULL;
    const char *bpm_filename = NULL;

    cpl_frameset *detflats_lin_raw = NULL;
    cpl_propertylist **detflats_lin_hdr = NULL;
    cpl_frameset *detflats_mon_raw = NULL;
    cpl_propertylist **detflats_mon_hdr = NULL;
    qmost_flatstats *lin_stats = NULL;
    qmost_flatstats *mon_stats = NULL;
    int iin, nin;
    int ilin, ilint, nlin = 0, nlinuse = 0;
    int imon, imont, nmon = 0, nmonuse = 0;
    int nbpmuse = 0;

    const cpl_frame *this_frame;
    const char *this_tag;

    cpl_frame *detflat_frame;
    const char *detflat_filename;
    const char *detflat_ref_filename = NULL;

    const cpl_parameter *par;
    int docorr = 1;
    int nord = 4;
    int niter = 3;
    double clipthr = 5.0;
    double underexp = 1000.0;
    double overexp = 60000.0;
    int mingood = 5;
    double lthr = 8.0;
    double hthr = 8.0;
    double badfrac = 0.25;
    int nbpmmax = 20;

    cpl_size raw_extension;
    int arm, detlive;
    const char *arm_extname;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;

    const cpl_frame *raw_frame;
    cpl_propertylist *detflat_hdr = NULL;

    cpl_propertylist *qclist = NULL;
    cpl_propertylist *qctmp = NULL;
    cpl_propertylist *qcuse;

    cpl_imagelist *detflats_lin_img = NULL;
    cpl_image *img = NULL;

    cpl_table *linearity_tbl = NULL;

    cpl_mask *bpm = NULL;

#undef TIDY
#define TIDY                                                            \
    if(detflats_lin_raw != NULL) {                                      \
        cpl_frameset_delete(detflats_lin_raw);                          \
        detflats_lin_raw = NULL;                                        \
    }                                                                   \
    if(detflats_lin_hdr != NULL) {                                      \
        for(ilint = 0; ilint < nlin; ilint++) {                         \
            if(detflats_lin_hdr[ilint] != NULL) {                       \
                cpl_propertylist_delete(detflats_lin_hdr[ilint]);       \
            }                                                           \
        }                                                               \
        cpl_free(detflats_lin_hdr);                                     \
        detflats_lin_hdr = NULL;                                        \
    }                                                                   \
    if(lin_stats != NULL) {                                             \
        for(ilint = 0; ilint < nlinuse; ilint++) {                      \
            qmost_flatstats_free(lin_stats + ilint);                     \
        }                                                               \
        cpl_free(lin_stats);                                            \
        lin_stats = NULL;                                               \
    }                                                                   \
    if(detflats_mon_raw != NULL) {                                      \
        cpl_frameset_delete(detflats_mon_raw);                          \
        detflats_mon_raw = NULL;                                        \
    }                                                                   \
    if(detflats_mon_hdr != NULL) {                                      \
        for(imont = 0; imont < nmon; imont++) {                         \
            if(detflats_mon_hdr[imont] != NULL) {                       \
                cpl_propertylist_delete(detflats_mon_hdr[imont]);       \
            }                                                           \
        }                                                               \
        cpl_free(detflats_mon_hdr);                                     \
        detflats_mon_hdr = NULL;                                        \
    }                                                                   \
    if(mon_stats != NULL) {                                             \
        for(imont = 0; imont < nmonuse; imont++) {                      \
            qmost_flatstats_free(mon_stats + imont);                    \
        }                                                               \
        cpl_free(mon_stats);                                            \
        mon_stats = NULL;                                               \
    }                                                                   \
    if(master_bpm != NULL) {                                            \
        cpl_mask_delete(master_bpm);                                    \
        master_bpm = NULL;                                              \
    }                                                                   \
    if(master_bias_img != NULL) {                                       \
        cpl_image_delete(master_bias_img);                              \
        master_bias_img = NULL;                                         \
    }                                                                   \
    if(master_bias_var != NULL) {                                       \
        cpl_image_delete(master_bias_var);                              \
        master_bias_var = NULL;                                         \
    }                                                                   \
    if(master_dark_img != NULL) {                                       \
        cpl_image_delete(master_dark_img);                              \
        master_dark_img = NULL;                                         \
    }                                                                   \
    if(master_dark_var != NULL) {                                       \
        cpl_image_delete(master_dark_var);                              \
        master_dark_var = NULL;                                         \
    }                                                                   \
    if(detflat_hdr != NULL) {                                           \
        cpl_propertylist_delete(detflat_hdr);                           \
        detflat_hdr = NULL;                                             \
    }                                                                   \
    if(qclist != NULL) {                                                \
        cpl_propertylist_delete(qclist);                                \
        qclist = NULL;                                                  \
    }                                                                   \
    if(qctmp != NULL) {                                                 \
        cpl_propertylist_delete(qctmp);                                 \
        qctmp = NULL;                                                   \
    }                                                                   \
    if(detflats_lin_img != NULL) {                                      \
        cpl_imagelist_delete(detflats_lin_img);                         \
        detflats_lin_img = NULL;                                        \
    }                                                                   \
    if(img != NULL) {                                                   \
        cpl_image_delete(img);                                          \
        img = NULL;                                                     \
    }                                                                   \
    if(linearity_tbl != NULL) {                                         \
        cpl_table_delete(linearity_tbl);                                \
        linearity_tbl = NULL;                                           \
    }                                                                   \
    if(bpm != NULL) {                                                   \
        cpl_mask_delete(bpm);                                           \
        bpm = NULL;                                                     \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    master_bias_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_BIAS);
    master_dark_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_DARK);
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);

    /* Create framesets of inputs */
    detflats_lin_raw = cpl_frameset_new();
    detflats_mon_raw = cpl_frameset_new();

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_DETECTOR_FLAT_LIN)) {
            cpl_frameset_insert(detflats_lin_raw,
                                cpl_frame_duplicate(this_frame));
        }
        else if(!strcmp(this_tag, QMOST_RAW_DETECTOR_FLAT_MON)) {
            cpl_frameset_insert(detflats_mon_raw,
                                cpl_frame_duplicate(this_frame));
        }
    }

    /* Check we have some */
    nlin = cpl_frameset_get_size(detflats_lin_raw);
    if(nlin < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_RAW_DETECTOR_FLAT_LIN);
    }

    /* Get number of monitoring flats.  These are optional so it's not
     * an error if there aren't any. */
    nmon = cpl_frameset_get_size(detflats_mon_raw);

    /* Retrieve linearity analysis parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.docorr");
    if(par != NULL)
        docorr = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.nord");
    if(par != NULL)
        nord = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.niter");
    if(par != NULL)
        niter = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.clipthr");
    if(par != NULL)
        clipthr = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.underexp");
    if(par != NULL)
        underexp = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.overexp");
    if(par != NULL)
        overexp = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".linearity.mingood");
    if(par != NULL)
        mingood = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".bpm.lthr");
    if(par != NULL)
        lthr = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".bpm.hthr");
    if(par != NULL)
        hthr = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".bpm.badfrac");
    if(par != NULL)
        badfrac = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".bpm.nmax");
    if(par != NULL)
        nbpmmax = cpl_parameter_get_int(par);

    /* Read FITS primary headers of all input files */
    detflats_lin_hdr = cpl_calloc(nlin, sizeof(cpl_propertylist *));

    for(ilin = 0; ilin < nlin; ilin++) {
        detflat_frame = cpl_frameset_get_position(detflats_lin_raw, ilin);
        if(detflat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get linearity sequence "
                                         "detector flat %d from SOF",
                                         ilin+1);
        }

        detflat_filename = cpl_frame_get_filename(detflat_frame);
        if(detflat_filename == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get linearity sequence "
                                         "detector flat %d filename",
                                         ilin+1);
        }
        
        if(ilin == 0) {
            detflat_ref_filename = detflat_filename;
        }

        detflats_lin_hdr[ilin] = cpl_propertylist_load(detflat_filename,
                                                        0);
        if(detflats_lin_hdr[ilin] == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS primary "
                                         "header for linearity sequence "
                                         "detector flat %d",
                                         ilin+1);
        }
    }

    if(nmon > 0) {
        detflats_mon_hdr = cpl_calloc(nmon, sizeof(cpl_propertylist *));

        for(imon = 0; imon < nmon; imon++) {
            detflat_frame = cpl_frameset_get_position(detflats_mon_raw, imon);
            if(detflat_frame == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't get monitor sequence "
                                             "detector flat %d from SOF",
                                             imon+1);
            }
            
            detflat_filename = cpl_frame_get_filename(detflat_frame);
            if(detflat_filename == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't get monitor sequence "
                                             "detector flat %d filename",
                                             imon+1);
            }
            
            detflats_mon_hdr[imon] = cpl_propertylist_load(detflat_filename,
                                                            0);
            if(detflats_mon_hdr[imon] == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load FITS primary "
                                             "header for monitor sequence "
                                             "detector flat %d",
                                             imon+1);
            }
        }
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(detflats_lin_hdr[0],
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    /* Create outputs */
    linearity_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_LINEARITY,
        CPL_FRAME_TYPE_TABLE,
        NULL);
    if(linearity_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_LINEARITY);
    }

    linearity_filename = cpl_frame_get_filename(linearity_frame);

    bpm_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_BPM,
        CPL_FRAME_TYPE_IMAGE,
        NULL);
    if(bpm_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_BPM);
    }

    bpm_filename = cpl_frame_get_filename(bpm_frame);

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        detflat_hdr = cpl_propertylist_load(
            detflat_ref_filename,
            raw_extension);
        if(detflat_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(detflat_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(detflat_hdr, &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            /* Detector is not live, write out dummy HDUs to products
             * and move on. */
            if(qmost_dfs_save_table_extension(linearity_frame,
                                              detflat_hdr,
                                              arm_extname,
                                              NULL,
                                              NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "linearity table %s[%s]",
                                             linearity_filename,
                                             arm_extname);
            }
            
            if(qmost_dfs_save_mask_extension(bpm_frame,
                                             detflat_hdr,
                                             arm_extname,
                                             NULL,
                                             NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy BPM "
                                             "%s[%s]",
                                             bpm_filename,
                                             arm_extname);
            }

            cpl_propertylist_delete(detflat_hdr);
            detflat_hdr = NULL;

            continue;
        }

        cpl_msg_info(cpl_func,
                     "Processing extension %lld = %s",
                     raw_extension, arm_extname);
        cpl_msg_indent_more();

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(master_bpm_frame != NULL) {
            if(qmost_load_master_mask(master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* Process monitor sequence.  These can be discarded after
         * computing the statistics we need. */
        if(nmon > 0) {
            mon_stats = cpl_calloc(nmon, sizeof(qmost_flatstats));

            nmonuse = 0;
            for(imon = 0; imon < nmon; imon++) {
                raw_frame = cpl_frameset_get_position(detflats_mon_raw,
                                                      imon);
                
                qctmp = cpl_propertylist_duplicate(qclist);
                qcuse = qctmp;
                
                if(qmost_ccdproc(raw_frame,
                                 raw_extension,
                                 master_bpm,
                                 master_bias_img,
                                 master_bias_var,
                                 master_dark_img,
                                 master_dark_var,
                                 NULL,
                                 NULL,
                                 0,
                                 NULL,
                                 1,
                                 NULL,
                                 QMOST_LIN_NEVER,
                                 &img,
                                 NULL,
                                 qcuse) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "CCD processing failed for "
                                                 "monitor file %d "
                                                 "extension %lld",
                                                 imon+1,
                                                 raw_extension);
                }
                
                if(qmost_flatstats_compute(mon_stats + nmonuse,
                                           img,
                                           detflats_mon_hdr[imon],
                                           qcuse) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "computation of statistics "
                                                 "failed for monitor file %d "
                                                 "extension %lld",
                                                 imon+1,
                                                 raw_extension);
                }


                if(qmost_flatstats_check(mon_stats + nmonuse,
                                         underexp,
                                         overexp)) {
                    /* Add to list */
                    nmonuse++;
                }
                else {
                    /* Free */
                    qmost_flatstats_free(mon_stats + nmonuse);
                }
                
                cpl_image_delete(img);
                img = NULL;
                
                cpl_propertylist_delete(qctmp);
                qctmp = NULL;
            }
        }

        /* Process linearity sequence */
        lin_stats = cpl_calloc(nlin, sizeof(qmost_flatstats));

        detflats_lin_img = cpl_imagelist_new();

        nlinuse = 0;
        nbpmuse = 0;
        for(ilin = 0; ilin < nlin; ilin++) {
            raw_frame = cpl_frameset_get_position(detflats_lin_raw,
                                                  ilin);

            if(ilin > 0) {
                qctmp = cpl_propertylist_duplicate(qclist);
                qcuse = qctmp;
            }
            else {
                qcuse = qclist;
            }

            if(qmost_ccdproc(raw_frame,
                             raw_extension,
                             master_bpm,
                             master_bias_img,
                             master_bias_var,
                             master_dark_img,
                             master_dark_var,
                             NULL,
                             NULL,
                             0,
                             NULL,
                             1,
                             NULL,
                             QMOST_LIN_NEVER,
                             &img,
                             NULL,
                             qcuse) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "CCD processing failed for "
                                             "linearity file %d "
                                             "extension %lld",
                                             ilin+1,
                                             raw_extension);
            }

            if(qmost_flatstats_compute(lin_stats + nlinuse,
                                       img,
                                       detflats_lin_hdr[ilin],
                                       qcuse) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "computation of statistics "
                                             "failed for linearity file %d "
                                             "extension %lld",
                                             ilin+1,
                                             raw_extension);
            }

            if(qmost_flatstats_check(lin_stats + nlinuse,
                                     underexp,
                                     overexp)) {
                /* Considers every other flat for use in BPM to reduce
                 * the number to something more reasonable.  According
                 * to the calibration plan we get a total of 35 in the
                 * linearity sequence, so this results in about 15 for
                 * the BPM after applying the under- and over-exposure
                 * limits.  The limit nbpmmax is also applied. */
                if(nlinuse % 2 == 0 &&
                   nbpmuse < nbpmmax) {
                    /* Add to list */
                    cpl_imagelist_set(detflats_lin_img, img, nbpmuse);
                    /* now owned by imagelist */

                    nbpmuse++;
                }
                else {
                    cpl_image_delete(img);
                    img = NULL;
                }

                nlinuse++;
            }
            else {
                /* Free */
                qmost_flatstats_free(lin_stats + nlinuse);

                cpl_image_delete(img);
                img = NULL;
            }

            if(qctmp != NULL) {
                cpl_propertylist_delete(qctmp);
                qctmp = NULL;
            }
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_bpm) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        /* Check we got enough */
        if(nlinuse < mingood) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "there were not enough good flats: "
                                         "%d < %d",
                                         nlinuse, mingood);
        }

        /* Linearity fit */
        if(qmost_linfit(lin_stats,
                        nlinuse,
                        mon_stats,
                        nmonuse,
                        docorr,
                        &linearity_tbl,
                        qclist,
                        nord,
                        niter,
                        clipthr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "linearity fit failed for "
                                         "extension %lld",
                                         raw_extension);
        }

        /* Make BPM */
        if(qmost_gen_bpm(detflats_lin_img,
                         qclist,
                         lthr,
                         hthr,
                         badfrac,
                         &bpm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "BPM generation failed for "
                                         "extension %lld",
                                         raw_extension);
        }

        /* Save linearity table */
        if(qmost_dfs_save_table_extension(linearity_frame,
                                          detflat_hdr,
                                          arm_extname,
                                          qclist,
                                          linearity_tbl) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save linearity table "
                                         "%s[%s]",
                                         linearity_filename,
                                         arm_extname);
        }

        /* Save BPM */
        if(qmost_dfs_save_mask_extension(bpm_frame,
                                         detflat_hdr,
                                         arm_extname,
                                         qclist,
                                         bpm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save BPM "
                                         "%s[%s]",
                                         bpm_filename,
                                         arm_extname);
        }

        /* Clean up */
        cpl_imagelist_delete(detflats_lin_img);
        detflats_lin_img = NULL;

        cpl_table_delete(linearity_tbl);
        linearity_tbl = NULL;

        cpl_mask_delete(bpm);
        bpm = NULL;

        cpl_propertylist_delete(detflat_hdr);
        detflat_hdr = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        for(ilin = 0; ilin < nlinuse; ilin++) {
            qmost_flatstats_free(lin_stats + ilin);
        }
        
        cpl_free(lin_stats);
        lin_stats = NULL;

        if(mon_stats != NULL) {
            for(imon = 0; imon < nmonuse; imon++) {
                qmost_flatstats_free(mon_stats + imon);
            }

            cpl_free(mon_stats);
            mon_stats = NULL;
        }

        cpl_msg_indent_less();
    }

    /* Clean up */
    cpl_frameset_delete(detflats_lin_raw);
    detflats_lin_raw = NULL;

    for(ilin = 0; ilin < nlin; ilin++) {
        if(detflats_lin_hdr[ilin] != NULL) {
            cpl_propertylist_delete(detflats_lin_hdr[ilin]);
        }
    }

    cpl_free(detflats_lin_hdr);
    detflats_lin_hdr = NULL;

    cpl_frameset_delete(detflats_mon_raw);
    detflats_mon_raw = NULL;

    if(nmon > 0) {
        for(imon = 0; imon < nmon; imon++) {
            if(detflats_mon_hdr[imon] != NULL) {
                cpl_propertylist_delete(detflats_mon_hdr[imon]);
            }
        }
        
        cpl_free(detflats_mon_hdr);
        detflats_mon_hdr = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_linearity_analyse_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* Linearity fit parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".linearity.docorr",
                                  CPL_TYPE_BOOL,
                                  "Use monitor sequence to correct for "
                                  "lamp brightness variations.",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.docorr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".linearity.nord",
                                  CPL_TYPE_INT,
                                  "The order of the polynomial used to "
                                  "fit the linearity solution.",
                                  RECIPE_NAME,
                                  4);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.nord");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".linearity.niter",
                                  CPL_TYPE_INT,
                                  "The number of rejection iterations "
                                  "to use in the linearity fit.",
                                  RECIPE_NAME,
                                  3);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.niter");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".linearity.clipthr",
                                  CPL_TYPE_DOUBLE,
                                  "The number of sigma for outlier "
                                  "rejection.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.clipthr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".linearity.underexp",
                                  CPL_TYPE_DOUBLE,
                                  "A minimum count level in ADU to be "
                                  "used to reject underexposed flats.",
                                  RECIPE_NAME,
                                  1000.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.underexp");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".linearity.overexp",
                                  CPL_TYPE_DOUBLE,
                                  "A saturation level in ADU to be "
                                  "used to reject overexposed flats.",
                                  RECIPE_NAME,
                                  60000.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.overexp");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".linearity.mingood",
                                  CPL_TYPE_INT,
                                  "The minimum number of good images "
                                  "left after removing under and over "
                                  "exposed images needed to run the "
                                  "algorithm.",
                                  RECIPE_NAME,
                                  5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "linearity.mingood");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".bpm.lthr",
                                  CPL_TYPE_DOUBLE,
                                  "Lower threshold for flagging a bad "
                                  "pixel in number of sigma.",
                                  RECIPE_NAME,
                                  8.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "bpm.lthr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".bpm.hthr",
                                  CPL_TYPE_DOUBLE,
                                  "Upper threshold for flagging a bad "
                                  "pixel in number of sigma.",
                                  RECIPE_NAME,
                                  8.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "bpm.hthr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".bpm.badfrac",
                                  CPL_TYPE_DOUBLE,
                                  "The fraction (0.0-1.0) of exposures on "
                                  "which a pixel is discordant in order "
                                  "for it to be defined as bad.",
                                  RECIPE_NAME,
                                  0.25);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "bpm.badfrac");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".bpm.nmax",
                                  CPL_TYPE_INT,
                                  "The maximum number of linearity sequence "
                                  "flats to use in BPM analysis",
                                  RECIPE_NAME,
                                  20);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "bpm.nmax");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/**@}*/
