/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/* 
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_dfs.h"
#include "qmost_blk.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_blk       qmost_blk
 * 
 * Utilities for dealing with binned data.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Determine image binning from FITS header.
 *
 * @param   in_hdr     (Given)    Input FITS header.
 * @param   specbin    (Returned) Pointer to variable to receive
 *                                binning in spectral direction.
 * @param   spatbin    (Returned) Pointer to variable to receive
 *                                binning in spatial direction.
 * @param   binned     (Returned) Pointer to receive boolean flag
 *                                specifying if image is binned.
 *
 * @return  void
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_isbinned (
    cpl_propertylist *in_hdr,
    int *specbin,
    int *spatbin,
    int *binned)
{
    cpl_errorstate prestate;

    /* Read the relevant DRS keywords written by qmost_ccdproc */
    prestate = cpl_errorstate_get();

    if(qmost_cpl_propertylist_get_int(in_hdr,
                                      "ESO DRS SPECBIN",
                                      specbin) != CPL_ERROR_NONE ||
       qmost_cpl_propertylist_get_int(in_hdr,
                                      "ESO DRS SPATBIN",
                                      spatbin) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);  /* clear */

        cpl_msg_warning(cpl_func,
                        "DRS binning keywords are missing or unreadable");

        *specbin = 1;
        *spatbin = 1;
        *binned = 0;
    }
    else {
        *binned = (*specbin > 1 || *spatbin > 1);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Update bad pixel mask to account for binning.
 *
 * @param   in_bpm     (Given)    Input bad pixel mask.
 * @param   blkfac     (Given)    Binning factors in each axis.
 * 
 * @return  cpl_mask *            The updated bad pixel mask.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_mask *qmost_maskblk (
    cpl_mask *in_bpm,
    int blkfac[2])
{
    cpl_mask *result = NULL;

    int jout,iout,i,j,outind,sum,ii,jj,inind;
    long naxis[2], bufax[2];
    cpl_binary *bpm, *buf;

    /* If no binning is done, then just copy the input and return */

    if (blkfac[0] == 1 && blkfac[1] == 1) {
        result = cpl_mask_duplicate(in_bpm);
        return result;
    }

    /* Define the axis lengths and get the buffer */

    naxis[0] = cpl_mask_get_size_x(in_bpm);
    naxis[1] = cpl_mask_get_size_y(in_bpm);

    bufax[0] = naxis[0]/blkfac[0];
    bufax[1] = naxis[1]/blkfac[1];

    result = cpl_mask_new(bufax[0], bufax[1]);

    bpm = cpl_mask_get_data(in_bpm);
    buf = cpl_mask_get_data(result);

    /* Do the blocking now */

    jout = -1;
    for (j = 0; j < naxis[1]; j += blkfac[1]) {
        jout++;
        iout = -1;
        for (i = 0; i < naxis[0]; i += blkfac[0]) {
            iout++;
            outind = jout*bufax[0] + iout;
            sum = 0;
            for (jj = j; jj < j+blkfac[1]; jj++) {
                for (ii = i; ii < i+blkfac[0]; ii++) {
                    inind = jj*naxis[0] + ii;
                    sum += bpm[inind];
                }
            }
            buf[outind] = (sum > 0);
        }
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Update flat to account for binning in spectral direction.
 *
 * @param   data      (Given)     Input unbinned data array.
 * @param   var       (Given)     Input unbinned variance array.
 * @param   naxis     (Given)     Input image dimensions.
 * @param   nx        (Given)     Original x dimension of image.
 * @param   specbin   (Given)     Binning factor in spectral
 *                                direction.
 * @param   pixoff    (Given)     Pixel offset 0 or 1 within bin. ?
 * @param   buf       (Returned)  Output binned data array.
 * @param   bufv      (Returned)  Output binned variance array.
 * @param   bufax     (Modified)  Output image dimensions.
 * 
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_flatblk(
    float *data,
    float *var,
    long naxis[2],
    int nx,
    int specbin,
    int pixoff,
    float **buf,
    float **bufv,
    long bufax[2])
{
    int jout,j,iout,i,outind,nc,ii,inind;
    float sum,sumv;

    /* If no binning is done, then just get out of here */

    if (specbin == 1)
        return;

    /* Define the axis lengths. This depends on whether there are any
       'incomplete' bins at either end */

    bufax[0] = nx;
    bufax[1] = naxis[1];

    /* Get the buffers you need */

    *buf = cpl_calloc(bufax[0]*bufax[1],sizeof(float));
    *bufv = cpl_calloc(bufax[0]*bufax[1],sizeof(float));

    for (i = 0; i < bufax[0]*bufax[1]; i++) {
        (*buf)[i] = 1.0;
        (*bufv)[i] = 1.0;
    }

    /* Do the blocking now */

    jout = -1;
    for (j = 0; j < naxis[1]; j++) {
        jout++;
        iout = -1;
        for (i = -pixoff; i < naxis[0]; i += specbin) {
            iout++;
            if(iout >= nx)
                break;
            outind = jout*bufax[0] + iout;
            sum = 0.0;
            sumv = 0.0;
            nc = 0;
            for (ii = i; ii < i+specbin; ii++) {
                if (ii < 0 || ii >= naxis[0])
                    break;
                inind = j*naxis[0] + ii;
                sum += data[inind];
                sumv += var[inind];
                nc++;
            }
            if (nc > 0) {
                (*buf)[outind] = sum/(float)nc;
                (*bufv)[outind] = sumv/pow((double)nc,2);
            } else {
                (*buf)[outind] = 1.0;
                (*bufv)[outind] = 0.0;
            }
        }
    }
}
    
/**@}*/

/*

$Log$
Revision 1.2  2019/03/21 11:48:43  jrl
modified algorithm in flatblk to ensure we get them same spectral length

Revision 1.1  2019/02/25 10:55:52  jrl
New entry


*/
