/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_CONSTANTS_H
#define QMOST_CONSTANTS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/**
 * @ingroup qmost_constants
 *
 * @brief   Enumerated constants for the FIB_ROOT values used to
 *          specify the spectrograph.
 *
 * The values for this constant are specified in the definition of the
 * FIBINFO table in the Facility Control System – Data Management
 * System Interface Control Document VIS-ICD-4MOST-47110-1314-0001,
 * Section 5.
 */

enum _qmost_spec {
    /** High Resolution Spectrograph, HRS */
    QMOST_SPEC_HRS   = 1,

    /** Low Resolution Spectrograph-A, LRS-A */
    QMOST_SPEC_LRS_A = 2,

    /** Low Resolution Spectrograph-B, LRS-B */
    QMOST_SPEC_LRS_B = 3
};

/**
 * @ingroup qmost_constants
 *
 * @brief   The number of spectrographs.
 */

#define QMOST_NUM_SPEC   3

/**
 * @ingroup qmost_constants
 *
 * @brief   Enumerated constants for the spectrograph arms.
 *
 * The order of these constants defines the order the FITS extensions
 * are written to the output files, which is specified to be in the
 * order red, green, blue by the Data Reduction Pipeline Description
 * document VIS-DER-4MOST-47110-1410-0002.
 */

enum _qmost_arm {
    /** Red arm */
    QMOST_ARM_RED    = 1,
    /** Green arm */
    QMOST_ARM_GREEN  = 2,
    /** Blue arm */
    QMOST_ARM_BLUE   = 3
};

/**
 * @ingroup qmost_constants
 *
 * @brief   The number of arms in each spectrograph.
 */

#define QMOST_NUM_ARMS   3

/**
 * @ingroup qmost_constants
 *
 * @brief   The number of fibres per slitlet.
 */

#define QMOST_NFIB_PER_SLITLET   29

/**
 * @ingroup qmost_constants
 *
 * @brief   Saturation threshold in ADU used to flag saturated pixels.
 */

#define QMOST_SATURATE 65534

/**
 * @ingroup qmost_constants
 *
 * @brief   Values used in the rejection map emitted by
 *          qmost_extract_psf.
 */

enum _qmost_wmask {
    /** Good pixel */
    QMOST_WMASK_GOOD  =   0,
    /** Hot pixel */
    QMOST_WMASK_HOT   =   1,
    /** Cold pixel */
    QMOST_WMASK_COLD  =   2,
    /** Pixel flagged affected by cosmic ray */
    QMOST_WMASK_CR1   =   3,
    /** Discordant pixel next to cosmic ray pixel */
    QMOST_WMASK_CR2   =   4,
    /** Pixel with level above defined saturation */
    QMOST_WMASK_SAT   =   5,
    /** Pixel flagged discordant in psf fitting */
    QMOST_WMASK_DISC  =   6,
    /** Pixel flagged for unknown reason */
    QMOST_WMASK_OTHER = 100
};

/**
 * @ingroup qmost_constants
 *
 * @brief   Speed of light in km/s.
 */

#define QMOST_SPEEDOFLIGHT    299792.458

/**
 * @ingroup qmost_constants
 *
 * @brief   Codes for type of statistic in qmost_imcombine_lite.
 */

enum _qmost_combtype {
    /** Mean combine. */
    QMOST_MEANCALC   = 1,
    /** Median combine. */
    QMOST_MEDIANCALC = 2
};

/**
 * @ingroup qmost_constants
 *
 * @brief   Custom error code raised when we attempt to use a dummy
 *          extension.
 *
 * This error code is thrown when we attempt to use a dummy extension
 * in processing.  We need to be able to trap it in some cases so we
 * use a custom error code extending the standard CPL errors.
 */

#define QMOST_ERROR_DUMMY CPL_ERROR_EOL

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_get_rebin_params (
    int spec,
    int arm,
    double *minwave,
    double *maxwave,
    double *dlam);

cpl_error_code qmost_get_skysubwave (
    int spec,
    int arm,
    double *minwave,
    double *maxwave);

CPL_END_DECLS

#endif  /* QMOST_CONSTANTS_H */
