/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_polynm.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_polynm_test  Unit test of qmost_polynm
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_polynm
 */
/*----------------------------------------------------------------------------*/

#define TOL 1.0e-9

static void test_qmost_polynm(void)
{
    double xp[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
    double yp[9];
    double realcfa[4] = { 1, 2, 3, 4 };
    double realcfb[4] = { 0, 1, 2, 3 };

    cpl_size i, n, icoef, ncoef, degree;
    cpl_polynomial *poly;
    double value, mean;

    n = sizeof(xp) / sizeof(xp[0]);
    ncoef = sizeof(realcfa) / sizeof(realcfa[0]);
    degree = ncoef - 1;

    /* Evaluate the correct answer */
    mean = 0;

    for(i = 0; i < n; i++) {
        yp[i] = ((realcfa[3] * xp[i] + realcfa[2]) * xp[i] + realcfa[1]) * xp[i] + realcfa[0];
        mean += yp[i];
    }

    mean /= n;

    /* NULL inputs */
    poly = qmost_polynm(xp, NULL, n, 1, 0);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    poly = qmost_polynm(NULL, yp, n, 1, 0);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Zero length input */
    poly = qmost_polynm(xp, yp, 0, 1, 0);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* Length 1 input */
    poly = qmost_polynm(xp, yp, 1, 1, 0);
    cpl_test_nonnull(poly);

    icoef = 0;
    value = cpl_polynomial_get_coeff(poly, &icoef);
    cpl_test_abs(value, yp[0], TOL);

    cpl_polynomial_delete(poly);

    /* Not enough points for number of coefficients */
    poly = qmost_polynm(xp, yp, 1, 3, 0);
    cpl_test_null(poly);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* Zero degree */
    poly = qmost_polynm(xp, yp, n, 1, 0);
    cpl_test_nonnull(poly);

    /* Check coefficient, it should be about mean(y) */
    icoef = 0;
    value = cpl_polynomial_get_coeff(poly, &icoef);
    cpl_test_abs(value, mean, TOL);

    cpl_polynomial_delete(poly);

    /* Valid input */
    poly = qmost_polynm(xp, yp, n, degree+1, 0);
    cpl_test_nonnull(poly);

    /* Check coefficients are what we put in */
    for(icoef = 0; icoef < ncoef; icoef++) {
        value = cpl_polynomial_get_coeff(poly, &icoef);
        cpl_test_abs(value, realcfa[icoef], TOL);
    }

    cpl_polynomial_delete(poly);

    /* Fit without degree zero coefficient gives different answer */
    for(i = 0; i < n; i++) {
        yp[i] = ((realcfb[3] * xp[i] + realcfb[2]) * xp[i] + realcfb[1]) * xp[i] + realcfb[0];
    }

    poly = qmost_polynm(xp, yp, n, degree+1, 1);
    cpl_test_nonnull(poly);

    for(icoef = 0; icoef < ncoef; icoef++) {
        value = cpl_polynomial_get_coeff(poly, &icoef);
        cpl_test_abs(value, realcfb[icoef], TOL);
    }

    cpl_polynomial_delete(poly);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_polynm module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_polynm();

    return cpl_test_end(0);
}

/**@}*/
