/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_constants.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_constants  qmost_constants
 * 
 * Static instrument-specific defaults for processing parameters.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_constants.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Return the output wavelength binning parameters.
 *
 * @param   spec              (Given)    One of the QMOST_
 * @param   arm               (Given)    One of the QMOST_ARM_* values
 *                                       specifying which arm this is.
 * @param   minwave           (Modified) The minimum wavelength to use
 *                                       when rebinning.
 * @param   maxwave           (Modified) The maximum wavelength to use
 *                                       when rebinning.
 * @param   dlam              (Modified) The wavelength increment to
 *                                       use when rebinning.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If the output pointers are null.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If spec or arm aren't one of the
 *                                    recognised values.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_get_rebin_params (
    int spec,
    int arm,
    double *minwave,
    double *maxwave,
    double *dlam)
{
    /* Tables of rebinning parameters */
    static const double minwaves[QMOST_NUM_SPEC][QMOST_NUM_ARMS] = {
        /* RED    GREEN   BLUE */
        { 6100.0, 5160.0, 3926.0 },  /* HRS */
        { 6970.0, 5360.0, 3700.0 },  /* LRS-A */
        { 6970.0, 5360.0, 3700.0 }   /* LRS-B */
    };

    static const double maxwaves[QMOST_NUM_SPEC][QMOST_NUM_ARMS] = {
        /* RED    GREEN   BLUE */
        { 6788.0, 5730.0, 4350.0 },  /* HRS */
        { 9500.0, 7130.0, 5445.0 },  /* LRS-A */
        { 9500.0, 7130.0, 5445.0 }   /* LRS-B */
    };

    static const double dlams[QMOST_NUM_SPEC] = {
        0.05,  /* HRS */
        0.25,  /* LRS-A */
        0.25   /* LRS-B */
    };

    cpl_ensure_code(minwave != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(maxwave != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dlam != NULL, CPL_ERROR_NULL_INPUT);

    if(spec < 1 || spec > QMOST_NUM_SPEC) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "invalid spec: %d", arm);
    }

    if(arm < 1 || arm > QMOST_NUM_ARMS) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "invalid arm: %d", arm);
    }

    *minwave = minwaves[spec-1][arm-1];
    *maxwave = maxwaves[spec-1][arm-1];
    *dlam = dlams[spec-1];

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Return the wavelength range to be used for sky background
 *          correction.
 *
 * @param   spec              (Given)    One of the QMOST_
 * @param   arm               (Given)    One of the QMOST_ARM_* values
 *                                       specifying which arm this is.
 * @param   minwave           (Modified) The minimum wavelength to use
 *                                       when rebinning.
 * @param   maxwave           (Modified) The maximum wavelength to use
 *                                       when rebinning.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If the output pointers are null.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If spec or arm aren't one of the
 *                                    recognised values.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_get_skysubwave (
    int spec,
    int arm,
    double *minwave,
    double *maxwave)
{
    /* Tables of background correction wavelength region parameters */
    static const double minwaves[QMOST_NUM_SPEC][QMOST_NUM_ARMS] = {
        /* RED    GREEN   BLUE */
        { 6100.0, 5160.0, 0.0 },  /* HRS */
        { 6970.0, 5500.0, 0.0 },  /* LRS-A */
        { 6970.0, 5500.0, 0.0 }   /* LRS-B */
    };

    static const double maxwaves[QMOST_NUM_SPEC][QMOST_NUM_ARMS] = {
        /* RED    GREEN   BLUE */
        { 6788.0, 5730.0, 0.0 },  /* HRS */
        { 9500.0, 7130.0, 0.0 },  /* LRS-A */
        { 9500.0, 7130.0, 0.0 }   /* LRS-B */
    };

    cpl_ensure_code(minwave != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(maxwave != NULL, CPL_ERROR_NULL_INPUT);

    if(spec < 1 || spec > QMOST_NUM_SPEC) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "invalid spec: %d", arm);
    }

    if(arm < 1 || arm > QMOST_NUM_ARMS) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "invalid arm: %d", arm);
    }

    *minwave = minwaves[spec-1][arm-1];
    *maxwave = maxwaves[spec-1][arm-1];

    return CPL_ERROR_NONE;
}

/**@}*/
