/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "hdrl.h"
#include "qmost_constants.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_utils.h"
#include "qmost_waveinfo.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_fibtab    qmost_fibtab
 *
 * FIBINFO table utility routines
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_fibtab.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load the FIBINFO table from an input frame.
 *
 * The fibre information, or "FIBINFO" table is loaded from the given
 * input frame by searching for an extension with EXTNAME="FIBINFO" in
 * the FITS header.
 *
 * When loading a raw file, the shuffle argument should be specified
 * and set equal to the appropriate QMOST_ARM_* constant defined in
 * qmost_constants.h, which will cause the rows of the input FIBINFO
 * table to be filtered to retain only those fibres connected to the
 * spectrograph being processed, based on the FIB_ROOT column.
 * Entries for the simultaneous calibration fibres (which are not
 * present in the FIBINFO tables of the raw data, but are present on
 * the spectrograph slit) are also added to the start and end of the
 * table, and the entries sorted into the correct order so they match
 * the order the fibre images appear along the spatial axis of the
 * detector.  Any relevant information from the METROLOGY table is
 * also cross-matched based on the fibre ID (FIB_ID) and added to the
 * FIBINFO table by appending extra columns.
 *
 * @param   inframe    (Given)    Input frame.
 * @param   shuffle    (Given)    If non-zero, then the rows will be
 *                                selected by spectrograph.  The value
 *                                given should be the appropriate
 *                                FIB_ROOT for the spectrograph.
 * @param   tab        (Returned) The loaded table.
 * @param   hdr        (Returned) FITS header information.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If no FIBINFO extension was
 *                                    found in the input file, or one
 *                                    of the required columns didn't
 *                                    exist.
 * @retval  CPL_ERROR_FILE_IO         If the input file can't be read.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the FIBINFO extension doesn't
 *                                    contain a valid binary table.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If one of the FIBINFO table
 *                                    columns had an incorrect data
 *                                    type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO INS SHUTn STAT</b>
 *
 * @par Input FIBINFO table columns:
 *   - <b>FIB_ID</b>
 *   - <b>FIB_ROOT</b>
 *   - <b>SLIT_POS</b>
 *
 * @par Output FIBINFO table columns:
 *   The following columns are populated for the simultaneous
 *   calibration fibres.
 *   - <b>FIB_ID</b>: The unique fibre ID of the object.  Populated
 *     with the otherwise invalid flag value of zero for the
 *     simultaneous calibration fibres.
 *   - <b>FIB_ST</b>: Fibre status.  Populated with a value equal to
 *     the simultaneous calibration fibre shutter status (ESO INS
 *     SHUTn STAT), i.e. 1 meaning parked but can see light if the
 *     shutter is open, and 0 meaning disabled and cannot see light if
 *     the shutter is closed.
 *   - <b>FIB_USE</b>: Fibre use.  Populated with a special value of
 *     zero for simultaneous calibration fibre.
 *   - <b>FIB_ROOT</b>: Specifies where the fibre is routed.
 *     Populated with the spectrograph ID from the parameter shuffle.
 *   - <b>SLIT_POS</b>: Slit position of the fibre.  The simultaneous
 *     calibration files are given their correct slit positions 1-5
 *     and 818-822.
 *
 * @note    The shuffle algorithm also deletes entries for which there
 *          is no fibre.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibtabload (
    const cpl_frame *inframe,
    int shuffle,
    cpl_table **tab,
    cpl_propertylist **hdr)
{
    const char *filename;
    cpl_table *intab = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_table *mettab = NULL;
    cpl_propertylist *methdr = NULL;
    cpl_table *shuftab = NULL;
    cpl_propertylist *reflist = NULL;

    cpl_size inext, metext;
    cpl_size nrowstmp;
    int row;

    int *met_fib_id = NULL;
    int nrowsmet = 0;

    int *wantrows = NULL;
    cpl_array *metnames = NULL;
    int fib_id, isp, ifp, im;

    int icol, ncols, rmet;
    const char *colname, *colunit;
    cpl_type coltype;

    cpl_propertylist *prihdr = NULL;
    char *key = NULL;
    cpl_errorstate prestate;
    int shut_open, min_pos, nsimucal = 0, isnull, fib_st;
    int slit_pos;

#undef TIDY
#define TIDY                                    \
    if(intab != NULL) {                         \
        cpl_table_delete(intab);                \
        intab = NULL;                           \
    }                                           \
    if(inhdr != NULL) {                         \
        cpl_propertylist_delete(inhdr);         \
        inhdr = NULL;                           \
    }                                           \
    if(mettab != NULL) {                        \
        cpl_table_delete(mettab);               \
        mettab = NULL;                          \
    }                                           \
    if(methdr != NULL) {                        \
        cpl_propertylist_delete(methdr);        \
        methdr = NULL;                          \
    }                                           \
    if(shuftab != NULL) {                       \
        cpl_table_delete(shuftab);              \
        shuftab = NULL;                         \
    }                                           \
    if(reflist != NULL) {                       \
        cpl_propertylist_delete(reflist);       \
        reflist = NULL;                         \
    }                                           \
    if(wantrows != NULL) {                      \
        cpl_free(wantrows);                     \
        wantrows = NULL;                        \
    }                                           \
    if(metnames != NULL) {                      \
        cpl_array_delete(metnames);             \
        metnames = NULL;                        \
    }                                           \
    if(prihdr != NULL) {                        \
        cpl_propertylist_delete(prihdr);        \
        prihdr = NULL;                          \
    }                                           \
    if(key != NULL) {                           \
        cpl_free(key);                          \
        key = NULL;                             \
    }

    /* Check required inputs and outputs */
    cpl_ensure_code(inframe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tab, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(hdr, CPL_ERROR_NULL_INPUT);

    /* Get filename */
    filename = cpl_frame_get_filename(inframe);
    if(filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    /* Get extension number */
    inext = cpl_fits_find_extension(filename,
                                    QMOST_FIBINFO_EXTNAME);
    if(inext < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "error trying to find %s in %s",
                                     QMOST_FIBINFO_EXTNAME,
                                     filename);
    }
    else if(inext == 0) {
        /* No FIBINFO extension.  This is not an error, but we emit a
         * warning. */
        cpl_msg_warning(cpl_func,
                        "%s extension is missing in input file %s",
                        QMOST_FIBINFO_EXTNAME,
                        filename);

        *tab = NULL;
        *hdr = NULL;

        TIDY;
        return CPL_ERROR_NONE;
    }

    /* Read input table */
    intab = cpl_table_load(filename,
                           inext,
                           1);
    if(intab == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO table %s[%lld]",
                                     filename,
                                     inext);
    }

    /* and FITS header */
    inhdr = cpl_propertylist_load(filename,
                                  inext);
    if(inhdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO header %s[%lld]",
                                     filename,
                                     inext);
    }

    /* Check for METROLOGY */
    metext = cpl_fits_find_extension(filename,
                                     QMOST_METROLOGY_EXTNAME);
    if(metext < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "error trying to find %s in %s",
                                     QMOST_METROLOGY_EXTNAME,
                                     filename);
    }
    else if(metext > 0) {
        /* Read METROLOGY */
        mettab = cpl_table_load(filename,
                                metext,
                                1);
        if(mettab == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load METROLOGY "
                                         "table %s[%lld]",
                                         filename,
                                         metext);
        }

        /* and FITS header */
        methdr = cpl_propertylist_load(filename,
                                       metext);
        if(methdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load METROLOGY "
                                         "header %s[%lld]",
                                         filename,
                                         metext);
        }

        /* Append HIERARCH ESO OCS MT headers */
        if(cpl_propertylist_copy_property_regexp(inhdr,
                                                 methdr,
                                                 "^ESO OCS MT .*$",
                                                 0) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't append METROLOGY "
                                         "headers");
        }

        cpl_propertylist_delete(methdr);
        methdr = NULL;

        /* Remove any NULL FIB_IDs from table */
        cpl_table_unselect_all(mettab);

        if(cpl_table_or_selected_invalid(mettab, "FIB_ID") < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to select invalid "
                                         "FIB_ID in METROLOGY");
        }

        cpl_table_erase_selected(mettab);
        cpl_table_select_all(mettab);

        /* Ensure sorted on FIB_ID */
        reflist = cpl_propertylist_new();
        cpl_propertylist_append_bool(reflist, "FIB_ID", 0);

        if(cpl_table_sort(mettab, reflist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to sort by FIB_ID in "
                                         "METROLOGY table");
        }

        cpl_propertylist_delete(reflist);
        reflist = NULL;

        /* Array of metrology FIB_IDs */
        met_fib_id = cpl_table_get_data_int(mettab, "FIB_ID");
        if(met_fib_id == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to get FIB_ID from "
                                         "METROLOGY table");
        }

        nrowsmet = cpl_table_get_nrow(mettab);
    }

    if(shuffle) {
        /* Select all rows with FIB_ROOT equal to the requested value and
           FIB_ST > 0 */
        cpl_table_select_all(intab);

        if(cpl_table_and_selected_int(intab,
                                      "FIB_ROOT",
                                      CPL_EQUAL_TO,
                                      shuffle) < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to select rows with "
                                         "FIB_ROOT=%d in FIBINFO table",
                                         shuffle);
        }

        /* Extract selected rows */
        shuftab = cpl_table_extract_selected(intab);
        if(!shuftab) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to extract selected rows "
                                         "in FIBINFO table");
        }

        /* Restore selection on both tables */
        cpl_table_select_all(intab);
        cpl_table_select_all(shuftab);

        /* Sort by slit position */
        reflist = cpl_propertylist_new();
        cpl_propertylist_append_bool(reflist, "SLIT_POS", 0);
      
        if(cpl_table_sort(shuftab, reflist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to sort by SLIT_POS in "
                                         "output FIBINFO table");
        }

        cpl_propertylist_delete(reflist);
        reflist = NULL;

        /* Copy METROLOGY information if we have one */
        if(mettab != NULL) {
            /* List of rows we need */
            nrowstmp = cpl_table_get_nrow(shuftab);

            wantrows = cpl_malloc(nrowstmp * sizeof(int));

            for(row = 0; row < nrowstmp; row++) {
                fib_id = cpl_table_get_int(shuftab,
                                           "FIB_ID",
                                           row,
                                           &isnull);
                if(isnull < 0) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to read FIB_ID for "
                                                 "row %d",
                                                 row+1);
                }
                else if(isnull == 0) {
                    isp = 0;
                    ifp = nrowsmet;

                    while(isp < ifp) {
                        im = (isp + ifp) / 2;
                        if(met_fib_id[im] < fib_id) {
                            isp = im + 1;
                        }
                        else {
                            ifp = im;
                        }
                    }

                    if(isp >= 0 && isp < nrowsmet &&
                       met_fib_id[isp] == fib_id) {
                        wantrows[row] = isp;
                    }
                    else {
                        wantrows[row] = -1;
                    }
                }
                else {
                    wantrows[row] = -1;
                }
            }

            /* Process column by column */
            metnames = cpl_table_get_column_names(mettab);

            ncols = cpl_array_get_size(metnames);

            for(icol = 0; icol < ncols; icol++) {
                colname = cpl_array_get_string(metnames, icol);

                /* Skip FIB_ID and FIB_ST, which are already there. */
                if(!strcmp(colname, "FIB_ID") ||
                   !strcmp(colname, "FIB_ST")) {
                    continue;
                }

                /* Create column in output */
                coltype = cpl_table_get_column_type(mettab, colname);
                colunit = cpl_table_get_column_unit(mettab, colname);

                if(cpl_table_new_column(shuftab,
                                        colname,
                                        coltype) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "failed to create "
                                                 "column %d", icol+1);
                }

                if(colunit != NULL) {
                    if(cpl_table_set_column_unit(shuftab,
                                                 colname,
                                                 colunit) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "failed to set "
                                                     "column %d units",
                                                     icol+1);
                    }
                }

                /* Copy data */
                for(row = 0; row < nrowstmp; row++) {
                    rmet = wantrows[row];

                    if(wantrows[row] >= 0) {
                        if(qmost_cpl_table_copy_cell(shuftab,
                                                     colname,
                                                     row,
                                                     mettab,
                                                     colname,
                                                     rmet) != CPL_ERROR_NONE) {
                            TIDY;
                            return cpl_error_set_message(cpl_func,
                                                         cpl_error_get_code(),
                                                         "failed to copy "
                                                         "column %d "
                                                         "contents for "
                                                         "row %d",
                                                         icol+1,
                                                         row+1);
                        }
                    }
                }
            }

            cpl_array_delete(metnames);
            metnames = NULL;

            cpl_free(wantrows);
            wantrows = NULL;
        }

        /* There might be 0 entries in the table, trap this so we
         * don't error out trying to read the first one. */
        if(cpl_table_get_nrow(shuftab) > 0) {
            /* Infer the number of simucal fibres from the smallest
             * SLIT_POS value seen in the table.  This is numbered from 1
             * including the simucal fibres so the number of simucals at
             * the bottom of the slit is min(SLIT_POS) - 1.  We assume
             * the number at the top of the slit is the same. */
            min_pos = cpl_table_get_int(shuftab,
                                        "SLIT_POS",
                                        0,
                                        &isnull);
            if(isnull < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not read %s for row %d",
                                             "SLIT_POS", 1);
            }
            else if(isnull > 0) {
                min_pos = 1;
            }

            nsimucal = min_pos - 1;
            if(nsimucal < 0) {
                nsimucal = 0;
            }
        }

        /* Special case to detect off by one error in raw files.
         * This happened because the number and layout of fibres on
         * the spectrograph slit was changed going from simulated to
         * real data, but the raw file metadata were not updated to
         * reflect this change. */
        if(nsimucal == 5) {
            /* For real data, this is incorrect, there are actually 6.
             * The entries for HRS and LRS-B are also reversed.  This
             * kludge fixes the problems by renumbering the column.
             * CAUTION: this is backwards-incompatible, and will
             * break any data sets where the number of simucals should
             * actually be 5, such as simulated data. */
            cpl_msg_warning(cpl_func,
                            "SLIT_POS numbering error detected. "
                            "Applying workaround.");

            nrowstmp = cpl_table_get_nrow(shuftab);

            for(row = 0; row < nrowstmp; row++) {
                slit_pos = cpl_table_get_int(shuftab,
                                             "SLIT_POS",
                                             row,
                                             &isnull);
                if(isnull < 0) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "could not read %s "
                                                 "for row %d",
                                                 "SLIT_POS", row);
                }
                else if(isnull == 0 && slit_pos != 0) {
                    if(shuffle == QMOST_SPEC_LRS_A) {
                        /* Add 1 for LRS-A */
                        slit_pos++;
                    }
                    else {
                        /* Equivalent to adding 1 and then reversing
                         * the numbering for HRS and LRS-B. */
                        slit_pos = 824 - slit_pos;
                    }

                    cpl_table_set_int(shuftab,
                                      "SLIT_POS",
                                      row,
                                      slit_pos);
                }
            }

            nsimucal++;

            /* Sort by slit position */
            reflist = cpl_propertylist_new();
            cpl_propertylist_append_bool(reflist, "SLIT_POS", 0);

            if(cpl_table_sort(shuftab, reflist) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to sort by SLIT_POS in "
                                             "output FIBINFO table");
            }

            cpl_propertylist_delete(reflist);
            reflist = NULL;
        }

        if(nsimucal > 0) {
            /* Figure out if simucal fibres are illuminated using PHDU
             * keywords.  We first check if any of the lamps are on, and
             * then if the appropriate shutter is open. */
            prihdr = cpl_propertylist_load(filename, 0);
            if(prihdr == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load primary FITS "
                                             "header for %s",
                                             filename);
            }

            key = cpl_sprintf("ESO INS SHUT%d STAT", shuffle);
            if(key == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't format shutter FITS "
                                             "header key for %d",
                                             shuffle);
            }

            prestate = cpl_errorstate_get();

            if(qmost_cpl_propertylist_get_int(prihdr,
                                              key,
                                              &shut_open) != CPL_ERROR_NONE) {
                cpl_msg_warning(cpl_func,
                                "couldn't read simucal shutter status "
                                "%s[0]:%s (%s), assuming open",
                                filename,
                                key,
                                cpl_error_get_message());

                cpl_errorstate_set(prestate);

                fib_st = 1;
            }
            else {
                fib_st = shut_open ? 1 : 0;
            }

            cpl_free(key);
            key = NULL;

            cpl_propertylist_delete(prihdr);
            prihdr = NULL;

            /* Insert blanks for the simucal fibres at start, set FIB_USE=0,
               and FIB_ROOT and SLIT_POS to appropriate values. */
            if(qmost_cpl_table_insert_blank_window(
                   shuftab,
                   0,
                   nsimucal) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to insert %d rows for "
                                             "simucal fibres at %d",
                                             nsimucal, 0);
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_ID",
                                                0,
                                                nsimucal,
                                                0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_ID for "
                                             "simucal fibres");
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_ST",
                                                0,
                                                nsimucal,
                                                fib_st) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_ST for "
                                             "simucal fibres");
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_USE",
                                                0,
                                                nsimucal,
                                                0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_USE for "
                                             "simucal fibres");
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_ROOT",
                                                0,
                                                nsimucal,
                                                shuffle) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_ROOT for "
                                             "simucal fibres");
            }

            for(row = 0; row < nsimucal; row++) {
                if(cpl_table_set_int(shuftab,
                                     "SLIT_POS",
                                     row,
                                     row + 1) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set SLIT_POS for "
                                                 "simucal fibre %d",
                                                 row + 1);
                }
            }

            /* Insert blanks for the simucal fibres at end, set FIB_USE=0
               for simucal, and FIB_ROOT and SLIT_POS to appropriate
               values. */
            nrowstmp = cpl_table_get_nrow(shuftab);

            if(qmost_cpl_table_insert_blank_window(
                   shuftab,
                   nrowstmp,
                   nsimucal) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to insert %d rows for "
                                             "simucal fibres at %lld",
                                             nsimucal, nrowstmp);
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_ID",
                                                nrowstmp,
                                                nsimucal,
                                                0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_ID for "
                                             "simucal fibres");
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_ST",
                                                nrowstmp,
                                                nsimucal,
                                                fib_st) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_ST for "
                                             "simucal fibres");
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_USE",
                                                nrowstmp,
                                                nsimucal,
                                                0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_USE for "
                                             "simucal fibres");
            }

            if(cpl_table_fill_column_window_int(shuftab,
                                                "FIB_ROOT",
                                                nrowstmp,
                                                nsimucal,
                                                shuffle) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set FIB_ROOT for "
                                             "simucal fibres");
            }

            for(row = 0; row < nsimucal; row++) {
                if(cpl_table_set_int(shuftab,
                                     "SLIT_POS",
                                     nrowstmp + row,
                                     nrowstmp + row + 1) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set SLIT_POS for "
                                                 "simucal fibre %lld",
                                                 nrowstmp + row + 1);
                }
            }
        }

        *tab = shuftab;

        cpl_table_delete(intab);
        intab = NULL;
    }
    else {
        *tab = intab;
    }

    if(mettab != NULL) {
        cpl_table_delete(mettab);
        mettab = NULL;
    }

    *hdr = inhdr;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Save the FIBINFO table to an output frame.
 *
 * @param   intab      (Given)    FIBINFO table to save.
 * @param   inhdr      (Given)    The corresponding FITS header.
 * @param   outframe   (Given)    Output frame.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no filename set in
 *                                    the output frame.
 * @retval  CPL_ERROR_FILE_NOT_CREATED  If the file can't be written
 *                                      or isn't a valid FITS file.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @par Output FITS Headers:
 *   - <b>EXTNAME</b>
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibtabsave (
    cpl_table *intab,
    cpl_propertylist *inhdr,
    const cpl_frame *outframe)
{
    cpl_propertylist *outhdr = NULL;
    const char *filename;

#undef TIDY
#define TIDY                                    \
    if(outhdr) {                                \
        cpl_propertylist_delete(outhdr);        \
        outhdr = NULL;                          \
    }

    /* Check required inputs and outputs */
    cpl_ensure_code(intab, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(outframe, CPL_ERROR_NULL_INPUT);

    /* Make output header */
    outhdr = cpl_propertylist_new();

    /* Set EXTNAME */
    if(cpl_propertylist_update_string(outhdr,
                                      "EXTNAME",
                                      QMOST_FIBINFO_EXTNAME) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set FITS header EXTNAME = %s",
                                     QMOST_FIBINFO_EXTNAME);
    }

    /* Copy in other headers */
    if(inhdr != NULL) {
        if(cpl_propertylist_copy_property_regexp(outhdr,
                                                 inhdr,
                                                 QMOST_REGEXP_HDUCOPY,
                                                 1) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not copy input file headers");
        }
    }

    /* Get filename */
    filename = cpl_frame_get_filename(outframe);
    if(filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    /* Write out table */
    if(cpl_table_save(intab,
                      NULL,
                      outhdr,
                      filename,
                      CPL_IO_EXTEND) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not save %s table extension to %s",
                                     QMOST_FIBINFO_EXTNAME,
                                     filename);
    }

    cpl_propertylist_delete(outhdr);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create a dummy FIBINFO table.
 *
 * A minimal dummy FIBINFO table is created with FIB_ID and FIB_ST
 * columns so it can be used to emit fibre flat normalisation
 * information for OB or SKY fibre flats when processing raw files
 * without a FIBINFO table (lab data).
 *
 * @param   nfib       (Given)    The number of fibres.
 * @param   tab        (Returned) The new table.
 * @param   hdr        (Returned) FITS header information.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  No FIBINFO extension was found
 *                                    in the input file.
 *
 * @par Output FIBINFO Table Columns:
 *   - <b>FIB_ID</b>: A dummy fibre ID, where the fibres are simply
 *     assigned a fibre ID equal to the row number in the table.
 *   - <b>FIB_ST</b>: Dummy fibre status, flagging all fibres as
 *     illuminated (allocated and functional).
 *   - <b>FIB_USE</b>: Dummy fibre use, flagging all fibres as
 *     science.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibtab_dummy (
    int nfib,
    cpl_table **tab,
    cpl_propertylist **hdr)
{
    int ifib;

    cpl_ensure_code(tab != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(hdr != NULL, CPL_ERROR_NULL_INPUT);

    *tab = NULL;
    *hdr = NULL;

#undef TIDY
#define TIDY                                    \
    if(*tab != NULL) {                          \
        cpl_table_delete(*tab);                 \
        *tab = NULL;                            \
    }                                           \
    if(*hdr != NULL) {                          \
        cpl_propertylist_delete(*hdr);          \
        *hdr = NULL;                            \
    }

    /* Create outputs */
    *tab = cpl_table_new(nfib);
    *hdr = cpl_propertylist_new();

    /* Create and populate FIB_ID, FIB_ST, FIB_USE */
    cpl_table_new_column(*tab, "FIB_ID", CPL_TYPE_INT);
    cpl_table_new_column(*tab, "FIB_ST", CPL_TYPE_INT);
    cpl_table_new_column(*tab, "FIB_USE", CPL_TYPE_INT);

    for(ifib = 0; ifib < nfib; ifib++) {
        cpl_table_set_int(*tab, "FIB_ID", ifib, ifib+1);
        cpl_table_set_int(*tab, "FIB_ST", ifib, 2);
        cpl_table_set_int(*tab, "FIB_USE", ifib, 1);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Add the extra pipeline-generated FIBINFO table columns.
 *
 * @param   in_tbl     (Given)    Input FIBINFO table.
 * @param   pri_hdr    (Given)    Input FITS primary header, used to
 *                                retrieve the telescope location and
 *                                pointing information.
 * @param   exptime    (Given)    The total exposure time in seconds.
 * @param   mjdmid     (Given)    The MJD of the exposure midpoint.
 * @param   out_tbl    (Returned) Output FIBINFO table.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If one of the required input
 *                                    FITS headers was not found.
 * @retval  CPL_ERROR_INVALID_TYPE    If one of the output float or
 *                                    double FIBINFO columns already
 *                                    exists in the table with an
 *                                    incompatible data type.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If one of the required input
 *                                    FITS headers has an incorrect
 *                                    data type, or if one of the
 *                                    output integer FIBINFO columns
 *                                    already exists in the table with
 *                                    an incompatible data type.
 *
 * @note    The table is recreated in order to force the order of the
 *          columns written to the FITS file to be correct (per the
 *          specification defined by the ICDs and the DRPD) because
 *          CPL does not provide any other means to control the order
 *          of the columns in the table.
 *
 * @par Input FITS Header Information:
 *   - <b>DEC</b>
 *   - <b>ESO TEL GEOELEV</b>
 *   - <b>ESO TEL GEOLAT</b>
 *   - <b>ESO TEL GEOLON</b>
 *   - <b>RA</b>
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_USE</b>
 *   - <b>OBJ_DEC</b>
 *   - <b>OBJ_RA</b>
 *
 * @par Output FIBINFO Table Columns:
 *   - <b>NSPEC</b>: The number of the spectrum in extracted order,
 *     numbering from 1 for the first spectrum in the spatial
 *     direction.
 *   - <b>HELIO_COR</b> (km/s): The barycentric velocity correction
 *     for the object in this fibre.
 *   - <b>EXPTIME</b> (s): The total exposure time for the object.
 *
 * @par Created FIBINFO Table Columns Populated Elsewhere:
 *   - <b>NLIN_ARC_R</b>
 *   - <b>NLIN_ARC_G</b>
 *   - <b>NLIN_ARC_B</b>
 *   - <b>RMS_ARC_R</b>
 *   - <b>RMS_ARC_G</b>
 *   - <b>RMS_ARC_B</b>
 *   - <b>FWHM_R</b>
 *   - <b>FWHM_G</b>
 *   - <b>FWHM_B</b>
 *   - <b>WAVE_COR_R</b>
 *   - <b>WAVE_COR_G</b>
 *   - <b>WAVE_COR_B</b>
 *   - <b>WAVE_CORRMS_R</b>
 *   - <b>WAVE_CORRMS_G</b>
 *   - <b>WAVE_CORRMS_B</b>
 *   - <b>SNR_R</b>
 *   - <b>SNR_G</b>
 *   - <b>SNR_B</b>
 *   - <b>MEANFLUX_R</b>
 *   - <b>MEANFLUX_G</b>
 *   - <b>MEANFLUX_B</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibtab_newcols (
    cpl_table *in_tbl,
    cpl_propertylist *pri_hdr,
    double exptime,
    double mjdmid,
    cpl_table **out_tbl)
{
    char *ttype[] = {"NLIN_ARC_R","NLIN_ARC_G","NLIN_ARC_B",
                     "RMS_ARC_R","RMS_ARC_G","RMS_ARC_B",
                     "FWHM_R","FWHM_G","FWHM_B",
                     "HELIO_COR",
                     "WAVE_COR_R","WAVE_CORRMS_R",
                     "WAVE_COR_G","WAVE_CORRMS_G",
                     "WAVE_COR_B","WAVE_CORRMS_B",
                     "EXPTIME",
                     "SNR_R","SNR_G","SNR_B",
                     "MEANFLUX_R","MEANFLUX_G","MEANFLUX_B"};
    char *tunit[] = {"","","",
                     "Angstrom","Angstrom","Angstrom",
                     "Angstrom","Angstrom","Angstrom",
                     "km/s",
                     "Angstrom","Angstrom",
                     "Angstrom","Angstrom",
                     "Angstrom","Angstrom",
                     "s",
                     "","","",
                     "ADU","ADU","ADU"};
    cpl_type cpltype[] = {CPL_TYPE_INT,CPL_TYPE_INT,CPL_TYPE_INT,
                          CPL_TYPE_DOUBLE,CPL_TYPE_DOUBLE,CPL_TYPE_DOUBLE,
                          CPL_TYPE_FLOAT,CPL_TYPE_FLOAT,CPL_TYPE_FLOAT,
                          CPL_TYPE_FLOAT,
                          CPL_TYPE_DOUBLE,CPL_TYPE_DOUBLE,
                          CPL_TYPE_DOUBLE,CPL_TYPE_DOUBLE,
                          CPL_TYPE_DOUBLE,CPL_TYPE_DOUBLE,
                          CPL_TYPE_FLOAT,
                          CPL_TYPE_FLOAT,CPL_TYPE_FLOAT,CPL_TYPE_FLOAT,
                          CPL_TYPE_FLOAT,CPL_TYPE_FLOAT,CPL_TYPE_FLOAT};
    int inew, nnew;
    int irow, nrows, icol, ncols;

    cpl_table *eop_tbl = NULL;

    cpl_array *colnames = NULL;
    const char *colname;

    double latitude, longitude, height;
    double point_ra, point_dec;
    int point_valid, badcoords;

    double ra, dec, velcor;
    int fib_use, isnull, ra_isnull, dec_isnull;

    double stpd, ctpd, sd, cd, s, c, denom, xi, xn, r;

    cpl_ensure_code(in_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pri_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_tbl != NULL, CPL_ERROR_NULL_INPUT);

    *out_tbl = NULL;

#undef TIDY
#define TIDY                                    \
    if(*out_tbl != NULL) {                      \
       cpl_table_delete(*out_tbl);              \
       *out_tbl = NULL;                         \
    }                                           \
    if(colnames != NULL) {                      \
        cpl_array_delete(colnames);             \
        colnames = NULL;                        \
    }                                           \
    if(eop_tbl != NULL) {                       \
        cpl_table_delete(eop_tbl);              \
        eop_tbl = NULL;                         \
    }

    /* Number of new columns */
    nnew = sizeof(ttype) / sizeof(ttype[0]);

    /* Telescope location */
    if(cpl_propertylist_has(pri_hdr, "ESO TEL GEOLAT")) {
        if(qmost_cpl_propertylist_get_double(pri_hdr,
                                             "ESO TEL GEOLAT",
                                             &latitude) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO TEL GEOLAT");
        }
    }
    else {
        latitude = -24.6157;
        cpl_msg_warning(cpl_func,
                        "telescope latitude missing, assuming default = %.4f",
                        latitude);
    }

    if(cpl_propertylist_has(pri_hdr, "ESO TEL GEOLON")) {
        if(qmost_cpl_propertylist_get_double(pri_hdr,
                                             "ESO TEL GEOLON",
                                             &longitude) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO TEL GEOLON");
        }
    }
    else {
        longitude = -70.3976;
        cpl_msg_warning(cpl_func,
                        "telescope longitude missing, assuming default = %.4f",
                        longitude);
    }

    if(cpl_propertylist_has(pri_hdr, "ESO TEL GEOELEV")) {
        if(qmost_cpl_propertylist_get_double(pri_hdr,
                                             "ESO TEL GEOELEV",
                                             &height) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO TEL GEOELEV");
        }
    }
    else {
        height = 2530;
        cpl_msg_warning(cpl_func,
                        "telescope elevation missing, assuming default = %.0f",
                        height);
    }

    /* These headers are in the data model, but not the simulations at
     * present.  ESO TEL TARG ALPHA and DELTA are populated in both,
     * but as far as I can tell, it's impossible to read the bizarre
     * format of these correctly using cpl_propertylist.  They are
     * base-60, so must be read as strings, but are missing the quotes
     * so get automatically typed as doubles by cpl_propertylist_load.
     * This could potentially corrupt the value (because rounding in
     * base 10 is in general not correct in base 60).  Lacking a way
     * to use these, and since they're only needed to provide a
     * fallback for objects where the FIBINFO OBJ_RA/DEC are NULL, we
     * just let these objects have null HELIO_COR in QC if the
     * following headers are missing. */
    if(cpl_propertylist_has(pri_hdr, "RA") &&
       cpl_propertylist_has(pri_hdr, "DEC")) {
        if(qmost_cpl_propertylist_get_double(pri_hdr,
                                             "RA",
                                             &point_ra) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read RA");
        }

        if(qmost_cpl_propertylist_get_double(pri_hdr,
                                             "DEC",
                                             &point_dec) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read DEC");
        }

        stpd = sin(point_dec * CPL_MATH_RAD_DEG);
        ctpd = cos(point_dec * CPL_MATH_RAD_DEG);

        point_valid = 1;
    }
    else {
        point_valid = 0;

        /* These aren't actually used but some compilers might not be
         * smart enough to figure it out so we initialize them here to
         * ensure there isn't a warning. */
        stpd = 0;
        ctpd = 0;
    }

    /* The Earth orientation parameter table is not needed at this
     * precision level.  The input is mandatory, so here we patch it
     * out by giving a table of zeros so the interpolated results are
     * all zero. */
    eop_tbl = cpl_table_new(2);

    cpl_table_new_column(eop_tbl, "MJD", CPL_TYPE_DOUBLE);
    cpl_table_new_column(eop_tbl, "PMX", CPL_TYPE_DOUBLE);
    cpl_table_new_column(eop_tbl, "PMY", CPL_TYPE_DOUBLE);
    cpl_table_new_column(eop_tbl, "DUT", CPL_TYPE_DOUBLE);

    cpl_table_set(eop_tbl, "MJD", 0, mjdmid - 2);
    cpl_table_set(eop_tbl, "MJD", 1, mjdmid + 2);
    
    cpl_table_fill_column_window(eop_tbl, "PMX", 0, 2, 0);
    cpl_table_fill_column_window(eop_tbl, "PMY", 0, 2, 0);
    cpl_table_fill_column_window(eop_tbl, "DUT", 0, 2, 0);

    /* Create output table */
    nrows = cpl_table_get_nrow(in_tbl);

    *out_tbl = cpl_table_new(nrows);

    /* Create NSPEC in the first position */
    if(!cpl_table_has_column(*out_tbl, "NSPEC")) {
        if(cpl_table_new_column(*out_tbl,
                                "NSPEC",
                                CPL_TYPE_INT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not create column NSPEC");
        }
    }

    for(irow = 0; irow < nrows; irow++) {
        if(cpl_table_set_int(*out_tbl,
                             "NSPEC",
                             irow,
                             irow+1) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set NSPEC for row %d",
                                         irow+1);
        }
    }

    /* Copy the other columns from the input */
    ncols = cpl_table_get_ncol(in_tbl);

    colnames = cpl_table_get_column_names(in_tbl);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);

        if(cpl_table_duplicate_column(*out_tbl,
                                      colname,
                                      in_tbl,
                                      colname) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not duplicate column %d",
                                         icol+1);
        }
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    /* Create the remaining columns we're supposed to put at the end */
    for(inew = 0; inew < nnew; inew++) {
        if(!cpl_table_has_column(*out_tbl, ttype[inew])) {
            if(cpl_table_new_column(*out_tbl,
                                    ttype[inew],
                                    cpltype[inew]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create column "
                                             "%s",
                                             ttype[inew]);
            }
            
            if(cpl_table_set_column_unit(*out_tbl,
                                         ttype[inew],
                                         tunit[inew]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set units for column "
                                             "%s to %s",
                                             ttype[inew],
                                             tunit[inew]);
            }
        }

        if(cpltype[inew] == CPL_TYPE_INT) {
            /* Fill with zeros */
            if(cpl_table_fill_column_window_int(*out_tbl,
                                                ttype[inew],
                                                0,
                                                nrows,
                                                0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not zero fill int "
                                             "column %s",
                                             ttype[inew]);
            }
        }
    }

    /* Define column names */
#ifdef OPR2
#define COLNAME_OBJ_RA "TRG_RA"
#define COLNAME_OBJ_DEC "TRG_DEC"
#else
#define COLNAME_OBJ_RA "OBJ_RA"
#define COLNAME_OBJ_DEC "OBJ_DEC"
#endif

    /* Loop over objects */
    for(irow = 0; irow < nrows; irow++) {
        /* Set EXPTIME */
        if(cpl_table_set(*out_tbl,
                         "EXPTIME",
                         irow,
                         exptime) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set EXPTIME for row %d",
                                         irow+1);
        }

        /* Check FIB_USE */
        fib_use = cpl_table_get_int(*out_tbl,
                                    "FIB_USE",
                                    irow,
                                    &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read %s for row %d",
                                         "FIB_USE", irow+1);
        }
        else if(isnull > 0) {
            fib_use = 1;
        }

        /* If this is a simucal fibre, set to zero and skip the rest */
        if(fib_use == 0) {
            if(cpl_table_set(*out_tbl,
                             "HELIO_COR",
                             irow,
                             0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set HELIO_COR "
                                             "for row %d",
                                             irow+1);
            }

            continue;
        }

        /* Get coordinates */
        ra = cpl_table_get(*out_tbl,
                           COLNAME_OBJ_RA,
                           irow,
                           &ra_isnull);
        if(ra_isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read %s for row %d",
                                         COLNAME_OBJ_RA, irow+1);
        }

        dec = cpl_table_get(*out_tbl,
                            COLNAME_OBJ_DEC,
                            irow,
                            &dec_isnull);
        if(dec_isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read %s for row %d",
                                         COLNAME_OBJ_RA, irow+1);
        }

        /* Check coordinates */
        badcoords = 0;

        if(ra_isnull == 0 && dec_isnull == 0) {
            if(point_valid) {
                /* Compute standard coordinates in degrees */
                sd = sin(dec * CPL_MATH_RAD_DEG);
                cd = cos(dec * CPL_MATH_RAD_DEG);
                
                s = sin((ra - point_ra) * CPL_MATH_RAD_DEG);
                c = cos((ra - point_ra) * CPL_MATH_RAD_DEG);
                
                denom = (stpd * sd + ctpd * cd * c) * CPL_MATH_RAD_DEG;
                
                xi = s * cd / denom;
                xn = (ctpd * sd - stpd * cd * c) / denom;
                
                r = hypot(xi, xn);
                
                if(r > 5.0) {
                    /* Too far off */
                    cpl_msg_warning(cpl_func,
                                    "coords for %d seem bad at r=%.3f",
                                    irow+1, r);
                    badcoords = 1;
                }
            }
            /* else, cannot check, so assume good */
        }
        else {
            /* One or both of them null.  This happens with the
             * simucal fibres so the level of the message is info so
             * as to avoid creating unnecessary alarm. */
            cpl_msg_info(cpl_func,
                         "coords for %d are NULL",
                         irow+1);

            badcoords = 1;
        }

        /* Can we fix them if bad? */
        if(badcoords && point_valid) {
            ra = point_ra;
            dec = point_dec;
            badcoords = 0;
        }

        /* If we got usable coordinates, proceed to calculation of
         * barycentric correction. */
        if(!badcoords) {
            /* IMO, since these are angles and should only be used as
             * the arguments of periodic trigonometric functions like
             * sin and cos in the calculation, values outside of the
             * nominal ranges are valid, so here we make sure they are
             * wrapped appropriately such that they will always pass
             * the range checks in the hdrl_barycorr_compute function.
             * Arguably we should also do this to latitude and
             * longitude, but there's something wrong with the input
             * file headers if these aren't in range so I left them as
             * they were. */ 
            ra = ra >= 0 ? fmod(ra, 360) : fmod(fmod(ra, 360) + 360, 360);
            dec = remainder(dec, 360.0);
            
            /* Now compute barycentric correction.  Refraction isn't
             * used so the values of those parameters don't matter.
             * This is missing some relativistic terms, so doesn't
             * agree perfectly with my routine, but the results are
             * fine for present purposes.  Note that this is
             * barycentric, not heliocentric (as in 4L1).  We
             * therefore set SPECSYS accordingly in the main recipe .
             * I didn't change the FIBINFO column name to avoid
             * breaking the data model, and it's close enough anyway. */
            if(hdrl_barycorr_compute(ra, dec,
                                     eop_tbl,
                                     mjdmid, 0,
                                     longitude, latitude, height,
                                     1013.25, 0.0, 0.5, 0.55,
                                     &velcor) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not compute barycorr "
                                             "for row %d", irow+1);
            }

            /* Write out */
            if(cpl_table_set(*out_tbl,
                             "HELIO_COR",
                             irow,
                             velcor / 1000.0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set HELIO_COR "
                                             "for row %d",
                                             irow+1);
            }
        }
    }

    cpl_table_delete(eop_tbl);
    eop_tbl = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Add the arc QC information for each arm to FIBINFO.
 *
 * @param   fibinfo_tbl     (Modified) FIBINFO table.
 * @param   arm             (Given)    One of the QMOST_ARM_*
 *                                     constants specifying which arm
 *                                     we're doing.
 * @param   master_wave_tbl (Given)    The master wavelength
 *                                     solution table.
 * @param   ob_wave_tbl     (Given)    The OB wavelength solution
 *                                     table, or NULL if none.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the parameter arm is invalid,
 *                                    or one of the waveinfo tables is
 *                                    invalid.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the numbers of fibres in
 *                                        the master and OB-level
 *                                        wavelength solutions don't
 *                                        match.
 * @retval  CPL_ERROR_INVALID_TYPE    If one of the output FIBINFO
 *                                    columns already exists in the
 *                                    table with an incompatible data
 *                                    type.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required input or
 *                                    output pointers was NULL.
 *
 * @par Output FIBINFO Table Columns:
 *   - <b>NLIN_ARC_a</b>: The number of good arc lines used in the
 *     wavelength solution for this fibre in arm "a" (R, G or B,
 *    depending on the value of the parameter arm).
 *   - <b>RMS_ARC_a</b> (A): The RMS residual from the wavelength
 *     solution polynomial fit in arm "a" (R, G or B, depending on the
 *     value of the parameter arm).
 *   - <b>FWHM_a</b> (A): The mean FWHM (in the spectral direction) of
 *     the arc lines in this fibre in arm "a" (R, G or B, depending on
 *     the value of the parameter arm).
 *   - <b>WAVE_COR_a</b> (A): The wavelength offset applied to the
 *     master wavelength solution in arm "a" (R, G or B, depending on
 *     the value of the parameter arm) based on the OB-level
 *     wavelength solution, if ob_wave_tbl is given.  Otherwise,
 *     NULL.
 *   - <b>WAVE_CORRMS_a</b> (A): The corresponding RMS of the
 *     wavelength offset in arm "a" (R, G or B, depending on the value
 *     of the parameter arm).
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibtab_arcqc (
    cpl_table *fibinfo_tbl,
    int arm,
    cpl_table *master_wave_tbl,
    cpl_table *ob_wave_tbl)
{
    const char *arm_extname;
    char arm_ltr;
    int irow, nrows;

    int master_nwv = 0;
    qmost_waveinfo *master_wv = NULL;
    int ob_nwv = 0;
    qmost_waveinfo *ob_wv = NULL;

    enum {
        COL_NLIN_ARC = 0,
        COL_RMS_ARC,
        COL_FWHM,
        COL_WAVE_COR,
        COL_WAVE_CORRMS,
        NCOLS
    };

    char *colnames[NCOLS];
    int icol, icolt, has;

    char *tunit[NCOLS] = { "",
                           "Angstrom",
                           "Angstrom",
                           "Angstrom",
                           "Angstrom" };
    cpl_type cpltype[NCOLS] = { CPL_TYPE_INT,
                                CPL_TYPE_DOUBLE,
                                CPL_TYPE_FLOAT,
                                CPL_TYPE_DOUBLE,
                                CPL_TYPE_DOUBLE };

    float *fwhmbuf = NULL;
    int iline, nmed;

    cpl_errorstate prestate;
    float medfwhm;

    double offset;
    cpl_size degree;

    cpl_ensure_code(fibinfo_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(master_wave_tbl != NULL, CPL_ERROR_NULL_INPUT);

    for(icol = 0; icol < NCOLS; icol++) {
        colnames[icol] = NULL;
    }

#undef TIDY
#define TIDY                                    \
    if(master_wv != NULL) {                     \
        qmost_wvclose(master_nwv, &master_wv);  \
        master_wv = NULL;                       \
        master_nwv = 0;                         \
    }                                           \
    if(ob_wv != NULL) {                         \
        qmost_wvclose(ob_nwv, &ob_wv);          \
        ob_wv = NULL;                           \
        ob_nwv = 0;                             \
    }                                           \
    for(icolt = 0; icolt < NCOLS; icolt++) {    \
        if(colnames[icolt] != NULL) {           \
            cpl_free(colnames[icolt]);          \
            colnames[icolt] = NULL;             \
        }                                       \
    }                                           \
    if(fwhmbuf != NULL) {                       \
        cpl_free(fwhmbuf);                      \
        fwhmbuf = NULL;                         \
    }

    /* Get the appropriate letter for the arm */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not determine EXTNAME "
                                     "for arm %d", arm);
    }
    
    arm_ltr = arm_extname[0];

    /* Form column names */
#undef MAKECOL
#define MAKECOL(number, name)                                           \
    colnames[number] = cpl_sprintf(name "_%c", arm_ltr)

    MAKECOL(COL_NLIN_ARC, "NLIN_ARC");
    MAKECOL(COL_RMS_ARC, "RMS_ARC");
    MAKECOL(COL_FWHM, "FWHM");
    MAKECOL(COL_WAVE_COR, "WAVE_COR");
    MAKECOL(COL_WAVE_CORRMS, "WAVE_CORRMS");

    for(icol = 0; icol < NCOLS; icol++) {
        if(colnames[icol] == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format column "
                                         "names for arm %d", arm);
        }

        /* Is column present? */
        has = cpl_table_has_column(fibinfo_tbl, colnames[icol]);
        if(has < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not check column "
                                         "exists");
        }
        else if(has == 0) {
            /* Not present, create it */
            if(cpl_table_new_column(fibinfo_tbl,
                                    colnames[icol],
                                    cpltype[icol]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create column %d",
                                             icol+1);
            }
            
            if(cpl_table_set_column_unit(fibinfo_tbl,
                                         colnames[icol],
                                         tunit[icol]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set units for column "
                                             "%d to %s",
                                             icol+1,
                                             tunit[icol]);
            }
        }
    }

    /* Load master wavelength solution */
    if(qmost_wvopen(master_wave_tbl,
                    &master_nwv,
                    &master_wv) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load wavelength "
                                     "solution for arm %d", arm);
    }

    /* Load OB wavelength solution, if given */
    if(ob_wave_tbl != NULL) {
        if(qmost_wvopen(ob_wave_tbl, &ob_nwv, &ob_wv) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not load wavelength "
                                         "solution for arm %d", arm);
        }

        if(master_nwv != ob_nwv) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "master and OB wavelength "
                                         "solution numbers of fibres "
                                         "don't match: %d != %d",
                                         master_nwv, ob_nwv);
        }
    }

    /* For each spectrum: */
    nrows = qmost_min(cpl_table_get_nrow(fibinfo_tbl), master_nwv);

    for(irow = 0; irow < nrows; irow++) {
        /* Add arc fit diagnostics */
        if(cpl_table_set_int(fibinfo_tbl,
                             colnames[COL_NLIN_ARC],
                             irow,
                             master_wv[irow].ngood) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set NLIN_ARC_%c "
                                         "for row %d",
                                         arm_ltr, irow);
        }
        cpl_table_set(fibinfo_tbl, colnames[COL_RMS_ARC], irow,
                      master_wv[irow].fit_rms);

        /* FWHM, aka spectral resolution, using good lines from fit */
        if(master_wv[irow].nlines > 0) {
            fwhmbuf = cpl_malloc(master_wv[irow].nlines * sizeof(float));
                
            nmed = 0;

            for(iline = 0; iline < master_wv[irow].nlines; iline++) {
                if(master_wv[irow].fit_flag[iline] == 0) {
                    fwhmbuf[nmed] = master_wv[irow].fwhm[iline];
                    nmed++;
                }
            }
            
            if(nmed > 0) {
                prestate = cpl_errorstate_get();

                if(qmost_med(fwhmbuf, NULL, nmed,
                             &medfwhm) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                    medfwhm = 0;
                }

                cpl_table_set(fibinfo_tbl, colnames[COL_FWHM], irow,
                              medfwhm);
            }

            cpl_free(fwhmbuf);
            fwhmbuf = NULL;
        }

        /* OB wavelength solution offset and RMS, if available */
        if(ob_wv != NULL && ob_wv[irow].coefs != NULL) {
            /* Give offset at reference pixel as offset */
            degree = 0;
            offset = cpl_polynomial_get_coeff(ob_wv[irow].coefs, &degree);

            cpl_table_set(fibinfo_tbl, colnames[COL_WAVE_COR], irow,
                          offset);

            /* RMS is the fit RMS */
            cpl_table_set(fibinfo_tbl, colnames[COL_WAVE_CORRMS], irow,
                          ob_wv[irow].fit_rms);
        }
    }

    qmost_wvclose(master_nwv, &master_wv);
    master_wv = NULL;
    master_nwv = 0;

    if(ob_wv != NULL) {
        qmost_wvclose(ob_nwv, &ob_wv);
        ob_wv = NULL;
        ob_nwv = 0;
    }

    for(icol = 0; icol < NCOLS; icol++) {
        if(colnames[icol] != NULL) {
            cpl_free(colnames[icol]);
            colnames[icol] = NULL;
        }
    }

    return CPL_ERROR_NONE;
}

/**@}*/
