/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_polynm.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_polynm    qmost_polynm
 *
 * Polynomial fitting convenience routines
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Fit simple 1D polynomial.
 *
 * Fit a simple 1D polynomial to arrays of x and y.  Wraps CPL routines.
 *
 * @param   xp          (Given)    x data points.
 * @param   yp          (Given)    y data points.
 * @param   npts        (Given)    Number of data points.
 * @param   ncoefs      (Given)    Number of coefficients = degree of
 *                                 polynomial + 1.
 * @param   ilim        (Given)    The lowest degree coefficient to be
 *                                 fit.  If ilim = 0, then all
 *                                 coefficients will be fit. If ilim =
 *                                 1, then the fit will start with the
 *                                 linear term and the constant term
 *                                 will be assumed to be zero.
 *
 * @return  cpl_polynomial  The resulting polynomial.
 *
 * @retval  NULL            If an error occurred.
 *
 * @par CPL error codes raised
 *   - CPL_ERROR_DATA_NOT_FOUND if there were no data or not enough
 *     data (less data points than coefficients) for the polynomial
 *     fit.
 *   - CPL_ERROR_ILLEGAL_INPUT if the number of coefficients is less
 *     than 1.
 *   - CPL_ERROR_SINGULAR_MATRIX if the polynomial solution is
 *     singular (not enough distinct samples to fit).
 *
 * @note    In practical use, errors CPL_ERROR_DATA_NOT_FOUND and
 *          CPL_ERROR_SINGULAR_MATRIX must be trapped by the caller to
 *          prevent singular matrices from causing a crash.  In order
 *          to do this, the call needs to be preceeded by
 *          cpl_errorstate_get() to save the error state, then after
 *          NULL is returned cpl_error_get_code() can be used to check
 *          the error code and invoke cpl_errorstate_set() using the
 *          previously saved cpl_errorstate if either of these errors
 *          are returned to reset the error.  In case they are, some
 *          domain-specific recovery action should be taken, such as
 *          creating a polynomial with all coefficients equal to
 *          zero.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_polynomial *qmost_polynm (
    double *xp,
    double *yp,
    int npts,
    int ncoefs,
    int ilim)
{
    cpl_matrix *m_x = NULL;
    cpl_vector *v_y = NULL;
    cpl_polynomial *coefs = NULL;
    cpl_size mindeg, maxdeg;

#undef TIDY
#define TIDY                                    \
    if(coefs) {                                 \
        cpl_polynomial_delete(coefs);           \
        coefs = NULL;                           \
    }                                           \
    if(m_x) {                                   \
        cpl_matrix_unwrap(m_x);                 \
        m_x = NULL;                             \
    }                                           \
    if(v_y) {                                   \
        cpl_vector_unwrap(v_y);                 \
        v_y = NULL;                             \
    }

    /* Check number of coefficients is valid */
    if(ncoefs < 1) {
        TIDY;
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "invalid number of coefficients: %d",
                              ncoefs);
        return NULL;
    }

    /* Check we got some data */
    if(npts < 1) {
        TIDY;
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                              "no data for polynomial fit");
        return NULL;
    }

    /* Create the polynomial first */
    coefs = cpl_polynomial_new(1);

    /* Now wrap the inputs */
    m_x = cpl_matrix_wrap(1, npts, xp);
    if(m_x == NULL) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not wrap input xp");
        return NULL;
    }

    v_y = cpl_vector_wrap(npts, yp);
    if(v_y == NULL) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "could not wrap input yp");
        return NULL;
    }

    mindeg = ilim;
    maxdeg = ncoefs - 1;

    if(cpl_polynomial_fit(coefs,
                          m_x,
                          NULL,
                          v_y,
                          NULL,
                          0,
                          &mindeg,
                          &maxdeg) != CPL_ERROR_NONE) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "polynomial fit error");
        return NULL;
    }

    cpl_matrix_unwrap(m_x);
    m_x = NULL;

    cpl_vector_unwrap(v_y);
    v_y = NULL;

    return coefs;
}

/**@}*/
