/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#define _POSIX_C_SOURCE   200809L   /* For mkstemp()       */
#define _DARWIN_C_SOURCE

#define TESTSTR "This is not a FITS file.\n"

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_dfs_test  Unit test of qmost_dfs
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_dfs_set_groups(void)
{
    /* Simulate data */
    const char *const filename[] = {"rawb.fits",
                                    "rawd.fits",
                                    "rawdf.fits",
                                    "rawff.fits",
                                    "calb.fits",
                                    "calrb.fits",
                                    "calmb.fits",
                                    "calmd.fits",
                                    "calmdf.fits",
                                    "calrft.fits",
                                    "calsm.fits",
                                    "calft.fits",
                                    "calfm.fits",
                                    "calff.fits"};
    const char *const tag[] = {QMOST_RAW_BIAS,
                               QMOST_RAW_DARK,
                               QMOST_RAW_DETECTOR_FLAT,
                               QMOST_RAW_FIBRE_FLAT_DAY,
                               QMOST_CALIB_MASTER_BPM,
                               QMOST_CALIB_REFERENCE_BIAS,
                               QMOST_PRO_MASTER_BIAS,
                               QMOST_PRO_MASTER_DARK,
                               QMOST_PRO_MASTER_DETECTOR_FLAT,
                               QMOST_CALIB_REFERENCE_FIBRE_TRACE,
                               QMOST_CALIB_SLIT_MASK,
                               QMOST_PRO_FIBRE_TRACE,
                               QMOST_PRO_FIBRE_MASK,
                               QMOST_PROC_FIBRE_FLAT};
    cpl_frame_group const expected_group[] = {CPL_FRAME_GROUP_RAW,
                                              CPL_FRAME_GROUP_RAW,
                                              CPL_FRAME_GROUP_RAW,
                                              CPL_FRAME_GROUP_RAW,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB,
                                              CPL_FRAME_GROUP_CALIB};
    const size_t N = sizeof(filename) / sizeof(filename[0]);

    cpl_frameset *frames = cpl_frameset_new();
    cpl_error_code code;

    cpl_test_eq(sizeof(tag) / sizeof(tag[0]), N);
    cpl_test_eq(sizeof(expected_group) / sizeof(expected_group[0]), N);
    cpl_test_nonnull(frames);

    /* Test with invalid input */
    code = qmost_dfs_set_groups(NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Test with valid input */

    /* Call the function - first with an empty frameset */
    code = qmost_dfs_set_groups(frames);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for (size_t i = 0; i < N; i++) {
        cpl_frame *frame = cpl_frame_new();

        code = cpl_frame_set_filename(frame, filename[i]);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        code = cpl_frame_set_tag(frame, tag[i]);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        code = cpl_frameset_insert(frames, frame);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
    }

    /* Call the function */
    code = qmost_dfs_set_groups(frames);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Verify results */
    for (size_t i = 0; i < N; i++) {
        const cpl_frame *frame = cpl_frameset_get_position_const(frames, i);

        cpl_test_nonnull(frame);

        cpl_test_eq(cpl_frame_get_group(frame), expected_group[i]);
    }

    cpl_frameset_delete(frames);

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_product_filename
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_product_filename(void)
{
    char *filename = NULL;

    filename = qmost_dfs_product_filename(QMOST_SPEC_HRS, "TEST");
    cpl_test_nonnull(filename);

    cpl_test(!strcmp(filename, "QMOST_TEST_HRS.fits"));

    cpl_free(filename);
    filename = NULL;

    filename = qmost_dfs_product_filename(QMOST_SPEC_LRS_A, "REST");
    cpl_test_nonnull(filename);

    cpl_test(!strcmp(filename, "QMOST_REST_LRS-A.fits"));

    cpl_free(filename);
    filename = NULL;

    filename = qmost_dfs_product_filename(QMOST_SPEC_LRS_B, "LEST");
    cpl_test_nonnull(filename);

    cpl_test(!strcmp(filename, "QMOST_LEST_LRS-B.fits"));

    cpl_free(filename);
    filename = NULL;

    filename = qmost_dfs_product_filename(-1, "DUMMY");
    cpl_test_nonnull(filename);

    cpl_test(!strcmp(filename, "QMOST_DUMMY_UNKNOWN.fits"));

    cpl_free(filename);
    filename = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_setup_product
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_setup_product(void)
{
    cpl_frameset *frameset = NULL;
    cpl_parameterlist *parlist = NULL;
    const cpl_frame *frame = NULL;
    cpl_frame *dummyframe = NULL;
    char rawfile[] = "test-qmost_dfs_XXXXXX";
    char outfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *plist = NULL;
    cpl_propertylist *applist = NULL;

    /* Empty frameset */
    frameset = cpl_frameset_new();

    /* Empty parameter list */
    parlist = cpl_parameterlist_new();

    /* Create temporary FITS file for raw input.  Populate some
       minimal FITS headers. */
    fd = mkstemp(rawfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, "ORIGIN", "TEST");
    cpl_propertylist_update_string(plist, "TELESCOP", "TEST");
    cpl_propertylist_update_string(plist, "INSTRUME", "TEST");
    cpl_propertylist_update_string(plist, "OBJECT", "TEST");
    cpl_propertylist_update_double(plist, "RA", 0);
    cpl_propertylist_update_double(plist, "DEC", 0);
    cpl_propertylist_update_string(plist, "RADECSYS", "ICRS");
    cpl_propertylist_update_string(plist, "ORIGFILE", "SHOULD_BE_REMOVED");
    cpl_propertylist_save(plist, rawfile, CPL_IO_CREATE);
    cpl_propertylist_delete(plist);

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* NULL inputs */
    frame = qmost_dfs_setup_product(NULL, parlist,
                                    "TEST", outfile, "PRO.TEST",
                                    CPL_FRAME_TYPE_IMAGE,
                                    NULL);
    cpl_test_null(frame);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    frame = qmost_dfs_setup_product(frameset, parlist,
                                    NULL, outfile, "PRO.TEST",
                                    CPL_FRAME_TYPE_IMAGE,
                                    NULL);
    cpl_test_null(frame);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    frame = qmost_dfs_setup_product(frameset, parlist,
                                    "TEST", NULL, "PRO.TEST",
                                    CPL_FRAME_TYPE_IMAGE,
                                    NULL);
    cpl_test_null(frame);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    frame = qmost_dfs_setup_product(frameset, parlist,
                                    "TEST", outfile, NULL,
                                    CPL_FRAME_TYPE_IMAGE,
                                    NULL);
    cpl_test_null(frame);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Empty SOF */
    frame = qmost_dfs_setup_product(frameset, parlist,
                                    "TEST", outfile, "PRO.TEST",
                                    CPL_FRAME_TYPE_IMAGE,
                                    NULL);
    cpl_test_null(frame);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* Populated SOF */
    dummyframe = cpl_frame_new();
    cpl_frame_set_filename(dummyframe, rawfile);
    cpl_frame_set_tag(dummyframe, "TEST");
    cpl_frame_set_type(dummyframe, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_group(dummyframe, CPL_FRAME_GROUP_RAW);
    cpl_frameset_insert(frameset, dummyframe);
    dummyframe = NULL;

    frame = qmost_dfs_setup_product(frameset, parlist,
                                    "TEST", outfile, "PRO.TEST",
                                    CPL_FRAME_TYPE_IMAGE,
                                    NULL);
    cpl_test_nonnull(frame);
    cpl_test_error(CPL_ERROR_NONE);

    /* Rudimentary check of processed header */
    plist = cpl_propertylist_load(outfile, 0);
    cpl_test_nonnull(plist);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(cpl_propertylist_has(plist, "ORIGIN"), 1);
    cpl_test_eq_string(cpl_propertylist_get_string(plist, "ORIGIN"), "TEST");
    cpl_test_eq(cpl_propertylist_has(plist, "ORIGFILE"), 0);
    cpl_test_eq(cpl_propertylist_has(plist, "PIPEFILE"), 1);

    cpl_propertylist_delete(plist);

    /* With extra headers and science to test those branches */
    applist = cpl_propertylist_new();
    cpl_propertylist_update_string(applist, "USERDEF", "TEST");

    frame = qmost_dfs_setup_product(frameset, parlist,
                                    "TEST", outfile, QMOST_PRO_SCIENCE,
                                    CPL_FRAME_TYPE_IMAGE,
                                    applist);
    cpl_test_nonnull(frame);
    cpl_test_error(CPL_ERROR_NONE);

    plist = cpl_propertylist_load(outfile, 0);
    cpl_test_nonnull(plist);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(cpl_propertylist_has(plist, "ORIGIN"), 1);
    cpl_test_eq_string(cpl_propertylist_get_string(plist, "ORIGIN"), "TEST");
    cpl_test_eq(cpl_propertylist_has(plist, "ORIGFILE"), 0);
    cpl_test_eq(cpl_propertylist_has(plist, "PIPEFILE"), 1);
    cpl_test_eq(cpl_propertylist_has(plist, "USERDEF"), 1);
    cpl_test_eq_string(cpl_propertylist_get_string(plist, "USERDEF"), "TEST");

    cpl_propertylist_delete(plist);

    cpl_propertylist_delete(applist);
    applist = NULL;

    /* Clean up */
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    unlink(rawfile);
    unlink(outfile);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_setup_product_default
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_setup_product_default(void)
{
    const cpl_frame *frame = NULL;

    /* Here we test failure to cover the code path but ensure no file
     * gets saved.  It would have a fixed name and location, which I
     * believe isn't permitted in unit tests. */
    frame = qmost_dfs_setup_product_default(NULL, NULL,
                                            "TEST", QMOST_SPEC_HRS, "PRO.TEST",
                                            CPL_FRAME_TYPE_IMAGE,
                                            NULL);
    cpl_test_null(frame);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_setup_product_extension_header
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_setup_product_extension_header(void)
{
    cpl_propertylist *inhdr = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_propertylist *outhdr = NULL;

    /* Set up input header with some stuff to strip and some not */
    inhdr = cpl_propertylist_new();

    cpl_propertylist_update_bool(inhdr, "SIMPLE", 1);
    cpl_propertylist_update_string(inhdr, "EXTNAME", "OLD");
    cpl_propertylist_update_string(inhdr, "KEEP", "ME");

    /* QC with some extra */
    qclist = cpl_propertylist_new();

    cpl_propertylist_update_string(qclist, "QC", "TEST");

    /* NULL inputs */
    outhdr = qmost_dfs_setup_product_extension_header(inhdr, NULL, 0, NULL);
    cpl_test_null(outhdr);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Input header only */
    outhdr = qmost_dfs_setup_product_extension_header(inhdr, "NEW", 0, NULL);

    cpl_test_eq(cpl_propertylist_has(outhdr, "SIMPLE"), 0);
    cpl_test_eq_string(cpl_propertylist_get_string(outhdr, "EXTNAME"), "NEW");
    cpl_test_eq(cpl_propertylist_has(outhdr, "KEEP"), 1);

    cpl_propertylist_delete(outhdr);

    /* Also with QC */
    outhdr = qmost_dfs_setup_product_extension_header(inhdr, "NEW", 0, qclist);

    cpl_test_eq(cpl_propertylist_has(outhdr, "SIMPLE"), 0);
    cpl_test_eq_string(cpl_propertylist_get_string(outhdr, "EXTNAME"), "NEW");
    cpl_test_eq(cpl_propertylist_has(outhdr, "KEEP"), 1);
    cpl_test_eq(cpl_propertylist_has(outhdr, "QC"), 1);

    cpl_propertylist_delete(outhdr);

    cpl_propertylist_delete(inhdr);
    cpl_propertylist_delete(qclist);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_save_image_extension
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_save_image_extension(void)
{
    cpl_frame *frame = NULL;
    char outfile[] = "test-qmost_dfs_XXXXXX";
    char notafitsfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *dummy = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_image *image = NULL;
    cpl_image *imchk = NULL;
    cpl_error_code code;

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create FITS file with dummy PHDU */
    dummy = cpl_propertylist_new();
    cpl_propertylist_save(dummy, outfile, CPL_IO_CREATE);
    cpl_propertylist_delete(dummy);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* Empty header */
    inhdr = cpl_propertylist_new();

    /* Test image */
    image = cpl_image_fill_test_create(512, 256);

    /* NULL inputs */
    code = qmost_dfs_save_image_extension(NULL, inhdr, "TEST", NULL,
                                          image, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_dfs_save_image_extension(frame, inhdr, NULL, NULL,
                                          image, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set in frame */
    code = qmost_dfs_save_image_extension(frame, inhdr, "TEST", NULL,
                                          image, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid inputs, should save image */
    cpl_frame_set_filename(frame, outfile);

    code = qmost_dfs_save_image_extension(frame, inhdr, "TEST", NULL,
                                          image, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check we can load it */
    imchk = cpl_image_load(outfile, CPL_TYPE_DOUBLE, 0, 1);
    cpl_test_nonnull(imchk);

    cpl_test_image_abs(image, imchk, 0.1);

    cpl_image_delete(imchk);

    /* Test saving a dummy extension */
    code = qmost_dfs_save_image_extension(frame, inhdr, "DUMMY", NULL,
                                          NULL, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_frame_delete(frame);
    unlink(outfile);

    /* Now try an invalid file */
    fd = mkstemp(notafitsfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0) {
        if(write(fd, TESTSTR, strlen(TESTSTR)) != strlen(TESTSTR)) {
            cpl_msg_warning(cpl_func, "writing test file failed");
        }
        close(fd);
    }

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, notafitsfile);

    code = qmost_dfs_save_image_extension(frame, inhdr, "TEST", NULL,
                                          image, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    cpl_frame_delete(frame);
    unlink(notafitsfile);

    /* Clean up */
    cpl_image_delete(image);
    cpl_propertylist_delete(inhdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_save_image_and_var
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_save_image_and_var(void)
{
    cpl_frame *frame = NULL;
    char outfile[] = "test-qmost_dfs_XXXXXX";
    char notafitsfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *dummy = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_image *image = NULL;
    cpl_image *var = NULL;
    cpl_propertylist *hdrchk = NULL;
    cpl_image *imchk = NULL;
    cpl_image *varchk = NULL;
    cpl_error_code code;

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create FITS file with dummy PHDU */
    dummy = cpl_propertylist_new();
    cpl_propertylist_save(dummy, outfile, CPL_IO_CREATE);
    cpl_propertylist_delete(dummy);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* Empty header */
    inhdr = cpl_propertylist_new();

    /* Test image */
    image = cpl_image_fill_test_create(512, 256);
    var = cpl_image_duplicate(image);
    cpl_image_threshold(var, 0, DBL_MAX, 0, DBL_MAX);

    /* NULL inputs */
    code = qmost_dfs_save_image_and_var(NULL, inhdr, "TEST", NULL,
                                        image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_dfs_save_image_and_var(frame, inhdr, NULL, NULL,
                                        image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set in frame */
    code = qmost_dfs_save_image_and_var(frame, inhdr, "TEST", NULL,
                                        image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid inputs, should save */
    cpl_frame_set_filename(frame, outfile);

    code = qmost_dfs_save_image_and_var(frame, inhdr, "TEST", NULL,
                                        image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check we can load image */
    imchk = cpl_image_load(outfile, CPL_TYPE_DOUBLE, 0, 1);
    cpl_test_nonnull(imchk);
    cpl_test_image_abs(imchk, image, 1);
    cpl_image_delete(imchk);

    hdrchk = cpl_propertylist_load(outfile, 1);
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "EXTNAME"),
                       "TEST");
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "ERRDATA"),
                       "TEST_var");
    cpl_propertylist_delete(hdrchk);

    /* and variance */
    varchk = cpl_image_load(outfile, CPL_TYPE_DOUBLE, 0, 2);
    cpl_test_nonnull(varchk);
    cpl_test_image_abs(varchk, var, 1);
    cpl_image_delete(varchk);

    hdrchk = cpl_propertylist_load(outfile, 2);
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "EXTNAME"),
                       "TEST_var");
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "SCIDATA"),
                       "TEST");
    cpl_propertylist_delete(hdrchk);

    /* Test saving a dummy pair of extensions */
    code = qmost_dfs_save_image_and_var(frame, inhdr, "DUMMY", NULL,
                                        NULL, NULL, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_frame_delete(frame);
    unlink(outfile);

    /* Now try an invalid file */
    fd = mkstemp(notafitsfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0) {
        if(write(fd, TESTSTR, strlen(TESTSTR)) != strlen(TESTSTR)) {
            cpl_msg_warning(cpl_func, "writing test file failed");
        }
        close(fd);
    }

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, notafitsfile);

    code = qmost_dfs_save_image_and_var(frame, inhdr, "TEST", NULL,
                                        image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    cpl_frame_delete(frame);
    unlink(notafitsfile);

    /* Clean up */
    cpl_image_delete(image);
    cpl_image_delete(var);
    cpl_propertylist_delete(inhdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_save_image_and_ivar
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_save_image_and_ivar(void)
{
    cpl_frame *frame = NULL;
    char outfile[] = "test-qmost_dfs_XXXXXX";
    char notafitsfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *dummy = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_image *image = NULL;
    cpl_image *ivar = NULL;
    cpl_propertylist *hdrchk = NULL;
    cpl_image *imchk = NULL;
    cpl_image *var = NULL;
    double *ddata = NULL;
    int ipix, npix;
    cpl_image *ivarchk = NULL;
    cpl_image *tmpimg = NULL;
    cpl_error_code code;

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create FITS file with dummy PHDU */
    dummy = cpl_propertylist_new();
    cpl_propertylist_save(dummy, outfile, CPL_IO_CREATE);
    cpl_propertylist_delete(dummy);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* Empty header */
    inhdr = cpl_propertylist_new();

    /* Test image */
    image = cpl_image_fill_test_create(512, 256);
    var = cpl_image_duplicate(image);
    cpl_image_threshold(var, 0, DBL_MAX, 0, DBL_MAX);

    /* NULL inputs */
    code = qmost_dfs_save_image_and_ivar(NULL, inhdr, "TEST_DATA", NULL,
                                         image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_dfs_save_image_and_ivar(frame, inhdr, NULL, NULL,
                                         image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set in frame */
    code = qmost_dfs_save_image_and_ivar(frame, inhdr, "TEST_DATA", NULL,
                                         image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid inputs, should save */
    cpl_frame_set_filename(frame, outfile);

    code = qmost_dfs_save_image_and_ivar(frame, inhdr, "TEST_DATA", NULL,
                                         image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check we can load image */
    imchk = cpl_image_load(outfile, CPL_TYPE_DOUBLE, 0, 1);
    cpl_test_nonnull(imchk);
    cpl_test_image_abs(imchk, image, 1);
    cpl_image_delete(imchk);

    hdrchk = cpl_propertylist_load(outfile, 1);
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "EXTNAME"),
                       "TEST_DATA");
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "ERRDATA"),
                       "TEST_IVAR");
    cpl_propertylist_delete(hdrchk);

    /* and inverse variance */
    ivar = cpl_image_duplicate(var);

    ddata = cpl_image_get_data_double(ivar);
    if(ddata != NULL) {
        npix = (cpl_image_get_size_x(ivar) *
                cpl_image_get_size_y(ivar));

        for(ipix = 0; ipix < npix; ipix++) {
            if(ddata[ipix] != 0) {
                ddata[ipix] = 1.0 / ddata[ipix];
            }
        }
    }

    ivarchk = cpl_image_load(outfile, CPL_TYPE_DOUBLE, 0, 2);
    cpl_test_nonnull(ivarchk);
    cpl_test_image_abs(ivarchk, ivar, 1);
    cpl_image_delete(ivarchk);

    cpl_image_delete(ivar);

    hdrchk = cpl_propertylist_load(outfile, 2);
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "EXTNAME"),
                       "TEST_IVAR");
    cpl_test_eq_string(cpl_propertylist_get_string(hdrchk, "SCIDATA"),
                       "TEST_DATA");
    cpl_propertylist_delete(hdrchk);

    /* float variance should also work */
    tmpimg = cpl_image_cast(var, CPL_TYPE_FLOAT);

    code = qmost_dfs_save_image_and_ivar(frame, inhdr, "TEST_DATA", NULL,
                                         image, tmpimg, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(tmpimg);
    tmpimg = NULL;

    /* but integer should not */
    tmpimg = cpl_image_cast(var, CPL_TYPE_INT);

    code = qmost_dfs_save_image_and_ivar(frame, inhdr, "TEST_DATA", NULL,
                                         image, tmpimg, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_image_delete(tmpimg);
    tmpimg = NULL;

    /* _DATA not in extname */
    code = qmost_dfs_save_image_and_ivar(frame, inhdr, "TEST", NULL,
                                         image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Test saving a dummy pair of extensions */
    code = qmost_dfs_save_image_and_ivar(frame, inhdr, "DUMMY_DATA", NULL,
                                         NULL, NULL, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_frame_delete(frame);
    unlink(outfile);

    /* Now try an invalid file */
    fd = mkstemp(notafitsfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0) {
        if(write(fd, TESTSTR, strlen(TESTSTR)) != strlen(TESTSTR)) {
            cpl_msg_warning(cpl_func, "writing test file failed");
        }
        close(fd);
    }

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, notafitsfile);

    code = qmost_dfs_save_image_and_var(frame, inhdr, "TEST_DATA", NULL,
                                        image, var, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    cpl_frame_delete(frame);
    unlink(notafitsfile);

    /* Clean up */
    cpl_image_delete(image);
    cpl_image_delete(var);
    cpl_propertylist_delete(inhdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_save_imagelist_extension
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_save_imagelist_extension(void)
{
    cpl_frame *frame = NULL;
    char outfile[] = "test-qmost_dfs_XXXXXX";
    char notafitsfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *dummy = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_imagelist *imagelist = NULL;
    cpl_imagelist *imchk = NULL;
    cpl_error_code code;

    int iimg, nimg = 2;
    cpl_image *imtmpa, *imtmpb;

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create FITS file with dummy PHDU */
    dummy = cpl_propertylist_new();
    cpl_propertylist_save(dummy, outfile, CPL_IO_CREATE);
    cpl_propertylist_delete(dummy);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* Empty header */
    inhdr = cpl_propertylist_new();

    /* Test imagelist with a couple of planes */
    imagelist = cpl_imagelist_new();

    for(iimg = 0; iimg < nimg; iimg++) {
        cpl_imagelist_set(imagelist,
                          cpl_image_fill_test_create(512, 256),
                          iimg);
    }

    /* NULL inputs */
    code = qmost_dfs_save_imagelist_extension(NULL, inhdr, "TEST", NULL,
                                              imagelist, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_dfs_save_imagelist_extension(frame, inhdr, NULL, NULL,
                                              imagelist, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set in frame */
    code = qmost_dfs_save_imagelist_extension(frame, inhdr, "TEST", NULL,
                                              imagelist, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid inputs, should save imagelist */
    cpl_frame_set_filename(frame, outfile);

    code = qmost_dfs_save_imagelist_extension(frame, inhdr, "TEST", NULL,
                                              imagelist, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check we can load it */
    imchk = cpl_imagelist_load(outfile, CPL_TYPE_DOUBLE, 1);
    cpl_test_nonnull(imchk);

    cpl_test_eq(cpl_imagelist_get_size(imchk), nimg);

    for(iimg = 0; iimg < nimg; iimg++) {
        imtmpa = cpl_imagelist_get(imagelist, iimg);
        imtmpb = cpl_imagelist_get(imchk, iimg);
        cpl_test_nonnull(imtmpb);

        cpl_test_image_abs(imtmpa, imtmpb, 0.1);
    }

    cpl_imagelist_delete(imchk);

    /* Test saving a dummy extension */
    code = qmost_dfs_save_imagelist_extension(frame, inhdr, "DUMMY", NULL,
                                              NULL, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_frame_delete(frame);
    unlink(outfile);

    /* Now try an invalid file */
    fd = mkstemp(notafitsfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0) {
        if(write(fd, TESTSTR, strlen(TESTSTR)) != strlen(TESTSTR)) {
            cpl_msg_warning(cpl_func, "writing test file failed");
        }
        close(fd);
    }

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, notafitsfile);

    code = qmost_dfs_save_imagelist_extension(frame, inhdr, "TEST", NULL,
                                              imagelist, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    cpl_frame_delete(frame);
    unlink(notafitsfile);

    /* Clean up */
    cpl_imagelist_delete(imagelist);
    cpl_propertylist_delete(inhdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_save_mask_extension
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_save_mask_extension(void)
{
    cpl_frame *frame = NULL;
    char outfile[] = "test-qmost_dfs_XXXXXX";
    char notafitsfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *dummy = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_mask *mask = NULL;
    cpl_mask *mskchk = NULL;
    cpl_error_code code;

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create FITS file with dummy PHDU */
    dummy = cpl_propertylist_new();
    cpl_propertylist_save(dummy, outfile, CPL_IO_CREATE);
    cpl_propertylist_delete(dummy);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* Empty header */
    inhdr = cpl_propertylist_new();

    /* Test mask */
    mask = cpl_mask_new(512, 256);

    /* NULL inputs */
    code = qmost_dfs_save_mask_extension(NULL, inhdr, "TEST", NULL,
                                         mask);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_dfs_save_mask_extension(frame, inhdr, NULL, NULL,
                                         mask);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set in frame */
    code = qmost_dfs_save_mask_extension(frame, inhdr, "TEST", NULL,
                                         mask);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid inputs, should save mask */
    cpl_frame_set_filename(frame, outfile);

    code = qmost_dfs_save_mask_extension(frame, inhdr, "TEST", NULL,
                                         mask);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check we can load it */
    mskchk = cpl_mask_load(outfile, 0, 1);
    cpl_test_nonnull(mskchk);
    cpl_mask_delete(mskchk);

    /* Test saving a dummy extension */
    code = qmost_dfs_save_mask_extension(frame, inhdr, "DUMMY", NULL,
                                         NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_frame_delete(frame);
    unlink(outfile);

    /* Now try an invalid file */
    fd = mkstemp(notafitsfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0) {
        if(write(fd, TESTSTR, strlen(TESTSTR)) != strlen(TESTSTR)) {
            cpl_msg_warning(cpl_func, "writing test file failed");
        }
        close(fd);
    }

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, notafitsfile);

    code = qmost_dfs_save_mask_extension(frame, inhdr, "TEST", NULL,
                                         mask);
    cpl_test_eq_error(code, CPL_ERROR_FILE_IO);

    cpl_frame_delete(frame);
    unlink(notafitsfile);

    /* Clean up */
    cpl_mask_delete(mask);
    cpl_propertylist_delete(inhdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dfs_save_table_extension
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_dfs_save_table_extension(void)
{
    cpl_frame *frame = NULL;
    char outfile[] = "test-qmost_dfs_XXXXXX";
    char notafitsfile[] = "test-qmost_dfs_XXXXXX";
    int fd = -1;
    cpl_propertylist *dummy = NULL;
    cpl_propertylist *inhdr = NULL;
    cpl_table *table = NULL;
    cpl_table *tbchk = NULL;
    cpl_error_code code;

    /* Create temporary FITS file for output. */
    fd = mkstemp(outfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0)
        close(fd);

    /* Create FITS file with dummy PHDU */
    dummy = cpl_propertylist_new();
    cpl_propertylist_save(dummy, outfile, CPL_IO_CREATE);
    cpl_propertylist_delete(dummy);

    /* Create empty frame */
    frame = cpl_frame_new();

    /* Empty header */
    inhdr = cpl_propertylist_new();

    /* Test table */
    table = cpl_table_new(1);
    cpl_table_new_column(table, "TEST", CPL_TYPE_INT);

    /* NULL inputs */
    code = qmost_dfs_save_table_extension(NULL, inhdr, "TEST", NULL,
                                          table);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_dfs_save_table_extension(frame, inhdr, NULL, NULL,
                                          table);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No filename set in frame */
    code = qmost_dfs_save_table_extension(frame, inhdr, "TEST", NULL,
                                          table);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Valid inputs, should save table */
    cpl_frame_set_filename(frame, outfile);

    code = qmost_dfs_save_table_extension(frame, inhdr, "TEST", NULL,
                                          table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check we can load it */
    tbchk = cpl_table_load(outfile, 1, 0);
    cpl_test_nonnull(tbchk);
    cpl_table_delete(tbchk);

    /* Test saving a dummy table */
    code = qmost_dfs_save_table_extension(frame, inhdr, "DUMMY", NULL,
                                          NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_frame_delete(frame);
    unlink(outfile);

    /* Now try an invalid file */
    fd = mkstemp(notafitsfile);
    qmost_test_ge(fd, 0);
    if(fd >= 0) {
        if(write(fd, TESTSTR, strlen(TESTSTR)) != strlen(TESTSTR)) {
            cpl_msg_warning(cpl_func, "writing test file failed");
        }
        close(fd);
    }

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, notafitsfile);

    /* Why this one throws a different error than the rest is beyond
     * me, but it does. */
    code = qmost_dfs_save_table_extension(frame, inhdr, "TEST", NULL,
                                          table);
    cpl_test_eq_error(code, CPL_ERROR_FILE_NOT_CREATED);

    cpl_frame_delete(frame);
    unlink(notafitsfile);

    /* Clean up */
    cpl_table_delete(table);
    cpl_propertylist_delete(inhdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_dfs module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_dfs_set_groups();
    test_qmost_dfs_product_filename();
    test_qmost_dfs_setup_product();
    test_qmost_dfs_setup_product_default();
    test_qmost_dfs_setup_product_extension_header();
    test_qmost_dfs_save_image_extension();
    test_qmost_dfs_save_image_and_var();
    test_qmost_dfs_save_image_and_ivar();
    test_qmost_dfs_save_imagelist_extension();
    test_qmost_dfs_save_mask_extension();
    test_qmost_dfs_save_table_extension();

    return cpl_test_end(0);
}

/**@}*/
