/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_filt2d.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_filt2d_test  Unit test of qmost_filt2d
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_bfilt2
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_bfilt2(void)
{
    int x, nx = 256;
    int y, ny = 512;

    cpl_image *img = NULL;
    cpl_image *tmpimg = NULL;
    float *buf;
    cpl_binary *bpm;

    int icalc, axis;
    int calcs[2] = { QMOST_MEANCALC, QMOST_MEDIANCALC };

    cpl_error_code code;

    /* Make an image with a gradient and some bad pixels */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    buf = cpl_image_get_data_float(img);
    cpl_test_nonnull(buf);

    bpm = cpl_mask_get_data(cpl_image_get_bpm(img));
    cpl_test_nonnull(bpm);

    if(buf != NULL && bpm != NULL) {
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                buf[y*nx+x] += 1024 + (x-nx/2) + (y-ny/2);
                bpm[y*nx+x] = (x+y) % 1000 == 0;
            }
        }
    }

    /* And a larger block of badness so we can test the no valid
     * pixels in block case */
    for(x = 0; x < 32; x++) {
        bpm[42*nx+x] = 1;
    }
    for(x = nx-32; x < nx; x++) {
        bpm[42*nx+x] = 1;
    }

    /* NULL input test */
    code = qmost_bfilt2(NULL, 16, QMOST_MEDIANCALC, 1);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Wrong type */
    tmpimg = cpl_image_cast(img, CPL_TYPE_DOUBLE);

    code = qmost_bfilt2(tmpimg, 16, QMOST_MEDIANCALC, 1);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_image_delete(tmpimg);
    tmpimg = NULL;

    /* These should all work and remove the gradient, which we can
     * check by division. */
    for(icalc = 0; icalc < 2; icalc++) {
        for(axis = 1; axis <= 2; axis++) {
            tmpimg = cpl_image_duplicate(img);
            
            code = qmost_bfilt2(tmpimg, 16, calcs[icalc], axis);
            cpl_test_eq_error(code, CPL_ERROR_NONE);
            
            cpl_image_divide(tmpimg, img);
            
            qmost_test_image_float_abs(tmpimg, 1.0, 0, 0.1);

            cpl_image_delete(tmpimg);
            tmpimg = NULL;
        }
    }

    cpl_image_delete(img);
    img = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_filt2d module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_bfilt2();

    return cpl_test_end(0);
}

/**@}*/
