/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_imcombine_lite.h"
#include "qmost_linear.h"
#include "qmost_lininfo.h"
#include "qmost_pfits.h"
#include "qmost_polynm.h"
#include "qmost_sort.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_linear  qmost_linear
 *
 * Linearity fitting and correction.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_linear.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Linearity look-up table minimum ADU level. */

#define LUTMIN -1000

/* Linearity look-up table maximum ADU level. */

#define LUTMAX 65535

/*----------------------------------------------------------------------------*/
/**
 * @brief   Measure detector flat counts in each amplifier.
 *
 * The given detector flat frame is analysed and the first argument is
 * populated with measurements of the detector flat counts (ADU) in
 * each amplifier and the corresponding sigma.  Information about the
 * image sections analysed is also recorded.
 *
 * The image analysis is conducted on an already trimmed image from
 * qmost_ccdproc() by histogramming the pixel values in the part of
 * the image corresponding to the illuminated pixels of the amplifier
 * under analysis, and taking the median and a robust estimate of
 * sigma based on the quartiles of the intensity distribution using
 * qmost_skylevel_image().
 *
 * Memory associated with the qmost_flatstats structure should be
 * freed by the caller by calling qmost_flatstats_free.
 *
 * @param   self             (Modified) The qmost_flatstats object to
 *                                      populate.
 * @param   img              (Given)    The detector flat image to be
 *                                      analysed.
 * @param   pri_hdr          (Given)    The FITS primary header of the
 *                                      detector flat.  The keywords
 *                                      used are MJD-OBS and EXPTIME.
 * @param   qclist           (Given)    The IMAGE extension FITS
 *                                      header from qmost_ccdproc.
 *                                      The DRS headers NAMPS and AMPn
 *                                      SECT written by qmost_ccdproc
 *                                      are read from this header to
 *                                      determine how many amps to
 *                                      analyse and where they are
 *                                      located in the images.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS header keywords was not
 *                                        found.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the amplifier section
 *                                        FITS header keyword value
 *                                        can't be understood or is
 *                                        invalid (e.g. out of
 *                                        bounds).
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If one of the required input
 *                                        FITS header keyword values
 *                                        had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS AMPn SECT</b>
 *   - <b>ESO DRS NAMPS</b>
 *   - <b>ESO EXPTIME</b>
 *   - <b>EXPTIME</b>
 *   - <b>MJD-OBS</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_flatstats_compute (
    qmost_flatstats *self,
    cpl_image *img,
    cpl_propertylist *pri_hdr,
    cpl_propertylist *qclist)
{
    int iamp;
    char *key = NULL;
    const char *strval;
    cpl_image *amp_img = NULL;

    /* Check for NULL pointers */
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pri_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialize output */
    memset(self, 0, sizeof(qmost_flatstats));

#undef TIDY
#define TIDY                                    \
    qmost_flatstats_free(self);                  \
    if(key != NULL) {                           \
        cpl_free(key);                          \
        key = NULL;                             \
    }                                           \
    if(amp_img != NULL) {                       \
        cpl_image_delete(amp_img);              \
        amp_img = NULL;                         \
    }

    /* Read PHDU information */
    if(qmost_pfits_get_exptime(pri_hdr,
                               &(self->exptime)) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read exposure time "
                                     "from PHDU for monitor image");
    }
    if(qmost_pfits_get_mjd_obs(pri_hdr,
                               &(self->mjd)) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read MJD-OBS from "
                                     "PHDU for monitor image");
    }

    /* Determine number of amps */
    if(qmost_cpl_propertylist_get_int(qclist,
                                      "ESO DRS NAMPS",
                                      &(self->namps)) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read ESO DRS NAMPS "
                                     "from IMAGE extension header");
    }

    /* Allocate space for results */
    self->xmin = cpl_calloc(self->namps, sizeof(int));
    self->xmax = cpl_calloc(self->namps, sizeof(int));
    self->ymin = cpl_calloc(self->namps, sizeof(int));
    self->ymax = cpl_calloc(self->namps, sizeof(int));
    self->median = cpl_calloc(self->namps, sizeof(float));
    self->sigma = cpl_calloc(self->namps, sizeof(float));

    for(iamp = 0; iamp < self->namps; iamp++) {
        /* Get image section corresponding to this amp */
        key = cpl_sprintf("ESO DRS AMP%d SECT", iamp + 1);
        if(key == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format AMP SECT for "
                                         "amp %d", iamp+1);
        }

        strval = cpl_propertylist_get_string(qclist, key);
        if(strval == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO DRS AMP%d SECT "
                                         "from IMAGE extension header",
                                         iamp+1);
        }

        cpl_free(key);
        key = NULL;

        if(sscanf(strval,
                  "[%d:%d,%d:%d]",
                  &(self->xmin[iamp]),
                  &(self->xmax[iamp]),
                  &(self->ymin[iamp]),
                  &(self->ymax[iamp])) != 4) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "couldn't understand amp section "
                                         "string '%s'", strval);
        }

        /* Extract and analyse amp */
        amp_img = cpl_image_extract(img,
                                    self->xmin[iamp],
                                    self->ymin[iamp],
                                    self->xmax[iamp],
                                    self->ymax[iamp]);
        if(amp_img == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to extract image "
                                         "section for amp %d",
                                         iamp+1);
        }
        
        if(qmost_skylevel_image(amp_img,
                                -1000,  /* in case debiased */
                                65535,
                                -FLT_MAX, FLT_MAX, 0,
                                &(self->median[iamp]),
                                &(self->sigma[iamp])) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute sky level "
                                         "for amp %d",
                                         iamp+1);
        }
        
        cpl_image_delete(amp_img);
        amp_img = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Check if a flat is within acceptable exposure range.
 *
 * The given qmost_flatstats object is used to evaluate the exposure
 * level to check if the flat is underexposed or overexposed.  All
 * amplifiers are required to be within the specified exposure range
 * for the flat to be accepted.
 *
 * @param   self             (Modified) The qmost_flatstats object to
 *                                      check.
 * @param   underexp         (Given)    A minimum count level in ADU
 *                                      to reject underexposed flats.
 *                                      If the median counts in a flat
 *                                      are below this level, the flat
 *                                      is rejected.
 * @param   overexp          (Given)    A maximum count level in ADU
 *                                      to be used to reject
 *                                      overexposed flats.  If the
 *                                      median counts in a flat are
 *                                      above this level, the flat is
 *                                      rejected.
 *
 * @return  int              True if the flat is good, false if it
 *                           should be rejected.
 *
 * @note    The exposure times are not checked, but the user might
 *          wish to check that they exceed a minimum exposure time if
 *          the exposure time might be non-uniform, for example due to
 *          shutter shading in systems with leaf shutters or similar
 *          designs where finite shutter transfer time leads to a
 *          non-uniform exposure.  This effect can be significant
 *          (larger than the non-linearity we're trying to measure)
 *          for short exposure times.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

int qmost_flatstats_check (
    qmost_flatstats *self,
    float underexp,
    float overexp)
{
    int iamp;
    int is_ok = 1;

    for(iamp = 0; iamp < self->namps; iamp++) {
        if(self->median[iamp] < underexp ||
           self->median[iamp] >= overexp) {
            is_ok = 0;
            break;
        }
    }

    return is_ok;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Free a qmost_flatstats object.
 *
 * @param   self             (Modified) The qmost_flatstats object to
 *                                      free.
 *
 * @return  void
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_flatstats_free (
    qmost_flatstats *self)
{
    if(self == NULL) {
        return;
    }

    if(self->xmin != NULL) {
        cpl_free(self->xmin);
        self->xmin = NULL;
    }
    if(self->xmax != NULL) {
        cpl_free(self->xmax);
        self->xmax = NULL;
    }
    if(self->ymin != NULL) {
        cpl_free(self->ymin);
        self->ymin = NULL;
    }
    if(self->ymax != NULL) {
        cpl_free(self->ymax);
        self->ymax = NULL;
    }
    if(self->median != NULL) {
        cpl_free(self->median);
        self->median = NULL;
    }
    if(self->sigma != NULL) {
        cpl_free(self->sigma);
        self->sigma = NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Measure linearity from detector flats taken with an
 *          exposure ramp.
 *
 * A set of detector flats of a constant light source taken with
 * variable exposure times are analysed to measure detector linearity
 * based on the flat counts.  The results are recorded in a lininfo
 * table as a set of polynomial coefficients expressing linearised
 * counts (ADU) as a function of raw measured counts.
 * 
 * The statistics of the flats should first be computed with
 * qmost_flatstats_compute, under or over-exposed flats discarded
 * using qmost_flatstats_check, and the resulting list of
 * qmost_flatstats structures for the good flats can then be passed
 * into this routine to perform the linearity fit.
 *
 * These can optionally be interleaved with lamp brightness monitoring
 * exposures of fixed exposure time to measure and correct for any
 * variations in lamp brightness.  These exposures should be processed
 * in the same way into a separate list of qmost_flatstats structures.
 *
 * @param   lin_stats        (Given)    A list of nfiles
 *                                      qmost_flatstats structures
 *                                      containing the statistics of
 *                                      the linearity ramp detector
 *                                      flat images to be analysed.
 * @param   nfiles           (Given)    The number of qmost_flatstats
 *                                      structures for the linearity
 *                                      ramp to be analysed.
 * @param   mon_stats        (Given)    An optional list of nmon
 *                                      qmost_flatstats structures
 *                                      containing the lamp brightness
 *                                      monitoring sequence
 *                                      measurements, or NULL if
 *                                      none.
 * @param   nmon             (Given)    The number of lamp brightness
 *                                      monitoring measurements to be
 *                                      analysed.
 * @param   docorr           (Given)    If true, use monitor sequence
 *                                      to correct for lamp brightness
 *                                      variations.
 * @param   tbl              (Returned) The resulting linearity table.
 * @param   qclist           (Modified) The IMAGE extension FITS
 *                                      header to be updated with QC
 *                                      information.  The DRS headers
 *                                      NAMPS and AMPn SECT written by
 *                                      qmost_ccdproc are read from
 *                                      this header to determine how
 *                                      many amps to analyse and where
 *                                      they are located in the images.
 * @param   nord             (Given)    The degree of polynomial to
 *                                      fit to the linearity.
 * @param   niter            (Given)    The number of rejection
 *                                      iterations to use.
 * @param   clipthr          (Given)    The number of sigma beyond
 *                                      which a point is considered an
 *                                      outlier and rejected from the
 *                                      fit.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If there are not enough
 *                                        flats to fit the requested
 *                                        degree polynomial, or not
 *                                        enough monitor flats.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the amplifier section
 *                                        FITS header keyword value
 *                                        couldn't be understood.
 * @retval  CPL_ERROR_INCOMPTAIBLE_INPUT  If the number of amplifiers
 *                                        in the input files doesn't
 *                                        match.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 *
 * @par Output QC Parameters:
 *   - <b>DETFLAT MAX</b> (ADU): The maximum level of the flat fields
 *     that were combined to form the linearity table.
 *   - <b>DETFLAT MAXFILE</b>: The index (numbering from 1) of the
 *     flat with the highest count level in the linearity sequence.
 *   - <b>DETFLAT MIN</b> (ADU): The minimum level of the flat fields
 *     that were combined to form the linearity table.
 *   - <b>DETFLAT MINFILE</b>: The index (numbering from 1) of the
 *     flat with the lowest count level in the linearity sequence.
 *   - <b>LIN NL1K</b> (percent): The estimated non-linearity at 1000
 *     ADU averaged over all of the amplifiers.
 *   - <b>LIN NL5K</b> (percent): The estimated non-linearity at 5000
 *     ADU averaged over all of the amplifiers.
 *   - <b>LIN NL10K</b> (percent): The estimated non-linearity at
 *     10000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL20K</b> (percent): The estimated non-linearity at
 *     20000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL30K</b> (percent): The estimated non-linearity at
 *     30000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL40K</b> (percent): The estimated non-linearity at
 *     40000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL50K</b> (percent): The estimated non-linearity at
 *     50000 ADU averaged over all of the amplifiers.
 *   - <b>LIN RMS MAX</b> (ADU): The RMS of the linearity fit for the
 *     amplifier with the worst linearity fit.
 *   - <b>LIN RMS MAXAMP</b>: The amplifier with the worst linearity
 *     fit.
 *   - <b>LIN RMS MEAN</b> (ADU): The RMS of the linearity fit
 *     averaged over all of the amplifiers.
 *   - <b>LIN RMS MIN</b> (ADU): The RMS of the linearity fit for the
 *     amplifier with the best linearity fit.
 *   - <b>LIN RMS MINAMP</b>: The amplifier with the best linearity
 *     fit.
 *
 * @note    This routine uses the flat counts versus exposure time,
 *          which is typically more precise than the textbook "photon
 *          transfer" method based on measuring the detector gain at
 *          each step up the ramp using the observed Poisson noise in
 *          the flat.  Since the flat counts themselves are used it is
 *          therefore extremely important that the light source is
 *          stable and that the exposure times are precisely
 *          controlled and accurately known.  A monitor sequence
 *          should be used if the light source might vary, and ideally
 *          the order of the linearity ramp should also be randomised
 *          to reduce the chance of systematic errors.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_linfit (
    qmost_flatstats *lin_stats,
    int nfiles,
    qmost_flatstats *mon_stats,
    int nmon,
    int docorr,
    cpl_table **tbl,
    cpl_propertylist *qclist,
    int nord,
    int niter,
    float clipthr)
{
    double *exptimes = NULL;
    double *mjds = NULL;
    double *mon_mjds = NULL;
    double *mon_facs = NULL;
    qmost_lininfo *lins = NULL;
    double *xfit = NULL;
    double *yfit = NULL;
    double *resid = NULL;
    cpl_polynomial *slcoefs = NULL;

    cpl_errorstate prestate;

    int ifile;
    int iamp, iampt, namps = 0;

    int imon;
    double meancts;
    double midpt;

    double mon_fac_min = -1, mon_fac_max = -1;
    double mon_fac_ssq, mon_fac_rms = 0;

    int isp, ifp, im;
    double fac, a, b, d;

    qmost_lininfo *lin;

    float median;
    double val, medoff, sigoff;
    int iiter, ngoodc;
    cpl_size degree;

    double sum, sum2;

    /* Check for NULL arguments or other serious issues */
    cpl_ensure_code(lin_stats != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nfiles > 0, CPL_ERROR_DATA_NOT_FOUND);
    cpl_ensure_code(tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialize this for garbage collection */
    *tbl = NULL;

#undef TIDY
#define TIDY                                            \
    if(exptimes != NULL) {                              \
        cpl_free(exptimes);                             \
        exptimes = NULL;                                \
    }                                                   \
    if(mjds != NULL) {                                  \
        cpl_free(mjds);                                 \
        mjds = NULL;                                    \
    }                                                   \
    if(mon_mjds != NULL) {                              \
        cpl_free(mon_mjds);                             \
        mon_mjds = NULL;                                \
    }                                                   \
    if(mon_facs != NULL) {                              \
        cpl_free(mon_facs);                             \
        mon_facs = NULL;                                \
    }                                                   \
    if(lins != NULL) {                                  \
        for(iampt = 0; iampt < namps; iampt++) {        \
            qmost_lindelinfo(lins + iampt);             \
        }                                               \
                                                        \
        cpl_free(lins);                                 \
        lins = NULL;                                    \
    }                                                   \
    if(xfit != NULL) {                                  \
        cpl_free(xfit);                                 \
        xfit = NULL;                                    \
    }                                                   \
    if(yfit != NULL) {                                  \
        cpl_free(yfit);                                 \
        yfit = NULL;                                    \
    }                                                   \
    if(resid != NULL) {                                 \
        cpl_free(resid);                                \
        resid = NULL;                                   \
    }                                                   \
    if(slcoefs != NULL) {                               \
        cpl_polynomial_delete(slcoefs);                 \
        slcoefs = NULL;                                 \
    }                                                   \
    if(*tbl != NULL) {                                  \
        cpl_table_delete(*tbl);                         \
        *tbl = NULL;                                    \
    }

    /* Check we have enough files */
    if(nfiles < nord) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "%d flats are not enough to fit a %d "
                                     "degree polynomial",
                                     nfiles, nord);
    }

    /* Fetch PHDU information and check number of amps */
    exptimes = cpl_malloc(nfiles * sizeof(double));
    mjds = cpl_malloc(nfiles * sizeof(double));

    namps = lin_stats[0].namps;

    for(ifile = 0; ifile < nfiles; ifile++) {
        exptimes[ifile] = lin_stats[ifile].exptime;
        mjds[ifile] = lin_stats[ifile].mjd;

        if(lin_stats[ifile].namps != namps) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "number of amps doesn't match "
                                         "for file %d: %d != %d",
                                         ifile+1,
                                         lin_stats[ifile].namps,
                                         namps);
        }
    }

    /* Get monitor series, if given */
    if(mon_stats != NULL && nmon > 0) {
        mon_mjds = cpl_malloc(nmon * sizeof(double));
        mon_facs = cpl_malloc(nmon * sizeof(double));

        for(imon = 0; imon < nmon; imon++) {
            mon_mjds[imon] = mon_stats[imon].mjd;

            meancts = 0;
            for(iamp = 0; iamp < mon_stats[imon].namps; iamp++) {
                meancts += mon_stats[imon].median[iamp];
            }

            meancts /= mon_stats[imon].namps;

            mon_facs[imon] = meancts / mon_stats[imon].exptime;
        }

        if(nmon < 2) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "there were not enough usable "
                                         "monitor flats");
        }

        /* Sort */
        qmost_sort_dd(mon_mjds, mon_facs, nmon);

        /* Normalise to the one closest to midpoint, record range and rms */
        midpt = mon_facs[nmon/2];

        mon_fac_min = -1;
        mon_fac_max = -1;
        mon_fac_ssq = 0;

        for(imon = 0; imon < nmon; imon++) {
            mon_facs[imon] /= midpt;

            if(imon == 0 || mon_facs[imon] < mon_fac_min) {
                mon_fac_min = mon_facs[imon];
            }
            if(imon == 0 || mon_facs[imon] > mon_fac_max) {
                mon_fac_max = mon_facs[imon];
            }

            mon_fac_ssq += (mon_facs[imon] - 1.0) * (mon_facs[imon] - 1.0);
        }

        mon_fac_rms = sqrt(mon_fac_ssq / nmon);
    }

    /* Allocate array of lininfo structures */
    lins = cpl_calloc(namps, sizeof(qmost_lininfo));

    /* Allocate workspace */
    xfit = cpl_malloc(nfiles * sizeof(double));
    yfit = cpl_malloc(nfiles * sizeof(double));
    resid = cpl_malloc(nfiles * sizeof(double));

    /* Analyse each amp separately */
    for(iamp = 0; iamp < namps; iamp++) {
        lin = lins + iamp;

        /* Allocate lininfo structure */
        if(qmost_lincrinfo(nfiles, nord, lin) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not create lininfo "
                                         "structure for amp %d",
                                         iamp+1);
        }

        lin->amp = iamp + 1;
        lin->xmin = lin_stats[0].xmin[iamp];
        lin->xmax = lin_stats[0].xmax[iamp];
        lin->ymin = lin_stats[0].ymin[iamp];
        lin->ymax = lin_stats[0].ymax[iamp];

        /* Analyse images */
        for(ifile = 0; ifile < nfiles; ifile++) {
            median = lin_stats[ifile].median[iamp];

            /* Adjustment from monitoring sequence */
            if(mon_stats != NULL && nmon > 0) {
                /* Find leftmost element */
                isp = 0;
                ifp = nmon;
                
                while(isp < ifp) {
                    im = (isp + ifp) / 2;
                    if(mon_mjds[im] < mjds[ifile]) {
                        isp = im + 1;
                    }
                    else {
                        ifp = im;
                    }
                }
                
                if(isp < 1) {
                    /* Before first, set equal to first */
                    fac = mon_facs[0];
                }
                else if(isp >= nmon) {
                    /* After last, set equal to last */
                    fac = mon_facs[nmon-1];
                }
                else {
                    /* Linear interpolation */
                    a = mon_mjds[isp] - mjds[ifile];
                    b = mjds[ifile] - mon_mjds[isp-1];
                    d = a + b;
                    if(d > DBL_EPSILON) {
                        fac = (mon_facs[isp-1] * a + mon_facs[isp] * b) / d;
                    }
                    else {
                        /* Dates nearly equal, take average */
                        fac = 0.5 * (mon_facs[isp-1] + mon_facs[isp]);
                    }
                }
            }
            else {
                fac = 1.0;
            }

            lin->med[ifile] = median;
            lin->exptime[ifile] = exptimes[ifile];
            lin->moncorr[ifile] = fac;
            lin->fit_flag[ifile] = 0;
        }

        /* Fit of effective exposure time (after correction for lamp
         * brightness variations) as a function of median counts. */
        for(iiter = 0; iiter < niter; iiter++) {
            /* Filtered list for fitting */
            lin->ngood = 0;

            for(ifile = 0; ifile < nfiles; ifile++) {
                if(!lin->fit_flag[ifile]) {
                    xfit[lin->ngood] = lin->med[ifile];
                    yfit[lin->ngood] = lin->exptime[ifile]*(docorr ? lin->moncorr[ifile] : 1.0);
                    lin->ngood++;
                }
            }

            /* Polynomial fit.  Not enough points or singular matrix
             * are converted into non-fatal error messages where all
             * coefficients are zero. */
            if(lin->coefs != NULL) {
                cpl_polynomial_delete(lin->coefs);
                lin->coefs = NULL;
            }

            prestate = cpl_errorstate_get();

            lin->coefs = qmost_polynm(xfit, yfit, lin->ngood, nord+1, 1);
            if(lin->coefs == NULL) {
                switch(cpl_error_get_code()) {
                case CPL_ERROR_DATA_NOT_FOUND:
                    cpl_errorstate_set(prestate);

                    cpl_msg_warning(cpl_func,
                                    "not enough valid points for degree %d "
                                    "polynomial fit for amp %d iteration "
                                    "%d: %d",
                                    nord, iamp+1, iiter+1, lin->ngood);

                    lin->coefs = cpl_polynomial_new(1);

                    degree = 1;
                    cpl_polynomial_set_coeff(lin->coefs, &degree, 1.0);

                    break;
                case CPL_ERROR_SINGULAR_MATRIX:
                    cpl_errorstate_set(prestate);

                    cpl_msg_warning(cpl_func,
                                  "singular matrix in degree %d polynomial "
                                  "fit for amp %d iteration %d "
                                  "with %d data points",
                                  nord, iamp+1, iiter+1, lin->ngood);

                    lin->coefs = cpl_polynomial_new(1);

                    degree = 1;
                    cpl_polynomial_set_coeff(lin->coefs, &degree, 1.0);

                    break;
                default:
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "polynomial fit failed for "
                                                 "amp %d",
                                                 iamp+1);
                }
            }

            /* Get out if this is the last iteration */
            if(iiter == niter-1) {
                break;
            }

            /* Evaluate fit residuals */
            for(ifile = 0; ifile < nfiles; ifile++) {
                val = cpl_polynomial_eval_1d(lin->coefs,
                                             lin->med[ifile], NULL);
                resid[ifile] = lin->exptime[ifile]*(docorr ? lin->moncorr[ifile] : 1.0) - val;
            }

            prestate = cpl_errorstate_get();

            if(qmost_dmedmad(resid, NULL, nfiles,
                             &medoff, &sigoff) != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);
                medoff = 0;
                sigoff = 0;
            }

            sigoff *= CPL_MATH_STD_MAD;

            /* Clip */
            ngoodc = 0;

            for(ifile = 0; ifile < nfiles; ifile++) {
                if(fabs(resid[ifile] - medoff) > clipthr * sigoff) {
                    lin->fit_flag[ifile] = 1;
                }
                else {
                    ngoodc++;
                }
            }

            /* If nothing new was clipped, we are done */
            if(lin->ngood == ngoodc) {
                break;
            }
        }

        /* Normalise by dividing out the first degree coefficient */
        degree = 1;
        val = cpl_polynomial_get_coeff(lin->coefs, &degree);
        
        cpl_polynomial_multiply_scalar(lin->coefs, lin->coefs, 1.0 / val);

        /* Calculate linearised counts and fit a straight line vs
         * effective exposure time to evaluate fit rms.  Again,
         * failure to fit is trapped, but this time we stay quiet
         * given that we would have emitted a warning above. */
        ngoodc = 0;

        for(ifile = 0; ifile < nfiles; ifile++) {
            val = cpl_polynomial_eval_1d(lin->coefs,
                                         lin->med[ifile], NULL);

            lin->linmed[ifile] = val;

            if(!lin->fit_flag[ifile]) {
                xfit[ngoodc] = lin->exptime[ifile]*(docorr ? lin->moncorr[ifile] : 1.0);
                yfit[ngoodc] = val;
                ngoodc++;
            }
        }

        prestate = cpl_errorstate_get();

        slcoefs = qmost_polynm(xfit, yfit, ngoodc, 2, 0);
        if(slcoefs == NULL) {
            switch(cpl_error_get_code()) {
            case CPL_ERROR_DATA_NOT_FOUND:
                cpl_errorstate_set(prestate);
                slcoefs = cpl_polynomial_new(1);

                degree = 1;
                cpl_polynomial_set_coeff(slcoefs, &degree, 1.0);

                break;
            case CPL_ERROR_SINGULAR_MATRIX:
                cpl_errorstate_set(prestate);
                slcoefs = cpl_polynomial_new(1);

                degree = 1;
                cpl_polynomial_set_coeff(slcoefs, &degree, 1.0);

                break;
            default:
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             cpl_error_get_code(),
                                             "linear fit failed for "
                                             "amp %d",
                                             iamp+1);
            }
        }

        /* Calculate RMS residual */
        sum = 0.0;
        sum2 = 0.0;
        ngoodc = 0;
        for (ifile = 0; ifile < nfiles; ifile++) {
            val = cpl_polynomial_eval_1d(
                slcoefs,
                lin->exptime[ifile]*(docorr ? lin->moncorr[ifile] : 1.0),
                NULL);
            resid[ifile] = lin->linmed[ifile] - val;

            if(!lin->fit_flag[ifile]) {
                sum += resid[ifile];
                sum2 += resid[ifile]*resid[ifile];
                ngoodc++;
            }
        }

        if(ngoodc > 0) {
            sum /= (double)ngoodc;
            sum2 = sum2/(double)ngoodc - sum*sum;
        }
        else {
            sum = 0;
            sum2 = 0;
        }
        
        lin->rms = sqrt(sum2 > 0 ? sum2 : 0);

        cpl_polynomial_delete(slcoefs);
        slcoefs = NULL;
    }

    cpl_free(xfit);
    xfit = NULL;

    cpl_free(yfit);
    yfit = NULL;

    cpl_free(resid);
    resid = NULL;

    cpl_free(exptimes);
    exptimes = NULL;

    cpl_free(mjds);
    mjds = NULL;

    if(mon_mjds != NULL) {
        cpl_free(mon_mjds);
        mon_mjds = NULL;
    }

    if(mon_facs != NULL) {
        cpl_free(mon_facs);
        mon_facs = NULL;
    }

    /* Write out linearity table and QC */
    if(qmost_lincrtab(tbl, nfiles, nord) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create linearity table");
    }

    if(qmost_linwrite(*tbl, qclist, lins, namps) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not write linearity table");
    }

    /* Monitor correction diagnostic information */
    if(mon_stats != NULL && nmon > 0) {
        cpl_msg_info(cpl_func,
                     "Lamp brightness variation rms, min, max: "
                     "%.1f, %.1f, %.1f %%",
                     mon_fac_rms * 100,
                     (mon_fac_min - 1.0) * 100,
                     (mon_fac_max - 1.0) * 100);
    }

    /* Clean up */
    for(iamp = 0; iamp < namps; iamp++) {
        qmost_lindelinfo(lins + iamp);
    }
    
    cpl_free(lins);
    lins = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Perform linearity correction on an image.
 *
 * The given linearity table is read and the polynomial correction for
 * each amplifier applied to the pixels of that amplifier.  This
 * routine is intended for use on images that were already processed
 * by qmost_ccdproc(), and is used after linearity measurement to
 * apply the new linearity correction.  For all other purposes the
 * linearity correction functionality inside qmost_ccdproc() should be
 * used instead.
 *
 * @param   img              (Modified) The image to be corrected.
 *                                      The data type must be float or
 *                                      double.
 * @param   hdr              (Given)    The IMAGE extension FITS
 *                                      header to be updated with QC
 *                                      information.  The DRS headers
 *                                      NAMPS and AMPn SECT written by
 *                                      qmost_ccdproc are read from
 *                                      this header to determine how
 *                                      many amps to analyse and where
 *                                      they are located in the images.
 * @param   linearity_tbl    (Given)    The linearity correction
 *                                      table.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE If the linearity table
 *                                        amplifier coordinate ranges
 *                                        are outside the image.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS header keywords was not
 *                                        found or the linearity table
 *                                        is empty.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If one of the input FITS
 *                                        header keyword values or the
 *                                        linearity table is invalid.
 * @retval  CPL_ERROR_INCOMPTAIBLE_INPUT  If the number of amplifiers
 *                                        in the linearity table
 *                                        doesn't match the image
 *                                        being processed.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_UNSUPPORTED_MODE    If the data type of the
 *                                        image is not float or
 *                                        double.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If one of the input FITS
 *                                        header keyword values had an
 *                                        incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS AMPn SECT</b>
 *   - <b>ESO DRS NAMPS</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_lincor (
    cpl_image *img,
    cpl_propertylist *hdr,
    cpl_table *linearity_tbl)
{
    int nx, ny;
    cpl_type type;

    qmost_lininfo *lins = NULL;
    int iamp, iampt, namps, nampstab = 0;
    int xmin, xmax, ymin, ymax;

    char *key = NULL;
    const char *strval;

    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(linearity_tbl != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                            \
    if(lins != NULL) {                                  \
        for(iampt = 0; iampt < nampstab; iampt++) {     \
            qmost_lindelinfo(lins + iampt);             \
        }                                               \
        cpl_free(lins);                                 \
        lins = NULL;                                    \
    }                                                   \
    if(key != NULL) {                                   \
        cpl_free(key);                                  \
        key = NULL;                                     \
    }

    /* Get image dimensions and type */
    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);
    type = cpl_image_get_type(img);

    /* Determine number of amps from image header */
    if(qmost_cpl_propertylist_get_int(hdr,
                                      "ESO DRS NAMPS",
                                      &namps) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read ESO DRS NAMPS "
                                     "from IMAGE extension header");
    }

    /* Read linearity table */
    if(qmost_linchk(linearity_tbl) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "invalid linearity table");
    }

    if(qmost_linread(linearity_tbl, &lins, &nampstab) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read linearity table");
    }

    /* Check we have linearity coefficients for all amps */
    if(namps > nampstab) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "linearity table doesn't have "
                                     "enough amps: image %d table %d",
                                     namps, nampstab);
    }

    /* Process amplifiers individually */
    for(iamp = 0; iamp < namps; iamp++) {
        /* Get image section corresponding to this amp */
        key = cpl_sprintf("ESO DRS AMP%d SECT", iamp + 1);
        if(key == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format AMP SECT for "
                                         "amp %d", iamp+1);
        }

        strval = cpl_propertylist_get_string(hdr, key);
        if(strval == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read ESO DRS AMP%d SECT "
                                         "from IMAGE extension header",
                                         iamp+1);
        }

        cpl_free(key);
        key = NULL;

        if(sscanf(strval,
                  "[%d:%d,%d:%d]",
                  &(xmin),
                  &(xmax),
                  &(ymin),
                  &(ymax)) != 4) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "couldn't understand amp section "
                                         "string '%s'", strval);
        }

        /* Bounds check image section for this amplifier */
        if(xmin < 1 || xmin > nx ||
           xmax < 1 || xmax > nx ||
           ymin < 1 || ymin > ny ||
           ymax < 1 || ymax > ny) {
            TIDY;
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "amplifier %d section "
                                         "[%d:%d,%d:%d] out of bounds "
                                         "for %dx%d image",
                                         iamp+1,
                                         xmin, xmax, ymin, ymax,
                                         nx, ny);
        }

        /* Call the appropriate version of the correction routine */
        switch(type) {
        case CPL_TYPE_DOUBLE:
            qmost_lincor1_double(cpl_image_get_data_double(img),
                                 nx,
                                 xmin-1,
                                 xmax-1,
                                 ymin-1,
                                 ymax-1,
                                 lins + iamp);
            break;
        case CPL_TYPE_FLOAT:
            qmost_lincor1_float(cpl_image_get_data_float(img),
                                nx,
                                xmin-1,
                                xmax-1,
                                ymin-1,
                                ymax-1,
                                lins + iamp);
            break;
        default:
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
                                         "linearity correction of a %s "
                                         "image is not supported",
                                         cpl_type_get_name(type));
            break;
        }
    }

    for(iampt = 0; iampt < nampstab; iampt++) {
        qmost_lindelinfo(lins + iampt);
    }

    cpl_free(lins);
    lins = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Perform linearity correction on a double image array.
 *
 * @param   data             (Modified) The image to be corrected.
 * @param   nx               (Given)    The size of the image in x.
 * @param   ixl              (Given)    The first x pixel to correct
 *                                      numbering from 0.
 * @param   ixh              (Given)    The last x pixel to correct
 *                                      numbering from 0.
 * @param   iyl              (Given)    The first y pixel to correct
 *                                      numbering from 0.
 * @param   iyh              (Given)    The last y pixel to correct
 *                                      numbering from 0.
 * @param   lin              (Given)    The lininfo structure
 *                                      containing the linearity
 *                                      correction polynomial
 *                                      coefficients.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_lincor1_double (
    double *data,
    int nx,
    int ixl,
    int ixh,
    int iyl,
    int iyh,
    qmost_lininfo *lin)
{
    int iy,ix,i,nlut,idat1,idat2;
    double ldat1,ldat2,slope,inter;
    double *lut = NULL;
    unsigned char *lutdef = NULL;

    /* Create a look up table to speed things up. If a value has been used
       before, then we can just reuse it rather than to evaluate the 
       polynomial again. */

    nlut = LUTMAX - LUTMIN + 1;

    lut = cpl_calloc(nlut,sizeof(double));
    lutdef = cpl_calloc(nlut,sizeof(unsigned char));
        
    /* OK, loop through the data and do the linearisation */

    for(iy = iyl; iy <= iyh; iy++) {
        for(ix = ixl; ix <= ixh; ix++) {
            i = iy*nx+ix;

            idat1 = qmost_max(qmost_min((int)data[i],LUTMAX),LUTMIN);
            idat2 = qmost_max(qmost_min((int)data[i]+1,LUTMAX),LUTMIN);
            if (lutdef[idat1-LUTMIN]) {
                ldat1 = lut[idat1-LUTMIN];
            } else {
                ldat1 = cpl_polynomial_eval_1d(lin->coefs,(double)idat1,NULL);
                lut[idat1-LUTMIN] = ldat1;
                lutdef[idat1-LUTMIN] = 1;
            }
            if (lutdef[idat2-LUTMIN]) {
                ldat2 = lut[idat2-LUTMIN];
            } else {
                ldat2 = cpl_polynomial_eval_1d(lin->coefs,(double)idat2,NULL);
                lut[idat2-LUTMIN] = ldat2;
                lutdef[idat2-LUTMIN] = 1;
            }
            slope = ldat2 - ldat1;
            inter = ldat2 - slope*(double)idat2;
            data[i] = slope*data[i] + inter;
        }
    }

    /* Tidy and exit */
    
    cpl_free(lut);
    cpl_free(lutdef);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Perform linearity correction on a float image array.
 *
 * @param   data             (Modified) The image to be corrected.
 * @param   nx               (Given)    The size of the image in x.
 * @param   ixl              (Given)    The first x pixel to correct
 *                                      numbering from 0.
 * @param   ixh              (Given)    The last x pixel to correct
 *                                      numbering from 0.
 * @param   iyl              (Given)    The first y pixel to correct
 *                                      numbering from 0.
 * @param   iyh              (Given)    The last y pixel to correct
 *                                      numbering from 0.
 * @param   lin              (Given)    The lininfo structure
 *                                      containing the linearity
 *                                      correction polynomial
 *                                      coefficients.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_lincor1_float (
    float *data,
    int nx,
    int ixl,
    int ixh,
    int iyl,
    int iyh,
    qmost_lininfo *lin)
{
    int iy,ix,i,nlut,idat1,idat2;
    float ldat1,ldat2,slope,inter;
    float *lut = NULL;
    unsigned char *lutdef = NULL;

    /* Create a look up table to speed things up. If a value has been used
       before, then we can just reuse it rather than to evaluate the 
       polynomial again. */

    nlut = LUTMAX - LUTMIN + 1;

    lut = cpl_calloc(nlut,sizeof(float));
    lutdef = cpl_calloc(nlut,sizeof(unsigned char));
        
    /* OK, loop through the data and do the linearisation */

    for(iy = iyl; iy <= iyh; iy++) {
        for(ix = ixl; ix <= ixh; ix++) {
            i = iy*nx+ix;

            idat1 = qmost_max(qmost_min((int)data[i],LUTMAX),LUTMIN);
            idat2 = qmost_max(qmost_min((int)data[i]+1,LUTMAX),LUTMIN);
            if (lutdef[idat1-LUTMIN]) {
                ldat1 = lut[idat1-LUTMIN];
            } else {
                ldat1 = cpl_polynomial_eval_1d(lin->coefs,(double)idat1,NULL);
                lut[idat1-LUTMIN] = ldat1;
                lutdef[idat1-LUTMIN] = 1;
            }
            if (lutdef[idat2-LUTMIN]) {
                ldat2 = lut[idat2-LUTMIN];
            } else {
                ldat2 = cpl_polynomial_eval_1d(lin->coefs,(double)idat2,NULL);
                lut[idat2-LUTMIN] = ldat2;
                lutdef[idat2-LUTMIN] = 1;
            }
            slope = ldat2 - ldat1;
            inter = ldat2 - slope*(float)idat2;
            data[i] = slope*data[i] + inter;
        }
    }

    /* Tidy and exit */
    
    cpl_free(lut);
    cpl_free(lutdef);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Make bad pixel mask from a list of detector flats.
 *
 * The given detector flats are normalised to the same background
 * level and median combined using qmost_imcombine_lite().  The
 * individual flats are then compared to the median flat by dividing
 * the flat under comparison by the median flat, and the median and
 * robustly estimated sigma of the resulting ratio image computed.
 * Outlying pixels in the ratio image are flagged using the specified
 * thresholds, and a count of how many times each detector pixel was
 * an outlier is accumulated over all of the flats.  Pixels flagged as
 * an outlier on more than badfrac of the exposures (or at least two,
 * exposures, if this number is smaller) are flagged as bad (true) in
 * the returned bad pixel mask.
 *
 * The detector flats should have different exposure times.  In
 * practice, the linearity ramp is used.
 *
 * @param   flats              (Given)    The list of flats.
 * @param   qclist             (Modified) A propertylist to write QC
 *                                        headers into.
 * @param   lthr               (Given)    The lower threshold in sigma
 *                                        to flag outlying pixels in
 *                                        flats.
 * @param   hthr               (Given)    The upper threshold in sigma
 *                                        to flag outlying pixels in
 *                                        flats.
 * @param   badfrac            (Given)    The fraction of exposures
 *                                        (between 0 and 1) on which a
 *                                        pixel is discordant in order
 *                                        for it to be flagged as bad.
 * @param   bpm                (Returned) The resulting bad pixel
 *                                        mask.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If the imagelist of flats is
 *                                        empty.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the flats have a data
 *                                        type other than float.
 *
 * @par Output QC Parameters:
 *   - <b>BPM BADFRAC</b>: The fraction of pixels on the detector
 *     flagged as bad.
 *   - <b>BPM NBAD</b>: The number of pixels flagged as bad in the bad
 *     pixel mask.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean background level of the
 *     framesthat were combined.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_gen_bpm (
    cpl_imagelist *flats,
    cpl_propertylist *qclist,
    float lthr,
    float hthr,
    float badfrac,
    cpl_mask **bpm)
{
    cpl_image *medflat = NULL;
    int *badct = NULL;

    int iflat, nflats;
    cpl_image *img;
    cpl_image *ratio = NULL;
    float skylev, sigma, low, high;

    int nx, ny, ipix, npix, nbad, nbmax;
    float *ratiobuf;
    unsigned char *bpmbuf;

    cpl_ensure_code(flats != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bpm != NULL, CPL_ERROR_NULL_INPUT);

    *bpm = NULL;

#undef TIDY
#define TIDY                                    \
    if(medflat != NULL) {                       \
        cpl_image_delete(medflat);              \
        medflat = NULL;                         \
    }                                           \
    if(ratio != NULL) {                         \
        cpl_image_delete(ratio);                \
        ratio = NULL;                           \
    }                                           \
    if(badct != NULL) {                         \
        cpl_free(badct);                        \
        badct = NULL;                           \
    }                                           \
    if(*bpm != NULL) {                          \
        cpl_mask_delete(*bpm);                  \
        *bpm = NULL;                            \
    }

    nflats = cpl_imagelist_get_size(flats);

    /* Stack flats */
    if(qmost_imcombine_lite(flats, NULL, NULL,
                            QMOST_MEDIANCALC, 2, 0, 0, 5.0,
                            0, 0.0, 0.0,
                            &medflat, NULL, NULL) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not stack flats");
    }

    /* Normalise median flat to unity */
    if(qmost_skylevel_image(medflat, -1000, 65535,
                            -FLT_MAX, FLT_MAX, 0,
                            &skylev, &sigma) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not determine stacked "
                                     "flat level");
    }

    if(skylev != 0.0) {
        cpl_image_divide_scalar(medflat, skylev);
    }

    /* Compare flats to the median to flag outlying pixels.  Keep
     * track of how many times each pixel is flagged.  XXX - the
     * statistics may need to be done separately per amplifier to
     * avoid running into problems with multimodal distribution. */
    nx = cpl_image_get_size_x(medflat);
    ny = cpl_image_get_size_y(medflat);
    npix = nx * ny;
    
    badct = cpl_calloc(npix, sizeof(int));
    
    for(iflat = 0; iflat < nflats; iflat++) {
        /* Get from imagelist */
        img = cpl_imagelist_get(flats, iflat);

        /* Divide by normalised median flat */
        ratio = cpl_image_divide_create(img, medflat);
        
        /* Statistics of ratio image.  Since we normalised the median
         * flat to unity above, these are still within the range of a
         * 16-bit unsigned integer, so we can use the faster
         * histogram-based analysis. */
        if(qmost_skylevel_image(ratio, -1000, 65535,
                                -FLT_MAX, FLT_MAX, 0,
                                &skylev, &sigma) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not determine ratio "
                                         "flat level");
        }

        /* Decide thresholds */
        low = skylev - lthr * sigma;
        high = skylev + hthr * sigma;

        /* Count pixels outside range */
        ratiobuf = cpl_image_get_data_float(ratio);
        if(ratiobuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer "
                                         "to ratio image");
        }

        for(ipix = 0; ipix < npix; ipix++) {
            if(ratiobuf[ipix] < low || ratiobuf[ipix] > high) {
                badct[ipix]++;
            }
        }

        cpl_image_delete(ratio);
        ratio = NULL;
    }

    cpl_image_delete(medflat);
    medflat = NULL;
    
    /* Create BPM.  Define bad pixels as any pixel flagged as an
     * outlier in more than "badfrac" of the flats. */
    *bpm = cpl_mask_new(nx, ny);
    bpmbuf = cpl_mask_get_data(*bpm);
    
    nbmax = qmost_max(2, qmost_nint(badfrac * nflats));
    
    nbad = 0;
    for(ipix = 0; ipix < npix; ipix++) {
        if(badct[ipix] >= nbmax) {
            bpmbuf[ipix] = 1;
            nbad++;
        }
        else {
            bpmbuf[ipix] = 0;
        }
    }

    cpl_free(badct);
    badct = NULL;

    /* Write QC */
    cpl_propertylist_update_int(qclist, "ESO QC BPM NBAD", nbad);
    cpl_propertylist_set_comment(qclist, "ESO QC BPM NBAD",
                                 "Number of bad pixels");

    cpl_propertylist_update_double(qclist, "ESO QC BPM BADFRAC",
                                   ((double) nbad) / npix);
    cpl_propertylist_set_comment(qclist, "ESO QC BPM BADFRAC",
                                 "Fraction of pixels bad");

    return CPL_ERROR_NONE;
}

/**@}*/
