/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

#include "qmost_constants.h"
#include "qmost_filt2d.h"
#include "qmost_sort.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_filt2d  qmost_filt2d
 *
 * Median and mean filtering of a 2d image.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_filt2d.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/* Structure used by medavg to keep state between calls of the
 * internal statistics routines. */ 

typedef struct {
    float sum;
    float sumw;
    int   naver;
    float nextw;
    float lastw;
    float nextval;
    float lastval;
    short int nextc;
    short int lastc;
} statmon;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void docols (
    float *data,
    unsigned char *bpm,
    int nx,
    int ny,
    int filter, 
    int stat);

static void dorows(
    float *data,
    unsigned char *bpm,
    int nx,
    int ny,
    int filter, 
    int stat);

static void dostat (
    float *data,
    unsigned char *bpm,
    unsigned char *goodval,
    int npts,
    int nfilt,
    int whichstat,
    float *darray,
    unsigned char *barray,
    int *ipoint,
    float *mabuf,
    float *ybuf,
    unsigned char *ybbuf);

static void wraparound (
    float *data,
    unsigned char *bpm,
    int npts,
    int nfilt, 
    int whichstat,
    float *darray,
    unsigned char *barray,
    int *ipoint,
    float *mabuf,
    statmon *sm,
    float *ybuf,
    unsigned char *ybbuf);

static void medavg (
    float *array,
    unsigned char *bpm,
    int *ipoint,
    float *buf,
    int npix, 
    int whichstat,
    int newl,
    statmon *sm,
    float *outval, 
    unsigned char *outbp);

static void quickie (
    float *array,
    unsigned char *iarray,
    int *iarray2, 
    int lll,
    int narray);

static void plugholes (
    float *data,
    unsigned char *bpm,
    int nx);

/*----------------------------------------------------------------------------*/
/**
 * @brief   2D image filtering by applying successive 1D filters.
 *
 * This routine filters a 2D image by applying a 1D filter in each
 * axis sequentially.  The order of the filters is specified by a
 * function argument.
 *
 * This routine can be used instead of a full 2D block filter for
 * cases where such a treatment is not needed.  The sequential 1D
 * filtering is much faster.  The rows/columns are also independent
 * and are therefore done in parallel using OpenMP.
 *
 * @param   img              (Modified) The image to filter.  Will be
 *                                      modified in place.  The data
 *                                      type must be CPL_TYPE_FLOAT.
 * @param   filt             (Given)    The filter kernel size to
 *                                      use.
 * @param   stat             (Given)    QMOST_MEANCALC: use mean
 *                                      (boxcar) filtering.
 *                                      QMOST_MEDIANCALC: use median
 *                                      filtering.
 * @param   axis             (Given)    If 1, filter rows first and
 *                                      then columns.  Otherwise
 *                                      filter columns first then
 *                                      rows.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the image was not of type
 *                                        CPL_TYPE_FLOAT.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_bfilt2 (
    cpl_image *img,
    int filt,
    int stat,
    int axis)
{
    float *data = NULL;
    unsigned char *bpm = NULL;
    int nx, ny;

    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);

    /* If filter kernel is 1 pixel or less, there's nothing to do, so
     * return immediately. */
    if(filt <= 1) {
        return CPL_ERROR_NONE;
    }

    /* Get image */
    data = cpl_image_get_data_float(img);
    if(data == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer to image");
    }

    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);

    /* Get BPM */
    bpm = cpl_mask_get_data(cpl_image_get_bpm(img));
    if(bpm == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get pointer to BPM");
    }

    if(axis == 1) {
        dorows(data,bpm,nx,ny,filt,stat);
        docols(data,bpm,nx,ny,filt,stat);
    }
    else {
        docols(data,bpm,nx,ny,filt,stat);
        dorows(data,bpm,nx,ny,filt,stat);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Apply 1D filter to each column of an image.
 *
 * @param   data             (Modified) The image array to filter.
 *                                      Will be modified in place.
 * @param   bpm              (Given)    The corresponding bad pixel
 *                                      mask.
 * @param   nx               (Given)    The size of the image in x.
 * @param   ny               (Given)    The size of the image in y.
 * @param   filter           (Given)    The filter kernel size to
 *                                      use.
 * @param   stat             (Given)    QMOST_MEANCALC: use mean
 *                                      (boxcar) filtering.
 *                                      QMOST_MEDIANCALC: use median
 *                                      filtering.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void docols (
    float *data,
    unsigned char *bpm,
    int nx,
    int ny,
    int filter, 
    int stat)
{
    int j,k,indx,nn;

    float *dbuf;
    unsigned char *bbuf;
    unsigned char *goodval;
    float *t;
    unsigned char *b;

    float *darray;
    unsigned char *barray;
    int *ipoint;
    float *mabuf;
    float *ybuf;
    unsigned char *ybbuf;

    /* check to make sure the filter size is non-zero and odd */

    if (filter <= 0)
        return;

    if (filter % 2 == 0)
        filter++;

#pragma omp parallel default(none) private(dbuf, bbuf, goodval, t, b, darray, barray, ipoint, mabuf, ybuf, ybbuf, nn, j, indx) shared(data, bpm, nx, ny, filter, stat)
    {
        dbuf = cpl_malloc(ny*sizeof(float));
        bbuf = cpl_malloc(ny*sizeof(unsigned char));
        goodval = cpl_malloc(ny*sizeof(unsigned char));
        t = cpl_malloc(ny*sizeof(float));
        b = cpl_malloc(ny*sizeof(unsigned char));

        darray = cpl_malloc(filter*sizeof(float));
        barray = cpl_malloc(filter*sizeof(unsigned char));
        ipoint = cpl_malloc(filter*sizeof(int));
        mabuf = cpl_malloc(qmost_max(3,filter)*sizeof(float));
        ybuf = cpl_malloc((ny+filter)*sizeof(float));
        ybbuf = cpl_malloc((ny+filter)*sizeof(unsigned char));

#pragma omp for
        for (k = 0; k < nx; k++) {
            memset(goodval,0,ny*sizeof(unsigned char));
            nn = 0;
            for (j = 0; j < ny; j++) {
                indx = j*nx + k;
                if (bpm[indx] == 0) {
                    dbuf[nn] = data[indx];
                    bbuf[nn++] = 0;
                }
            } 
            if (nn < filter) {
                continue;
            }
            dostat(dbuf,bbuf,goodval,nn,filter,stat,
                   darray,barray,ipoint,mabuf,ybuf,ybbuf);
            nn = 0;
            for (j = 0; j < ny; j++) {
                indx = j*nx + k;
                if (bpm[indx] == 0) {
                    t[j] = dbuf[nn++];
                    b[j] = 0;
                } else {
                    t[j] = -999.0;
                    b[j] = 1;
                }
            }
            plugholes(t,b,ny);
            nn = 0;
            for (j = 0; j < ny; j++) {
                indx = j*nx + k;
                data[indx] = t[j];
            }
        }

        cpl_free(dbuf);
        cpl_free(bbuf);
        cpl_free(goodval);
        cpl_free(t);
        cpl_free(b);

        cpl_free(darray);
        cpl_free(barray);
        cpl_free(ipoint);
        cpl_free(mabuf);
        cpl_free(ybuf);
        cpl_free(ybbuf);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Apply 1D filter to each row of an image.
 *
 * @param   data             (Modified) The image array to filter.
 *                                      Will be modified in place.
 * @param   bpm              (Given)    The corresponding bad pixel
 *                                      mask.
 * @param   nx               (Given)    The size of the image in x.
 * @param   ny               (Given)    The size of the image in y.
 * @param   filter           (Given)    The filter kernel size to
 *                                      use.
 * @param   stat             (Given)    QMOST_MEANCALC: use mean
 *                                      (boxcar) filtering.
 *                                      QMOST_MEDIANCALC: use median
 *                                      filtering.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void dorows (
    float *data,
    unsigned char *bpm,
    int nx,
    int ny,
    int filter, 
    int stat)
{
    int j,k,indx,nn;

    float *dbuf;
    unsigned char *bbuf;
    unsigned char *goodval;
    float *t;
    unsigned char *b;

    float *darray;
    unsigned char *barray;
    int *ipoint;
    float *mabuf;
    float *ybuf;
    unsigned char *ybbuf;

    /* check to make sure the filter size is non-zero and odd */

    if (filter <= 0)
        return;

    if (filter % 2 == 0)
        filter++;

#pragma omp parallel default(none) private(dbuf, bbuf, goodval, t, b, darray, barray, ipoint, mabuf, ybuf, ybbuf, nn, j, indx) shared(data, bpm, nx, ny, filter, stat)
    {
        dbuf = cpl_malloc(nx*sizeof(float));
        bbuf = cpl_malloc(nx*sizeof(unsigned char));
        goodval = cpl_malloc(nx*sizeof(unsigned char));
        t = cpl_malloc(nx*sizeof(float));
        b = cpl_malloc(nx*sizeof(unsigned char));

        darray = cpl_malloc(filter*sizeof(float));
        barray = cpl_malloc(filter*sizeof(unsigned char));
        ipoint = cpl_malloc(filter*sizeof(int));
        mabuf = cpl_malloc(qmost_max(3,filter)*sizeof(float));
        ybuf = cpl_malloc((nx+filter)*sizeof(float));
        ybbuf = cpl_malloc((nx+filter)*sizeof(unsigned char));

#pragma omp for
        for (k = 0; k < ny; k++) {
            memset(goodval,0,nx*sizeof(unsigned char));
            nn = 0;
            for (j = 0; j < nx; j++) {
                indx = k*nx + j;
                if (bpm[indx])
                    continue;
                dbuf[nn] = data[indx];
                bbuf[nn++] = 0;
            }
            if (nn < filter) {
                continue;
            }
            dostat(dbuf,bbuf,goodval,nn,filter,stat,
                   darray,barray,ipoint,mabuf,ybuf,ybbuf);
            nn = 0;
            for (j = 0; j < nx; j++) {
                indx = k*nx + j;
                if (bpm[indx] == 0) {
                    t[j] = dbuf[nn++];
                    b[j] = 0;
                } else {
                    t[j] = -999.0;
                    b[j] = 1;
                }
            }
            plugholes(t,b,nx);
            for (j = 0; j < nx; j++) {
                indx = k*nx + j;
                data[indx] = t[j];
            }
        }

        cpl_free(dbuf);
        cpl_free(bbuf);
        cpl_free(goodval);
        cpl_free(t);
        cpl_free(b);

        cpl_free(darray);
        cpl_free(barray);
        cpl_free(ipoint);
        cpl_free(mabuf);
        cpl_free(ybuf);
        cpl_free(ybbuf);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Apply 1D filter to a 1D array.
 *
 * @param   data             (Modified) The 1D data array.  Will be
 *                                      modified in place.
 * @param   bpm              (Given)    The corresponding 1D input bad
 *                                      pixel mask.
 * @param   goodval          (Modified) A caller allocated array to
 *                                      receive the 1D output bad
 *                                      pixel mask.
 * @param   npts             (Given)    The size of the 1D array.
 * @param   nfilt            (Given)    The filter kernel size to
 *                                      use.
 * @param   whichstat        (Given)    QMOST_MEANCALC: use mean
 *                                      (boxcar) filtering.
 *                                      QMOST_MEDIANCALC: use median
 *                                      filtering.
 * @param   darray           (Modified) Caller-allocated float
 *                                      workspace array of length
 *                                      nfilt.
 * @param   barray           (Modified) Caller-allocated byte
 *                                      workspace array of length
 *                                      nfilt.
 * @param   ipoint           (Modified) Caller-allocated int
 *                                      workspace array of length
 *                                      nfilt.
 * @param   mabuf            (Modified) Caller-allocated float
 *                                      workspace of length
 *                                      max(3,nfilt).
 * @param   ybuf             (Modified) Caller-allocated float
 *                                      workspace of length
 *                                      npts+nfilt.
 * @param   ybbuf            (Modified) Caller-allocated byte
 *                                      workspace of length
 *                                      npts+nfilt.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void dostat (
    float *data,
    unsigned char *bpm,
    unsigned char *goodval,
    int npts,
    int nfilt,
    int whichstat,
    float *darray,
    unsigned char *barray,
    int *ipoint,
    float *mabuf,
    float *ybuf,
    unsigned char *ybbuf)
{
    int jl,jh,j,ifree,i;
    unsigned char bval;
    float val;
    statmon sm;

    /* Do the wrap around and load the data into an oversized array*/

    wraparound(data,bpm,npts,nfilt,whichstat,
               darray,barray,ipoint,mabuf,
               &sm,ybuf,ybbuf);

    /* Start doing the filtering...*/

    memcpy(darray,ybuf,nfilt*sizeof(float));
    memcpy(barray,ybbuf,nfilt*sizeof(unsigned char));
    for (j = 0; j < nfilt; j++)
        ipoint[j] = j;
    ifree = 0;
    medavg(darray,barray,ipoint,mabuf,nfilt,whichstat,-1,&sm,&val,&bval);
    if (! bval) 
        data[0] = val;
    goodval[0] = bval;
    jl = nfilt;
    jh = nfilt + npts - 2;
    for (j = jl; j <= jh; j++) {
        for (i = 0; i < nfilt; i++) {
            if (ipoint[i] == 0) {
                ifree = i;
                ipoint[i] = nfilt - 1;
                sm.lastval = darray[ifree];
                sm.lastw = 0.0;
                sm.lastc = 0;
                if (barray[ifree] == 0) {
                    sm.lastw = 1.0;
                    sm.lastc = 1;
                }                
                darray[ifree] = ybuf[j];
                barray[ifree] = ybbuf[j];
                sm.nextval = darray[ifree];
                sm.nextw = 0.0;
                sm.nextc = 0;
                if (barray[ifree] == 0) {
                    sm.nextw = 1.0;
                    sm.nextc = 1;
                }                
            } else
                ipoint[i]--;
        }
        medavg(darray,barray,ipoint,mabuf,nfilt,whichstat,ifree,&sm,&val,&bval);
        if (! bval) 
            data[j-jl+1] = val;
        goodval[j-jl+1] = bval;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Extend a 1D data array for filtering by reflection.
 *
 * The given 1D array is extended beyond its start and end pixels by
 * half the size of the filter kernel to allow these pixels to be
 * filtered.  This is done by reflecting the ends of the array about
 * the first and last pixel.
 *
 * @param   data             (Modified) The input array.
 * @param   bpm              (Given)    The bad pixel mask.
 * @param   npts             (Given)    The size of the arrays.
 * @param   nfilt            (Given)    The filter kernel size to
 *                                      use.
 * @param   whichstat        (Given)    QMOST_MEANCALC: use mean
 *                                      (boxcar) filtering.
 *                                      QMOST_MEDIANCALC: use median
 *                                      filtering.
 * @param   darray           (Modified) Caller-allocated float
 *                                      workspace array of length
 *                                      nfilt.
 * @param   barray           (Modified) Caller-allocated byte
 *                                      workspace array of length
 *                                      nfilt.
 * @param   ipoint           (Modified) Caller-allocated int
 *                                      workspace array of length
 *                                      nfilt.
 * @param   mabuf            (Modified) Caller-allocated float
 *                                      workspace of length
 *                                      max(3,nfilt).
 * @param   sm               (Given)    A structure used by function
 *                                      medavg to keep state between
 *                                      calls.
 * @param   ybuf             (Returned) The resulting extended array.
 * @param   ybbuf            (Returned) The corresponding bad pixel
 *                                      mask.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void wraparound (
    float *data,
    unsigned char *bpm,
    int npts,
    int nfilt, 
    int whichstat,
    float *darray,
    unsigned char *barray,
    int *ipoint,
    float *mabuf,
    statmon *sm,
    float *ybuf,
    unsigned char *ybbuf)
{
    float xmns,xmnf;
    int i1,ilow,i;
    unsigned char bxmns,bxmnf;

    /* Do some padding at the edges */

    i1 = nfilt/2;
    ilow = qmost_max(3,nfilt/4);
    ilow = (ilow/2)*2 + 1;

    /* Do the wrap around.*/

    memcpy(darray,data,ilow*sizeof(float));
    memcpy(barray,bpm,ilow*sizeof(unsigned char));
    medavg(darray,barray,ipoint,mabuf,ilow,whichstat,-1,sm,&xmns,&bxmns);
    memcpy(darray,data+npts-ilow,ilow*sizeof(float));
    memcpy(barray,bpm+npts-ilow,ilow*sizeof(unsigned char));
    medavg(darray,barray,ipoint,mabuf,ilow,whichstat,-1,sm,&xmnf,&bxmnf);
    for (i = 0; i < i1; i++) {
        if (! bxmns) {
            ybuf[i] = 2.0*xmns - data[i1+ilow-i-1];
            ybbuf[i] = bpm[i1+ilow-i-1];
        } else {
            ybuf[i] = data[i1+ilow-i-1];
            ybbuf[i] = 1;
        }
        if (! bxmnf) {
            ybuf[npts+i1+i] = 2.0*xmnf - data[npts-i-ilow-1];
            ybbuf[npts+i1+i] = bpm[npts-i-ilow-1];
        } else {
            ybuf[npts+i1+i] = data[npts-i-ilow-1];
            ybbuf[npts+i1+i] = 1;
        }
    }

    /* Now place the full line into the buffer */

    memcpy(ybuf+i1,data,npts*sizeof(float));
    memcpy(ybbuf+i1,bpm,npts*sizeof(unsigned char));
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Apply the filter to compute a new filtered output
 *          element.
 *
 * This routine is called repeatedly for each output pixel to be
 * produced, where the argument newl should start at -1 to initialize
 * the calculation and then called for each output pixel.  The value
 * of newl must be strictly increasing each call.
 *
 * @param   array            (Modified) The array to average.
 * @param   bpm              (Given)    The bad pixel mask.
 * @param   ipoint           (Given)    An array of pointers which
 *                                      will be maintained sorted in
 *                                      parallel with the data array
 *                                      used by the median filter.
 * @param   buf              (Modified) Caller-allocated float
 *                                      workspace of length npix.
 * @param   npix             (Given)    The size of the array.
 * @param   whichstat        (Given)    QMOST_MEANCALC: use mean
 *                                      (boxcar) filtering.
 *                                      QMOST_MEDIANCALC: use median
 *                                      filtering.
 * @param   newl             (Given)    The index of the element to be
 *                                      computed, or -1 to initialize.
 * @param   sm               (Given)    A structure used to keep state
 *                                      between calls.
 * @param   outval           (Returned) The resulting filtered value.
 * @param   outbp            (Returned) The corresponding bad pixel
 *                                      mask value.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void medavg (
    float *array,
    unsigned char *bpm,
    int *ipoint,
    float *buf,
    int npix, 
    int whichstat,
    int newl,
    statmon *sm,
    float *outval, 
    unsigned char *outbp) { 

    int m,i;

    /* If there is a new element and that new element is bad, then
       the buffer is already sorted from last time (just one element
       sorter */
    
    m = 0;
    if (whichstat == QMOST_MEDIANCALC) {
        if (newl == -1) 
            qmost_sort_fui(array,bpm,ipoint,npix);
        else  
            quickie(array,bpm,ipoint,newl,npix);

        /* Now put everything that's good in the buffer */

        m = 0;
        for (i = 0; i < npix; i++) {
            if (bpm[i] == 0) {
                buf[m] = array[i];
                m++;
            }
        }

        if (m == 0) {
            /* If they were all bad, then send a null result back */
            *outval = 0.0;
            *outbp = 1;
        } else {
            /* Otherwise calculate the relevant stat */
            *outval = buf[m/2];
            *outbp = 0;
        }

    } else if (whichstat == QMOST_MEANCALC) {
        if (newl == -1) {
            sm->sum = 0.0;
            sm->sumw = 0.0;
            sm->naver = 0;
            for (i = 0; i < npix; i++) {
                if (bpm[i] == 0) {
                    sm->sum += array[i];
                    sm->sumw += 1.0;
                    sm->naver += 1;
                }
            }
            m = sm->naver;
        } else {
            sm->sum += (sm->nextw*sm->nextval - sm->lastw*sm->lastval);
            sm->sumw += (sm->nextw - sm->lastw);
            sm->naver += (sm->nextc - sm->lastc);
            m = sm->naver;
        }

        if (m == 0) {
            /* If they were all bad, then send a null result back */
            *outval = 0.0;
            *outbp = 1;
        } else {
            /* Otherwise calculate the relevant stat */
            *outval = sm->sum/sm->sumw;
            *outbp = 0;
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Locate insertion point in sorted array x for element l and
 *          rearrange array as needed to maintain sort order.
 *
 * @param   array      (Modified) The sorted array.
 * @param   iarray     (Modified) Parallel array of integers
 *                                corresponding to elements of x.
 * @param   iarray2    (Modified) Parallel array of integers
 *                                corresponding to elements of x.
 * @param   lll        (Given)    Element to be inserted.
 * @param   narray     (Given)    Number of sorted elements.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void quickie (
    float *array,
    unsigned char *iarray,
    int *iarray2, 
    int lll,
    int narray)
{
    float test;
    int i,j,npt,it2;
    unsigned char it;

    test = array[lll];
    it = iarray[lll];
    it2 = iarray2[lll];
    j = -1;
    for (i = 0; i < narray; i++) {
        if (i != lll && test <= array[i]) {
            j = i;
            break;
        }
    }
    if (j == -1) 
        j = narray;
    if (j - 1 == lll)
        return;

    if (j - lll < 0) {
        npt = lll - j;
        for (i = 0; i < npt; i++) {
            array[lll-i] = array[lll-i-1];
            iarray[lll-i] = iarray[lll-i-1];
            iarray2[lll-i] = iarray2[lll-i-1];
        }
        array[j] = test;
        iarray[j] = it;
        iarray2[j] = it2;
    } else {
        j--;
        npt = j - lll;
        if (npt != 0) {
            for (i = 0; i < npt; i++) {
                array[lll+i] = array[lll+i+1];
                iarray[lll+i] = iarray[lll+i+1];
                iarray2[lll+i] = iarray2[lll+i+1];
            }
        }
        array[j] = test;
        iarray[j] = it;
        iarray2[j] = it2;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Remove bad pixels from a 1d array by linear interpolation
 *          and extrapolation.
 *
 * @param   data       (Modified) The input array.
 * @param   bpm        (Given)    The bad pixel mask.
 * @param   nx         (Given)    The size of the arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void plugholes (
    float *data,
    unsigned char *bpm,
    int nx)
{
    int i,ifirst,ilast,i1,i2,j;
    float nc,d1,d2,t1,t2,slope;

    /* First of all, find the first good value in the array */

    i = 0;
    while (i < nx && bpm[i] != 0)
        i++;
    ifirst = i;

    /* If all the values in the array are bad, then do nothing */

    if (ifirst == nx)
        return;

    /* Find the last good value in the array */

    i = nx - 1;
    while (i >= 0 && bpm[i] != 0) 
        i--;
    ilast = i;

    /* Right, now start from the first good value and fill in any holes in the
       middle part of the array */

    i = ifirst;
    while (i <= ilast) {
        if (bpm[i] == 0) {
            i++;
            continue;
        }
        i1 = i - 1;
        while (bpm[i] != 0) 
            i++;
        i2 = i;
        nc = (float)(i2 - i1 + 1);
        d1 = data[i1];
        d2 = data[i2];
        for (j = i1+1; j <= i2-1; j++) {
            t1 = 1.0 - (float)(j - i1)/nc;
            t2 = 1.0 - t1;
            data[j] = t1*d1 + t2*d2;
        }
    }

    /* Now the left bit... */

    if (ifirst > 0) {
        slope = data[ifirst+1] - data[ifirst];
        for (j = 0; j < ifirst; j++)
            data[j] = slope*(float)(j - ifirst) + data[ifirst];
    }

    /* Now the right bit... */

    if (ilast < nx - 1) {
        slope = data[ilast] - data[ilast-1];
        for (j = ilast; j < nx; j++) 
            data[j] = slope*(float)(j - ilast) + data[ilast];
    }
}

/**@}*/

/*

$Log$
Revision 1.3  2022/11/15 10:16:49  jmi
Parallelised qmost_bfilt2, qmost_hanning_2d, and qmost_blockfilt_2d.
Fixed missing free of dbuf and MEDIANCALC vs the correct MEDIANFILTER
constant in qmost_blockfilt2d causing it to do mean filtering when
median was intended.

Revision 1.2  2019/02/25 10:39:08  jrl
New memory allocation scheme

Revision 1.1  2018/02/20 12:39:09  jim
new entry


*/
