/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_constants.h"
#include "qmost_filt1d.h"
#include "qmost_pca.h"
#include "qmost_skysub.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_skysub  qmost_skysub
 * 
 * Sky subtraction.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_skysub.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static float getscale (
    float *wave,
    float *inspec,
    float *skyspec,
    float *skymask,
    long nx,
    float wmin,
    float wmax);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do scaled sky subtraction with PCA residual removal.
 *
 * An input image is given with spectra of both objects and sky.  The
 * sky fibres are identified from the FIB_USE column of the given
 * FIBINFO table and their spectra are combined into a high SNR master
 * sky.  The master sky is scaled by comparing the sky emission lines
 * in the master and object spectra.  A scaled sky is subtracted from
 * each spectrum.  The sky emission lines in the object spectra are
 * then analysed using a PCA algorithm to identify and remove common
 * sky line residuals.
 *
 * @param   spec_img     (Modified) The input 2D array of spectra to
 *                                  sky subtract.  The data type must
 *                                  be CPL_TYPE_FLOAT.
 * @param   spec_var     (Modified) The corresponding variance array.
 *                                  The data type must be
 *                                  CPL_TYPE_FLOAT.
 * @param   spec_hdr     (Modified) The FITS header for the input 2D
 *                                  spectra with the spectral WCS.
 * @param   fibinfo_tbl  (Given)    The FIBINFO table for the input
 *                                  spectra.
 * @param   neigen       (Given)    The maximum number of eigenvectors
 *                                  to use, or -1 to select a suitable
 *                                  default equal to the number of sky
 *                                  fibres.
 * @param   smoothing    (Given)    A smoothing box in Angstroms to be
 *                                  used in smoothing the fitted
 *                                  continuum.
 * @param   doscale      (Given)    If non-zero, then then initial
 *                                  mean sky will be used to determine
 *                                  a scaling factor for the zeroth
 *                                  order sky subtraction.
 * @param   wmin         (Given)    The minimum wavelength to be used
 *                                  in the analysis (Angstroms), or 0
 *                                  to use the minimum wavelength of
 *                                  the spectra.
 * @param   wmax         (Given)    The maximum wavelength to be used
 *                                  in the analysis (Angstroms), or 0
 *                                  to use the maximum wavelength of
 *                                  the spectra.
 * @param   resid_filt   (Given)    If non-zero, do extra filtering of
 *                                  residuals.
 * @param   diags        (Modified) A pointer to a caller allocated
 *                                  qmost_skysub_diags structure to
 *                                  receive diagnostic information for
 *                                  saving to the optional diagnostic
 *                                  output files, or NULL if these
 *                                  outputs are not required.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If there were no usable sky
 *                                        fibres, or if one of the
 *                                        required input FITS headers
 *                                        or FIBINFO table columns was
 *                                        missing.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the dimensions of the
 *                                        inputs don't match.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the input image data type was
 *                                        not float, or if one of the
 *                                        input FITS header keyword
 *                                        values or FIBINFO table
 *                                        columns had an incorrect
 *                                        data type.
 *
 * @par Input FITS Header Information:
 *   - <b>CD1_1</b>
 *   - <b>CRPIX1</b>
 *   - <b>CRVAL1</b>
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_USE</b>
 *
 * @par Output DRS Headers:
 *   - <b>PCANUM</b>: The number of eigenvectors used in the PCA
 *     analysis.
 *   - <b>PCASUB</b>: Set to true if PCA correction was applied during
 *     sky subtraction.
 *   - <b>PCATOTAL</b> (percent):  The percentage of the total
 *     residual sky variation that was subtracted.
 *   - <b>SKYSUB</b>: Boolean flag, true if sky subtraction was
 *     enabled and the spectra were sky subtracted.
 *
 * @par Output QC Parameters:
 *   - <b>SKY CONT MED</b> (ADU): Median sky continuum level in the
 *     sky fibres. 
 *   - <b>SKY CONT RMS</b> (ADU): RMS variation in sky level between
 *     the sky fibres. 
 *   - <b>SKY NUM</b>: The number of skies found, including sky fibres
 *     and any weakly exposed targets used to supplement the sky
 *     information for the sky residual PCA.
 *   - <b>SKY NUSED</b>: The number of skies used in the sky residual
 *     PCA.
 *   - <b>SKY RESID MED</b>: The median of the sky emission line
 *     residuals after correction, relative to the expected noise.
 *   - <b>SKY RESID RMS</b>: The RMS variation of the sky emission
 *     line residuals after correction, relative to the expected
 *     noise.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_skysub_vshiftpca (
    cpl_image *spec_img,
    cpl_image *spec_var,
    cpl_propertylist *spec_hdr,
    cpl_table *fibinfo_tbl,
    int neigen,
    float smoothing,
    int doscale,
    float wmin,
    float wmax,
    int resid_filt,
    qmost_skysub_diags *diags)
{
    int anynul,i,j,ind1,ind2,nmed,nlin,niter,k,n01,n99,nskymask,usepca;
    int i1,i2,isum,doshift,neigen_used,n,nev;
    int nsky,jj,jc,jf,medfilt,linfilt,j1,j2,nsky_final,ns_all;
    long naxis[2],naxisv[2],nrows;
    float wave1,crpix1,dispersion,*imgdata,sigma;
    float lowcut,highcut;
    float sum,warp,*outputspec,*outputvar,wave,dw,w1,w2,medsky,madsky;
    float medval,madval,wm1,wm2,objsigma,wavewarped,delmed,skycontflux;
    float wmed,contfrac,pcatotal,diagvar;
    float *specvar = NULL;
    double *dbuf;
    float *fbuf;

    cpl_errorstate prestate;
    cpl_error_code code;

    int nresid, jsky;
    float residmed, residsig;

    /* Separate out allocated variables for easier garbage collection */
    float *skyspec = NULL;
    float *skyvar = NULL;
    float *wavein = NULL;
    float *vshifts = NULL;
    float *skyscale = NULL;
    int *targuse = NULL;
    int *skyfibs = NULL;
    float *allskys = NULL;
    float *allskyvars = NULL;
    float *meds = NULL;
    float *sigs = NULL;
    unsigned char *skybpm = NULL;
    float *buf = NULL;
    float *bufv = NULL;
    float *skymad = NULL;
    float *skycont = NULL;
    float *skyline = NULL;
    float *skymask = NULL;
    float *objline = NULL;
    float *objcont = NULL;
    float *skyline_shift = NULL;
    float *skymask_shift = NULL;
    float *contflux = NULL;
    float *chisqu = NULL;
    float *copydata = NULL;
    int *chuck = NULL;
    double *tempsky = NULL;
    double **covar = NULL;
    double *eigenvalues = NULL;
    double *eigenfrac = NULL;
    double **eigenvectors1 = NULL;
    float *ecumulate = NULL;
    float *diffs = NULL;
    double **eigenvectors2 = NULL;
    float *recon = NULL;
    float *shifted_skyvar = NULL;
    float *residbuf = NULL;

    /* These are privately allocated and freed inside the parallel for
     * loop so should not be included in TIDY */
    float *shifted_sky = NULL;
    double **shifted_eigens = NULL;
    double *shifted_eigens_buf = NULL;
    double *looptempsky = NULL;
    float *tempskyf = NULL;
    float *loopskycont = NULL;
    float *loopskyline = NULL;
    float *tmpmask = NULL;

    /* Check for NULL arguments, particularly the outputs so we can
     * initialize them for garbage collection. */
    cpl_ensure_code(spec_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_var != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fibinfo_tbl != NULL, CPL_ERROR_NULL_INPUT);

    if(diags != NULL) {
        memset(diags, 0, sizeof(qmost_skysub_diags));
    }

#undef TIDY
#define TIDY                                            \
    if(skyspec != NULL) {                               \
        cpl_free(skyspec);                              \
        skyspec = NULL;                                 \
    }                                                   \
    if(skyvar != NULL) {                                \
        cpl_free(skyvar);                               \
        skyvar = NULL;                                  \
    }                                                   \
    if(wavein != NULL) {                                \
        cpl_free(wavein);                               \
        wavein = NULL;                                  \
    }                                                   \
    if(vshifts != NULL) {                               \
        cpl_free(vshifts);                              \
        vshifts = NULL;                                 \
    }                                                   \
    if(skyscale != NULL) {                              \
        cpl_free(skyscale);                             \
        skyscale = NULL;                                \
    }                                                   \
    if(targuse != NULL) {                               \
        cpl_free(targuse);                              \
        targuse = NULL;                                 \
    }                                                   \
    if(skyfibs != NULL) {                               \
        cpl_free(skyfibs);                              \
        skyfibs = NULL;                                 \
    }                                                   \
    if(allskys != NULL) {                               \
        cpl_free(allskys);                              \
        allskys = NULL;                                 \
    }                                                   \
    if(allskyvars != NULL) {                            \
        cpl_free(allskyvars);                           \
        allskyvars = NULL;                              \
    }                                                   \
    if(meds != NULL) {                                  \
        cpl_free(meds);                                 \
        meds = NULL;                                    \
    }                                                   \
    if(sigs != NULL) {                                  \
        cpl_free(sigs);                                 \
        sigs = NULL;                                    \
    }                                                   \
    if(skybpm != NULL) {                                \
        cpl_free(skybpm);                               \
        skybpm = NULL;                                  \
    }                                                   \
    if(buf != NULL) {                                   \
        cpl_free(buf);                                  \
        buf = NULL;                                     \
    }                                                   \
    if(bufv != NULL) {                                  \
        cpl_free(bufv);                                 \
        bufv = NULL;                                    \
    }                                                   \
    if(skymad != NULL) {                                \
        cpl_free(skymad);                               \
        skymad = NULL;                                  \
    }                                                   \
    if(skycont != NULL) {                               \
        cpl_free(skycont);                              \
        skycont = NULL;                                 \
    }                                                   \
    if(skyline != NULL) {                               \
        cpl_free(skyline);                              \
        skyline = NULL;                                 \
    }                                                   \
    if(skymask != NULL) {                               \
        cpl_free(skymask);                              \
        skymask = NULL;                                 \
    }                                                   \
    if(objline != NULL) {                               \
        cpl_free(objline);                              \
        objline = NULL;                                 \
    }                                                   \
    if(objcont != NULL) {                               \
        cpl_free(objcont);                              \
        objcont = NULL;                                 \
    }                                                   \
    if(skyline_shift != NULL) {                         \
        cpl_free(skyline_shift);                        \
        skyline_shift = NULL;                           \
    }                                                   \
    if(skymask_shift != NULL) {                         \
        cpl_free(skymask_shift);                        \
        skymask_shift = NULL;                           \
    }                                                   \
    if(contflux != NULL) {                              \
        cpl_free(contflux);                             \
        contflux = NULL;                                \
    }                                                   \
    if(chisqu != NULL) {                                \
        cpl_free(chisqu);                               \
        chisqu = NULL;                                  \
    }                                                   \
    if(copydata != NULL) {                              \
        cpl_free(copydata);                             \
        copydata = NULL;                                \
    }                                                   \
    if(chuck != NULL) {                                 \
        cpl_free(chuck);                                \
        chuck = NULL;                                   \
    }                                                   \
    if(tempsky != NULL) {                               \
        cpl_free(tempsky);                              \
        tempsky = NULL;                                 \
    }                                                   \
    if(covar != NULL) {                                 \
        if(covar[0] != NULL) {                          \
            cpl_free(covar[0]);                         \
        }                                               \
        cpl_free(covar);                                \
        covar = NULL;                                   \
    }                                                   \
    if(eigenvalues != NULL) {                           \
        cpl_free(eigenvalues);                          \
        eigenvalues = NULL;                             \
    }                                                   \
    if(eigenfrac != NULL) {                             \
        cpl_free(eigenfrac);                            \
        eigenfrac = NULL;                               \
    }                                                   \
    if(eigenvectors1 != NULL) {                         \
        if(eigenvectors1[0] != NULL) {                  \
            cpl_free(eigenvectors1[0]);                 \
        }                                               \
        cpl_free(eigenvectors1);                        \
        eigenvectors1 = NULL;                           \
    }                                                   \
    if(ecumulate != NULL) {                             \
        cpl_free(ecumulate);                            \
        ecumulate = NULL;                               \
    }                                                   \
    if(diffs != NULL) {                                 \
        cpl_free(diffs);                                \
        diffs = NULL;                                   \
    }                                                   \
    if(eigenvectors2 != NULL) {                         \
        if(eigenvectors2[0] != NULL) {                  \
            cpl_free(eigenvectors2[0]);                 \
        }                                               \
        cpl_free(eigenvectors2);                        \
        eigenvectors2 = NULL;                           \
    }                                                   \
    if(recon != NULL) {                                 \
        cpl_free(recon);                                \
        recon = NULL;                                   \
    }                                                   \
    if(shifted_skyvar != NULL) {                        \
        cpl_free(shifted_skyvar);                       \
        shifted_skyvar = NULL;                          \
    }                                                   \
    if(diags != NULL) {                                 \
        if(diags->eigenvectors != NULL) {               \
            cpl_image_delete(diags->eigenvectors);      \
            diags->eigenvectors = NULL;                 \
        }                                               \
        if(diags->eigeninfo != NULL) {                  \
            cpl_table_delete(diags->eigeninfo);         \
            diags->eigeninfo = NULL;                    \
        }                                               \
        if(diags->orig_img != NULL) {                   \
            cpl_image_delete(diags->orig_img);          \
            diags->orig_img = NULL;                     \
        }                                               \
        if(diags->orig_var != NULL) {                   \
            cpl_image_delete(diags->orig_var);          \
            diags->orig_var = NULL;                     \
        }                                               \
        if(diags->skyinfo != NULL) {                    \
            cpl_table_delete(diags->skyinfo);           \
            diags->skyinfo = NULL;                      \
        }                                               \
        if(diags->comb_img != NULL) {                   \
            cpl_image_delete(diags->comb_img);          \
            diags->comb_img = NULL;                     \
        }                                               \
        if(diags->comb_var != NULL) {                   \
            cpl_image_delete(diags->comb_var);          \
            diags->comb_var = NULL;                     \
        }                                               \
        if(diags->subt_img != NULL) {                   \
            cpl_image_delete(diags->subt_img);          \
            diags->subt_img = NULL;                     \
        }                                               \
        if(diags->subt_var != NULL) {                   \
            cpl_image_delete(diags->subt_var);          \
            diags->subt_var = NULL;                     \
        }                                               \
    }                                                   \
    if(residbuf != NULL) {                              \
        cpl_free(residbuf);                             \
        residbuf = NULL;                                \
    }

/* These macros are used to check the FIB_USE column */
#define ISSKY(a) ((a) == 3)
#define ISTARGET(a) ((a) == 1)
#define USEFORSKY(a) ((a) == 3 || (a) == 1)
#define CORRECTSKY(a) ((a) != 0)

    /* Get input 2D spectra */
    naxis[0] = cpl_image_get_size_x(spec_img);
    naxis[1] = cpl_image_get_size_y(spec_img);

    imgdata = cpl_image_get_data_float(spec_img);
    if(imgdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Now the variance array */
    naxisv[0] = cpl_image_get_size_x(spec_var);
    naxisv[1] = cpl_image_get_size_y(spec_var);

    if(naxisv[0] != naxis[0] ||
       naxisv[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxisv[0], naxisv[1]);
    }

    specvar = cpl_image_get_data_float(spec_var);
    if(specvar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D variance image");
    }

    /* Get the wavelength information */
    if(qmost_cpl_propertylist_get_float(spec_hdr,
                                        "CRVAL1",
                                        &wave1) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read required keyword "
                                     "CRVAL1 from profile FITS header");
    }

    if(qmost_cpl_propertylist_get_float(spec_hdr,
                                        "CRPIX1",
                                        &crpix1) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read required keyword "
                                     "CRPIX1 from profile FITS header");
    }

    if(qmost_cpl_propertylist_get_float(spec_hdr,
                                        "CD1_1",
                                        &dispersion) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read required keyword "
                                     "CD1_1 from profile FITS header");
    }

    /* Get some workspace */
    skyspec = cpl_calloc(naxis[0], sizeof(float));
    skyvar = cpl_calloc(naxis[0], sizeof(float));
    wavein = cpl_calloc(naxis[0], sizeof(float));
    vshifts = cpl_calloc(naxis[1], sizeof(float));
    skyscale = cpl_calloc(naxis[1], sizeof(float));

    /* Create a wavelength array */
    for (i = 0; i < naxis[0]; i++) {
	wavein[i] = wave1 + (((float) i) + 1.0 - crpix1) * dispersion;
    }

    if (wmin == 0.0 && wmax == 0.0) {
        wmin = wavein[0];
        wmax = wavein[naxis[0]-1];
    }
    
    /* Read FIB_USE from FIBINFO */
    nrows = cpl_table_get_nrow(fibinfo_tbl);
    if(nrows != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "FIBINFO table has "
                                     "%ld fibres, "
                                     "number of spectra is %ld, "
                                     "these must match",
                                     nrows,
                                     naxis[1]);
    }

    targuse = cpl_malloc(nrows*sizeof(int));

    for(i = 0; i < nrows; i++) {
        targuse[i] = cpl_table_get_int(fibinfo_tbl, 
                                      "FIB_USE",
                                      i,
                                      &anynul);
        if(anynul < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d from FIBINFO",
                                         "FIB_ID",
                                         i+1);
        }
        else if(anynul > 0) {  /* NULL */
            targuse[i] = -1;
        }
    }

    /* Work out which ones are sky fibres */

    skyfibs = cpl_malloc(nrows*sizeof(int));

    nsky = 0;
    for (i = 0; i < nrows; i++) {
        if (ISSKY(targuse[i])) {
            skyfibs[nsky] = i + 1;
            nsky++;
        }
    }

    /* If there are no sky fibres, then get out of here now */

    if (nsky < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "no sky fibres found in input "
                                     "FIBINFO table");
    }

    /* Set default number of eigenvectors if requested */

    if(neigen < 0) {
        neigen = nsky;
    }

    /* Reallocate the sky fibres and get some space for the sky data */
    
    allskys = cpl_malloc(nsky*naxis[0]*sizeof(float));
    allskyvars = cpl_malloc(nsky*naxis[0]*sizeof(float));

    /* Look at input sky images and see which ones might be contaminated */
    
    meds = cpl_calloc(nsky,sizeof(float)); 
    sigs = cpl_calloc(nsky,sizeof(float));
    skybpm = cpl_calloc(naxis[0],sizeof(unsigned char));

    for (j = 0; j < nsky; j++) {
	ind1 = j*naxis[0];
	ind2 = (skyfibs[j]-1)*naxis[0];
        memcpy(allskys+ind1,imgdata+ind2,naxis[0]*sizeof(float));
        memcpy(allskyvars+ind1,specvar+ind2,naxis[0]*sizeof(float));
        memset(skybpm,0,naxis[0]*sizeof(unsigned char));
        for (i = 0; i < naxis[0]; i++) {
            if (allskyvars[ind1+i] == 0)
                skybpm[i] = 1;
        }
        lowcut = -1.0e10;
        highcut = 1.0e10;
        for (i = 0; i < 3; i++) {
            prestate = cpl_errorstate_get();
            code = qmost_medmadcut(allskys+ind1,skybpm,(int)naxis[0],lowcut,
                                   highcut,&medsky,&madsky);
            if(code != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);

                medsky = 0.0;
                madsky = 0.0;

                break;
            }
            lowcut = medsky - 4.44*madsky;
            highcut = medsky + 4.44*madsky;
        }
        meds[j] = medsky;
        sigs[j] = 1.48*madsky;                
    }

    cpl_free(skybpm);
    skybpm = NULL;

    prestate = cpl_errorstate_get();

    if(qmost_medmad(meds,NULL,nsky,&medsky,&madsky) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
        medsky = 0;
        madsky = 0;
    }
    
    /* Make a first pass sky by just averaging the sky fibres together. 
       NB: we're assuming these are all on the same wavelength scale so
       we don't have to do any interpolation here. */

    buf = cpl_malloc(nsky*sizeof(float));
    bufv = cpl_malloc(nsky*sizeof(float));
    skymad = cpl_calloc(naxis[0],sizeof(float));

    for (j = 0; j < naxis[0]; j++) {
        i1 = 0;
	for (i = 0; i < nsky; i++) {
            ind2 = (skyfibs[i]-1)*naxis[0] + j;
            if (specvar[ind2] != 0.0) {
                buf[i1] = imgdata[ind2];
                bufv[i1++] = specvar[ind2];
            }
        }
        if (i1 > 0) {
            prestate = cpl_errorstate_get();

            if(qmost_medmad(buf,NULL,i1,&medval,&madval) != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);
                medval = 0;
                madval = 0;
            }

            madval *= 1.48;
            skymad[j] = madval;
            sum = 0.0;
            isum = 0;
            for (k = 0; k < i1; k++) {
                if (fabsf(buf[k] - medval) < 3.0*madval) {
                    sum += buf[k];
                    isum++;
                }
            }
            if (isum > (nsky+3)/4) {
                skyspec[j] = sum/(float)isum;
                skyvar[j] = madval*madval/(float)isum;
            } else {
                skyspec[j] = 0.0;
                skyvar[j] = 0.0;
            }
        } else {
	    skyspec[i] = 0.0;
            skyvar[i] = 0.0;
        }
    }

    cpl_free(buf);
    buf = NULL;

    cpl_free(bufv);
    bufv = NULL;

    /* Smooth over a given smoothing box and get a background rms */

    nmed = (int)(smoothing/fabs(dispersion) + 0.5);
    nmed = 2*(nmed/2) + 1;
    nlin = nmed/3;
    nlin = 2*(nlin/2) + 1;
    niter = 3;

    skycont = cpl_calloc(naxis[0],sizeof(float));
    skyline = cpl_calloc(naxis[0],sizeof(float));
    skymask = cpl_calloc(naxis[0],sizeof(float));

    n01 = qmost_nint(0.01*(float)naxis[0]);
    n99 = qmost_nint(0.99*(float)naxis[0]);
    for (i = 0; i < naxis[0]; i++) {
        if (skyvar[i] == 0.0) {
            skymask[i] = 100.0;
        } else if (i < n01 || i > n99) {
            skymask[i] = 50.0;
        }
    }
    qmost_skyfilt(naxis[0],skyspec,skycont,skyline,skymask,&sigma,nlin,nmed,
                  niter,3.0,1.0);

    prestate = cpl_errorstate_get();

    if(qmost_med(skycont,NULL,naxis[0],&skycontflux) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
        skycontflux = 0;
    }

    /* Now do the same thing for the individual spectra */

    objline = cpl_calloc(naxis[0]*naxis[1],sizeof(float));
    objcont = cpl_calloc(naxis[0]*naxis[1],sizeof(float));

#pragma omp parallel default(none) private(tmpmask, i, ind1, objsigma) shared(naxis, targuse, imgdata, specvar, n01, n99, objcont, objline, nlin, nmed, niter, skyvar)
    {
        tmpmask = cpl_malloc(naxis[0]*sizeof(float));

#pragma omp for
        for (j = 0; j < naxis[1]; j++) {
            ind1 = j*naxis[0];

            if (USEFORSKY(targuse[j])) {
                for (i = 0; i < naxis[0]; i++) {
                    tmpmask[i] = 0.0;
                    if (specvar[ind1+i] == 0.0)
                        tmpmask[i] = 100.0;
                    if (i < n01 || i > n99)
                        tmpmask[i] = 50.0;
                }

                qmost_skyfilt(naxis[0],imgdata+ind1,objcont+ind1,objline+ind1,
                              tmpmask,&objsigma,nlin,nmed,niter,3.0,1.0);
                for (i = 0; i < naxis[0]; i++) {
                    if (skyvar[i] == 0)
                        objline[ind1+i] = 0.0;
                }

            }
        }

        cpl_free(tmpmask);
        tmpmask = NULL;
    }

    /* Redefine sky mask. Grow any lines to an extra 1 pixel on either side */

    nskymask = 0;
    for (i = 0; i < naxis[0]; i++) {
        if (skyvar[i] == 0.0) {
            skyline[i] = 0.0;
            skyspec[i] = skycont[i];
        }
        if (skyline[i] > 10.0*sigma && wavein[i] > 5400.0) {
            skymask[i] = 100.0;
            nskymask++;
            skyline[i] = qmost_max(-5.0*sigma,skyline[i]);
        } else {
            skymask[i] = 0.0;
        }
    }
    for (i = 1; i < naxis[0]-2; i++) {
        if (skymask[i] > 50.0) {
            for (j = i-1; j <= i+1; j++) {
                if (j == i)
                    continue;
                if (skymask[j] == 0.0) {
                    skymask[j] = 50.0;
                    nskymask++;
                }
            }
        }
    }

    cpl_msg_debug(cpl_func,
                  "sky sigma and nskymask %f %d",
                  sigma,nskymask);

    /* If hardly any lines were found, modify the sky mask so that the whole 
       spectrum will be used to work out the emission line scale factor */

    usepca = 1;
    if (nskymask < 25) {
        usepca = 0;
        for (i = 0; i < naxis[0]; i++) 
            if (skyvar[i] != 0.0)
                skymask[i] = 1.0;
    }

    /* Right...work out the emission line scale factors. There several
       possible combinations depending upon whether we want to use the 
       emission lines to work out the scale factor and whether we have
       a velocity shift or not. NB: this is done for all spectra, not 
       just the sky fibres */

    skyline_shift = cpl_malloc(naxis[0]*sizeof(float));
    skymask_shift = cpl_malloc(naxis[0]*sizeof(float));

    for (j = 0; j < naxis[1]; j++) {
        if (! CORRECTSKY(targuse[j])) {
            skyscale[j] = 0.0;
            continue;
        }
        ind1 = j*naxis[0];

        /* We don't want to do scaling, so by default the scale factors are
           all 1.0 */

        if (! doscale) {
            skyscale[j] = 1.0;
            continue;
        }

        /* Do we have a velocity shift in the object spectrum? */
        
        doshift = (vshifts[j] != 0.0);

        /* If there is a velocity shift, then we need to create a temporary
           shifted sky. If not, then we can use the sky emission lines
           and the mask as the are... */

        if (! doshift) {
	    skyscale[j] = getscale(wavein,objline+ind1,skyline,skymask,
                                   naxis[0],wmin,wmax);
        } else {
            warp = 1.0 + vshifts[j]/QMOST_SPEEDOFLIGHT;
            for (i = 0; i < naxis[0]; i++) {
                wavewarped = wavein[i]/warp;
                ind1 = (int)((wavewarped - wave1)/dispersion + crpix1 - 1);
                if (ind1 >= naxis[0]) {
                    skyline_shift[i] = skyline[naxis[0]-1];
                    skymask_shift[i] = skymask[naxis[0]-1];
                } else if (ind1 < 0) {
                    skyline_shift[i] = skyline[0];
                    skymask_shift[i] = skymask[0];
                } else {
                    ind2 = qmost_min(naxis[0]-1,ind1+1);
                    dw = wavewarped - wavein[ind1];
                    w1 = 1.0 - dw/dispersion;
                    w2 = 1.0 - w1;
                    skyline_shift[i] = w1*skyline[ind1] + w2*skyline[ind2];
                    skymask_shift[i] = w1*skymask[ind1] + w2*skymask[ind2];
                }
            }
            wm1 = wmin/warp;
            wm2 = wmax/warp;
            skyscale[j] = getscale(wavein,objline+ind1,skyline_shift,
                                   skymask_shift,naxis[0],wm1,wm2);
        }
    }

    j = 0;

    cpl_free(skyline_shift);
    skyline_shift = NULL;

    cpl_free(skymask_shift);
    skymask_shift = NULL;

    cpl_free(objline);
    objline = NULL;

    /* Subtract off scaled sky and look at what's left */

    objline = cpl_calloc(naxis[1],sizeof(float));
    contflux = cpl_calloc(naxis[1],sizeof(float));
    chisqu = cpl_calloc(naxis[1],sizeof(float));
    buf = cpl_calloc(naxis[0],sizeof(float));
    copydata = cpl_malloc(naxis[0]*naxis[1]*sizeof(float));

    memcpy(copydata,imgdata,naxis[0]*naxis[1]*sizeof(float));

    j1 = (int)(0.01*(float)naxis[0] + 0.5);
    j2 = (int)(0.99*(float)naxis[0] + 0.5);
    for (i = 0; i < naxis[1]; i++) {
        jc = 0;
        jf = 0;
        jj = 0;
        for (j = 0; j < naxis[0]; j++) {
            ind1 = i*naxis[0] + j;
            if (specvar[ind1] == 0.0) {
                imgdata[ind1] = 0.0;
                objcont[ind1] = 0.0;
            } else {
                imgdata[ind1] -= skyspec[j]*skyscale[i];
                objcont[ind1] -= skycont[j]*skyscale[i];
                if (j > j1 && j < j2) {
                    buf[jj++] = objcont[ind1];
                    if (skymask[j] > 0.0) {
                        jf++;
                        objline[i] += (imgdata[ind1] - objcont[ind1]);
                    }
                    if (skymad[j] > 0.0 && skyvar[j] > 0.0) {
                        jc++;
                        if (ISSKY(targuse[i])) {
                            chisqu[i] += pow(imgdata[ind1]/skymad[j],2.0);
                        } else {
                            chisqu[i] += pow((imgdata[ind1]-objcont[ind1])/skymad[j],2.0);
                        }
                    }
                }
            }
        }
        if (jj > 0) {
            prestate = cpl_errorstate_get();

            if(qmost_med(buf,NULL,jj,contflux+i) != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);
                contflux[i] = 0;
            }
        }
        if (jc > 0)
            chisqu[i] /= (float)jc;
        if (jf > 0)
            objline[i] /= (float)jf;        
    }

    cpl_free(buf);
    buf = NULL;

    /* Do some extra residual filtering if requested */

    if (resid_filt) {
        buf = cpl_malloc(naxis[1]*sizeof(float));
        medfilt = 21;
        linfilt = 7;
        for (j = 0; j < naxis[0]; j++) {
            if (skymask[j] > 0.0) {
                for (i = 0; i < naxis[1]; i++) {
                    ind1 = i*naxis[0] + j;
                    buf[i] = imgdata[ind1] - objcont[ind1];
                    if (specvar[ind1] == 0.0)
                        buf[i] = -1000.0;
                }
                qmost_filt1d(buf,naxis[1],medfilt,linfilt,-1000.0);

                prestate = cpl_errorstate_get();

                if(qmost_med(buf,NULL,naxis[1],&wmed) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                    wmed = 0;
                }

                for (i = 0; i < naxis[1]; i++) {
                    ind1 = i*naxis[0] + j;
                    if (specvar[ind1] != 0.0) 
                        imgdata[ind1] -= (buf[i] - wmed);
                }
            }
        }

        cpl_free(buf);
        buf = NULL;
    }

    /* OK, make a final judgement on which skies to use in the analysis */

    nsky_final = 0; /* The number to include in the analysis */
    ns_all = 0; /* The number to include in the analysis + any skies that
                   are rejected */
    contfrac = 0.1;    
    for (i = 0; i < naxis[1]; i++) {
        if (ISSKY(targuse[i])) {
            ns_all++;
            if (chisqu[i] < 3.0 || usepca == 0) 
                nsky_final++;
        } else if (ISTARGET(targuse[i]) && 
                   fabs(contflux[i]) < contfrac*skycontflux &&
                   fabs(objline[i]) < 0.02*skycontflux &&
                   chisqu[i] < 3.0) {
            ns_all++;
            nsky_final++;
        }
    }
    if (nsky_final > 2*nsky)
        contfrac = 0.01;

    /* Redo this with possible new value of contfrac and keep a copy of all
       of the skies and targets used along with a flag */

    allskys = cpl_realloc(allskys,ns_all*naxis[0]*sizeof(float));
    allskyvars = cpl_realloc(allskyvars,ns_all*naxis[0]*sizeof(float));
    chuck = cpl_malloc(ns_all*sizeof(float));
    skyfibs = cpl_realloc(skyfibs,ns_all*sizeof(int));

    nsky_final = 0;
    ns_all = 0;
    for (i = 0; i < naxis[1]; i++) {
        ind1 = i*naxis[0];
        ind2 = ns_all*naxis[0];
        if (ISSKY(targuse[i])) {
            memcpy(allskys+ind2,copydata+ind1,naxis[0]*sizeof(float));
            memcpy(allskyvars+ind2,specvar+ind1,naxis[0]*sizeof(float));
            chuck[ns_all] = 1;
            skyfibs[ns_all] = i + 1;
            if (chisqu[i] < 3.0 || usepca == 0) {  
                nsky_final++;
                chuck[ns_all] = 0;
            }
            ns_all++;
        } else if (ISTARGET(targuse[i]) &&
                   (fabs(contflux[i]) < contfrac*skycontflux) &&
                   (fabs(objline[i]) < 0.02*skycontflux) &&
                   chisqu[i] < 3.0) {
            memcpy(allskys+ind2,copydata+ind1,naxis[0]*sizeof(float));
            memcpy(allskyvars+ind2,specvar+ind1,naxis[0]*sizeof(float));
            skyfibs[ns_all] = i + 1;
            chuck[ns_all] = 2;
            ns_all++;
            nsky_final++;
        }
    }

    cpl_free(copydata);
    copydata = NULL;

    cpl_msg_debug(cpl_func,
                  "Final no. of skies to be used: %d %d",
                  nsky_final,ns_all);

    if(nsky_final < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "no skies remain, check FIBINFO table");
    }

    allskys = cpl_realloc(allskys,ns_all*naxis[0]*sizeof(float));
    allskyvars = cpl_realloc(allskyvars,ns_all*naxis[0]*sizeof(float));
    chuck = cpl_realloc(chuck,ns_all*sizeof(float));
    skyfibs = cpl_realloc(skyfibs,ns_all*sizeof(int));

    /* Get some workspace for the PCA work */

    tempsky = cpl_calloc(nsky_final*naxis[0],sizeof(double));

    /* Form the eigen vectors from the sky subtracted sky spectra. Start
       subtracting of the scaled mean sky from all the sky fibres */

    n = 0;
    diagvar = 0.0;
    for (j = 0; j < ns_all; j++) {
        ind1 = (skyfibs[j]-1)*naxis[0];
        if (chuck[j] == 0) {
 	   for (i = 0; i < naxis[0]; i++) { 
             tempsky[n*naxis[0]+i] = imgdata[ind1+i];
	     if (skymask[i] > 0.0)
	       diagvar = diagvar + specvar[ind1+i];
	   }
           n++;
        } else if (chuck[j] == 2) {
	   for (i = 0; i < naxis[0]; i++) {
             tempsky[n*naxis[0]+i] = imgdata[ind1+i] - objcont[ind1+i];
	     if (skymask[i] > 0.0)
	       diagvar = diagvar + specvar[ind1+i];
	   }
           n++;
        }
    }

    /* Form the covariance matrix */

    nsky = nsky_final;
    covar = qmost_pca_form_covar(tempsky,skymask,nsky,(int)naxis[0]);

    /* sky <noise> contribution to diagonal = eigenvalue offset */

    if(isfinite(diagvar)) {
        diagvar = diagvar/(float)nsky;
    }
    else {
        /* I've seen this overflow in cases where something went wrong
         * in spectral extraction and the extracted spectra are
         * nonsense values.  This traps the infinity to prevent a
         * crash later due to cpl_propertylist / cfitsio being fussy
         * about nans and infinities. */
        diagvar = 0;
    }

    cpl_msg_debug(cpl_func,
                  "No. of sky fibres used = %d %f",
                  nsky,diagvar);

    /* Get the eigens */

    qmost_pca_get_eigen(covar,nsky,&eigenvalues,&eigenfrac,
                        &eigenvectors1,diagvar);

    cpl_free(covar[0]);
    cpl_free(covar);
    covar = NULL;

    nev = nsky;

    /* Work out how many eigenvectors to include */

    buf = cpl_malloc(nev*sizeof(float));

    ecumulate = cpl_malloc(nev*sizeof(float));
    ecumulate[0] = eigenfrac[0];

    j = 0;

    diffs = cpl_malloc(nev*sizeof(float));
    diffs[0] = 1.0;

    for (i = 1; i < nev; i++) {
      ecumulate[i] = ecumulate[i-1] + qmost_max(0.0,eigenfrac[i]);
        diffs[i] = 100.0*(eigenfrac[i-1] - eigenfrac[i]);
        if (diffs[i] > 0.001 && diffs[i] < 1.0)
            buf[j++] = diffs[i];
    }
    if (j > 0) {
        prestate = cpl_errorstate_get();

        if(qmost_med(buf,NULL,j,&delmed) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            delmed = 0;
        }
    } else {
        delmed = 0.01;
    }
    i = 0;
    while (ecumulate[i] < 0.98 && i < qmost_min(neigen,nev) - 1
	                       && diffs[i] > delmed) 
        i++;
    pcatotal = 100.0*ecumulate[i];
    neigen_used = i + 1;
    cpl_msg_debug(cpl_func,
                  "Eigen info: %d %f %f",
                  neigen_used,pcatotal,delmed);

    cpl_free(buf);
    buf = NULL;

    cpl_free(diffs);
    diffs = NULL;

    /* Now we need to transform the eigenvectors */

    eigenvectors2 = qmost_pca_trans_eigen(eigenvectors1,tempsky,skymask,nev,
                                          (int)naxis[0]);

    cpl_free(eigenvectors1[0]);
    cpl_free(eigenvectors1);
    eigenvectors1 = NULL;

    cpl_free(tempsky);
    tempsky = NULL;

    /* Get space for the reconstructed skies, for a shifted mean sky and
       for shifted eigenvectors */
    
    recon = cpl_calloc(naxis[0]*naxis[1],sizeof(float));
    shifted_skyvar = cpl_malloc(naxis[0]*naxis[1]*sizeof(float));

    /* Loop for each fibre */
#pragma omp parallel default(none) private(shifted_sky, shifted_eigens, shifted_eigens_buf, looptempsky, tempskyf, warp, outputspec, outputvar, i, wave, ind1, w1, w2, i1, i2, dw, k, loopskycont, loopskyline, sigma) shared(naxis, targuse, nev, vshifts, imgdata, specvar, wavein, wave1, crpix1, dispersion, skyspec, skyvar, shifted_skyvar, eigenvectors2, skymask, nlin, nmed, niter, usepca, recon, neigen_used)
    {
        shifted_sky = cpl_malloc(naxis[0]*sizeof(float));
        shifted_eigens = cpl_malloc(nev*sizeof(double *));
        shifted_eigens_buf = cpl_malloc(nev*naxis[0]*sizeof(double));
        looptempsky = cpl_malloc(naxis[0]*sizeof(double));
        tempskyf = cpl_malloc(naxis[0]*sizeof(float));
        loopskycont = cpl_malloc(naxis[0]*sizeof(float));
        loopskyline = cpl_malloc(naxis[0]*sizeof(float));

#pragma omp for
        for (j = 0; j < naxis[1]; j++) {
            if (! CORRECTSKY(targuse[j]))
                continue;

            for (i = 0; i < nev; i++)
                shifted_eigens[i] = shifted_eigens_buf + i*naxis[0];

            warp = 1.0 + vshifts[j]/QMOST_SPEEDOFLIGHT;
            outputspec = imgdata + j*naxis[0];
            outputvar = specvar + j*naxis[0];

            /* Shift the mean sky and the neigen eigenvectors by the velocity */

            for (i = 0; i < naxis[0]; i++) {
                wave = wavein[i]/warp;
                ind1 = (int)((wave - wave1)/dispersion + crpix1 - 1);
                if (ind1 >= naxis[0]) {
                    w1 = 1.0;
                    w2 = 0.0;
                    i1 = naxis[0] - 1;
                    i2 = naxis[0] - 1;
                } else if (ind1 < 0) {
                    w1 = 1.0; 
                    w2 = 0.0;
                    i1 = 0;
                    i2 = 0;
                } else {
                    i1 = ind1;
                    i2 = qmost_min(naxis[0]-1,ind1+1);
                    dw = wave - wavein[ind1];
                    w1 = 1.0 - dw/dispersion;
                    w2 = 1.0 - w1;
                }
                shifted_sky[i] = w1*skyspec[i1] + w2*skyspec[i2];
                shifted_skyvar[j*naxis[0]+i] = w1*skyvar[i1] + w2*skyvar[i2];
                for (k = 0; k < nev; k++) 
                    shifted_eigens[k][i] = w1*eigenvectors2[k][i1] +
                        w2*eigenvectors2[k][i2];
                if (specvar[j*naxis[0]+i] == 0.0) {
                    tempskyf[i] = 0.0;
                } else {
                    tempskyf[i] = imgdata[j*naxis[0]+i] ;
                }
                looptempsky[i] = (double)tempskyf[i];
            }

            /* Filter out the continuum */

            qmost_skyfilt(naxis[0],tempskyf,loopskycont,loopskyline,skymask,
                          &sigma,nlin,nmed,niter,3.0,1.0);
            for (i = 0; i < naxis[0]; i++)
                looptempsky[i] -= (double)loopskycont[i];

            /* Reconstruct a sky now */
            if (usepca)
                qmost_pca_recon_spec(shifted_eigens,looptempsky,(int)naxis[0],
                                     skymask,specvar+j*naxis[0],neigen_used,
                                     recon+j*naxis[0]);
        
            /* Correct the spectrum now */
        
            for (i = 0; i < naxis[0]; i++) {
                if (shifted_skyvar[j*naxis[0]+i] == 0.0)
                    recon[j*naxis[0]+i] = 0.0;
                if (specvar[j*naxis[0]+i] == 0.0) {
                    outputspec[i] = 0.0;
                    outputvar[i] = 0.0;
                } else {
                    outputspec[i] -= recon[j*naxis[0]+i];
                    outputvar[i] += shifted_skyvar[j*naxis[0]+i];
                }
            }
        }

        cpl_free(shifted_sky);
        shifted_sky = NULL;

        cpl_free(shifted_eigens);
        shifted_eigens = NULL;

        cpl_free(shifted_eigens_buf);
        shifted_eigens_buf = NULL;

        cpl_free(looptempsky);
        looptempsky = NULL;

        cpl_free(tempskyf);
        tempskyf = NULL;

        cpl_free(loopskycont);
        loopskycont = NULL;

        cpl_free(loopskyline);
        loopskyline = NULL;
    }

    /* Write out diagnostic information if requested */
    if(diags != NULL) {
        /* Eigenvectors */
        diags->eigenvectors = cpl_image_new(naxis[0], nev+1, CPL_TYPE_DOUBLE);
        dbuf = cpl_image_get_data_double(diags->eigenvectors);
        if(dbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get double pointer to "
                                         "eigenvectors");
        }

        for(i = 0; i < naxis[0]; i++) {
            dbuf[i] = skyspec[i];
        }

        for (j = 0; j < nev; j++) {
            memcpy(dbuf + (j+1) * naxis[0],
                   eigenvectors2[j],
                   naxis[0] * sizeof(double));
        }

        /* Eigeninfo table giving eigenvalues */
        diags->eigeninfo = cpl_table_new(nev+1);
        cpl_table_new_column(diags->eigeninfo, "vecnum", CPL_TYPE_INT);
        cpl_table_new_column(diags->eigeninfo, "eigenvalue", CPL_TYPE_FLOAT);
        cpl_table_new_column(diags->eigeninfo, "eigenfraction", CPL_TYPE_FLOAT);
        cpl_table_new_column(diags->eigeninfo, "ecumulate", CPL_TYPE_FLOAT);

        cpl_table_set_int(diags->eigeninfo, "vecnum", 0, 0);
        cpl_table_set(diags->eigeninfo, "eigenvalue", 0, 0);
        cpl_table_set(diags->eigeninfo, "eigenfraction", 0, 0);
        cpl_table_set(diags->eigeninfo, "ecumulate", 0, 0);

        for (j = 0; j < nev; j++) {
            cpl_table_set_int(diags->eigeninfo, "vecnum", j+1, j+1);
            cpl_table_set(diags->eigeninfo, "eigenvalue", j+1, eigenvalues[j]);
            cpl_table_set(diags->eigeninfo, "eigenfraction", j+1, eigenfrac[j]);
            cpl_table_set(diags->eigeninfo, "ecumulate", j+1, ecumulate[j]);
        }

        /* Original sky spectra and variance */
        diags->orig_img = cpl_image_new(naxis[0], ns_all, CPL_TYPE_FLOAT);
        fbuf = cpl_image_get_data_float(diags->orig_img);
        if(fbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "original sky spectra");
        }

        memcpy(fbuf, allskys, naxis[0] * ns_all * sizeof(float));

        diags->orig_var = cpl_image_new(naxis[0], ns_all, CPL_TYPE_FLOAT);
        fbuf = cpl_image_get_data_float(diags->orig_var);
        if(fbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "original sky variance");
        }

        memcpy(fbuf, allskyvars, naxis[0] * ns_all * sizeof(float));

        /* Sky information table */
        diags->skyinfo = cpl_table_new(ns_all);
        cpl_table_new_column(diags->skyinfo, "specnum", CPL_TYPE_INT);
        cpl_table_new_column(diags->skyinfo, "skymed", CPL_TYPE_FLOAT);
        cpl_table_new_column(diags->skyinfo, "skysig", CPL_TYPE_FLOAT);
        cpl_table_new_column(diags->skyinfo, "badsky", CPL_TYPE_INT);

        cpl_table_set_column_unit(diags->skyinfo, "skymed", "ADU");
        cpl_table_set_column_unit(diags->skyinfo, "skysig", "ADU");

        i = 0;
        for (j = 0; j < ns_all; j++) {
            cpl_table_set_int(diags->skyinfo, "specnum", j, skyfibs[j]);

            /* Median and sigma for sky fibres only */
            if(ISSKY(targuse[skyfibs[j]-1])) {
                cpl_table_set(diags->skyinfo, "skymed", j, meds[i]);
                cpl_table_set(diags->skyinfo, "skysig", j, sigs[i]);
                i++;
            }

            cpl_table_set_int(diags->skyinfo, "badsky", j, chuck[j]);
        }

        /* Mean sky spectrum and variance */
        diags->comb_img = cpl_image_new(naxis[0], 1, CPL_TYPE_FLOAT);
        fbuf = cpl_image_get_data_float(diags->comb_img);
        if(fbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "mean sky spectrum");
        }

        memcpy(fbuf, skyspec, naxis[0] * sizeof(float));

        diags->comb_var = cpl_image_new(naxis[0], 1, CPL_TYPE_FLOAT);
        fbuf = cpl_image_get_data_float(diags->comb_var);
        if(fbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "mean sky spectrum variance");
        }

        memcpy(fbuf, skyvar, naxis[0] * sizeof(float));

        /* Residuals of masked sky subtraction and variance */
        diags->subt_img = cpl_image_new(naxis[0], naxis[1], CPL_TYPE_FLOAT);
        fbuf = cpl_image_get_data_float(diags->subt_img);
        if(fbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "subtracted sky spectrum");
        }

        memcpy(fbuf, recon, naxis[0] * naxis[1] * sizeof(float));

        diags->subt_var = cpl_image_new(naxis[0], naxis[1], CPL_TYPE_FLOAT);
        fbuf = cpl_image_get_data_float(diags->subt_var);
        if(fbuf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "subtracted sky spectrum variance");
        }

        memcpy(fbuf, shifted_skyvar, naxis[0] * naxis[1] * sizeof(float));
    }

    /* Set DRS headers in output to say what we did */
    cpl_propertylist_update_bool(spec_hdr, "ESO DRS SKYSUB", 1);
    cpl_propertylist_set_comment(spec_hdr, "ESO DRS SKYSUB",
                                 "Spectra are sky subtracted");

    cpl_propertylist_update_bool(spec_hdr, "ESO DRS PCASUB", usepca);
    cpl_propertylist_set_comment(spec_hdr, "ESO DRS PCASUB",
                                 "PCA correction applied");

    cpl_propertylist_update_int(spec_hdr, "ESO DRS PCANUM", neigen_used);
    cpl_propertylist_set_comment(spec_hdr, "ESO DRS PCANUM",
                                 "Number of PCA eigenvectors used");

    cpl_propertylist_update_float(spec_hdr, "ESO DRS PCATOTAL", pcatotal);
    cpl_propertylist_set_comment(spec_hdr, "ESO DRS PCATOTAL",
                                 "Percentage sky residual variation "
                                 "accounted for");

    /* Set QC headers in output */
    cpl_propertylist_update_float(spec_hdr,
                                  "ESO QC SKY CONT MED",
                                  skycontflux);
    cpl_propertylist_set_comment(spec_hdr,
                                 "ESO QC SKY CONT MED",
                                 "[ADU] Median sky continuum level");

    cpl_propertylist_update_float(spec_hdr,
                                  "ESO QC SKY CONT RMS",
                                  1.48*madsky);
    cpl_propertylist_set_comment(spec_hdr,
                                 "ESO QC SKY CONT RMS",
                                 "[ADU] RMS variation in sky level");

    cpl_propertylist_update_int(spec_hdr,
                                "ESO QC SKY NUM",
                                ns_all);
    cpl_propertylist_set_comment(spec_hdr,
                                 "ESO QC SKY NUM",
                                 "Number of skies found");

    cpl_propertylist_update_int(spec_hdr,
                                "ESO QC SKY NUSED",
                                nsky_final);
    cpl_propertylist_set_comment(spec_hdr,
                                 "ESO QC SKY NUSED",
                                 "Number of skies used");

    /* Sky line residual statistics for QC */
    residbuf = cpl_malloc(ns_all * naxis[0] * sizeof(float));

    nresid = 0;

    for(jsky = 0; jsky < ns_all; jsky++) {
        j = skyfibs[jsky] - 1;

        outputspec = imgdata + j*naxis[0];
        outputvar = specvar + j*naxis[0];

        for(i = 0; i < naxis[0]; i++) {
            if(skymask[i] > 40.0 &&
               outputvar[i] > 0) {
                residbuf[nresid] = outputspec[i] / sqrt(outputvar[i]);
                nresid++;
            }
        }
    }

    if(nresid > 0) {
        prestate = cpl_errorstate_get();
        if(qmost_medmad(residbuf, NULL, nresid,
                        &residmed, &residsig) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);

            residmed = 0;
            residsig = 0;
        }
        else {
            residsig *= 1.48;
        }
    }
    else {
        residmed = 0;
        residsig = 0;
    }

    cpl_propertylist_update_float(spec_hdr,
                                  "ESO QC SKY RESID MED",
                                  residmed);
    cpl_propertylist_set_comment(spec_hdr,
                                 "ESO QC SKY RESID MED",
                                 "Median normalised sky line residual");

    cpl_propertylist_update_float(spec_hdr,
                                  "ESO QC SKY RESID RMS",
                                  residsig);
    cpl_propertylist_set_comment(spec_hdr,
                                 "ESO QC SKY RESID RMS",
                                 "RMS normalised sky line residual");

    cpl_free(residbuf);
    residbuf = NULL;

    cpl_free(targuse);
    targuse = NULL;

    cpl_free(skycont);
    skycont = NULL;

    cpl_free(skyline);
    skyline = NULL;

    cpl_free(skymask);
    skymask = NULL;

    cpl_free(eigenvectors2[0]);
    cpl_free(eigenvectors2);
    eigenvectors2 = NULL;

    cpl_free(eigenvalues);
    eigenvalues = NULL;
    cpl_free(eigenfrac);
    eigenfrac = NULL;
    cpl_free(ecumulate);
    ecumulate = NULL;

    cpl_free(allskys);
    allskys = NULL;
    cpl_free(allskyvars);
    allskyvars = NULL;

    cpl_free(meds);
    meds = NULL;
    cpl_free(sigs);
    sigs = NULL;
    cpl_free(chuck);
    chuck = NULL;
    cpl_free(skyfibs);
    skyfibs = NULL;

    cpl_free(recon);
    recon = NULL;

    cpl_free(shifted_skyvar);
    shifted_skyvar = NULL;

    cpl_free(skyspec);
    skyspec = NULL;
    cpl_free(skyvar);
    skyvar = NULL;
    cpl_free(wavein);
    wavein = NULL;
    cpl_free(vshifts);
    vshifts = NULL;
    cpl_free(skyscale);
    skyscale = NULL;
    cpl_free(skymad);
    skymad = NULL;
    cpl_free(objline);
    objline = NULL;
    cpl_free(objcont);
    objcont = NULL;
    cpl_free(contflux);
    contflux = NULL;
    cpl_free(chisqu);
    chisqu = NULL;

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief   Work out a scale factor by looking at emission features.
 *
 * Compare emission features in the mean sky and an input spectrum to
 * work out a scale factor.  Restrict the analysis to a wavelength
 * window if you want.
 *
 * @param   wave         (Given)    An array with the wavelengths of
 *                                  each pixel of the spectra.
 * @param   inspec       (Given)    The object spectrum.
 * @param   skyspec      (Given)    The master sky spectrum.
 * @param   skymask      (Given)    A mask array marking where sky
 *                                  lines are located.
 *                                  sky lines are.
 * @param   nx           (Given)    The number of pixels in the
 *                                  arrays.
 * @param   wmin         (Given)    The lower wavelength limit of the
 *                                  window used in the analysis.
 * @param   wmax         (Given)    The upper wavelength limit of the
 *                                  window used in the analysis.
 *
 * @return  float giving the calculated scale factor.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static float getscale (
    float *wave,
    float *inspec,
    float *skyspec,
    float *skymask,
    long nx,
    float wmin,
    float wmax)
{
    float gap,scalecurr,scalemin,sum[3],a,b,c,offset,bestsum,summin;
    int iter,niter=5,iflag,j;

    /* Do niter iterations. Scalecurr is the best guess of the scale 
       factor and gap is the current offset for the iteration */

    gap = 1.0;
    scalecurr = 1.0;
    scalemin = 1.0;
    bestsum = 1.0e20;
    for (iter = 1; iter <= niter; iter++) {
	gap /= 2.0;
	memset(sum,0,3*sizeof(float));
	iflag = 0;
	for (j = 0; j < nx; j++) {
	    if (wave[j] >= wmin && wave[j] <= wmax && skymask[j] > 0.0) {
		sum[0] += fabs(inspec[j] - (scalecurr - gap)*skyspec[j]);
		sum[1] += fabs(inspec[j] - scalecurr*skyspec[j]);
		sum[2] += fabs(inspec[j] - (scalecurr + gap)*skyspec[j]);
		iflag = 1;
	    }
	}
	if (! iflag) {
	    scalemin = 1.0;
	    break;
	}

	/* Work out a new scale factor by fitting a parabola to the 
           summed noise versus offset from the current scale. Find 
           the minimum point */

        a = sum[1];
        b = 0.5*(sum[2] - sum[0]);
        c = 0.5*(sum[2] + sum[0] - 2*a);
	offset = -0.5*b/c;
	offset = qmost_max(qmost_min(1.0,offset),-1.0);
	summin = a + b*offset + c*offset*offset;

        /* If the estimate of the noise is better for this iteration than
           it was for the previous one, then update the scale factor and
           iterate again. Otherwise, keep the old estimate and get out of
           here */
        
        if (summin < bestsum) {
            bestsum = summin;
            scalemin = scalecurr + offset*gap;
            scalecurr = scalemin;
        } else {
            break;
        }
    }

    /* Return the best estimate */
    
    return(scalemin);
}

/**@}*/
    
/*

$Log$

Revision 1.23  20230922  jmi
Changed skyvar calculation to a more robust empirical method.

Revision 1.22  20221118  jmi
Added a trap for the case where no skies remain after selection.  This
was causing a segfault due to double free.

Revision 1.21  20221114  jmi
Parallelised the loops doing skyfilt over all fibres.

Revision 1.20  20211103  mji
added allowance for sky <noise> offset diagvar in eigenvalues 

Revision 1.19  20210412  mji
Aligned with qmost version by removing skyline scaling and adding 5400A cut 

Revision 1.18  20190816  mji
Fixed bug in variance scaling for linear interp, removed doshift option
and explicitly set skyscale to 1.0 as bug in skyline scaling for calibrators

Revision 1.17  2019/04/05 12:56:01  jrl
Modified to rectify a bug that caused segmentation fault in some cases

Revision 1.16  2019/04/01 16:06:45  jrl
Modfied vshiftpca to bring it into line with Mike's latest version

Revision 1.15  2019/02/25 10:46:44  jrl
New memory allocation scheme,

Revision 1.14  2018/10/12 10:09:46  jrl
Added qmost_skysub_vshiftpca

Revision 1.13  2018/08/17 00:12:48  jim
Fixed small bug in sky variance in resampling

Revision 1.12  2017/10/05 09:10:41  jim
Write the sky scaling factor into the fibinfo table

Revision 1.11  2017/08/10 08:26:38  jim
fixed memory free'ing problem

Revision 1.10  2017/08/08 07:43:48  jim
Add code to filter out skies that might be affected by nearby stars. Add
skyinfo table to sky output file

Revision 1.9  2017/06/02 13:56:12  jim
Fixed bug in the normalisation of the sky variance

Revision 1.8  2017/05/23 08:56:38  jim
Modified to ensure that we don't try and do any sky subtraction on pixels that
have a zero variance (bad pixels)

Revision 1.7  2017/05/22 11:21:15  jim
fixed bug in error message reporting

Revision 1.6  2017/01/17 09:01:43  jim
Fixed memory bug. Also now copies over the fibre table

Revision 1.5  2016/10/26 11:20:40  jim
Added docs

Revision 1.4  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.3  2016/07/21 08:33:52  jim
offset is now optional. Also option to save sky in a separate file

Revision 1.2  2016/07/11 15:02:18  jim
Lots of changes to remove redundant stuff for GES days. Also modified
to return a FITS table with the sky scale and offset information

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry

*/
