/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_doarcs.h"
#include "qmost_traceinfo.h"
#include "qmost_waveinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_doarcs_test  Unit test of qmost_doarcs
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_doarcs_ref
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_doarcs_ref(void)
{
    cpl_image *in_spec = NULL;
    cpl_propertylist *in_hdr = NULL;
    cpl_image *ref_wave_surface = NULL;
    cpl_table *linelist_tbl = NULL;
    cpl_table *trace_tbl = NULL;
    cpl_table *full_trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_table *first_wave_tbl = NULL;
    cpl_table *second_wave_tbl = NULL;
    cpl_error_code code;

    int nx = 512;
    int ny = 2048;

    int nspec = 5;
    int nlines = 32;
    int nord = 5;

    double lam_min = 5000;
    double lam_max = 6000;

    double angppix = (lam_max - lam_min) / (ny - 1);

    double yfwhm = 2.0;
    double pkht = 1000;

    float *buf = NULL;

    int iline;
    int ix, iy, ispec, yl, yh;
    double lam, ycen, ypoly, yw, dy;

    double xtr;
    qmost_traceinfo tr;
    cpl_size zero;

    qmost_waveinfo *wv = NULL;
    double wvtst;

    /* Create a simple line list and simulate arc spectrum */
    linelist_tbl = cpl_table_new(nlines);
    cpl_table_new_column(linelist_tbl, "Wavelength", CPL_TYPE_DOUBLE);

    in_spec = cpl_image_new(ny, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(in_spec, -8, 8);

    buf = cpl_image_get_data_float(in_spec);

    yw = 0.5*CPL_MATH_FWHM_SIG*CPL_MATH_FWHM_SIG / (yfwhm*yfwhm);

    for(iline = 0; iline < nlines; iline++) {
        /* Wavelength of injected line */
        lam = lam_min + (iline+0.5) * ((lam_max - lam_min) / nlines);
        /* add a bit of variation so spacing isn't completely uniform */
        lam += (iline % 2 ? 5.0 : -5.0) * ((double) iline) / nlines;

        /* Spectral pixel of line centre */
        ycen = 1 + (lam - lam_min) / angppix;

        /* Inject line */
        for(ispec = 0; ispec < nspec; ispec++) {
            ypoly = 0;

            yl = ycen + ypoly - 3*yfwhm;
            if(yl < 0)
                yl = 0;
            if(yl >= ny)
                yl = ny-1;
            
            yh = ycen + ypoly + 3*yfwhm;
            if(yh < 0)
                yh = 0;
            if(yh >= ny)
                yh = ny-1;
            
            for(iy = yl; iy <= yh; iy++) {
                dy = (iy+1) - (ycen + ypoly);
                buf[ispec*ny+iy] += pkht * exp(-yw * dy*dy);
            }
        }

        /* Set in line list but line 9 missing */
        if(iline != 9) {
            cpl_table_set_double(linelist_tbl, "Wavelength", iline, lam);
        }
    }

    in_hdr = cpl_propertylist_new();

    /* Create ref surface */
    ref_wave_surface = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    buf = cpl_image_get_data_float(ref_wave_surface);

    for(iy = 0; iy < ny; iy++) {
        for(ix = 0; ix < nx; ix++) {
            buf[iy*nx+ix] = lam_min + iy * ((lam_max - lam_min) / ny);
        }
    }

    /* Trace tables, one with some bad fibres and the other all live */
    code = qmost_trcreate(&trace_tbl, 0, 2);
    code = qmost_trcreate(&full_trace_tbl, 0, 2);

    for(ispec = 0; ispec < nspec; ispec++) {
        xtr = (ispec+0.5) * nx / nspec;

        memset(&tr, 0, sizeof(tr));
        tr.specnum = ispec+1;
        tr.live = (ispec + 1) % 2;  /* alternate live and not */
        tr.yst = 1;
        tr.yfn = ny;
        tr.yref = 0.5 * ny;
        tr.trrms = 0;
        tr.nord = 0;
        tr.coefs = cpl_polynomial_new(1);
        zero = 0;
        cpl_polynomial_set_coeff(tr.coefs, &zero, xtr);
        tr.npos = 2;
        tr.xpos = cpl_malloc(2 * sizeof(double));
        tr.xpos[0] = xtr;
        tr.xpos[1] = xtr;
        tr.ypos = cpl_malloc(2 * sizeof(double));
        tr.ypos[0] = 1;
        tr.ypos[1] = ny;
        tr.fwhm = cpl_malloc(2 * sizeof(float));
        tr.fwhm[0] = 3;
        tr.fwhm[1] = 3;
        tr.peak = cpl_malloc(2 * sizeof(float));
        tr.peak[0] = 10000;
        tr.peak[1] = 10000;
        tr.contrast = cpl_malloc(2 * sizeof(float));
        tr.contrast[0] = 1.0;
        tr.contrast[1] = 1.0;
        qmost_trwrite1(trace_tbl, tr, ispec+1);

        /* create alt table for test where all fibres live */
        tr.live = 1;
        qmost_trwrite1(full_trace_tbl, tr, ispec+1);

        cpl_polynomial_delete(tr.coefs);
        cpl_free(tr.xpos);
        cpl_free(tr.ypos);
        cpl_free(tr.fwhm);
        cpl_free(tr.peak);
        cpl_free(tr.contrast);
    }

    trace_hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(trace_hdr, "MINYST", 1);
    cpl_propertylist_update_int(trace_hdr, "MAXYFN", ny);

    /* This should work */
    code = qmost_doarcs_ref(in_spec,
                            in_hdr,
                            ref_wave_surface,
                            NULL,
                            NULL,
                            linelist_tbl,
                            trace_tbl,
                            trace_hdr,
                            10.0,
                            3.0,
                            nord,
                            1.0,
                            5.0,
                            &first_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(first_wave_tbl);

    if(first_wave_tbl) {
        /* Make sure it's valid */
        code = qmost_wvchk(first_wave_tbl);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        /* Check length */
        cpl_test_eq(cpl_table_get_nrow(first_wave_tbl), nspec);

        for(ispec = 0; ispec < nspec; ispec++) {
            /* Read entry */
            wv = cpl_malloc(sizeof(qmost_waveinfo));

            code = qmost_wvread1(first_wave_tbl, ispec+1, wv);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            /* Check entry */
            cpl_test_eq(wv->live, (ispec + 1) % 2);

            if(wv->live) {
                cpl_test_eq(wv->nord, nord);

                cpl_test_abs(wv->medresid, 0, 1.0);
                cpl_test_abs(wv->fit_rms, 0, 1.0);

                cpl_test_abs(wv->wave1, lam_min, 1.0);
                cpl_test_abs(wv->waven, lam_max, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 1.0 - wv->xref, NULL);
                cpl_test_abs(wvtst, lam_min, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 0, NULL);
                cpl_test_abs(wvtst, lam_min + (wv->xref - 1) * angppix, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, ny - wv->xref, NULL);
                cpl_test_abs(wvtst, lam_max, 1.0);
            }

            qmost_wvclose(1, &wv);
        }
    }

    /* As should this, where we give what we got last time as a
       reference wavelength solution.  Increase the match distance
       this time so we false match the extra line and trigger those
       branches.  Grid search is turned off. */
    code = qmost_doarcs_ref(in_spec,
                            in_hdr,
                            ref_wave_surface,
                            first_wave_tbl,
                            trace_hdr,
                            linelist_tbl,
                            trace_tbl,
                            trace_hdr,
                            10.0,
                            3.0,
                            nord,
                            50.0,
                            0.0,
                            &second_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(second_wave_tbl);

    if(second_wave_tbl) {
        /* Make sure it's valid */
        code = qmost_wvchk(second_wave_tbl);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        /* Check length */
        cpl_test_eq(cpl_table_get_nrow(second_wave_tbl), nspec);

        for(ispec = 0; ispec < nspec; ispec++) {
            /* Read entry */
            wv = cpl_malloc(sizeof(qmost_waveinfo));

            code = qmost_wvread1(second_wave_tbl, ispec+1, wv);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            /* Check entry */
            cpl_test_eq(wv->live, (ispec + 1) % 2);

            if(wv->live) {
                cpl_test_eq(wv->nord, nord);

                cpl_test_abs(wv->medresid, 0, 1.0);
                cpl_test_abs(wv->fit_rms, 0, 1.0);

                cpl_test_abs(wv->wave1, lam_min, 1.0);
                cpl_test_abs(wv->waven, lam_max, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 1.0 - wv->xref, NULL);
                cpl_test_abs(wvtst, lam_min, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 0, NULL);
                cpl_test_abs(wvtst, lam_min + (wv->xref - 1) * angppix, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, ny - wv->xref, NULL);
                cpl_test_abs(wvtst, lam_max, 1.0);
            }

            qmost_wvclose(1, &wv);
        }
    }

    cpl_table_delete(second_wave_tbl);
    second_wave_tbl = NULL;

    /* Test all live with ref made with some not live, this should
     * still work using adjacent fibres as backup. */
    code = qmost_doarcs_ref(in_spec,
                            in_hdr,
                            ref_wave_surface,
                            first_wave_tbl,
                            trace_hdr,
                            linelist_tbl,
                            full_trace_tbl,
                            trace_hdr,
                            10.0,
                            3.0,
                            nord,
                            1.0,
                            5.0,
                            &second_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(second_wave_tbl);

    if(second_wave_tbl) {
        /* Make sure it's valid */
        code = qmost_wvchk(second_wave_tbl);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        /* Check length */
        cpl_test_eq(cpl_table_get_nrow(second_wave_tbl), nspec);

        for(ispec = 0; ispec < nspec; ispec++) {
            /* Read entry */
            wv = cpl_malloc(sizeof(qmost_waveinfo));

            code = qmost_wvread1(second_wave_tbl, ispec+1, wv);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            /* Check entry */
            cpl_test_eq(wv->live, 1);

            if(wv->live) {
                cpl_test_eq(wv->nord, nord);

                cpl_test_abs(wv->medresid, 0, 1.0);
                cpl_test_abs(wv->fit_rms, 0, 1.0);

                cpl_test_abs(wv->wave1, lam_min, 1.0);
                cpl_test_abs(wv->waven, lam_max, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 1.0 - wv->xref, NULL);
                cpl_test_abs(wvtst, lam_min, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 0, NULL);
                cpl_test_abs(wvtst, lam_min + (wv->xref - 1) * angppix, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, ny - wv->xref, NULL);
                cpl_test_abs(wvtst, lam_max, 1.0);
            }

            qmost_wvclose(1, &wv);
        }
    }

    cpl_table_delete(second_wave_tbl);
    second_wave_tbl = NULL;

    cpl_table_delete(first_wave_tbl);
    first_wave_tbl = NULL;

    cpl_image_delete(in_spec);
    cpl_propertylist_delete(in_hdr);
    cpl_image_delete(ref_wave_surface);
    cpl_table_delete(linelist_tbl);
    cpl_table_delete(trace_tbl);
    cpl_table_delete(full_trace_tbl);
    cpl_propertylist_delete(trace_hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_doarcs_ob
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_doarcs_ob(void)
{
    cpl_image *in_spec = NULL;
    cpl_propertylist *in_hdr = NULL;
    cpl_image *ref_wave_surface = NULL;
    cpl_table *linelist_tbl = NULL;
    cpl_table *trace_tbl = NULL;
    cpl_table *full_trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_table *ref_wave_tbl = NULL;
    cpl_table *ob_wave_tbl = NULL;
    cpl_error_code code;

    int nx = 512;
    int ny = 2048;

    int nspec = 5;
    int nlines = 32;
    int nord = 5;

    double lam_min = 5000;
    double lam_max = 6000;

    double angppix = (lam_max - lam_min) / (ny - 1);

    double yfwhm = 2.0;
    double pkht = 1000;

    float *buf = NULL;

    int iline;
    int ix, iy, ispec, yl, yh;
    double lam, ycen, ypoly, yw, dy;

    double xtr;
    qmost_traceinfo tr;
    cpl_size zero;

    qmost_waveinfo *wv = NULL;
    double wvtst;

    /* Create a simple line list and simulate arc spectrum */
    linelist_tbl = cpl_table_new(nlines);
    cpl_table_new_column(linelist_tbl, "Wavelength", CPL_TYPE_DOUBLE);

    in_spec = cpl_image_new(ny, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(in_spec, -8, 8);

    buf = cpl_image_get_data_float(in_spec);

    yw = 0.5*CPL_MATH_FWHM_SIG*CPL_MATH_FWHM_SIG / (yfwhm*yfwhm);

    for(iline = 0; iline < nlines; iline++) {
        /* Wavelength of injected line */
        lam = lam_min + (iline+0.5) * ((lam_max - lam_min) / nlines);
        /* add a bit of variation so spacing isn't completely uniform */
        lam += (iline % 2 ? 5.0 : -5.0) * ((double) iline) / nlines;

        /* Spectral pixel of line centre */
        ycen = 1 + (lam - lam_min) / angppix;

        /* Inject line */
        for(ispec = 0; ispec < nspec; ispec++) {
            ypoly = 0;

            yl = ycen + ypoly - 3*yfwhm;
            if(yl < 0)
                yl = 0;
            if(yl >= ny)
                yl = ny-1;
            
            yh = ycen + ypoly + 3*yfwhm;
            if(yh < 0)
                yh = 0;
            if(yh >= ny)
                yh = ny-1;
            
            for(iy = yl; iy <= yh; iy++) {
                dy = (iy+1) - (ycen + ypoly);
                buf[ispec*ny+iy] += pkht * exp(-yw * dy*dy);
            }
        }

        /* Set in line list but line 9 missing */
        if(iline != 9) {
            cpl_table_set_double(linelist_tbl, "Wavelength", iline, lam);
        }
    }

    in_hdr = cpl_propertylist_new();

    /* Create ref surface */
    ref_wave_surface = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    buf = cpl_image_get_data_float(ref_wave_surface);

    for(iy = 0; iy < ny; iy++) {
        for(ix = 0; ix < nx; ix++) {
            buf[iy*nx+ix] = lam_min + iy * ((lam_max - lam_min) / ny);
        }
    }

    /* Trace tables, one with some bad fibres and the other all live */
    code = qmost_trcreate(&trace_tbl, 0, 2);
    code = qmost_trcreate(&full_trace_tbl, 0, 2);

    for(ispec = 0; ispec < nspec; ispec++) {
        xtr = (ispec+0.5) * nx / nspec;

        memset(&tr, 0, sizeof(tr));
        tr.specnum = ispec+1;
        tr.live = (ispec + 1) % 2;  /* alternate live and not */
        tr.yst = 1;
        tr.yfn = ny;
        tr.yref = 0.5 * ny;
        tr.trrms = 0;
        tr.nord = 0;
        tr.coefs = cpl_polynomial_new(1);
        zero = 0;
        cpl_polynomial_set_coeff(tr.coefs, &zero, xtr);
        tr.npos = 2;
        tr.xpos = cpl_malloc(2 * sizeof(double));
        tr.xpos[0] = xtr;
        tr.xpos[1] = xtr;
        tr.ypos = cpl_malloc(2 * sizeof(double));
        tr.ypos[0] = 1;
        tr.ypos[1] = ny;
        tr.fwhm = cpl_malloc(2 * sizeof(float));
        tr.fwhm[0] = 3;
        tr.fwhm[1] = 3;
        tr.peak = cpl_malloc(2 * sizeof(float));
        tr.peak[0] = 10000;
        tr.peak[1] = 10000;
        tr.contrast = cpl_malloc(2 * sizeof(float));
        tr.contrast[0] = 1.0;
        tr.contrast[1] = 1.0;
        qmost_trwrite1(trace_tbl, tr, ispec+1);

        /* create alt table for test where all fibres live */
        tr.live = 1;
        qmost_trwrite1(full_trace_tbl, tr, ispec+1);

        cpl_polynomial_delete(tr.coefs);
        cpl_free(tr.xpos);
        cpl_free(tr.ypos);
        cpl_free(tr.fwhm);
        cpl_free(tr.peak);
        cpl_free(tr.contrast);
    }

    trace_hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(trace_hdr, "MINYST", 1);
    cpl_propertylist_update_int(trace_hdr, "MAXYFN", ny);

    /* Make reference wavelength solution */
    code = qmost_doarcs_ref(in_spec,
                            in_hdr,
                            ref_wave_surface,
                            NULL,
                            NULL,
                            linelist_tbl,
                            trace_tbl,
                            trace_hdr,
                            10.0,
                            3.0,
                            nord,
                            1.0,
                            5.0,
                            &ref_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(ref_wave_tbl);

    /* Now test OB-level */
    code = qmost_doarcs_ob(in_spec,
                           in_hdr,
                           ref_wave_tbl,
                           in_hdr,
                           linelist_tbl,
                           trace_tbl,
                           trace_hdr,
                           10.0,
                           3.0,
                           nord,
                           1.0,
                           5.0,
                           &ob_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(ob_wave_tbl);

    if(ob_wave_tbl) {
        /* Make sure it's valid */
        code = qmost_wvchk(ob_wave_tbl);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        /* Check length */
        cpl_test_eq(cpl_table_get_nrow(ob_wave_tbl), nspec);

        for(ispec = 0; ispec < nspec; ispec++) {
            /* Read entry */
            wv = cpl_malloc(sizeof(qmost_waveinfo));

            code = qmost_wvread1(ob_wave_tbl, ispec+1, wv);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            /* Check entry */
            cpl_test_eq(wv->live, (ispec + 1) % 2);

            if(wv->live) {
                cpl_test_eq(wv->nord, nord);

                cpl_test_abs(wv->medresid, 0, 1.0);
                cpl_test_abs(wv->fit_rms, 0, 1.0);

                cpl_test_abs(wv->wave1, lam_min, 1.0);
                cpl_test_abs(wv->waven, lam_max, 1.0);

                /* The coefficients compute a correction to the master
                   so should be near zero here. */
                wvtst = cpl_polynomial_eval_1d(wv->coefs, 1.0 - wv->xref, NULL);
                cpl_test_abs(wvtst, 0, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 0, NULL);
                cpl_test_abs(wvtst, 0, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, ny - wv->xref, NULL);
                cpl_test_abs(wvtst, 0, 1.0);
            }

            qmost_wvclose(1, &wv);
        }
    }

    cpl_table_delete(ob_wave_tbl);
    ob_wave_tbl = NULL;

    /* Test all live at OB and ref with some not live, this should
     * still work using adjacent fibres as backup. */
    code = qmost_doarcs_ob(in_spec,
                           in_hdr,
                           ref_wave_tbl,
                           in_hdr,
                           linelist_tbl,
                           full_trace_tbl,
                           trace_hdr,
                           10.0,
                           3.0,
                           nord,
                           1.0,
                           5.0,
                           &ob_wave_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(ob_wave_tbl);

    if(ob_wave_tbl) {
        /* Make sure it's valid */
        code = qmost_wvchk(ob_wave_tbl);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        /* Check length */
        cpl_test_eq(cpl_table_get_nrow(ob_wave_tbl), nspec);

        for(ispec = 0; ispec < nspec; ispec++) {
            /* Read entry */
            wv = cpl_malloc(sizeof(qmost_waveinfo));

            code = qmost_wvread1(ob_wave_tbl, ispec+1, wv);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            /* Check entry */
            cpl_test_eq(wv->live, 1);

            if(wv->live) {
                cpl_test_eq(wv->nord, nord);

                cpl_test_abs(wv->medresid, 0, 1.0);
                cpl_test_abs(wv->fit_rms, 0, 1.0);

                cpl_test_abs(wv->wave1, lam_min, 1.0);
                cpl_test_abs(wv->waven, lam_max, 1.0);

                /* The coefficients compute a correction to the master
                   so should be near zero here. */
                wvtst = cpl_polynomial_eval_1d(wv->coefs, 1.0 - wv->xref, NULL);
                cpl_test_abs(wvtst, 0, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, 0, NULL);
                cpl_test_abs(wvtst, 0, 1.0);

                wvtst = cpl_polynomial_eval_1d(wv->coefs, ny - wv->xref, NULL);
                cpl_test_abs(wvtst, 0, 1.0);
            }

            qmost_wvclose(1, &wv);
        }
    }

    cpl_table_delete(ob_wave_tbl);
    ob_wave_tbl = NULL;

    cpl_table_delete(ref_wave_tbl);
    ref_wave_tbl = NULL;

    cpl_image_delete(in_spec);
    cpl_propertylist_delete(in_hdr);
    cpl_image_delete(ref_wave_surface);
    cpl_table_delete(linelist_tbl);
    cpl_table_delete(trace_tbl);
    cpl_table_delete(full_trace_tbl);
    cpl_propertylist_delete(trace_hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fpmeasure
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fpmeasure(void)
{
    cpl_image *in_spec = NULL;
    cpl_propertylist *in_hdr = NULL;
    cpl_table *wave_tbl = NULL;
    cpl_propertylist *wave_hdr = NULL;
    cpl_table *linelist_tbl = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_error_code code;

    int ny = 2048;
    int nspec = 15;

    double lam_min = 6000;
    double lam_max = 6500;

    double angppix = (lam_max - lam_min) / (ny - 1);

    double pkht = 10000;
    double refl = 0.5;  /* sets finesse */
    double dnu = 375.0;  /* GHz */

    float *buf = NULL;

    qmost_waveinfo wv;
    cpl_size deg;

    int ispec, iy;
    double nfac, wave, nu, s;

    int iline, nlines;
    double mode;

    /* Simulate FPE spectrum.  For simplicity we skip convolution by
     * the instrumetal profile so the finesse needs to be fairly low
     * to stop the lines from becoming too sharp.  The fibres are
     * staggered a bit in wavelength so some of the lines at the end
     * don't match to ensure those branches are covered. */
    in_spec = cpl_image_new(ny, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(in_spec, -8, 8);

    buf = cpl_image_get_data_float(in_spec);
    if(buf != NULL) {
        for(ispec = 0; ispec < nspec; ispec++) {
            nfac = 1.0 - refl;
            nfac *= nfac;
            
            for(iy = 0; iy < ny; iy++) {
                /* Frequency in GHz */
                wave = lam_min + ispec + angppix * iy;
                nu = 10000 * QMOST_SPEEDOFLIGHT / wave;
                
                s = sin(CPL_MATH_PI * nu / dnu);
                
                /* FPE equation */
                buf[ispec*ny+iy] = pkht * nfac / (nfac + 4 * refl * s * s);
            }
        }
    }

    in_hdr = cpl_propertylist_new();

    /* Wavelength solution table */
    code = qmost_wvcreate(&wave_tbl, 1, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        memset(&wv, 0, sizeof(wv));

        wv.specnum = ispec+1;
        wv.live = ispec != 7 ? 1 : 0;  /* flag #7 */
        wv.nord = 1;
        wv.coefs = cpl_polynomial_new(1);

        /* Put in a small wavelength error between fibres so the rms
           is non-zero. */
        deg = 0;
        cpl_polynomial_set_coeff(wv.coefs, &deg,
                                 lam_min + ispec + 0.5*ny*angppix +
                                 1.0e-3 * (ispec-nspec/2));

        deg = 1;
        cpl_polynomial_set_coeff(wv.coefs, &deg,
                                 angppix);

        wv.xref = 0.5*ny + 1;
        wv.ngood = 2;
        wv.medresid = 0;
        wv.fit_rms = 0;
        wv.wave1 = lam_min;
        wv.waven = lam_max;
        wv.dwave = angppix;
        wv.nlines = 2;
        wv.xpos = cpl_calloc(2, sizeof(double));
        wv.fwhm = cpl_calloc(2, sizeof(float));
        wv.wave_calc = cpl_calloc(2, sizeof(double));
        wv.wave_true = cpl_calloc(2, sizeof(double));
        wv.fit_flag = cpl_calloc(2, sizeof(unsigned char));
        wv.wave_cor = cpl_calloc(2, sizeof(double));
        wv.peak = cpl_calloc(2, sizeof(float));
        wv.contrast = cpl_calloc(2, sizeof(float));

        qmost_wvwrite1(wave_tbl, ispec+1, wv);

        cpl_polynomial_delete(wv.coefs);
        cpl_free(wv.xpos);
        cpl_free(wv.fwhm);
        cpl_free(wv.wave_calc);
        cpl_free(wv.wave_true);
        cpl_free(wv.fit_flag);
        cpl_free(wv.wave_cor);
        cpl_free(wv.peak);
        cpl_free(wv.contrast);
    }

    /* Add two blank fibres, 7 flagged by the waveinfo, 8 not so will
     * be analysed but has nothing in it */
    if(buf != NULL) {
        memset(buf + 7 * ny, 0, ny*sizeof(float));
        memset(buf + 8 * ny, 0, ny*sizeof(float));
    }

    wave_hdr = cpl_propertylist_new();

    /* This should work */
    qclist = cpl_propertylist_new();

    code = qmost_fpmeasure(in_spec, in_hdr,
                           wave_tbl, wave_hdr,
                           5.0, 1.0, &linelist_tbl, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check line list */
    nlines = cpl_table_get_nrow(linelist_tbl);
    cpl_test(nlines > 0);

    for(iline = 0; iline < nlines; iline++) {
        wave = cpl_table_get(linelist_tbl, "Wavelength", iline, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        /* Frequency */
        nu = 10000 * QMOST_SPEEDOFLIGHT / wave;

        /* FP mode */
        mode = nu / dnu;

        /* Check it's close to an integer, and therefore corresponds
         * to a valid FP mode. */
        cpl_test(fabs(remainder(mode, 1.0)) < 0.01);
    }

    cpl_table_delete(linelist_tbl);
    linelist_tbl = NULL;

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_image_delete(in_spec);
    in_spec = NULL;

    cpl_propertylist_delete(in_hdr);
    in_hdr = NULL;

    cpl_table_delete(wave_tbl);
    wave_tbl = NULL;

    cpl_propertylist_delete(wave_hdr);
    wave_hdr = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_fpcorr
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_fpcorr(void)
{
    cpl_propertylist *in_hdr = NULL;
    cpl_propertylist *linelist_hdr = NULL;
    float prescoef = 0.5, prescorr;
    cpl_error_code code;

    /* Empty headers */
    in_hdr = cpl_propertylist_new();
    linelist_hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_fpcorr(NULL, linelist_hdr, prescoef, &prescorr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fpcorr(in_hdr, NULL, prescoef, &prescorr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_fpcorr(in_hdr, linelist_hdr, prescoef, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* With empty headers, result should be zero */
    prescorr = 42.0;

    code = qmost_fpcorr(in_hdr, linelist_hdr, prescoef, &prescorr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(prescorr, 0);

    /* Add keyword, but wrong type */
    cpl_propertylist_update_string(in_hdr, "ESO INS PRES16 VAL", "invalid");
    cpl_propertylist_update_string(linelist_hdr, "ESO INS PRES16 VAL", "invalid");

    code = qmost_fpcorr(in_hdr, linelist_hdr, prescoef, &prescorr);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* Correct one of them */
    cpl_propertylist_delete(in_hdr);
    in_hdr = NULL;

    in_hdr = cpl_propertylist_new();
    cpl_propertylist_update_float(in_hdr, "ESO INS PRES16 VAL", 4.0);

    code = qmost_fpcorr(in_hdr, linelist_hdr, prescoef, &prescorr);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    /* Correct the other, should now work (int -> float should happen
     * automatically and not raise a type error). */
    cpl_propertylist_delete(linelist_hdr);
    linelist_hdr = NULL;

    linelist_hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(linelist_hdr, "ESO INS PRES16 VAL", 1);

    /* The answer should be prescoef * delta p */
    prescorr = 42.0;

    code = qmost_fpcorr(in_hdr, linelist_hdr, prescoef, &prescorr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(prescorr, 3*prescoef, FLT_EPSILON);

    cpl_propertylist_delete(in_hdr);
    in_hdr = NULL;

    cpl_propertylist_delete(linelist_hdr);
    linelist_hdr = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_doarcs module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_doarcs_ref();
    test_qmost_doarcs_ob();
    test_qmost_fpmeasure();
    test_qmost_fpcorr();

    return cpl_test_end(0);
}

/**@}*/
