/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_constants.h"
#include "qmost_extract_psf.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_extract_psf  qmost_extract_psf
 * 
 * PSF extraction of fibre spectra.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_extract_psf.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Number of symmetric bands to consider in covariance matrix.
 * 1 = diagonal, 2 = tridiagonal, etc. */
#undef NBAND
#define NBAND 2

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/* Structure used to store the PSF and related parameters for a fibre. */

typedef struct {
    int ixleft;
    int ixright;
    int ndata;
    int nmax;
    int live;
    float *data;
    float *sdata;
} profinfo;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void evalfit(
    profinfo *ps,
    double *amps,
    int ntrace,
    float *pfit,
    float *pvar,
    int nx);

static int profopen(
    profinfo *p,
    float crval1,
    float tpos,
    int live, 
    float cd1_1,
    float crpix1,
    int spatbin,
    int nx,
    float *ppos,
    float *pspos,
    int np);

static void dogs(
    double *a,
    double *b,
    double *sol,
    double *sol_old,
    int n,
    int restr);

/*----------------------------------------------------------------------------*/
/**
 * @brief   PSF weighted spectral extraction.
 *
 * Spectra in a 2D image are extracted by constructing the full
 * spatial profiles at each spectral pixel, and using the profile
 * values and signal to noise ratio (based on the supplied variance
 * image) as weights when summing the spatial pixels.  Provided the
 * trace information and profiles are correct, this produces an
 * "optimal" maximum likelihood solution to the problem of determining
 * the 1D spectrum.  This implementation extracts all of the fibres
 * simultaneously, allowing crosstalk to be corrected by computing the
 * overlap integrals between the spatial profiles and solving the
 * corresponding matrix problem to determine the flux in all of the
 * fibres, accounting for the overlap and unequal count levels in each
 * fibre.  Individual outlying pixels not well-fit by the profile, for
 * example due to cosmic ray hits, are iteratively flagged and
 * rejected from the profile fit.
 *
 * @param   in_img             (Given)    The input image, including
 *                                        bad pixel mask.  The data
 *                                        type must be
 *                                        CPL_TYPE_FLOAT.
 * @param   in_var             (Given)    The input variance image.
 *                                        The data type must be
 *                                        CPL_TYPE_FLOAT.
 * @param   in_crmask          (Given)    A cpl_mask flagging the
 *                                        locations of detected cosmic
 *                                        ray hits, or NULL if none.
 * @param   in_hdr             (Given)    The FITS header of the input
 *                                        image.
 * @param   prof_img           (Given)    An image list containing the
 *                                        fibre PSFs at each spectral
 *                                        pixel.  The structure of
 *                                        this argument is described
 *                                        in a note for the routine
 *                                        that produces it, see
 *                                        qmost_model_psf_full().
 * @param   prof_var           (Given)    The corresponding variances
 *                                        of the fibre PSFs.
 * @param   prof_hdr           (Given)    The FITS header of the
 *                                        profile file with WCS.
 * @param   trace_tbl          (Given)    The relevant trace table for
 *                                        the 2d spectra.
 * @param   trace_hdr          (Given)    The FITS header of the trace
 *                                        table.
 * @param   doover             (Given)    If true, correct for overlap
 *                                        of fibre profiles (crosstalk
 *                                        correction).  If false, this
 *                                        correction is disabled and
 *                                        standard PSF weighted
 *                                        extraction is used (diagonal
 *                                        overlap matrix).
 * @param   niter              (Given)    The number of rejection
 *                                        iterations used in the fit.
 * @param   crrej_thr          (Given)    The rejection threshold for
 *                                        the detection of cosmic ray
 *                                        events in the spectra. 
 * @param   out_spec_img       (Returned) The output image containing
 *                                        the extracted spectra.  NULL
 *                                        is returned if there are no
 *                                        fibres in the trace file.
 * @param   out_spec_var       (Returned) The corresponding variance
 *                                        array for the extracted
 *                                        spectra.  NULL is returned
 *                                        if there are no fibres in
 *                                        the trace file.
 * @param   out_spec_hdr       (Modified) A caller allocated property
 *                                        list to receive output DRS
 *                                        FITS headers.  Some of these
 *                                        are needed by subsequent
 *                                        processing stages if binning
 *                                        was used.
 * @param   out_rej            (Returned) A rejection map showing
 *                                        which pixels were flagged as
 *                                        bad or cosmics.  This is an
 *                                        image the same shape as the
 *                                        input image, of data type
 *                                        CPL_TYPE_INT, containing the
 *                                        QMOST_WMASK_* flag values
 *                                        defined in
 *                                        qmost_constants.h.  NULL is
 *                                        returned if there are no
 *                                        fibres in the trace file.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If if one of the required
 *                                        input FITS header keywords
 *                                        was not found.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the dimensions of the
 *                                        inputs don't match.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the input image data type
 *                                        was not float, or if an
 *                                        input FITS header keyword
 *                                        value had an incorrect data
 *                                        type.
 *
 * @note    The output spectra each occupy a single row in a 2d
 *          image. The input spectra are assumed to be oriented so
 *          that the wavelength direction runs along the Y axis.
 *
 * @par Input FITS Header Information:
 *   - <b>CD1_1</b>
 *   - <b>CRPIX1</b>
 *   - <b>CRVAL1</b>
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS WVSCLFAC</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *   - <b>WVSCLFAC</b>
 *
 * @par Output DRS Headers:
 *   - <b>MAXYFN</b>: The last physical detector spectral pixel in the
 *     extracted spectra.
 *   - <b>MINYST</b>: The first physical detector spectral pixel in
 *     the extracted spectra.
 *   - <b>WVCRV</b>: The physical (unbinned) spectral pixel coordinate
 *     of the middle of the first (binned) pixel in the extracted
 *     spectrum.  Needed for wavelength calibration of binned data.
 *   - <b>WVNOFF</b>: Physical (unbinned) spectral pixel offset of
 *     (binned) trace start position.  Needed for wavelength
 *     calibration of binned data.
 *
 * @par Output QC Parameters:
 *   - <b>EXT GOF MED</b>: The median goodness of fit of the profile
 *     in PSF extraction.
 *   - <b>EXT NUM REJECTED</b>: The number of rejected pixels during
 *     PSF extraction.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_extract_psf(
    cpl_image *in_img,
    cpl_image *in_var,
    cpl_propertylist *in_hdr,
    cpl_mask *in_crmask,
    cpl_imagelist *prof_img,
    cpl_imagelist *prof_var,
    cpl_propertylist *prof_hdr,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    int doover,
    int niter,
    float crrej_thr,
    cpl_image **out_spec_img,
    cpl_image **out_spec_var,
    cpl_propertylist *out_spec_hdr,
    cpl_image **out_rej)
{
    int npts,ispec,i,j,k,atleast1,npl,jx,jy,ind2;
    int ip,ix1,ix2,ip2,iter,nrej,nswap,ind,iy1,iy2,changed;
    int specbin,spatbin,isbinned,jp,jp2,noff,nchisq;
    long naxis_p[3],naxis_pv[3],nps,miny,maxy;
    long naxis[2],naxis_v[2],naxis_cr[2],naxis_tmp[2];
    float crval1,cd1_1;
    float sum,*ydata,*yvar;
    float scfac,diff,v_diff,thr,xpos,tcrv,chisq;
    float crpix1,pval;
    double sumyp,sumvp2;
    unsigned char *crline;
    int npmax, ib, iblim;
    cpl_type ptype;
    cpl_errorstate prestate;
    int looperror;
    float medgof;

    /* These pointers are extracted from our input or output CPL objects */
    float *imgdata = NULL;
    float *imgvar = NULL;
    cpl_mask *in_bpm = NULL;
    cpl_binary *bpmdata = NULL;
    cpl_binary *crmaskdata = NULL;
    const cpl_image *this_prof_img = NULL;
    const cpl_image *this_prof_var = NULL;
    const float *pin = NULL;
    const float *psin = NULL;
    float *odata = NULL;
    float *odatav = NULL;
    cpl_binary *obpm = NULL;
    cpl_binary *obpmv = NULL;
    int *orej = NULL;

    /* Thread-private allocations, garbage collected in parallel section */
    double *b;
    double *amps;
    double *amps_old;
    float *vars;
    double *covar;
    float *pfit;
    float *pvar;
    float *pdata;
    float *psdata;
    profinfo *ps;
    float *psbuf;

    /* Garbage collected in main routine */
    int ntrace = 0;
    qmost_traceinfo *tr = NULL;

    unsigned char *rejdata = NULL;
    float *linegof = NULL;
    unsigned char *linebpm = NULL;

    /* Check for NULL arguments */
    cpl_ensure_code(in_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_var != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_var != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(prof_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_var != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_rej != NULL, CPL_ERROR_NULL_INPUT);

    /* Now initialize outputs to NULL for garbage collection */
    *out_spec_img = NULL;
    *out_spec_var = NULL;
    *out_rej = NULL;

    /* This intentionally doesn't garbage collect variables used
     * inside the parallel section. */
#undef TIDY
#define TIDY                                    \
    if(tr != NULL) {                            \
        qmost_trclose(ntrace, &tr);             \
        ntrace = 0;                             \
        tr = NULL;                              \
    }                                           \
    if(rejdata != NULL) {                       \
        cpl_free(rejdata);                      \
        rejdata = NULL;                         \
    }                                           \
    if(*out_spec_img != NULL) {                 \
        cpl_image_delete(*out_spec_img);        \
        *out_spec_img = NULL;                   \
    }                                           \
    if(*out_spec_var != NULL) {                 \
        cpl_image_delete(*out_spec_var);        \
        *out_spec_var = NULL;                   \
    }                                           \
    if(*out_rej != NULL) {                      \
        cpl_image_delete(*out_rej);             \
        *out_rej = NULL;                        \
    }                                           \
    if(linegof != NULL) {                       \
        cpl_free(linegof);                      \
        linegof = NULL;                         \
    }                                           \
    if(linebpm != NULL) {                       \
        cpl_free(linebpm);                      \
        linebpm = NULL;                         \
    }

    /* Get image */
    naxis[0] = cpl_image_get_size_x(in_img);
    naxis[1] = cpl_image_get_size_y(in_img);

    npts = naxis[0]*naxis[1];

    /* Get pointer to image data */
    imgdata = cpl_image_get_data_float(in_img);
    if(imgdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    /* Check error is the correct size and get pointer */
    naxis_v[0] = cpl_image_get_size_x(in_var);
    naxis_v[1] = cpl_image_get_size_y(in_var);

    if(naxis_v[0] != naxis[0] ||
       naxis_v[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input image and variance dimensions "
                                     "don't match: (%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxis_v[0], naxis_v[1]);
    }

    imgvar = cpl_image_get_data_float(in_var);
    if(imgvar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input variance image");
    }

    /* Get bad pixel mask */
    in_bpm = cpl_image_get_bpm(in_img);
    bpmdata = cpl_mask_get_data(in_bpm);
    if(bpmdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "input BPM");
    }

    /* If there is a cosmic ray mask available, then open it now. */
    if(in_crmask != NULL) {
        naxis_cr[0] = cpl_mask_get_size_x(in_crmask);
        naxis_cr[1] = cpl_mask_get_size_y(in_crmask);
        
        if(naxis_cr[0] != naxis[0] ||
           naxis_cr[1] != naxis[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "input image and cosmic ray mask "
                                         "dimensions don't match: "
                                         "(%ld, %ld) != (%ld, %ld)",
                                         naxis[0], naxis[1],
                                         naxis_cr[0], naxis_cr[1]);
        }
        
        crmaskdata = cpl_mask_get_data(in_crmask);
        if(crmaskdata == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get pointer to "
                                         "input cosmic ray mask");
        }
    }
    
    /* Open the trace table */
    if(qmost_tropen(trace_tbl, trace_hdr,
                    &ntrace, &tr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to load trace table");
    }  
    
    if(ntrace < 1) {
        TIDY;
        return CPL_ERROR_NONE;
    }

    /* Check all planes in the profile imagelists have the same size
     * and the correct type.  We need to do this here before we enter
     * the parallel section to ensure the fetches in the parallel
     * section won't fail. */
    naxis_p[2] = cpl_imagelist_get_size(prof_img);
    naxis_pv[2] = cpl_imagelist_get_size(prof_var);

    if(naxis_pv[2] != naxis_p[2]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "profile image and variance numbers "
                                     "of planes don't match: %ld != %ld",
                                     naxis_p[2], naxis_pv[2]);
    }

    this_prof_img = cpl_imagelist_get_const(prof_img, 0);
    naxis_p[0] = cpl_image_get_size_x(this_prof_img);
    naxis_p[1] = cpl_image_get_size_y(this_prof_img);

    ptype = cpl_image_get_type(this_prof_img);
    if(ptype != CPL_TYPE_FLOAT) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "profile image %d type "
                                     "is not float",
                                     1);
    }

    this_prof_var = cpl_imagelist_get_const(prof_var, 0);
    naxis_tmp[0] = cpl_image_get_size_x(this_prof_var);
    naxis_tmp[1] = cpl_image_get_size_y(this_prof_var);

    if(naxis_tmp[0] != naxis_p[0] ||
       naxis_tmp[1] != naxis_p[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "first profile image and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis_p[0], naxis_p[1],
                                     naxis_tmp[0], naxis_tmp[1]);
    }

    ptype = cpl_image_get_type(this_prof_var);
    if(ptype != CPL_TYPE_FLOAT) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "profile variance %d type "
                                     "is not float",
                                     1);
    }

    for(ip = 1; ip < naxis_p[2]; ip++) {
        this_prof_img = cpl_imagelist_get_const(prof_img, ip);
        naxis_tmp[0] = cpl_image_get_size_x(this_prof_img);
        naxis_tmp[1] = cpl_image_get_size_y(this_prof_img);
    
        if(naxis_tmp[0] != naxis_p[0] ||
           naxis_tmp[1] != naxis_p[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "profile image %d dimensions "
                                         "are inconsistent: "
                                         "(%ld, %ld) != (%ld, %ld)",
                                         ip+1,
                                         naxis_p[0], naxis_p[1],
                                         naxis_tmp[0], naxis_tmp[1]);
        }

        ptype = cpl_image_get_type(this_prof_img);
        if(ptype != CPL_TYPE_FLOAT) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "profile image %d type "
                                         "is not float",
                                         ip+1);
        }

        this_prof_var = cpl_imagelist_get_const(prof_var, ip);
        naxis_tmp[0] = cpl_image_get_size_x(this_prof_var);
        naxis_tmp[1] = cpl_image_get_size_y(this_prof_var);

        if(naxis_tmp[0] != naxis_p[0] ||
           naxis_tmp[1] != naxis_p[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "profile variance %d dimensions "
                                         "are inconsistent: "
                                         "(%ld, %ld) != (%ld, %ld)",
                                         ip+1,
                                         naxis_p[0], naxis_p[1],
                                         naxis_tmp[0], naxis_tmp[1]);
        }

        ptype = cpl_image_get_type(this_prof_var);
        if(ptype != CPL_TYPE_FLOAT) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "profile variance %d type "
                                         "is not float",
                                         ip+1);
        }
    }

    /* Check the number of fibres */
    if(naxis_p[1] != ntrace) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "profile has %ld fibres but "
                                     "trace has %d fibres",
                                     naxis_p[1], ntrace);

    }

    /* And the number of planes, which should equal the number of
     * spectral pixels accounting for binning */
    if(naxis_p[2] != naxis[1]*specbin) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "profile has %ld spectral pixels but "
                                     "image has %ld",
                                     naxis_p[1], naxis[1]*specbin);
    }

    /* Get WCS information from profile header */
    if(qmost_cpl_propertylist_get_float(prof_hdr,
                                        "CRVAL1",
                                        &crval1) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read required keyword "
                                     "CRVAL1 from profile FITS header");
    }

    if(qmost_cpl_propertylist_get_float(prof_hdr,
                                        "CRPIX1",
                                        &crpix1) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read required keyword "
                                     "CRPIX1 from profile FITS header");
    }

    if(qmost_cpl_propertylist_get_float(prof_hdr,
                                        "CD1_1",
                                        &cd1_1) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read required keyword "
                                     "CD1_1 from profile FITS header");
    }

    if(cpl_propertylist_has(prof_hdr, "ESO DRS WVSCLFAC")) {
        if(qmost_cpl_propertylist_get_float(prof_hdr,
                                            "ESO DRS WVSCLFAC",
                                            &scfac) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read required keyword "
                                         "ESO DRS WVSCLFAC from profile "
                                         "FITS header");
        }
    }
    else if(cpl_propertylist_has(prof_hdr, "WVSCLFAC")) {
        if(qmost_cpl_propertylist_get_float(prof_hdr,
                                            "WVSCLFAC",
                                            &scfac) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read required keyword "
                                         "WVSCLFAC from profile FITS header");
        }
    }
    else {
        scfac = 1.0;
    }

    /* Create an array for the rejected pixels.  This is a bit
       different than it would be in 4L1 because our inputs are binary
       cpl_mask objects rather than the flag values used there, so we
       recreate the appropriate flag values. */
    rejdata = cpl_calloc(naxis[0]*naxis[1],sizeof(unsigned char));

    for (i = 0; i < naxis[0]*naxis[1]; i++) {
        /* Bad pixels are flagged COLD (the distinction between HOT
           and COLD doesn't really seem important and the latter
           seems more likely on a CCD). */
        if (bpmdata[i])
            rejdata[i] = QMOST_WMASK_COLD;

        /* Cosmics are flagged CR1 to match what qmost_cosmic would
           have done. */
        if (crmaskdata != NULL && crmaskdata[i])
 	    rejdata[i] = QMOST_WMASK_CR1;

        /* qmost_cosmic would also flag saturated pixels using this
           threshold so we recreate it here. */
        if (imgdata[i] > 0.95 * QMOST_SATURATE)
            rejdata[i] = QMOST_WMASK_SAT;
    }

    /* What size will the output spectra be? */

    maxy = tr[0].maxyfn/(float)specbin;
    miny = tr[0].minyst/(float)specbin;
    nps = maxy - miny + 1;

    /* Where should we start? */

    ip2 = -1;
    tcrv = 1.0;
    noff = 0;
    for (ip = 1; ip <= naxis[1]; ip++) {
        for (ispec = 0; ispec < ntrace; ispec++) {
            if (specbin > 1) {
                jp = (ip - 1)*specbin + 1;
                jp2 = jp + specbin - 1;
                if (tr[ispec].live) {
                    if ((jp >= tr[ispec].yst || jp2 >= tr[ispec].yst) &&
                        jp <= tr[ispec].yfn) {
                        ip2 = ip;
                        tcrv = (float)(jp2 + jp)*0.5 - (float)tr[ispec].yst + 1;
                        noff = tr[ispec].yst - jp;
                        break;
                    }
                }
            } else {
                if (tr[ispec].live && ip >= tr[ispec].yst && ip <= tr[ispec].yfn) {
                    ip2 = ip;
                    break;
                }
            }
        }
        if (ip2 > -1)
            break;
    }

    /* Create outputs */
    *out_spec_img = cpl_image_new(nps, ntrace, CPL_TYPE_FLOAT);
    *out_spec_var = cpl_image_new(nps, ntrace, CPL_TYPE_FLOAT);
    if(*out_spec_img == NULL || *out_spec_var == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output spectrum "
                                     "array of size (%ld, %d)",
                                     nps, ntrace);
    }

    odata = cpl_image_get_data_float(*out_spec_img);
    odatav = cpl_image_get_data_float(*out_spec_var);
    obpm = cpl_mask_get_data(cpl_image_get_bpm(*out_spec_img));
    obpmv = cpl_mask_get_data(cpl_image_get_bpm(*out_spec_var));
    if(odata == NULL || odatav == NULL ||
       obpm == NULL || obpmv == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get pointer to "
                                     "output spectrum array");
    }

    /* Array for output line goodness of fit */
    linegof = cpl_malloc(nps * sizeof(float));
    linebpm = cpl_malloc(nps * sizeof(unsigned char));

    for(ip = 0; ip < nps; ip++) {
        linebpm[ip] = 1;
    }

    /* Maximum size of interpolated profile: number of actual pixels
     * plus one extra at each end to allow for interpolation. */
    npmax = ceil(naxis_p[0] * cd1_1 / spatbin);
    npmax += 2;

    /* Round up to multiple of 4 so the split pieces of the array are
     * aligned on 16-byte boundaries (e.g. SSE). */
    npmax = 4 * (npmax / 4 + (npmax % 4 ? 1 : 0));

    /* Loop for each spectral row of pixels */
    looperror = 0;

#pragma omp parallel default(none) private(covar,b,amps,amps_old,vars,pdata,psdata,i,atleast1,j,k,npl,ispec,ydata,yvar,ib,iblim,sum,sumyp,sumvp2,pfit,pvar,ix1,ix2,pval,ps,psbuf,crline,iter,nrej,nswap,diff,v_diff,chisq,nchisq,thr,jp,xpos,this_prof_img,this_prof_var,pin,psin) shared(ip2,naxis,naxis_p,specbin,spatbin,tr,ntrace,prof_img,prof_var,scfac,imgdata,imgvar,rejdata,crval1,crpix1,cd1_1,crrej_thr,doover,niter,nps,odata,odatav,obpm,obpmv,linegof,linebpm,npmax,looperror)
    {
        /* Get some memory for the fitting routines */

        b = cpl_malloc(ntrace*sizeof(double));
        amps = cpl_malloc(ntrace*sizeof(double));
        amps_old = cpl_malloc(ntrace*sizeof(double));
        vars = cpl_malloc(ntrace*sizeof(float));
        covar = cpl_malloc(ntrace*NBAND*sizeof(double *));
        pfit = cpl_malloc(naxis[0]*sizeof(float));
        pvar = cpl_malloc(naxis[0]*sizeof(float));

        /* Get some memory for the profiles */

        pdata = cpl_malloc(naxis_p[0]*ntrace*sizeof(float));
        psdata = cpl_malloc(naxis_p[0]*ntrace*sizeof(float));

        ps = cpl_calloc(ntrace,sizeof(profinfo));
        psbuf = cpl_malloc(2*npmax*ntrace * sizeof(float));

        for(i = 0; i < ntrace; i++) {
            ps[i].nmax = npmax;
            ps[i].data = psbuf + 2*i * npmax;
            ps[i].sdata = psbuf + (2*i + 1) * npmax;
        }

#pragma omp for
        for (ip = ip2; ip <= naxis[1]; ip++) {
            jp = (ip - 1)*specbin + 1;

            /* Initialize pointers we use to get stuff from CPL objects */
            this_prof_img = NULL;
            pin = NULL;
            this_prof_var = NULL;
            psin = NULL;

            /* If a particular fibre is dead or if we are currently outside
             * the spectral limits of the current fibre, we fill it with
             * zeros so they don't contribute to the final fit. */

            atleast1 = 0;
            for (ispec = 0; ispec < ntrace; ispec++) {
                if (tr[ispec].live &&
                    jp >= tr[ispec].yst && jp <= tr[ispec].yfn) {
                    atleast1 = 1;
                }
            }
        
            /* If there is at least one profile to fit we continue at
             * the current position... */
        
            if (! atleast1)
                continue;

            /* Clear arrays */

            memset(amps, 0, ntrace*sizeof(double));
            memset(vars, 0, ntrace*sizeof(float));
            memset(pdata, 0, naxis_p[0]*ntrace*sizeof(float));
            memset(psdata, 0, naxis_p[0]*ntrace*sizeof(float));

            /* Get the appropriate plane from the profile */
            this_prof_img = cpl_imagelist_get_const(prof_img, jp-1);
            pin = cpl_image_get_data_float_const(this_prof_img);

            this_prof_var = cpl_imagelist_get_const(prof_var, jp-1);
            psin = cpl_image_get_data_float_const(this_prof_var);

            npl = naxis_p[0]*naxis_p[1];

            for (k = 0; k < npl; k++) {
                pdata[k] = pin[k] / scfac;
                psdata[k] = psin[k] / scfac;
            }

            /* Ok loop for the pixels along the slit and get the
             * data. Zero the fitting arrays */

            ydata = imgdata + (ip-1)*naxis[0];
            yvar = imgvar + (ip-1)*naxis[0];
            crline = rejdata + (ip-1)*naxis[0];
            memset(covar,0,ntrace*NBAND*sizeof(double));

            /* Get the profile info for this spectral position */

            for (i = 0; i < ntrace; i++) {
                xpos = qmost_tracexpos(tr[i],(double)jp);
                if(profopen(ps+i,crval1,xpos,
                            tr[i].live,cd1_1,crpix1,spatbin,naxis[0],
                            pdata+i*naxis_p[0],
                            psdata+i*naxis_p[0],naxis_p[0]) < 0) {
#pragma omp atomic write
                    looperror = 1;
                }
            }

            /* Start the iteration loop */

            chisq = 0.0;
            nchisq = 0;

            for (iter = 0; iter < niter; iter++) {

                /* Do the summations */

                iblim = doover ? NBAND : 1;

                for(j = 0; j < ntrace; j++) {
                    b[j] = 0.0;
                    vars[j] = 0.0;

                    if(!ps[j].live) {
                        continue;
                    }

                    /* First the overlap integrals that go into the
                     * covariance matrix. The matrix is stored in
                     * symmetric band diagonal form, where the array
                     * is of size ntrace x NBAND, and the fibre index
                     * is the most rapidly varying. The rows of the
                     * array (least rapidly varying index) contain the
                     * diagonal in index zero, followed by the first,
                     * second, etc. off diagonal. Unused elements are
                     * set to zero above. */

                    for(ib = 0, i = j; ib < iblim && i < ntrace; ib++, i++) {
                        /* No overlap at all */
                        if(ps[i].ixleft > ps[j].ixright ||
                           ps[i].ixright < ps[j].ixleft || 
                           !ps[i].live) {
                            continue;
                        }

                        /* Some overlap */
                        ix1 = qmost_max(ps[i].ixleft, ps[j].ixleft);
                        ix2 = qmost_min(ps[i].ixright, ps[j].ixright);

                        sum = 0.0;
                        for(k = ix1; k <= ix2; k++) {
                            if(crline[k-1] == QMOST_WMASK_GOOD ||
                               crline[k-1] == QMOST_WMASK_SAT) {
                                sum += qmost_max(0.0,ps[i].data[k-ps[i].ixleft]) * qmost_max(0.0,ps[j].data[k-ps[j].ixleft]);
                            }
                        }

                        covar[j*NBAND+ib] = (double)sum;
                    }

                    /* Now the vector on the RHS and variance */

                    sumyp = 0.0;
                    sumvp2 = 0.0;
                    ix1 = ps[j].ixleft;
                    ix2 = ps[j].ixright;
                    for (k = ix1; k <= ix2; k++) {
                        pval = qmost_max(0.0,ps[j].data[k-ix1]);
                        if (crline[k-1] != QMOST_WMASK_GOOD &&
                            crline[k-1] != QMOST_WMASK_SAT) {
                            continue;
                        }
                        sumyp += (double)pval*(double)ydata[k-1];

                        /* Variance sum for maximum likelihood weights
                         * (profile * inverse variance). */
                        if(yvar[k-1] != 0.0) {
                            sumvp2 += (double)pval*(double)pval/yvar[k-1];
                        }
                    }

                    b[j] = sumyp;

                    /* Resulting variance for maximum likeliood.  The
                     * reason for using this (by default) is it's less
                     * prone to being affected by an ill determined
                     * PSF model.  The profile weighted variance can
                     * produce spurious results if the PSF is a bad fit,
                     * which has often been the case in 4MOST due to
                     * difficulty of determining it. */
                    if(sumvp2 != 0.0) {
                        vars[j] = 1.0 / sumvp2;
                    }
                }

                /* Do the Gauss-Seidel */

                dogs(covar,b,amps,amps_old,ntrace,NBAND-1);

                /* Evaluate model */

                evalfit(ps, amps, ntrace, pfit, pvar, naxis[0]);

                /* Terminate loop here on last iteration, the results of
                 * running the rejection aren't used and could make the
                 * rejection map inaccurate. */
                if(iter == niter - 1) {
                    break;
                }

                /* Look for pixels to delete...*/

                nrej = 0;
                nswap = 0;
                for (k = 0; k < naxis[0]; k++) {
                    if(pvar[k] > 0.0) {
                        v_diff = qmost_max(1.0,yvar[k]+pvar[k]);
                        thr = crrej_thr * sqrt(v_diff);
                        diff = ydata[k] - pfit[k];
                        if (diff >= thr) {
                            if (crline[k] == QMOST_WMASK_GOOD ||
                                crline[k] == QMOST_WMASK_SAT) {
                                /* Set bad */
                                crline[k] = QMOST_WMASK_DISC;
                                nrej++;
                            }
                        }
                        else if (diff < thr/2.0 && 
                                 (crline[k] == QMOST_WMASK_CR1 ||
                                  crline[k] == QMOST_WMASK_SAT ||
                                  crline[k] == QMOST_WMASK_HOT ||
                                  crline[k] == QMOST_WMASK_DISC)) {
                            /* Set good */
                            crline[k] = QMOST_WMASK_GOOD;
                            nswap++;
                        }
                    }
                }
                if (nrej == 0 && nswap == 0)
                    break;
            }

            /* Create the output arrays. Similar to extract_tram we flag
             * pixels where we didn't do any extraction in the BPM
             * (e.g. off ends, or fibre not live) so they can be excluded
             * from the statistics. */

            for (j = 0; j < ntrace; j++) {
                if (tr[j].live && jp >= tr[j].yst && jp <= tr[j].yfn) {
                    odata[j*nps+ip-ip2] = (float)amps[j];
                    odatav[j*nps+ip-ip2] = vars[j];
                    obpm[j*nps+ip-ip2] = 0;
                    obpmv[j*nps+ip-ip2] = 0;
                } else {
                    odata[j*nps+ip-ip2] = 0.0;
                    odatav[j*nps+ip-ip2] = 0.0;
                    obpm[j*nps+ip-ip2] = 1;
                    obpmv[j*nps+ip-ip2] = 1;
                }
            }

            /* Goodness of fit statistic */
            chisq = 0.0;
            nchisq = 0;

            for (k = 0; k < naxis[0]; k++) {
                if(pvar[k] > 0.0) {
                    v_diff = qmost_max(1.0,yvar[k]+pvar[k]);
                    thr = crrej_thr * sqrt(v_diff);
                    diff = ydata[k] - pfit[k];

                    /* Statistics of good pixels */
                    if(crline[k] == QMOST_WMASK_GOOD) {
                        chisq += diff*diff / v_diff;
                        nchisq++;
                    }
                }
            }

            if(chisq > 0 && nchisq > 0) {
                linegof[ip-ip2] = sqrt(chisq / nchisq);
                linebpm[ip-ip2] = 0;
            }
            else {
                linegof[ip-ip2] = 0.0;
                linebpm[ip-ip2] = 1;
            }
        }

        /* Tidy up */
        cpl_free(ps);
        ps = NULL;

        cpl_free(psbuf);
        psbuf = NULL;

        cpl_free(covar);
        covar = NULL;

        cpl_free(pdata);
        pdata = NULL;

        cpl_free(psdata);
        psdata = NULL;

        cpl_free(b);
        b = NULL;

        cpl_free(amps);
        amps = NULL;

        cpl_free(amps_old);
        amps_old = NULL;

        cpl_free(vars);
        vars = NULL;

        cpl_free(pfit);
        pfit = NULL;

        cpl_free(pvar);
        pvar = NULL;
    }

    if(looperror != 0) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                     "profile was truncated");
    }

    /* Look at rejected pixels. If any of the pixels are marked as WMASK_OTHER
       we swap those back to WMASK_CR1. If any of those marked WMASK_DISC touch
       one that is marked WMASK_CR1, then we assume it is part of a cosmic ray
       that wasn't detected the first time around and mark it WMASK_CR2 */

    nrej = 0;
    for (i = 0; i < npts; i++) {
        if (rejdata[i] == QMOST_WMASK_OTHER)
            rejdata[i] = QMOST_WMASK_CR1;
	if (rejdata[i] != 0)
            nrej++;
    }

    nswap = 0;
    for (j = 0; j < naxis[1]; j++) {
        for (i = 0; i < naxis[0]; i++) {
            ind = j*naxis[0] + i;
            if (rejdata[ind] == QMOST_WMASK_DISC) {
                ix1 = qmost_max(0,i-1);
                ix2 = qmost_min(naxis[0]-1,i+1);
                iy1 = qmost_max(0,j-1);
                iy2 = qmost_min(naxis[1]-1,j+1);
                changed = 0;
                for (jy = iy1; jy <= iy2; jy++) {
                    for (jx = ix1; jx <= ix2; jx++) {
                        ind2 = jy*naxis[0] + jx;
                        if (rejdata[ind2] == QMOST_WMASK_CR1) {
                            rejdata[ind] = QMOST_WMASK_CR2;
                            changed++;
                            nswap++;
                        }
                        if (changed)
                            break;
                    }
                    if (changed)
                        break;
                }
            }
        }
    }

    /* Write FITS headers */
    cpl_propertylist_update_int(out_spec_hdr,
                                "ESO QC EXT NUM REJECTED",
                                nrej);
    cpl_propertylist_set_comment(out_spec_hdr,
                                 "ESO QC EXT NUM REJECTED",
                                 "Number of rejected pixels");

    prestate = cpl_errorstate_get();

    if(qmost_med(linegof, linebpm, nps, &medgof) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
    }
    else {
        cpl_propertylist_update_float(out_spec_hdr,
                                      "ESO QC EXT GOF MED",
                                      medgof);
        cpl_propertylist_set_comment(out_spec_hdr,
                                     "ESO QC EXT GOF MED",
                                     "Median goodness of fit");
    }

    cpl_free(linegof);
    linegof = NULL;

    cpl_free(linebpm);
    linebpm = NULL;

    cpl_propertylist_update_int(out_spec_hdr,
                                "ESO DRS MINYST",
                                tr[0].minyst);
    cpl_propertylist_set_comment(out_spec_hdr,
                                 "ESO DRS MINYST",
                                 "[pix] Minimum yst");
    cpl_propertylist_update_int(out_spec_hdr,
                                "ESO DRS MAXYFN",
                                tr[0].maxyfn);
    cpl_propertylist_set_comment(out_spec_hdr,
                                 "ESO DRS MAXYFN",
                                 "[pix] Maximum yfn");

    if (specbin > 1) {
        cpl_propertylist_update_float(out_spec_hdr,
                                      "ESO DRS WVCRV",
                                      tcrv);
        cpl_propertylist_set_comment(out_spec_hdr,
                                     "ESO DRS WVCRV",
                                     "[pix] Extraction zeropoint offset");

        cpl_propertylist_update_int(out_spec_hdr,
                                    "ESO DRS WVNOFF",
                                    noff);
        cpl_propertylist_set_comment(out_spec_hdr,
                                     "ESO DRS WVNOFF",
                                     "[pix] Extraction blocking offset");
    }

    /* Write rejection map */
    *out_rej = cpl_image_new(naxis[0], naxis[1], CPL_TYPE_INT);
    if(*out_rej == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output rejection "
                                     "map of size (%ld, %d)",
                                     nps, ntrace);
    }

    orej = cpl_image_get_data_int(*out_rej);
    if(orej == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get pointer to "
                                     "output rejection map");
    }

    for (i = 0; i < npts; i++) {
        orej[i] = rejdata[i];
    }

    cpl_free(rejdata);
    rejdata = NULL;

    qmost_trclose(ntrace,&tr);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Evaluate fitted profile and profile variance.
 *
 * Computes the value of the profile and profile variance at a given
 * position along the spatial direction, summed over all fibres to
 * account for the overlapping profile wings of the fibres.
 *
 * @param   ps                 (Given)    The profinfo structures for
 *                                        each fibre at the given
 *                                        spectral position.
 * @param   amps               (Given)    The profile amplitudes (aka
 *                                        the extracted counts, or the
 *                                        coefficients multiplying the
 *                                        normalised profile) for each
 *                                        fibfre at the given spectral
 *                                        position.
 * @param   ntrace             (Given)    The number of fibres.
 * @param   pfit               (Modified) Array giving the resulting
 *                                        value of the summed profile
 *                                        fit at each x pixel.
 * @param   pvar               (Modified) Array giving the resulting
 *                                        value of the summed profile
 *                                        variance at each x pixel.
 * @param   nx                 (Given)    The size of the image in x.
 *
 * @return  void
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void evalfit(
    profinfo *ps,
    double *amps,
    int ntrace,
    float *pfit,
    float *pvar,
    int nx)
{
    int itrace;
    int xl, xh, x;
    float sigma;

    /* Clear output arrays */
    memset(pfit, 0, nx * sizeof(float));
    memset(pvar, 0, nx * sizeof(float));

    /* Sum profiles for each trace */
    for(itrace = 0; itrace < ntrace; itrace++) {
        if(!ps[itrace].live) {
            continue;
        }

        xl = ps[itrace].ixleft;
        xh = ps[itrace].ixright;

        for(x = xl; x <= xh; x++) {
            pfit[x-1] += amps[itrace] * qmost_max(0.0, ps[itrace].data[x-xl]);

            sigma = amps[itrace] * qmost_max(0.0, ps[itrace].sdata[x-xl]);
            pvar[x-1] += sigma * sigma;
        }
    }

    /* 2.5% lower bound clip on profile variance estimate */
    for(x = 0; x < nx; x++) {
        pvar[x] = qmost_max(pvar[x], 0.0025 * pfit[x]*pfit[x]);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Open a profinfo structure for a given fibre.
 *
 * A profinfo structure is opened for a given fibre and some default
 * information is written to it. This includes the interpolated
 * profile for a given spectral position.
 *
 * @param   p                  (Modified) The profinfo structure for a
 *                                        given fibre.
 * @param   crval1             (Given)    The true pixel position of
 *                                        the profile pixel reference
 *                                        point.
 * @param   tpos               (Given)    The trace position for the
 *                                        fibre at this spectral
 *                                        location.
 * @param   live               (Given)    True if this is a live
 *                                        fibre, false otherwise.
 * @param   cd1_1              (Given)    The increment in 'real'
 *                                        pixels of each pixel
 *                                        represented in the input
 *                                        profile array.
 * @param   crpix1             (Given)    The profile pixel reference
 *                                        point.
 * @param   spatbin            (Given)    The spatial binning factor
 *                                        used in the exposure.
 * @param   nx                 (Given)    The size of the image in x.
 * @param   ppos               (Given)    The full subsampled profile
 *                                        for the given fibre at this
 *                                        wavelength position.
 * @param   pspos              (Given)    The full subsampled profile
 *                                        variance for the given fibre
 *                                        at this wavelength position.
 * @param   np                 (Given)    The number of profile pixels.
 *
 * @return  int    0 if OK, -1 if buffer was not large enough.
 *
 * @note    The input profiles are sampled at the subpixel
 *          level. Hence for a given pixel x_p in the profile space
 *          (where 1 <= x_p <= np) the location in the real image of
 *          that pixel is given by: 
 *          x_real = crval1 + (x_p - crpix1)*cd1_1 + tpos
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static int profopen(
    profinfo *p,
    float crval1,
    float tpos,
    int live, 
    float cd1_1,
    float crpix1,
    int spatbin,
    int nx,
    float *ppos,
    float *pspos,
    int np)
{
    int rv = 0;
    float xleft,xright,sum,pval,perr,pvalsum,perrsum;
    int i,n;
    float x,p1,w1,w2;
    int ip1,ip2;

    /* Clear arrays */

    memset(p->data, 0, p->nmax * sizeof(float));
    memset(p->sdata, 0, p->nmax * sizeof(float));

    /* If this guy isn't live then we can just give it null values */

    p->live = live;
    if (! live) {
        p->ixleft = 0;
        p->ixright = 0;
        p->ndata = 0;
        return rv;
    }

    /* Set up the limits of the current profile */

    xleft = (crval1 + (1.0 - crpix1)*cd1_1 + tpos - 0.5) / spatbin + 0.5;
    xright = (crval1 + ((float)np - crpix1)*cd1_1 + tpos - 0.5) / spatbin + 0.5;
    p->ixleft = (int)(xleft + 0.4);
    p->ixright = (int)(xright + 0.6);

    if(p->ixleft < 1) {
        p->ixleft = 1;
    }
    else if(p->ixleft > nx) {
        p->ixleft = nx;
    }

    if(p->ixright < 1) {
        p->ixright = 1;
    }
    else if(p->ixright > nx) {
        p->ixright = nx;
    }

    p->ndata = (p->ixright - p->ixleft + 1);

    /* Truncate if above limit.  This should never happen if the limit
     * is set correctly. */

    if(p->ndata > p->nmax) {
        p->ndata = p->nmax;
        rv = -1;
    }

    /* Set up the data array. Make sure it's normalised */

    sum = 0.0;
    for (n = 0; n < p->ndata; n++) {
        pvalsum = 0.0;
        perrsum = 0.0;
        for (i = 0; i < spatbin; i++) {
            /* Input x position in unbinned image */
            x = (p->ixleft + n - 1) * spatbin + 1 + i;

            /* Work out which profile pixel matches the input x
             * position. */
            p1 = (x - crval1 - tpos) / cd1_1 + crpix1;

            /* Work out the contribution of adjacent pixels and do a
             * linear weighting. */
            ip1 = floor(p1);
            ip2 = ip1 + 1;

            if(ip1 < 0) {  /* off left */
                pval = 0.0;
                perr = 0.0;
            }
            else if(ip1 == 0) {  /* within the left-most subpixel */
                /* Workaround for the edge of the profile array.  In
                 * most cases there's another profile here for the
                 * next fibre, so we give half of the subpixel to each
                 * fibre. */
                pval = 0.5*qmost_max(0.0,ppos[0]);
                perr = 0.5*qmost_max(0.0,pspos[0]);
            }
            else if(ip1 == np) {  /* within the right-most subpixel */
                /* This is the corresponding workaround at the other
                 * side. */
                pval = 0.5*qmost_max(0.0,ppos[np-1]);
                perr = 0.5*qmost_max(0.0,pspos[np-1]);
            }
            else if(ip1 > np) {  /* off right */
                pval = 0.0;
                perr = 0.0;
            }
            else {  /* normal case within array, linear interpolation */
                w2 = (p1 - (float)ip1);
                w1 = 1.0 - w2;

                pval = qmost_max(0.0,w1*ppos[ip1-1] + w2*ppos[ip2-1]);
                perr = qmost_max(0.0,w1*pspos[ip1-1] + w2*pspos[ip2-1]);
            }

            pvalsum += pval;
            perrsum += perr;
        }
	p->data[n] = pvalsum;
	p->sdata[n] = perrsum;
        sum += p->data[n];
    }
    if (sum != 0.0) {
        for (n = 0; n < p->ndata; n++) {
            p->data[n] /= sum;
            p->sdata[n] /= sum;
        }
    }

    return rv;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do Gauss-Seidel approximation.
 *
 * Do the Gauss-Seidel approximation of the solution to the  system of
 * equations in Ax = b, where A is square. Works best for sparse,
 * nearly diagonal matrices.
 *
 * @param   a                  (Given)    The input matrix in
 *                                        symmetric band diagonal
 *                                        form, dimension n x NBAND.
 * @param   b                  (Given)    The input rhs vector of the
 *                                        equation.
 * @param   sol                (Modified) The solution vector x.
 * @param   sol_old            (Modified) Workspace array of length
 *                                        n.
 * @param   n                  (Given)    The length of a side of A.
 * @param   restr              (Given)    Value >= 0. If != 0, then
 *                                        this is the maximum number
 *                                        of off diagonal positions
 *                                        that will be considered in
 *                                        the solution. If A is nearly
 *                                        diagonal, this speeds up the
 *                                        solution enormously.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void dogs(
    double *a,
    double *b,
    double *sol,
    double *sol_old,
    int n,
    int restr)
{
    double delta,damp,change,cmax,ds;
    int iter,j,i,i1,i2,k;

    /* Supply default for restr if needed */

    if(restr == 0) {
        restr = NBAND;
    }

    /* Make a copy of the input vector */

    memcpy(sol,b,n*sizeof(double));

    /* Iteration loop */

    damp = 1.0;
    cmax = 1.0e10;
    for (iter = 0; iter <= 10; iter++) {
        change = 0.0;
        memcpy(sol_old,sol,n*sizeof(double));
        for (j = 0; j < n; j++) {
            delta = 0.0;
            i1 = qmost_max(0,j-restr);
            i2 = qmost_min(n,j+restr+1);

            /* In band diagonal form, the element Aji is in positions
             * (j-i,i) if i < j, and (i-j,j) if j > i, noting we only
             * calculated it for the upper triangle in the caller. */
            for (i = i1; i < j; i++) {
                k = j - i;
                delta += damp*a[i*NBAND+k]*sol[i];
            }
            for (i = j+1; i < i2; i++) {
                k = i - j;
                delta += damp*a[j*NBAND+k]*sol[i];
            }

            if (a[j*NBAND] != 0.0) {
                sol[j] = (b[j] - delta)/a[j*NBAND];
                ds = fabs(sol[j] - sol_old[j]);
                if (ds > cmax)
                    cmax = ds;
                change += ds;
            }
        }
        change /= (double)n;
        if (change < 1.0e-6 && cmax < 0.01)
            break;
    }
}

/**@}*/

/*

$Log$
Revision 1.28  20230920  mji
switched to alternative ML diag var estimator to counter spurious
results due to PSF problems with Jim method

Revision 1.27  20230306  mji
modified p->ixleft = (int)(xleft + 0.4) and p->ixright = (int)(xright + 0.6)
to give more accurate profile boundaries

Revision 1.26  20230206  mji
fixed sub-sample edge issue to deal with mismatch between data and
model profiles yielding slightly different overlap integrals as a
function of number of sub-samples used [aka wiggles problem]

Revision 1.25  20221218  mji
removed the iter>0 constraint on rejection for RHS vector
which improves CR performance [unsure why it was set]

Revision 1.24  20221109  mji
allowed previously flagged discrepant pixels to be reflagged as
good in subsequent iterations, changed reflagging threshold criterion,
trapped -ve going pfit in threshold test

Revision 1.23  20221014  mji
Added in pragma traps for thread conflicts during i/o to
attempt fix of occasional random extracted rows being all zero

Revision 1.22  20210420  mji
Added in better support for cr and other forms of rejection 
and updated rejection test to use variance information

Revision 1.21  2019/02/25 10:34:23  jrl
New memory allocation scheme. Added binning

Revision 1.20  2018/08/28 12:02:21  jrl
Modified to check for negative profile values

Revision 1.19  2018/07/15 14:33:01  jim
Fixed multithreading private variables

Revision 1.18  2018/06/27 14:33:47  jim
small mods to fix compiler warnings

Revision 1.17  2018/06/27 09:42:18  jim
Add cosmic ray masking and rejection

Revision 1.16  2018/02/20 12:45:15  jim
Added cosmic ray mask to argument list partially worked it into the algorithm.
Iteration still not done

Revision 1.15  2018/01/16 10:13:09  jim
superficial changes

Revision 1.14  2017/10/06 09:43:06  jim
Moved a few things around to help with multithreading

Revision 1.13  2017/10/05 09:08:04  jim
main fitting calculation now done as doubles

Revision 1.12  2017/08/08 07:44:23  jim
Added a few diagnostics

Revision 1.11  2017/05/22 11:18:11  jim
Fixed bug in error reporting. Also fixed a bug where random values are given
to parts of a spectrum where there should be no data.

Revision 1.10  2017/04/13 10:55:36  jim
Modified to use OpenMP

Revision 1.9  2017/03/23 12:09:10  jim
Modified to use new compressed version of the input profile cube

Revision 1.8  2017/03/14 11:01:45  jim
Now gets information on the ends of the trace from the trace structure

Revision 1.7  2017/01/17 08:58:10  jim
Modified to trap zero divides

Revision 1.6  2016/10/23 15:55:38  jim
Added docs

Revision 1.5  2016/10/03 14:52:39  jim
Fixed bug that caused NaNs in the extracted variance in fibres next
to gaps

Revision 1.4  2016/07/11 14:58:31  jim
Fixed bug which occurs if a fibre is missing

Revision 1.3  2016/07/06 11:03:10  jim
Modified the input file naming. Improved speed.

Revision 1.2  2016/05/20 09:58:57  jim
Uses new version of trace file

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry


*/
