/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_ffnorm_fib.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_ffnorm_fib  qmost_ffnorm_fib
 * 
 * Fibre flat field normalisation functions.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_ffnorm_fib.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void plugholes(float *data, unsigned char *bpm, int nx);
static int ff_meanvarcut(float *data, float *datav, int npts, float lcut,
                         float hcut, float *mean, float *meanv, float *sig);
static void ff_filt1d(float *data, float *var, float aval, long nx, int smooth);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Normalise fibre flat spectra and prepare them for use in
 *          flat fielding.
 *
 * This routine is used to prepare fibre flat spectra for use in flat
 * fielding the data.  The input spectra should be extracted and
 * rebinned to a common wavelength scale prior to calling the
 * routine.  In its most basic form with the three flags resptrack,
 * smooth and rescale all set to false, this routine normalises all of
 * the individual fibre flat spectra to an average of unity.
 *
 * Setting the resptrack flag to true determines an average spectral
 * response (the combination of the spectrum of the lamp and any
 * spectral response common to all fibres such as blaze) and removes
 * it from the spectra prior to normalisation.  For 4MOST, this is
 * needed to remove the strong spectral features in the laser driven
 * light source ("lamp"), particularly the broad Xenon emission lines
 * in the red arm.
 *
 * Setting the rescale flag to true normalises the fibres to an
 * ensemble average (over all of the fibres) of unity, preserving
 * their relative throughputs in the result.  When dividing by the
 * resulting flat to apply the correction, this causes the relative
 * throughput differences between the fibres to also be removed
 * (corrected).
 *
 * Setting the smooth parameter to a non-zero value applies an
 * additional clipped boxcar smoothing filter with spectral window
 * (in pixels) specified by the value of the smooth parameter, to
 * reduce noise.
 *
 * Information about the fibre normalisation and throughput derived
 * during processing is saved to numbered columns in the FIBINFO table
 * for information and for later use during processing.  In particular
 * the MED3_* columns are used in qmost_obffcor_fib() to apply the
 * OB-level fibre throughput correction to correct for any changes in
 * the throughput caused by the fibre spine tilt used to position the
 * fibres for the science exposure.
 *
 * @param   spec_img           (Modified) The extracted fibre flat
 *                                        field spectra to process as
 *                                        a 2D image.  The data type
 *                                        must be CPL_TYPE_FLOAT.
 *                                        Will be updated in place to
 *                                        return the result.
 * @param   spec_var           (Modified) The variance of the
 *                                        extracted fibre flat field
 *                                        spectra.  The data type must
 *                                        be CPL_TYPE_FLOAT.  Will be
 *                                        updated in place to return
 *                                        the result.
 * @param   spec_hdr           (Modified) FITS header to populate with
 *                                        QC.
 * @param   arm                (Given)    One of the QMOST_ARM_*
 *                                        constants saying which arm
 *                                        we're processing.
 * @param   fibinfo_tbl        (Modified) The FIBINFO table, or NULL
 *                                        if none.  Will be populated
 *                                        with relative fibre
 *                                        throughputs if given.
 * @param   resptrack          (Given)    If set, then a median
 *                                        response will also
 *                                        calculated from all the good
 *                                        spectra and then removed
 *                                        from each.
 * @param   smooth             (Given)    If set, this is the size of
 *                                        the smoothing box.
 * @param   rescale            (Given)    If set, then the fibres will
 *                                        be rescaled by their
 *                                        relative throughputs. If
 *                                        not, then all fibres will be
 *                                        scaled to a median of one.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If all spectra are flagged
 *                                        bad or one of the input
 *                                        FIBINFO table columns
 *                                        doesn't exist.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the parameter arm isn't
 *                                        valid.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the dimensions of the
 *                                        spectrum 2D image,
 *                                        variance arrays, and FIBINFO
 *                                        table don't match.
 * @retval  CPL_ERROR_INVALID_TYPE        If one of the output FIBINFO
 *                                        table columns already exists
 *                                        but doesn't have a numerical
 *                                        data type.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the data type of the
 *                                        input image was not float,
 *                                        or one of the input FIBINFO
 *                                        columns had an incorrect
 *                                        data type.
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_ID</b>
 *   - <b>FIB_ST</b>
 *   - <b>FIB_USE</b>
 *
 * @par Output DRS Headers:
 *   - <b>FRESCALE</b>: Were fibres rescaled by their relative
 *     throughputs?  This reports the value of the rescale flag to
 *     qmost_ffnorm_fib.
 *
 * @par Output QC Parameters:
 *   - <b>FIBFLAT ENS MAX</b> (ADU): The maximum of the average fluxes
 *     of all of the individual fibre flats in the current image
 *     before normalisation.
 *   - <b>FIBFLAT ENS MAXFIB</b>: The spectrum with the maximum
 *     average flux.
 *   - <b>FIBFLAT ENS MAXSPC</b>: The fibre ID of the fibre with the
 *     maximum average flux.
 *   - <b>FIBFLAT ENS MED</b> (ADU): The median of the average fluxes
 *     of all of the individual fibre flats in the current image
 *     before normalisation. Used to assess the relative throughput of
 *     the fibres.
 *   - <b>FIBFLAT ENS MIN</b> (ADU): The minimum of the average fluxes
 *     of all of the individual fibre flats in the current image
 *     before normalisation.
 *   - <b>FIBFLAT ENS MINFIB</b>: The spectrum with the minimum
 *     average flux.
 *   - <b>FIBFLAT ENS MINSPC</b>: The fibre ID of the fibre with the
 *     minimum average flux.
 *   - <b>FIBFLAT ENS RMS</b> (ADU): The robustly-estimated RMS of the
 *     average fluxes of all of the individual fibre flats in the
 *     current image before normalisation. Used to assess consistency
 *     of the throughput of the fibres.
 *
 * @par Output FIBINFO Table Columns:
 *   - <b>MED1_a</b> (ADU): The measured median flux per spectral
 *     pixel down the fibre in the fibre flat in arm "a" (R, G or B,
 *     depending on the value of parameter arm).
 *   - <b>MED2_a</b> (ADU): The median flux per spectral pixel in the
 *     fibre in arm "a" (R, G or B, depending on the value of
 *     parameter arm) after dividing out the average spectral response
 *     to remove the lamp spectrum.
 *   - <b>MED3_a</b>: The median normalised flux in the fibre in arm
 *     "a" (R, G or B, depending on the value of parameter arm) after
 *     rescaling by the relative fibre throughputs or normalisation to
 *     1 depending on the value of the flag rescale.
 *   - <b>NORMLEVEL_a</b>: The multiplicative scaling factor used to
 *     correct for the relative throughput of the fibre in arm "a" (R,
 *     G or B, depending on the value of parameter arm).  Equivalent
 *     to the reciprocal of the relative throughput of the fibre.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_ffnorm_fib(
    cpl_image *spec_img,
    cpl_image *spec_var,
    cpl_propertylist *spec_hdr,
    int arm,
    cpl_table *fibinfo_tbl,
    int resptrack,
    int smooth,
    int rescale)
{
    int anynul,fib_st,fib_use,i,n,j,ind,minsp,minfib,maxsp,maxfib;
    int nthr,maxfirst,minlast,ndiv;
    int nspec;
    long naxis[2],naxisv[2],npts,nrows;
    float *data,*datav,*dat,medmed,val,minval,maxval;
    float sigmad,*datv,sum,med,mad,lowcut,highcut;
    float meanold,mean,sig,diff,aa,bb,dd,mvar;
    unsigned char *b,allbad;
    cpl_errorstate prestate;
    int nbad,nnot_for_med;

    const char *arm_extname;
    char arm_ltr;

    /* Separate these out to make garbage collection easier */

    float *snr = NULL;
    float *meds1 = NULL;
    float *meds2 = NULL;
    float *meds3 = NULL;
    float *scales = NULL;
    float *buf = NULL;
    float *medprof = NULL;
    float *medprofv = NULL;
    float *arr = NULL;
    float *arrv = NULL;
    unsigned char *bpm = NULL;
    unsigned char *bad_spec = NULL;
    unsigned char *not_for_med = NULL;
    unsigned char *medprof_bpm = NULL;
    int *fibnos = NULL;

    char *colname1 = NULL;
    char *colname2 = NULL;
    char *colname3 = NULL;
    char *colnamen = NULL;

    /* Check for NULL arguments */
    cpl_ensure_code(spec_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_var, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(snr != NULL) {                           \
        cpl_free(snr);                          \
        snr = NULL;                             \
    }                                           \
    if(meds1 != NULL) {                         \
        cpl_free(meds1);                        \
        meds1 = NULL;                           \
    }                                           \
    if(meds2 != NULL) {                         \
        cpl_free(meds2);                        \
        meds2 = NULL;                           \
    }                                           \
    if(meds3 != NULL) {                         \
        cpl_free(meds3);                        \
        meds3 = NULL;                           \
    }                                           \
    if(scales != NULL) {                        \
        cpl_free(scales);                       \
        scales = NULL;                          \
    }                                           \
    if(buf != NULL) {                           \
        cpl_free(buf);                          \
        buf = NULL;                             \
    }                                           \
    if(medprof != NULL) {                       \
        cpl_free(medprof);                      \
        medprof = NULL;                         \
    }                                           \
    if(medprofv != NULL) {                      \
        cpl_free(medprofv);                     \
        medprofv = NULL;                        \
    }                                           \
    if(arr != NULL) {                           \
        cpl_free(arr);                          \
        arr = NULL;                             \
    }                                           \
    if(arrv != NULL) {                          \
        cpl_free(arrv);                         \
        arrv = NULL;                            \
    }                                           \
    if(bpm != NULL) {                           \
        cpl_free(bpm);                          \
        bpm = NULL;                             \
    }                                           \
    if(bad_spec != NULL) {                      \
        cpl_free(bad_spec);                     \
        bad_spec = NULL;                        \
    }                                           \
    if(not_for_med != NULL) {                   \
        cpl_free(not_for_med);                  \
        not_for_med = NULL;                     \
    }                                           \
    if(medprof_bpm != NULL) {                   \
        cpl_free(medprof_bpm);                  \
        medprof_bpm = NULL;                     \
    }                                           \
    if(fibnos != NULL) {                        \
        cpl_free(fibnos);                       \
        fibnos = NULL;                          \
    }                                           \
    if(colname1 != NULL) {                      \
        cpl_free(colname1);                     \
        colname1 = NULL;                        \
    }                                           \
    if(colname2 != NULL) {                      \
        cpl_free(colname2);                     \
        colname2 = NULL;                        \
    }                                           \
    if(colname3 != NULL) {                      \
        cpl_free(colname3);                     \
        colname3 = NULL;                        \
    }                                           \
    if(colnamen != NULL) {                      \
        cpl_free(colnamen);                     \
        colnamen = NULL;                        \
    }

    /* Get input 2D spectra */
    naxis[0] = cpl_image_get_size_x(spec_img);
    naxis[1] = cpl_image_get_size_y(spec_img);

    npts = naxis[0]*naxis[1];

    data = cpl_image_get_data_float(spec_img);
    if(data == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Now the error array */
    naxisv[0] = cpl_image_get_size_x(spec_var);
    naxisv[1] = cpl_image_get_size_y(spec_var);

    if(naxisv[0] != naxis[0] ||
       naxisv[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxisv[0], naxisv[1]);
    }

    datav = cpl_image_get_data_float(spec_var);
    if(datav == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D variance image");
    }

    /* Get arm now so we can check it prior to modifying anything */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not determine EXTNAME "
                                     "for arm %d", arm);
    }
    
    arm_ltr = arm_extname[0];

    /* Get a load of the memory that we're going to need during this... Doing
       it now will make garbage collection much easier */

    snr = cpl_malloc(npts * sizeof(float));
    meds1 = cpl_calloc(naxis[1], sizeof(float));
    meds2 = cpl_calloc(naxis[1], sizeof(float));
    meds3 = cpl_calloc(naxis[1], sizeof(float));
    scales = cpl_calloc(naxis[1], sizeof(float));
    buf = cpl_malloc(naxis[0] * sizeof(float));
    medprof = cpl_malloc(naxis[0] * sizeof(float));
    medprofv = cpl_malloc(naxis[0] * sizeof(float));
    arr = cpl_malloc(naxis[1] * sizeof(float));
    arrv = cpl_malloc(naxis[1] * sizeof(float));
    bpm = cpl_calloc(npts, sizeof(unsigned char));
    bad_spec = cpl_calloc(naxis[1], sizeof(unsigned char));
    not_for_med = cpl_calloc(naxis[1], sizeof(unsigned char));
    medprof_bpm = cpl_calloc(naxis[0], sizeof(unsigned char));

    /* Calculate variance and SNR */

    memset(snr,0,npts*sizeof(float));
    for (i = 0; i < npts; i++) {
        if (datav[i] > 0.0)
            snr[i] = data[i]/sqrt(datav[i]);
    }

    /* Get the fibre number info from the fibre table (if there is
     * one) and check fibres are illuminated, flag bad if not */

    if(fibinfo_tbl != NULL) {
        nrows = cpl_table_get_nrow(fibinfo_tbl);
        if(nrows != naxis[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "fibre flat FIBINFO table has "
                                         "%ld fibres, "
                                         "number of spectra is %ld, "
                                         "these must match",
                                         nrows,
                                         naxis[1]);
        }

        fibnos = cpl_malloc(nrows*sizeof(int));

        nbad = 0;
        nnot_for_med = 0;

        for(j = 0; j < nrows; j++) {
            fibnos[j] = cpl_table_get_int(fibinfo_tbl, 
                                          "FIB_ID",
                                          j,
                                          &anynul);
            if(anynul < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d from FIBINFO",
                                             "FIB_ID",
                                             j+1);
            }
            else if(anynul > 0) {  /* NULL */
                fibnos[j] = -1;
            }

            fib_st = cpl_table_get_int(fibinfo_tbl,
                                       "FIB_ST",
                                       j,
                                       &anynul);
            if(anynul < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d from FIBINFO",
                                             "FIB_ST",
                                             j+1);
            }
            else if(anynul > 0) {  /* NULL */
                fib_st = -1;
            }

            fib_use = cpl_table_get_int(fibinfo_tbl,
                                        "FIB_USE",
                                        j,
                                        &anynul);
            if(anynul < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d from FIBINFO",
                                             "FIB_USE",
                                             j+1);
            }
            else if(anynul > 0) {  /* NULL */
                fib_use = -1;
            }

            /* Bad fibres */
            if(fib_st == 0) {
                bad_spec[j] = 1;
                nbad++;
            }

            /* Fibres we shouldn't use in median spectrum or median
             * normalisation: bad or simucal, where simucal shouldn't
             * be used because they have different illumination. */
            if(fib_st == 0 || fib_use == 0) {
                not_for_med[j] = 1;
                nnot_for_med++;
            }
        }

        /* Special case where all fibres are simucals, in this case we
         * allow use of the simucals. */
        if(nnot_for_med >= nrows && nbad < nrows) {
            memcpy(not_for_med, bad_spec, naxis[1]*sizeof(unsigned char));
        }

    } else {
        fibnos = cpl_malloc(naxis[1]*sizeof(int));
        for (j = 0; j < naxis[1]; j++)
            fibnos[j] = j+1;
    }

    /* Create a bad pixel mask for each fibre */

    maxfirst = 0;
    minlast = naxis[0]-1;
    for (i = 0; i < naxis[1]; i++) {
        dat = datav + i*naxis[0];
        b = bpm + i*naxis[0];
        for (j = 0; j < naxis[0]; j++) {
            if (dat[j] == 0.0)
                b[j] = 1;
        }
        qmost_sumbpm(b,(int)naxis[0],&j);
        if(j == naxis[0]) {
            bad_spec[i] = 1;
        }
    }
    qmost_sumbpm(bad_spec,(int)naxis[1],&j);
    allbad = (j == naxis[1]);
    if (allbad) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "all spectra flagged bad");
    }

    /* Get the medians of each spectrum. Ignore dead fibres */

    minval = 1.0e10;
    maxval = -1.0e10;
    minsp = -1;
    maxsp = -1;
    minfib = -1;
    maxfib = -1;
    nspec = 0;
    for (i = 0; i < naxis[1]; i++) {
        if (bad_spec[i])
            continue;
	dat = data + i*naxis[0];
        b = bpm + i*naxis[0];
        n = 0;
        for (j = maxfirst; j <= minlast; j++)
            if (b[j] == 0)
                buf[n++] = dat[j];

        prestate = cpl_errorstate_get();

        if(qmost_med(buf,NULL,n,&val) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            val = 0;
        }

	nspec++;
        meds1[i] = val;
        if (val < minval) {
            minval = val;
            minsp = i + 1;
            minfib = fibnos[i];
        }
        if (val > maxval) {
            maxval = val;
            maxsp = i + 1;
            maxfib = fibnos[i];
        }
    }

    cpl_free(fibnos);
    fibnos = NULL;

    /* If we're taking out the median response, then do that here. Start by
       defining it over the spectral region where at least 5% of the fibres
       can contribute to it. */

    nthr = qmost_max(1,nspec/20);

    if (resptrack) {
        nbad = 0;
        for (j = 0; j < naxis[0]; j++) {
            n = 0;
            for (i = 0; i < naxis[1]; i++) {
                if (not_for_med[i] || bpm[i*naxis[0] + j])
                    continue;
                dat = data + i*naxis[0];
                datv = datav + i*naxis[0];
                arr[n] = dat[j]/meds1[i];
                arrv[n++] = datv[j]/pow(meds1[i],2.0);
            }
            if (n > nthr) {
                prestate = cpl_errorstate_get();

                if(qmost_medmad(arr,NULL,n,&med,&mad) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                    med = 0;
                    mad = 0;
                }

                mad *= 1.48;

                lowcut = med - 3.0*mad;
                highcut = med + 3.0*mad;
                meanold = 1.0e10;
                sig = 1.0e10;
                for (i = 0; i < 3; i++) {
                    if(ff_meanvarcut(arr,arrv,n,lowcut,highcut,&mean,
                                     &mvar,&sig) != 0) {
                        mean = med;
                        mvar = mad*mad / n;
                        break;
                    }
                    diff = fabs((mean - meanold)/mean);
                    if (diff < 0.01)
                        break;
                    lowcut = mean - 3.0*sig;
                    highcut = mean + 3.0*sig;
                    meanold = mean;
                }
                medprof[j] = mean;
                medprofv[j] = mvar;
                medprof_bpm[j] = 0;
            } else {
                medprof[j] = 0.0;
                medprofv[j] = 0.0;
                medprof_bpm[j] = 1;
                nbad++;
            }
        }

        if(nbad < naxis[0]) {
            /* Extrapolate the ends and fill in any bad pixels in the
             * middle (there shouldn't be any of the latter, but you
             * never know...) */
            plugholes(medprof,medprof_bpm,naxis[0]);
            plugholes(medprofv,medprof_bpm,naxis[0]);

            /* Now remove this average response from each fibre */

            for (j = 0; j < naxis[1]; j++) {
                if (bad_spec[j])
                    continue;
                dat = data + j*naxis[0];
                datv = datav + j*naxis[0];
                b = bpm + j*naxis[0];
                sum = 0.0;
                ndiv = 0;
                for (i = 0; i < naxis[0]; i++) {
                    if (b[i] == 0) {
                        dd = dat[i]/medprof[i];
                        aa = datv[i];
                        bb = medprofv[i];
                        dat[i] = dd;
                        datv[i] = (aa + dd*dd*bb) / (medprof[i]*medprof[i]);
                        sum += medprof[i];
                        ndiv++;
                    }
                }
                if(ndiv > 0) {
                    sum /= (float)ndiv;
                    for (i = 0; i < naxis[0]; i++) {
                        if (b[i] == 0) {
                            dat[i] *= sum;
                            datv[i] *= pow(sum,2.0);
                        }
                    }
                }

                /* Work out final median */

                n = 0;
                for (i = maxfirst; i <= minlast; i++) {
                    if (b[i] == 0) {
                        buf[n++] = dat[i];
                    }
                }

                prestate = cpl_errorstate_get();

                if(qmost_med(buf,NULL,n,&val) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                    val = 0;
                }

                meds2[j] = val;
            }
        }
        else {
            memcpy(meds2,meds1,naxis[1]*sizeof(float));
        }
    } else {
        memcpy(meds2,meds1,naxis[1]*sizeof(float));
    }
    
    /* If rescaling to the fibre throughputs, work out the median of the 
       medians and normalise the data array. If this is a bad fibre then 
       replace it with 1s. */

    if (rescale) {
        prestate = cpl_errorstate_get();

        if(qmost_medmad(meds2,not_for_med,naxis[1],
                        &medmed,&sigmad) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            medmed = 1.0;
            sigmad = 0.0;
        }

        sigmad *= 1.48;

        aa = 1.0/medmed;
        bb = pow(aa,2.0);
        for (i = 0; i < naxis[1]; i++) {
            scales[i] = aa;
            ind = i*naxis[0];
            if (! bad_spec[i]) {
                for (j = 0; j < naxis[0]; j++) {
                    if (bpm[j+ind] == 0) {
                        data[j+ind] *= aa;
                        datav[j+ind] *= bb;
                    }
                }
            } else {
                for (j = 0; j < naxis[0]; j++) {
                    data[j+ind] = 1.0;
                    datav[j+ind] = 0.0;
                }
            }
        }
    } else {
        for (i = 0; i < naxis[1]; i++) {
            if (bad_spec[i])
                continue;
            dat = data + i*naxis[0];
            datv = datav + i*naxis[0];
            b = bpm + i*naxis[0];
            n = 0;
            for (j = maxfirst; j <= minlast; j++)
                if (b[j] == 0)
                    buf[n++] = dat[j];

            prestate = cpl_errorstate_get();

            if(qmost_med(buf,NULL,n,&val) != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);
                val = 1.0;
            }

            bb = pow(val,2.0);
            scales[i] = 1.0 / val;
            for (j = 0; j < naxis[0]; j++) {
                dat[j] /= val;
                datv[j] /= bb;
            }
        }
    }

    /* Work out a final median for each spectrum */

    for (i = 0; i < naxis[1]; i++) {
        if (bad_spec[i])
            continue;

        dat = data + i*naxis[0];
        b = bpm + i*naxis[0];
        n = 0;
        for (j = maxfirst; j <= minlast; j++)
            if (b[j] == 0)
                buf[n++] = dat[j];

        prestate = cpl_errorstate_get();

        if(qmost_med(buf,NULL,n,&val) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            val = 0;
        }

        meds3[i] = val;
    }

    /* Rescale by SNR to find output variance */

    for (i = 0; i < npts; i++) {
        if (snr[i] == 0.0) {
            datav[i] = 0.0;
        } else {
            datav[i] = pow(data[i]/snr[i],2.0);
        }
    }

    /* If smoothing is requested, then do that now */

    if (smooth) {
#pragma omp parallel for default(none) private(i, dat, datv, val) shared(data, datav, naxis, meds3, bad_spec, smooth)
        for (i = 0; i < naxis[1]; i++) {
            dat = data + i*naxis[0];
            datv = datav + i*naxis[0];
	    val = meds3[i];
            if (! bad_spec[i]) 
	        ff_filt1d(dat,datv,val,naxis[0],smooth);
        }
    }           

    /* Add some QC information to the header */

    if(spec_hdr != NULL) {
        cpl_propertylist_update_float(spec_hdr,
                                      "ESO QC FIBFLAT ENS MED",
                                      medmed);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MED",
                                     "[ADU] Ensemble median flux");

        cpl_propertylist_update_float(spec_hdr,
                                      "ESO QC FIBFLAT ENS RMS",
                                      sigmad);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS RMS",
                                     "[ADU] Sigma of median flux");

        cpl_propertylist_update_float(spec_hdr,
                                      "ESO QC FIBFLAT ENS MIN",
                                      minval);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MIN",
                                     "[ADU] Minimum median flux");

        cpl_propertylist_update_int(spec_hdr,
                                    "ESO QC FIBFLAT ENS MINSPC",
                                    minsp);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MINSPC",
                                     "Spectrum with min median flux");

        cpl_propertylist_update_int(spec_hdr,
                                    "ESO QC FIBFLAT ENS MINFIB",
                                    minfib);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MINFIB",
                                     "Fibre ID with min median flux");

        cpl_propertylist_update_float(spec_hdr,
                                      "ESO QC FIBFLAT ENS MAX",
                                      maxval);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MAX",
                                     "[ADU] Maximum median flux");

        cpl_propertylist_update_int(spec_hdr,
                                    "ESO QC FIBFLAT ENS MAXSPC",
                                    maxsp);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MAXSPC",
                                     "Spectrum with max median flux");

        cpl_propertylist_update_int(spec_hdr,
                                    "ESO QC FIBFLAT ENS MAXFIB",
                                    maxfib);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO QC FIBFLAT ENS MAXFIB",
                                     "Fibre ID with max median flux");

        cpl_propertylist_update_int(spec_hdr,
                                    "ESO DRS FRESCALE",
                                    rescale);
        cpl_propertylist_set_comment(spec_hdr,
                                     "ESO DRS FRESCALE",
                                     rescale ?
                                     "Normalised to ensemble median=1" :
                                     "Normalised to individual median=1");
    }

    /* Add some stuff to the fibtable if you can */

    if(fibinfo_tbl != NULL) {
        nrows = cpl_table_get_nrow(fibinfo_tbl);
        
        colname1 = cpl_sprintf("MED1_%c", arm_ltr);
        colname2 = cpl_sprintf("MED2_%c", arm_ltr);
        colname3 = cpl_sprintf("MED3_%c", arm_ltr);
        colnamen = cpl_sprintf("NORMLEVEL_%c", arm_ltr);
        
        if(!cpl_table_has_column(fibinfo_tbl, colname1)) {
            if(cpl_table_new_column(fibinfo_tbl,
                                    colname1,
                                    CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create FIBINFO column "
                                             "MED1 for arm %c", arm_ltr);
            }

	    if(cpl_table_set_column_unit(fibinfo_tbl,
					 colname1,
					 "ADU") != CPL_ERROR_NONE) {
		TIDY;
		return cpl_error_set_message(cpl_func, cpl_error_get_code(),
					     "could not set MED1 unit "
					     "for arm %c", arm_ltr);
	    }
        }

        if(!cpl_table_has_column(fibinfo_tbl, colname2)) {
            if(cpl_table_new_column(fibinfo_tbl,
                                    colname2,
                                    CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create FIBINFO column "
                                             "MED2 for arm %c", arm_ltr);
            }

	    if(cpl_table_set_column_unit(fibinfo_tbl,
					 colname2,
					 "ADU") != CPL_ERROR_NONE) {
		TIDY;
		return cpl_error_set_message(cpl_func, cpl_error_get_code(),
					     "could not set MED2 unit "
					     "for arm %c", arm_ltr);
	    }
        }

        if(!cpl_table_has_column(fibinfo_tbl, colname3)) {
            if(cpl_table_new_column(fibinfo_tbl,
                                    colname3,
                                    CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create FIBINFO column "
                                             "MED3 for arm %c", arm_ltr);
            }
        }

        if(!cpl_table_has_column(fibinfo_tbl, colnamen)) {
            if(cpl_table_new_column(fibinfo_tbl,
                                    colnamen,
                                    CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create FIBINFO column "
                                             "NORMLEVEL for arm %c", arm_ltr);
            }
        }

        for(j = 0; j < nrows; j++) {
            if(cpl_table_set(fibinfo_tbl,
                             colname1,
                             j,
                             meds1[j]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to write MED1 for row "
                                             "%d to FIBINFO", j+1);
            }

            if(cpl_table_set(fibinfo_tbl,
                             colname2,
                             j,
                             meds2[j]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to write MED2 for row "
                                             "%d to FIBINFO", j+1);
            }

            if(cpl_table_set(fibinfo_tbl,
                             colname3,
                             j,
                             meds3[j]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to write MED3 for row "
                                             "%d to FIBINFO", j+1);
            }

            if(cpl_table_set(fibinfo_tbl,
                             colnamen,
                             j,
                             scales[j]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to write NORMLEVEL "
                                             "for row %d to FIBINFO", j+1);
            }
        }

        cpl_free(colname1);
        colname1 = NULL;

        cpl_free(colname2);
        colname2 = NULL;

        cpl_free(colname3);
        colname3 = NULL;

        cpl_free(colnamen);
        colnamen = NULL;
    }

    /* Tidy and exit */
    cpl_free(snr);
    snr = NULL;

    cpl_free(meds1);
    meds1 = NULL;

    cpl_free(meds2);
    meds2 = NULL;

    cpl_free(meds3);
    meds3 = NULL;

    cpl_free(scales);
    scales = NULL;

    cpl_free(buf);
    buf = NULL;

    cpl_free(medprof);
    medprof = NULL;

    cpl_free(medprofv);
    medprofv = NULL;

    cpl_free(arr);
    arr = NULL;

    cpl_free(arrv);
    arrv = NULL;

    cpl_free(bpm);
    bpm = NULL;

    cpl_free(bad_spec);
    bad_spec = NULL;

    cpl_free(not_for_med);
    not_for_med = NULL;

    cpl_free(medprof_bpm);
    medprof_bpm = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Correct relative fibre throughputs using OB or twilight
 *          flat.
 *
 * Spectra that have been flat field corrected using a master fibre
 * flat with routine qmost_ffdiv_fib() are further corrected for
 * changes in the relative throughputs of the fibres measured using an
 * OB-level or twilight fibre flat field.  The MED3_* columns of the
 * input FIBINFO table are used to obtain the correction factor, which
 * is re-normalised to an ensemble average of unity and then divided
 * out of the spectra.
 *
 * This routine is used for two purposes in 4MOST.  The twilight flat
 * fields measure the "ground truth" relative throughputs of the
 * fibres, and are used to correct for illumination differences
 * between the fibres from the "light sabres" at the telescope
 * secondary used to feed calibration light into the science fibres
 * for the internal "daytime" fibre flats used to make the master
 * fibre flat.  These are obtained in the same "neutral" fibre
 * positions as the master fibre flat.  The OB-level fibre flats are
 * used to correct for throughput changes caused by tilting the fibre
 * spines to position them on the science targets (relative to their
 * "neutral" positions).
 *
 * @param   spec_img           (Modified) The spectra to be flat
 *                                        fielded as a 2D image.  The
 *                                        data type must be
 *                                        CPL_TYPE_FLOAT.
 * @param   spec_var           (Modified) The variance of the
 *                                        spectra.  The data type must
 *                                        be CPL_TYPE_FLOAT.
 * @param   trace_tbl          (Given)    The trace table used in
 *                                        spectral extraction.
 * @param   arm                (Given)    One of the QMOST_ARM_*
 *                                        constants saying which arm
 *                                        we're processing.
 * @param   obffn_fibinfo      (Given)    The FIBINFO table from the
 *                                        normalised OB-level fibre
 *                                        flat.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required
 *                                        FIBINFO table columns is
 *                                        missing.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the trace table isn't
 *                                        valid.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the trace or FIBINFO
 *                                        table dimensions don't match
 *                                        the spectrum, or the
 *                                        spectrum 2D image and
 *                                        variance array dimensions
 *                                        don't match.
 * @retval  CPL_ERROR_INVALID_TYPE        If the input FIBINFO table
 *                                        columns had an incorrect
 *                                        (non-numerical) data type.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the data type of the
 *                                        input image was not float.
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_ST</b>
 *   - <b>FIB_USE</b>
 *   - <b>MED3_a</b> (a = R, G, or B depending on parameter arm).
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_obffcor_fib(
    cpl_image *spec_img,
    cpl_image *spec_var,
    cpl_table *trace_tbl,
    int arm,
    cpl_table *obffn_fibinfo)
{
    int i,j,anynul;
    long naxis[2],naxisv[2],nx,ny,nrows;
    float *idata,*ivar,*id,*iv,medmed,fac,fac2;
    const char *arm_extname;
    char arm_ltr;
    int fib_st,fib_use;

    float *meds = NULL;
    unsigned char *skip = NULL;
    qmost_traceinfo *tr = NULL;
    char *colname = NULL;

    cpl_errorstate prestate;

    /* Check for NULL arguments */
    cpl_ensure_code(spec_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_var, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(obffn_fibinfo, CPL_ERROR_NULL_INPUT);    

#undef TIDY
#define TIDY                                    \
    if(meds != NULL) {                          \
        cpl_free(meds);                         \
        meds = NULL;                            \
    }                                           \
    if(skip != NULL) {                          \
        cpl_free(skip);                         \
        skip = NULL;                            \
    }                                           \
    if(tr != NULL) {                            \
        qmost_trclose(1, &tr);                  \
        tr = NULL;                              \
    }                                           \
    if(colname != NULL) {                       \
        cpl_free(colname);                      \
        colname = NULL;                         \
    }

    /* Get input 2D spectra */
    naxis[0] = cpl_image_get_size_x(spec_img);
    naxis[1] = cpl_image_get_size_y(spec_img);

    idata = cpl_image_get_data_float(spec_img);
    if(idata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Now the error array */
    naxisv[0] = cpl_image_get_size_x(spec_var);
    naxisv[1] = cpl_image_get_size_y(spec_var);

    if(naxisv[0] != naxis[0] ||
       naxisv[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxisv[0], naxisv[1]);
    }

    ivar = cpl_image_get_data_float(spec_var);
    if(ivar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D variance image");
    }

    nx = naxis[0];
    ny = naxis[1];

    /* Check number of fibres in FIBINFO is the same as the number of
       spectra (ny) */
    nrows = cpl_table_get_nrow(obffn_fibinfo);
    if(nrows != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "OB fibre flat FIBINFO table has "
                                     "%ld fibres, "
                                     "number of spectra is %ld, "
                                     "these must match",
                                     nrows,
                                     naxis[1]);
    }

    /* Open the trace and check that the number of fibres is the same
       as the number of spectra (ny) */

    if(qmost_trchk(trace_tbl) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "bad trace table");
    }

    nrows = cpl_table_get_nrow(trace_tbl);
    if(nrows != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "trace table has %ld fibres, "
                                     "number of spectra is %ld, "
                                     "these must match",
                                     nrows,
                                     naxis[1]);
    }

    /* Get some memory for the data arrays */

    meds = cpl_malloc(ny*sizeof(float));
    skip = cpl_calloc(ny,sizeof(unsigned char));

    /* Read factors from flat FIBINFO */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not determine EXTNAME "
                                     "for arm %d", arm);
    }
    
    arm_ltr = arm_extname[0];
    
    colname = cpl_sprintf("MED3_%c", arm_ltr);

    for(i = 0; i < ny; i++) {
        meds[i] = cpl_table_get(obffn_fibinfo, 
                                colname,
                                i,
                                &anynul);
        if(anynul < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read MED3 column "
                                         "for row %d from OB fibre flat "
                                         "FIBINFO",
                                         i+1);
        }
        else if(anynul > 0) {  /* NULL */
            meds[i] = NAN;
        }
    }

    cpl_free(colname);
    colname = NULL;

    /* Figure out which fibres we can use.  This considers both
     * whether they are flagged as bad in the trace table, and whether
     * they were illuminated in the flat based on the FIBINFO.  The
     * extra check is needed to deal with twilight flats where the
     * simucal fibres aren't illuminated in the flat, so they need to
     * be excluded. */

    for (i = 0; i < ny; i++) {
        tr = cpl_calloc(1, sizeof(qmost_traceinfo));

        if(qmost_trread1(trace_tbl, i+1, tr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read trace table "
                                         "row %d",
                                         i+1);
        }

        fib_st = cpl_table_get_int(obffn_fibinfo,
                                   "FIB_ST",
                                   i,
                                   &anynul);
        if(anynul < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d from FIBINFO",
                                         "FIB_ST",
                                         i+1);
        }
        else if(anynul > 0) {  /* NULL */
            fib_st = -1;
        }

        fib_use = cpl_table_get_int(obffn_fibinfo,
                                    "FIB_USE",
                                    i,
                                    &anynul);
        if(anynul < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d from FIBINFO",
                                         "FIB_USE",
                                         i+1);
        }
        else if(anynul > 0) {  /* NULL */
            fib_use = -1;
        }

        /* Don't use or correct bad fibres or simucal fibres */
        if (!tr->live || fib_st == 0 || fib_use == 0)
            skip[i] = 1;

        qmost_trclose(1, &tr);
        tr = NULL;
    }

    /* Work out the ensemble median of the flat fields */

    prestate = cpl_errorstate_get();

    if(qmost_med(meds,skip,ny,&medmed) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
        medmed = 0;
    }

    if(medmed != 0) {
        for (i = 0; i < ny; i++)
            if (! skip[i])
                meds[i] /= medmed;
    }
            
    /* Loop for each spectrum. Do nothing if the fibre should be skipped */

    for (i = 0; i < ny; i++) {
        if (skip[i])
            continue;

	/* Do the flat field correction */

	id = idata + i*nx;
	iv = ivar + i*nx;
        fac = meds[i];
        fac2 = meds[i]*meds[i];
        if(fac != 0 && fac2 != 0) {
            for (j = 0; j < nx; j++) {
                if (iv[j] != 0.0) {
                    id[j] /= fac;
                    iv[j] /= fac2;
                }
            }
        }
    }

    cpl_free(meds);
    meds = NULL;

    cpl_free(skip);
    skip = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Divide spectra by normalised flats.
 *
 * Divide each spectrum in an image by its corresponding normalised
 * master fibre flat field spectrum.
 *
 * @param   spec_img           (Modified) The spectra to be flat
 *                                        fielded as a 2D image.  The
 *                                        data type must be
 *                                        CPL_TYPE_FLOAT.
 * @param   spec_var           (Modified) The variance of the
 *                                        spectra.  The data type must
 *                                        be CPL_TYPE_FLOAT.
 * @param   spec_hdr           (Given)    The corresponding FITS
 *                                        header.
 * @param   trace_tbl          (Given)    The trace table used in
 *                                        spectral extraction.
 * @param   ffnorm_img         (Given)    The normalised master fibre
 *                                        flat.
 * @param   ffnorm_var         (Given)    The variance in the
 *                                        normalised master fibre
 *                                        flat.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS header keywords was not
 *                                        found.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the trace table isn't
 *                                        valid.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the trace table
 *                                        dimensions don't match 
 *                                        the spectrum, or the
 *                                        spectrum 2D image, variance,
 *                                        normalised flat and variance
 *                                        array dimensions don't
 *                                        match.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the data type of the
 *                                        input image was not float,
 *                                        or one of the required input
 *                                        FITS header keyword values
 *                                        had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS WVNOFF</b>
 *   - <b>WVNOFF</b>
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_ffdiv_fib(
    cpl_image *spec_img,
    cpl_image *spec_var,
    cpl_propertylist *spec_hdr,
    cpl_table *trace_tbl,
    cpl_image *ffnorm_img,
    cpl_image *ffnorm_var)
{
    int i,j,specbin,spatbin,isbinned,pixoff;
    long naxis[2],naxisv[2],bnaxis[2],nx,ny,nrows;
    float *idata,*fdata,*ivar,*fvar,*id,*fd,*iv,*fv,d1,d2,v1,v2;

    float *bfdata = NULL;
    float *bfvar = NULL;
    qmost_traceinfo *tr = NULL;

    /* Check for NULL arguments */
    cpl_ensure_code(spec_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_var, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(spec_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ffnorm_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ffnorm_var, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(bfdata != NULL) {                        \
        cpl_free(bfdata);                       \
        bfdata = NULL;                          \
    }                                           \
    if(bfvar != NULL) {                         \
        cpl_free(bfvar);                        \
        bfvar = NULL;                           \
    }                                           \
    if(tr != NULL) {                            \
        qmost_trclose(1, &tr);                  \
        tr = NULL;                              \
    }

    /* Get input 2D spectra */
    naxis[0] = cpl_image_get_size_x(spec_img);
    naxis[1] = cpl_image_get_size_y(spec_img);

    idata = cpl_image_get_data_float(spec_img);
    if(idata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    /* Now the error array */
    naxisv[0] = cpl_image_get_size_x(spec_var);
    naxisv[1] = cpl_image_get_size_y(spec_var);

    if(naxisv[0] != naxis[0] ||
       naxisv[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxisv[0], naxisv[1]);
    }

    ivar = cpl_image_get_data_float(spec_var);
    if(ivar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D variance image");
    }

    nx = naxis[0];
    ny = naxis[1];

    qmost_isbinned(spec_hdr,&specbin,&spatbin,&isbinned);

    /* Get flat */
    naxis[0] = cpl_image_get_size_x(ffnorm_img);
    naxis[1] = cpl_image_get_size_y(ffnorm_img);

    if(naxis[1] != ny) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and fibre flat "
                                     "number of spectra "
                                     "don't match: %ld != %ld",
                                     ny, naxis[1]);
    }

    fdata = cpl_image_get_data_float(ffnorm_img);
    if(fdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input normalised fibre flat");
    }

    /* Now the error array */
    naxisv[0] = cpl_image_get_size_x(ffnorm_var);
    naxisv[1] = cpl_image_get_size_y(ffnorm_var);

    if(naxisv[0] != naxis[0] ||
       naxisv[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "fibre flat spectrum and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxisv[0], naxisv[1]);
    }

    fvar = cpl_image_get_data_float(ffnorm_var);
    if(fvar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D variance image");
    }

    /* Open the trace and check that the number of fibres is the same
       as the number of spectra (ny) */

    if(qmost_trchk(trace_tbl) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "bad trace table");
    }

    nrows = cpl_table_get_nrow(trace_tbl);
    if(nrows != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "trace table has %ld fibres, "
                                     "number of spectra is %ld, "
                                     "these must match",
                                     nrows,
                                     naxis[1]);
    }

    /* Redefine the flat field images so the reflect any spectral binning
       that make have taken place */

    if (specbin > 1) {
        /* Get WVNOFF.  Default to 0 if not present in header. */
        if(cpl_propertylist_has(spec_hdr, "ESO DRS WVNOFF")) {
            if(qmost_cpl_propertylist_get_int(spec_hdr,
                                              "ESO DRS WVNOFF",
                                              &pixoff) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not read "
                                             "ESO DRS WVNOFF from "
                                             "input FITS header");
            }
        }
        else if(cpl_propertylist_has(spec_hdr, "WVNOFF")) {
            if(qmost_cpl_propertylist_get_int(spec_hdr,
                                              "WVNOFF",
                                              &pixoff) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not read "
                                             "WVNOFF from "
                                             "input FITS header");
            }
        }
        else {
            pixoff = 0;
        }

        qmost_flatblk(fdata,fvar,naxis,nx,specbin,pixoff,
                      &bfdata,&bfvar,bnaxis);

        if(bfdata && bfvar) {
            /* Swap them */
            fdata = bfdata;
            fvar = bfvar;

            memcpy(naxis, bnaxis, sizeof(naxis));
        }
    }

    if(naxis[0] != nx) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and fibre flat "
                                     "wavelength axis "
                                     "doesn't match: %ld != %ld",
                                     nx, naxis[0]);
    }

    /* Loop for each spectrum. Do nothing if the fibre is dead */

    for (i = 1; i <= ny; i++) {
        tr = cpl_calloc(1, sizeof(qmost_traceinfo));

        if(qmost_trread1(trace_tbl, i, tr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read trace table "
                                         "row %d",
                                         i);
        }

        if (! tr->live) {
            qmost_trclose(1,&tr);
            continue;
        }
        qmost_trclose(1,&tr);

        /* Do the flat field correction */

        id = idata + (i-1)*nx;
        fd = fdata + (i-1)*nx;
        iv = ivar + (i-1)*nx;
        fv = fvar + (i-1)*nx;
        for (j = 0; j < nx; j++) {
            d1 = id[j];
            d2 = fd[j];
            v1 = iv[j];
            v2 = fv[j];
            if (v1 != 0.0) {
                if (d2 <= 0.1) {
                    /* Flag as bad */
                    iv[j] = 0.0;
                } else {
                    id[j] = d1 / d2;
                    iv[j] = (v1 + id[j]*id[j]*v2) / (d2*d2);
                }
            }
        }
    }

    if(bfdata) {
        cpl_free(bfdata);
        bfdata = NULL;
    }

    if(bfvar) {
        cpl_free(bfvar);
        bfvar = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Interpolate out bad pixels in a 1D data array.
 *
 * @param   data               (Modified) The data to update.
 * @param   bpm                (Given)    Bad pixel mask, flagging bad
 *                                        pixels with non-zero values.
 * @param   nx                 (Given)    The number of pixels in the
 *                                        arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void plugholes(float *data, unsigned char *bpm, int nx) {
    int i,ifirst,ilast,i1,i2,j;
    float nc,d1,d2,t1,t2,slope,med1,med2,inter;
    cpl_errorstate prestate;

    /* First of all, find the first good value in the array */

    i = 0;
    while (i < nx && bpm[i] != 0)
        i++;
    ifirst = i;

    /* If all the values in the array are bad, then do nothing */

    if (ifirst == nx)
        return;

    /* Find the last good value in the array */

    i = nx - 1;
    while (i >= 0 && bpm[i] != 0)
        i--;
    ilast = i;

    /* Right, now start from the first good value and fill in any holes in the
       middle part of the array */

    i = ifirst;
    while (i <= ilast) {
        if (bpm[i] == 0) {
            i++;
            continue;
        }
        i1 = i - 1;
        while (bpm[i] != 0)
            i++;
        i2 = i;
        nc = (float)(i2 - i1);
        d1 = data[i1];
        d2 = data[i2];
        for (j = i1+1; j <= i2-1; j++) {
            t1 = 1.0 - (float)(j - i1)/nc;
            t2 = 1.0 - t1;
            data[j] = qmost_max(0.0,t1*d1 + t2*d2);
        }
    }

    /* Now the left bit... */

    if (ifirst > 0) {
        prestate = cpl_errorstate_get();

        if(qmost_med(data+ifirst,NULL,5,&med1) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            med1 = 0;
        }

        if(qmost_med(data+ifirst+5,NULL,5,&med2) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            med2 = 0;
        }

        slope = 0.2*(med2 - med1);
        inter = med2 - slope*(float)(ifirst+7.5);
        for (j = 0; j < ifirst; j++)
	  data[j] = qmost_max(0.0,slope*(float)j + inter);
    }

    /* Now the right bit... */

    if (ilast < nx - 1) {
        prestate = cpl_errorstate_get();

        if(qmost_med(data+ilast-5,NULL,5,&med1) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            med1 = 0;
        }

        if(qmost_med(data+ilast-10,NULL,5,&med2) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            med2 = 0;
        }

        slope = 0.2*(med2 - med1);
        inter = med2 - slope*(float)(ilast-7.5);
        for (j = ilast+1; j < nx; j++)
	  data[j] = qmost_max(0.0,slope*(float)j + inter);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Find the mean and mean variance of a float data array. Cut
 *          data outside a defined data window.
 *
 * The mean and mean variance of a float data array is found in the
 * usual way. Pixels that are outside a predefined data window are
 * removed.
 *
 * @param   data               (Given)    The input data.
 * @param   datav              (Given)    The input data variance.
 * @param   npts               (Given)    The number of pixels in the
 *                                        input data arrays.
 * @param   lcut               (Given)    The lowest allowable value.
 * @param   hcut               (Given)    The highest allowable value.
 * @param   mean               (Returned) The value of the mean.
 * @param   meanv              (Returned) The variance in the mean.
 * @param   sig                (Returned) The standard deviation.
 *
 * @return  int
 *
 * @retval  0     Success.
 * @retval  -1    If there were no good pixels.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static int ff_meanvarcut(float *data, float *datav, int npts, float lcut,
                         float hcut, float *mean, float *meanv, float *sig) {
    int i,n;
    double sum,sumv,sum2;

    /* Separate sections depending on whether there is a BPM or not */

    sum = 0.0;
    sumv = 0.0;
    sum2 = 0.0;
    n = 0;
    for (i = 0; i < npts; i++) {
        if (data[i] > lcut && data[i] < hcut && datav[i] != 0.0) {
            sum += data[i];
            sum2 += pow(data[i],2.0);
            sumv += datav[i];
            n++;
        }
    }

    /* Check whether we can do the mean and sigma calculations */

    if (n > 1) {
        sum /= (double)n;
        *mean = (float)sum;
        sumv /= ((double)n) * ((double)n);
        *meanv = (float)sumv;
        *sig = sum2/(double)n - sum*sum;
        if (*sig > 0.0)
            *sig = sqrt(*sig);
        else
            *sig = 0.01;
    } else {
        *mean = 0.0;
        *meanv = 0.0;
        return -1;
    }
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Smooth a 1d array and its associated variance array.
 *
 * Smooth a 1d array and its associated variance array. A variance of
 * zero indicates a bad pixel.
 *
 * @param   data               (Modified) The input/output data.
 * @param   var                (Modified) The input/output data
 *                                        variance.
 * @param   aval               (Given)    The median fibre response.
 * @param   nx                 (Given)    The number of pixels in the
 *                                        input data arrays.
 * @param   smooth             (Given)    The size of the smoothing
 *                                        box.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void ff_filt1d(float *data, float *var, float aval, long nx, int smooth) {
    int i,nfilt,il,n;
    float mean,meanv,sig,lcut,hcut,*d,*v;
    float *ybuf = NULL;
    float *ybufv = NULL;

    /* Make sure you have an odd number of pixels in the filter window */

    if (smooth <= 1)
        return;
    nfilt = smooth;
    if (! nfilt % 2)
        nfilt++;
    if (nx <= nfilt)
        return;
    il = nfilt/2;

    /* Allocate buffers */

    ybuf = cpl_calloc(nx,sizeof(float));
    ybufv = cpl_calloc(nx,sizeof(float));

    /* Do the filtering now */

    for (i = 0; i < il; i++) {
	n = il + i;
	d = data;
	v = var;
	lcut = 0.5*aval;
	hcut = 1.5*aval;
	if (ff_meanvarcut(d,v,n,lcut,hcut,&mean,&meanv,&sig) != 0) {
	    ybuf[i] = aval;
	    ybufv[i] = 0.0;
	} else {
	    lcut = mean - 2.0*sig;
	    hcut = mean + 2.0*sig;
	    if (ff_meanvarcut(d,v,n,lcut,hcut,&mean,&meanv,&sig) != 0) {
		ybuf[i] = aval;
		ybufv[i] = 0.0;
	    } else {
		ybuf[i] = mean;
		ybufv[i] = meanv;
	    }
	}
    }
    for (i = il; i < nx-il; i++) {
	d = data + i - il;
	v = var + i - il;
	lcut = 0.5*aval;
	hcut = 1.5*aval;
	if (ff_meanvarcut(d,v,nfilt,lcut,hcut,&mean,&meanv,&sig) != 0) {
	    ybuf[i] = aval;
	    ybufv[i] = 0.0;
	} else {
	    lcut = mean - 2.0*sig;
	    hcut = mean + 2.0*sig;
	    if (ff_meanvarcut(d,v,nfilt,lcut,hcut,&mean,&meanv,&sig) != 0) {
		ybuf[i] = aval;
		ybufv[i] = 0.0;
	    } else {
		ybuf[i] = mean;
		ybufv[i] = meanv;
	    }
	}
    }
    n = nfilt;
    for (i = nx-il; i < nx; i++) {
	n--;
	d = data + nx - n;	
	v = var + nx - n;
	lcut = 0.5*aval;
	hcut = 1.5*aval;
	if (ff_meanvarcut(d,v,n,lcut,hcut,&mean,&meanv,&sig) != 0) {
	    ybuf[i] = aval;
	    ybufv[i] = 0.0;
	} else {
	    lcut = mean - 2.0*sig;
	    hcut = mean + 2.0*sig;
	    if (ff_meanvarcut(d,v,n,lcut,hcut,&mean,&meanv,&sig) != 0) {
		ybuf[i] = aval;
		ybufv[i] = 0.0;
	    } else {
		ybuf[i] = mean;
		ybufv[i] = meanv;
	    }
	}
    }    
    memcpy(data,ybuf,nx*sizeof(float));
    memcpy(var,ybufv,nx*sizeof(float));
    cpl_free(ybuf);
    cpl_free(ybufv);
}

/**@}*/

/* 

$Log$
Revision 1.22  20210721  mji
fixed bug in meanvarcut routine and made ff_filt1d more robust 

Revision 1.21  20210406  mji
fixed fibre scaling bug introduced by writing info to PHU in wrong place

Revision 1.20  20191014  mji
modified threshold no. of spectra needed to be scalable 

Revision 1.19  2019/03/21 11:47:46  jrl
modified call to _flatblk

Revision 1.18  2019/02/25 10:36:38  jrl
New memory allocation scheme. Modified smoothing

Revision 1.17  2018/11/29 12:42:59  jrl
Adds name of fibre flat to header of corrected file as FFLATCOR

Revision 1.16  2018/10/12 10:04:49  jrl
Modified to try and ensure the snr in the flats remain the same even when
they are modified

Revision 1.15  2018/09/19 11:26:24  jrl
Fixed bug where OBFBFLAT is written as a float rather than a string

Revision 1.14  2018/08/01 10:41:07  jim
A little change to make the calculation of the mean profile a bit more robust

Revision 1.13  2018/08/01 09:09:47  jim
Fixed problem in meansigcut that caused NaNs

Revision 1.12  2018/07/15 14:33:34  jim
added obfibcor routine

Revision 1.11  2018/06/27 09:42:57  jim
Added smoothing option

Revision 1.10  2018/01/16 10:13:54  jim
Modified _div module so that it handles zero in flat fields correctly

Revision 1.9  2017/12/20 16:45:01  jim
Normalisation now done with wavelength space

Revision 1.8  2017/10/05 09:09:04  jim
Added median and normalised levels to fibinfo table of normalised flat
output file

Revision 1.7  2017/05/23 08:55:53  jim
Modified both routines to take into account the fact that we shouldn't be doing
anything to or with bad pixels (i.e. those who have zero for variance)

Revision 1.6  2017/05/22 11:19:24  jim
Fixed to deal with possible missing fibtable

Revision 1.5  2017/01/17 09:00:00  jim
Modified to add QC info to the header of the normalised flat

Revision 1.4  2016/10/23 15:55:38  jim
Added docs

Revision 1.3  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.2  2016/07/06 11:03:41  jim
Modified to change the way input file are specified

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry


*/
