/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

#include "qmost_constants.h"
#include "qmost_imcombine_lite.h"
#include "qmost_sort.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_imcombine_lite  qmost_imcombine_lite
 *
 * 2D image combination.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_imcombine_lite.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/* Structure to store pointers and metadata for a single image. */

typedef struct {
    cpl_image *img;
    cpl_image *var;
    int      has_var;
    long     nx;
    long     ny;
    long     npts;
    float    *data;
    float    *vdata;
    long     ndata;
    float    exptime;
    float    expadj;
    float    skyaverage;
    float    skynoise;
    float    skyadj;
    float    skyrenorm;
    int      nrej;
    int      nrejx;
} litestrct;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void medcalc_lite (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    float avskynoise,
    int scaletype);

static void meancalc_lite (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    float avskynoise,
    int scaletype);

static void xclip_med (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    int scaletype);

static void xclip_mean (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    int scaletype);

static cpl_error_code qmost_imcombine_skylevel (
    cpl_image *in_image,
    int minlev,
    int maxlev,
    int fibre,
    float *skylev,
    float *sigma);

/*----------------------------------------------------------------------------*/
/**
 * @brief   General purpose 2D image combination routine.
 *
 * A list of input images are combined to form an output image.
 * Offsets in data values are calculated automatically and the input
 * data are biased or scaled to bring them onto a common background
 * level depending on the value of the parameter scaletype.
 *
 * Image combination can be done using clipped mean or median
 * depending on the value of the parameter combtype, with the
 * threshold specified by the parameter thresh in units of the
 * background sigma.  The appropriate sigma value to use during
 * clipping is computed from a robust estimator of the image
 * background based on histogramming and combined with the variance
 * from the given list of variance images if supplied, using the
 * larger of the the background noise or the square root of the input
 * pixel variance for the clipping threshold.
 *
 * An extra rejection cycle using the noise level estimated
 * individually in each pixel from the scatter of the multiple
 * exposures can optionally also be applied.
 *
 * The images are assumed to be aligned in pixel space and there is no
 * support for resampling.
 *
 * @param   in_imglist       (Given)    An imagelist of images to be
 *                                      combined.  The data type must
 *                                      be CPL_TYPE_FLOAT.
 * @param   in_varlist       (Given)    An optional imagelist giving
 *                                      the corresponding variance
 *                                      arrays for each input image,
 *                                      or NULL if none.  The data
 *                                      type be CPL_TYPE_FLOAT.
 * @param   in_explist       (Given)    The exposure time for each
 *                                      image when using scaletype=3.
 *                                      Ignored otherwise and can be
 *                                      NULL if not using this scaling
 *                                      method.
 * @param   combtype         (Given)    QMOST_MEANCALC: the output
 *                                      pixel will be the mean of the
 *                                      input pixels.
 *                                      QMOST_MEDIANCALC: the output
 *                                      pixel will be the median of
 *                                      the input pixels.
 * @param   scaletype        (Given)    0: no biasing or scaling is
 *                                      done.  1: input images are
 *                                      biased by the relative offset
 *                                      of their background values
 *                                      before combining.  2: input
 *                                      images are scaled by the
 *                                      relative ratios of their
 *                                      background values before
 *                                      combining.  3: input images
 *                                      are scaled by their exposure
 *                                      times (for dark frames).
 * @param   fibre            (Given)    If true, treat as fibre
 *                                      spectra where the majority of
 *                                      pixels aren't illuminated.
 * @param   xrej             (Given)    If true, then an extra
 *                                      rejection cycle is performed.
 * @param   thresh           (Given)    The threshold level in terms
 *                                      of background noise for
 *                                      rejection.
 * @param   checkexp         (Given)    If true, check median count
 *                                      level on image.
 * @param   underexp         (Given)    A minimum count level in ADU
 *                                      to reject underexposed data.
 * @param   overexp          (Given)    A maximum count level in ADU
 *                                      to reject overexposed data.
 * @param   out_img          (Returned) The resulting combined image.
 * @param   out_var          (Returned) The corresponding variance
 *                                      array, or NULL if not needed.
 * @param   out_hdr          (Modified) A caller allocated
 *                                      propertylist to receive QC
 *                                      header information, or NULL if
 *                                      not needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If there were no input
 *                                        images.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the combtype parameter
 *                                        value is invalid.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the image or variance
 *                                        dimensions don't match.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If the input image or
 *                                        variance data types aren't
 *                                        float.
 *
 * @par Output QC Parameters:
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean background level of the
 *     framesthat were combined.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *
 * @note    The variance calculation is only done if both the input
 *          imagelist (in_varlist) and output image (out_var) are
 *          given.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_imcombine_lite (
    cpl_imagelist *in_imglist,
    cpl_imagelist *in_varlist,
    double *in_explist,
    int combtype,
    int scaletype,
    int fibre,
    int xrej,
    float thresh,
    int checkexp,
    float underexp,
    float overexp,
    cpl_image **out_img,
    cpl_image **out_var,
    cpl_propertylist *out_hdr)
{
    int ifile, nfiles = 0, nfiles_var, ngood;

    litestrct *fileptrs = NULL;
    litestrct *ff;

    litestrct ofbuf;
    litestrct *ofptr = NULL;

    float *all_skylevel = NULL;
    float *all_skynoise = NULL;

    long nxvar, nyvar;
    float skylevel, skynoise, sumsky, sumsig;

    float *work = NULL;

    cpl_image *tmp_img = NULL;
    int ipix;

    cpl_errorstate prestate;
    float min_skylevel, max_skylevel, mean_skylevel, rms_skylevel;
    float mean_skynoise, rms_skynoise;

    cpl_ensure_code(in_imglist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_img != NULL, CPL_ERROR_NULL_INPUT);

    *out_img = NULL;

    if(out_var != NULL) {
        *out_var = NULL;
    }

#undef TIDY
#define TIDY                                    \
    if(fileptrs != NULL) {                      \
        cpl_free(fileptrs);                     \
        fileptrs = NULL;                        \
    }                                           \
    if(all_skylevel != NULL) {                  \
        cpl_free(all_skylevel);                 \
        all_skylevel = NULL;                    \
    }                                           \
    if(all_skynoise != NULL) {                  \
        cpl_free(all_skynoise);                 \
        all_skynoise = NULL;                    \
    }                                           \
    if(*out_img != NULL) {                      \
        cpl_image_delete(*out_img);             \
        *out_img = NULL;                        \
    }                                           \
    if(out_var != NULL && *out_var != NULL) {   \
        cpl_image_delete(*out_var);             \
        *out_var = NULL;                        \
    }                                           \
    if(work != NULL) {                          \
        cpl_free(work);                         \
        work = NULL;                            \
    }                                           \
    if(tmp_img != NULL) {                       \
        cpl_image_delete(tmp_img);              \
        tmp_img = NULL;                         \
    }

    /* Check the number of images */
    nfiles = cpl_imagelist_get_size(in_imglist);
    if(nfiles < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "there were no input images");
    }

    if(in_varlist != NULL) {
        nfiles_var = cpl_imagelist_get_size(in_varlist);
        if(nfiles != nfiles_var) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image and variance list sizes don't "
                                         "match: %d != %d",
                                         nfiles, nfiles_var);
        }
    }

    /* Check we have needed exptime argument if using it */
    if(in_explist == NULL && scaletype == 3) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                                     "exposure times must be given "
                                     "when scaling by exposure time");
    }

    /* Get some file structures and storage for QC */
    fileptrs = cpl_calloc(nfiles,sizeof(litestrct));
    all_skylevel = cpl_calloc(nfiles,sizeof(float));
    all_skynoise = cpl_calloc(nfiles,sizeof(float));

    /* Fill in information for each input */
    sumsky = 0.0;
    sumsig = 0.0;
    ngood = 0;
    for (ifile = 0; ifile < nfiles; ifile++) {
        ff = fileptrs + ngood;

        if(in_varlist != NULL && out_var != NULL) {
            ff->has_var = 1;
        } else {
            ff->has_var = 0;
        }

        if(in_explist != NULL) {
            ff->exptime = in_explist[ifile];
        }
        else {
            ff->exptime = 1.0;
        }

	ff->expadj = fileptrs[0].exptime / ff->exptime;

        /* Get image */
        ff->img = cpl_imagelist_get(in_imglist, ifile);
        if(ff->img == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't fetch image %d",
                                         ifile+1);
        }

        /* Get the image parameters and check them */
        ff->nx = cpl_image_get_size_x(ff->img);
        ff->ny = cpl_image_get_size_y(ff->img);
        if(ff->nx != fileptrs[0].nx ||
           ff->ny != fileptrs[0].ny) {
            cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                  "image %d size doesn't match: "
                                  "(%ld, %ld) != (%ld, %ld)",
                                  ifile+1,
                                  ff->nx, ff->ny,
                                  fileptrs[0].nx,
                                  fileptrs[0].ny);
            TIDY;
            return cpl_error_get_code();
        }

	ff->npts = ff->nx * ff->ny;

        /* Get pointer */
        ff->data = cpl_image_get_data_float(ff->img);
        if(ff->data == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "input image %d",
                                         ifile+1);
        }

        /* Get background level and noise */
        skynoise = -1.0;

        if(qmost_imcombine_skylevel(ff->img, -1000, 65535, fibre,
				    &skylevel,
				    &skynoise) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not compute background "
                                         "for input image %d",
                                         ifile+1);
        }

        /* Check exposure if we're doing that */
        if(checkexp) {
            if(skylevel < underexp || skylevel >= overexp) {
                cpl_msg_warning(cpl_func,
                                "File %d median counts/pixel = %.0f, BAD",
                                ifile+1,
                                skylevel);
                continue;
            }
            else {
                cpl_msg_debug(cpl_func,
                              "File %d median counts/pixel = %.0f, OK",
                              ifile+1,
                              skylevel);
            }
        }

        /* Scale by exposure time if we're doing that */
        if(scaletype == 3) {
            skylevel *= ff->expadj;
            skynoise *= ff->expadj;
        }

        /* Accumulate statistics for QC */
        all_skylevel[ifile] = skylevel;
        all_skynoise[ifile] = skynoise;

	/* If the skynoise is negative, then this frame was all nulls */
	if(skynoise < 0.0) {
	    continue;
	}

	/* Otherwise keep going */
	ngood++;
	ff->skyaverage = skylevel;
	ff->skynoise = skynoise;
	sumsky += skylevel;
	sumsig += skynoise;

	/* Get variance */
        if(ff->has_var) {
            ff->var = cpl_imagelist_get(in_varlist, ifile);
            if(ff->var == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't fetch variance "
                                             "array %d",
                                             ifile+1);
            }

            /* Check size */
            nxvar = cpl_image_get_size_x(ff->var);
            nyvar = cpl_image_get_size_y(ff->var);
            if(nxvar != ff->nx ||
               nyvar != ff->ny) {
                cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                      "variance %d size doesn't match image: "
                                      "(%ld, %ld) != (%ld, %ld)",
                                      ifile+1,
                                      nxvar, nyvar,
                                      ff->nx, ff->ny);
                TIDY;
                return cpl_error_get_code();
            }

            /* Get pointer */
            ff->vdata = cpl_image_get_data_float(ff->var);
            if(ff->vdata == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get float pointer to "
                                             "input variance array %d",
                                             ifile+1);
            }
        }
        else {
            ff->vdata = NULL;
        }
    }

    if(ngood <= 0) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "there were no good exposures");
    }

    /* Create output */
    *out_img = cpl_image_new(fileptrs[0].nx, fileptrs[0].ny, CPL_TYPE_FLOAT);
    if(*out_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output image");
    }

    ofptr = &ofbuf;
    memset(ofptr, 0, sizeof(ofbuf));

    ofptr->nx = fileptrs[0].nx;
    ofptr->ny = fileptrs[0].ny;
    ofptr->npts = fileptrs[0].npts;
    ofptr->data = cpl_image_get_data_float(*out_img);
    if(ofptr->data == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer to "
                                     "output image array");
    }

    if(fileptrs[0].has_var) {
        *out_var = cpl_image_new(fileptrs[0].nx, fileptrs[0].ny,
                                 CPL_TYPE_FLOAT);
        if(*out_var == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not create output "
                                         "variance array");
        }

        ofptr->vdata = cpl_image_get_data_float(*out_var);
        if(ofptr->vdata == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get float pointer to "
                                         "output variance array");
        }

        ofptr->has_var = 1;
    }
    else {
        ofptr->vdata = NULL;
        ofptr->has_var = 0;
    }
        
    /* If there is only one left then just move the data across */
    if(ngood == 1) {
        memcpy(ofptr->data, fileptrs[0].data,
               ofptr->npts * sizeof(float));
        if(ofptr->has_var) {
            memcpy(ofptr->vdata, fileptrs[0].vdata,
                   ofptr->npts * sizeof(float));
        }
    }
    else {
        /* Initialize to zero */
        memset(ofptr->data, 0, ofptr->npts * sizeof(float));
        if(ofptr->has_var) {
            memset(ofptr->vdata, 0, ofptr->npts * sizeof(float));
        }
    }

    /* If all the input frames were zero then the value of 'sumsky' should 
       also be zero. In that case, don't do any averaging. Just write out
       the data as an array of zeros. */

    if(sumsky != 0.0 || ngood > 1) {
	/* Work out median background and noise. Then create background 
	   zeropoint or scale factor, depending upon which was requested 
	   in the call. */
	work = cpl_malloc(ngood * sizeof(float));

	for(ifile = 0; ifile < ngood; ifile++) {
	    work[ifile] = fileptrs[ifile].skyaverage;
        }

        prestate = cpl_errorstate_get();

	if(qmost_med(work,NULL,ngood,&sumsky) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            sumsky = 0;
        }

	for(ifile = 0; ifile < ngood; ifile++) {
	    work[ifile] = fileptrs[ifile].skynoise;
        }

	if(qmost_med(work,NULL,ngood,&sumsig) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            sumsig = 0;
        }

        cpl_free(work);
        work = NULL;

	switch (scaletype) {
	case 1:
	    for(ifile = 0; ifile < ngood; ifile++) {
		fileptrs[ifile].skyadj = sumsky - fileptrs[ifile].skyaverage;
            }
	    break;
	case 2:
	    for(ifile = 0; ifile < ngood; ifile++) {
		if(fileptrs[ifile].skyaverage != 0.0) {
		    fileptrs[ifile].skyadj = (sumsky /
                                                fileptrs[ifile].skyaverage);
                }
		else {
		    fileptrs[ifile].skyadj = 1.0;
                }
	    }
	    break;	    
	case 3:
	    for(ifile = 0; ifile < ngood; ifile++) {
		fileptrs[ifile].skyadj = sumsky - fileptrs[ifile].skyaverage;
            }
	    break;
	default:
	    for(ifile = 0; ifile < ngood; ifile++) {
		fileptrs[ifile].skyadj = 0.0;
            }
	    break;
	}

	/* Now do the averaging/medianing */
        switch(combtype) {
        case QMOST_MEDIANCALC:
	    medcalc_lite(ngood,fileptrs,ofptr,thresh,sumsig,scaletype);
            break;
        case QMOST_MEANCALC:
	    meancalc_lite(ngood,fileptrs,ofptr,thresh,sumsig,scaletype);
            break;
        default:
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "unrecognised averaging type: %d",
                                         combtype);
        }

	/* Do the extra clipping if requested */
	if(xrej) {
	    /* First get sky background and sigma from output data */
            skynoise = -1.0;

            if(qmost_imcombine_skylevel(*out_img, -1000, 65535, fibre,
					&skylevel,
					&skynoise) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not compute background "
                                             "for output image");
            }

	    ofptr->skyaverage = skylevel;
	    ofptr->skynoise = skynoise;

	    /* Rescale the normalising factor and work out a new estimate
  	       of the background noise */
	    for(ifile = 0; ifile < ngood; ifile++) {
		ff = fileptrs + ifile;

		ff->skyrenorm = ff->skyaverage / ofptr->skyaverage;

                tmp_img = cpl_image_duplicate(ff->img);
                work = cpl_image_get_data_float(tmp_img);

		switch (scaletype) {
		case 1:
		    for(ipix = 0; ipix < ff->npts; ipix++) {
			work[ipix] -= ofptr->data[ipix] - ofptr->skyaverage;
                    }
		    break;
		case 2:
		    for(ipix = 0; ipix < ff->npts; ipix++) {
 		        work[ipix] -=  ff->skyrenorm*(ofptr->data[ipix] -
                                                      ofptr->skyaverage);
                    }
		    break;
		case 3:
		    for(ipix = 0; ipix < ff->npts; ipix++) {
			work[ipix] -= ofptr->data[ipix] - ofptr->skyaverage;
                    }
		    break;
	        case 0:
		    for(ipix = 0; ipix < ff->npts; ipix++) {
			work[ipix] -= ofptr->data[ipix] - ofptr->skyaverage;
                    }
		    break;
		}

                if(qmost_imcombine_skylevel(tmp_img, -65536, 65535, fibre,
					    &skylevel,
					    &skynoise) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "could not compute "
                                                 "background for "
                                                 "temporary image");
                }

		ff->skynoise = skynoise;

                work = NULL;
                cpl_image_delete(tmp_img);
                tmp_img = NULL;
	    }

	    /* Now do the extra clip... */

	    switch(combtype) {
	    case QMOST_MEDIANCALC:
		xclip_med(ngood,fileptrs,ofptr,thresh,scaletype);
		break;
	    case QMOST_MEANCALC:
		xclip_mean(ngood,fileptrs,ofptr,thresh,scaletype);
		break;
            default:
                break;
	    }
	}
    }

    /* Populate QC */
    if(out_hdr != NULL) {
        min_skylevel = all_skylevel[0];
        max_skylevel = all_skylevel[0];

        for(ifile = 1; ifile < nfiles; ifile++) {
            if(all_skylevel[ifile] < min_skylevel) {
                min_skylevel = all_skylevel[ifile];
            }
            if(all_skylevel[ifile] > max_skylevel) {
                max_skylevel = all_skylevel[ifile];
            }
        }

        prestate = cpl_errorstate_get();

        if(qmost_meansig(all_skylevel, NULL, nfiles,
                         &mean_skylevel, &rms_skylevel) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            mean_skylevel = 0;
            rms_skylevel = 0;
        }

        if(qmost_meansig(all_skynoise, NULL, nfiles,
                         &mean_skynoise, &rms_skynoise) != CPL_ERROR_NONE) {
            cpl_errorstate_set(prestate);
            mean_skynoise = 0;
            rms_skynoise = 0;
        }

        cpl_propertylist_update_int(out_hdr,
                                    "ESO QC IMCOMBINE NUM INPUTS",
                                    nfiles);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE NUM INPUTS",
                                     "Number of input files to combine");

        cpl_propertylist_update_int(out_hdr,
                                    "ESO QC IMCOMBINE NUM COMBINED",
                                    ngood);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE NUM COMBINED",
                                     "Number of files combined");

        cpl_propertylist_update_float(out_hdr,
                                      "ESO QC IMCOMBINE MEAN",
                                      mean_skylevel);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE MEAN",
                                     "[ADU] Mean background in input images");

        cpl_propertylist_update_float(out_hdr,
                                      "ESO QC IMCOMBINE RMS",
                                      rms_skylevel);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE RMS",
                                     "[ADU] RMS background in input images");

        cpl_propertylist_update_float(out_hdr,
                                      "ESO QC IMCOMBINE MIN",
                                      min_skylevel);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE MIN",
                                     "[ADU] Minimum background in input images");

        cpl_propertylist_update_float(out_hdr,
                                      "ESO QC IMCOMBINE MAX",
                                      max_skylevel);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE MAX",
                                     "[ADU] Maximum background in input images");

        cpl_propertylist_update_float(out_hdr,
                                      "ESO QC IMCOMBINE NOISE MEAN",
                                      mean_skynoise);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE NOISE MEAN",
                                     "[ADU] Mean background noise in input images");

        cpl_propertylist_update_int(out_hdr,
                                    "ESO QC IMCOMBINE NUM REJECTED",
                                    ofptr->nrej);
        cpl_propertylist_set_comment(out_hdr,
                                     "ESO QC IMCOMBINE NUM REJECTED",
                                     "Total pixels rejected");
    }

    cpl_free(fileptrs);
    fileptrs = NULL;

    cpl_free(all_skylevel);
    all_skylevel = NULL;

    cpl_free(all_skynoise);
    all_skynoise = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate the median combination.
 *
 * @param   nf               (Given)    The number of input files.
 * @param   fileptrs         (Given)    The list of structures with
 *                                      information about the input
 *                                      files.
 * @param   ofptr            (Modified) The structure with the
 *                                      information about the output
 *                                      file.
 * @param   thresh           (Given)    The rejection threshold in
 *                                      units of sky noise.
 * @param   avskynoise       (Given)    The average sky noise.
 * @param   scaletype        (Given)    The type of biassing/scaling
 *                                      being done between images.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void medcalc_lite (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    float avskynoise,
    int scaletype)
{
    int nf1,nf2,nfm,nrejmax,is_even,nrej,nremain,nm,nmm,is_even2,k;
    int is_small,dovar,ntrej,nrejr,nok;
    long ix,iy,i,i2;
    float value,value2,pixnoise;
    float *work = NULL;
    float *workv = NULL;
    
    /* Set up a few useful variables */

    dovar = fileptrs->has_var;
    is_small = (nf <= 5);
    is_even = ! (nf & 1);
    nf1 = nf/2 - 1;
    nf2 = nf1 + 1;
    nfm = (nf+1)/2 - 1;
    if (is_small) 
	nrejmax = 1;
    else
	nrejmax = nf/2;

    /* Ok, loop for each pixel... */
    ntrej = 0;

#pragma omp parallel reduction(+:ntrej) default(none) private(work, workv, ix, i2, i, nrejr, k, value, nrej, nok, pixnoise, nremain, nm, nmm, is_even2, value2) shared(nf, fileptrs, ofptr, thresh, avskynoise, scaletype, dovar, is_small, is_even, nf1, nf2, nfm, nrejmax)
    {
        /* Get a workspace */
        work = cpl_malloc(nf*sizeof(float));
        if (dovar) {
            workv = cpl_malloc(nf*sizeof(float));
        }
        else {
            workv = NULL;
        }

#pragma omp for
        for(iy = 0; iy < fileptrs->ny; iy++) {
            nrejr = 0;

            for(ix = 0; ix < fileptrs->nx; ix++) {
                i2 = iy * fileptrs->nx + ix;
                i = i2;
	    
                /* Scale or shift data */

                switch (scaletype) {
                case 0:
                    for (k = 0; k < nf; k++) {
                        work[k] = ((fileptrs+k)->data)[i];
                        if (dovar) 
                            workv[k] = ((fileptrs+k)->vdata)[i];
                    }
                    break;
                case 1:
                    for (k = 0; k < nf; k++) { 
                        work[k] = ((fileptrs+k)->data)[i] + (fileptrs+k)->skyadj;
                        if (dovar) 
                            workv[k] = ((fileptrs+k)->vdata)[i];
                    }
                    break;
                case 2:
                    for (k = 0; k < nf; k++) {
                        work[k] = (((fileptrs+k)->data)[i])*(fileptrs+k)->skyadj;
                        if (dovar) 
                            workv[k] = (((fileptrs+k)->vdata)[i])*pow((fileptrs+k)->skyadj,2.0);
                    }
                    break;
                case 3:
                    for (k = 0; k < nf; k++) {
                        work[k] = ((fileptrs+k)->expadj * (fileptrs+k)->data[i] +
                                   (fileptrs+k)->skyadj);
                        if (dovar) 
                            workv[k] = ((fileptrs+k)->expadj *
                                        (fileptrs+k)->expadj *
                                        (fileptrs+k)->vdata[i]);
                    }
                    break;
                }

                /* Sort data and get the median */

                if (dovar) 
                    qmost_sort_ff(work,workv,nf);
                else
                    qmost_sort_f(work,nf);
                if (is_even)
                    value = 0.5*(work[nf1] + work[nf2]);
                else 
                    if (is_small) 
                        value = work[nfm];
                    else 
                        value = 0.25*(work[nfm-1] + work[nfm+1]) + 0.5*work[nfm];

                /* Enter a rejection loop  */

                nrej = 0;
                /* cliplev1 = value - thresh*avskynoise; 
                   cliplev2 = value + thresh*avskynoise;
                   while (nrej < nrejmax && work[nf-nrej-1] > cliplev2)
                   nrej++; */
                nok = 0;
                while (nrej < nrejmax && nok == 0) {
                    if(dovar) {
                        pixnoise = qmost_max(avskynoise,sqrt(qmost_max(1.0,workv[nf-nrej-1])));
                    } else {
                        pixnoise = avskynoise;
                    }
                    if (work[nf-nrej-1] > value + thresh*pixnoise)
                        nrej++;
                    else
                        nok = 1;
                }

                /* If you've clipped any, then recalculate the median...*/

                if (nrej > 0) {
                    nremain = nf - nrej;
                    nm = nremain/2 - 1;
                    nmm = (nremain + 1)/2 - 1;
                    is_even2 = !(nremain & 1);
                    if (is_even2) 
                        value = 0.5*(work[nm] + work[nm+1]);
                    else 
                        if (nremain <= 3) 
                            value = work[nmm];
                        else 
                            value = 0.5*work[nmm] + 0.25*(work[nmm-1] + work[nmm+1]);
                }

                /* Now the variance */

                if (dovar) {
                    nremain = nf - nrej;
                    value2 = 0.0;
                    for (i = 0; i < nremain; i++)
                        value2 += workv[i];
                    value2 = 0.5*CPL_MATH_PI*value2/powf((float)nremain,2.0);
                }

                /* Store the result away */

                ofptr->data[i2] = value;
                if (dovar)
                    ofptr->vdata[i2] = value2;

                nrejr += nrej;
            }

            ntrej += nrejr;
        }

        /* Get rid of workspace */
        
        cpl_free(work);
        work = NULL;

        if (dovar) {
            cpl_free(workv);
            workv = NULL;
        }
    }

    ofptr->nrej = ntrej;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate the mean combination.
 *
 * @param   nf               (Given)    The number of input files.
 * @param   fileptrs         (Given)    The list of structures with
 *                                      information about the input
 *                                      files.
 * @param   ofptr            (Modified) The structure with the
 *                                      information about the output
 *                                      file.
 * @param   thresh           (Given)    The rejection threshold in
 *                                      units of sky noise.
 * @param   avskynoise       (Given)    The average sky noise.
 * @param   scaletype        (Given)    The type of biassing/scaling
 *                                      being done between images.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void meancalc_lite (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    float avskynoise,
    int scaletype)
{
    int nf2,k,nrejmax,kk,krem,dovar,nrej,nrejr,ntrej;
    long i,i2,ix,iy;
    float value,value2,resid,maxresid,pixnoise;
    float *work = NULL;
    float *workv = NULL;
    unsigned char *iflag = NULL;

    dovar = fileptrs->has_var;

    /* Ok, loop for each pixel... */
    nrejmax = nf - 1;
    ntrej = 0;

#pragma omp parallel reduction(+:ntrej) default(none) private(work, workv, iflag, nrejr, ix, i2, i, k, value, nrej, kk, maxresid, krem, resid, pixnoise, value2, nf2) shared(nf, dovar, nrejmax, fileptrs, ofptr, thresh, avskynoise, scaletype)
    {
        /* Get a workspace */
        work = cpl_malloc(nf*sizeof(float));
        if (dovar) {
            workv = cpl_malloc(nf*sizeof(float));
        }
        else {
            workv = NULL;
        }
        iflag = cpl_malloc(nf*sizeof(unsigned char));

#pragma omp for
        for(iy = 0; iy < fileptrs->ny; iy++) {
            nrejr = 0;

            for(ix = 0; ix < fileptrs->nx; ix++) {
                i2 = iy * fileptrs->nx + ix;
                i = i2;

                /* Scale or shift data */

                switch (scaletype) {
                case 0:
                    for (k = 0; k < nf; k++) {
                        work[k] = ((fileptrs+k)->data)[i];
                        if (dovar) 
                            workv[k] = ((fileptrs+k)->vdata)[i];
                    }
                    break;
                case 1:
                    for (k = 0; k < nf; k++) {
                        work[k] = ((fileptrs+k)->data)[i] + (fileptrs+k)->skyadj;
                        if (dovar) 
                            workv[k] = ((fileptrs+k)->vdata)[i];
                    }
                    break;
                case 2:
                    for (k = 0; k < nf; k++) {
                        work[k] = (((fileptrs+k)->data)[i])*(fileptrs+k)->skyadj;
                        if (dovar) 
                            workv[k] = (((fileptrs+k)->vdata)[i])*pow((fileptrs+k)->skyadj,2.0);
                    }
                    break;
                case 3:
                    for (k = 0; k < nf; k++) {
                        work[k] = ((fileptrs+k)->expadj * (fileptrs+k)->data[i] +
                                   (fileptrs+k)->skyadj);
                        if (dovar) 
                            workv[k] = ((fileptrs+k)->expadj *
                                        (fileptrs+k)->expadj *
                                        (fileptrs+k)->vdata[i]);
                    }
                    break;
                }
	
                /* Get the mean */

                value = 0.0;
                for (k = 0; k < nf; k++) {
                    value += work[k];
                    iflag[k] = 0;
                }
                value /= (float)nf;

                /* Enter a rejection loop. Reject pixels one at a time. If there
                   are only two, then chose the positive one since it's likely to be
                   a cosmic ray*/

                nrej = 0;

                for (kk = 0; kk < nrejmax; kk++) {
                    maxresid = -1.0e30;
                    krem = -1;
                    for (k = 0; k < nf; k++) {
                        if (iflag[k] == 1)
                            continue;
                        resid = fabs(work[k] - value);
                        if (dovar) {
                            pixnoise = qmost_max(avskynoise,sqrt(qmost_max(1.0,workv[k])));
                        } else {
                            pixnoise = avskynoise;
                        }
                        if (resid > thresh*pixnoise) {
                            if (nf <= 2) 
                                resid = work[k] - value;
                            if (resid > maxresid) {
                                krem = k;
                                maxresid = resid;
                            }
                        }
                    } 
                    if (krem == -1)
                        break;
                    value2 = 0.0;
                    iflag[krem] = 1;
                    nrej++;
                    nf2 = 0;
                    for (k = 0; k < nf; k++) {
                        if (iflag[k] == 0) {
                            value2 += work[k];
                            nf2++;
                        }
                    }
                    value = value2/(float)nf2;
                }

                /* Store the result away */

                ofptr->data[i2] = value;

                /* And the variance... */

                if (dovar) {
                    value2 = 0.0;
                    nf2 = 0;
                    for (k = 0; k < nf; k++) {
                        if (iflag[k] == 0) {
                            value2 += workv[k];
                            nf2++;
                        }
                    }
                    value2 /= powf((float)nf2,2.0);
                    ofptr->vdata[i2] = value2;
                }

                nrejr += nrej;
            }

            ntrej += nrejr;
        }

        /* Get rid of workspace */
        cpl_free(work);
        work = NULL;

        if(workv != NULL) {
            cpl_free(workv);
            workv = NULL;
        }

        cpl_free(iflag);
        iflag = NULL;
    }

    ofptr->nrej = ntrej;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do the extra clipping pass for median combinations.
 *
 * @param   nf               (Given)    The number of input files.
 * @param   fileptrs         (Given)    The list of structures with
 *                                      information about the input
 *                                      files.
 * @param   ofptr            (Modified) The structure with the
 *                                      information about the output
 *                                      file.
 * @param   thresh           (Given)    The rejection threshold in
 *                                      units of sky noise.
 * @param   scaletype        (Given)    The type of biassing/scaling
 *                                      being done between images.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void xclip_med (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    int scaletype)
{
    int nf1,nf2,nfm,nrejmax,is_even,k,is_even2,nrej,nremain,nm,nmm,dovar;
    int nrejr, ntrej;
    float value,value2,cliplev2,doff;
    long i,i2,ix,iy;
    litestrct *ff=NULL;
    float *work = NULL;
    float *work2 = NULL;
    float *nork = NULL;
    float *dork = NULL;

    /* Set up a few useful variables */

    dovar = fileptrs->has_var;
    nf1 = nf/2 - 1;
    nf2 = nf1 + 1;
    nfm = (nf + 1)/2 - 1;
    nrejmax = nf/2;
    is_even = !(nf & 1);

    /* Loop for each input pixel now... */
    ntrej = 0;

#pragma omp parallel reduction(+:ntrej) default(none) private(work, nork, dork, work2, nrejr, ix, i2, i, doff, k, ff, value, value2, cliplev2, nrej, nremain, nm, nmm, is_even2) shared(nf, fileptrs, ofptr, thresh, scaletype, dovar, nf1, nf2, nfm, nrejmax, is_even)
    {
        /* Get some workspace */
        work = cpl_malloc(nf*sizeof(float));
        nork = cpl_malloc(nf*sizeof(float));
        dork = cpl_malloc(nf*sizeof(float));
        
        if (dovar) {
            work2 = cpl_malloc(nf*sizeof(float));
        }
        else {
            work2 = NULL;
        }

#pragma omp for
        for(iy = 0; iy < fileptrs->ny; iy++) {
            nrejr = 0;

            for(ix = 0; ix < fileptrs->nx; ix++) {
                i2 = iy * fileptrs->nx + ix;
                i = i2;

                /* Scale or shift data. NB: We have to reassaign nork every time
                   as the array is cosorted for each pixel */

                switch (scaletype) {
                case 0:
                    doff = ofptr->data[i2] - ofptr->skyaverage;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = ff->data[i];
                        nork[k] = ff->skynoise;
                        work[k] = dork[k] - doff;
                        if (dovar)
                            work2[k] = ff->vdata[i];
                    }
                    break;
                case 1:
                    doff = ofptr->data[i2] - ofptr->skyaverage;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = ff->data[i] + ff->skyadj;
                        nork[k] = ff->skynoise;
                        work[k] = dork[k] - doff;
                        if (dovar)
                            work2[k] = ff->vdata[i];
                    }
                    break;
                case 2:
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        doff = (ofptr->data[i2]*ff->skyrenorm - 
                                ff->skyaverage)*ff->skyadj;
                        dork[k] = (ff->data[i])*ff->skyadj;
                        nork[k] = ff->skynoise*ff->skyadj;
                        work[k] = dork[k] - doff;
                        if (dovar)
                            work2[k] = ff->vdata[i]*powf(ff->skyadj,2.0);
                    }
                    break;
                case 3:
                    doff = ofptr->data[i2] - ofptr->skyaverage;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = ff->expadj * ff->data[i] + ff->skyadj;
                        nork[k] = ff->skynoise;
                        work[k] = dork[k] - doff;
                        if (dovar)
                            work2[k] = ff->expadj * ff->expadj * ff->vdata[i];
                    }
                    break;
                }	

                /* Sort and get a first pass median */

                if (dovar)
                    qmost_sort_ffff(work,nork,dork,work2,nf);
                else
                    qmost_sort_fff(work,nork,dork,nf);
                if (is_even)
                    value = 0.5*(work[nf1] + work[nf2]);
                else 
                    if (nf < 5) 
                        value = work[nfm];
                    else 
                        value = 0.25*(work[nfm-1] + work[nfm+1]) + 0.5*work[nfm];
                value2 = 0.0;
                if (dovar) {
                    for (i = 0; i < nf; i++)
                        value2 += work2[i];
                    value2 *= 0.5*CPL_MATH_PI/powf((float)nf,2.0);
                }
                /* Do upper clipping */

                /* cliplev1 = value - thresh*nork[nf-1]; */
                cliplev2 = value + thresh*nork[nf-1];
                nrej = 0;
                while (nrej < nrejmax && work[nf-nrej-1] > cliplev2) {
/* 	while (nrej < nrejmax &&  */
/* 	       (work[nf-nrej-1] > cliplev2 || work[nf-nrej-1] < cliplev1)) { */
                    nrej++;
                    /* cliplev1 = value - thresh*nork[nf-nrej-1]; */
                    cliplev2 = value + thresh*nork[nf-nrej-1];
                }

                /* If there were any clipped out, the re-estimate the value */

                if (nrej > 0) {
                    nremain = nf - nrej;
                    nm = nremain/2 - 1;
                    nmm = (nremain + 1)/2 - 1;
                    is_even2 = !(nremain & 1);
                    qmost_sort_f(dork,nm);
                    if (is_even2) 
                        value = 0.5*(dork[nm] + dork[nm+1]);
                    else 
                        if (nremain < 3) 
                            value = dork[nmm];
                        else 
                            value = 0.5*dork[nmm] + 0.25*(dork[nmm-1] + dork[nmm+1]);
                    if (dovar) {
                        value2 = 0.0;
                        for (i = 0; i < nremain; i++)
                            value2 += work2[i];
                        value2 *= 0.5*CPL_MATH_PI/powf((float)nf,2.0);
                    }
            
                    /* Store the result away */

                    ofptr->data[i2] = value;
                    if (dovar)
                        ofptr->vdata[i2] = value2;
                }

                nrejr += nrej;
            }

            ntrej += nrejr;
        }

        /* Ditch workspace */
        cpl_free(work);
        work = NULL;

        cpl_free(nork);
        nork = NULL;

        cpl_free(dork);
        dork = NULL;

        if(work2 != NULL) {
            cpl_free(work2);
            work2 = NULL;
        }
    }

    ofptr->nrejx = ntrej;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do the extra clipping pass for mean combinations.
 *
 * @param   nf               (Given)    The number of input files.
 * @param   fileptrs         (Given)    The list of structures with
 *                                      information about the input
 *                                      files.
 * @param   ofptr            (Modified) The structure with the
 *                                      information about the output
 *                                      file.
 * @param   thresh           (Given)    The rejection threshold in
 *                                      units of sky noise.
 * @param   scaletype        (Given)    The type of biassing/scaling
 *                                      being done between images.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void xclip_mean (
    int nf,
    litestrct *fileptrs,
    litestrct *ofptr,
    float thresh,
    int scaletype)
{
    int k,nf2,kk,krem,nrejmax,dovar,nrej,nrejr,ntrej;
    float value,value2,resid,maxresid,doff;
    long i,i2,ix,iy;
    litestrct *ff=NULL;
    float *nork = NULL;
    float *work = NULL;
    float *dork = NULL;
    float *work2 = NULL;
    unsigned char *iflag = NULL;

    dovar = fileptrs->has_var;
    nrejmax = nf/2;

    /* Get some workspace */
    nork = cpl_malloc(nf*sizeof(float));

    /* Set up the noise estimates for each frame. NB: scaletype=3 is scale
       by exposure time. But this has already been done earlier in the main
       routine. */

    switch(scaletype) {
    case 0:
	for (k = 0; k < nf; k++) {
	    ff = fileptrs + k;
	    nork[k] = ff->skynoise;
	}
	break;
    case 1:
	for (k = 0; k < nf; k++) {
	    ff = fileptrs + k;
	    nork[k] = ff->skynoise;
	}
	break;
    case 2:
	for (k = 0; k < nf; k++) {
	    ff = fileptrs + k;
	    nork[k] = ff->skynoise*ff->skyadj;
	}
	break;
    case 3:
	for (k = 0; k < nf; k++) {
	    ff = fileptrs + k;
	    nork[k] = ff->skynoise;
	}
	break;
    }
		
    /* Loop for each input pixel now... */
    ntrej = 0;

#pragma omp parallel reduction(+:ntrej) default(none) private(work, dork, work2, iflag, nrejr, ix, i2, i, doff, k, ff, value, value2, nrej, kk, maxresid, krem, resid, nf2) shared(nf, fileptrs, ofptr, thresh, scaletype, dovar, nrejmax, nork)
    {
        /* Get some workspace */
        work = cpl_malloc(nf*sizeof(float));
        dork = cpl_malloc(nf*sizeof(float));
        
        if (dovar) {
            work2 = cpl_malloc(nf*sizeof(float));
        }
        else {
            work2 = NULL;
        }
        
        iflag = cpl_malloc(nf*sizeof(unsigned char));

#pragma omp for
        for(iy = 0; iy < fileptrs->ny; iy++) {
            nrejr = 0;

            for(ix = 0; ix < fileptrs->nx; ix++) {
                i2 = iy * fileptrs->nx + ix;
                i = i2;

                /* Scale or shift data */

                switch (scaletype) {
                case 0:
                    doff = ofptr->data[i2] - ofptr->skyaverage;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = ff->data[i];
                        work[k] = dork[k] - doff;
                        iflag[k] = 0;
                        if (dovar)
                            work2[k] = ff->vdata[i];
                    }
                    break;
                case 1:
                    doff = ofptr->data[i2] - ofptr->skyaverage;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = ff->data[i] + ff->skyadj;
                        work[k] = dork[k] - doff;
                        iflag[k] = 0;
                        if (dovar)
                            work2[k] = ff->vdata[i];
                    }
                    break;
                case 2:
                    doff = (ofptr->data[i2]*ofptr->skyrenorm - ofptr->skyaverage)*ofptr->skyadj;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = (ff->data[i])*ff->skyadj;
                        work[k] = dork[k] - doff;
                        iflag[k] = 0;
                        if (dovar)
                            work2[k] = ff->vdata[i]*powf(ff->skyadj,2.0);
                    }
                    break;
                case 3:
                    doff = ofptr->data[i2] - ofptr->skyaverage;
                    for (k = 0; k < nf; k++) {
                        ff = fileptrs + k;
                        dork[k] = ff->expadj * ff->data[i] + ff->skyadj;
                        work[k] = dork[k] - doff;
                        iflag[k] = 0;
                        if (dovar)
                            work2[k] = ff->expadj * ff->expadj * ff->vdata[i];
                    }
                    break;
                }	

                /* Get a first pass mean */

                value = 0.0;
                for (k = 0; k < nf; k++) 
                    value += work[k];
                value /= (float)nf;
                value2 = 0.0;

                /* Enter a rejection loop. Reject pixels one at a time  */

                nrej = 0;
                for (kk = 0; kk < nrejmax; kk++) {
                    maxresid = -1.0e30;
                    krem = -1;
                    for (k = 0; k < nf; k++) {
                        if (iflag[k] == 1)
                            continue;
                        resid = fabs(work[k] - value);
                        if (resid > thresh*nork[k]) {
                            if (nf <= 2) 
                                resid = work[k] - value;
                            if (resid > maxresid) {
                                krem = k;
                                maxresid = resid;
                            }
                        }
                    }

                    if (krem == -1)
                        break;

                    /* Another pixel is rejected... */
            
                    iflag[krem] = 1;
                    nrej++;
            
                    nf2 = 0;
                    value = 0;
                    for (k = 0; k < nf; k++) {
                        if (iflag[k] == 0) {
                            value += dork[k];
                            nf2++;
                        }
                    }
                    if (nf2 != 0) {
                        value /= (float)nf2;
                        if (dovar) {
                            value2 = 0.0;
                            for (k = 0; k < nf; k++) {
                                if (iflag[k] == 0)
                                    value2 += work2[k];
                            }
                            value2 /= powf((float)nf2,2.0);
                        }
                    } else
                        break;
                }
                if (nrej > 0) {
                    ofptr->data[i] = value;
                    if (dovar)
                        ofptr->vdata[i] = value2;
                }

                nrejr += nrej;
            }

            ntrej += nrejr;
        }

        cpl_free(work);
        work = NULL;

        cpl_free(dork);
        dork = NULL;

        if(work2 != NULL) {
            cpl_free(work2);
            work2 = NULL;
        }

        cpl_free(iflag);
        iflag = NULL;
    }

    ofptr->nrejx = ntrej;

    /* Ditch workspace and get out of here */

    cpl_free(nork);
    nork = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate median and sigma of integer image data.
 *
 * Histogramming is used to compute a precise median and robustly
 * estimated sigma of integer image data.  The histogram is computed
 * over the range specified by "minlev" and "maxlev", any pixels
 * outside this range are clamped to these limits.
 *
 * This is a specialised version of the generic routine used inside
 * qmost_imcombine_lite to implement a workaround for fibre flat
 * spectra where the illumination level is calculated by computing the
 * statistics on the 90th percentile of the pixel intensity
 * distribution to reduce the influence of unilluminated pixels.
 *
 * @param   in_image   (Given)    Input image to process.  The data
 *                                type must be one that can be cast to
 *                                float.
 * @param   minlev     (Given)    The lower limit of the histogram.
 * @param   maxlev     (Given)    The upper limit of the histogram.
 * @param   fibre      (Given)    If true, compute statistics on
 *                                90th percentile of pixel intensity
 *                                distribution for use with fibre
 *                                flat spectra.  Otherwise use
 *                                standard skylevel analysis.
 * @param   skylev     (Returned) The resulting median sky level.
 * @param   sigma      (Returned) The resulting sigma (1.48*MAD).
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If minlev > maxlev.
 * @retval  CPL_ERROR_NULL_INPUT      If the image or one of the
 *                                    output pointers are NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input image data type was
 *                                    not float.
 *
 * If there are no pixels within the specified range, skylev and sigma
 * are left unchanged but no error is raised.  This condition can be
 * detected by setting them to a known value that won't be seen in the
 * image (for example, NAN).
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_imcombine_skylevel (
    cpl_image *in_image,
    int minlev,
    int maxlev,
    int fibre,
    float *skylev,
    float *sigma)
{
    cpl_image *tmp_image = NULL;
    cpl_image *image;

    float *map = NULL;
    cpl_binary *mask = NULL;
    int npix;

    int *hist = NULL;
    int nhist, hmin, hmax, mpix, v, p;
 
    float f;
    float toplev;
    
    cpl_ensure_code(in_image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skylev != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sigma != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(tmp_image) {                             \
        cpl_image_delete(tmp_image);            \
        tmp_image = NULL;                       \
    }                                           \
    if(hist) {                                  \
        cpl_free(hist);                         \
        hist = NULL;                            \
    }

    /* Check if image is float, cast if not */
    if(cpl_image_get_type(in_image) == CPL_TYPE_FLOAT) {
        image = in_image;
    }
    else {
        tmp_image = cpl_image_cast(in_image, CPL_TYPE_FLOAT);
        if(tmp_image == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't cast input image "
                                         "to float");
        }

        image = tmp_image;
    }

    /* Get image */
    map = cpl_image_get_data_float(image);
    if(map == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    npix = cpl_image_get_size_x(image) * cpl_image_get_size_y(image);

    mask = cpl_mask_get_data(cpl_image_get_bpm(image));
    if(mask == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get bad pixel mask "
                                     "for input image");
    }

    /* Compute size of histogram, allocate workspace and clear */
    nhist = maxlev - minlev + 1;

    if(nhist < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "couldn't get bad pixel mask "
                                     "for input image");
    }

    hist = cpl_calloc(nhist, sizeof(int));

    /* Accumulate pixels */
    hmin = nhist-1;
    hmax = 0;
    
    mpix = 0;
    
    for(p = 0; p < npix; p++) {
        /* Good, finite pixels only, ignore NaNs and infinities */
        if(mask[p] == 0 && isfinite(map[p])) {
            f = rintf(map[p]) - minlev;
            
            if(f < 0) {
                v = 0;
            }
            else if(f > nhist-1) {
                v = nhist-1;
            }
            else {
                v = f;
            }
            
            hist[v]++;
            mpix++;
            
            if(v < hmin) {
                hmin = v;
            }
            if(v > hmax) {
                hmax = v;
            }
        }
    }
    
    if(mpix > 0) {
        qmost_skylevel(hist, hmin, hmax, mpix,
		       -FLT_MAX, FLT_MAX, 0,
                       skylev, sigma, &toplev);

	*skylev += minlev;
        toplev += minlev;
        
	if(fibre) {
	    /* Recompute for pixels above 90th percentile */
            memset(hist, 0, nhist * sizeof(int));

            hmin = nhist-1;
            hmax = 0;
    
            mpix = 0;

	    for(p = 0; p < npix; p++) {
		if(mask[p] == 0 && isfinite(map[p]) && map[p] >= toplev) {
		    f = rintf(map[p]) - minlev;
            
		    if(f < 0) {
			v = 0;
		    }
		    else if(f > nhist-1) {
			v = nhist-1;
		    }
		    else {
			v = f;
		    }
            
		    hist[v]++;
		    mpix++;
            
		    if(v < hmin) {
			hmin = v;
		    }
		    if(v > hmax) {
			hmax = v;
		    }
		}
	    }
    
	    if(mpix > 0) {
		qmost_skylevel(hist, hmin, hmax, mpix,
			       -FLT_MAX, FLT_MAX, 0,
			       skylev, sigma, NULL);
        
		*skylev += minlev;
	    }
	}
    }

    TIDY;

    return CPL_ERROR_NONE;
}


/**@}*/

/*

$Log$
Revision 1.6  20210106  mji
Added in use of var info in pixel rejection in meanclip and medianclip

Revision 1.5  2019/02/25 10:41:47  jrl
New memory allocation scheme

Revision 1.4  2018/10/12 10:07:07  jrl
Commented out _stamp calls

Revision 1.3  2017/12/20 16:45:47  jim
Modified to propagate variances properly

Revision 1.2  2016/10/23 15:55:38  jim
Added docs

Revision 1.1  2016/08/24 11:53:06  jim
new


*/
