/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_constants.h"
#include "qmost_pfits.h"
#include "qmost_rebin_spectra.h"
#include "qmost_utils.h"
#include "qmost_waveinfo.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_rebin_spectra  qmost_rebin_spectra
 * 
 * Wavelength calibrate and rebin spectra to uniform wavelength scale.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_rebin_spectra.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void dorebin_lin_off(
    float *in,
    float *inv,
    long nx,
    float *out,
    float *outv,
    long nxout,
    int specbin,
    int xst,
    qmost_waveinfo wv,
    int wv_xst,
    qmost_waveinfo wvoff[2],
    int wvoff_xst[2],
    double veloff,
    double waveoff,
    double waveout1,
    double dwave_out,
    int extrap,
    float tcrv);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Wavelength calibrate and rebin spectra onto a linear
 *          wavelength axis.
 *
 * This routine wavelength calibrates the given spectra by rebinning
 * them onto a linear wavelength axis defined by the given start and
 * end wavelength and dispersion, which are typically fixed values
 * for each arm of each spectrograph, defined in qmost_constants.c.
 * If a dispersion is not specified, a suitable default will be chosen
 * based on the dispersion measured from the wavelength solution.  In
 * either case, the dispersion is in unbinned pixels; if the input
 * spectra are spectrally binned, the requested dispersion is
 * multiplied by the spectral binning factor when deciding the
 * dispersion to use in the output file.
 *
 * All spectra in the rows of the input image are rebinned onto the
 * same wavelength axis, and so the wavelengths can be described by a
 * single linear World Coordinate System (WCS), which is returned in
 * the output FITS header propertylist argument.
 *
 * Flux is conserved in rebinning and proceeds using drizzling if the
 * output spectral pixels are larger than the input pixels, or if they
 * are smaller, by linear interpolation in flux density (multiplied
 * thereafter by the spectral dispersion to return to flux).
 *
 * @param   in_spec_img      (Given)    The input spectra to rebin,
 *                                      as an image.  The data type
 *                                      must be CPL_TYPE_FLOAT.
 * @param   in_spec_var      (Given)    The variance of the input
 *                                      spectra.  The data type
 *                                      must be CPL_TYPE_FLOAT.
 * @param   in_spec_hdr      (Given)    The FITS header for the input
 *                                      spectra.
 * @param   wave_tbl         (Given)    The wavelength solution
 *                                      table.
 * @param   wave_hdr         (Given)    The FITS header for the
 *                                      wavelength solution table.
 * @param   ob_wave_tbl1     (Given)    An optional wavelength
 *                                      solution table giving the
 *                                      first OB level corrrection to
 *                                      the wavelength solution, or
 *                                      NULL.
 * @param   ob_wave_hdr1     (Given)    The FITS header for the
 *                                      first OB level wavelength
 *                                      solution table.
 * @param   ob_wave_tbl2     (Given)    An optional wavelength
 *                                      solution table giving the
 *                                      second OB level correction to
 *                                      the wavelength solution, or
 *                                      NULL.
 * @param   ob_wave_hdr2     (Given)    The FITS header for the
 *                                      second OB level wavelength
 *                                      solution table.
 * @param   waveoff          (Given)    A list of wavelength offsets
 *                                      (in Angstroms), one per input
 *                                      spectrum. See notes.
 * @param   veloff           (Given)    A list of velocity offsets (in
 *                                      km/s), one per input
 *                                      spectrum. See notes.
 * @param   noff             (Given)    The number of offsets in the
 *                                      above lists.
 * @param   wave1            (Given)    The lower wavelength endpoint
 *                                      for the output in Angstroms.
 * @param   wave2            (Given)    The upper wavelength endpoint
 *                                      for the output in Angstroms.
 * @param   dlam             (Given)    The number of Angstroms per
 *                                      pixel for the dispersion. If
 *                                      the user gives a value of 0.0,
 *                                      then a mean dispersion will be
 *                                      worked out from the dispersion
 *                                      solutions and applied.
 * @param   out_spec_img     (Returned) The rebinned spectra.
 * @param   out_spec_var     (Returned) The corresponding variance
 *                                      array for the rebinned
 *                                      spectra.
 * @param   out_spec_hdr     (Modified) The WCS fits headers needed to
 *                                      describe the rebinned spectra.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the dimensions of the
 *                                        input spectra, wavelength
 *                                        solution tables, or offsets
 *                                        don't match.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH       If one of the input FITS
 *                                        header keyword values had an
 *                                        incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MINYST</b>
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS WVCRV</b>
 *   - <b>WVCRV</b>
 *
 * @par Output FITS Headers:
 *   - <b>CD1_1</b>
 *   - <b>CD1_2</b>
 *   - <b>CD2_1</b>
 *   - <b>CD2_2</b>
 *   - <b>CRPIX1</b>
 *   - <b>CRPIX2</b>
 *   - <b>CRVAL1</b>
 *   - <b>CRVAL2</b>
 *   - <b>CTYPE1</b>
 *   - <b>CTYPE2</b>
 *   - <b>CUNIT1</b>
 *   - <b>CUNIT2</b>
 *
 * @note    The wavelength offset is applied so that it is added to
 *          the wavelength calculated from the wavelength
 *          solution. The velocity offset (for example a heliocentric
 *          correction) is positive as a redshift as usual.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_rebin_spectra_off(
    cpl_image *in_spec_img,
    cpl_image *in_spec_var,
    cpl_propertylist *in_spec_hdr,
    cpl_table *wave_tbl,
    cpl_propertylist *wave_hdr,
    cpl_table *ob_wave_tbl1,
    cpl_propertylist *ob_wave_hdr1,
    cpl_table *ob_wave_tbl2,
    cpl_propertylist *ob_wave_hdr2,
    double *waveoff, 
    double *veloff,
    int noff,
    double wave1, 
    double wave2,
    double dlam,
    cpl_image **out_spec_img,
    cpl_image **out_spec_var,
    cpl_propertylist *out_spec_hdr)
{
    int i,j,specbin,spatbin,isbinned;
    long naxis[2],naxisv[2],naxout[2];
    float *inspectrum, *outspectrum;
    float *invar, *outvar;
    float *in, *out, *inv, *outv;
    float tcrv;
    qmost_waveinfo w, ww[2];
    double dwave,voff,lamoff;

    int minyst = 1, wv_minyst = 1;
    int ob_wv_minyst[2] = { 1, 1 };

    /* These are garbage collected and need to be initialized */
    qmost_waveinfo *wv = NULL;
    int nwv = 0;

    qmost_waveinfo *wvoff[2] = { NULL, NULL };
    int nwoff[2] = { 0, 0 };

    /* Check for NULL arguments, particularly the outputs so we can
       initialize them for garbage collection. */
    cpl_ensure_code(in_spec_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_spec_var, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_spec_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wave_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wave_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(waveoff, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(veloff, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_img, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_var, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_spec_hdr, CPL_ERROR_NULL_INPUT);

    /* Initialize these for garbage collection */
    *out_spec_img = NULL;
    *out_spec_var = NULL;

#undef TIDY
#define TIDY                                    \
    if(wv != NULL) {                            \
        qmost_wvclose(nwv, &wv);                \
        wv = NULL;                              \
    }                                           \
    if(wvoff[0] != NULL) {                      \
        qmost_wvclose(nwoff[0], &(wvoff[0]));   \
        wvoff[0] = NULL;                        \
    }                                           \
    if(wvoff[1] != NULL) {                      \
        qmost_wvclose(nwoff[1], &(wvoff[1]));   \
        wvoff[1] = NULL;                        \
    }                                           \
    if(*out_spec_img != NULL) {                 \
        cpl_image_delete(*out_spec_img);        \
        *out_spec_img = NULL;                   \
    }                                           \
    if(*out_spec_var != NULL) {                 \
        cpl_image_delete(*out_spec_var);        \
        *out_spec_var = NULL;                   \
    }

    /* Get input 2D spectra */
    naxis[0] = cpl_image_get_size_x(in_spec_img);
    naxis[1] = cpl_image_get_size_y(in_spec_img);

    inspectrum = cpl_image_get_data_float(in_spec_img);
    if(inspectrum == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D spectrum image");
    }

    qmost_isbinned(in_spec_hdr,&specbin,&spatbin,&isbinned);

    /* Now the variance array */

    naxisv[0] = cpl_image_get_size_x(in_spec_var);
    naxisv[1] = cpl_image_get_size_y(in_spec_var);

    if(naxisv[0] != naxis[0] ||
       naxisv[1] != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input 2D spectrum and variance "
                                     "dimensions don't match: "
                                     "(%ld, %ld) != (%ld, %ld)",
                                     naxis[0], naxis[1],
                                     naxisv[0], naxisv[1]);
    }

    invar = cpl_image_get_data_float(in_spec_var);
    if(invar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input 2D variance image");
    }

    /* Start y coord used for extraction */
    if(qmost_pfits_get_minyst(in_spec_hdr, &minyst) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read minimum yst");
    }

    /* Now the wavelength solution table */

    if(qmost_wvopen(wave_tbl, &nwv, &wv) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "problem reading wavelength "
                                     "solution table");
    }

    /* If the image and tables don't match then get out of here */

    if (nwv < naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input image has %ld spectra, "
                                     "but wavelength solution has %d, "
                                     "not enough wavelength solutions",
                                     naxis[1], nwv);
    }
    else if (nwv != naxis[1]) {
        cpl_msg_warning(cpl_func,
                        "Mismatched wavelength solution: "
                        "input image has %ld spectra, "
                        "but wavelength solution has %d ",
                        naxis[1], nwv);
    }

    /* Start y coord used for wave solution, default if not given in
     * file is to assume it's the same as the spectrum. */
    wv_minyst = minyst;

    if(qmost_pfits_get_minyst(wave_hdr, &wv_minyst) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read minimum yst");
    }

    /* Get wavelength solution offsets (OB level wavelength
       solutions) */

    if(ob_wave_tbl1 != NULL) {
        if(qmost_wvopen(ob_wave_tbl1,
                        &(nwoff[0]),&(wvoff[0])) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "problem reading OB level "
                                         "wavelength offset table 1");
        }
        if(nwoff[0] != naxis[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "input image has %ld spectra, "
                                         "but OB level wavelength offset "
                                         "table 1 has %d, these must match",
                                         naxis[1], nwoff[0]);
        }

        ob_wv_minyst[0] = minyst;
        
        if(qmost_pfits_get_minyst(ob_wave_hdr1,
                                  &(ob_wv_minyst[0])) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read minimum yst");
        }
    } else {
        nwoff[0] = 0;
        wvoff[0] = NULL;
    }

    if(ob_wave_tbl2 != NULL) {
        if(qmost_wvopen(ob_wave_tbl2,
                        &(nwoff[1]),&(wvoff[1])) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "problem reading OB level "
                                         "wavelength offset table 2");
        }
        if(nwoff[1] != naxis[1]) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "input image has %ld spectra, "
                                         "but OB level wavelength offset "
                                         "table 2 has %d, these must match",
                                         naxis[1], nwoff[1]);
        }

        ob_wv_minyst[1] = minyst;
        
        if(qmost_pfits_get_minyst(ob_wave_hdr2,
                                  &(ob_wv_minyst[1])) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read minimum yst");
        }
    } else {
        nwoff[1] = 0;
        wvoff[1] = NULL;
    }
    
    /* Check that there are the same number of offsets as spectra */

    if (noff != naxis[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "input image has %ld spectra, "
                                     "but there are %d offsets, "
                                     "these must match",
                                     naxis[1], noff);
    }

    /* Work out mean dispersion and wavelength endpoints so that all 
       spectra can be rebinned to a common wavelength solution */

    dwave = (dlam == 0.0 ? qmost_wvmeandisp(nwv,wv) : dlam);
    tcrv = 1.0;
    if (isbinned) {
        /* wave1 += 0.5*(double)(specbin-1)*dlam; */
        /* wave2 -= 0.5*(double)(specbin-1)*dlam;      */       
        dwave *= (double)specbin;

        if(cpl_propertylist_has(in_spec_hdr, "ESO DRS WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_spec_hdr,
                                                "ESO DRS WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read ESO DRS WVCRV "
                                             "from input FITS header");
            }
        }
        else if(cpl_propertylist_has(in_spec_hdr, "WVCRV")) {
            if(qmost_cpl_propertylist_get_float(in_spec_hdr,
                                                "WVCRV",
                                                &tcrv) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't read WVCRV "
                                             "from input FITS header");
            }
        }
    }
    naxout[0] = qmost_nint((wave2 - wave1)/dwave) + 1;
    naxout[1] = naxis[1];

    /* Create outputs */

    *out_spec_img = cpl_image_new(naxout[0], naxout[1], CPL_TYPE_FLOAT);
    *out_spec_var = cpl_image_new(naxout[0], naxout[1], CPL_TYPE_FLOAT);
    if(*out_spec_img == NULL || *out_spec_var == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create output spectrum "
                                     "array of size (%ld, %ld)",
                                     naxout[0], naxout[1]);
    }

    outspectrum = cpl_image_get_data_float(*out_spec_img);
    outvar = cpl_image_get_data_float(*out_spec_var);
    if(outspectrum == NULL || outvar == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get float pointer to "
                                     "output spectrum array");
    }

    /* Right, loop for each spectrum */

    for (i = 0; i < naxis[1]; i++) {
	w = wv[i];
	if (! w.live)
	    continue;

        memset(ww, 0, sizeof(ww));

        for (j = 0; j < 2; j++) {
            if (wvoff[j] != NULL)
                ww[j] = wvoff[j][i];
        }

        /* Get input and output pointers */
                    
	in = inspectrum + i*naxis[0];
	out = outspectrum + i*naxout[0];
        inv = invar + i*naxis[0];
        outv = outvar + i*naxout[0];

	/* Do the rebinning */

	voff = veloff[i];
	lamoff = waveoff[i];

	dorebin_lin_off(in,inv,naxis[0],
                        out,outv,naxout[0],
                        specbin,
                        minyst,
                        w,wv_minyst,ww,ob_wv_minyst,
                        voff,lamoff,
                        wave1,dwave,
                        0,tcrv);
    }

    /* Create output WCS headers */
    cpl_propertylist_update_string(out_spec_hdr, "CTYPE1", "AWAV");
    cpl_propertylist_set_comment(out_spec_hdr, "CTYPE1",
                                    "Grating dispersion function");

    cpl_propertylist_update_string(out_spec_hdr, "CTYPE2", "NSPEC");
    cpl_propertylist_set_comment(out_spec_hdr, "CTYPE2",
                                 "Number of spectrum");

    cpl_propertylist_update_string(out_spec_hdr, "CUNIT1", "Angstrom");
    cpl_propertylist_update_string(out_spec_hdr, "CUNIT2", " ");

    cpl_propertylist_update_double(out_spec_hdr, "CRVAL1", wave1);
    cpl_propertylist_set_comment(out_spec_hdr, "CRVAL1",
                                 "Wavelength zeropoint");

    cpl_propertylist_update_double(out_spec_hdr, "CRVAL2", 1);

    cpl_propertylist_update_double(out_spec_hdr, "CRPIX1", 1);
    cpl_propertylist_set_comment(out_spec_hdr, "CRPIX1",
                                 "Pixel zeropoint");

    cpl_propertylist_update_double(out_spec_hdr, "CRPIX2", 1);
    cpl_propertylist_set_comment(out_spec_hdr, "CRPIX2",
                                 "Pixel zeropoint");

    cpl_propertylist_update_double(out_spec_hdr, "CD1_1", dwave);
    cpl_propertylist_set_comment(out_spec_hdr, "CD1_1",
                                 "Wavelength increment");

    cpl_propertylist_update_double(out_spec_hdr, "CD1_2", 0);
    cpl_propertylist_update_double(out_spec_hdr, "CD2_1", 0);
    cpl_propertylist_update_double(out_spec_hdr, "CD2_2", 1);

    qmost_wvclose(nwv, &wv);
    if(wvoff[0] != NULL) {
        qmost_wvclose(nwoff[0], &(wvoff[0]));
    }
    if(wvoff[1] != NULL) {
        qmost_wvclose(nwoff[1], &(wvoff[1]));
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Do a linear wavelength rebinning operation with offsets.
 *
 * Rebin a single spectrum onto a linear wavelength scale using
 * wavelength and velocity offsets.
 *
 * @param   in         (Given)    The input array to be rebinned.
 * @param   inv        (Given)    The input variance array to be
 *                                rebinned.
 * @param   nx         (Given)    The number of pixels in the input
 *                                array.
 * @param   out        (Modified) The output array to hold the
 *                                rebinned data.
 * @param   outv       (Modified) The output array to hold the
 *                                rebinned variance data.
 * @param   nxout      (Given)    The number of pixels in the output
 *                                array.
 * @param   specbin    (Given)    Spectral binning factor if used.
 * @param   xst        (Given)    
 * @param   wv         (Given)    The wavelength solution information
 *                                for the master.
 * @param   wv_xst     (Given)
 * @param   wvoff      (Given)    The wavelength solution correction
 *                                information for at least two OB
 *                                level arcs.
 * @param   wvoff_xst  (Given)  
 * @param   veloff     (Given)    The velocity offset.
 * @param   waveoff    (Given)    The wavelength offset.
 * @param   waveout1   (Given)    The wavelength lower limit for the
 *                                output array.
 * @param   dwave_out  (Given)    The wavelength pixel size in
 *                                angstroms/pixel.
 * @param   extrap     (Given)    If set, then if the requested
 *                                wavelength range is greater than the
 *                                wavelength range covered by the
 *                                wavelength solution, then the extra
 *                                pixels will be set to an
 *                                extrapolated value of the last good
 *                                pixels. If not set, then these will
 *                                be given a value of 0.0.
 * @param   tcrv       (Given)    Physical pixel offset in case of
 *                                binning.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void dorebin_lin_off(
    float *in,
    float *inv,
    long nx,
    float *out,
    float *outv,
    long nxout,
    int specbin,
    int xst,
    qmost_waveinfo wv,
    int wv_xst,
    qmost_waveinfo wvoff[2],
    int wvoff_xst[2],
    double veloff,
    double waveoff,
    double waveout1,
    double dwave_out,
    int extrap,
    float tcrv)
{
    double *wavein_l = NULL;
    double *wavein_r = NULL;
    double *wavein = NULL;

    double dxr,xx,waveout,warp,woff_l,woff_r,waveout_l,waveout_r;
    float evalue,sum,dl,wt,sumw,sumv,evaluev,psin,cf_fac,delx;
    int i,j,j1,j2,ilook,jlook,firstgood,lastgood;

    if(nx < 1) {
        return;
    }
    
    /* Get workspace to hold the wavelength array for the input data */
    wavein_l = cpl_malloc(nx*sizeof(double));
    wavein_r = cpl_malloc(nx*sizeof(double));
    wavein = cpl_malloc(nx*sizeof(double));

    /* Left edge of first input pixel */
    dxr = (double)wv.xref;
    warp = 1.0 + veloff/QMOST_SPEEDOFLIGHT;

    i = 0;

    if (specbin == 1) {
        xx = ((double)(i+1) - dxr);
    }
    else {
        xx = (double)tcrv + (double)(specbin*i) - dxr;
    }
    wavein_l[i] = cpl_polynomial_eval_1d(wv.coefs,
                                         xx + (xst - wv_xst) -
                                         0.5*(double)specbin,NULL);
    woff_l = 0.0;
    for (j = 0; j < 2; j++) {
        if (wvoff[j].coefs != NULL && wvoff[j].ngood > 0) {
            woff_l += cpl_polynomial_eval_1d(wvoff[j].coefs,
                                             xx + (xst - wvoff_xst[j]) -
                                             0.5*(double)specbin,
                                             NULL);
        }
    }
    wavein_l[i] = warp*(wavein_l[i] + woff_l + waveoff);

    /* Left edge of pixel i = right edge of pixel i-1 */
    for (i = 1; i < nx; i++) {
        if (specbin == 1) {
            xx = ((double)(i+1) - dxr);
        }
        else {
            xx = (double)tcrv + (double)(specbin*i) - dxr;
        }
	wavein_l[i] = cpl_polynomial_eval_1d(wv.coefs,
                                             xx + (xst - wv_xst) -
                                             0.5*(double)specbin,NULL);
        woff_l = 0.0;
        for (j = 0; j < 2; j++) {
            if (wvoff[j].coefs != NULL && wvoff[j].ngood > 0) {
                woff_l += cpl_polynomial_eval_1d(wvoff[j].coefs,
                                                 xx + (xst - wvoff_xst[j]) -
                                                 0.5*(double)specbin,
                                                 NULL);
            }
        }
	wavein_l[i] = warp*(wavein_l[i] + woff_l + waveoff);
        wavein_r[i-1] = wavein_l[i];
        wavein[i-1] = 0.5*(wavein_l[i-1] + wavein_r[i-1]);
    }

    /* Right edge of last pixel */
    i = nx - 1;

    if (specbin == 1) {
        xx = ((double)(i+1) - dxr);
    }
    else {
        xx = (double)tcrv + (double)(specbin*i) - dxr;
    }
    
    wavein_r[i] = cpl_polynomial_eval_1d(wv.coefs,
                                         xx + (xst - wv_xst) +
                                         0.5*(double)specbin,NULL);
    woff_r = 0.0;
    for (j = 0; j < 2; j++) {
        if (wvoff[j].coefs != NULL && wvoff[j].ngood > 0) {
            woff_r += cpl_polynomial_eval_1d(wvoff[j].coefs,
                                             xx + (xst - wvoff_xst[j]) +
                                             0.5*(double)specbin,
                                             NULL);
        }
    }
    
    wavein_r[i] = warp*(wavein_r[i] + woff_r + waveoff);
    wavein[i] = 0.5*(wavein_l[i] + wavein_r[i]);
    
    /* Loop for each output pixel */

    firstgood = nxout;
    lastgood = -1;
    ilook = 0;
    jlook = 0;
    for (i = 0; i < nxout; i++) {
	waveout = waveout1 + dwave_out*(double)i;
        waveout_l = waveout - 0.5*dwave_out;
        waveout_r = waveout + 0.5*dwave_out;
	if (waveout_r < wavein_l[0] || waveout_l > wavein_r[nx-1]) {
	    out[i] = 0.0;
            outv[i] = 0.0;
	    continue;
	}
	if (i < firstgood)
	    firstgood = i;
	if (i > lastgood)
	    lastgood = i;

	/* Find the input pixel just blueward of the current output pixel */

	j1 = -1;
	j2 = nx;
	for (j = ilook; j < nx; j++) {
            if (waveout_l >= wavein_l[j] && waveout_l < wavein_r[j]) {
                j1 = j;
                break;
            }
        }
        for (j = qmost_max(j1, 0); j < nx; j++) {
            if (waveout_r > wavein_l[j] && waveout_r <= wavein_r[j]) {
                j2 = j;
                break;
            }
        }
        if (j1 == -1 && j == nx) {
            j1 = nx - 2;
            j2 = j1 + 1;
        } else {
            j1 = qmost_max(0,j1);
            j2 = qmost_min((nx-1),j2);
        }
	ilook = j1;

        /* Work out the ratio of the pixel sizes to create a factor we can
           use to ensure we conserve flux */

        psin = 0.5*((wavein_r[j1] - wavein_l[j1]) +
                    (wavein_r[j2] - wavein_l[j2]));
        cf_fac = dwave_out/psin;

        /* Swap between two different interpolation schemes. The first is a
           drizzle and the second is a straight forward linear interpolation */

        if (cf_fac > 1.0) {  /* disabled || drizzle to be sure */
            
            /* Look at the various situations. Start with the output pixel being
               totally inside the input pixel */

            if (j1 == j2) {
                out[i] = cf_fac*in[j1];
#ifdef REBIN_VARIANCE_INTERPOLATION
                outv[i] = cf_fac*inv[j1];
#else
                outv[i] = cf_fac*cf_fac*inv[j1];
#endif  /* REBIN_VARIANCE_INTERPOLATION */

                /* There are overlaps */

            } else {
                sum = 0.0;
                sumv = 0.0;
                sumw = 0.0;
                for (j = j1+1; j <= j2-1; j++) {
                    if (inv[j] == 0.0)
                        continue;
                    sum += in[j];
                    sumv += inv[j];
                    sumw += 1.0;
                }
                if (inv[j1] != 0.0) {
                    dl = wavein_r[j1] - waveout_l;
                    wt = dl/(wavein_r[j1] - wavein_l[j1]);
                    sum += in[j1]*wt;
#ifdef REBIN_VARIANCE_INTERPOLATION
                    sumv += inv[j1]*wt;
#else
                    sumv += inv[j1]*wt*wt;
#endif  /* REBIN_VARIANCE_INTERPOLATION */
                    sumw += wt;
                }
                if (inv[j2] != 0.0) {
                    dl = waveout_r - wavein_l[j2];
                    wt = dl/(wavein_r[j2] - wavein_l[j2]);
                    sum += in[j2]*wt;
#ifdef REBIN_VARIANCE_INTERPOLATION
                    sumv += inv[j2]*wt;
#else
                    sumv += inv[j2]*wt*wt;
#endif  /* REBIN_VARIANCE_INTERPOLATION */
                    sumw += wt;
                }
                if (sumw != 0.0) {
                    out[i] = cf_fac*sum/sumw;
#ifdef REBIN_VARIANCE_INTERPOLATION
                    outv[i] = qmost_max(0.0,cf_fac*sumv/sumw);
#else
                    outv[i] = qmost_max(0.0,cf_fac*cf_fac*sumv/(sumw*sumw));
#endif  /* REBIN_VARIANCE_INTERPOLATION */
                } else {
                    out[i] = 0.0;
                    outv[i] = 0.0;
                }
            }

        } else {

        /* Linear interpolation ensure we have the two adjacent points */
            
            j1 = -1;
	    for (j = jlook; j < nx; j++) {
	        if (wavein[j] > waveout) {
		    j1 = j - 1;
		    break;
	        }
	    }
            if (j1 == -1 && j == nx) {
                out[i] = 0.0;
                outv[i] = 0.0;
                continue;
            }
            if (j1 == nx - 1) {
                j2 = j1;
                j1--;
            } else {
                j1 = qmost_max(0,j1);
                j2 = qmost_min((nx-1),j1+1);
            }
	    jlook = j1;

            /* Make sure these are two good points */
            
            if (inv[j1] == 0.0 || inv[j2] == 0.0 || j1 == j2) {
                out[i] = 0.0;
                outv[i] = 0.0;
                continue;
            }

            /* Work out the linear fit to the two adjacent points */

	    delx = (waveout - wavein[j1])/(wavein[j2] - wavein[j1]);
	    delx = qmost_min(1.0,qmost_max(0.0,delx));
	    out[i] = cf_fac*(in[j1]*(1.0 -delx) + in[j2]*delx);
#ifdef REBIN_VARIANCE_INTERPOLATION
	    outv[i] = cf_fac*(inv[j1]*(1.0 -delx) + inv[j2]*delx);
#else
	    outv[i] = cf_fac*cf_fac*(inv[j1]*(1.0 -delx)*(1.0 -delx) +
                                     inv[j2]*delx*delx);
#endif  /* REBIN_VARIANCE_INTERPOLATION */
        }
    }

    /* Pad out the ends */

    if (firstgood > -1) {
        evalue = (extrap ? out[firstgood] : 0.0);
        evaluev = (extrap ? outv[firstgood] : 0.0);
	for (i = 0; i < firstgood; i++) {
	    out[i] = evalue;
            outv[i] = evaluev;
        }
    }
    if (lastgood < nxout) {
        evalue = (extrap ? out[lastgood] : 0.0);
        evaluev = (extrap ? outv[lastgood] : 0.0);
	for (i = nxout-1; i > lastgood; i--) {
	    out[i] = evalue;
            outv[i] = evaluev;
        }
    }

    /* Tidy and exit */

    cpl_free(wavein_l);
    wavein_l = NULL;

    cpl_free(wavein_r);
    wavein_r = NULL;

    cpl_free(wavein);
    wavein = NULL;
}

/**@}*/

/*

$Log$
Revision 1.19  20230125 jmi
Fixed out of bounds read in dorebin_lin_off when locating the 
end wavelength of the first output pixel.  The logic surrounding this
was also fixed and now considers the first and last input pixels.

Revision 1.18  20221224 jmi
Made other requested FITS header changes from DXU comments.

Revision 1.17  20211006 mji
sorted out calibration filenames in PHU

Revision 1.16  20191002 mji
Replaced Jim's arcane way of doing linear interpolation with simpler method
and got rid of spurious -ve output variances

Revision 1,15  20190820 mji
Fixed bug in calling routine dorebin_lin_off viz swapped tcrv and drizzle 
Added automatic choice for drizzle or linear but left drizzle-only option 
Fixed further bug in linear interpolation and tidied up code
Disabled warp factor as designed for heliocentric wavelength correction

Revision 1.14  2019/04/04 13:51:20  jrl
Added non-drizzle linear interpolation option

Revision 1.13  2019/02/25 10:44:08  jrl
New memory allocation scheme, binning added

Revision 1.12  2018/11/29 12:43:44  jrl
Modified to fix bug in scaling of the variance

Revision 1.11  2018/10/25 06:11:54  jrl
Modified rebinning routine to multiply by factor that conserves flux

Revision 1.10  2018/10/23 11:00:04  jrl
Updated rebinning routine. It now does the spectrum and variance together. The
rebinning algorithm is based on a linear drizzle. Finally bad pixels are
taken into account too.

Revision 1.9  2018/10/12 10:09:03  jrl
mainly superficial modifications. Added extra linear interpolation scheme, but
it's commented out for now

Revision 1.8  2018/09/19 11:28:34  jrl
qmost_rebin_spectra_off modified to include ob level wavelength solutions to
tweak the master solution

Revision 1.7  2018/08/01 09:10:12  jim
removed some commented out code

Revision 1.6  2017/05/31 11:54:18  jim
Fixed so that places where there are no contributions from any spectra are
now set to zero rather than some inter/extrapolation

Revision 1.5  2016/10/26 11:20:40  jim
Added docs

Revision 1.4  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.3  2016/07/11 15:00:38  jim
Removed references to deprecated waveinfo items. Fixed EXTNAME problem

Revision 1.2  2016/07/06 11:04:53  jim
Modified main rebinning routine to use new wavelength file format.

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry


*/
