/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_doarcs.h"
#include "qmost_extract_tram.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_rebin_spectra.h"
#include "qmost_scattered.h"
#include "qmost_utils.h"
#include "qmost_waveinfo.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_fpe_analyse"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_fpe_analyse_description[] =
    "Combine individual simufpe frames, remove scattered light, and do a\n"
    "boxcar extraction.  Identify emission features in the extracted\n"
    "spectrum and measure their wavelengths using the given simuarc\n"
    "wavelength solution.  Emit a Fabry-Perot line list.\n\n"
    "The master ThAr wavelength solution is produced from simuarc\n"
    "frames using the recipe qmost_arc_analyse with a ThAr line list.\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                   Req/Opt? Tag\n"
    "----------------------------- -------- --------------------\n"
    "simufpe frames                Required " QMOST_RAW_FIBRE_WAVE_SIMUFPE "\n"
    "Master bias frame             Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame             Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master detector flat          Optional " QMOST_PRO_MASTER_DETECTOR_FLAT "\n"
    "Master bad pixel mask         Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Master linearity table        Optional " QMOST_PRO_LINEARITY "\n"
    "Trace table                   Required " QMOST_PRO_FIBRE_TRACE "\n"
    "Fibre mask                    Required " QMOST_PRO_FIBRE_MASK "\n"
    "Simuarc wavelength solution   Required " QMOST_PRO_SIMUARC_WAVE "\n"
    "\n"
    "Outputs:\n\n"
    "Description                   Tag\n"
    "----------------------------- -----------------\n"
    "FPE line list                 " QMOST_PRO_FPE_LINELIST "\n"
    "\n"
    "Optional processed outputs if keep=true:\n\n"
    "Description                   Tag\n"
    "----------------------------- -----------------\n"
    "Stacked FPE frame             " QMOST_PROC_FIBRE_WAVE "\n"
    "Extracted FPE spectrum        " QMOST_PRO_FPE_SPECTRUM "\n"
    "\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the FPE line list for\n"
    "LRS-A would be QMOST_" QMOST_PRO_FPE_LINELIST "_LRS-B.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_fpe_analyse,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Analyse simufpe frames to produce a Fabry-Perot line list.",
                  qmost_fpe_analyse_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_fpe_analyse    qmost_fpe_analyse
 *
 * @brief Analyse simufpe frames to produce a Fabry-Perot line list.
 *
 * @par Name: 
 *   qmost_fpe_analyse
 * @par Purpose:
 *   Combine individual simufpe frames, remove scattered light, and do
 *   a boxcar extraction.  Identify emission features in the extracted
 *   spectrum and measure their wavelengths using the given master
 *   ThAr wavelength solution.  Emit a Fabry-Perot line list.
 * @par Type:
 *   Spectroscopic calibration
 * @par Parameters:
 *   - @b keep (bool, default false):
 *     If this flag is set, optional products generated during
 *     execution of the recipe are also saved. In the case of
 *     qmost_fpe_analyse, this saves the processed FPE image and the
 *     extracted FPE spectrum.
 *   - @b ccdproc.swapx (string, default 100010101):
 *     String of 0 or 1 specifying for each arm of each spectrograph
 *     if we should flip the x axis to correct the wavelength order of
 *     the spectral axis.
 *   - @b imcombine.combtype (int, default 1):
 *     Determines the type of combination that is done to form the
 *     stacked arc frame.  Can take the following values:
 *       - (1): The output pixels are kappa sigma clipped means of
 *              the input pixels.
 *       - (2): The output pixels are medians of the input pixels.
 *   - @b imcombine.scaletype (int, default 0):
 *     Scaling to use when combining multiple images to form the
 *     stacked arc frame.  Can take the following values:
 *       - (0): No biasing or scaling is done.
 *       - (1): An additive offset is applied to remove any
 *              difference in background levels.
 *       - (2): A multiplicative scaling is applied to remove any
 *              difference in background levels.
 *       - (3): Images are scaled by their relative exposure times
 *              and an additive offset is applied to remove any
 *              remaining difference in background levels.
 *   - @b imcombine.xrej (bool, default false):
 *     If set, then an extra rejection cycle will be run.
 *   - @b imcombine.thresh (float, default 5.0):
 *     The rejection threshold for clipping during the combination in
 *     units of the background sigma.
 *   - @b scattered.enable (bool, default true):
 *     Enable scattered light removal.
 *   - @b scattered.nbsize (int, default 256):
 *     Size of the background smoothing cells for determining
 *     scattered light in pixels.
 *   - @b extract.width (int, default 5):
 *     Boxcar width in pixels over which the summation will be done
 *     at each spectral pixel during extraction.
 *   - @b arc.detthr (float, default 5.0):
 *     Detection threshold used to find emission features in the
 *     extracted arc spectrum in units of the background sigma.
 *   - @b arc.matchwindow (float, default 1.5):
 *     Maximum deviation between the measured positions of an arc line
 *     between different fibres for it to be considered the same line,
 *     in units of the spectral resolution element (evaluated for a
 *     nominal resolving power of 6000 at 6000A for LRS, and 18000 at
 *     6000A for HRS, scaling with wavelength such that it remains
 *     constant in terms of resolving power).
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b FIBRE_WAVE_SIMUFPE: Raw simufpe frames.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_DETECTOR_FLAT (optional): Master detector flat.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b LINEARITY (optional): Master linearity table.
 *    - @b FIBRE_TRACE (required): Trace table.
 *    - @b FIBRE_MASK (required): Fibre mask.
 *    - @b SIMUARC_WAVE (required): Simuarc wavelength solution.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b FPE_LINELIST: The resulting line list table of measured
 *          FPE line wavelengths.
 *
 *   - The following intermediate product files are optional and are
 *     only emitted if keep=true is specified in the recipe parameters.
 *     - @b PROC_FIBRE_WAVE: Processed 2D arc image (prior to spectral
 *          extraction).  Can be used to assess the quality of the
 *          initial CCD processing, such as bias, dark, detector flat
 *          field correction, scattered light correction.
 *     - @b FPE_SPECTRUM: Extracted FPE spectrum prior to wavelength
 *          calibration.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     FPE line list for LRS-A would be @c
 *     QMOST_FPE_LINELIST_LRS-B.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean residual background level
 *     in the illuminated region of the overscan subtracted frames
 *     that were combined.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>EXT FLUX MED</b> (ADU): The median of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX RMS</b> (ADU): The robustly-estimated RMS of
 *     average extracted flux in each fibre.
 *   - <b>EXT FLUX MIN</b> (ADU): The minimum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MINSPC</b>: The fibre with the minimum extracted
 *     flux.
 *   - <b>EXT FLUX MAX</b> (ADU): The maximum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MAXSPC</b>: The fibre with the maximum extracted
 *     flux.
 *   - <b>EXT SN MED</b>: The median of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN RMS</b>: The robustly-estimated RMS of the average
 *     signal to noise ratio in each fibre.
 *   - <b>EXT SN MIN</b>: The minimum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MINSPC</b>: The fibre with the minimum signal to
 *     noise ratio.
 *   - <b>EXT SN MAX</b>: The maximum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MAXSPC</b>: The fibre with the maximum signal to
 *     noise ratio.
 *   - <b>SCATTL MED</b> (ADU): The median scattered light level on
 *     the detector outside the region illuminated by the fibres.
 *   - <b>SCATTL RMS</b> (ADU): The robustly-estimated RMS of the
 *     scattered light level on the detector outside the region
 *     illuminated by the fibres.
 *   - <b>SCATTL RESID MED</b> (ADU): The median residual scattered
 *     light background remaining after subtraction.
 *   - <b>SCATTL RESID RMS</b> (ADU): The RMS residual of the
 *     scattered light background remaining after subtraction.
 *   - <b>FPE NFIB MEAN</b>: The mean number of fibres each FPE line
 *     was detected in.
 *   - <b>FPE NFIB MAX</b>: The maximum number of fibres an FPE line
 *     was detected in.
 *   - <b>FPE RMS MEAN</b> (A): The mean RMS scatter of the wavelength
 *     in an FPE line.
 *   - <b>FPE RMS MIN</b> (A): The RMS scatter of the wavelength for
 *     the FPE line with the smallest RMS scatter.
 *   - <b>FPE RMS MAX</b> (A): The RMS scatter of the wavelength for
 *     the FPE line with the highest RMS scatter.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No simufpe frames in the input frameset.
 *   - No master fibre trace, mask, or simuarc wavelength solution.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 *   - If the trace table for an active detector is empty.
 * @par Non-Fatal Error Conditions:
 *   - No FPE lines are detected.
 *   - Missing bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> qmost_extract_tram;
 *   SIMUARC_WAVE -> qmost_fpmeasure;
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_WAVE_SIMUFPE -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> qmost_extract_tram;
 *     qmost_extract_tram -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_fpmeasure;
 *
 *     qmost_fpmeasure -> FPE_LINELIST;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_WAVE;
 *   qmost_extract_qc -> FPE_SPECTRUM;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_fpmeasure -> QC1;
 * 
 *   FIBRE_WAVE_SIMUFPE [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   SIMUARC_WAVE [shape="box" fillcolor="#fff5ce"]
 *   FPE_LINELIST [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_WAVE [shape="box" fillcolor="#ffffff"]
 *   FPE_SPECTRUM [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_fpe_analyse(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    const cpl_frame *master_bias_frame = NULL;
    const cpl_frame *master_dark_frame = NULL;
    const cpl_frame *master_detflat_frame = NULL;
    const cpl_frame *master_bpm_frame = NULL;
    const cpl_frame *master_lintab_frame = NULL;
    const cpl_frame *fibre_trace_frame = NULL;
    const cpl_frame *fibre_mask_frame = NULL;
    const cpl_frame *simuarc_wave_frame = NULL;

    const cpl_frame *proc_arc_frame = NULL;
    const char *proc_arc_filename = NULL;

    const cpl_frame *extracted_arc_frame = NULL;
    const char *extracted_arc_filename = NULL;

    const cpl_frame *linelist_frame = NULL;
    const char *linelist_filename = NULL;

    cpl_frameset *arcs_raw = NULL;
    int nin, iin, nff;

    const cpl_frame *this_frame;
    const char *this_tag;

    cpl_frame *arc_ref;
    const char *arc_ref_filename;

    const cpl_parameter *par;
    int keep = 0;
    const char *swapx_table = QMOST_DEFAULT_SWAPX_TABLE;
    int combtype = QMOST_MEANCALC;
    int scaletype = 0;
    int xrej = 0;
    double combthresh = 5.0;
    int scattered_enable = 1;
    int scattered_nbsize = 256;
    int extract_width = 5;
    float detthr = 5.0;
    float matchwindow = 1.5;

    cpl_size raw_extension;
    int spec, arm, detlive;
    const char *arm_extname;
    char *extname = NULL;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_image *master_detflat_img = NULL;
    cpl_image *master_detflat_var = NULL;
    cpl_table *master_lintab = NULL;

    cpl_image *arc_img = NULL;
    cpl_image *arc_var = NULL;
    cpl_propertylist *arc_hdr = NULL;

    cpl_propertylist *qclist = NULL;

    cpl_table *fibinfo_tbl = NULL;
    cpl_propertylist *fibinfo_hdr = NULL;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_mask *fibre_mask = NULL;

    int irow, nrows, nrowsf, nrowst, fibuse, isnull;

    cpl_table *simuarc_wave_tbl = NULL;
    cpl_propertylist *simuarc_wave_hdr = NULL;
    cpl_table *linelist_tbl = NULL;

    cpl_image *extracted_arc_img = NULL;
    cpl_image *extracted_arc_var = NULL;

#undef TIDY
#define TIDY                                            \
    if(arcs_raw) {                                      \
        cpl_frameset_delete(arcs_raw);                  \
        arcs_raw = NULL;                                \
    }                                                   \
    if(fibinfo_tbl) {                                   \
        cpl_table_delete(fibinfo_tbl);                  \
        fibinfo_tbl = NULL;                             \
    }                                                   \
    if(fibinfo_hdr) {                                   \
        cpl_propertylist_delete(fibinfo_hdr);           \
        fibinfo_hdr = NULL;                             \
    }                                                   \
    if(master_bpm) {                                    \
        cpl_mask_delete(master_bpm);                    \
        master_bpm = NULL;                              \
    }                                                   \
    if(master_bias_img) {                               \
        cpl_image_delete(master_bias_img);              \
        master_bias_img = NULL;                         \
    }                                                   \
    if(master_bias_var) {                               \
        cpl_image_delete(master_bias_var);              \
        master_bias_var = NULL;                         \
    }                                                   \
    if(master_dark_img) {                               \
        cpl_image_delete(master_dark_img);              \
        master_dark_img = NULL;                         \
    }                                                   \
    if(master_dark_var) {                               \
        cpl_image_delete(master_dark_var);              \
        master_dark_var = NULL;                         \
    }                                                   \
    if(master_detflat_img) {                            \
        cpl_image_delete(master_detflat_img);           \
        master_detflat_img = NULL;                      \
    }                                                   \
    if(master_detflat_var) {                            \
        cpl_image_delete(master_detflat_var);           \
        master_detflat_var = NULL;                      \
    }                                                   \
    if(master_lintab) {                                 \
        cpl_table_delete(master_lintab);                \
        master_lintab = NULL;                           \
    }                                                   \
    if(arc_img) {                                       \
        cpl_image_delete(arc_img);                      \
        arc_img = NULL;                                 \
    }                                                   \
    if(arc_var) {                                       \
        cpl_image_delete(arc_var);                      \
        arc_var = NULL;                                 \
    }                                                   \
    if(arc_hdr) {                                       \
        cpl_propertylist_delete(arc_hdr);               \
        arc_hdr = NULL;                                 \
    }                                                   \
    if(qclist) {                                        \
        cpl_propertylist_delete(qclist);                \
        qclist = NULL;                                  \
    }                                                   \
    if(trace_tbl) {                                     \
        cpl_table_delete(trace_tbl);                    \
        trace_tbl = NULL;                               \
    }                                                   \
    if(trace_hdr) {                                     \
        cpl_propertylist_delete(trace_hdr);             \
        trace_hdr = NULL;                               \
    }                                                   \
    if(fibre_mask) {                                    \
        cpl_mask_delete(fibre_mask);                    \
        fibre_mask = NULL;                              \
    }                                                   \
    if(simuarc_wave_tbl) {                              \
        cpl_table_delete(simuarc_wave_tbl);             \
        simuarc_wave_tbl = NULL;                        \
    }                                                   \
    if(simuarc_wave_hdr) {                              \
        cpl_propertylist_delete(simuarc_wave_hdr);      \
        simuarc_wave_hdr = NULL;                        \
    }                                                   \
    if(linelist_tbl) {                                  \
        cpl_table_delete(linelist_tbl);                 \
        linelist_tbl = NULL;                            \
    }                                                   \
    if(extracted_arc_img) {                             \
        cpl_image_delete(extracted_arc_img);            \
        extracted_arc_img = NULL;                       \
    }                                                   \
    if(extracted_arc_var) {                             \
        cpl_image_delete(extracted_arc_var);            \
        extracted_arc_var = NULL;                       \
    }                                                   \
    if(extname) {                                       \
        cpl_free(extname);                              \
        extname = NULL;                                 \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    master_bias_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_BIAS);
    master_dark_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_DARK);
    master_detflat_frame = cpl_frameset_find_const(frameset,
                                                         QMOST_PRO_MASTER_DETECTOR_FLAT);
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);
    master_lintab_frame = cpl_frameset_find_const(frameset,
                                                  QMOST_PRO_LINEARITY);

    /* Trace table and fibre mask, required */
    fibre_trace_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_FIBRE_TRACE);
    if(fibre_trace_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_TRACE);
    }

    fibre_mask_frame = cpl_frameset_find_const(frameset,
                                               QMOST_PRO_FIBRE_MASK);
    if(fibre_mask_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_MASK);
    }

    /* Master wavelength solution, required */
    simuarc_wave_frame = cpl_frameset_find_const(frameset,
                                                 QMOST_PRO_SIMUARC_WAVE);
    if(simuarc_wave_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_SIMUARC_WAVE);
    }

    /* Create frameset of input raw arcs */
    arcs_raw = cpl_frameset_new();

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_FIBRE_WAVE_SIMUFPE)) {
            cpl_frameset_insert(arcs_raw, cpl_frame_duplicate(this_frame));
        }
    }

    /* Check we have some */
    nff = cpl_frameset_get_size(arcs_raw);
    if(nff < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_RAW_FIBRE_WAVE_SIMUFPE);
    }

    /* Switch to save processed products */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".keep");
    if(par != NULL)
        keep = cpl_parameter_get_bool(par);

    /* Retrieve ccdproc parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ccdproc.swapx");
    if(par != NULL)
        swapx_table = cpl_parameter_get_string(par);

    /* Retrieve imcombine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.combtype");
    if(par != NULL)
        combtype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.scaletype");
    if(par != NULL)
        scaletype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.xrej");
    if(par != NULL)
        xrej = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.thresh");
    if(par != NULL)
        combthresh = cpl_parameter_get_double(par);

    /* Retrieve scattered light removal parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.enable");
    if(par != NULL)
        scattered_enable = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.nbsize");
    if(par != NULL)
        scattered_nbsize = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.width");
    if(par != NULL)
        extract_width = cpl_parameter_get_int(par);

    /* Retrieve doarcs parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.detthr");
    if(par != NULL)
        detthr = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.matchwindow");
    if(par != NULL)
        matchwindow = cpl_parameter_get_double(par);

    /* Get first arc to use as reference */
    arc_ref = cpl_frameset_get_position(arcs_raw, 0);
    if(arc_ref == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first arc "
                                     "from SOF");
    }

    /* Get filename of first arc */
    arc_ref_filename = cpl_frame_get_filename(arc_ref);
    if(arc_ref_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first arc frame");
    }

    /* Extract the primary FITS header of the reference */
    arc_hdr = cpl_propertylist_load(arc_ref_filename,
                                    0);
    if(arc_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(arc_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    cpl_propertylist_delete(arc_hdr);
    arc_hdr = NULL;

    /* Get fibinfo */
    if(qmost_fibtabload(arc_ref,
                        spec,
                        &fibinfo_tbl,
                        &fibinfo_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO table "
                                     "from first input file");
    }

    /* Create outputs */
    linelist_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_FPE_LINELIST,
        CPL_FRAME_TYPE_TABLE,
        NULL);
    if(linelist_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_FPE_LINELIST);
    }

    linelist_filename = cpl_frame_get_filename(linelist_frame);

    if(keep) {
        proc_arc_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PROC_FIBRE_WAVE,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(proc_arc_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PROC_FIBRE_WAVE);
        }

        proc_arc_filename = cpl_frame_get_filename(proc_arc_frame);

        extracted_arc_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PRO_FPE_SPECTRUM,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(extracted_arc_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PRO_FPE_SPECTRUM);
        }
        
        extracted_arc_filename = cpl_frame_get_filename(extracted_arc_frame);
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        arc_hdr = cpl_propertylist_load(arc_ref_filename,
                                        raw_extension);
        if(arc_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(arc_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(arc_hdr,
                                   &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            if(proc_arc_frame != NULL) {
                if(qmost_dfs_save_image_and_var(proc_arc_frame,
                                                arc_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "stacked arc frame %s[%s]",
                                                 proc_arc_filename,
                                                 arm_extname);
                }
            }

            if(extracted_arc_frame != NULL) {
                if(qmost_dfs_save_image_and_var(extracted_arc_frame,
                                                arc_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "extracted arc %s[%s]",
                                                 extracted_arc_filename,
                                                 arm_extname);
                }
            }

            extname = cpl_sprintf("linelist_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format linelist "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_table_extension(linelist_frame,
                                              arc_hdr,
                                              extname,
                                              NULL,
                                              NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "linelist %s[%s]",
                                             linelist_filename,
                                             arm_extname);
            }

            cpl_free(extname);
            extname = NULL;
        
            cpl_propertylist_delete(arc_hdr);
            arc_hdr = NULL;

            continue;
        }

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(master_detflat_frame != NULL) {
            if(qmost_load_master_image_and_var(master_detflat_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_detflat_img,
                                               &master_detflat_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master detector "
                                             "flat frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_detflat_img = NULL;
            master_detflat_var = NULL;
        }

        if(master_bpm_frame != NULL) {
            if(qmost_load_master_mask(master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        if(master_lintab_frame != NULL) {
            if(qmost_load_master_table(master_lintab_frame,
                                       arm_extname,
                                       &master_lintab,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "linearity table");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* CCD processing and stacking */
        if(qmost_ccdproc_and_combine(arcs_raw,
                                     raw_extension,
                                     master_bpm,
                                     master_bias_img,
                                     master_bias_var,
                                     master_dark_img,
                                     master_dark_var,
                                     master_detflat_img,
                                     master_detflat_var,
                                     1,
                                     swapx_table,
                                     1,
                                     master_lintab,
                                     QMOST_LIN_AFTER_BIAS,
                                     combtype,
                                     scaletype,
                                     1,  /* fibre spectra */
                                     xrej,
                                     combthresh,
                                     0,
                                     0.0,
                                     0.0,
                                     &arc_img,
                                     &arc_var,
                                     qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "basic CCD processing failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_detflat_img != NULL) {
            cpl_image_delete(master_detflat_img);
            master_detflat_img = NULL;
        }

        if(master_detflat_var != NULL) {
            cpl_image_delete(master_detflat_var);
            master_detflat_var = NULL;
        }

        if(master_bpm != NULL) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        if(master_lintab != NULL) {
            cpl_table_delete(master_lintab);
            master_lintab = NULL;
        }

        /* Get trace table and fibre mask */
        extname = cpl_sprintf("trace_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format trace "
                                         "EXTNAME string");
        }
        
        if(qmost_load_master_table(fibre_trace_frame,
                                   extname,
                                   &trace_tbl,
                                   &trace_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load trace");
        }
        
        cpl_free(extname);
        extname = NULL;

        extname = cpl_sprintf("mask_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format fibre mask "
                                         "EXTNAME string");
        }

        if(qmost_load_master_mask(fibre_mask_frame,
                                  extname,
                                  &fibre_mask,
                                  NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load fibre mask");
        }
        
        cpl_free(extname);
        extname = NULL;

        /* Populate scattered light QC before we remove it */
        if(qmost_scattered_qc(arc_img,
                              qclist,
                              fibre_mask,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "scattered light QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(scattered_enable) {
            /* Remove scattered light */
            if(qmost_scattered(arc_img,
                               qclist,
                               fibre_mask,
                               scattered_nbsize,
                               1) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "removal of scattered light "
                                             "with box size %d failed "
                                             "for extension %lld",
                                             scattered_nbsize,
                                             raw_extension);
            }
        }
        
        /* Save stacked, scattered light corrected arc if requested */
        if(proc_arc_frame != NULL) {
            if(qmost_dfs_save_image_and_var(proc_arc_frame,
                                            arc_hdr,
                                            arm_extname,
                                            qclist,
                                            arc_img,
                                            arc_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save stacked arc "
                                             "frame %s[%s]",
                                             proc_arc_filename,
                                             arm_extname);
            }
        }

        /* We should only analyse the simucal fibres.  Using the
         * FIBINFO table, unset the live flag in the trace table for
         * anything else.  If there's no FIBINFO table assume all
         * fibres are illuminated by ThAr such as in lab data. */
        if(fibinfo_tbl != NULL) {
            nrowsf = cpl_table_get_nrow(fibinfo_tbl);
            nrowst = cpl_table_get_nrow(trace_tbl);

            nrows = qmost_min(nrowsf, nrowst);

            for(irow = 0; irow < nrows; irow++) {
                fibuse = cpl_table_get_int(fibinfo_tbl,
                                           "FIB_USE",
                                           irow,
                                           &isnull);
                if(isnull < 0) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "could not read FIB_USE "
                                                 "for FIBINFO row %d",
                                                 irow+1);
                }
                else if(isnull == 0 && fibuse > 0) {
                    if(cpl_table_set_int(trace_tbl,
                                         "fiblive",
                                         irow,
                                         0) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "could not set fiblive "
                                                     "in trace table row %d",
                                                     irow+1);
                    }
                }
            }
        }

        /* Extract arc */
        if(qmost_extract_tram(arc_img,
                              arc_var,
                              qclist,
                              trace_tbl,
                              trace_hdr,
                              extract_width,
                              &extracted_arc_img,
                              &extracted_arc_var,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectral extraction failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        /* Populate QC */
        if(qmost_extract_qc(extracted_arc_img,
                            extracted_arc_var,
                            qclist,
                            NULL,
                            arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "extracted spectrum QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(extracted_arc_frame != NULL) {
            /* Save extracted arc */
            if(qmost_dfs_save_image_and_var(extracted_arc_frame,
                                            arc_hdr,
                                            arm_extname,
                                            qclist,
                                            extracted_arc_img,
                                            extracted_arc_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save extracted "
                                             "arc %s[%s]",
                                             extracted_arc_filename,
                                             arm_extname);
            }
        }

        /* Load master wavelength solution */
        extname = cpl_sprintf("wave_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format wave "
                                         "EXTNAME string");
        }
        
        if(qmost_load_master_table(simuarc_wave_frame,
                                   extname,
                                   &simuarc_wave_tbl,
                                   &simuarc_wave_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load master "
                                         "wavelength solution");
        }
        
        cpl_free(extname);
        extname = NULL;

        if(qmost_fpmeasure(extracted_arc_img,
                           qclist,
                           simuarc_wave_tbl,
                           simuarc_wave_hdr,
                           detthr,
                           spec == QMOST_SPEC_HRS ?
                           matchwindow / 3.0 :
                           matchwindow,
                           &linelist_tbl,
                           qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "OB arc line analysis failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        cpl_table_delete(trace_tbl);
        trace_tbl = NULL;

        cpl_propertylist_delete(trace_hdr);
        trace_hdr = NULL;

        cpl_mask_delete(fibre_mask);
        fibre_mask = NULL;

        /* Save line list */
        extname = cpl_sprintf("linelist_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format linelist "
                                         "EXTNAME string");
        }

        if(qmost_dfs_save_table_extension(linelist_frame,
                                          arc_hdr,
                                          extname,
                                          qclist,
                                          linelist_tbl) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save linelist "
                                         "%s[%s]",
                                         linelist_filename,
                                         arm_extname);
        }

        cpl_free(extname);
        extname = NULL;

        /* Clean up */
        cpl_image_delete(arc_img);
        arc_img = NULL;

        cpl_image_delete(arc_var);
        arc_var = NULL;

        cpl_propertylist_delete(arc_hdr);
        arc_hdr = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        cpl_table_delete(simuarc_wave_tbl);
        simuarc_wave_tbl = NULL;

        cpl_propertylist_delete(simuarc_wave_hdr);
        simuarc_wave_hdr = NULL;

        cpl_table_delete(linelist_tbl);
        linelist_tbl = NULL;

        cpl_image_delete(extracted_arc_img);
        extracted_arc_img = NULL;

        cpl_image_delete(extracted_arc_var);
        extracted_arc_var = NULL;
    }

    if(proc_arc_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                proc_arc_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to processed "
                                             "arc frame %s",
                                             proc_arc_filename);
            }
        }
    }

    if(extracted_arc_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                extracted_arc_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to extracted "
                                             "arc frame %s",
                                             extracted_arc_filename);
            }
        }
    }

    if(fibinfo_tbl != NULL) {
        if(qmost_fibtabsave(fibinfo_tbl,
                            fibinfo_hdr,
                            linelist_frame) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save FIBINFO table "
                                         "to linelist frame %s",
                                         linelist_filename);
        }
    }

    /* Clean up */
    cpl_frameset_delete(arcs_raw);
    arcs_raw = NULL;

    if(fibinfo_tbl != NULL) {
        cpl_table_delete(fibinfo_tbl);
        fibinfo_tbl = NULL;
    }

    if(fibinfo_hdr != NULL) {
        cpl_propertylist_delete(fibinfo_hdr);
        fibinfo_hdr = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_fpe_analyse_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* Switch to save processed products */
    par = cpl_parameter_new_value(RECIPE_NAME".keep",
                                  CPL_TYPE_BOOL,
                                  "Save optional processed products",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "keep");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* ccdproc parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".ccdproc.swapx",
                                  CPL_TYPE_STRING,
                                  "String of 0 or 1 specifying for each "
                                  "arm of each spectrograph if we should "
                                  "flip the x axis to correct the "
                                  "wavelength order of the spectral axis.",
                                  RECIPE_NAME,
                                  QMOST_DEFAULT_SWAPX_TABLE);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ccdproc.swapx");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* imcombine parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.combtype",
                                  CPL_TYPE_INT,
                                  "Stacking method when combining multiple "
                                  "images.  1: mean, 2: median.",
                                  RECIPE_NAME,
                                  QMOST_MEANCALC);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.combtype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.scaletype",
                                  CPL_TYPE_INT,
                                  "Scaling to use when combining multiple "
                                  "images.  0: no biasing or scaling, "
                                  "1: bias by relative offset in background "
                                  "levels, 2: scale by ratio of background "
                                  "levels, 3: scale by exposure time and "
                                  "bias by relative offset in background.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.scaletype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.xrej",
                                  CPL_TYPE_BOOL,
                                  "Do extra rejection cycle when combining "
                                  "images.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.xrej");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold in terms of "
                                  "background noise for rejection when "
                                  "combining images.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Scattered light removal parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".scattered.enable",
                                  CPL_TYPE_BOOL,
                                  "Enable scattered light removal",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.enable");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".scattered.nbsize",
                                  CPL_TYPE_INT,
                                  "Size of the background smoothing cells "
                                  "for determining scattered light in pixels.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.nbsize");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Extraction parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".extract.width",
                                  CPL_TYPE_INT,
                                  "Boxcar width in pixels over which the "
                                  "summation will be done at each spectral "
                                  "pixel during extraction.",
                                  RECIPE_NAME,
                                  5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.width");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);
    
    /* Arc analyse parameters */  
    par = cpl_parameter_new_value(RECIPE_NAME".arc.detthr",
                                  CPL_TYPE_DOUBLE,
                                  "Detection threshold used to find emission "
                                  "features in the extracted arc spectrum in "
                                  "units of the background sigma.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.detthr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".arc.matchwindow",
                                  CPL_TYPE_DOUBLE,
                                  "Maximum deviation between the "
                                  "measured positions of an arc "
                                  "line between fibres for it to "
                                  "be considered the same line, in "
                                  "units of the spectral resolution "
                                  "element.",
                                  RECIPE_NAME,
                                  1.5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.matchwindow");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/**@}*/
