/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_dataset.h"
#include "sph_dataset_stats.h"
#include "sph_ifs_instrument_flat.h"
#include "sph_ifs_lenslet_model.h"
#include "sph_master_frame.h"
#include "sph_framecombination.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ifs_tags.h"
#include "sph_ifs_keywords.h"
#include "sph_ldt.h"
#include "sph_keyword_manager.h"
#include "sph_create_flat.h"
#include "sph_create_super_flat.h"

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IFS_INSTRUMENT_FLAT_GENERAL;
extern sph_error_code SPH_IFS_INSTRUMENT_FLAT_FRAMES_MISSING;

cpl_table*
sph_ifs_instrument_flat_create_ifu_tab(sph_ldt* ldt);

static sph_master_frame*
sph_ifs_ins_flat(sph_ifs_instrument_flat* self, cpl_frameset* rawframes,
        sph_master_frame* dark, cpl_mask* bads, cpl_image** linbads,
        cpl_vector** flux, cpl_vector** counts, cpl_vector** flux_stdev) {
    sph_master_frame* temp_master = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    if (self->nofit) {
        if (self->make_badpix) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Can not create badpixels when the --nofit option is selected.");
        } else {
            double median = 0.0;

            temp_master = sph_framecombination_master_frame_from_cpl_frameset(
                    self->rawframes, self->coll_alg,
                    self->framecomb_parameterlist);
            if (dark) {
                sph_master_frame_subtract_master_frame(temp_master, dark);
            }
            if (bads) {
                sph_master_frame_set_bads_from_mask(temp_master, bads);
            }
            median = sph_master_frame_get_mean(temp_master, NULL);
            sph_master_frame_divide_double(temp_master, median);
            sph_master_frame_mask_tolerance(temp_master,
                    self->badpix_lowtolerance, self->badpix_uptolerance);
        }
    } else {
        temp_master = sph_create_flat(rawframes, self->robust_fit, bads, dark,
                linbads, self->badpix_lowtolerance, self->badpix_uptolerance,
                self->badpix_chisqtolerance,
                flux,
                counts,
                0,
                flux_stdev);
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return temp_master;
}

cpl_table*
sph_ifs_instrument_flat_create_ifu_tab(sph_ldt* ldt) {
    cpl_table* tab = NULL;
    int ll = 0;
    double val = 0.0;
    double rms = 0.0;
    int bpix = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;cpl_ensure(ldt,
            CPL_ERROR_NULL_INPUT, NULL);

    tab = cpl_table_new(ldt->nlens);
    cpl_table_new_column(tab, SPH_IFS_INSTRUMENT_MODEL_LENSID, CPL_TYPE_INT);
    cpl_table_new_column(tab, SPH_IFS_INSTRUMENT_MODEL_FLATVAL,
            CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, SPH_IFS_INSTRUMENT_MODEL_FLATBADPIX,
            CPL_TYPE_INT);
    cpl_table_new_column(tab, SPH_IFS_INSTRUMENT_MODEL_FLATRMS,
            CPL_TYPE_DOUBLE);

    for (ll = 0; ll < ldt->nlens; ++ll) {
        if (ldt->arr[ll]) {
            bpix = sph_lenslet_descriptor_get_overall_bad(ldt->arr[ll]);
            if (bpix == 0) {
                val = sph_lenslet_descriptor_get_median(ldt->arr[ll], &rms);
            } else {
                val = 0.0;
                rms = SPH_MASTER_FRAME_BAD_RMS;
            }
            cpl_table_set(tab, SPH_IFS_INSTRUMENT_MODEL_LENSID, ll,
                    ldt->arr[ll]->lensid);
            cpl_table_set(tab, SPH_IFS_INSTRUMENT_MODEL_FLATVAL, ll, val);
            cpl_table_set(tab, SPH_IFS_INSTRUMENT_MODEL_FLATBADPIX, ll, bpix);
            cpl_table_set(tab, SPH_IFS_INSTRUMENT_MODEL_FLATRMS, ll, rms);
        }
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return tab;
}

static sph_error_code sph_ifs_instrument_flat_save_ifuflat(const char* filename,
        cpl_frameset* inframes, cpl_parameterlist* inparams,
        sph_pixel_description_table* pdt, sph_ifs_lenslet_model* model,
        sph_master_frame* insflat, cpl_propertylist* pli) {
    cpl_propertylist* ins_plist = NULL;
    cpl_image* nonlin_image = NULL;
    cpl_mask* nonlin_mask = NULL;
    sph_master_frame* ifu_flat = NULL;
    sph_ldt* ldt = NULL;
    cpl_table* tab = NULL;
    double rms = 0.0;

    cpl_ensure_code(pdt, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(model, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(insflat, CPL_ERROR_NULL_INPUT);
    ins_plist = cpl_propertylist_new();

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Creating IFU flat using PDT...");
    ldt = sph_ldt_new_from_pdt_master_frame(pdt, sph_ifs_lenslet_model_duplicate(model), insflat);
    cpl_ensure_code(ldt, cpl_error_get_code());
    ifu_flat = sph_ldt_collapse_interpolate(ldt, 0.0, 1.0);
    cpl_ensure_code(ifu_flat, cpl_error_get_code());

    tab = sph_ifs_instrument_flat_create_ifu_tab(ldt);
    cpl_ensure_code(tab, cpl_error_get_code());

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Saving...");
    SPH_RAISE_CPL
    sph_ldt_delete(ldt);
    ldt = NULL;
    cpl_propertylist_append_double(ins_plist,
            SPH_COMMON_KEYWORD_FLAT_MEAN_COUNT,
            sph_master_frame_get_mean(ifu_flat, &rms));
    cpl_propertylist_append_double(ins_plist, SPH_COMMON_KEYWORD_FLAT_FPN, rms);
    cpl_propertylist_append_int(ins_plist, SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
            sph_master_frame_get_nbads(ifu_flat));
    nonlin_image = sph_master_frame_get_rms(ifu_flat);
    nonlin_mask = sph_master_frame_get_badpixelmask(ifu_flat);
    cpl_image_reject_from_mask(nonlin_image, nonlin_mask);
    cpl_propertylist_append_double(ins_plist,
            SPH_COMMON_KEYWORD_FLAT_NONLIN_FACTOR,
            cpl_image_get_mean(nonlin_image));
    cpl_image_delete(nonlin_image);
    nonlin_image = NULL;
    cpl_mask_delete(nonlin_mask);
    nonlin_mask = NULL;
    sph_utils_simple_copy_singular(pli,ins_plist);
    sph_master_frame_save_dfs(ifu_flat, filename, inframes, NULL, inparams,
            SPH_IFS_TAG_IFU_FLAT_CALIB, SPH_RECIPE_NAME_IFS_IFF,
            SPH_PIPELINE_NAME_IFS, ins_plist);

    /* Remove non-table FITS cards */
    cpl_propertylist_erase_regexp(ins_plist, "^(BZERO|BSCALE|BUNIT|BLANK"
                                  "|DATAMIN|DATAMAX)$", 0);
    /* Make any final header updates */
    sph_utils_update_header(ins_plist);
    cpl_table_save(tab, NULL, ins_plist, filename, CPL_IO_EXTEND);
    cpl_propertylist_delete(ins_plist);
    ins_plist = NULL;
    sph_master_frame_delete(ifu_flat);
    ifu_flat = NULL;
    cpl_table_delete(tab);
    tab = NULL;
    return cpl_error_get_code();
}
static sph_error_code sph_ifs_instrument_flat_save_insflat(
        sph_ifs_instrument_flat* self, sph_master_frame* insflat,
        cpl_propertylist* ins_plist) {
    cpl_image* nonlin_image = NULL;
    cpl_mask* nonlin_mask = NULL;
    double rms = 0.0;

    cpl_propertylist_append_double(ins_plist,
            SPH_COMMON_KEYWORD_FLAT_MEAN_COUNT,
            sph_master_frame_get_mean(insflat, &rms));
    cpl_propertylist_append_double(ins_plist, SPH_COMMON_KEYWORD_FLAT_FPN, rms);
    cpl_propertylist_append_int(ins_plist, SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
            sph_master_frame_get_nbads(insflat));
    nonlin_image = sph_master_frame_get_rms(insflat);
    nonlin_mask = sph_master_frame_get_badpixelmask(insflat);
    cpl_image_reject_from_mask(nonlin_image, nonlin_mask);
    cpl_propertylist_append_double(ins_plist,
            SPH_COMMON_KEYWORD_FLAT_NONLIN_FACTOR,
            cpl_image_get_mean(nonlin_image));
    cpl_image_delete(nonlin_image);
    nonlin_image = NULL;
    cpl_mask_delete(nonlin_mask);
    nonlin_mask = NULL;
    sph_master_frame_save_dfs(insflat, self->instrument_flat_filename,
            self->inframes, NULL, self->inparams, SPH_IFS_TAG_FLAT_CALIB,
            SPH_RECIPE_NAME_IFS_IFF, SPH_PIPELINE_NAME_IFS, ins_plist);
    return cpl_error_get_code();
}

static sph_master_frame*
sph_ifs_ifu_flat(sph_pixel_description_table* wpdt, sph_master_frame* insflat,
        sph_master_frame* ldff1, sph_master_frame* ldff2,
        sph_master_frame* ldff3, sph_master_frame* ldff4,
        sph_ifs_lenslet_model* model, double ditherx, double dithery) {
    sph_master_frame* temp_master = NULL;
    sph_master_frame* result = NULL;
    cpl_image* ill = NULL;

    int			used_ins_flat = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    temp_master = sph_create_super_flat(wpdt, ldff1, ldff2, ldff3, ldff4);

    if ( !temp_master ) {
    	SPH_ERROR_RAISE_WARNING(cpl_error_get_code(),"Could not create super flat. Am using the instrument flat instead.");
        result = sph_master_frame_duplicate(insflat);
    	cpl_error_reset();
    	used_ins_flat = 1;
    }
    else {
    	result = sph_master_frame_duplicate(insflat);
    	sph_master_frame_divide_master_frame(result, temp_master);
    	sph_master_frame_delete(temp_master);
    	temp_master = NULL;
    }
    if (model) {
        SPH_INFO_MSG("Dividing by illumination pattern...");
        sph_ifs_lenslet_model_create_pdt_images(model, NULL, NULL, &ill, NULL,
                ditherx, dithery);
        temp_master = sph_master_frame_new_from_cpl_image(ill);
        if (result) {
            sph_master_frame_divide_master_frame(result, temp_master);
        }
        sph_master_frame_delete(temp_master);
        temp_master = NULL;
    } else {
        SPH_INFO_MSG(
                "Am not dividing illumination pattern. The result may contain artefacts (usually a 'wavy' structure).");
    }
    if (result && !used_ins_flat ) {
        sph_master_frame_quality_check(result);
        sph_master_frame_divide_double(result,
                sph_master_frame_get_mean(result, NULL));
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_instrument_flat_run Run the Instrument Flat Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * instrument flat recipe for IFS.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ifs_instrument_flat.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ifs_instrument_flat_run(sph_ifs_instrument_flat* self) {
    const cpl_frame* background_dark_frame
        = self->master_background_frame != NULL
        ? self->master_background_frame
        : self->master_dark_frame;
    sph_master_frame* dark = NULL;
    sph_master_frame* sdffbb = NULL;
    sph_master_frame* preamp_flat = NULL;
    sph_master_frame* ldff1 = NULL;
    sph_master_frame* ldff2 = NULL;
    sph_master_frame* ldff3 = NULL;
    sph_master_frame* ldff4 = NULL;
    sph_master_frame* ldffbb = NULL;
    sph_master_frame* insflat = NULL;
    sph_master_frame* ifuflat = NULL;
    cpl_image* linbad = NULL;
    cpl_mask* badpix = NULL;
    cpl_propertylist* ins_plist = NULL;
    cpl_propertylist* pli = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    cpl_propertylist* plist = NULL;
    cpl_vector* flux = NULL;
    cpl_vector* counts = NULL;
    cpl_vector* flux_stdev = NULL;
    double dx = 0.0;
    double dy = 0.0;


	pli = cpl_propertylist_load(
	                    cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
	                    0); // read template property list of first frame to copy singular keys later

    if (self->robust_fit) {
        if (cpl_frameset_get_size(self->rawframes) < 5) {
            SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_INPUT,
                    "The robust fitting is enabled. "
                    "For this to work, a minimum of 5 raw frames needs"
                    "to be provided. Switching robust fitting off.");
            self->robust_fit = 0;
        }
    }
    if (!self->nofit) {
        if (cpl_frameset_get_size(self->rawframes) < 2) {
            SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_INPUT,
                    "Only one input frame provided, "
                    "setting --nofit to TRUE.");
            self->nofit = 1;
        }
    }

    if (background_dark_frame) {
        dark = sph_master_frame_load_(background_dark_frame, 0);
    }
    if (self->preamp_flat) {
        preamp_flat = sph_master_frame_load_(self->preamp_flat, 0);
    }
    if (self->master_dff_short_frame) {
        sdffbb = sph_master_frame_load_(self->master_dff_short_frame, 0);
    }
    ins_plist = cpl_propertylist_new();

    if (self->wave_calib_frame == NULL) {
        SPH_INFO_MSG(
                "The recipe is called without a wavelength calibration frame "
                "and is hence executed in instrument flat rather than IFU flat mode");
        if (self->specpos_frame) {
            pdt = sph_pixel_description_table_load(
                    cpl_frame_get_filename(self->specpos_frame));
        } else {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Can not process flat since I "
                            "have neither a spectra position frame, nor a wavelength calibration frame."
                            "I need at least one of these.");
            if (dark) {
                sph_master_frame_delete(dark);
            }
            return SPH_ERROR_GENERAL;
        }
        if (pdt) {
            badpix = sph_pixel_description_table_get_mask(pdt);
            sph_pixel_description_table_delete(pdt);
            pdt = NULL;
        }

        if (!badpix) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not extract the badpixel mask for the spectra.");
        } else {
            cpl_mask_not(badpix);
        }
        if (self->make_badpix) {
            insflat = sph_ifs_ins_flat(self, self->rawframes, dark, badpix,
                    &linbad, &flux, &counts, &flux_stdev);
            sph_master_frame_divide_double(insflat,
                    sph_master_frame_get_median(insflat, NULL));
        } else {
            insflat = sph_ifs_ins_flat(self, self->rawframes, dark, badpix,
                    NULL, &flux, &counts, &flux_stdev);
            sph_master_frame_divide_double(insflat,
                    sph_master_frame_get_median(insflat, NULL));
        }

        if (insflat) {
            if (preamp_flat) {
                sph_master_frame_divide_master_frame(insflat, preamp_flat);
            }
            if (sdffbb) {
                sph_master_frame_divide_master_frame(insflat, sdffbb);
            }
            if (counts) {
                sph_keyword_manager_vector2proplist(ins_plist, counts,
                        SPH_COMMON_KEYWORD_FLAT_LAMP_COUNTS);
                cpl_vector_delete(counts);
                counts = NULL;
            }
            if (flux) {
                sph_keyword_manager_vector2proplist(ins_plist, flux,
                        SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX);
                cpl_vector_delete(flux);
                flux = NULL;
            }
            if (flux_stdev) {
                sph_keyword_manager_vector2proplist(ins_plist, flux_stdev,
                        SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX_STDEV);
                cpl_vector_delete(flux_stdev);
                flux_stdev = NULL;
            }
            sph_utils_simple_copy_singular(pli,ins_plist);
            cpl_ensure(
                    sph_ifs_instrument_flat_save_insflat(self,insflat,ins_plist) == CPL_ERROR_NONE,
                    cpl_error_get_code(), cpl_error_get_code());
        }
        if (linbad) {
            cpl_propertylist_append_double(ins_plist,
                    SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                    cpl_image_get_flux(linbad));
            cpl_propertylist_append_string(ins_plist,
                    SPH_COMMON_KEYWORD_PRO_CATG,
                    SPH_IFS_TAG_NON_LINEAR_PIXELMAP_CALIB);
            cpl_dfs_save_image(self->inframes, NULL, self->inparams,
                    self->inframes, NULL, linbad, CPL_BPP_8_UNSIGNED,
                    SPH_RECIPE_NAME_IFS_IFF, ins_plist, NULL,
                    SPH_PIPELINE_NAME_IFS, self->badpix_filename);
            cpl_image_delete(linbad);
            linbad = NULL;
        }
        cpl_mask_delete(badpix);
        badpix = NULL;
        sph_master_frame_delete(insflat);
        insflat = NULL;
    } else {
        SPH_INFO_MSG("The recipe is called with a wavelength calibration frame "
        "and is hence executed in IFU flat mode");

        plist = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(self->wave_calib_frame), 0);

        lensmodel = sph_ifs_lenslet_model_new_from_propertylist(plist);

        cpl_propertylist_delete(plist);
        plist = NULL;

        pdt = sph_pixel_description_table_load(
                cpl_frame_get_filename(self->wave_calib_frame));
        cpl_ensure_code(pdt, cpl_error_get_code());

        badpix = sph_pixel_description_table_get_mask(pdt);
        cpl_mask_not(badpix);
        insflat = sph_ifs_ins_flat(self, self->rawframes, dark, badpix, NULL,
                &flux, &counts, &flux_stdev);
        cpl_ensure(insflat, cpl_error_get_code(), cpl_error_get_code());
        sph_master_frame_divide_double(insflat,
                sph_master_frame_get_median(insflat, NULL));

        if (flux) {
            sph_keyword_manager_vector2proplist(ins_plist, flux,
                    SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX);
            cpl_vector_delete(flux);
            flux = NULL;
        }
        if (counts) {
            sph_keyword_manager_vector2proplist(ins_plist, counts,
                    SPH_COMMON_KEYWORD_FLAT_LAMP_COUNTS);
            cpl_vector_delete(counts);
            counts = NULL;
        }
        if (flux_stdev) {
            sph_keyword_manager_vector2proplist(ins_plist, flux_stdev,
                    SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX_STDEV);
            cpl_vector_delete(flux_stdev);
            flux_stdev = NULL;
        }
        sph_utils_simple_copy_singular(pli,ins_plist);
        cpl_ensure(
                sph_ifs_instrument_flat_save_insflat(self,insflat,ins_plist) == CPL_ERROR_NONE,
                cpl_error_get_code(), cpl_error_get_code());

        ldff1 = sph_master_frame_load_(self->master_dff_long1_frame, 0);
        ldff2 = sph_master_frame_load_(self->master_dff_long2_frame, 0);
        ldff3 = sph_master_frame_load_(self->master_dff_long3_frame, 0);
        if (self->master_dff_long4_frame) {
            ldff4 = sph_master_frame_load_(self->master_dff_long4_frame, 0);
        }
        if ( self->master_dff_longbb_frame ) {
            ldffbb = sph_master_frame_load_(self->master_dff_longbb_frame, 0);
        }
        if (ldff1 && ldff2 && ldff3) {
            if (self->use_illum) {
                plist = sph_keyword_manager_load_properties(
                        cpl_frame_get_filename(
                                cpl_frameset_get_first(self->rawframes)), 0);
                if (cpl_propertylist_has(plist, SPH_COMMON_KEYWORD_SPH_DITHERX)
                        && cpl_propertylist_has(plist,
                                SPH_COMMON_KEYWORD_SPH_DITHERY)) {
                    dx = cpl_propertylist_get_double(plist,
                            SPH_COMMON_KEYWORD_SPH_DITHERX);
                    dy = cpl_propertylist_get_double(plist,
                            SPH_COMMON_KEYWORD_SPH_DITHERY);

                    ifuflat = sph_ifs_ifu_flat(pdt, insflat, ldff1, ldff2,
                            ldff3, ldff4, lensmodel, dx, dy);
                } else {
                    SPH_ERROR_RAISE_ERR(
                            CPL_ERROR_ILLEGAL_INPUT,
                            "Could not find diterhing keywords in raw frame(s).");
                }
            } else {
                ifuflat = sph_ifs_ifu_flat(pdt, insflat, ldff1, ldff2, ldff3,
                        ldff4, NULL, 0.0, 0.0);
            }

        } else {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "Am missing a (or all) important calibration flats.");
        }
        sph_master_frame_delete(ldff1);
        ldff1 = NULL;
        sph_master_frame_delete(ldff2);
        ldff2 = NULL;
        sph_master_frame_delete(ldff3);
        ldff3 = NULL;
        sph_master_frame_delete(ldff4);
        ldff4 = NULL;
        sph_master_frame_delete(ldffbb);
        ldffbb = NULL;
        sph_master_frame_delete(insflat);
        insflat = NULL;

        if (ifuflat) {
            sph_ifs_instrument_flat_save_ifuflat(self->ifu_filename,
                    self->inframes, self->inparams, pdt, lensmodel, ifuflat, pli);
            sph_master_frame_delete(ifuflat);
            ifuflat = NULL;
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Done with IFU flat.");
        }
        sph_pixel_description_table_delete(pdt);
        pdt = NULL;
        cpl_mask_delete(badpix);
        badpix = NULL;
        sph_master_frame_delete(insflat);
        insflat = NULL;
    }
    if (dark) {
        sph_master_frame_delete(dark);
        dark = NULL;
    }
    sph_master_frame_delete(sdffbb);
    sdffbb = NULL;
    sph_master_frame_delete(preamp_flat);
    preamp_flat = NULL;
    if(pli){
    	cpl_propertylist_delete(pli);
    	pli = NULL;
    }
    cpl_propertylist_delete(ins_plist);
    ins_plist = NULL;
    sph_ifs_lenslet_model_delete(lensmodel);
    lensmodel = NULL;
    return cpl_error_get_code();
}

/**@}*/
