/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_ifs_science_dr.h"
#include "sph_common_keywords.h"
#include "sph_ifs_keywords.h"
#include "sph_dataset.h"
#include "sph_dataset_stats.h"
#include "sph_ifs_simple_adi_common.h"
//#include "sph_ifs_spec_deconv.h"
#include "sph_spec_deconv.h"
#include "sph_extract_angles.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ifs_tags.h"
#include "sph_ldt.h"
#include "sph_framecombination.h"
#include "sph_framesorter.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_ifs_science.h"
#include "sph_frame_validator.h"
#include "sph_transform.h"
#include "sph_common_science.h"
#include "sph_raw_image_corrector_data.h"

#include <cpl.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IFS_SCIENCE_DR_GENERAL;
extern sph_error_code SPH_IFS_SCIENCE_DR_FRAMES_MISSING;

static cpl_error_code
sph_ifs_science_dr_find_max_angle(cpl_frameset* inframes, double* maxangle);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_science_dr_run Run method for the IFS science_dr recipe
 * @par
 * This module provides the main run method for the IFS science recipe.
 * The science_dr recipe is the core recipe for IFS, and one of a few recipes
 * that transform the detector data into wavelength cubes.
 * For that purpose it makes use of the shared sph_ifs_science.c module.
 * @par
 * In its current implementation the recipe has no other responsibility,
 * that to reduce the raw data and create wavelength cubes for every
 * input image (if dithering is on) or for every input file (if dithering is off).
 * Please see the sph_ifs_science.c module for more information.
 * @par
 * The recipe performs no frame combination. For IFS the responsibility for
 * combining frames is shifted to other higher level recipes,
 * e.g. sph_ifs_simple_adi.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_error_code sph_ifs_science_dr_run(sph_ifs_science_dr* self) {
    const cpl_frame* background_dark_frame
        = self->master_background_frame != NULL
        ? self->master_background_frame
        : self->master_dark_frame;
    sph_ifs_lenslet_model* lensmodel = NULL;
    cpl_propertylist* plist = NULL;
    sph_frame_validator_result fcheck = SPH_FRAME_VALIDATOR_ERROR;
    cpl_frameset* result_frameset = NULL;
    cpl_frame* arawframe = NULL;
    double maxangle = 0.0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    fcheck = sph_frame_validator_warn_keyword_tolerance(self->inframes, 0,
            SPH_IFS_KEYWORD_BENCH_TEMP, 0.5, NULL);
    if (fcheck == SPH_FRAME_VALIDATOR_NOT_VALIDATED) {
        SPH_ERROR_RAISE_WARNING(CPL_ERROR_INCOMPATIBLE_INPUT,
                "The frames provided do not all have the same "
                "bench temperature. Check your raw frames/detector to "
                "ensure all is OK."
                "Will proceed with recipe execution anyway...");
    }
    if (fcheck == SPH_FRAME_VALIDATOR_ERROR) {
        SPH_ERROR_RAISE_WARNING(
                CPL_ERROR_INCOMPATIBLE_INPUT,
                "The frames provided do not have the "
                "%s keyword. Check your raw frames/detector to "
                "ensure all is OK. "
                "Will proceed with recipe execution anyway...", SPH_IFS_KEYWORD_BENCH_TEMP);
    }

    plist = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(self->wave_calib_frame), 0);
//    SPH_INFO_MSG("PIPE-6081: @ L128 of sph_ifs_science_dr_run, plist from self->wave_calib_frame contains:");
//    if (cpl_propertylist_has(plist, "DATE-OBS")) {
//        printf("DATE-OBS: %s\n", cpl_property_get_string(cpl_propertylist_get_property(plist, "DATE-OBS")));
//    }
//    if (cpl_propertylist_has(plist, "ESO DET SEQ UTC")) {
//        printf("ESO DET SEQ UTC: %s\n", cpl_property_get_string(cpl_propertylist_get_property(plist, "ESO DET SEQ UTC")));
//    }
//    printf("Press ENTER to continue...");
//    getchar();

    lensmodel = sph_ifs_lenslet_model_new_from_propertylist(plist);

//    SPH_INFO_MSG("PIPE-6081: @ L138 of sph_ifs_science_dr_run, plist from self->wave_calib_frame contains:");
//    if (cpl_propertylist_has(plist, "DATE-OBS")) {
//        printf("DATE-OBS: %s\n", cpl_property_get_string(cpl_propertylist_get_property(plist, "DATE-OBS")));
//    }
//    if (cpl_propertylist_has(plist, "ESO DET SEQ UTC")) {
//        printf("ESO DET SEQ UTC: %s\n", cpl_property_get_string(cpl_propertylist_get_property(plist, "ESO DET SEQ UTC")));
//    }
//    printf("Press ENTER to continue...");
//    getchar();

    cpl_propertylist_delete(plist);
    plist = NULL;

    {
    	sph_raw_image_corrector_data corrector_data;
    	corrector_data.badpixco_apply = self->badpixcoapply;
    	corrector_data.crosstalkco_apply = self->xtalkapply;
    	corrector_data.largescalecrosstalkco_apply = self->xtalklsapply;
    	corrector_data.badpixco_border = self->badpix_border;
    	corrector_data.badpixco_thresh_absolute = self->badpix_threshold;
    	corrector_data.badpixco_thresh_factor = self->badpix_fac;
    	corrector_data.crosstalkco_bfac = self->bfac;
    	corrector_data.crosstalkco_lg_scale_corr_window = self->lgscalewin;
    	corrector_data.crosstalkco_powfac = self->powfac;
    	corrector_data.crosstalkco_sepmax = self->sepmax;
    	corrector_data.crosstalkco_smoothing_fwhm = self->smoothing;
    	corrector_data.crosstalkco_stephist = self->stephist;
    	corrector_data.crosstalkco_threshold = self->threshold;

        result_frameset = sph_ifs_science_reduce_cubes(self->raw_science_frameset,
                self->inframes, self->inparams, lensmodel, background_dark_frame,
                self->static_badpixel_frame, self->wave_calib_frame,
                self->preamp_flat, self->master_ifu_flat_frame,
                self->lenslet_distortion_frame, self->master_dff_long1_frame,
                self->master_dff_long2_frame, self->master_dff_long3_frame,
                self->master_dff_long4_frame, self->master_dff_longbb_frame,
                self->master_dff_short_frame, SPH_RECIPE_NAME_IFS_SCIENCE_DR,
                SPH_IFS_TAG_SCIENCE_DR_CALIB, self->use_illum,
				&corrector_data);
    }

//    cpl_propertylist * test_plist;
//    SPH_INFO_MSG("PIPE-6081 - Result frameset at L181 of sph_ifs_science_dr_run:");
//    cpl_frameset_dump(result_frameset, stdout);
//    arawframe = cpl_frameset_get_first(result_frameset);
//    while (arawframe) {
//        cpl_frame_dump(arawframe, stdout);
//        test_plist = cpl_propertylist_load(cpl_frame_get_filename(arawframe), 0);
//        if (cpl_propertylist_has(test_plist, "DATE-OBS")) {
//            printf("DATE-OBS:        %s\n", cpl_property_get_string(cpl_propertylist_get_property(test_plist, "DATE-OBS")));
//        }
//        if (cpl_propertylist_has(test_plist, "ESO DET SEQ UTC")) {
//            printf("ESO DET SEQ UTC: %s\n", cpl_property_get_string(cpl_propertylist_get_property(test_plist, "ESO DET SEQ UTC")));
//        }
//        // Iterate
//        arawframe = cpl_frameset_get_next(result_frameset);
//    }
//    printf("Press ENTER to continue...");
//    getchar();

    if (self->spec_deconv) {
        arawframe = cpl_frameset_get_first(result_frameset);
        while (arawframe) {
            cpl_frame_set_group(arawframe, CPL_FRAME_GROUP_CALIB);
//            SPH_INFO_MSG("PIPE-6081 - CPL frame debug info after set_group call:");
//            cpl_frame_dump(arawframe, stdout);
//            printf("Press ENTER to continue...");
//            getchar();
            arawframe = cpl_frameset_get_next(result_frameset);
        }

        SPH_ERROR_RAISE_INFO(CPL_ERROR_CONTINUE,
                "Applying spectra deconvolutions...");

        sph_spec_deconv(self->inframes, result_frameset, self->inparams,
                self->framecomb_parameterlist, SPH_IFS_TAG_SPEC_DECONV_CALIB,
                SPH_RECIPE_NAME_IFS_SCIENCE_DR, self->spec_deconv_filename, 5, 0, 0.0,
                0.0, 1.0, 2.0, result_frameset);

        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT
    }

    sph_ifs_science_dr_find_max_angle(result_frameset, &maxangle);

    if (self->use_adi == 1
            || (self->use_adi == 2
                    && maxangle * CPL_MATH_DEG_RAD > self->min_adi_angle)) {
        SPH_ERROR_RAISE_INFO(CPL_ERROR_CONTINUE, "Applying simple ADI...");
        sph_ifs_simple_adi_common(self->inframes, result_frameset,
                self->inparams, SPH_IFS_TAG_SIMPLE_ADI_CALIB,
                SPH_RECIPE_NAME_IFS_SCIENCE_DR, self->science_dr_filename,
                self->coll_alg, SPH_TRANSFORM_METHOD_FFT, 0.0, 0.0, 0.0, 0.0,
                0.0);
    }
    EXIT: cpl_frameset_delete(result_frameset);
    result_frameset = NULL;
    sph_ifs_lenslet_model_delete(lensmodel);
    lensmodel = NULL;
    return cpl_error_get_code();
}

static cpl_error_code sph_ifs_science_dr_find_max_angle(cpl_frameset* inframes,
        double* maxangle) {
    cpl_frame* aframe = NULL;
    double minang = 0.0;
    double maxang = 0.0;
    cpl_vector* tmpangles = NULL;
    cpl_ensure_code(inframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(maxangle, CPL_ERROR_NULL_INPUT);

    aframe = cpl_frameset_get_first(inframes);

    while (aframe) {
        tmpangles = sph_extract_angles_from_cube(aframe, 0.0, 0.0);
        if (tmpangles == NULL) {
            return cpl_error_set_where(cpl_func);
        }
        cpl_vector_sort(tmpangles, CPL_SORT_ASCENDING);
        maxang =
                cpl_vector_get_max(tmpangles) > maxang ?
                        cpl_vector_get_max(tmpangles) : maxang;
        minang =
                cpl_vector_get_min(tmpangles) < minang ?
                        cpl_vector_get_min(tmpangles) : minang;
        aframe = cpl_frameset_get_next(inframes);
        cpl_vector_delete(tmpangles);
        tmpangles = NULL;
    }

    *maxangle = maxang - minang;
    return CPL_ERROR_NONE;
}
/**@}*/
