/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ifs_subtract_dark_scaled.h"
#include "sph_common_keywords.h"

#include <cpl.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Subtract an if needed scaled dark frame from a master_frame
 * @param    self      The master frame to subtract from
 * @param    mdark     The dark frame to subtract
 * @return   CPL_ERROR_NONE or the relevant error code on error
 *
 * This function subtracts the dark frame mdark from self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ifs_subtract_dark_scaled(sph_master_frame* self,
                                            const sph_master_frame* mdark) {
    double dit_dark = 0.0;
    double dit_self = 0.0;
    cpl_boolean dits_ok = CPL_FALSE;
    sph_master_frame* tmpdark = NULL;
    const sph_master_frame* usedark;

    cpl_ensure_code(self != NULL,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mdark != NULL, CPL_ERROR_NULL_INPUT);

    if (self->properties == NULL) {
        cpl_msg_warning(cpl_func, "Raw-frame has no propertylist");
    } else if (mdark->properties == NULL) {
        cpl_msg_warning(cpl_func, "Dark-frame has no propertylist");
    } else if (!cpl_propertylist_has(self->properties,
                                     SPH_COMMON_KEYWORD_SEQ1DIT)) {
        cpl_msg_warning(cpl_func, "Raw-frame has no " SPH_COMMON_KEYWORD_SEQ1DIT
                        " card");
    } else if (!cpl_propertylist_has(mdark->properties,
                                     SPH_COMMON_KEYWORD_SEQ1DIT)) {
        cpl_msg_warning(cpl_func, "Dark-frame has no " SPH_COMMON_KEYWORD_SEQ1DIT
                        " card");
    } else {

        dit_self = cpl_propertylist_get_double(self->properties,
                                               SPH_COMMON_KEYWORD_SEQ1DIT);
        dit_dark = cpl_propertylist_get_double(mdark->properties,
                                               SPH_COMMON_KEYWORD_SEQ1DIT);

        if (dit_self <= 0.0 || dit_dark <= 0.0) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "Non-positive DIT ("
                                         SPH_COMMON_KEYWORD_SEQ1DIT
                                         ") Raw-DIT=%f. dark-DIT=%f.",
                                         dit_dark, dit_self);
        }
        dits_ok = CPL_TRUE;

        if (dit_self != dit_dark) {
#ifdef SPH_IFS_DIT_MATCH
            const double SPH_IFS_SUBTRACT_DARK_SCALED_MIN_DIT = 2.0;

            if (dit_dark <= SPH_IFS_SUBTRACT_DARK_SCALED_MIN_DIT &&
                dit_self >  SPH_IFS_SUBTRACT_DARK_SCALED_MIN_DIT ) {
                cpl_msg_warning(cpl_func, "The DITs do not match for dark "
                                "subtraction: Should not substract a dark with "
                                "a short DIT (%f) from a raw frame with a long "
                                "DIT (%f). Please consider a different dark."
                                "Subtracting dark as is.", dit_dark, dit_self);
            } else if (dit_self <= SPH_IFS_SUBTRACT_DARK_SCALED_MIN_DIT &&
                       dit_dark  > SPH_IFS_SUBTRACT_DARK_SCALED_MIN_DIT ) {
                cpl_msg_warning(cpl_func, "The DITs do not match for dark "
                                "subtraction: Should not substract a dark with "
                                "a long DIT (%f) from a raw frame with a short "
                                "DIT (%f). Please consider a different dark."
                                "Subtracting dark as is.", dit_dark, dit_self);
            } else {
#endif
                /* Need to scale the dark */
                cpl_msg_info(cpl_func, "Multiplying dark by DIT-ratio: %g/%g "
                             "= %g", dit_self, dit_dark, dit_self / dit_dark);

                tmpdark = sph_master_frame_duplicate(mdark);
                sph_master_frame_multiply_double(tmpdark, dit_self / dit_dark);
#ifdef SPH_IFS_DIT_MATCH
            }
#endif
        } else {
            cpl_msg_info(cpl_func, "Subtracting dark with matching DIT: %g",
                         dit_self);
        }
    }

    if (!dits_ok) {
        cpl_msg_warning(cpl_func, "Could not obtain DIT ("
                        SPH_COMMON_KEYWORD_SEQ1DIT ") from dark and/or raw "
                        "frame, so a scaling of the dark is not possible. "
                        "Subtracting dark as it is");
    }

    usedark = tmpdark ? tmpdark : mdark;
    sph_master_frame_subtract_master_frame(self, usedark);
    
    sph_master_frame_delete(tmpdark);

    return cpl_error_get_code() ?
        cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/**@}*/
